/*
 * Driver for EEPROM
 * songwei4@lenovo.com
 * 2014-3-21
 */
#include <linux/i2c.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <linux/cdev.h>
#include <linux/uaccess.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include "kd_camera_hw.h"
#include "eeprom.h"
#include "eeprom_define.h"
#include "cat24c16.h"
#include <asm/system.h>  // for SMP
#include <mach/mt_pm_ldo.h>
#include <linux/dma-mapping.h>
#include "pmic_drv.h"

#define EEPROM_DEBUG  0
#ifdef EEPROM_DEBUG
#define EEPROMDB printk
#else
#define EEPROMDB(x,...)
#endif

#define EEPROM_I2C_BUSNUM 0
#define EEPROM_DRVNAME "EEPROM_CAT24C16"
#define IMX214_DOVDD_SUPPLY_GPIO (GPIO105|0x80000000)
static struct i2c_board_info __initdata kd_eeprom_dev={ I2C_BOARD_INFO("EEPROM_CAT24C16", CAT24C16_DEVICE_ID>>1)};
static const struct i2c_device_id EEPROM_i2c_id[] = {{EEPROM_DRVNAME,0},{}};
static DEFINE_SPINLOCK(g_EEPROMLock); // for SMP
/*******************************************************************************
 *
 ********************************************************************************/
static struct i2c_client * g_pstI2Cclient = NULL;

static dev_t g_EEPROMdevno = MKDEV(EEPROM_DEV_MAJOR_NUMBER,0);
static struct cdev * g_pEEPROM_CharDrv = NULL;

static struct class *EEPROM_class = NULL;
static atomic_t g_EEPROMatomic=ATOMIC_INIT(0);
/*******************************************************************************
 *
 ********************************************************************************/
#define PAGE_SIZE 16
static int device_page_wirte(u16 internal_addr, u32 length, u8 *pdata)
{
	int  i4RetValue = 0;
	int one_time_len,write_len=0;

	u32 phyAddr = 0;
	u8 *buf = NULL;

	if(internal_addr+length>=0x800)
	{
		EEPROMDB("[CAT24C16 EEPROM] Write Error!! device not supprt address >= 0x800!! \n" );
		return -1;
	}

	//alloct DMA memory
	phyAddr = 0;
	buf = dma_alloc_coherent(0, PAGE_SIZE+1, &phyAddr, GFP_KERNEL);
	if (NULL == buf) {
		EEPROMDB("eeprom cat24c16 Not enough memory \n");
		return -1;
	}

	do{
		//first caculate length we want to wirte.
		one_time_len=PAGE_SIZE-(internal_addr&0xf);
		if(write_len+one_time_len>length)
			one_time_len-=(write_len+one_time_len-length);
		
		//then set slave device addr and data to send.
		spin_lock(&g_EEPROMLock); //for SMP
		g_pstI2Cclient->addr=((CAT24C16_DEVICE_ID>>1)&0x78)|((internal_addr>>8)&0x7);
		g_pstI2Cclient->ext_flag = (g_pstI2Cclient->ext_flag)|(I2C_DMA_FLAG);
		spin_unlock(&g_EEPROMLock); //for SMP

		buf[0]=internal_addr;
		memcpy(buf+1, pdata, one_time_len);
		i4RetValue = i2c_master_send(g_pstI2Cclient,(u8 *)phyAddr,one_time_len+1);
		if (i4RetValue != (one_time_len+1))
		{
			EEPROMDB("[CAT24C16 EEPROM] I2C write  failed!! \n");
			dma_free_coherent(0,PAGE_SIZE+1, buf, phyAddr);
			return -1;
		}

		//update params.
		pdata+=one_time_len;
		internal_addr+=one_time_len;
		write_len+=one_time_len;

		//wait for device write finish and this need on write cycle.
		//for Twr singnal --> write data form buffer to memory.
		mdelay(5); 
	}while(write_len!=length);
	
	dma_free_coherent(0,PAGE_SIZE+1, buf, phyAddr);

	return 0;
}

static int device_byte_write(u16 internal_addr,u8 data)
{
	int  i4RetValue = 0;
	char puSendCmd[2]={(char)(internal_addr&0xff),(char)data};

	if(internal_addr>=0x800)
	{
		EEPROMDB("[CAT24C16 EEPROM] Write Error!! device not supprt address >= 0x800!! \n" );
		return -1;
	}
	
	//then set slave device addr and data to send.
	spin_lock(&g_EEPROMLock); //for SMP
	g_pstI2Cclient->addr=((CAT24C16_DEVICE_ID>>1)&0x78)|((internal_addr>>8)&0x7);
	g_pstI2Cclient->ext_flag = (g_pstI2Cclient->ext_flag)&(~I2C_DMA_FLAG);
	spin_unlock(&g_EEPROMLock); //for SMP

	i4RetValue = i2c_master_send(g_pstI2Cclient, puSendCmd,2);
	if (i4RetValue != 2)
	{
		EEPROMDB("[CAT24C16 EEPROM] I2C write  failed!! \n");
		return -1;
	}

	//wait for device write finish and this need on write cycle.
	//for Twr singnal --> write data form buffer to memory.
	mdelay(5); 

	return 0;
}
static int iReadEEPROM(u16 a_u2Addr, u32 ui4_length, u8 * a_puBuff)
{
	int  i4RetValue = 0;
	char puSendCmd=a_u2Addr&0xff;

	if(ui4_length > 8)
	{
		EEPROMDB("[CAT24C16EEPROM] exceed I2c-mt65xx.c 8 bytes limitation\n");
		return -1;
	}
	spin_lock(&g_EEPROMLock); //for SMP
	g_pstI2Cclient->addr=((CAT24C16_DEVICE_ID>>1)&0x78)|((a_u2Addr>>8)&0x7);
	g_pstI2Cclient->ext_flag = (g_pstI2Cclient->ext_flag)&(~I2C_DMA_FLAG);
	spin_unlock(&g_EEPROMLock); // for SMP

	struct i2c_adapter *adap =g_pstI2Cclient->adapter; 
	struct i2c_msg msg[2];

	//dummy write.
	msg[0].addr=g_pstI2Cclient->addr;
	msg[0].flags=g_pstI2Cclient->flags & I2C_M_TEN;
	msg[0].len=1;
	msg[0].buf=&puSendCmd;
	msg[0].timing=g_pstI2Cclient->timing;
	msg[0].ext_flag=g_pstI2Cclient->ext_flag;
	//read.	
	msg[1].addr  =g_pstI2Cclient->addr;
	msg[1].flags =g_pstI2Cclient->flags & I2C_M_TEN;
	msg[1].flags |= I2C_M_RD;
	msg[1].len   =ui4_length;
	msg[1].buf   =(char *)a_puBuff;
	msg[1].timing=g_pstI2Cclient->timing;
	msg[1].ext_flag=g_pstI2Cclient->ext_flag;
	i4RetValue = i2c_transfer(adap, msg, 2);
	
	return (i4RetValue >0) ? 0 : i4RetValue;
}
static int iReadData(unsigned int  ui4_offset, unsigned int  ui4_length, unsigned char * pinputdata)
{
	int  i4RetValue = 0;
	unsigned int  i4ResidueDataLength;
	u32 u4IncOffset = 0,read_len=0;
	u32 u4CurrentOffset;
	u8 * pBuff;

	if (ui4_offset + ui4_length >= 0x800)
	{
		EEPROMDB("[CAT24C16EEPROM] Read Error!! not supprt address >= 0x2000!! \n" );
		return -1;
	}

	i4ResidueDataLength = ui4_length;
	u4CurrentOffset = ui4_offset;
	pBuff = pinputdata;
	do 
	{
		if(i4ResidueDataLength >= 8)
			read_len=8;
		else 
			read_len=i4ResidueDataLength;

		i4RetValue = iReadEEPROM((u16)u4CurrentOffset, read_len, pBuff);
		if (i4RetValue != 0)
		{
			EEPROMDB("[EEPROM] I2C iReadData failed!! \n");
			return -1;
		}           
		u4IncOffset += read_len;
		i4ResidueDataLength -= read_len;
		u4CurrentOffset = ui4_offset + u4IncOffset;
		pBuff = pinputdata + u4IncOffset;
	}while (i4ResidueDataLength > 0);
	return 0;
}


static int iWriteData(unsigned int  ui4_offset, unsigned int  ui4_length, unsigned char * pinputdata)
{
	if(pinputdata==NULL)
	{
		EEPROMDB("[CAT24C16 EEPROM] Read Error!! input data is empty! \n" );
		return -1;
	}
	if(ui4_length<16)
	{
		int i;
		for(i=0;i<ui4_length;i++)
			if(device_byte_write((u16)ui4_offset+i,pinputdata[i])!=0)
			{
				printk("cat24c16 byte write error, have writed [%d] byte[s]----\n",i);
				return -1;
			}

		return 0;
	}else{

		return device_page_wirte((u16)ui4_offset,ui4_length,pinputdata);
	}
		
}

/*******************************************************************************
 *
 ********************************************************************************/
#define NEW_UNLOCK_IOCTL
#ifndef NEW_UNLOCK_IOCTL
static int EEPROM_Ioctl(struct inode * a_pstInode,
		struct file * a_pstFile,
		unsigned int a_u4Command,
		unsigned long a_u4Param)
#else 
static long EEPROM_Ioctl(
		struct file *file, 
		unsigned int a_u4Command, 
		unsigned long a_u4Param
		)
#endif
{
	int i4RetValue = 0;
	u8 * pBuff = NULL;
	u8 * pWorkingBuff = NULL;
	stEEPROM_INFO_STRUCT *ptempbuf;

#ifdef EEPROMGETDLT_DEBUG
	struct timeval ktv1, ktv2;
	unsigned long TimeIntervalUS;
#endif

	if(_IOC_NONE == _IOC_DIR(a_u4Command))
	{
	}
	else
	{
		pBuff = (u8 *)kmalloc(sizeof(stEEPROM_INFO_STRUCT),GFP_KERNEL);

		if(NULL == pBuff)
		{
			EEPROMDB("[CAT24C16 EEPROM] ioctl allocate mem failed\n");
			return -ENOMEM;
		}

		if(_IOC_WRITE & _IOC_DIR(a_u4Command))
		{
			if(copy_from_user((u8 *) pBuff , (u8 *) a_u4Param, sizeof(stEEPROM_INFO_STRUCT)))
			{    //get input structure address
				kfree(pBuff);
				EEPROMDB("[CAT24C16 EEPROM] ioctl copy from user failed\n");
				return -EFAULT;
			}
		}
	}

	ptempbuf = (stEEPROM_INFO_STRUCT *)pBuff;
	pWorkingBuff = (u8*)kmalloc(ptempbuf->u4Length,GFP_KERNEL); 
	if(NULL == pWorkingBuff)
	{
		kfree(pBuff);
		EEPROMDB("[CAT24C16 EEPROM] ioctl allocate mem failed\n");
		return -ENOMEM;
	}
	EEPROMDB("[CAT24C16 EEPROM] init Working buffer address 0x%8x  command is 0x%8x\n", (u32)pWorkingBuff, (u32)a_u4Command);


	if(copy_from_user((u8*)pWorkingBuff ,  (u8*)ptempbuf->pu1Params, ptempbuf->u4Length))
	{
		kfree(pBuff);
		kfree(pWorkingBuff);
		EEPROMDB("[CAT24C16 EEPROM] ioctl copy from user failed\n");
		return -EFAULT;
	} 

	switch(a_u4Command)
	{
		case EEPROMIOC_S_WRITE:    
			EEPROMDB("[CAT24C16 EEPROM] Write CMD \n");
#ifdef EEPROMGETDLT_DEBUG
			do_gettimeofday(&ktv1);
#endif            
			i4RetValue = iWriteData((u16)ptempbuf->u4Offset, ptempbuf->u4Length, pWorkingBuff);
#ifdef EEPROMGETDLT_DEBUG
			do_gettimeofday(&ktv2);
			if(ktv2.tv_sec > ktv1.tv_sec)
			{
				TimeIntervalUS = ktv1.tv_usec + 1000000 - ktv2.tv_usec;
			}
			else
			{
				TimeIntervalUS = ktv2.tv_usec - ktv1.tv_usec;
			}
			printk("Write data %d bytes take %lu us\n",ptempbuf->u4Length, TimeIntervalUS);
#endif            
			break;
		case EEPROMIOC_G_READ:
			EEPROMDB("[CAT24C16 EEPROM] Read CMD \n");
#ifdef EEPROMGETDLT_DEBUG            
			do_gettimeofday(&ktv1);
#endif 
			EEPROMDB("[EEPROM] offset %d \n", ptempbuf->u4Offset);
			EEPROMDB("[EEPROM] length %d \n", ptempbuf->u4Length);
			EEPROMDB("[EEPROM] Before read Working buffer address 0x%8x \n", (u32)pWorkingBuff);

			i4RetValue = iReadData((u16)ptempbuf->u4Offset, ptempbuf->u4Length, pWorkingBuff);
			EEPROMDB("[CAT24C16 EEPROM] After read Working buffer data  0x%4x \n", *pWorkingBuff);


#ifdef EEPROMGETDLT_DEBUG
			do_gettimeofday(&ktv2);
			if(ktv2.tv_sec > ktv1.tv_sec)
			{
				TimeIntervalUS = ktv1.tv_usec + 1000000 - ktv2.tv_usec;
			}
			else
			{
				TimeIntervalUS = ktv2.tv_usec - ktv1.tv_usec;
			}
			printk("Read data %d bytes take %lu us\n",ptempbuf->u4Length, TimeIntervalUS);
#endif            

			break;
		default :
			EEPROMDB("[CAT24C16 EEPROM] No CMD \n");
			i4RetValue = -EPERM;
			break;
	}

	if(_IOC_READ & _IOC_DIR(a_u4Command))
	{
		//copy data to user space buffer, keep other input paremeter unchange.
		EEPROMDB("[CAT24C16 EEPROM] to user length %d \n", ptempbuf->u4Length);
		EEPROMDB("[CAT24C16 EEPROM] to user  Working buffer address 0x%8x \n", (u32)pWorkingBuff);
		if(copy_to_user((u8 __user *) ptempbuf->pu1Params , (u8 *)pWorkingBuff , ptempbuf->u4Length))
		{
			kfree(pBuff);
			kfree(pWorkingBuff);
			EEPROMDB("[CAT24C16 EEPROM] ioctl copy to user failed\n");
			return -EFAULT;
		}
	}

	kfree(pBuff);
	kfree(pWorkingBuff);
	return i4RetValue;
}


static u32 g_u4Opened = 0;
//#define
//Main jobs:
// 1.check for device-specified errors, device not ready.
// 2.Initialize the device if it is opened for the first time.
static int EEPROM_Open(struct inode * a_pstInode, struct file * a_pstFile)
{
	EEPROMDB("[CAT24C16 EEPROM] EEPROM_Open\n");
	spin_lock(&g_EEPROMLock);
	if(g_u4Opened)
	{
		spin_unlock(&g_EEPROMLock);
		return -EBUSY;
	}
	else
	{
		g_u4Opened = 1;
		atomic_set(&g_EEPROMatomic,0);
	}
	spin_unlock(&g_EEPROMLock);

	//dovdd control by gpio dc-dc.
	mt_set_gpio_mode(IMX214_DOVDD_SUPPLY_GPIO,GPIO_MODE_00);
	mt_set_gpio_dir(IMX214_DOVDD_SUPPLY_GPIO,GPIO_DIR_OUT);
	mt_set_gpio_out(IMX214_DOVDD_SUPPLY_GPIO,GPIO_OUT_ONE);
	return 0;
}

//Main jobs:
// 1.Deallocate anything that "open" allocated in private_data.
// 2.Shut down the device on last close.
// 3.Only called once on last time.
// Q1 : Try release multiple times.
static int EEPROM_Release(struct inode * a_pstInode, struct file * a_pstFile)
{
	spin_lock(&g_EEPROMLock);
	g_u4Opened = 0;
	atomic_set(&g_EEPROMatomic,0);
	spin_unlock(&g_EEPROMLock);

	mt_set_gpio_out(IMX214_DOVDD_SUPPLY_GPIO,GPIO_OUT_ZERO);
	return 0;
}

static const struct file_operations g_stEEPROM_fops =
{
	.owner = THIS_MODULE,
	.open = EEPROM_Open,
	.release = EEPROM_Release,
	//.ioctl = EEPROM_Ioctl
	.unlocked_ioctl = EEPROM_Ioctl
};

#define EEPROM_DYNAMIC_ALLOCATE_DEVNO 1
inline static int RegisterEEPROMCharDrv(void)
{
	struct device* EEPROM_device = NULL;

#if EEPROM_DYNAMIC_ALLOCATE_DEVNO
	if( alloc_chrdev_region(&g_EEPROMdevno, 0, 1,EEPROM_DRVNAME) )
	{
		EEPROMDB("[CAT24C16 EEPROM] Allocate device no failed\n");

		return -EAGAIN;
	}
#else
	if( register_chrdev_region(  g_EEPROMdevno , 1 , EEPROM_DRVNAME) )
	{
		EEPROMDB("[CAT24C16 EEPROM] Register device no failed\n");

		return -EAGAIN;
	}
#endif

	//Allocate driver
	g_pEEPROM_CharDrv = cdev_alloc();

	if(NULL == g_pEEPROM_CharDrv)
	{
		unregister_chrdev_region(g_EEPROMdevno, 1);

		EEPROMDB("[CAT24C16 EEPROM] Allocate mem for kobject failed\n");

		return -ENOMEM;
	}

	//Attatch file operation.
	cdev_init(g_pEEPROM_CharDrv, &g_stEEPROM_fops);

	g_pEEPROM_CharDrv->owner = THIS_MODULE;

	//Add to system
	if(cdev_add(g_pEEPROM_CharDrv, g_EEPROMdevno, 1))
	{
		EEPROMDB("[CAT24C16 EEPROM] Attatch file operation failed\n");

		unregister_chrdev_region(g_EEPROMdevno, 1);

		return -EAGAIN;
	}

	EEPROM_class = class_create(THIS_MODULE, "EEPROMdrv");
	if (IS_ERR(EEPROM_class)) {
		int ret = PTR_ERR(EEPROM_class);
		EEPROMDB("Unable to create class, err = %d\n", ret);
		return ret;
	}
	EEPROM_device = device_create(EEPROM_class, NULL, g_EEPROMdevno, NULL, EEPROM_DRVNAME);

	return 0;
}

inline static void UnregisterEEPROMCharDrv(void)
{
	//Release char driver
	cdev_del(g_pEEPROM_CharDrv);

	unregister_chrdev_region(g_EEPROMdevno, 1);

	device_destroy(EEPROM_class, g_EEPROMdevno);
	class_destroy(EEPROM_class);
}


//////////////////////////////////////////////////////////////////////
#if EEPROM_DEBUG
static unsigned char test_data[256]={"lenovo test cat24c16 0x1,0x2,3,4,5,6,7,8,9,a,b,c,d,e,f,sw-test!!lenovo test cat24c16 0x1,0x2,3,4,5,6,7,8,9,a,b,c,d,e,f,sw-test!!lenovo test cat24c16 0x1,0x2,3,4,5,6,7,8,9,a,b,c,d,e,f,sw-test!!lenovo test cat24c16 0x1,0x2,3,4,5,6,7,8,9,a,b,c,d,e,f,sw-test!!"};
static unsigned int test_off=0;test_len=16;
static char wr_flag;
static ssize_t show_cat24c16_STATUS(struct device *dev,struct device_attribute *attr, char *buf)
{
	unsigned char test_buf[256]={0};
	if(test_len>256)
		test_len=256;
	printk("---[%s][%d]--------[%d-%d]\n",__FUNCTION__,__LINE__,test_off,test_len);	

	iReadData(test_off,test_len,test_buf);	
	sprintf(buf,"[%s]",test_buf);
	return test_len+3;
}
static ssize_t store_cat24c16_STATUS(struct device *dev,struct device_attribute *attr, const char *buf, size_t size)
{
	if(buf!=NULL)
		sscanf(buf,"%d %d %c",&test_off,&test_len,&wr_flag);
	printk("-------cat24c16---%d--%d--%c--\n",test_off,test_len,wr_flag);
	if(wr_flag=='w')
	{
		if(test_len>256)
			test_len=256;
		iWriteData(test_off,test_len,test_data);	
	}	
	return size;
}
static DEVICE_ATTR(cat24c16_STATUS, 0664, show_cat24c16_STATUS, store_cat24c16_STATUS);
#endif
static int EEPROM_i2c_probe(struct i2c_client *client, const struct i2c_device_id *id) {             
	int i4RetValue = 0;
	EEPROMDB("------[CAT24C16 EEPROM] Attach I2C--I2C addr=[0x%x]\n",client->addr);

	//get sensor i2c client
	spin_lock(&g_EEPROMLock); //for SMP
	g_pstI2Cclient = client;
	g_pstI2Cclient->addr = CAT24C16_DEVICE_ID>>1;
	spin_unlock(&g_EEPROMLock); // for SMP    

	EEPROMDB("[EEPROM] g_pstI2Cclient->addr = 0x%8x \n",g_pstI2Cclient->addr);
	//Register char driver
	i4RetValue = RegisterEEPROMCharDrv();

	if(i4RetValue){
		EEPROMDB("[CAT24C16 EEPROM] register char device failed!\n");
		return i4RetValue;
	}

#if EEPROM_DEBUG
	//create sys entry.
	device_create_file(&(client->dev), &dev_attr_cat24c16_STATUS);
#endif
	EEPROMDB("[CAT24C16 EEPROM] Attached!! \n");
	return 0;                                                                                       
} 

static int EEPROM_i2c_remove(struct i2c_client *client)
{
	return 0;
}

static struct i2c_driver EEPROM_i2c_driver = {
	.probe = EEPROM_i2c_probe,                                   
	.remove = EEPROM_i2c_remove,                           
	//   .detect = EEPROM_i2c_detect,                           
	.driver.name = EEPROM_DRVNAME,
	.id_table = EEPROM_i2c_id,                             
};


static int EEPROM_probe(struct platform_device *pdev)
{
	return i2c_add_driver(&EEPROM_i2c_driver);
}

static int EEPROM_remove(struct platform_device *pdev)
{
	i2c_del_driver(&EEPROM_i2c_driver);
	return 0;
}

// platform structure
static struct platform_driver g_stEEPROM_Driver = {
	.probe		= EEPROM_probe,
	.remove	= EEPROM_remove,
	.driver		= {
		.name	= EEPROM_DRVNAME,
		.owner	= THIS_MODULE,
	}
};


static struct platform_device g_stEEPROM_Device = {
	.name = EEPROM_DRVNAME,
	.id = 0,
	.dev = {
	}
};

static int __init EEPROM_i2C_init(void)
{
	i2c_register_board_info(EEPROM_I2C_BUSNUM, &kd_eeprom_dev, 1);
	if(platform_driver_register(&g_stEEPROM_Driver)){
		EEPROMDB("failed to register CAT24C16 EEPROM driver\n");
		return -ENODEV;
	}

	if (platform_device_register(&g_stEEPROM_Device))
	{
		EEPROMDB("failed to register CAT24C16 EEPROM driver, 2nd time\n");
		return -ENODEV;
	}	

	return 0;
}

static void __exit EEPROM_i2C_exit(void)
{
	platform_driver_unregister(&g_stEEPROM_Driver);
}

module_init(EEPROM_i2C_init);
module_exit(EEPROM_i2C_exit);

MODULE_DESCRIPTION("EEPROM driver");
MODULE_AUTHOR("Lenovo <songwei4@lenovo.com>");
MODULE_LICENSE("GPL");


