/*
 * Copyright (C) 2008 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <fcntl.h>
#include <errno.h>
#include <math.h>
#include <poll.h>
#include <unistd.h>
#include <dirent.h>
#include <sys/select.h>
#include <cutils/log.h>
#include <utils/BitSet.h>
#include <math.h>

#include "CwMcuSensor.h"

#define LOG_TAG "CwMcuHAL"

/*****************************************************************************/

CwMcuSensor::CwMcuSensor()
    : SensorBase(NULL, "CwMcuSensor"),
      mEnabled(0),
      mInputReader(4),
      mHasPendingEvent(false){

    mPendingEvents[CW_ACCELERATION].version = sizeof(sensors_event_t);
    mPendingEvents[CW_ACCELERATION].sensor = CW_ACCELERATION;
    mPendingEvents[CW_ACCELERATION].type = SENSOR_TYPE_ACCELEROMETER;
	
    mPendingEvents[CW_MAGNETIC].version = sizeof(sensors_event_t);
    mPendingEvents[CW_MAGNETIC].sensor = CW_MAGNETIC;
    mPendingEvents[CW_MAGNETIC].type = SENSOR_TYPE_MAGNETIC_FIELD;
	
    mPendingEvents[CW_GYRO].version = sizeof(sensors_event_t);
    mPendingEvents[CW_GYRO].sensor = CW_GYRO;
    mPendingEvents[CW_GYRO].type = SENSOR_TYPE_GYROSCOPE;

	mPendingEvents[CW_LIGHT].version = sizeof(sensors_event_t);
	mPendingEvents[CW_LIGHT].sensor = CW_LIGHT;
	mPendingEvents[CW_LIGHT].type = SENSOR_TYPE_LIGHT;

	mPendingEvents[CW_PROXIMITY].version = sizeof(sensors_event_t);
	mPendingEvents[CW_PROXIMITY].sensor = CW_PROXIMITY;
	mPendingEvents[CW_PROXIMITY].type = SENSOR_TYPE_PROXIMITY;

    mPendingEvents[CW_ORIENTATION].version = sizeof(sensors_event_t);
    mPendingEvents[CW_ORIENTATION].sensor = CW_ORIENTATION;
    mPendingEvents[CW_ORIENTATION].type = SENSOR_TYPE_ORIENTATION;
    mPendingEvents[CW_ORIENTATION].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

    mPendingEvents[CW_ROTATIONVECTOR].version = sizeof(sensors_event_t);
    mPendingEvents[CW_ROTATIONVECTOR].sensor = CW_ROTATIONVECTOR;
    mPendingEvents[CW_ROTATIONVECTOR].type = SENSOR_TYPE_ROTATION_VECTOR;
    mPendingEvents[CW_ROTATIONVECTOR].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

    mPendingEvents[CW_LINEARACCELERATION].version = sizeof(sensors_event_t);
    mPendingEvents[CW_LINEARACCELERATION].sensor = CW_LINEARACCELERATION;
    mPendingEvents[CW_LINEARACCELERATION].type = SENSOR_TYPE_LINEAR_ACCELERATION;
    mPendingEvents[CW_LINEARACCELERATION].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

    mPendingEvents[CW_GRAVITY].version = sizeof(sensors_event_t);
    mPendingEvents[CW_GRAVITY].sensor = CW_GRAVITY;
    mPendingEvents[CW_GRAVITY].type = SENSOR_TYPE_GRAVITY;
    mPendingEvents[CW_GRAVITY].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;
    
    mPendingEvents[CW_PEDOMETER].version = sizeof(sensors_event_t);
    mPendingEvents[CW_PEDOMETER].sensor = CW_PEDOMETER;
    mPendingEvents[CW_PEDOMETER].type = SENSOR_TYPE_STEP_COUNTER;
    mPendingEvents[CW_PEDOMETER].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;
	mPendingEvents[CW_PEDOMETER].u64.step_counter = 0;
    
    mPendingEvents[CW_SNAP].version = sizeof(sensors_event_t);
    mPendingEvents[CW_SNAP].sensor = CW_SNAP;
    mPendingEvents[CW_SNAP].type = /*SENSOR_TYPE_ROTATION_VECTOR*/33171002;
    mPendingEvents[CW_SNAP].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;
    
    mPendingEvents[CW_SHAKE].version = sizeof(sensors_event_t);
    mPendingEvents[CW_SHAKE].sensor = CW_SHAKE;
    mPendingEvents[CW_SHAKE].type = /*SENSOR_TYPE_ROTATION_VECTOR*/33171003;
    mPendingEvents[CW_SHAKE].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;
    
    mPendingEvents[CW_TAP].version = sizeof(sensors_event_t);
    mPendingEvents[CW_TAP].sensor = CW_TAP;
    mPendingEvents[CW_TAP].type = /*SENSOR_TYPE_ROTATION_VECTOR*/33171004;
    mPendingEvents[CW_TAP].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

    mPendingEvents[CW_PEDOMETER_DETECTOR].version = sizeof(sensors_event_t);
    mPendingEvents[CW_PEDOMETER_DETECTOR].sensor = CW_PEDOMETER_DETECTOR;
    mPendingEvents[CW_PEDOMETER_DETECTOR].type = SENSOR_TYPE_ROTATION_VECTOR;
    mPendingEvents[CW_PEDOMETER_DETECTOR].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

    mPendingEvents[CW_MAGNETIC_UNCALIBRATED].version = sizeof(sensors_event_t);
    mPendingEvents[CW_MAGNETIC_UNCALIBRATED].sensor = CW_MAGNETIC_UNCALIBRATED;
    mPendingEvents[CW_MAGNETIC_UNCALIBRATED].type = SENSOR_TYPE_MAGNETIC_FIELD_UNCALIBRATED;
    mPendingEvents[CW_MAGNETIC_UNCALIBRATED].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

    mPendingEvents[CW_GYROSCOPE_UNCALIBRATED].version = sizeof(sensors_event_t);
    mPendingEvents[CW_GYROSCOPE_UNCALIBRATED].sensor = CW_GYROSCOPE_UNCALIBRATED;
    mPendingEvents[CW_GYROSCOPE_UNCALIBRATED].type = SENSOR_TYPE_GYROSCOPE_UNCALIBRATED;
    mPendingEvents[CW_GYROSCOPE_UNCALIBRATED].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

    mPendingEvents[CW_GAME_ROTATION_VECTOR].version = sizeof(sensors_event_t);
    mPendingEvents[CW_GAME_ROTATION_VECTOR].sensor = CW_GAME_ROTATION_VECTOR;
    mPendingEvents[CW_GAME_ROTATION_VECTOR].type = SENSOR_TYPE_GAME_ROTATION_VECTOR;
    mPendingEvents[CW_GAME_ROTATION_VECTOR].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

    mPendingEvents[CW_GEOMAGNETIC_ROTATION_VECTOR].version = sizeof(sensors_event_t);
    mPendingEvents[CW_GEOMAGNETIC_ROTATION_VECTOR].sensor = CW_GEOMAGNETIC_ROTATION_VECTOR;
    mPendingEvents[CW_GEOMAGNETIC_ROTATION_VECTOR].type = SENSOR_TYPE_GEOMAGNETIC_ROTATION_VECTOR;
    mPendingEvents[CW_GEOMAGNETIC_ROTATION_VECTOR].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

    mPendingEvents[CW_SIGNIFICANT_MOTION].version = sizeof(sensors_event_t);
    mPendingEvents[CW_SIGNIFICANT_MOTION].sensor = CW_SIGNIFICANT_MOTION;
    mPendingEvents[CW_SIGNIFICANT_MOTION].type = SENSOR_TYPE_SIGNIFICANT_MOTION;
    mPendingEvents[CW_SIGNIFICANT_MOTION].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

	mPendingEvents[CW_FLIP].version = sizeof(sensors_event_t);
    mPendingEvents[CW_FLIP].sensor = CW_FLIP;
    mPendingEvents[CW_FLIP].type = /*SENSOR_TYPE_ROTATION_VECTOR*/33171005;
    mPendingEvents[CW_FLIP].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

	mPendingEvents[CW_TWIST].version = sizeof(sensors_event_t);
    mPendingEvents[CW_TWIST].sensor = CW_TWIST;
    mPendingEvents[CW_TWIST].type = /*SENSOR_TYPE_ROTATION_VECTOR*/33171006;
    mPendingEvents[CW_TWIST].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

    mPendingEvents[CW_RGB].version = sizeof(sensors_event_t);
    mPendingEvents[CW_RGB].sensor = CW_RGB;
    mPendingEvents[CW_RGB].type = /*SENSOR_TYPE_ROTATION_VECTOR*/33171013;
    mPendingEvents[CW_RGB].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

    mPendingEvents[CW_PROXIMITY_GESTURE].version = sizeof(sensors_event_t);
    mPendingEvents[CW_PROXIMITY_GESTURE].sensor = CW_PROXIMITY_GESTURE;
    mPendingEvents[CW_PROXIMITY_GESTURE].type = /*SENSOR_TYPE_ROTATION_VECTOR*/33171011;
    mPendingEvents[CW_PROXIMITY_GESTURE].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

	mPendingEvents[CW_SCREEN_ON].version = sizeof(sensors_event_t);
    mPendingEvents[CW_SCREEN_ON].sensor = CW_SCREEN_ON;
    mPendingEvents[CW_SCREEN_ON].type = /*SENSOR_TYPE_ROTATION_VECTOR*/33171010;
    mPendingEvents[CW_SCREEN_ON].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

	mPendingEvents[CW_PROXIMITY_SCREEN_ON].version = sizeof(sensors_event_t);
    mPendingEvents[CW_PROXIMITY_SCREEN_ON].sensor = CW_PROXIMITY_SCREEN_ON;
    mPendingEvents[CW_PROXIMITY_SCREEN_ON].type = /*SENSOR_TYPE_ROTATION_VECTOR*/33171014;
    mPendingEvents[CW_PROXIMITY_SCREEN_ON].orientation.status = SENSOR_STATUS_ACCURACY_HIGH;

	mPendingEvents_flush.version = META_DATA_VERSION;
	mPendingEvents_flush.sensor = 0;
	mPendingEvents_flush.type = SENSOR_TYPE_META_DATA;

	ALOGE("--SENHAL-- fd =%d",data_fd);
    if (data_fd) {
        strcpy(input_sysfs_path, "/sys/class/input/");
        strcat(input_sysfs_path, input_name);
        strcat(input_sysfs_path, "/device/");
        input_sysfs_path_len = strlen(input_sysfs_path);
        enable(0, 1);

		int temp_data[255];
		strcpy(&input_sysfs_path[input_sysfs_path_len], "calibrator_cmd");
		ALOGE("--SENHAL-- CYWEE input_sysfs_path %s\n",input_sysfs_path);
		//cw_read_calibrator_file(CW_GYRO, SAVE_PATH_GYRO, temp_data);
		//cw_save_calibrator_sysfs(CW_GYRO, input_sysfs_path, temp_data);
/*
		int test_data[33];
		int read_data[33];
		for(int i=0; i<33; i++) {
			test_data[i] = i+31;
		}
		cw_save_calibrator_file(CW_GYRO, SAVE_PATH_GYRO, test_data);
		cw_read_calibrator_file(CW_GYRO, SAVE_PATH_GYRO, read_data);
*/
    }
}

CwMcuSensor::~CwMcuSensor() {
    if (mEnabled) {
        enable(0, 0);
    }
}

int CwMcuSensor::setInitialState() {

    return 0;
}

int CwMcuSensor::find_sensor(int32_t handle){
	int what = -1;
	return what;
}

int CwMcuSensor::enable(int32_t handle, int en) {

	int what = -1;
	int err = 0;
	int flags = en ? 1 : 0;
	int fd;
	char buf[10];
	what = handle;
	if (uint32_t(what) >= CW_SENSORS_ID_END)
		return -EINVAL;
	strcpy(&input_sysfs_path[input_sysfs_path_len], "enable");
	fd = open(input_sysfs_path, O_RDWR);
	if (fd >= 0) {
		sprintf(buf, "%d %d\n", what, flags);		
		err = write(fd, buf, sizeof(buf));
		close(fd);
		mEnabled &= ~(1<<what);
		mEnabled |= (uint32_t(flags)<<what);
	}
	
	ALOGE("--SENHAL-- enable() fd=%d sensors_id =%d enable =%d mEnabled=%d path=%s",fd ,what,flags,mEnabled,input_sysfs_path);
    return 0;
}

int CwMcuSensor::batch(int handle, int flags, int64_t period_ns, int64_t timeout)
{
	int what = -1;
	int fd = 0;
	char buf[128] = {0};
	int err = 0;
	int delay_ms = 0;
	int timeout_ms = 0;
	bool dryRun = false;

	ALOGE("--SENHAL-- %s in\n", __func__);

	what = handle;
	delay_ms = period_ns/1000000;
	timeout_ms = timeout/1000000;

	if(flags == SENSORS_BATCH_DRY_RUN)
	{
		ALOGE("--SENHAL-- SENSORS_BATCH_DRY_RUN~!!\n");
		dryRun = true;
	}

	if (uint32_t(what) >= CW_SENSORS_ID_END)
		return -EINVAL;

	if(flags == SENSORS_BATCH_WAKE_UPON_FIFO_FULL)
	{
		ALOGE("--SENHAL-- SENSORS_BATCH_WAKE_UPON_FIFO_FULL~!!\n");
	}

	switch(what)
	{
		//these sensors do not suport batch mode.
		case CW_AIRMOUSE:
		case CW_SNAP:
		case CW_SHAKE:
		case CW_TAP:
			return 0;
	}
/*
	if(dryRun == true)
	{
		return 0;
	}
*/
	strcpy(&input_sysfs_path[input_sysfs_path_len], "batch");

	fd = open(input_sysfs_path, O_RDWR);

	if(fd >= 0)
	{
		sprintf(buf, "%d %d %d %d\n", what, flags, delay_ms, timeout_ms);
		err = write(fd, buf, sizeof(buf));
		close(fd);
	}

	ALOGE("--SENHAL-- batch() fd=%d, sensors_id =%d, flags =%d, delay_ms= %d, timeout = %d, path=%s\n",fd , what, flags, delay_ms, timeout_ms, input_sysfs_path);

	return 0;
}

int CwMcuSensor::flush(int handle)
{
	int what = -1;
	int fd = 0;
	char buf[10] = {0};
	int err = 0;

	what = handle;

	if (uint32_t(what) >= CW_SENSORS_ID_END)
		return -EINVAL;

	strcpy(&input_sysfs_path[input_sysfs_path_len], "flush");

	fd = open(input_sysfs_path, O_RDWR);

	if(fd >= 0)
	{
		sprintf(buf, "%d\n", what);
		err = write(fd, buf, sizeof(buf));
		close(fd);
	}

	ALOGE("--SENHAL-- flush() fd=%d, sensors_id =%d, path=%s\n",fd , what, input_sysfs_path);

	return 0;
}

bool CwMcuSensor::hasPendingEvents() const {
    return mHasPendingEvent;
}

int CwMcuSensor::setDelay(int32_t handle, int64_t delay_ns)
{
	char buf[80];
    int fd;
    int what = 0;
    int delay_ms = (int)delay_ns/1000000;
	what = handle;
		if (uint32_t(what) >= CW_SENSORS_ID_END)
			return -EINVAL;
		strcpy(&input_sysfs_path[input_sysfs_path_len], "delay_ms");
		fd = open(input_sysfs_path, O_RDWR);
		if (fd >= 0) {
			sprintf(buf, "%d %d\n", what, delay_ms);		
			write(fd, buf, strlen(buf)+1);
			close(fd);
		}
		ALOGE("--SENHAL-- setDelay() sensors_id =%d delay_ms =%d path=%s",what,delay_ms,input_sysfs_path);
    return 0;
}

int CwMcuSensor::readEvents(sensors_event_t* data, int count)
{
	static int64_t prev_time;
	if (count < 1)
		return -EINVAL;
	ssize_t n = mInputReader.fill(data_fd);
	if (n < 0)
		return n;

	int numEventReceived = 0;
	input_event const* event;

	while (count && mInputReader.readEvent(&event)) {
		if (event->type == EV_ABS) {
			processEvent(event->code, event->value);
		} else if (event->type == EV_SYN) {
			int64_t time = timevalToNano(event->time);
			for (int j=0 ; count && mPendingMask && j<CW_SENSORS_ID_END ; j++) {
				if (mPendingMask & (1<<j)) {
					mPendingMask &= ~(1<<j);
					mPendingEvents[j].timestamp = time;
					if (mEnabled & (1<<j)) {
						*data++ = mPendingEvents[j];
						count--;
						numEventReceived++;
					}
				}
			}
			if(flush_event ==1)
			{
				ALOGE("--SENHAL-- send flush event~!!\n");
				flush_event = 0;
				mPendingEvents_flush.timestamp = time;
				*data++ = mPendingEvents_flush;
				count--;
				numEventReceived++;
			}
		} else {}
		mInputReader.next();
	}
	return numEventReceived;
}
void CwMcuSensor::processEvent(int code, int value){
	int sensorsid = 0;
	int index = 0;
	int16_t data = 0;
	uint16_t light = 0;

	sensorsid = (int)((uint32_t )value >> 16);

	if (sensorsid == CW_LIGHT || sensorsid == CW_RGB) {
		light |= value;
	} else {
		data |= value;
	}

	switch (code) {
		case CW_ABS_X:
			index = 0;
		break;
		case CW_ABS_Y:
			index = 1;
		break;
		case CW_ABS_Z:
			index = 2;
		break;
		case CW_ABS_X1:
			index = 3;
		break;
		case CW_ABS_Y1:
			index = 4;
		break;
		case CW_ABS_Z1:
			index = 5;
		break;
		case CW_ABS_TIMEDIFF:
			index = 6;
		break;
		case CW_ABS_M_ACCURACY:
			index = 7;
		break;
		default:
			return;
	}

	switch (sensorsid) {
		case CW_PEDOMETER:
			mPendingMask |= 1<<sensorsid;
			if (index == 0) {
				pedometer_l = (uint16_t)data;
			} else if (index == 1) {
				pedometer_h = (uint16_t)data;
				//mPendingEvents[sensorsid].u64.step_counter = (((uint64_t)pedometer_h)<<16) | (uint64_t)pedometer_l;
			}
			mPendingEvents[sensorsid].u64.step_counter = (((uint64_t)pedometer_h)<<16) | (uint64_t)pedometer_l;
			ALOGE("--SENHAL-- mPendingEvents[sensorsid].u64.step_counter = %lld", mPendingEvents[sensorsid].u64.step_counter);
			break;
		case CW_PROXIMITY:
			mPendingMask |= 1<<sensorsid;
			mPendingEvents[sensorsid].data[index] = (float)data;
                     //add LOG for value-change type sensor is ok. by zhuyawen
                     ALOGE("--SENHAL-- sensors_id=%d,data=%d,index=%d", sensorsid, data, index);
			break;
		case CW_LIGHT:
		case CW_RGB:
			mPendingMask |= 1<<sensorsid;
			mPendingEvents[sensorsid].data[index] = (float)light;
			break;
		case CW_ORIENTATION:
			mPendingMask |= 1<<sensorsid;
			mPendingEvents[sensorsid].data[index] = (float)data	* CONVERT_10;
			if (index == 7) {
				mPendingEvents[sensorsid].orientation.status = data;
				ALOGE("--SENHAL--mPendingEvents[sensorsid].orientation.status = %d", mPendingEvents[sensorsid].orientation.status);
			}
			break;
		case CW_ACCELERATION:
		case CW_GYRO:
		case CW_LINEARACCELERATION:
		case CW_GRAVITY:
			mPendingMask |= 1<<sensorsid;
			mPendingEvents[sensorsid].data[index] = (float)data	* CONVERT_100;
			break;
		case CW_MAGNETIC:
		case CW_MAGNETIC_UNCALIBRATED:
			mPendingMask |= 1<<sensorsid;
			mPendingEvents[sensorsid].data[index] = (float)data	* CONVERT_100;
			//for AKM debug
			//mPendingEvents[sensorsid].data[index] = (float)data*0.06f;
			if (index == 7) {
				mPendingEvents[sensorsid].magnetic.status = data;
				ALOGE("--SENHAL--mPendingEvents[sensorsid].magnetic.status = %d", mPendingEvents[sensorsid].magnetic.status);
			}
			break;
		case CW_ROTATIONVECTOR:
		case CW_GAME_ROTATION_VECTOR:
		case CW_GEOMAGNETIC_ROTATION_VECTOR:
			mPendingMask |= 1<<sensorsid;
			mPendingEvents[sensorsid].data[index] = (float)data	* CONVERT_10000;
			if (index == 2) {
					mPendingEvents[sensorsid].data[3] =
						1 - mPendingEvents[sensorsid].data[0]*mPendingEvents[sensorsid].data[0]
							- mPendingEvents[sensorsid].data[1]*mPendingEvents[sensorsid].data[1]
							- mPendingEvents[sensorsid].data[2]*mPendingEvents[sensorsid].data[2];
					mPendingEvents[sensorsid].data[3] = (mPendingEvents[sensorsid].data[3] > 0) ? (float)sqrt(mPendingEvents[sensorsid].data[3]) : 0;
				}
			ALOGE("--SENHAL-- sensors_id = %d, index = %d, data= %f", sensorsid, 3, mPendingEvents[sensorsid].data[3]);
			break;
		case CW_GYROSCOPE_UNCALIBRATED:
			mPendingMask |= 1<<sensorsid;
			mPendingEvents[sensorsid].data[index] = (float)data	* CONVERT_100;
			break;
		case CW_SNAP:
		case CW_SHAKE:
		case CW_TAP:
		case CW_FLIP:
		case CW_PROXIMITY_GESTURE:
		case CW_SIGNIFICANT_MOTION:
		case CW_PEDOMETER_DETECTOR:
		case CW_SCREEN_ON:
		case CW_PROXIMITY_SCREEN_ON:
			mPendingMask |= 1<<(sensorsid);
			mPendingEvents[sensorsid].data[0] = data;
			mPendingEvents[sensorsid].data[1] = 0;
			mPendingEvents[sensorsid].data[2] = 0;
			ALOGE("--SENHAL-- sensors_id=%d,data=%d", sensorsid, data);
			break;
		case CW_META_DATA:
			mPendingEvents_flush.meta_data.what = META_DATA_FLUSH_COMPLETE;
			mPendingEvents_flush.meta_data.sensor = data;
			flush_event = 1;
			ALOGE("--SENHAL-- sensors_id=%d,meta_data.what=%d, meta_data.sensor =%d", sensorsid, mPendingEvents_flush.meta_data.what, mPendingEvents_flush.meta_data.sensor);
			break;
	}
}

int CwMcuSensor::cw_read_calibrator_sysfs(int type, char *path, int *calib)
{
	int fd = 0;
	char value[1024];
	int readBytes = 0;

	if (type == CW_GYRO) {

		if ((fd = open(path, O_RDONLY)) < 0) {
			ALOGE("--SENHAL--  Can't open '%s': %s", path, strerror(errno));
			return -1;
		}

		if ((readBytes = read(fd, value, sizeof(value))) < 0) {
			close(fd);
			return -1;
		}

		sscanf(value, "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
			&calib[0],  &calib[1],  &calib[2],
			&calib[3],  &calib[4],  &calib[5],  &calib[6],  &calib[7],  &calib[8],  &calib[9],  &calib[10], &calib[11], &calib[12],
			&calib[13], &calib[14], &calib[15], &calib[16], &calib[17], &calib[18], &calib[19], &calib[20], &calib[21], &calib[22],
			&calib[23], &calib[24], &calib[25], &calib[26], &calib[27], &calib[28], &calib[29], &calib[30], &calib[31], &calib[32]);

		for (int i = 0; i < 33; i++) {
			ALOGE("--SENHAL-- read sysfs calib[%d] = %d\n", i, calib[i]);
		}
	}
	return 0;
}

int CwMcuSensor::cw_save_calibrator_sysfs(int type, char *path, int *calib)
{
	int i = 0;
	uint32_t data[33]={0};
	int fd = 0;
	int error =0;
	char value[1024];//255

	snprintf(value, sizeof(value), "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
			calib[0],  calib[1],  calib[2],
			calib[3],  calib[4],  calib[5],  calib[6],  calib[7],  calib[8],  calib[9],  calib[10], calib[11], calib[12],
			calib[13], calib[14], calib[15], calib[16], calib[17], calib[18], calib[19], calib[20], calib[21], calib[22],
			calib[23], calib[24], calib[25], calib[26], calib[27], calib[28], calib[29], calib[30], calib[31], calib[32]);

	ALOGE("--SENHAL-- value = %s\n", value);
	ALOGE("--SENHAL-- value size = %d\n", sizeof(value));

	if (type == CW_GYRO) {
		if ((fd = open(path, O_WRONLY)) < 0) {
			ALOGE("--SENHAL--  Can't open '%s': %s", path, strerror(errno));
		    return -1;
		}

		if (write(fd, value, sizeof(value)) < 0) {
			ALOGE("--SENHAL-- save calib to sysfs error: %d %s\n",fd, strerror(errno));
		    close(fd);
		    return -1;
		}
		close(fd);
	}
	return 0;
}

int CwMcuSensor::cw_read_calibrator_file(int type, char *path, int *calib)
{
	FILE *fp;
	int readBytes = 0;

	fp = fopen(path, "r");
	if (!fp) {
		ALOGE("open File failed: %s\n", strerror(errno));
		return 0;
	}
	rewind(fp);

	if (type == CW_GYRO) {
		calib[0] = 2;
		calib[1] = 1;
		calib[2] = CW_GYRO;

		readBytes = fscanf(fp, "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
			&calib[3],  &calib[4],  &calib[5],  &calib[6],  &calib[7],  &calib[8],  &calib[9],  &calib[10], &calib[11], &calib[12],
			&calib[13], &calib[14], &calib[15], &calib[16], &calib[17], &calib[18], &calib[19], &calib[20], &calib[21], &calib[22],
			&calib[23], &calib[24], &calib[25], &calib[26], &calib[27], &calib[28], &calib[29], &calib[30], &calib[31], &calib[32]);

		if (readBytes <= 0) {
			ALOGE("--SENHAL-- Read file error:%s\n", strerror(errno));
			return readBytes;
		} else {
			for (int i = 0; i < 33; i++) {
				ALOGE("--SENHAL-- calib[%d] = %d\n", i, calib[i]);
			}
		}
	}
	fclose(fp);

	return 	0;
}

int CwMcuSensor::cw_save_calibrator_file(int type, char *path, int *calib)
{
	FILE *fp = NULL;
	int readBytes = 0;
	char value[1024];

	fp = fopen(path, "w");
	if(!fp) {
		ALOGE("--SENHAL-- open calibrator file error!\n");
		return -1;
	}

	snprintf(value, sizeof(value), "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
			calib[0],  calib[1],  calib[2],
			calib[3],  calib[4],  calib[5],  calib[6],  calib[7],  calib[8],  calib[9],  calib[10], calib[11], calib[12],
			calib[13], calib[14], calib[15], calib[16], calib[17], calib[18], calib[19], calib[20], calib[21], calib[22],
			calib[23], calib[24], calib[25], calib[26], calib[27], calib[28], calib[29], calib[30], calib[31], calib[32]);

	readBytes = fprintf(fp,"%s",value);

	if (readBytes <= 0) {
		ALOGE("--SENHAL-- write calibrator file error!\n");
		fclose(fp);
		return -1;
	}

	fclose(fp);
	return 0;
}
