/*
 * Copyright (C) 2008 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");

 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG "Sensors"

#include <hardware/sensors.h>
#include <fcntl.h>
#include <errno.h>
#include <dirent.h>
#include <math.h>
#include <poll.h>
#include <pthread.h>
#include <stdlib.h>

#include <linux/input.h>

#include <utils/Atomic.h>
#include <utils/Log.h>

#include "CwMcuSensor.h"
#include "sensors.h"

/*****************************************************************************/

#define DELAY_OUT_TIME 0x7FFFFFFF

#define LIGHT_SENSOR_POLLTIME    2000000000
#define AKM_FTRACE 0
#define AKM_DEBUG 0
#define AKM_DATA 0

/*****************************************************************************/
#ifndef PROXIMITY_RANGE		
#define PROXIMITY_RANGE 		1.00f
#endif	
#ifndef PROXIMITY_RESOLUTION		
#define PROXIMITY_RESOLUTION  		1.0f	
#endif	
#ifndef PROXIMITY_POWER					
#define PROXIMITY_POWER			0.13f
#endif

/* The SENSORS Module */
static const struct sensor_t sSensorList[] = {
        { "Cywee Accelerometer Sensor",
			"CyWee Group Ltd.",
			1, CW_ACCELERATION,
			SENSOR_TYPE_ACCELEROMETER, RANGE_A, CONVERT_100, 0.23f, 20000, 0, 0, { } },
        { "Cywee Magnetic field Sensor",
			"CyWee Group Ltd.",
			1, CW_MAGNETIC,
			SENSOR_TYPE_MAGNETIC_FIELD, 200.0f, CONVERT_100, 6.8f, 20000, 0, 0, { } },
        { "Cywee Gyroscope Sensor",
			"CyWee Group Ltd.",
			1, CW_GYRO,
			SENSOR_TYPE_GYROSCOPE, 40, CONVERT_100, 6.1f, 20000, 0, 0, { } },
		{ "Cywee Light Sensor",
			"CyWee Group Ltd.",
			1, CW_LIGHT,
			SENSOR_TYPE_LIGHT, 10240.0f, 1.0f, 0.13f, 1190, 0, 0, { } },
		{ "Cywee Proximity Sensor",
			"CyWee Group Ltd.",
			1, CW_PROXIMITY,
			SENSOR_TYPE_PROXIMITY, PROXIMITY_RANGE, PROXIMITY_RESOLUTION, PROXIMITY_POWER, 1190, 0, 0, { } },
		{ "Cywee Orientation Sensor",
			"CyWee Group Ltd.",
			1, CW_ORIENTATION,
			SENSOR_TYPE_ORIENTATION, 360.0f, 0.1f, 13.0f, 20000, 0, 0, { } },
		{ "Cywee Rotation Vector",
			"CyWee Group Ltd.",
			1, CW_ROTATIONVECTOR,
			SENSOR_TYPE_ROTATION_VECTOR, 1.0f, 1.0f / (1<<24), 6.1f, 20000, 0, 0, { } },
		{ "Cywee Linear Acceleration",
			"CyWee Group Ltd.",
			1, CW_LINEARACCELERATION,
			SENSOR_TYPE_LINEAR_ACCELERATION, RANGE_A, RESOLUTION_A, 0.2f, 20000, 0, 0, { } },
		{ "Cywee Gravity",
			"CyWee Group Ltd.",
			1, CW_GRAVITY,
			SENSOR_TYPE_GRAVITY, GRAVITY_EARTH, (4.0f*9.81f)/256.0f, 0.2f, 20000, 0, 0, { } },
		{ "Cywee Step Counter",
			"CyWee Group Ltd.",
			1, CW_PEDOMETER,
			SENSOR_TYPE_STEP_COUNTER, 20000.0f, 1.0f, 6.1f, 20000, 0, 0, { } },
		{ "Cywee Snap",
			"CyWee Group Ltd.",
			1, CW_SNAP,
			/*SENSOR_TYPE_ROTATION_VECTOR*/33171002, 10.0f, 1.0f, 6.1f, 0, 0, 0, { } },
		{ "Cywee Shake",
			"CyWee Group Ltd.",
			1, CW_SHAKE,
			/*SENSOR_TYPE_ROTATION_VECTOR*/33171003, 10.0f, 1.0f, 6.1f, 0, 0, 0, { } },
		{ "Cywee Tap",
			"CyWee Group Ltd.",
			1, CW_TAP,
			/*SENSOR_TYPE_ROTATION_VECTOR*/33171004, 10.0f, 1.0f, 6.1f, 0, 0, 0, { } },
		{ "Cywee Step Detector",
			"CyWee Group Ltd.",
			1, CW_PEDOMETER_DETECTOR,
			SENSOR_TYPE_STEP_DETECTOR, 2.0f, 1.0f, 0, 0, 0, 0, { } },
		{ "Cywee Uncalibrated Magnetic Field Sensor",
			"CyWee Group Ltd.",
			1, CW_MAGNETIC_UNCALIBRATED,
			SENSOR_TYPE_MAGNETIC_FIELD_UNCALIBRATED, 200.0f, CONVERT_100, 6.8f, 20000, 0, 0, { } },
        { "Cywee Uncalibrated Gyroscope Sensor",
			"CyWee Group Ltd.",
			1, CW_GYROSCOPE_UNCALIBRATED,
			SENSOR_TYPE_GYROSCOPE_UNCALIBRATED, 40, CONVERT_100, 6.1f, 20000, 0, 0, { } },
		{ "Cywee Game Rotation Vector",
			"CyWee Group Ltd.",
			1, CW_GAME_ROTATION_VECTOR,
			SENSOR_TYPE_GAME_ROTATION_VECTOR, 1.0f, 1.0f / (1<<24), 6.1f, 20000, 0, 0, { } },
		{ "Cywee Geomagnetic Rotation Vector",
			"CyWee Group Ltd.",
			1, CW_GEOMAGNETIC_ROTATION_VECTOR,
			SENSOR_TYPE_GEOMAGNETIC_ROTATION_VECTOR, 1.0f, 1.0f / (1<<24), 6.1f, 20000, 0, 0, { } },
		{ "Cywee Significant Motions",
			"CyWee Group Ltd.",
			1, CW_SIGNIFICANT_MOTION,
			SENSOR_TYPE_SIGNIFICANT_MOTION, 2.0f, 1.0f, 0, 0, 0, 0, { } },
		{ "Cywee Flip",
			"CyWee Group Ltd.",
			1, CW_FLIP,
			/*SENSOR_TYPE_ROTATION_VECTOR*/33171005, 10.0f, 1.0f, 6.1f, 0, 0, 0, { } },
		{ "Cywee Twist",
			"CyWee Group Ltd.",
			1, CW_TWIST,
			/*SENSOR_TYPE_ROTATION_VECTOR*/33171006, 10.0f, 1.0f, 6.1f, 0, 0, 0, { } },
		{ "Cywee RGB",
			"CyWee Group Ltd.",
			1, CW_RGB,
			/*SENSOR_TYPE_ROTATION_VECTOR*/33171013, 1.0f, 1.0f, 6.1f, 0, 0, 0, { } },
		{ "Cywee Proximity Gesture",
			"CyWee Group Ltd.",
			1, CW_PROXIMITY_GESTURE,
			/*SENSOR_TYPE_ROTATION_VECTOR*/33171011, 1.0f, 1.0f, 6.1f, 0, 0, 0, { } },
		{ "Cywee Screen on",
			"CyWee Group Ltd.",
			1, CW_SCREEN_ON,
			/*SENSOR_TYPE_ROTATION_VECTOR*/33171010, 1.0f, 1.0f, 6.1f, 0, 0, 0, { } },
		{ "Cywee Proximity Screen on",
			"CyWee Group Ltd.",
			1, CW_PROXIMITY_SCREEN_ON,
			/*SENSOR_TYPE_ROTATION_VECTOR*/33171014, 1.0f, 1.0f, 6.1f, 0, 0, 0, { } }

};
static int open_sensors(const struct hw_module_t* module, const char* id,
                        struct hw_device_t** device);

static int sensors__get_sensors_list(struct sensors_module_t* module,
                                     struct sensor_t const** list)
{
        *list = sSensorList;
        return ARRAY_SIZE(sSensorList);
}

static struct hw_module_methods_t sensors_module_methods = {
        open: open_sensors
};

struct sensors_module_t HAL_MODULE_INFO_SYM = {
    .common = {
        .tag = HARDWARE_MODULE_TAG,
        .version_major = 1,
        .version_minor = 0,
        .id = SENSORS_HARDWARE_MODULE_ID,
        .name = "CyWee SENSORS Module",
        .author = "CyWee Group Ltd.",
        .methods = &sensors_module_methods,
		.dso = NULL,
		.reserved = {0}
        },
        get_sensors_list: sensors__get_sensors_list,
};

struct sensors_poll_context_t {
    sensors_poll_device_1_t device; // must be first

	sensors_poll_context_t();
	~sensors_poll_context_t();
    int activate(int handle, int enabled);
    int setDelay(int handle, int64_t ns);
    int pollEvents(sensors_event_t* data, int count);
    int batch(int handle, int flags, int64_t period_ns, int64_t timeout);
    int flush(int handle);

private:
    enum {
        cwmcu			= 0,
        numSensorDrivers,
        numFds,
    };

    static const size_t wake = numFds - 1;
    static const char WAKE_MESSAGE = 'W';
    struct pollfd mPollFds[numFds];
    int mWritePipeFd;
    SensorBase* mSensors[numSensorDrivers];

int handleToDriver(int handle) const {
	switch (handle) {
		case CW_ACCELERATION:
		case CW_MAGNETIC:
		case CW_GYRO:
		case CW_LIGHT:
		case CW_PROXIMITY:
		case CW_ORIENTATION:
		case CW_ROTATIONVECTOR:
		case CW_LINEARACCELERATION:
		case CW_GRAVITY:
		case CW_PEDOMETER:
		case CW_SNAP:
		case CW_SHAKE:
		case CW_TAP:
		case CW_MAGNETIC_UNCALIBRATED:
		case CW_GYROSCOPE_UNCALIBRATED:
		case CW_GAME_ROTATION_VECTOR:
		case CW_GEOMAGNETIC_ROTATION_VECTOR:
		case CW_SIGNIFICANT_MOTION:
		case CW_PEDOMETER_DETECTOR:
		case CW_FLIP:
		case CW_TWIST:
		case CW_RGB:
		case CW_PROXIMITY_GESTURE:
		case CW_SCREEN_ON:
		case CW_PROXIMITY_SCREEN_ON:
			return cwmcu;
	}
        return -EINVAL;
    }
};

/*****************************************************************************/

sensors_poll_context_t::sensors_poll_context_t()
{
    mSensors[cwmcu] = new CwMcuSensor();
    mPollFds[cwmcu].fd = mSensors[cwmcu]->getFd();
    mPollFds[cwmcu].events = POLLIN;
    mPollFds[cwmcu].revents = 0;

	ALOGE("--SENSORHAL-- mPollFds[cwmcu].fd = %d\n", mSensors[cwmcu]->getFd());

    int wakeFds[2];
    int result = pipe(wakeFds);
    //LOGE_IF(result<0, "error creating wake pipe (%s)", strerror(errno));
    fcntl(wakeFds[0], F_SETFL, O_NONBLOCK);
    fcntl(wakeFds[1], F_SETFL, O_NONBLOCK);
    mWritePipeFd = wakeFds[1];

    mPollFds[wake].fd = wakeFds[0];
    mPollFds[wake].events = POLLIN;
    mPollFds[wake].revents = 0;
}

sensors_poll_context_t::~sensors_poll_context_t() {
    for (int i=0 ; i<numSensorDrivers ; i++) {
        delete mSensors[i];
    }
    close(mPollFds[wake].fd);
    close(mWritePipeFd);
}

int sensors_poll_context_t::activate(int handle, int enabled) {
    int index = handleToDriver(handle);
    if (index < 0) return index;
    int err =  mSensors[index]->enable(handle, enabled);
    if (enabled && !err) {
        const char wakeMessage(WAKE_MESSAGE);
        int result = write(mWritePipeFd, &wakeMessage, 1);
    }
    return err;
}

int sensors_poll_context_t::setDelay(int handle, int64_t ns) {

    int index = handleToDriver(handle);
    if (index < 0) return index;
    return mSensors[index]->setDelay(handle, ns);
}

int sensors_poll_context_t::pollEvents(sensors_event_t* data, int count)
{
    int nbEvents = 0;
    int n = 0;
    do {
        // see if we have some leftover from the last poll()
        for (int i=0 ; count && i<numSensorDrivers ; i++) {
            SensorBase* const sensor(mSensors[i]);
            if ((mPollFds[i].revents & POLLIN) || (sensor->hasPendingEvents())) {
                int nb = sensor->readEvents(data, count);
                if (nb < count) {
                    // no more data for this sensor
                    mPollFds[i].revents = 0;
                }
                count -= nb;
                nbEvents += nb;
                data += nb;
            }
        }

        if (count) {
            // we still have some room, so try to see if we can get
            // some events immediately or just wait if we don't have
            // anything to return
            do {
                n = poll(mPollFds, numFds, nbEvents ? 0 : -1);
            } while (n < 0 && errno == EINTR);
            if (n<0) {
                return -errno;
            }
            if (mPollFds[wake].revents & POLLIN) {
                char msg;
                int result = read(mPollFds[wake].fd, &msg, 1);
                mPollFds[wake].revents = 0;
            }
        }
        // if we have events and space, go read them
    } while (n && count);

    return nbEvents;
}

int sensors_poll_context_t::batch(int handle, int flags, int64_t period_ns, int64_t timeout)
{
	ALOGE("--SENHAL-- %s\n", __func__);

	int index = handleToDriver(handle);

    if (index < 0)
		return index;

	ALOGE("--SENHAL-- handle = %d, flags = %d, period_ns = %lld, timeout = %lld\n", handle, flags, period_ns, timeout);

	int err = mSensors[index]->batch(handle, flags, period_ns, timeout);

    return err;
}

int sensors_poll_context_t::flush(int handle)
{
	ALOGE("--SENHAL-- %s\n", __func__);

	int index = handleToDriver(handle);

    if (index < 0)
		return index;

	int err = mSensors[index]->flush(handle);

    return 0;
}

/*****************************************************************************/

static int poll__close(struct hw_device_t *dev)
{
    sensors_poll_context_t *ctx = (sensors_poll_context_t *)dev;
    if (ctx) {
        delete ctx;
    }
    return 0;
}

static int poll__activate(struct sensors_poll_device_t *dev,
        int handle, int enabled) {
    sensors_poll_context_t *ctx = (sensors_poll_context_t *)dev;
    return ctx->activate(handle, enabled);
}

static int poll__setDelay(struct sensors_poll_device_t *dev,
        int handle, int64_t ns) {
    sensors_poll_context_t *ctx = (sensors_poll_context_t *)dev;
    return ctx->setDelay(handle, ns);
}

static int poll__poll(struct sensors_poll_device_t *dev,
        sensors_event_t* data, int count) {
    sensors_poll_context_t *ctx = (sensors_poll_context_t *)dev;
    return ctx->pollEvents(data, count);
}

static int poll__batch(struct sensors_poll_device_1* dev,
            int handle, int flags, int64_t period_ns, int64_t timeout)
{
	sensors_poll_context_t *ctx = (sensors_poll_context_t *)dev;
	return ctx->batch(handle, flags, period_ns, timeout);
}

static int poll__flush(struct sensors_poll_device_1* dev, int handle)
{
	sensors_poll_context_t *ctx = (sensors_poll_context_t *)dev;
	return ctx->flush(handle);
}

/*****************************************************************************/

/** Open a new instance of a sensor device using name */
static int open_sensors(const struct hw_module_t* module, const char* id,
                        struct hw_device_t** device)
{
        int status = -EINVAL;
        sensors_poll_context_t *dev = new sensors_poll_context_t();

        memset(&dev->device, 0, sizeof(sensors_poll_device_1_t));

        dev->device.common.tag = HARDWARE_DEVICE_TAG;
        dev->device.common.version  = SENSORS_DEVICE_API_VERSION_1_1;	//1.1 support batch, setdelay() is not used. SENSORS_DEVICE_API_VERSION_1_1
        dev->device.common.module   = const_cast<hw_module_t*>(module);
        dev->device.common.close    = poll__close;
        dev->device.activate        = poll__activate;
        dev->device.setDelay        = poll__setDelay;
        dev->device.poll            = poll__poll;

        dev->device.batch			= poll__batch;
        dev->device.flush			= poll__flush;

        *device = &dev->device.common;
        status = 0;

        return status;
}

