/*
 * drivers/leds/lp5521_leds.c
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive for
 * more details.
 *
 * lp5521 leds driver
 *
 *
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <linux/string.h>
#include <linux/ctype.h>
#include <linux/leds.h>
#include <linux/leds-mt65xx.h>
#include <linux/workqueue.h>
#include <linux/wakelock.h>
#include <linux/slab.h>
#include <linux/i2c.h>
#include <mach/mt_gpio.h>
#include <linux/uaccess.h>
#include <linux/fb.h>
#include <linux/debugfs.h>


/****************************************************************************
 * defined
 ***************************************************************************/
#define LED_NAME "rgbled"
#define I2C_MASTER_CLOCK       400
#ifdef GPIO_LED_EN
#undef GPIO_LED_EN
#endif
#define GPIO_LED_EN GPIO169


static int lp5521_is_init= 0;
static int debug_enable = 1;
static int last_brightness=0;

#define LEDS_DEBUG(format, args...) do{ \
	if(debug_enable) \
	{\
		printk(KERN_EMERG format,##args);\
	}\
}while(0)

struct lp5521_leds_priv {
	struct led_classdev cdev;
	struct work_struct work;
	int gpio;
	int level;
	int delay_on;
	int delay_off;
};
typedef struct
{
	void (*Charging_RGB_LED)(unsigned int value);
   
} LENOVO_LED_CONTROL_FUNCS;

//extern void lenovo_register_led_control(LENOVO_LED_CONTROL_FUNCS * ctrl);


/****************************************************************************
 * local functions
 ***************************************************************************/

static int	lp5521_leds_i2c_probe(struct i2c_client *client, const struct i2c_device_id *id);
//static int  lp5521_leds_i2c_detect(struct i2c_client *client, int kind, struct i2c_board_info *info);
static int  lp5521_leds_i2c_remove(struct i2c_client *client);


static int __init lp5521_leds_platform_probe(struct platform_device *pdev);
static int lp5521_leds_platform_remove(struct platform_device *pdev);

static void breath_red_led_on(void) ;
static void breath_green_led_on(void) ;
static void breath_blue_led_on(void) ;

static void red_led_on(void) ;
static void green_led_on(void) ;
static void blue_led_on(void) ;
////////////////

#define LP5521_I2C_ADDR 0x64
#define LP5521_BUS_NUM 3
struct i2c_client *lp5521_i2c_cilent = NULL;

static struct i2c_board_info __initdata lp5521_i2c_board_info = { I2C_BOARD_INFO("lp5521", (LP5521_I2C_ADDR >> 1))};

static const struct i2c_device_id lp5521_i2c_id[] = {{"lp5521",0}, {}};
static struct i2c_driver lp5521_i2c_driver = {
    .probe = lp5521_leds_i2c_probe,
    .remove = lp5521_leds_i2c_remove,
    .driver.name = "lp5521",
    .id_table = lp5521_i2c_id,
    //.address_list = (const unsigned short *) forces,
};


static struct platform_driver lp5521_leds_platform_driver = {
	.driver		= {
		.name	= "leds-lp5521",
		//.owner	= THIS_MODULE,
	},
	.probe		= lp5521_leds_platform_probe,
	.remove		= lp5521_leds_platform_remove,
	//.suspend	= lp5521_leds_platform_suspend,
	//.shutdown   = lp5521_leds_platform_shutdown,
};


static void lp5521_udelay(UINT32 us)
{
	udelay(us);
}

static void lp5521_mdelay(UINT32 ms)
{
	msleep(ms);
}

static int lp5521_i2c_txdata(char *txdata, int len)
{
    int ret;
    struct i2c_msg msg[] = {
            {
                .addr = lp5521_i2c_cilent->addr,
                .flags = 0,
                .len =len,
                .buf = txdata,
            },
    };
    if(lp5521_i2c_cilent != NULL) {
        ret = i2c_transfer(lp5521_i2c_cilent->adapter, msg, 1);
        if(ret < 0)
            pr_err("%s i2c write erro: %d\n", __func__, ret);
    } else {
        LEDS_DEBUG("lp5521_i2c_cilent null\n");
    }
    return ret;
}

static int lp5521_write_reg(u8 addr, u8 para)
{
	LEDS_DEBUG("[LED]%s\n", __func__);

    u8 buf[3];
    int ret = -1;

    buf[0] = addr;
    buf[1] = para;
    ret = lp5521_i2c_txdata(buf,2);
    if(ret < 0) {
        LEDS_DEBUG("%s write reg failed! addr=0x%x para=0x%x ret=%d\n", __func__,buf[0],buf[1],ret);
        return -1;
    }
    return 0;
}

static int lp5521_init(void)
{
	LEDS_DEBUG("[LED]+%s\n", __func__);

    mt_set_gpio_mode(GPIO_LED_EN,GPIO_MODE_GPIO);
    mt_set_gpio_dir(GPIO_LED_EN,GPIO_DIR_OUT);
    mt_set_gpio_out(GPIO_LED_EN,GPIO_OUT_ONE);

    lp5521_mdelay(10);

    lp5521_write_reg(0x00, 0x40);
    lp5521_mdelay(5);
    lp5521_write_reg(0x01, 0x15);//config R G B channel

	//lp5521_write_reg(0x02, 0x80);
	//lp5521_write_reg(0x03, 0xC0);
	//lp5521_write_reg(0x04, 0xFF);
#if 0
	lp5521_write_reg(0x50, 0x03);
	lp5521_write_reg(0x51, 0x7F);
	lp5521_write_reg(0x52, 0x4D);

	lp5521_write_reg(0x53, 0x00);  
	lp5521_write_reg(0x54, 0x03);  
	lp5521_write_reg(0x55, 0xFF);  
	lp5521_write_reg(0x56, 0x60);  
	lp5521_write_reg(0x57, 0x00);  

	lp5521_write_reg(0x01, 0X02);
	lp5521_write_reg(0x00, 0x42);
#endif
    lp5521_write_reg(0x08, 0x39);	

	lp5521_is_init = 1;

    LEDS_DEBUG("[LED]-%s\n", __func__);
	return 0;
}
void lp5521_off(void)
{
	LEDS_DEBUG("[LED]%s\n", __func__);

    lp5521_write_reg(0x00, 0x00);
    lp5521_mdelay(5);
    //lp5521_write_reg(0x01, 0x00);
}

void lp5521_rgb_factory_test(void)
{
	LEDS_DEBUG("[LED]%s\n", __func__);
#if 0
    lp5521_write_reg(0x00, 0x20);
    lp5521_write_reg(0x02, 0x01 << 5); //RGB mode

    lp5521_write_reg(0x06, 170); //DOUT3,R
    lp5521_write_reg(0x05, 130); //DOUT2,G
    lp5521_write_reg(0x04, 170); //DOUT1,B

    lp5521_write_reg(0x0C, 0x00);	//R
    lp5521_write_reg(0x12, 0x04);
    lp5521_write_reg(0x18, 0x08);  

    lp5521_write_reg(0x0B, 0x30);	//G
    lp5521_write_reg(0x11, 0x04);
    lp5521_write_reg(0x17, 0x08);  

    lp5521_write_reg(0x0A, 0x40);	//B
    lp5521_write_reg(0x10, 0x04);
    lp5521_write_reg(0x16, 0x08);  
    lp5521_write_reg(0x1C, 1);
    lp5521_write_reg(0x07, 1);	 

    lp5521_write_reg(0x1C, 1);
    lp5521_write_reg(0x01, 0x03); 
    lp5521_write_reg(0x1D, 0x07);
    lp5521_write_reg(0x07, 1);
#endif
    red_led_on();
    lp5521_mdelay(1000);
    green_led_on();
    lp5521_mdelay(1000);
    blue_led_on();
    lp5521_mdelay(1000);
}


static void LP5521_PowerOff_Charging_RGB_LED(unsigned int level)
{
    if(!lp5521_is_init) {
        lp5521_init();
    }
	printk("[JX] %s\n",__func__);
	//lp5521_proc_backlight_value(level,0,0);

}

static void breath_red_led_on(void) //Dout3
{
	    lp5521_write_reg(0x00, 0x40);
	    lp5521_mdelay(5);
	    lp5521_write_reg(0x01, 0x15);//config R G B channel

	lp5521_write_reg(0x10, 0x0B);
	lp5521_write_reg(0x11, 0x7F);
	lp5521_write_reg(0x12, 0x46);
	lp5521_write_reg(0x13, 0x00);  
	lp5521_write_reg(0x14, 0x0B);  
	lp5521_write_reg(0x15, 0xFF);  
	lp5521_write_reg(0x16, 0x7F);  
	lp5521_write_reg(0x17, 0x00);  
	lp5521_write_reg(0x18, 0x7F);  
	lp5521_write_reg(0x19, 0x00);  
	lp5521_write_reg(0x1A, 0x63);  
	lp5521_write_reg(0x1B, 0x00);  

	lp5521_write_reg(0x01, 0X20);
	lp5521_write_reg(0x00, 0x60);
}

static void breath_green_led_on(void) //Dout2
{
    lp5521_write_reg(0x00, 0x40);
    lp5521_mdelay(5);
    lp5521_write_reg(0x01, 0x15);//config R G B channel

	lp5521_write_reg(0x30, 0x0B);
	lp5521_write_reg(0x31, 0x7F);
	lp5521_write_reg(0x32, 0x46);
	lp5521_write_reg(0x33, 0x00);  
	lp5521_write_reg(0x34, 0x0B);  
	lp5521_write_reg(0x35, 0xFF);  
	lp5521_write_reg(0x36, 0x7F);  
	lp5521_write_reg(0x37, 0x00);  
	lp5521_write_reg(0x38, 0x7F);  
	lp5521_write_reg(0x39, 0x00);  
	lp5521_write_reg(0x3A, 0x63);  
	lp5521_write_reg(0x3B, 0x00);  
	
	lp5521_write_reg(0x01, 0X08);
	lp5521_write_reg(0x00, 0x48);
}

static void breath_blue_led_on(void) //Dout2
{
    lp5521_write_reg(0x00, 0x40);
    lp5521_mdelay(5);
    lp5521_write_reg(0x01, 0x15);//config R G B channel

	lp5521_write_reg(0x50, 0x0B);
	lp5521_write_reg(0x51, 0x7F);
	lp5521_write_reg(0x52, 0x46);
	lp5521_write_reg(0x53, 0x00);  
	lp5521_write_reg(0x54, 0x0B);  
	lp5521_write_reg(0x55, 0xFF);  
	lp5521_write_reg(0x56, 0x7F);  
	lp5521_write_reg(0x57, 0x00);  
	lp5521_write_reg(0x58, 0x7F);  
	lp5521_write_reg(0x59, 0x00);  
	lp5521_write_reg(0x5A, 0x63);  
	lp5521_write_reg(0x5B, 0x00);   

	lp5521_write_reg(0x01, 0X02);
	lp5521_write_reg(0x00, 0x42);
}

static void red_led_on(void)
{
    lp5521_write_reg(0x00, 0x40);
    lp5521_mdelay(5);
    lp5521_write_reg(0x01, 0x3F);

    lp5521_write_reg(0x03, 0x00);
    lp5521_write_reg(0x04, 0x00);
    lp5521_write_reg(0x02, 0x80);
}

static void green_led_on(void)
{
    lp5521_write_reg(0x00, 0x40);
    lp5521_mdelay(5);
    lp5521_write_reg(0x01, 0x3F);

    lp5521_write_reg(0x02, 0x00);
    lp5521_write_reg(0x04, 0x00);
    lp5521_write_reg(0x03, 0x80);
}

static void blue_led_on(void)
{
    lp5521_write_reg(0x00, 0x40);
    lp5521_mdelay(5);
    lp5521_write_reg(0x01, 0x3F);

    lp5521_write_reg(0x02, 0x00);
    lp5521_write_reg(0x03, 0x00);
    lp5521_write_reg(0x04, 0x80);
}

static void lp5521_led_work(struct work_struct *work)
{	
	struct lp5521_leds_priv	*led_dat =
		container_of(work, struct lp5521_leds_priv, work);
	LEDS_DEBUG("[LED]%s level=0x%x\n", __func__,led_dat->level);

	switch(led_dat->level)
	{
		case 0:
			lp5521_off();
			break;
		case 1:
			breath_red_led_on();
			break;
		case 2:
			breath_green_led_on();
			break;
		case 3:
			breath_blue_led_on();
			break;
		case 4:
			red_led_on();
			break;
		case 5:
			green_led_on();
			break;
		case 6:
			blue_led_on();
			break;
		case 7:
			lp5521_rgb_factory_test();
			break;
		default:
			break;
	}
}

void lp5521_led_set(struct led_classdev *led_cdev,enum led_brightness value)
{
	LEDS_DEBUG("[LED]%s value=0x%x\n", __func__,value);

	struct lp5521_leds_priv *led_data =
		container_of(led_cdev, struct lp5521_leds_priv, cdev);
	

    if(lp5521_i2c_cilent == NULL) {
        printk("lp5521_i2c_cilent null\n");
        return;
    }
    //cancel_work_sync(&led_data->work);
	led_data->level = value;
        
    if(!lp5521_is_init) {
        lp5521_init();
    }
    schedule_work(&led_data->work);
}

/*for factory test*/
static void lp5521_led_work_test(struct work_struct *work)
{
	struct lp5521_leds_priv	*led_data =
		container_of(work, struct lp5521_leds_priv, work);

if((led_data->level)==0)
	lp5521_off();
else
	lp5521_rgb_factory_test();
}

/*for factory test*/

void lp5521_led_set_test(struct led_classdev *led_cdev,enum led_brightness value)
{
	LEDS_DEBUG("[LED]%s value=%d\n", __func__,value);

	struct lp5521_leds_priv *led_data =
		container_of(led_cdev, struct lp5521_leds_priv, cdev);
	

    if(lp5521_i2c_cilent == NULL) {
        printk("lp5521_i2c_cilent null\n");
        return;
    }
    //cancel_work_sync(&led_data->work);
	led_data->level = value;
        
    if(!lp5521_is_init) {
        lp5521_init();
    }
    schedule_work(&led_data->work);
}


/*wangyq13 for led current start*/
static void lp5521_led_work_current(struct work_struct *work)
{
	struct lp5521_leds_priv	*led_data =
		container_of(work, struct lp5521_leds_priv, work);

if((led_data->level)==0)
	lp5521_off();
else if((led_data->level) <= 0xff)
	{
		lp5521_write_reg(0x05, led_data->level);
		lp5521_write_reg(0x06, led_data->level);
		lp5521_write_reg(0x07, led_data->level);
	}
else if((led_data->level) > 0xff)
	printk("lp5521_led_work_current larger than 0xff\n");
}

void lp5521_led_set_current(struct led_classdev *led_cdev,enum led_brightness value)
{
	LEDS_DEBUG("[LED]%s value=%d\n", __func__,value);

	struct lp5521_leds_priv *led_data =
		container_of(led_cdev, struct lp5521_leds_priv, cdev);
	

    if(lp5521_i2c_cilent == NULL) {
        printk("lp5521_i2c_cilent null\n");
        return;
    }
    //cancel_work_sync(&led_data->work);
	led_data->level = value;
        
    if(!lp5521_is_init) {
        lp5521_init();
    }
    schedule_work(&led_data->work);
}
/*wangyq13 for led current end*/

static int  lp5521_leds_i2c_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int ret=-1;
	LEDS_DEBUG("[LED]%s\n", __func__);
	lp5521_i2c_cilent = client;

    ret = lp5521_init();

    return ret;
 }

static int  lp5521_leds_i2c_remove(struct i2c_client *client)
{
   
   LEDS_DEBUG("[LED]%s\n", __func__);
    return 0;
}


struct lp5521_leds_priv *g_lp5521_leds_data[3];


static LENOVO_LED_CONTROL_FUNCS led_ctrl  = 
{
	.Charging_RGB_LED = LP5521_PowerOff_Charging_RGB_LED,
};


static int __init lp5521_leds_platform_probe(struct platform_device *pdev)
{
	int ret=0;
	int i;
	LEDS_DEBUG("[LED]%s\n", __func__);

	if(i2c_add_driver(&lp5521_i2c_driver))
	{
		printk("add i2c driver error %s\n",__func__);
		return -1;
	} 

	g_lp5521_leds_data[0] = kzalloc(sizeof(struct lp5521_leds_priv), GFP_KERNEL);
	if (!g_lp5521_leds_data[0]) {
		ret = -ENOMEM;
		goto err;
	}

	
	g_lp5521_leds_data[0]->cdev.name = LED_NAME;
	g_lp5521_leds_data[0]->cdev.brightness_set = lp5521_led_set;
	//g_lp5521_leds_data[0]->cdev.max_brightness = 0xffffffff;
	//g_lp5521_leds_data[0]->cdev.blink_set = lp5521_blink_set;
	INIT_WORK(&g_lp5521_leds_data[0]->work, lp5521_led_work);
	g_lp5521_leds_data[0]->gpio = GPIO_LED_EN;
	g_lp5521_leds_data[0]->level = 0;
	
	ret = led_classdev_register(&pdev->dev, &g_lp5521_leds_data[0]->cdev);
	if (ret)
		goto err;

	//for factory test
	g_lp5521_leds_data[1] = kzalloc(sizeof(struct lp5521_leds_priv), GFP_KERNEL);
	if (!g_lp5521_leds_data[1]) {
		ret = -ENOMEM;
		goto err;
	}

	g_lp5521_leds_data[1]->cdev.name = "test-led";
	g_lp5521_leds_data[1]->cdev.brightness_set = lp5521_led_set_test;
	g_lp5521_leds_data[1]->cdev.max_brightness = 0xff;
	INIT_WORK(&g_lp5521_leds_data[1]->work, lp5521_led_work_test);
	g_lp5521_leds_data[1]->gpio = GPIO_LED_EN;
	g_lp5521_leds_data[1]->level = 0;
	
	ret = led_classdev_register(&pdev->dev, &g_lp5521_leds_data[1]->cdev);
	//end for factory test


	//start for lenovo-sw wangyq13 for led current adjust 20140524
	g_lp5521_leds_data[2] = kzalloc(sizeof(struct lp5521_leds_priv), GFP_KERNEL);
	if (!g_lp5521_leds_data[2]) {
		ret = -ENOMEM;
		goto err;
	}

	g_lp5521_leds_data[2]->cdev.name = "current-led";
	g_lp5521_leds_data[2]->cdev.brightness_set = lp5521_led_set_current;
	g_lp5521_leds_data[2]->cdev.max_brightness = 0xff;
	INIT_WORK(&g_lp5521_leds_data[2]->work, lp5521_led_work_current);
	g_lp5521_leds_data[2]->gpio = GPIO_LED_EN;
	g_lp5521_leds_data[2]->level = 0;
	
	ret = led_classdev_register(&pdev->dev, &g_lp5521_leds_data[2]->cdev);
	//end for lenovo-sw wangyq13 for led current adjust 20140524
	
	if (ret)
		goto err;


//	lenovo_register_led_control(&led_ctrl);
	
	return 0;
	
err:

	for (i = 2; i >=0; i--) {
			if (!g_lp5521_leds_data[i])
				continue;
			led_classdev_unregister(&g_lp5521_leds_data[i]->cdev);
			cancel_work_sync(&g_lp5521_leds_data[i]->work);
			kfree(g_lp5521_leds_data[i]);
			g_lp5521_leds_data[i] = NULL;
		}
	i2c_del_driver(&lp5521_i2c_driver);

	return ret;
}

static int lp5521_leds_platform_remove(struct platform_device *pdev)
{
	struct lp5521_leds_priv *priv = dev_get_drvdata(&pdev->dev);
	LEDS_DEBUG("[LED]%s\n", __func__);

	dev_set_drvdata(&pdev->dev, NULL);

	kfree(priv);

	i2c_del_driver(&lp5521_i2c_driver);

	return 0;
}


/***********************************************************************************
* please add platform device in mt_devs.c
*
************************************************************************************/
static int __init lp5521_leds_init(void)
{
	int ret;

	LEDS_DEBUG("[LED]%s\n", __func__);

	if (i2c_register_board_info(LP5521_BUS_NUM, &lp5521_i2c_board_info, 1) !=0) {
		printk(" cann't register i2c %s\n",__func__);
		return -1;
	}

	ret = platform_driver_register(&lp5521_leds_platform_driver);
	if (ret)
	{
		printk("[LED]%s:drv:E%d\n", __func__,ret);
		return ret;
	}

	return ret;
}

static void __exit lp5521_leds_exit(void)
{
	LEDS_DEBUG("[LED]%s\n", __func__);

	i2c_del_driver(&lp5521_i2c_driver);

	platform_driver_unregister(&lp5521_leds_platform_driver);

	lp5521_is_init = 0;
}

module_param(debug_enable, int,0644);

module_init(lp5521_leds_init);
module_exit(lp5521_leds_exit);

MODULE_AUTHOR("jixu@lenovo.com");
MODULE_DESCRIPTION("lp5521 led driver");
MODULE_LICENSE("GPL");


