#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/input.h>
#include <linux/input-polldev.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/pm.h>
#include <linux/pm_runtime.h>
#include <linux/string.h>
#include <linux/gpio.h>
#include <linux/of_gpio.h>
#include <linux/regulator/consumer.h>
#include <linux/firmware.h>
#include <linux/timer.h>
#include <linux/wakelock.h>
#include <linux/fs.h>
#include <linux/file.h>
#include <asm/uaccess.h>
#ifdef CONFIG_FB
#include <linux/notifier.h>
#include <linux/fb.h>
#endif

#include "CwMcuSensor.h"
#include "lpc5400x_update.h"

static struct wake_lock psensor_wake_lock;
static struct wake_lock fwupdate_wake_lock;

/* FEATURE FLAGS SECTION START */
/* turn on to support dvt1 prototype */
#define CONFIG_SNR_HUB_DVT1_COMPATIBLE
/* turn on interrupt reporting if defined */
#define CWMCU_INTERRUPT
/* turn on to recover after detected continous i2c errors to avoid panic. */
//#define CONFIG_SNR_HUB_I2C_HACK
/* turn on this workaround flag to avoid light/gesture conflict align with fw before v08.25 */
//#define CONFIG_SNR_HUB_LG_HACK
/* turn on to avoid compass current leak align with fw before v08.25 */
//#define CONFIG_SNR_HUB_MAG_LEAK_HACK
/* TODO: Move to makefile to auto mark per build type */
#define CONFIG_SNS_HUB_DEBUG
//#define CONFIG_SNR_HUB_VT
#define PRES_FEATURE_DISABLED

/* FEATURE FLAGS SECTION END */

#define DBG_LOG                                 0
#define DBG_LOG_DATA                            1
#define DBG_LOG_BATCH                           2

#define VERBOSE(flag, id, format,...) do {            \
                                if (flagVerbose) \
                                    printk(KERN_DEBUG format, ##__VA_ARGS__); \
                                else if ((flag == DBG_LOG_DATA) && (sensor->sns_debug_log_list & (1<<id))) {\
                                    printk(KERN_DEBUG format, ##__VA_ARGS__); \
                                } \
                            } while (0)

#define TAG_PROX                 "PROX"
#define TAG_CONTAWARENESS        "CONTEXT-AWARENESS"
#define TAG_VOICE_TRIGGER        "VOICE"
#define TAG_SCREENON             "SCREENON"
#define TAG_GESTURE              "PROX-GESTURE"
#define TAG_SSNAP                "S-SNAP"

#define print_event(id, data) do { printk("CWMCU Event %s : %d\n", id, data); } while (0)

#define DPS_MAX                    (1 << (16 - 1))

/* Input poll interval in milliseconds */
#define CWMCU_POLL_INTERVAL        10//10
#define CWMCU_POLL_MAX            200//200
#define CWMCU_POLL_MIN            10//10


#define FT_VTG_MIN_UV        2600000
#define FT_VTG_MAX_UV        3300000
#define FT_I2C_VTG_MIN_UV    1800000
#define FT_I2C_VTG_MAX_UV    1800000
//#define FT_MIC_MIN_UV        3075000
//#define FT_MIC_MAX_UV        3075000

#define I2C_ERRS_MAX        10

#ifdef CONFIG_SNS_HUB_DEBUG
#define SUSPEND_ENABLE_LIST     ((1<<CW_SINGLE_SNAP) | (1<<CW_CONTEXT_AWARE) | (1<<CW_VOICE_TRIGGER) | (1<<CW_SCREEN_ON) | (1<<CW_PROXIMITY) )
#else
#define SUSPEND_ENABLE_LIST     ((1<<CW_CONTEXT_AWARE) | (1<<CW_VOICE_TRIGGER) | (1<<CW_SCREEN_ON) | (1<<CW_PROXIMITY) )
#endif

#ifdef CONFIG_SNR_HUB_VT
#define DMIC_REQ_ID_CODEC                0
#define DMIC_REQ_ID_SNSHUB               1
#define DMIC_REQ_PROC_RELEASE            0
#define DMIC_REQ_PROC_OPEN               1
#define SNSHUB_DMIC_OFF                  0
#define SNSHUB_DMIC_ON                   1
#endif

#define DEBUG_SUPPORT_POWER_KEY_EVENT           (1 << 0)
#define DEBUG_SENSOR_TYPE       ((SUSPEND_ENABLE_LIST) | (1<<(CW_PROXIMITY) | (1<<CW_PROXIMITY_GESTURE)))

static void cwmcu_send_powerkey(void);
static void cwmcu_restore_status(void);
static int cwmcu_check_sensor_alive(void);

static void cwmcu_screen_on_resume(void);
static void cwmcu_screen_off_suspend(void);

// Flag to enable/disable verbose message, for debug convinience.
static volatile int flagVerbose;
#ifdef CONFIG_SNS_HUB_DEBUG
static int queue_len = 0;
#endif

struct CWMCU_data {
    struct i2c_client *client;
    struct regulator *vdd;
    struct regulator *vcc_i2c;
#ifdef CONFIG_SNR_HUB_VT
    struct regulator *mic;
#endif
    struct input_dev *input;
    struct timeval previous;
    struct timeval now;
    int mcu_mode;
    int mcu_status;
    int mcu_power;
    int mcu_screen;
    int mcu_reset;

    struct timer_list timer;

    struct workqueue_struct *driver_wq;
    struct work_struct work;

    /* enable & batch list */
    uint32_t enabled_list;
    uint32_t batched_list;
    uint32_t suspend_enabled_list;
    uint32_t flush_list;
    uint32_t step_counter;

    uint32_t batch_timeout;
    int sensor_timeout[CW_SENSORS_ID_END];

    uint32_t interrupt_status;

    /* report time */
    int64_t    report_period[CW_SENSORS_ID_END];/* ms */
    uint32_t update_list;

    /* power status */
    int power_on_list;

    /* debug */
    int debug_count;

    /* calibrator */
    uint8_t cal_cmd;
    uint8_t cal_type;
    uint8_t cal_id;

    /* gpio */
    int irq_gpio;
    int mic_gpio;
    int wakeup_gpio;
    int reset_gpio;
    int reset_gpio_dvt1;
    int compass_reset_gpio;

    int cmd;
    uint32_t addr;
    int value;
    int mcu_slave_addr;
    int fwupdate_status;
    int cw_i2c_rw;    /* r = 0 , w = 1 */
    int cw_i2c_len;
    uint8_t cw_i2c_data[300];
    char fw_name[32];

    /* check mcu sensor */
    uint8_t check_muc_sensor_alive_flag;
    int poll_count_num;
    int preContinuousSensor[3][3];
    int curContinuousSensor[3][3];
    int sensor_error_count[3];
    int mcu_reset_error_count;
    int pre_reset_count;

#ifdef CONFIG_SNR_HUB_VT
    /* file send variable */
    uint8_t fs_status;
    int32_t fs_size;
    int32_t fs_progress;

    /* udt control */
    int udt_cmd;
    int udt_value;

    /* voice trigger */
    uint8_t pre_voice_trigger_value;
    int dmic_req_codec;
    int dmic_req_snshub;
    bool dmic_on;
    int dmic_codec_release;
    struct work_struct dmic_work;
    bool vt_enabled;
#endif

#ifdef CONFIG_FB
    struct notifier_block fb_notif;
#endif
    struct mutex lock;
    uint32_t sns_debug_flag;
    uint32_t sns_debug_log_list;

    /* workaround for light/gesture conflict */
#ifdef CONFIG_SNR_HUB_LG_HACK
    uint32_t light_x;
    uint32_t light_y;
    uint32_t light_z;
    bool light_keeped;
#endif

#ifdef CONFIG_SNS_HUB_DEBUG
    bool power_test_on;
#endif
    /* Assure not to access i2c if mcu still interrupt after suspend due to unknown intenal issue. */
    bool is_suspend; 
};

struct CWMCU_data *sensor;

/* normalmode: bool: If true, then mcu will reboot to normal mode, otherwise reboot to second-bootloader mode */
static void CWMCU_reset_mcu(bool normalmode)
{
    sensor->pre_reset_count = sensor->mcu_reset;

    if (normalmode) {
        gpio_direction_output(sensor->irq_gpio, 1);
        gpio_direction_output(sensor->mic_gpio, 1);
    } else {
        gpio_direction_output(sensor->irq_gpio, 0);
        gpio_direction_output(sensor->mic_gpio, 0);
    }
    gpio_direction_output(sensor->reset_gpio, 1);
#ifdef CONFIG_SNR_HUB_DVT1_COMPATIBLE
    gpio_direction_output(sensor->reset_gpio_dvt1, 1);
#endif
    usleep_range(19000, 20000);
    gpio_set_value(sensor->reset_gpio, 0);
#ifdef CONFIG_SNR_HUB_DVT1_COMPATIBLE
    gpio_set_value(sensor->reset_gpio_dvt1, 0);
#endif
    usleep_range(19000, 20000);
    gpio_set_value(sensor->reset_gpio, 1);
#ifdef CONFIG_SNR_HUB_DVT1_COMPATIBLE
    gpio_set_value(sensor->reset_gpio_dvt1, 1);
#endif
    usleep(10000);
    gpio_direction_input(sensor->irq_gpio);
    gpio_direction_input(sensor->mic_gpio);
}

/* helper function to avoid vulnerable qualcomm i2c timeout oops.
 * Each i2c ops status should be checked by this function.
 */
static void check_i2c_error_cnt(bool is_err)
{
#ifdef CONFIG_SNR_HUB_I2C_HACK
    static int i2c_err_num = 0;

    if (is_err) {
        i2c_err_num++;
        if (i2c_err_num > I2C_ERRS_MAX) {
            printk("CWMCU too much i2c err, reset mcu!!\n");
            CWMCU_reset_mcu(true);
            i2c_err_num = 0;
        }
    } else
        i2c_err_num = 0;
#endif
}

#ifdef CONFIG_SNR_HUB_VT
extern void msm8x16_codec_micbias1_enable(bool enable);

static int dmic_enable_for_snshub(bool enable)
{
    /*TODO: Add function enable check for voice wakeup*/
    printk("--CWMCU--: %s, enable = %d, dmic_on = %d.\n", __func__, enable, sensor->dmic_on);

    if (enable == DMIC_REQ_PROC_OPEN) {
        if (sensor->dmic_on == SNSHUB_DMIC_OFF) {
            printk("--CWMCU--: %s, change dmic status from [ off ] to [ on ].\n", __func__);
            msm8x16_codec_micbias1_enable(true);
            usleep(20000);
            gpio_direction_output(sensor->wakeup_gpio, 0);
            usleep(20000);
            sensor->dmic_on = SNSHUB_DMIC_ON;
        } else {
            printk("--CWMCU--: %s, change dmic status from [ on ] to [ on ].\n", __func__);
        }
    } else {
        if (sensor->dmic_on == SNSHUB_DMIC_ON) {
            printk("--CWMCU--: %s, change dmic status from [ on ] to [ off ].\n", __func__);
            msm8x16_codec_micbias1_enable(false);
            usleep(20000);
            gpio_direction_output(sensor->wakeup_gpio, 1);
            usleep(20000);
            sensor->dmic_on = SNSHUB_DMIC_OFF;
        } else {
            printk("--CWMCU--: %s, change dmic status from [ off ] to [ off ].\n", __func__);
        }
    }

    return 0;
}

static int snshub_dmic_status_process(int screen, int id, int proc)
{
    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        printk(KERN_DEBUG "--CWMCU--: %s, sensor->mcu_mode = %d\n", __func__, sensor->mcu_mode);
        return -1;
    }

    mutex_lock(&sensor->lock);
    printk("--CWMCU--: %s enter, req_codec = %d, req_snshub = %d, screen = %d, id = %d, proc = %d.\n",
        __func__, sensor->dmic_req_codec, sensor->dmic_req_snshub, screen, id, proc);

    switch (id) {
    case DMIC_REQ_ID_CODEC:
        if (proc == DMIC_REQ_PROC_OPEN) {
            dmic_enable_for_snshub(DMIC_REQ_PROC_RELEASE);
        } else {
            if ((screen == CW_SUSPEND) && (sensor->dmic_req_snshub))
                dmic_enable_for_snshub(DMIC_REQ_PROC_OPEN);
        }
        sensor->dmic_req_codec = proc;
        break;
    case DMIC_REQ_ID_SNSHUB:
        if (proc == DMIC_REQ_PROC_OPEN) {
            if ((screen == CW_SUSPEND) && (!sensor->dmic_req_codec))
                dmic_enable_for_snshub(DMIC_REQ_PROC_OPEN);
            else if (sensor->dmic_req_codec)
                printk("--CWMCU--: %s, snshub cannot open dmic because dmic was used by codec.\n", __func__);
            else if (screen != CW_NORMAL)
                printk("--CWMCU--: %s, snshub cannot open dmic because the screen status is not suspend.\n", __func__);
        } else {
            dmic_enable_for_snshub(DMIC_REQ_PROC_RELEASE);
        }
        sensor->dmic_req_snshub = proc;
        break;
    default:
        break;
    }

    printk("--CWMCU--: %s exit , req_codec = %d, req_snshub = %d.\n", __func__, sensor->dmic_req_codec, sensor->dmic_req_snshub);
    mutex_unlock(&sensor->lock);
    return 0;
}

static void snshub_dmic_worker(struct work_struct *work)
{
    printk("--CWMCU--: %s\n", __func__);
    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        printk(KERN_DEBUG "--CWMCU--%s sensor->mcu_mode = %d\n", __func__, sensor->mcu_mode);
        return;
    }

    if (sensor->dmic_codec_release)
        snshub_dmic_status_process(sensor->mcu_screen, DMIC_REQ_ID_CODEC, DMIC_REQ_PROC_RELEASE);
    else
        snshub_dmic_status_process(sensor->mcu_screen, DMIC_REQ_ID_CODEC, DMIC_REQ_PROC_OPEN);
}

static int snshub_early_suspend_check(void)
{
    printk("--CWMCU--: %s\n", __func__);
    if (sensor->enabled_list & (1<<CW_VOICE_TRIGGER)) {
        printk("--CWMCU--: %s, voice trigger --- request dmic.\n", __func__);
        snshub_dmic_status_process(sensor->mcu_screen, DMIC_REQ_ID_SNSHUB, DMIC_REQ_PROC_OPEN);
    }
    return 0;
}

static int snshub_screen_on_check(void)
{
    printk("--CWMCU--: %s\n", __func__);
    if (sensor->enabled_list & (1<<CW_VOICE_TRIGGER)) {
        printk("--CWMCU--: %s, voice trigger --- release dmic.\n", __func__);
        snshub_dmic_status_process(sensor->mcu_screen, DMIC_REQ_ID_SNSHUB, DMIC_REQ_PROC_RELEASE);
    }
    return 0;
}
#endif

void audiocodec_notify_snshub(bool mic_not_used_by_codec)
{
#ifdef CONFIG_SNR_HUB_VT
    printk("--CWMCU--: %s, mic_not_used_by_codec=%d\n", __func__, mic_not_used_by_codec);

    sensor->dmic_codec_release = mic_not_used_by_codec;
    schedule_work(&sensor->dmic_work);
#endif
}
EXPORT_SYMBOL(audiocodec_notify_snshub);


#ifdef CONFIG_FB
static int fb_notifier_callback(struct notifier_block *self,
                         unsigned long event, void *data)
{
    struct fb_event *evdata = data;
    int *blank;
//    struct mit_ts_info *ts =
//        container_of(self, struct mit_ts_info, fb_notif);
//    if (evdata && evdata->data && event == FB_EVENT_BLANK &&
    if (evdata && evdata->data && event == FB_EVENT_BLANK) {
        blank = evdata->data;
        printk(KERN_DEBUG "--CWMCU-- blank=%d\n", *blank);
        if (*blank == FB_BLANK_UNBLANK) {
            printk(KERN_DEBUG "--CWMCU-- fbnotify set to normal mode.\n");
            sensor->mcu_screen = CW_NORMAL;
#ifdef CONFIG_SNR_HUB_VT
            snshub_screen_on_check();
#endif
            //cwmcu_screen_on_resume();
        } else if (*blank == FB_BLANK_POWERDOWN) {
            printk(KERN_DEBUG "--CWMCU-- fbnotify set to early suspend mode.\n");
            sensor->mcu_screen = CW_SUSPEND;
#ifdef CONFIG_SNR_HUB_VT
            snshub_early_suspend_check();
#endif
            //cwmcu_screen_off_suspend();
        }
    }

    return 0;
}
#endif

static int CWMCU_i2c_write(struct CWMCU_data *sensor, u8 reg_addr, u8 *data, u8 len)
{
    int dummy;
    int i;
    for (i = 0; i < len; i++) {
        dummy = i2c_smbus_write_byte_data(sensor->client, reg_addr++, data[i]);
        check_i2c_error_cnt((dummy < 0));
        if (dummy < 0) {
            pr_err("CWMCU i2c write error =%d\n", dummy);
            return dummy;
        }
    }
    return 0;
}

/* Returns the number of read bytes on success */
static int CWMCU_i2c_read(struct CWMCU_data *sensor, u8 reg_addr, u8 *data, u8 len)
{
    int dummy;
    dummy = i2c_smbus_read_i2c_block_data(sensor->client, reg_addr, len, data);
    if (len)
        check_i2c_error_cnt((dummy == 0));
    return dummy;
}

/* write format    1.slave address  2.data[0]  3.data[1] 4.data[2] */
static int CWMCU_write_i2c_block(struct CWMCU_data *sensor, u8 reg_addr, const u8 *data, u8 len)
{
    int dummy;
    dummy = i2c_smbus_write_i2c_block_data(sensor->client, reg_addr, len, data);
    check_i2c_error_cnt((dummy < 0));
    if (dummy < 0) {
        pr_err("CWMCU i2c write error =%d\n", dummy);
        return dummy;
    }
    return 0;
}

#ifdef CONFIG_SNR_HUB_VT
static int CWMCU_write_i2c_multiple(struct CWMCU_data *sensor, u8 reg_addr, const u8 *data, int len)
{
    int dummy;
    unsigned char buffer[len+1];
    buffer[0] = reg_addr;

    memcpy(&buffer[1],data,len);

    dummy = i2c_master_send(sensor->client, buffer, len+1);
    check_i2c_error_cnt((dummy < 0));
    if (dummy < 0) {
        printk("CWMCU i2c master send fail! error=%d\n", dummy);
        return dummy;
    }
    return 0;
}
#endif


/*
static int CWMCU_write_serial(u8 *data, int len)
{
    int dummy;
    dummy = i2c_master_send(sensor->client, data, len);
    if (dummy < 0) {
        pr_err("i2c write error =%d\n", dummy);
        return dummy;
    }
    return 0;
}

static int CWMCU_read_serial(u8 *data, int len)
{
    int dummy;
    dummy = i2c_master_recv(sensor->client, data, len);
    if (dummy < 0) {
        pr_err("i2c read error =%d\n", dummy);
        return dummy;
    }
    return 0;
}
*/


#define CWMCU_TIMER_INTERVAL_MS   2000
void cwmcu_timer_callback(unsigned long data)
{
    //printk("--CWMCU-- timer=> cwmcu_timer_callback start!\n");
    cwmcu_check_sensor_alive();
    //printk("--CWMCU-- timer=> cwmcu_timer_callback end!\n");
    mod_timer(&sensor->timer, jiffies + msecs_to_jiffies(CWMCU_TIMER_INTERVAL_MS));
}

int cwmcu_init_timer(void)
{
    printk("--CWMCU-- timer=> init cwmcu timer\n");

    setup_timer(&sensor->timer, cwmcu_timer_callback, 0);
    mod_timer(&sensor->timer, jiffies + msecs_to_jiffies(40000));

    return 0;
}

void cleanup_my_timer(void)
{
    printk("--CWMCU-- timer=> cleanup cwmcu timer\n");
    del_timer(&sensor->timer);
}

static void cwmcu_send_powerkey(void)
{
    if ((sensor->mcu_screen == CW_SUSPEND) && (sensor->sns_debug_flag & (DEBUG_SUPPORT_POWER_KEY_EVENT))) {
        printk( "--CWMCU-- send power key!\n");
        input_report_key(sensor->input, KEY_POWER, 1);
        input_report_key(sensor->input, KEY_POWER, 0);
        input_sync(sensor->input);
        printk( "--CWMCU-- send power key end!\n");
    }
}

static int cwmcu_check_sensor_alive(void)
{
    int id = 0;

    if(sensor->check_muc_sensor_alive_flag==0)
        return 0;

    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        return 0;
    }

    /* check mcu init status */
    if( !(sensor->mcu_reset > sensor->pre_reset_count)) {
        printk("--CWMCU-- check mcu init status. [fail] count=%d\n", sensor->mcu_reset_error_count);
        sensor->mcu_reset_error_count++;
    }

    /*  check mcu sensor alive  */
    if(sensor->enabled_list & 0x07) {

        printk("--CWMCU-- check sensor alive. [start]\n");
        for(id=0;id<3;id++) {
            if(sensor->enabled_list & (1<<id)) {
                if( sensor->preContinuousSensor[id][0] == sensor->curContinuousSensor[id][0]
                    && sensor->preContinuousSensor[id][1] == sensor->curContinuousSensor[id][1]
                    && sensor->preContinuousSensor[id][2] == sensor->curContinuousSensor[id][2]) {
                    sensor->sensor_error_count[id]++;
                    printk("--CWMCU-- check sensor %d alive. [fail] count=%d\n",id,sensor->sensor_error_count[id]);
                } else {
                    sensor->preContinuousSensor[id][0] = sensor->curContinuousSensor[id][0];
                    sensor->preContinuousSensor[id][1] = sensor->curContinuousSensor[id][1];
                    sensor->preContinuousSensor[id][2] = sensor->curContinuousSensor[id][2];
                    printk("--CWMCU-- check sensor %d alive. [pass] count=%d\n",id,sensor->sensor_error_count[id]);
                    sensor->sensor_error_count[id] = 0;
                }
            }
        }

    }

    if(sensor->sensor_error_count[0]>20 || sensor->sensor_error_count[1]>20
        || sensor->mcu_reset_error_count > 10) {
        printk("--CWMCU-- check mcu status fail => [reset mcu]\n");
        sensor->sensor_error_count[0] = 0;
        sensor->sensor_error_count[1] = 0;
        sensor->sensor_error_count[2] = 0;
        sensor->mcu_reset_error_count = 0;
        CWMCU_reset_mcu(true);
    }

    return 0;
}

static void cwmcu_restore_status(void)
{
    int id,part;
    u8 list;
    uint8_t delay_ms = 0;
    uint8_t data[6] = {0};
    uint8_t sensorId = 0;
    uint32_t enable_list_temp;

    printk("--CWMCU-- %s\n", __func__);

#ifdef CONFIG_SNS_HUB_DEBUG
    list  = (sensor->power_test_on)? 0: 1;
    printk("CWMCU write CW_SET_INTERRUPT_ENABLE, %d\n", list);
    if (CWMCU_write_i2c_block(sensor, CW_SET_INTERRUPT_ENABLE, &list, 1) < 0) {
        printk("CWMCU Write CW_SET_INTERRUPT_ENABLE error.\n");
    }
#endif

    //enable sensor
    enable_list_temp = sensor->enabled_list & (~(1<<CW_VOICE_TRIGGER));
    for (id = 0; id < CW_SENSORS_ID_END; id++) {
        if (enable_list_temp & (1<<id)) {
            part = id / 8;
            list = (u8)(enable_list_temp>>(part*8));
            CWMCU_i2c_write(sensor, CW_ENABLE_REG+part, &list, 1);
            id = ((part+1)*8)-1;
        }
    }

    //set sensor report rate
    for (id = 0; id < CW_SENSORS_ID_END; id++) {
        if (sensor->enabled_list & (1<<id)) {
            delay_ms = (uint8_t)sensor->report_period[id];
            data[0] = (uint8_t)id;
            data[1] = delay_ms;
            CWMCU_write_i2c_block(sensor, CW_SENSOR_DELAY_SET, data, 2);
        }
    }

    /* flush status */
    for(id=0;id<CW_SENSORS_ID_END;id++) {
        if(sensor->flush_list & (1<<id)) {
            sensorId = (uint8_t)id;
            if (CWMCU_i2c_write(sensor, CW_BATCHFLUSH, &sensorId, 1) < 0) {
                printk("--CWMCU-- flush => i2c error~!!\n");
            }
        }
    }

    /* batch status */
    for(id=0;id<CW_SENSORS_ID_END;id++) {
        if(sensor->batched_list & (1<<id)) {
            data[0] = (uint8_t)id;
            memcpy(&data[1],&sensor->sensor_timeout[id],sizeof(uint8_t)*4);
            CWMCU_write_i2c_block(sensor, CW_BATCHTIMEOUT, data, 5);
        }
    }

    /* pedometer */
    memcpy(data,&sensor->step_counter,sizeof(uint8_t)*4);
    if(CWMCU_write_i2c_block(sensor, CW_SET_STEP_COUNTER, data, 4)<0) {
        printk("--CWMCU-- restore pedometer error!!\n");
    }

#ifdef CONFIG_SNR_HUB_VT
    /* voice trigger task */
    list = (sensor->vt_enabled)?1:0;
    if (CWMCU_i2c_write(sensor, CW_VOICE_TRIGGER_CTRL, &list, 1)) {
        printk("CWMCU: fail to send vt cmd\n");
    } 
#endif
}

static void cwmcu_check_mcu_enable(void)
{
    uint8_t data[6];
    uint32_t mcu_enable_list = 0;
    uint32_t android_enable_list = 0;

    printk("--CWMCU-- check mcu enable status.\n");

    if (CWMCU_i2c_read(sensor, CW_ENABLE_STATUS, data, 4) < 0) {
        printk("--CWMCU-- get mcu enable list error.\n");
    } else {
        mcu_enable_list = (uint32_t)data[3]<< 24 | (uint32_t)data[2]<< 16 | (uint32_t)data[1]<< 8 | data[0];
        mcu_enable_list &= ~(1<<CW_VOICE_TRIGGER);
        android_enable_list = sensor->enabled_list & (~(1<<CW_VOICE_TRIGGER));

        if(mcu_enable_list != android_enable_list) {
            cwmcu_restore_status();
        }
    }
}

static int cwmcu_get_version(void)
{
    int ret;
    uint8_t count;
    uint8_t data[6] = {0};

    printk("--CWMCU-- %s\n", __func__);

    for (count = 0; count < 3; count++) {
        if ((ret = CWMCU_i2c_read(sensor, CW_FWVERSION, data, 2)) >= 0) {
            printk("--CWMCU-- CHECK_FIRMWARE_VERSION : %d.%d\n", data[0],data[1]);
            return 0;
        }
        mdelay(20);
    }

    printk("--CWMCU-- Cannot get fw version, [ret:%d]\n", ret);
    return ret;
}


/*
    Sensor get data and report event
    format byte[0] = id
    format byte[1] & byte[2] = time diff [L/H]
    format byte[3]~byte[8] = data X,Y,Z [L/H]
*/
static int cwmcu_gesture_read(struct CWMCU_data *sensor)
{
    uint8_t data[6] = {0};
    uint32_t data_event[3]={0};
    int data_count = 0;
    int i = 0;

    if (sensor->mcu_mode == CW_BOOT || sensor->mcu_power == CW_BOOT) {
        return 0;
    }

    if (CWMCU_i2c_read(sensor, CW_READ_GESTURE_EVENT_COUNT, data, 1) >= 0) {
        data_count = data[0];
        //printk(KERN_DEBUG "--CWMCU-- read gesture count = %d\n", data_count);
        for (i = 0; i < data_count; i++) {
            /* read 2byte */
            if (CWMCU_i2c_read(sensor, CW_READ_GESTURE_EVENT_DATA, data, 6) >= 0) {
                printk(KERN_DEBUG "--CWMCU-- read gesture data = %d\n", data_count);
                if (data[0] != CWMCU_NODATA) {
                    data_event[0] = ((u32)data[0] << 16) | (((u32)data[1]));

                    //timestamp = (uint32_t)(data[5]<<24) | (uint32_t)(data[4]<<16) | (uint32_t)(data[3]<<8) | (uint32_t)data[2];
                    //printk("--CWMCU-- gesture timestamp=0x%x \n",timestamp);
                    //printk("--CWMCU-- gesture d2:0x%x | d3:0x%x | d4:0x%x | d5:0x%x\n",data[2],data[3],data[4],data[5]);

                    if(data[0]==CW_CONTEXT_AWARE) {
                        cwmcu_send_powerkey();
                        data_event[1] = (uint32_t)(data[5]<<24) | (uint32_t)(data[4]<<16);
                        data_event[2] = (uint32_t)(data[3]<<8) | (uint32_t)data[2];
                        input_report_abs(sensor->input, CW_ABS_TIMESTAMP_H, data_event[1]);
                        input_report_abs(sensor->input, CW_ABS_TIMESTAMP_L, data_event[2]);
                        //printk("--CWMCU--Normal castor H:%u | L:%u\n", data_event[1], data_event[2]);
                        print_event(TAG_CONTAWARENESS, data[1]);
                    }
                    if (data[0] == CW_SINGLE_SNAP) {
                        print_event(TAG_SSNAP, data[1]);
                        if ((data[1] == 1) || (data[1] == 2))
                            cwmcu_send_powerkey();
                    }
                    if ( (data[0] == CW_SCREEN_ON)){
                        print_event(TAG_SCREENON, data[1]);
                        if (data[1])
                            cwmcu_send_powerkey();
                    }

                    VERBOSE(0, data[0], "--CWMCU--Normal mgesture %d data -> x = %d\n", data[0], data[1]);
                    input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
                    input_sync(sensor->input);
                    input_report_abs(sensor->input, CW_ABS_X, 0xFF0000);
                    input_sync(sensor->input);
                }
            } else {
                printk(KERN_DEBUG "--CWMCU-- read gesture failed~!!\n");
            }
        }
    }

    return 1;
}

#ifdef CONFIG_SNR_HUB_VT
static int cwmcu_voice_trigger_read(struct CWMCU_data *sensor)
{
    uint8_t data[6] = {0};
    uint32_t data_event[3]={0};
    int data_count = 0;
    int i = 0;

    if (sensor->mcu_mode == CW_BOOT || sensor->mcu_power == CW_BOOT) {
        return 0;
    }

    if (CWMCU_i2c_read(sensor, CW_READ_SENSORY_EVENT_COUNT, data, 1) >= 0) {
        data_count = data[0];
        VERBOSE(DBG_LOG, 0, "--CWMCU-- read voice_trigger: count = %d\n", data_count);
        for (i = 0; i < data_count; i++) {
            /* read 2byte */
            if (CWMCU_i2c_read(sensor, CW_READ_SENSORY_EVENT_DATA, data, 2) >= 0) {
                VERBOSE(DBG_LOG, 0, "--CWMCU-- voice_trigger: read voice count = %d\n", data_count);
                if (data[0] != CWMCU_NODATA) {

                    if (sensor->pre_voice_trigger_value == data[1])
                        data[1] += 20;

                    data_event[0] = ((u32)data[0] << 16) | (((u32)data[1]));
                    print_event(TAG_VOICE_TRIGGER, data[1]%20);
                    VERBOSE(DBG_LOG, 0, "--CWMCU-- voice_triggerD -> data[0-1]= %x %x\n", data[0], data[1]%20);
                    cwmcu_send_powerkey();
                    input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
                    input_sync(sensor->input);

                    sensor->pre_voice_trigger_value = data[1];
                }
            } else {
                printk(KERN_DEBUG "--CWMCU-- read voice trigger failed~!!\n");
            }
        }
    }

    return 1;
}
#endif

static int cwmcu_error_log_read(struct CWMCU_data *sensor)
{
    uint8_t data[32] = {0};
    char log_message[32];
    int data_count = 0;
    int i = 0;

    if (sensor->mcu_mode == CW_BOOT || sensor->mcu_power == CW_BOOT) {
        return 0;
    }

    if (CWMCU_i2c_read(sensor, CW_ERROR_COUNT_GET, data, 1) >= 0) {
        data_count = data[0];
        printk("--CWMCU-- read error_log: count = %d\n", data_count);
        for (i = 0; i < data_count; i++) {
            /* read 32bytes */
            if (CWMCU_i2c_read(sensor, CW_ERROR_INFO_GET, data, 32) >= 0) {
                    memcpy(log_message,data,32);
                    printk("--CWMCU-- error_log => %s\n", log_message);
            } else {
                printk("--CWMCU-- read error_log failed.\n");
            }
        }
    }

    return 1;
}

static int cwmcu_debug_log_read(struct CWMCU_data *sensor)
{
    uint8_t data[32] = {0};
    char log_message[32];
    int data_count = 0;
    int i = 0;

    if (sensor->mcu_mode == CW_BOOT || sensor->mcu_power == CW_BOOT) {
        return 0;
    }

    if (CWMCU_i2c_read(sensor, CW_DEBUG_COUNT_GET, data, 1) >= 0) {
        data_count = data[0];
        printk("--CWMCU-- read debug_log: count = %d\n", data_count);
        for (i = 0; i < data_count; i++) {
            /* read 32bytes */
            memset(data,0,32);
            memset(log_message,0,32);
            if (CWMCU_i2c_read(sensor, CW_DEBUG_INFO_GET, data, 32) >= 0) {
                memcpy(log_message,data,32);
                printk("--CWMCU-- debug_log => %s\n", log_message);
            } else {
                printk("--CWMCU-- read debug_log failed.\n");
            }
        }
    }

    return 1;
}

static int cwmcu_batch_read(struct CWMCU_data *sensor)
{
    int i = 0;
    uint8_t data[20] = {0};
    uint32_t data_event[6] = {0};

    uint16_t batch_count = 0;

    if (sensor->mcu_mode == CW_BOOT || sensor->mcu_power == CW_BOOT) {
        return 0;
    }

    /* read the count of batch queue */
    if (CWMCU_i2c_read(sensor, CW_BATCHCOUNT, data, 2) >= 0) {
        batch_count = ((uint16_t)data[1] << 8) | (uint16_t)data[0];
        VERBOSE(DBG_LOG_BATCH, 0, "--CWMCU-- read batch count = %d\n", batch_count);
    } else {
        printk("--CWMCU-- read batch count fail.\n");
    }

    //4,294,967,295
    for (i = 0; i < batch_count; i++) {
        if (CWMCU_i2c_read(sensor, CW_BATCHEVENT, data, 11) >= 0) {
            /* check if there are no data from queue */
            if (data[0] != CWMCU_NODATA) {
                if (data[0] == CW_META_DATA) {
                    data_event[0] = ((u32)data[0] << 16) | ((u32)data[5] << 8) | (u32)data[6];
                    sensor->flush_list &= ~(1<<data[6]);
                    VERBOSE(DBG_LOG_BATCH, 0, "--CWMCU-- META_DATA => %02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x,%02x\n",
                        data[0],data[1],data[2],data[3],data[4],data[5],data[6],data[7],data[8],data[9],data[10]);
                    input_report_abs(sensor->input, CW_ABS_Z, data_event[0]);
                    input_sync(sensor->input);
                    input_report_abs(sensor->input, CW_ABS_Z, 0xFF0000);
                    input_sync(sensor->input);
                } else {
                    data_event[0] = ((u32)data[0] << 16) | ((u32)data[2] << 8) | (u32)data[1];
                    data_event[1] = ((u32)data[0] << 16) | ((u32)data[4] << 8) | (u32)data[3];
                    data_event[2] = ((u32)data[0] << 16) | ((u32)data[6] << 8) | (u32)data[5];
                    data_event[3] = (uint32_t)(data[10]<<24) | (uint32_t)(data[9]<<16);
                    data_event[4] = (uint32_t)(data[8] <<8)  | (uint32_t)(data[7]);
                    //data_event[3] = ((u32)data[0] << 16) | ((u32)data[8] << 8) | (u32)data[7];//timediff

                    /* check flush event */
                    input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
                    input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
                    input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
                    input_report_abs(sensor->input, CW_ABS_TIMESTAMP_H, data_event[3]);
                    input_report_abs(sensor->input, CW_ABS_TIMESTAMP_L, data_event[4]);
                    input_sync(sensor->input);
                    VERBOSE(DBG_LOG_BATCH, 0, "--CWMCU-- Batch data_event[0-4] => %u, %u, %u, H:%u, L:%u\n",
                            data_event[0], data_event[1], data_event[2], data_event[3],data_event[4]);

                    VERBOSE(DBG_LOG_BATCH, 0, "--CWMCU-- Batch raw data => %2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x,%2x\n",
                        data[0],data[1],data[2],data[3],data[4],data[5],data[6],data[7],data[8],data[9],data[10]);
                }
            }
        } else {
            printk("--CWMCU-- read batch event failed~!!\n");
        }
    }
    return 1;
}

#if 0
static int cwmcu_batch_timestamp_sync(struct CWMCU_data *sensor)
{
    uint8_t data[6] = {0};
    uint32_t data_event[3]={0};

    if (sensor->mcu_mode == CW_BOOT || sensor->mcu_power == CW_BOOT) {
        return 0;
    }

    /* read timestamp */
    if (CWMCU_i2c_read(sensor, CW_BATCHTIME_SYNC, data, 4) >= 0) {

        data_event[0] = (uint32_t)(data[3]<<24) | (uint32_t)(data[2]<<16);
        data_event[1] = (uint32_t)(data[1]<<8) | (uint32_t)data[0];
        //input_report_abs(sensor->input, CW_ABS_TIMESTAMP_H, data_event[1]);
        //input_report_abs(sensor->input, CW_ABS_TIMESTAMP_L, data_event[2]);
        printk("--CWMCU-- batch timestamp sync => H:%u | L:%u\n", data_event[0], data_event[1]);
    } else {
        printk("--CWMCU-- batch timestamp sync fail.\n");
    }

    return 1;
}
#endif



#ifndef CWMCU_INTERRUPT
static void cwmcu_check_sensor_update(void)
{
    int64_t tvusec, temp = 0;
    unsigned int tvsec;

    do_gettimeofday(&sensor->now);
    tvsec = sensor->now.tv_sec;
    tvusec = sensor->now.tv_usec;

    temp = (int64_t)(tvsec * 1000000LL) + tvusec;

    sensor->update_list = sensor->enabled_list;
}
#endif

static int CWMCU_read(struct CWMCU_data *sensor)
{
    int id_check = 0;
    uint8_t data[20] = {0};
    uint32_t data_event[7] = {0};

    uint8_t MCU_REG = 0;

    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        /* it will not get data if status is bootloader mode */
        return 0;
    }

#ifndef CWMCU_INTERRUPT
    cwmcu_check_sensor_update();
#endif

    if (sensor->update_list) {
        for (id_check = 0; id_check < CW_SENSORS_ID_END; id_check++) {
            if ((sensor->update_list & (1<<id_check)) && (sensor->sensor_timeout[id_check] == 0)) {
                switch (id_check) {
                case CW_ACCELERATION:
                case CW_MAGNETIC:
                case CW_GYRO:
                case CW_LIGHT:
                case CW_PROXIMITY:
                case CW_PRESSURE:
                case CW_ORIENTATION:
                case CW_ROTATIONVECTOR:
                case CW_LINEARACCELERATION:
                case CW_GRAVITY:
                case CW_GAME_ROTATION_VECTOR:
                case CW_GEOMAGNETIC_ROTATION_VECTOR:
                case CW_PROXIMITY_GESTURE:
                case CW_LIGHT_RGB:

                    if(id_check == CW_PROXIMITY_GESTURE) {
                        MCU_REG = CW_READ_PROXIMITY_GESTURE;
                    } else if(id_check == CW_LIGHT_RGB) {
                        MCU_REG = CW_READ_LIGHT_RGB;
                    } else {
                        MCU_REG = CWMCU_I2C_SENSORS_REG_START+id_check;
                    }

                    /* read 6byte */
                    if (CWMCU_i2c_read(sensor, MCU_REG, data, 6) >= 0) {
                        data_event[0] = ((u32)id_check << 16) | (((u32)data[1] << 8) | (u32)data[0]);
                        data_event[1] = ((u32)id_check << 16) | (((u32)data[3] << 8) | (u32)data[2]);
                        data_event[2] = ((u32)id_check << 16) | (((u32)data[5] << 8) | (u32)data[4]);

                        VERBOSE(0, id_check, "--CWMCU--Normal id=%d data -> x = %d, y = %d, z = %d\n"
                                    , id_check
                                    , (int16_t)((u32)data[1] << 8) | (u32)data[0]
                                    , (int16_t)((u32)data[3] << 8) | (u32)data[2]
                                    , (int16_t)((u32)data[5] << 8) | (u32)data[4]
                                    );
                        if (id_check == CW_MAGNETIC || id_check == CW_ORIENTATION) {
                            if (CWMCU_i2c_read(sensor, CW_ACCURACY, data, 1) >= 0) {
                                data_event[6] = ((u32)id_check << 16) | (u32)data[0];
                            }
                            VERBOSE(0, id_check, "--CWMCU--MAG ACCURACY = %d\n", data[0]);
                            input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
                            input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
                            input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
                            input_report_abs(sensor->input, CW_ABS_ACCURACY, data_event[6]);
                            input_sync(sensor->input);
                        } else if (id_check == CW_PROXIMITY) {
                            wake_lock_timeout(&psensor_wake_lock, 2*HZ);
                            input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
                            input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
                            input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
                            input_sync(sensor->input);
                            print_event(TAG_PROX, (int16_t)((u32)data[1] << 8) | (u32)data[0]);
#ifdef CONFIG_SNR_HUB_LG_HACK
                        } else if (id_check == CW_LIGHT) {
                            if (sensor->light_keeped) {
                                input_report_abs(sensor->input, CW_ABS_X, sensor->light_x);
                                input_report_abs(sensor->input, CW_ABS_Y, sensor->light_y);
                                input_report_abs(sensor->input, CW_ABS_Z, sensor->light_z);
                            } else {
                                input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
                                input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
                                input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
                                sensor->light_x = data_event[0];
                                sensor->light_y = data_event[1];
                                sensor->light_z = data_event[2];
                            }
                            input_sync(sensor->input);
#endif
                        } else if (id_check == CW_PROXIMITY_GESTURE) {
                            input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
                            input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
                            input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
                            input_sync(sensor->input);
                            print_event(TAG_GESTURE, (int16_t)((u32)data[1] << 8) | (u32)data[0]);
                        } else {
                            input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
                            input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
                            input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
                            input_sync(sensor->input);
                        }

                        /* reset x,y,z */
                        input_report_abs(sensor->input, CW_ABS_X, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_Y, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_Z, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_ACCURACY, 0xFF0000);
                        input_sync(sensor->input);

                        if(id_check == CW_ACCELERATION || id_check == CW_MAGNETIC || id_check == CW_GYRO) {
                            sensor->curContinuousSensor[id_check][0] = data_event[0];
                            sensor->curContinuousSensor[id_check][1] = data_event[1];
                            sensor->curContinuousSensor[id_check][2] = data_event[2];
                        }

                    } else {
                        printk("--CWMCU-- CWMCU_i2c_read error 0x%x~!!!\n", CWMCU_I2C_SENSORS_REG_START+id_check);
                    }
                    break;
                case CW_MAGNETIC_UNCALIBRATED:
                case CW_GYROSCOPE_UNCALIBRATED:
                    /* read 12byte */
                    if (CWMCU_i2c_read(sensor, CWMCU_I2C_SENSORS_REG_START+id_check, data, 12) >= 0) {
                        data_event[0] = ((u32)id_check << 16) | (((u32)data[1] << 8) | (u32)data[0]);
                        data_event[1] = ((u32)id_check << 16) | (((u32)data[3] << 8) | (u32)data[2]);
                        data_event[2] = ((u32)id_check << 16) | (((u32)data[5] << 8) | (u32)data[4]);
                        data_event[3] = ((u32)id_check << 16) | (((u32)data[7] << 8) | (u32)data[6]);
                        data_event[4] = ((u32)id_check << 16) | (((u32)data[9] << 8) | (u32)data[8]);
                        data_event[5] = ((u32)id_check << 16) | (((u32)data[11] << 8) | (u32)data[10]);

                        VERBOSE(0, id_check, "--CWMCU--Normal id=%d data -> x = %d, y = %d, z = %d, x_bios = %d, y_bios = %d, z_bios = %d,\n"
                                    , id_check
                                    , (int16_t)((u32)data[1] << 8) | (u32)data[0]
                                    , (int16_t)((u32)data[3] << 8) | (u32)data[2]
                                    , (int16_t)((u32)data[5] << 8) | (u32)data[4]
                                    , (int16_t)((u32)data[7] << 8) | (u32)data[6]
                                    , (int16_t)((u32)data[9] << 8) | (u32)data[8]
                                    , (int16_t)((u32)data[11] << 8) | (u32)data[10]
                                );
                        if (CWMCU_i2c_read(sensor, CW_ACCURACY, data, 1) >= 0) {
                            data_event[6] = ((u32)id_check << 16) | (u32)data[0];
                        }
                        input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
                        input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
                        input_report_abs(sensor->input, CW_ABS_Z, data_event[2]);
                        input_report_abs(sensor->input, CW_ABS_X1, data_event[3]);
                        input_report_abs(sensor->input, CW_ABS_Y1, data_event[4]);
                        input_report_abs(sensor->input, CW_ABS_Z1, data_event[5]);
                        input_report_abs(sensor->input, CW_ABS_ACCURACY, data_event[6]);
                        input_sync(sensor->input);

                        input_report_abs(sensor->input, CW_ABS_X, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_Y, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_Z, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_X1, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_Y1, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_Z1, 0xFF0000);
                        input_report_abs(sensor->input, CW_ABS_ACCURACY, 0xFF0000);
                        input_sync(sensor->input);
                    }
                    break;
                case CW_STEP_COUNTER:
                    /* read 6byte */
                    if (CWMCU_i2c_read(sensor, CWMCU_I2C_SENSORS_REG_START+id_check, data, 4) >= 0) {
                        data_event[0] = ((u32)id_check << 16) | (((u32)data[1] << 8) | (u32)data[0]);
                        data_event[1] = ((u32)id_check << 16) | (((u32)data[3] << 8) | (u32)data[2]);

                        sensor->step_counter = (uint32_t)data[3]<< 24 | (uint32_t)data[2]<< 16 | (uint32_t)data[1]<< 8 | data[0];

                        input_report_abs(sensor->input, CW_ABS_X, data_event[0]);
                        input_report_abs(sensor->input, CW_ABS_Y, data_event[1]);
                        input_sync(sensor->input);

                        VERBOSE(DBG_LOG_DATA, id_check, "--CWMCU--Normal id=%d data -> x = %d, y = %d, z = 0\n"
                                    , id_check
                                    , (int16_t)((u32)data[1] << 8) | (u32)data[0]
                                    , (int16_t)((u32)data[3] << 8) | (u32)data[2]);
                    } else {
                        printk("--CWMCU-- CWMCU_i2c_read error 0x%x~!!!\n", CWMCU_I2C_SENSORS_REG_START+id_check);
                    }
                    break;
                default:
                    break;
                }
            }
        }
    }

#ifndef CWMCU_INTERRUPT
    cwmcu_gesture_read(sensor);
#ifdef CONFIG_SNR_HUB_VT
    cwmcu_voice_trigger_read(sensor);
#endif
    cwmcu_batch_read(sensor);
    cwmcu_error_log_read(sensor);
    cwmcu_debug_log_read(sensor);
#endif

    return 0;
}

/*==========sysfs node=====================*/

static int active_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int enabled = 0;
    int sensors_id = 0;
    int error_msg = 0;
    u8 data = 0;
    u8 i = 0;
    uint8_t delay_ms = 0;
    u8 buffer[2] = {0};

    //if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
    //    return count;
    //}

    sscanf(buf, "%d %d\n", &sensors_id, &enabled);

#ifdef PRES_FEATURE_DISABLED
    if(sensors_id == CW_PRESSURE)
        return count;
#endif
    if(sensors_id == CW_VOICE_TRIGGER) {
#ifndef CONFIG_SNR_HUB_VT
        return count;
#else
        if (enabled) {
            printk( "--CWMCU-- voice trigger open.\n");
            snshub_dmic_status_process(sensor->mcu_screen, DMIC_REQ_ID_SNSHUB, DMIC_REQ_PROC_OPEN);
        } else {
            printk( "--CWMCU-- voice trigger close.\n");
            snshub_dmic_status_process(sensor->mcu_screen, DMIC_REQ_ID_SNSHUB, DMIC_REQ_PROC_RELEASE);
        }
        sensor->enabled_list &= ~(1<<sensors_id);
        sensor->enabled_list |= ((uint32_t)enabled)<<sensors_id;
        return count;
#endif
    }

    sensor->enabled_list &= ~(1<<sensors_id);
    sensor->enabled_list |= ((uint32_t)enabled)<<sensors_id;

#ifdef CONFIG_SNR_HUB_LG_HACK
    if (sensors_id == CW_PROXIMITY_GESTURE) {
        sensor->light_keeped = enabled;
        printk("CWMCU: gesture enabled %d, so setting light_keeped\n", enabled);
    }
#endif
    /* clean timeout value if sensor turn off */
    if (enabled == 0) {
        sensor->sensor_timeout[sensors_id] = 0;
    }

    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        return count;
    }

    i = sensors_id / 8;
    data = (u8)(sensor->enabled_list>>(i*8));

    error_msg += CWMCU_i2c_write(sensor, CW_ENABLE_REG+i, &data, 1);

    printk( "--CWMCU-- data =%d, i = %d, sensors_id=%d enable=%d  enable_list=0x%08X\n", data, i, sensors_id, enabled, sensor->enabled_list);

    delay_ms = (uint8_t)sensor->report_period[sensors_id];


    buffer[0] = (uint8_t)sensors_id;
    buffer[1] = delay_ms;

    CWMCU_write_i2c_block(sensor, CW_SENSOR_DELAY_SET, buffer, 2);

    return count;
}

static int active_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    return snprintf(buf, 16, "0x%08x\n", sensor->enabled_list);
}

static int interval_show(struct device *dev, struct device_attribute *attr, char *buf)
{
#ifdef CWMCU_INTERRUPT
    int id;
    uint8_t sensor_report_rate[CW_SENSORS_ID_END] = {0};
    char line[256] = {0};
    uint8_t sensorId;
    uint8_t data[2];

    for(id=0;id<CW_SENSORS_ID_END;id++)
    {
        sensorId = (uint8_t)id;
        if(CWMCU_i2c_write(sensor, CW_SENSOR_DELAY_ID_SET, &sensorId, 1) >= 0) {
            if (CWMCU_i2c_read(sensor, CW_SENSOR_DELAY_GET, data, 1) >= 0) {
                sensor_report_rate[id] = data[0];
            } else {
                printk("--CWMCU-- interval_show()=> Get sensor report rate error. sensorId=%d\n",sensorId);
            }
        } else {
            printk("--CWMCU-- interval_show()=> Set sensor ID error. sensorId=%d\n",sensorId);
        }
    }

#if 0
    for(id=0;id<CW_SENSORS_ID_END;id++) {
        printk("--CWMCU-- sensor report rate(ms) %d => %d\n",id, sensor_report_rate[id]);
    }
#endif

    for (id = 0; id < CW_SENSORS_ID_END; id++) {
        sprintf(line+id*7, "%02d:%3d\n", id, sensor_report_rate[id]);
    }
    return snprintf(buf, 255, "%s", line);
#else
    return snprintf(buf, 16, "%d\n", CWMCU_POLL_INTERVAL);
#endif
}

static int interval_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int val = 0;
    int sensors_id = 0;
    uint8_t delay_ms = 0;
    uint8_t data[2];

    //if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
    //    return count;
    //}

    sscanf(buf, "%d %d\n", &sensors_id , &val);
    if (val < CWMCU_POLL_MIN)
        val = CWMCU_POLL_MIN;

    sensor->report_period[sensors_id] = val;

    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        return count;
    }

    delay_ms = (uint8_t)val;

    data[0] = (uint8_t)sensors_id;
    data[1] = delay_ms;

    CWMCU_write_i2c_block(sensor, CW_SENSOR_DELAY_SET, data, 2);

    printk( "--CWMCU-- sensors_id=%d delay_ms=%d\n", sensors_id, delay_ms);
    return count;
}

static int batch_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int sensors_id = 0;
    int delay_ms = 0;
    int timeout = 0;
    int batch_mode = -1;
    uint8_t data[5] = {0};
    int id;

    //if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
    //    return count;
    //}

    printk("--CWMCU-- %s in~!!\n", __func__);

    sscanf(buf, "%d %d %d %d\n", &sensors_id, &batch_mode, &delay_ms, &timeout);
    printk("--CWMCU-- sensors_id = %d, batch_mode = %d, delay_ms = %d, timeout = %d\n", sensors_id, batch_mode, delay_ms, timeout);

    sensor->sensor_timeout[sensors_id] = timeout;
    sensor->report_period[sensors_id] = delay_ms;

    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        return count;
    }

    data[0] = (uint8_t)sensors_id;
    data[1] = (uint8_t)(timeout);
    data[2] = (uint8_t)(timeout >> 8);
    data[3] = (uint8_t)(timeout >> 16);
    data[4] = (uint8_t)(timeout >> 24);

    CWMCU_write_i2c_block(sensor, CW_BATCHTIMEOUT, data, 5);

    if (timeout > 0) {
        sensor->batched_list |= ((uint32_t)1<<sensors_id);
        sensor->batch_timeout = timeout;
    } else {
        sensor->batched_list &= ~((uint32_t)1<<sensors_id);
    }

    for(id=0;id<CW_SENSORS_ID_END;id++) {
        if(sensor->batched_list & (1<<id)) {
            if(sensor->sensor_timeout[id] < sensor->batch_timeout)
                sensor->batch_timeout = sensor->sensor_timeout[id];
        }
    }

    //Set sensor report rate
    if(sensor->enabled_list & (1<<sensors_id)) {
        data[0] = (uint8_t)sensors_id;
        data[1] = delay_ms;
        CWMCU_write_i2c_block(sensor, CW_SENSOR_DELAY_SET, data, 2);
    }

    printk( "--CWMCU-- BatchSet=> id = %d, timeout = %d, delay_ms = %d, batch_timeout = %d\n",
            sensors_id, timeout, delay_ms, sensor->batch_timeout);

    return count;
}

static int batch_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    return snprintf(buf, 255, "batched_list = 0x%08x, batch_timeout = %u\n"
                    ,sensor->batched_list, sensor->batch_timeout);
}

static int flush_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int sensors_id = 0;
    uint8_t data = 0;

    //if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
    //    return count;
    //}

    printk("--CWMCU-- %s in\n", __func__);

    sscanf(buf, "%d\n", &sensors_id);
    printk("--CWMCU-- flush => sensors_id = %d \n", sensors_id);

    sensor->flush_list &= ~(1<<sensors_id);
    sensor->flush_list |= (uint32_t)1<<sensors_id;

    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        return count;
    }

    data = (uint8_t)sensors_id;
    if (CWMCU_i2c_write(sensor, CW_BATCHFLUSH, &data, 1) < 0) {
        printk("--CWMCU-- flush => i2c error~!!\n");
    }

    return count;
}

static int flush_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("--CWMCU-- %s in\n", __func__);

    return snprintf(buf, 255, "flush_list = 0x%08x\n",sensor->flush_list);
}

#ifdef CONFIG_SNR_HUB_VT
static int send_binfile_to_mcu(struct i2c_client *client, char *binfile)
{
    const uint16_t fs_send_max_size = 256;
    struct file *filp = NULL;
    mm_segment_t pre_fs;
    u8 buffer[256];
    int reg_ret = 0;

    size_t offset = 0;
    u8 data[16] = {0};
    int ret, get_fs_status=0, get_fs_size=0, error_count=0;
    u8 fs_cmd=0;
    u8 remainder[256]={0};

    int remainder_size, bin_size=0, total_size=0;
    int fs_error_flag = 0;
    uint16_t offset_count = 0;

    u8 regValue;

    usleep_range(10000, 20000);

    /* send start cmd */
    if(strstr(binfile,"voice") != NULL) {
        fs_cmd = 2;//send gram.bin + net.bin
    }
    printk("--CWMCU-- FILESEND: bin file name: %s | fs_cmd=%d\n", binfile, fs_cmd);
    CWMCU_i2c_write(sensor, CW_FILE_SEND_CMD, &fs_cmd, 1);

    /* check fs status  */
    error_count=0;
    while(1) {
        error_count++;
        if (CWMCU_i2c_read(sensor, CW_FILE_SEND_STATUS, data, 1) >= 0) {
            get_fs_status = (int)data[0];
            printk("--CWMCU-- FILESEND: fs_status=%d \n",get_fs_status);
        }

        if(error_count>5 && get_fs_status==0) {
            printk("--CWMCU-- FILESEND: file send command not ready. fs_status=%d \n",get_fs_status);
            return 0;
        } else if(get_fs_status==2)//erase flash completed.
        {
            break;
        }
        usleep_range(20000, 40000);
    }

    /* write bin file to mcu */
    filp = filp_open("/data/voice.bin", O_RDONLY, 0);
    if(filp == NULL) {
        printk("--CWMCU-- FILESEND: filp_open error!!.\n");
        return 0;
    } else {
        pre_fs = get_fs();
        set_fs(KERNEL_DS);
        printk("--CWMCU-- FILESEND: filp_open success!!.\n");
    }

    /*  voice trigger control disable */
    regValue = 0;
    if (CWMCU_i2c_write(sensor, CW_VOICE_TRIGGER_CTRL, &regValue, 1) >= 0) {
        printk("CWMCU: set vt cmd disable.\n");
        reg_ret = regulator_disable(sensor->mic);
    } else {
        printk("CWMCU: fail to send vt cmd\n");
    }

    bin_size = filp->f_op->llseek(filp,0,SEEK_END);
    printk("--CWMCU-- FILESEND: filp_open size=%d \n",bin_size);
    ret = filp->f_op->llseek(filp,0,0);

    if((bin_size%fs_send_max_size)>0) {
        total_size = bin_size + (fs_send_max_size-(bin_size%fs_send_max_size));
    } else {
        total_size = bin_size;
    }
    memset(data,0,sizeof(data));
    memcpy(data,&total_size,4);
    printk("--CWMCU-- FILESEND: Bin file total size = %d | d[0-3]: %x %x %x %x\n",total_size, data[0],data[1],data[2],data[3]);

    if (CWMCU_write_i2c_block(sensor, CW_FILE_SEND_SET_FILESIZE, data, 4)==0) {
        printk("--CWMCU-- FILESEND: Send Bin file size success, total bytes= %d\n",total_size);
    } else {
        printk("--CWMCU-- FILESEND: Send Bin file size fail.\n");
    }

    sensor->fs_progress = (offset*100)/total_size;

    error_count=0;
    while (offset < bin_size) {
        fs_error_flag = 0;
        if(get_fs_status > 0 && get_fs_size == offset) {

            if( (bin_size - offset) >= fs_send_max_size) {
                ret = filp->f_op->read(filp, buffer, fs_send_max_size, &filp->f_pos);
                if (CWMCU_write_i2c_multiple(sensor, CW_FILE_SEND_WRITE, buffer, fs_send_max_size)==0) {
                    printk("--CWMCU-- FILESEND: send %d bytes success, offset=%d ret=%d\n",fs_send_max_size, offset, ret);
                } else {
                    printk("--CWMCU-- FILESEND: send %d bytes fail, offset=%d ret=%d\n",fs_send_max_size, offset, ret);
                    fs_error_flag = 1;
                }
            } else {
                remainder_size = bin_size - offset;
                memset(remainder,0xFF,fs_send_max_size);
                ret = filp->f_op->read(filp, buffer, remainder_size, &filp->f_pos);
                memcpy(remainder,buffer,remainder_size);
                if (CWMCU_write_i2c_multiple(sensor, CW_FILE_SEND_WRITE, remainder, fs_send_max_size)==0) {
                    printk("--CWMCU-- FILESEND: send %d r bytes success, offset=%d ret=%d\n",fs_send_max_size, offset, ret);
                } else {
                    printk("--CWMCU-- FILESEND: send %d bytes fail, offset=%d | remainder=%d\n",fs_send_max_size, offset, remainder_size);
                    fs_error_flag = 1;
                }
            }    
        } else {
            printk("--CWMCU-- FILESEND: get error size or status. \n");
            fs_error_flag = 1;
            memset(data,0,sizeof(data));
            if (CWMCU_i2c_read(sensor, CW_FILE_SEND_SIZE, data, 4) >= 0) {
                get_fs_size = (int32_t)((uint32_t)(data[3]<<24) | (uint32_t)(data[2]<<16) | (uint32_t)(data[1]<<8) | (uint32_t)data[0]);
            }
            printk("--CWMCU-- FILESEND: debug=> fs_size=%d offset=%d \n", get_fs_size, offset);
        }
        usleep_range(3000, 4000);

        /* Transfer data success */
        if(fs_error_flag == 0) {
            memset(data,0,sizeof(data));
            if (CWMCU_i2c_read(sensor, CW_FILE_SEND_SIZE, data, 4) >= 0) {
                get_fs_size = (int32_t)((uint32_t)(data[3]<<24) | (uint32_t)(data[2]<<16) | (uint32_t)(data[1]<<8) | (uint32_t)data[0]);
                printk("--CWMCU-- FILESEND: debug=> fs_size=%x %x %x %x | %d\n", data[3],data[2],data[1],data[0], get_fs_size);
            }
            if(get_fs_size == (offset+fs_send_max_size)) {
                offset = get_fs_size;
            }
            sensor->fs_progress = (offset*100)/total_size;
            printk("--CWMCU-- FILESEND: fs_status=%d fs_size=%d offset=%d | %d \n",get_fs_status, get_fs_size, offset, sensor->fs_progress);
            usleep_range(3000, 4000);
        }
        /* Transfer data fail and re-sent bin file  */
        else if(fs_error_flag == 1) {
            error_count++;
            if(error_count>20)
                break;

            offset_count = offset/fs_send_max_size;
            printk("--CWMCU-- FILESEND: Re-sent bin file from error byte. count=%d\n", offset_count);

            memset(data,0,sizeof(data));
            memcpy(data,&offset_count,2);
            if (CWMCU_write_i2c_block(sensor, CW_FILE_SEND_SET_FLASH_CNT, data, 2)==0) {
                printk("--CWMCU-- FILESEND: Re-sent Bin file flash count = %d | offset = %d\n",offset_count,offset);
            } else {
                printk("--CWMCU-- FILESEND: Re-sent Bin file flash count error.\n");
            }
            usleep_range(3000, 4000);
            memset(data,0,sizeof(data));
            if (CWMCU_i2c_read(sensor, CW_FILE_SEND_SIZE, data, 4) >= 0) {
                get_fs_size = (int32_t)((uint32_t)(data[3]<<24) | (uint32_t)(data[2]<<16) | (uint32_t)(data[1]<<8) | (uint32_t)data[0]);
                printk("--CWMCU-- FILESEND: Re-sent=> fs_size=%x %x %x %x | %d\n", data[3],data[2],data[1],data[0], get_fs_size);
            }
        }

    }

    set_fs(pre_fs);
    filp_close(filp,NULL);

    /*  voice trigger restore status  */
    regValue = 1;
    if(sensor->vt_enabled) {
        printk("CWMCU: start to restore vt cmd\n");
        if (CWMCU_i2c_write(sensor, CW_VOICE_TRIGGER_CTRL, &regValue, 1) >=0) {
            printk("CWMCU: set vt cmd enable.\n");
            reg_ret = regulator_enable(sensor->mic);
        }
    } else {
        printk("CWMCU: do not need to restore vt cmd\n");
    }

    if(error_count>20) {
        fs_cmd = 4;//reset fs status
        sensor->fs_progress = -1;
        CWMCU_i2c_write(sensor, CW_FILE_SEND_CMD, &fs_cmd, 1);
        return 0;
    }

    fs_cmd = 3;//send bin file finish.
    CWMCU_i2c_write(sensor, CW_FILE_SEND_CMD, &fs_cmd, 1);
    printk("--CWMCU-- FILESEND: send bin file finish. \n");

    return 1;
}

static int dmic_enable_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int data = 0;

    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        return count;
    }

    printk(KERN_DEBUG "--CWMCU-- %s, %s in\n", buf, __func__);

    if ((buf[0] != '1') && (buf[0] != '0')) {
        printk(KERN_DEBUG "--CWMCU-- dmic switch value error(%s) (0:release 1:sensorhub) !!\n", buf);
        return count;
    }
    sscanf(buf, "%d\n", &data);
    printk(KERN_DEBUG "--CWMCU-- dmic status from [%d] to [%d] (0:release 1:sensorhub)!!\n", sensor->dmic_req_snshub, data);

    if (data != sensor->dmic_req_snshub) {
        if (data == DMIC_REQ_PROC_OPEN)
            dmic_enable_for_snshub(DMIC_REQ_PROC_OPEN);
        else
            dmic_enable_for_snshub(DMIC_REQ_PROC_RELEASE);
    }
    sensor->dmic_req_snshub = data;

    return count;
}

static int dmic_enable_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk(KERN_DEBUG "--CWMCU-- %s snshub_req_dmic = %d (0:release 1:sensorhub), codec_req_dmic = %d, dmic_on = %d\n",
        __func__, sensor->dmic_req_snshub, sensor->dmic_req_codec, sensor->dmic_on);
    return  0;
}

static int dmic_data_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    uint8_t data[6] = {0};
    uint16_t dmic_value = 0;

    printk("CWMCU %s\n", __func__);

    if (CWMCU_i2c_read(sensor, CW_GET_DMIC_BUF, data, 2) >= 0) {
        dmic_value = (uint16_t)data[1] << 8 | data[0];
        printk("CWMCU: Get dmic data %d [0-1]=> %x %x\n", dmic_value, data[0], data[1]);
    }

    return snprintf(buf, 8, "%d\n", dmic_value);
}

static int udt_control_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    uint8_t data = 0;

    printk("--CWMCU-- %s\n", __func__);

    if (CWMCU_i2c_read(sensor, CW_FILE_SEND_GET_UDT_TYPE, &data, 1) >= 0) {
        printk("Get UDT Type: %d\n", data);
    }

    return snprintf(buf, 8, "%d\n", data);
}

static int udt_control_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    u8 data[16] = {0};
    int data_buffer = 0;
    u8 regVaule;

    sscanf(buf, "%d %d\n", &sensor->udt_cmd, &sensor->udt_value);
    regVaule = sensor->udt_value;
    printk("CWMCU udt_cmd=%d udt_value=%d\n", sensor->udt_cmd, sensor->udt_value);

    switch (sensor->udt_cmd) {

    case UDT_FS_CMD:
        printk("--CWMCU-- UDT_FS_CMD => cmd_value: %d\n",regVaule);
        if(CWMCU_i2c_write(sensor, CW_FILE_SEND_CMD, &regVaule, 1) < 0) {
            printk("--CWMCU-- Send udt command error!\n");
        }
        break;

    case UDT_FS_GET_STATUS:
        printk("--CWMCU-- UDT_FS_GET_STATUS \n");
        if (CWMCU_i2c_read(sensor, CW_FILE_SEND_STATUS, data, 1) >= 0) {
            data_buffer = (int)data[0];
            printk("--CWMCU-- fs_status=%d \n",data_buffer);
        }
        break;

    case UDT_FS_BURN_VOICE_TO_MCU:
        printk("--CWMCU-- UDT_FS_BURN_VOICE_TO_MCU \n");
        sensor->mcu_mode = CW_BOOT;
        send_binfile_to_mcu(sensor->client, "voice.bin");
        sensor->mcu_mode = CW_NORMAL;
        break;

    default:
        break;
    }
    return count;
}

static int vt_sv_param_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    int svScore=0, paramAOffset=0;
    uint8_t data[4];

    sscanf(buf, "%d %d\n", &svScore, &paramAOffset);
    
    memcpy(data,&svScore,sizeof(uint8_t)*2);
    memcpy(data+2,&paramAOffset,sizeof(uint8_t)*2);
    
    printk("--CWMCU-- %s write data= %02x %02x %02x %02x\n", __func__, data[0],data[1],data[2],data[3]);
    
    CWMCU_write_i2c_block(sensor, CW_SET_VT_SV_PARAM, data, 4);
    if (CWMCU_write_i2c_block(sensor, CW_SET_VT_SV_PARAM, data, 4) < 0) {
        printk("--CWMCU-- Write svScore & paramAOffset error.\n");
    } else {
        printk("--CWMCU-- Write svScore=%d  paramAOffset=%d success.\n", svScore, paramAOffset);
    }

    return count;
}

static int filesend_progress_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("--CWMCU-- %s\n", __func__);

    return snprintf(buf, 8, "%d\n", sensor->fs_progress);
}
#endif

static int firmware_update_cmd_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("--CWMCU-- %s in\n", __func__);
    return snprintf(buf, 8, "%d\n", sensor->fwupdate_status);
}

static int firmware_update_cmd_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    bool ret = false;
    sscanf(buf, "%d\n", &sensor->cmd);
    printk(KERN_DEBUG "CWMCU cmd=%d \n", sensor->cmd);

    switch (sensor->cmd) {

    case CHANGE_TO_BOOTLOADER_MODE:
        printk("CWMCU CHANGE_TO_BOOTLOADER_MODE\n");
        sensor->fwupdate_status = 1;
        wake_lock_timeout(&fwupdate_wake_lock, 20*HZ);
        disable_irq(sensor->client->irq);

        sensor->mcu_mode = CW_BOOT;
        CWMCU_reset_mcu(false);
        sensor->client->addr = 0x18;

        if (!lpc5400x_probebus(sensor->client, SL_I2C1, sensor->irq_gpio)) {
            printk( "CWMCU Firmware update bus detection failed!\n");
            sensor->fwupdate_status = -1;
        }
        break;

    case MCU_IMAGE_WRITE:
        printk("CWMCU WRITE MCU IMAGE\n");
#if 1
        if(sensor->fwupdate_status == -1) {
            printk("CWMCU check firmware update status error.\n");
            break;
        }
        ret = lpc5400x_updateFirmware(sensor->client, sensor->fw_name);

        sensor->client->addr = 0x3a;

        if (ret == true) {
            sensor->fwupdate_status = 2;
            sensor->mcu_mode = CW_NORMAL;
            sensor->mcu_status = CW_VAILD;
            enable_irq(sensor->client->irq);
        } else {
            printk("CWMCU write firmware image error.\n");
            sensor->fwupdate_status = -1;
            sensor->mcu_mode = CW_BOOT;
        }
#endif
        break;

    default:
        break;
    }
    return count;
}

static int set_mcu_cmd(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    u8 data[40] = {0};
    s16 data_buff[3] = {0};
    u8 regValue;
    u32 McuReg = 0x80;
#ifdef CONFIG_SNR_HUB_VT
    int reg_ret = 0;
#endif
    int i;
    int inValueA, inValueB;
    u8  cmData[6] = {0};

    sscanf(buf, "%d %x %d\n", &sensor->cmd, &sensor->addr, &sensor->value);
    sscanf(buf, "%d %d %d\n", &sensor->cmd, &inValueA, &inValueB);
    McuReg = sensor->addr;
    regValue = sensor->value;
    cmData[0] = inValueA;
    cmData[1] = inValueB;
    printk(KERN_DEBUG "CWMCU cmd=%d addr=0x%x value=%d\n", sensor->cmd, McuReg, sensor->value);

    switch (sensor->cmd) {

    case MCU_REGISTER_WRITE_CTRL:
        printk("CWMCU MCU_REGISTER_WRITE_CTRL\n");
        if (CWMCU_i2c_write(sensor, McuReg, &regValue, 1) < 0) {
            printk("--CWMCU-- Write MCU Reg error.\n");
        } else {
            printk("--CWMCU-- Write MCU Reg=0x%x: value=%d\n",McuReg,regValue);
        }
        break;

    case MCU_REGISTER_READ_CTRL:
        printk("CWMCU MCU_REGISTER_READ_CTRL\n");
        if (CWMCU_i2c_read(sensor, McuReg, data, regValue) < 0) {
            printk("--CWMCU-- Read MCU Reg error.\n");
        } else {
            printk("--CWMCU-- Read MCU Reg=0x%x: read_size=%d\n",McuReg,regValue);
            for(i=0; i<regValue; i++) {
                printk("Read: data[%d] = %2x\n",i,data[i]);
            }
        }
        break;

    case CHECK_ACC_DATA:
        printk(KERN_DEBUG "CWMCU CHECK_ACC_DATA\n");
        if (CWMCU_i2c_read(sensor, CW_READ_ACCELERATION, data, 6) >= 0) {
            data_buff[0] = (s16)(((u16)data[1] << 8) | (u16)data[0]);
            data_buff[1] = (s16)(((u16)data[3] << 8) | (u16)data[2]);
            data_buff[2] = (s16)(((u16)data[5] << 8) | (u16)data[4]);

            printk("--CWMCU-- ACC_DATA: x = %d, y = %d, z = %d\n",
                data_buff[0], data_buff[1], data_buff[2]);
        }
        break;

    case CHECK_MAG_DATA:
        printk(KERN_DEBUG "CWMCU CHECK_MAG_DATA\n");
        if (CWMCU_i2c_read(sensor, CW_READ_MAGNETIC, data, 6) >= 0) {
            data_buff[0] = (s16)(((u16)data[1] << 8) | (u16)data[0]);
            data_buff[1] = (s16)(((u16)data[3] << 8) | (u16)data[2]);
            data_buff[2] = (s16)(((u16)data[5] << 8) | (u16)data[4]);

            printk("--CWMCU-- MAG_DATA: x = %d, y = %d, z = %d\n",
                data_buff[0], data_buff[1], data_buff[2]);
        }
        break;

    case CHECK_GYRO_DATA:
        printk(KERN_DEBUG "CWMCU CHECK_GYRO_DATA\n");
        if (CWMCU_i2c_read(sensor, CW_READ_GYRO, data, 6) >= 0) {
            data_buff[0] = (s16)(((u16)data[1] << 8) | (u16)data[0]);
            data_buff[1] = (s16)(((u16)data[3] << 8) | (u16)data[2]);
            data_buff[2] = (s16)(((u16)data[5] << 8) | (u16)data[4]);

            printk("--CWMCU-- GYRO_DATA: d[0-6] = %2x %2x %2x %2x %2x %2x\n",
                data[0],data[1],data[2],data[3],data[4],data[5]);
        }
        break;

#ifdef CONFIG_SNR_HUB_VT
    case FILE_SEND_GRAMNET_TO_MCU:
        sensor->mcu_mode = CW_BOOT;
        send_binfile_to_mcu(sensor->client, "voice.bin");
        sensor->mcu_mode = CW_NORMAL;
        break;

    case VOICE_TRIGGER_CTRL:
        if (CWMCU_i2c_write(sensor, CW_VOICE_TRIGGER_CTRL, &regValue, 1)) {
            printk("CWMCU: fail to send vt cmd\n");
        } else {
            sensor->vt_enabled = (regValue != 0);
            if (sensor->vt_enabled)
                reg_ret = regulator_enable(sensor->mic);
            else
                reg_ret = regulator_disable(sensor->mic);
        }
        printk("CWMCU: vt regulator_ret = %d\n", reg_ret);
        break;
#endif

    case WDT_FEED_STATUS:
        printk("CWMCU CW_WDT_FEED_STATUS\n");
        if (CWMCU_i2c_write(sensor, CW_WDT_FEED_STATUS, &regValue, 1) < 0) {
            printk("--CWMCU-- Write MCU CW_WDT_FEED_STATUS error.\n");
        }
        break;

    case MCU_DBG_INT_SWITCH:
        printk("CWMCU CW_MCU_DBG_INT_SWITCH\n");
        if (CWMCU_i2c_write(sensor, CW_MCU_DBG_INT_SWITCH, &regValue, 1) < 0) {
            printk("CWMCU Write CW_MCU_DBG_INT_SWITCH error.\n");
        }
        break;

    case SET_PROXIMITY_THRESHOLD:
        printk("CWMCU CW_SET_PROXIMITY_THRESHOLD\n");
        printk("CWMCU set threshold: %d %d\n", cmData[0],cmData[1]);
        if (CWMCU_write_i2c_block(sensor, CW_SET_PROXIMITY_THRESHOLD, cmData, 2) < 0) {
            printk("--CWMCU-- Write MCU CW_SET_PROXIMITY_THRESHOLD error.\n");
        }
        break;

    case SET_INTERRUPT_ENABLE:
        printk("CWMCU SET_INTERRUPT_ENABLE\n");
#ifdef CONFIG_SNS_HUB_DEBUG
        if (CWMCU_write_i2c_block(sensor, CW_SET_INTERRUPT_ENABLE, &regValue, 1) < 0) {
            printk("--CWMCU-- Write CW_SET_INTERRUPT_ENABLE error.\n");
        } else {
            sensor->power_test_on = (regValue == 0);
            printk("CWMCU INT status=%d.\n", sensor->power_test_on);
        }
#endif
        break;

    case MCU_CHECK_SENSOR_ALIVE_CTRL:
        printk("CWMCU MCU_CHECK_SENSOR_ALIVE_CTRL\n");
        sensor->check_muc_sensor_alive_flag = regValue;
        printk("CWMCU set muc_sensor_alive_flag: %d\n", sensor->check_muc_sensor_alive_flag);
        break;

    default:
        break;
    }
    return count;
}

/* get calibrator data */
static int get_calibrator_data(struct device *dev, struct device_attribute *attr, char *buf)
{
    int i = 0;
    uint8_t data[32] = {0};
    int8_t status = 0;

    data[0] = sensor->cal_cmd;
    data[1] = sensor->cal_type;
    data[2] = sensor->cal_id;

    switch (sensor->cal_cmd) {

    case CWMCU_CALIBRATOR_STATUS:
        printk("--CWMCU-- CWMCU_CALIBRATOR_STATUS\n");
        if (CWMCU_i2c_read(sensor, CW_CALIBRATOR_STATUS, &status, 1) >= 0) {
            printk("--CWMCU-- calibrator status = %d\n", status);
            return sprintf(buf, "%d\n", status);
        } else {
            printk("--CWMCU-- calibrator => i2c error\n");
            return sprintf(buf, "calibrator i2c error\n");
        }
        break;

    case CWMCU_CALIBRATOR_ACCELERATION:
        printk("--CWMCU-- CWMCU_CALIBRATOR_ACCELERATION read data\n");
        if (CWMCU_i2c_read(sensor, CW_CALIBRATOR_GET_BIAS_ACC, &data[3], 6) <= 0) {
            printk("--CWMCU-- i2c calibrator read fail!!! [ACC]\n");
        }
        break;

    case CWMCU_CALIBRATOR_MAGNETIC:
/*
        printk("--CWMCU-- CWMCU_CALIBRATOR_MAGNETIC read data\n");
        if (CWMCU_i2c_read(sensor, CW_CALIBRATOR_GET_BIAS_MAG, &data[9], 30) <= 0) {
            printk("--CWMCU-- i2c calibrator read fail!!! [MAG]\n");
        }
*/
        break;

    case CWMCU_CALIBRATOR_GYRO:
        printk("--CWMCU-- CWMCU_CALIBRATOR_GYRO read data\n");
        if (CWMCU_i2c_read(sensor, CW_CALIBRATOR_GET_BIAS_GYRO, &data[3], 6) <= 0) {
            printk("--CWMCU-- i2c calibrator read fail!!! [GYRO]\n");
        }
        break;

    case CWMCU_CALIBRATOR_LIGHT:
        printk("--CWMCU-- CWMCU_CALIBRATOR_LIGHT read data\n");
        if (CWMCU_i2c_read(sensor, CW_CALIBRATOR_GET_BIAS_LIGHT, &data[3], 6) <= 0) {
            printk("--CWMCU-- i2c calibrator read fail!!! [LIGHT]\n");
        }
        break;

    case CWMCU_CALIBRATOR_PROXIMITY:
        printk("--CWMCU-- CWMCU_CALIBRATOR_PROXIMITY read data\n");
        if (CWMCU_i2c_read(sensor, CW_CALIBRATOR_GET_BIAS_PROXIMITY, &data[3], 8) <= 0) {
            printk("--CWMCU-- i2c calibrator read fail!!! [PROXIMITY]\n");
        }
        break;

    case CWMCU_CALIBRATOR_PRESSURE:
        printk("--CWMCU-- CWMCU_CALIBRATOR_PRESSURE read data\n");
        if (CWMCU_i2c_read(sensor, CW_CALIBRATOR_GET_BIAS_PRESSURE, &data[3], 6) <= 0) {
            printk("--CWMCU-- i2c calibrator read fail!!! [PRESSURE]\n");
        }
        break;
    }

    for (i = 0; i < 32; i++) {
        printk(KERN_DEBUG "--CWMCU-- castor read data[%d] = %u\n", i, data[i]);
    }

    return sprintf(buf, "%d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d\n",
            data[0], data[1], data[2],
            data[3], data[4], data[5], data[6], data[7], data[8], data[9], data[10], data[11], data[12],
            data[13], data[14], data[15], data[16], data[17], data[18], data[19], data[20], data[21], data[22],
            data[23], data[24], data[25], data[26], data[27], data[28], data[29], data[30], data[31]);

}

static int set_calibrator_data(struct device *dev,struct device_attribute *attr,const char *buf, size_t count)
{
    int i = 0;
    uint8_t data[32] = {0};
    int temp[32] = {0};

    char source[512];
    char *pch;
    int buf_count=0;

    char *myBuf= source;

    strcpy(source,buf);
    //printk("--CWMCU-- source = %s | count:%d\n", source, count);

    while ((pch = strsep(&myBuf, ", ")) != NULL) {
        buf_count++;
    }
    //printk("--CWMCU-- buf = %s | bufcount:%d\n", buf, buf_count);


    /* Command mode */
    if (buf_count == 3) {
        sscanf(buf, "%d %d %d",&temp[0], &temp[1], &temp[2]);
        sensor->cal_cmd = (uint8_t)temp[0];
        sensor->cal_type = (uint8_t)temp[1];
        sensor->cal_id = (uint8_t)temp[2];
        printk("--CWMCU-- Calibrator=> cmd:%d type:%d id:%d\n", sensor->cal_cmd, sensor->cal_type, sensor->cal_id);

        if (sensor->cal_cmd == CWMCU_CALIBRATOR_ENABLE) {
            /* type=> 1=calibrator 2=selftest */
            printk("--CWMCU-- Calibrator=> set calibrator info\n");
            CWMCU_i2c_write(sensor, CW_CALIBRATOR_TYPE, &sensor->cal_type, 1);
            CWMCU_i2c_write(sensor, CW_CALIBRATOR_SENSOR_ID, &sensor->cal_id, 1);
        } else {
            printk(KERN_DEBUG "--CWMCU-- set command\n");
            return count;
        }
    } else if (buf_count == 4) {  /* write calibrator bias to mcu */
        sscanf(buf, "%d %d %d %d\n",&temp[0], &temp[1], &temp[2],&temp[3]);

        for (i = 0; i < 4; i++) {
            data[i] = (uint8_t)temp[i];
        }

        sensor->cal_cmd = data[0];
        sensor->cal_type = data[1];
        sensor->cal_id = data[2];

        printk("--CWMCU-- Calibrator=> set command=%d , type=%d, sensorid=%d\n", sensor->cal_cmd, sensor->cal_type, sensor->cal_id);

        switch (sensor->cal_cmd) {

        case CWMCU_CALIBRATOR_WRITE_BIAS:
            printk("--CWMCU-- CWMCU_CALIBRATOR_ACCELERATION write data\n");
            CWMCU_i2c_write(sensor, CW_CALIBRATOR_SET_BIAS, &sensor->cal_id, 1);
            break;
        }
    } else {
        printk("--CWMCU-- input parameter incorrect !!! | %d\n",count);
        return count;
    }

    return count;
}

#define MAG_VENDOR_AKM           1
#define MAG_VENDOR_ALPS          2

static uint8_t mag_vendor = MAG_VENDOR_ALPS;
static void get_mag_vendor(void)
{

    printk("--CWMCU-- %s\n", __func__);

    if (CWMCU_i2c_read(sensor, CW_MAG_GET_VENDOR_TYPE, &mag_vendor, 1) >= 0) {
        printk("--CWMCU-- get mag vendor: %d\n", mag_vendor);

        if ((mag_vendor != MAG_VENDOR_AKM) && (mag_vendor != MAG_VENDOR_ALPS)) {
	    mag_vendor = MAG_VENDOR_ALPS;
            printk("--CWMCU-- mag vendor error, the default vendor is: %d\n", mag_vendor);
        }
    } else {
	mag_vendor = MAG_VENDOR_ALPS;
        printk("--CWMCU-- cannot get mag vendor, the default vendor is: %d\n", mag_vendor);
    }
}

static int get_magnetic_vendor(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("--CWMCU-- %s\n", __func__);
    printk("--CWMCU-- mag vendor (local): %d\n", mag_vendor);

    return snprintf(buf, 8, "%d\n", mag_vendor);
}


static int version_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    uint8_t data[6] = {0};

    printk("--CWMCU-- %s\n", __func__);

    if (cwmcu_get_version())
        sensor->mcu_status = CW_INVAILD;
    else {
        sensor->mcu_status = CW_VAILD;

        if (CWMCU_i2c_read(sensor, CW_FWVERSION, data, 2) >= 0) {
            printk("CHECK_FIRMWARE_VERSION : %d.%d\n", data[0],data[1]);
        }
    }

    return snprintf(buf, 8, "%d.%d\n", data[0],data[1]);
}


static int led_control_set(struct device *dev,struct device_attribute *attr,const char *buf, size_t count)
{
    int value;
    uint8_t data;

    sscanf(buf, "%d\n", &value);
    data = (uint8_t)value;
    printk("--CWMCU-- %s led value is %d\n", __func__, data);
    if (sensor->mcu_mode == CW_NORMAL && sensor->mcu_power == CW_NORMAL) {
        CWMCU_i2c_write(sensor, CW_LED_CTRL, &data, 1);
        if (CWMCU_i2c_write(sensor, CW_LED_CTRL, &data, 1) < 0)
            printk("led_control_set fail and CWMCU_i2c_write error\n");
    }
    return count;
}

static int power_control_set(struct device *dev,struct device_attribute *attr,const char *buf, size_t count)
{
    int value;
    uint8_t data;

    printk("--CWMCU-- %s\n", __func__);
    sscanf(buf, "%d\n", &value);
    data = (uint8_t)value;

    if(data==0xFF)
        sensor->mcu_power = CW_NORMAL;
    else
        sensor->mcu_power = CW_BOOT;

    CWMCU_i2c_write(sensor, CW_POWER_CTRL, &data, 1);

    return count;
}

static int firmware_name_set(struct device *dev,struct device_attribute *attr,const char *buf, size_t count)
{
    printk("--CWMCU-- %s\n", __func__);
    sscanf(buf, "%s\n", sensor->fw_name);

    printk("--CWMCU-- sensor->fw_path : %s\n", sensor->fw_name);

    return count;
}

static int firmware_name_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("--CWMCU-- %s : %s\n", __func__, sensor->fw_name);

    return snprintf(buf, 50, "%s\n", sensor->fw_name);
}

static int verbose_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    int i;
    printk("--CWMCU-- %s\n", __func__);

    for (i =0; i < 5; i++)
        printk("CWMCU DEBUG_ONLY -------------------------------------------------\n");
    return snprintf(buf, 50, "%d\n", flagVerbose);
}

static int verbose_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%d\n", &flagVerbose);
    printk("--CWMCU-- %s, parsed %d\n", __func__, flagVerbose);

    return count;
}

static char sensor_type_desc[CW_SENSORS_ID_END][31] = {
"acceleration",
"magnetic",
"gyro",
"light",
"proximity",
"pressure",
"orientation",
"rotationvector",
"linearacceleration",
"gravity",
"step_counter",
"step_detector",
"magnetic_uncalibrated",
"gyroscope_uncalibrated",
"game_rotation_vector",
"geomagnetic_rotation_vector",
"significant_motion",
"single_snap",
"double_snap",
"tap",
"context_aware",
"proximity_gesture",
"light_rgb",
"voice_trigger",
"screen_on",
};

static int sns_info_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    int len = 0;
    uint8_t id;
    uint8_t data[6];
    uint32_t mcu_enable_list = 0;

    printk("--CWMCU-- %s\n", __func__);


    if (CWMCU_i2c_read(sensor, CW_ENABLE_STATUS, data, 4) < 0) {
        printk("--CWMCU-- get mcu enable list error.\n");
    }
    mcu_enable_list = (uint32_t)data[3]<< 24 | (uint32_t)data[2]<< 16 | (uint32_t)data[1]<< 8 | data[0];

    len += sprintf(buf + len,
                        "mcu_mode                  : %d\n" \
                        "mcu_status                : %d\n" \
                        "mcu_power                 : %d\n" \
                        "mcu_screen                : %d\n" \
                        "mcu_reset                 : %d\n" \
                        "enabled_list              : 0x%x\n" \
                        "mcu_enabled_list          : 0x%x\n" \
                        "batched_list              : 0x%x\n" \
                        "update_list               : 0x%x\n" \
                        "suspend_enabled_list      : 0x%x\n" \
                        "suspend_enabled_list(pre) : 0x%x\n" \
                        "batch_timeout             : %u\n" \
                        "interrupt_status          : 0x%x\n" \
                        "power_on_list             : %d\n" 
#ifdef CONFIG_SNR_HUB_VT
                        "dmic_req_codec            : %d   (1:request  0:free)\n" \
                        "dmic_req_snshub           : %d   (1:request  0:free)\n" \
                        "dmic_on                   : %d   (1:open     0:release)\n" \
                        "dmic_codec_release        : %d   (1:free     0:request)\n" \
                        "vt_enabled                : %d\n"
#endif
                        ,
                        sensor->mcu_mode,
                        sensor->mcu_status,
                        sensor->mcu_power,
                        sensor->mcu_screen,
                        sensor->mcu_reset,
                        sensor->enabled_list,
                        mcu_enable_list,
                        sensor->batched_list,
                        sensor->update_list,
                        sensor->suspend_enabled_list,
                        SUSPEND_ENABLE_LIST,
                        sensor->batch_timeout,
                        sensor->interrupt_status,
#ifndef CONFIG_SNR_HUB_VT
                        sensor->power_on_list
#else
                        sensor->power_on_list,
                        sensor->dmic_req_codec,
                        sensor->dmic_req_snshub,
                        sensor->dmic_on,
                        sensor->dmic_codec_release,
                        sensor->vt_enabled
#endif
                            );

    len += sprintf(buf + len, "\n.......................................................................................................\n");
    len += sprintf(buf + len, "description                   id :  active  batch  update  suspend  suspend   report       sensor_batch\n");
    len += sprintf(buf + len, "                                 :                                  (pre)     period(ms)   timeout(ms)\n");
    for (id = 0; id < CW_SENSORS_ID_END; id++) {
        len += sprintf(buf + len, "%-30s%2d :%6s  %5s  %6s %7s  %7s   %7d           %7d\n",
                sensor_type_desc[id],
                id,
                (sensor->enabled_list & (1 << id)) ? "Y" : "",
                (sensor->batched_list & (1 << id)) ? "Y" : "",
                (sensor->update_list & (1 << id)) ? "Y" : "",
                (sensor->suspend_enabled_list & (1 << id)) ? "Y" : "",
                (SUSPEND_ENABLE_LIST & (1 << id)) ? "Y" : "",
                (uint8_t)sensor->report_period[id],
                sensor->sensor_timeout[id]);
    }
    return len;
}

static int sns_debug_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    printk("--CWMCU-- %s\n", __func__);

    return sprintf(buf, "0x%x\n%s\n", sensor->sns_debug_flag,
                        "bit 0: support power key event");
}

static int sns_debug_set(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    sscanf(buf, "%d\n", &sensor->sns_debug_flag);
    printk("--CWMCU-- %s, sns_debug_flag = 0x%x\n", __func__, sensor->sns_debug_flag);

    return count;
}



static int psensor_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    uint8_t data = 0;

    printk("--CWMCU-- %s\n", __func__);

    if (CWMCU_i2c_read(sensor, CW_PSENSOR_RAW_DATA_GET, &data, 1) >= 0) {
        printk("Get P-sensor RAWDATA: %d\n", data);
    }

    return snprintf(buf, 8, "%d\n", data);
}



static DEVICE_ATTR(enable, 0644, active_show, active_set);
static DEVICE_ATTR(delay_ms, 0644, interval_show, interval_set);
static DEVICE_ATTR(batch, 0644, batch_show, batch_set);
static DEVICE_ATTR(flush, 0644, flush_show, flush_set);

static DEVICE_ATTR(calibrator_cmd, 0644, get_calibrator_data, set_calibrator_data);

static DEVICE_ATTR(firmware_update_cmd, 0644, firmware_update_cmd_show, firmware_update_cmd_set);
static DEVICE_ATTR(firmware_name, 0644, firmware_name_show, firmware_name_set);
static DEVICE_ATTR(version, 0644, version_show, NULL);
static DEVICE_ATTR(mcu_cmd, 0644, NULL, set_mcu_cmd);

static DEVICE_ATTR(led_control, 0644, NULL, led_control_set);
static DEVICE_ATTR(power_control, 0644, NULL, power_control_set);
static DEVICE_ATTR(mag_vendor, 0644, get_magnetic_vendor, NULL);
#ifdef CONFIG_SNR_HUB_VT
static DEVICE_ATTR(fs_progress, 0644, filesend_progress_show, NULL);
static DEVICE_ATTR(udt_control, 0644, udt_control_show, udt_control_set);
static DEVICE_ATTR(dmic_data, 0644, dmic_data_show, NULL);
static DEVICE_ATTR(dmic_enable, 0644, dmic_enable_show, dmic_enable_set);
static DEVICE_ATTR(vt_sv_param, 0644, NULL, vt_sv_param_set);
#endif
static DEVICE_ATTR(psensor, 0644, psensor_show, NULL);
static DEVICE_ATTR(verbose, 0644, verbose_show, verbose_set);
static DEVICE_ATTR(sns_debug, 0644, sns_debug_show, sns_debug_set);
static DEVICE_ATTR(sns_info, 0644, sns_info_show, NULL);

static struct attribute *sysfs_attributes[] = {
    &dev_attr_enable.attr,
    &dev_attr_delay_ms.attr,
    &dev_attr_batch.attr,
    &dev_attr_flush.attr,
    &dev_attr_mcu_cmd.attr,
    &dev_attr_firmware_update_cmd.attr,
    &dev_attr_calibrator_cmd.attr,
    &dev_attr_version.attr,
    &dev_attr_led_control.attr,
    &dev_attr_power_control.attr,
    &dev_attr_firmware_name.attr,
    &dev_attr_mag_vendor.attr,
    &dev_attr_psensor.attr,
    &dev_attr_verbose.attr,
    &dev_attr_sns_debug.attr,
    &dev_attr_sns_info.attr,
#ifdef CONFIG_SNR_HUB_VT
    &dev_attr_fs_progress.attr,
    &dev_attr_udt_control.attr,
    &dev_attr_dmic_data.attr,
    &dev_attr_dmic_enable.attr,
    &dev_attr_vt_sv_param.attr,
#endif
    NULL
};

static struct attribute_group sysfs_attribute_group = {
    .attrs = sysfs_attributes
};

/*=======input device==========*/
static int CWMCU_init_input_device(struct CWMCU_data *sensor)
{
    int err = 0;

    sensor->input = input_allocate_device();
    if (!sensor->input) {
        printk("Failed to allocate gyro input device\n");
        err = -ENOMEM;
    }

    sensor->input->name = CWMCU_I2C_NAME;
    sensor->input->id.bustype = BUS_I2C;
    sensor->input->dev.parent = &sensor->client->dev;
//    sensor->input->evbit[0] = BIT_MASK(EV_ABS) | BIT_MASK(EV_ABS);
//    set_bit(EV_KEY, sensor->input->evbit);
//
//    /* send mouse event */
//    set_bit(BTN_MOUSE, sensor->input->keybit);
//    set_bit(EV_REL, sensor->input->evbit);
//    set_bit(REL_X, sensor->input->relbit);
//    set_bit(REL_Y, sensor->input->relbit);
//    set_bit(EV_MSC, sensor->input->evbit);
//    set_bit(MSC_SCAN, sensor->input->mscbit);
//    set_bit(BTN_LEFT, sensor->input->keybit);
//    set_bit(BTN_RIGHT, sensor->input->keybit);
//
//    input_set_capability(sensor->input, EV_KEY, 116);
//    input_set_capability(sensor->input, EV_KEY, 102);
    /*
    input_set_capability(idev, EV_KEY, 88);
    */
    set_bit(EV_ABS, sensor->input->evbit);
    input_set_abs_params(sensor->input, CW_ABS_X, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_Y, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_Z, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_X1, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_Y1, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_Z1, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_TIMEDIFF, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_ACCURACY, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_TIMESTAMP_H, -DPS_MAX, DPS_MAX, 0, 0);
    input_set_abs_params(sensor->input, CW_ABS_TIMESTAMP_L, -DPS_MAX, DPS_MAX, 0, 0);
//    input_set_abs_params(sensor->input, REL_X, -DPS_MAX, DPS_MAX, 0, 0);
//    input_set_abs_params(sensor->input, REL_Y, -DPS_MAX, DPS_MAX, 0, 0);

    err = input_register_device(sensor->input);
    if (err) {
        pr_err("Failed to register gyro input device\n");
        input_free_device(sensor->input);
        sensor->input = NULL;
    }

    return err;
}


static void cwmcu_screen_off_suspend()
{
    //int part;
    //u8 list;
    uint32_t enable_list_temp;
    u8 lstdat[4];

    printk("--CWMCU-- %s\n", __func__);

#ifdef CONFIG_SNS_HUB_DEBUG
    if (sensor->power_test_on) {
        sensor->is_suspend = true;
        printk("CWMCU: Power testing, do nothing.\n");
        return;
    }
#endif

    sensor->suspend_enabled_list = ( SUSPEND_ENABLE_LIST & sensor->enabled_list) | sensor->batched_list;
    printk("--CWMCU-- CWMCU_suspend: suspend_enabled_list = 0x%x\n", sensor->suspend_enabled_list);

    enable_list_temp = sensor->suspend_enabled_list & (~(1<<CW_VOICE_TRIGGER));
    /*
    for (part = 0; part < 4; part++) {
        list = (u8)(enable_list_temp>>(part*8));
        CWMCU_i2c_write(sensor, CW_ENABLE_REG+part, &list, 1);
    }
    */
    memcpy(lstdat, &enable_list_temp, sizeof(enable_list_temp));
    CWMCU_write_i2c_block(sensor, CW_SUSPEND_ENABLE, lstdat, 4);

    sensor->is_suspend = true; // Disable all following i2c ops in IRQ handler until resume.
    printk("CWMCU waitting for queued works done\n");
    flush_work(&sensor->work);
    printk("CWMCU all queued works done\n");
#ifdef CONFIG_SNR_HUB_MAG_LEAK_HACK
    usleep(20000);
    gpio_set_value(sensor->compass_reset_gpio, 0);
#endif

    //sensor->mcu_mode = CW_SUSPEND;

    printk("CWMCU suspend completed\n");
    return;
}

static void cwmcu_screen_on_resume()
{
    //int id,part;
    //u8 list;
    //uint8_t delay_ms = 0;
    //u8 data[2] = {0};
    uint32_t enable_list_temp;
    u8 lstdat[4];

    printk("--CWMCU-- %s\n", __func__);

#ifdef CONFIG_SNS_HUB_DEBUG
    queue_len = 0;
    if (sensor->power_test_on) {
        sensor->is_suspend = false;
        printk("CWMCU: Power testing, do nothing.\n");
        return;
    }
#endif

#ifdef CONFIG_SNR_HUB_MAG_LEAK_HACK
    gpio_direction_output(sensor->compass_reset_gpio, 0);
    udelay(5);
    gpio_set_value(sensor->compass_reset_gpio, 1);
    usleep(100);
#endif

    usleep(10000);
    sensor->is_suspend = false;

    //sensor->mcu_mode = CW_SUSPEND;
    sensor->suspend_enabled_list = (SUSPEND_ENABLE_LIST | sensor->enabled_list)^SUSPEND_ENABLE_LIST;
    printk("--CWMCU-- CWMCU_resume: suspend_enabled_list = 0x%x\n", sensor->suspend_enabled_list);

    enable_list_temp = sensor->enabled_list & (~(1<<CW_VOICE_TRIGGER));

    memcpy(lstdat, &enable_list_temp, sizeof(enable_list_temp));
    CWMCU_write_i2c_block(sensor, CW_RESUME_ENABLE, lstdat, 4);
/*
    //enable sensor
    for (id = 0; id < CW_SENSORS_ID_END; id++) {
        if (enable_list_temp & (1<<id)) {
            part = id / 8;
            list = (u8)(enable_list_temp>>(part*8));
            CWMCU_i2c_write(sensor, CW_ENABLE_REG+part, &list, 1);
            id = ((part+1)*8)-1;
        }
    }

    //set sensor report rate
    for (id = 0; id < CW_SENSORS_ID_END; id++) {
        if (sensor->suspend_enabled_list & (1<<id)) {
            delay_ms = (uint8_t)sensor->report_period[id];
            data[0] = (uint8_t)id;
            data[1] = delay_ms;
            CWMCU_write_i2c_block(sensor, CW_SENSOR_DELAY_SET, data, 2);
        }
    }
    */
    //sensor->mcu_mode = CW_NORMAL;
    printk("CWMCU resume completed\n");
}

static int CWMCU_suspend(struct device *dev)
{
    printk("--CWMCU-- %s\n", __func__);

    cwmcu_screen_off_suspend();

    return 0;
}

static int CWMCU_resume(struct device *dev)
{
    printk("--CWMCU-- %s\n", __func__);

    cwmcu_screen_on_resume();

    return 0;
}

#ifdef CWMCU_INTERRUPT
static irqreturn_t CWMCU_interrupt_thread(int irq, void *data)
{
    VERBOSE(DBG_LOG, 0, "--CWMCU--%s in\n", __func__);
    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        VERBOSE(DBG_LOG, 0, "--CWMCU--%s sensor->mcu_mode = %d\n", __func__, sensor->mcu_mode);
        return IRQ_HANDLED;
    }
    if (!sensor->is_suspend) {
        schedule_work(&sensor->work);
#ifdef CONFIG_SNS_HUB_DEBUG
        VERBOSE(DBG_LOG, 0, "--CWMCU--IRQ queued %d works\n", ++queue_len);
#endif
    }

    return IRQ_HANDLED;
}


/*  function for interrupt work  */
static void cwmcu_work_report(struct work_struct *work)
{
    uint8_t temp[6] = {0};
    char mStatus;

#ifdef CONFIG_SNS_HUB_DEBUG
    VERBOSE(DBG_LOG, 0, "--CWMCU-- cwmcu_work_report mcu_mode= %d mcu_status=%d, %d\n",sensor->mcu_mode, sensor->mcu_status, --queue_len);
#else
    VERBOSE(DBG_LOG, 0, "--CWMCU-- cwmcu_work_report mcu_mode= %d mcu_status=%d\n",sensor->mcu_mode, sensor->mcu_status);
#endif
    if ((sensor->mcu_mode == CW_BOOT) || sensor->mcu_status == CW_INVAILD || sensor->mcu_power == CW_BOOT) {
        printk(KERN_DEBUG "--CWMCU--%s sensor->mcu_mode = %d\n", __func__, sensor->mcu_mode);
        return;
    }

    /* check mcu interrupt status */
    if (CWMCU_i2c_read(sensor, CW_INTERRUPT_STATUS, temp, 6) >= 0) {
        sensor->interrupt_status = (u32)temp[1] << 8 | (u32)temp[0];
        sensor->update_list = (u32)temp[5] << 24 | (u32)temp[4] << 16 | (u32)temp[3] << 8 | (u32)temp[2];
        VERBOSE(DBG_LOG, 0, "--CWMCU-- sensor->interrupt_status~ = %d | update_list: 0x%8X\n", sensor->interrupt_status, sensor->update_list);
    } else {
        printk( "--CWMCU-- check interrupt_status failed~!!\n");
        sensor->interrupt_status = 0;
    }


    /* MCU reset */
    if (sensor->interrupt_status & (1<<INTERRUPT_INIT)) {
        if (CWMCU_i2c_read(sensor, CW_MCU_STATUS_GET, temp, 1) >= 0) {
            mStatus = (char)temp[0];
            if(mStatus==0) {
                printk( "--CWMCU-- Check MCU Reset => %d\n",sensor->mcu_reset);
                if(sensor->mcu_reset > 0) {
                    cwmcu_restore_status();
                    printk( "--CWMCU-- Check MCU restore status.\n");
                }
                sensor->mcu_reset++;
            } else if(mStatus==1) {
                cwmcu_check_mcu_enable();
            }
        } else {
            printk( "--CWMCU-- Check MCU status i2c fail.\n");
        }
    }


    /* read MCU error log data */
    if (sensor->interrupt_status & (1<<INTERRUPT_ERROR_LOG)) {
        cwmcu_error_log_read(sensor);
    }

    /* read MCU debug log data */
    if (sensor->interrupt_status & (1<<INTERRUPT_DEBUG_LOG)) {
        cwmcu_debug_log_read(sensor);
    }

    /* read sensor data of batch mode*/
    if (sensor->interrupt_status & (1<<INTERRUPT_BATCHTIMEOUT)
        || sensor->interrupt_status & (1<<INTERRUPT_BATCHFULL)
        || sensor->interrupt_status & (1<<INTERRUPT_BATCH_FLUSH)
        || sensor->interrupt_status & (1<<INTERRUPT_BATCH_TIMESTAMP_SYNC)) {
        cwmcu_batch_read(sensor);
    }

#if 0
    /* batch mode queue is full sync */
    if (sensor->interrupt_status & (1<<INTERRUPT_BATCH_TIMESTAMP_SYNC)) {
        cwmcu_batch_timestamp_sync(sensor);
    }
#endif

    /* read gesture event */
    if (sensor->interrupt_status & (1<<INTERRUPT_GESTURE)) {
        cwmcu_gesture_read(sensor);
    }

#ifdef CONFIG_SNR_HUB_VT
    /* read voice trigger event */
    if (sensor->interrupt_status & (1<<INTERRUPT_VOICETRIGGER)) {
        cwmcu_voice_trigger_read(sensor);
    }
#endif

    /* read sensor data of normal mode*/
    if (sensor->interrupt_status & (1<<INTERRUPT_DATAREADY)) {
        CWMCU_read(sensor);
    }
}
#endif

static int cwstm_parse_dt(struct device *dev, struct CWMCU_data *sensor)
{
    struct device_node *np = dev->of_node;
    int ret = 0;

    ret = of_get_named_gpio(np, "cwstm,irq-gpio", 0);
    if (ret < 0) {
        pr_err("failed to get \"cwstm,irq_gpio\"\n");
        goto err;
    }
    sensor->irq_gpio = ret;

    ret = of_get_named_gpio(np, "cwstm,mic-gpio", 0);
    if (ret < 0) {
        pr_err("failed to get \"cwstm,mic-gpio\"\n");
        goto err;
    }
    sensor->mic_gpio = ret;

    ret = of_get_named_gpio(np, "cwstm,reset-gpio", 0);
    if (ret < 0) {
        pr_err("failed to get \"reset\"\n");
        goto err;
    }
    sensor->reset_gpio = ret;

#ifdef CONFIG_SNR_HUB_DVT1_COMPATIBLE
    ret = of_get_named_gpio(np, "cwstm,reset-gpio-dvt1", 0);
    if (ret < 0) {
        pr_err("failed to get \"reset-dvt1\"\n");
        goto err;
    }
    sensor->reset_gpio_dvt1 = ret;
#endif

    ret = of_get_named_gpio(np, "cwstm,wakeup-gpio", 0);
    if (ret < 0) {
        pr_err("failed to get \"wakeup\"\n");
        goto err;
    }
    sensor->wakeup_gpio = ret;

    ret = of_get_named_gpio(np, "cwstm,compass-reset-gpio", 0);
    if (ret < 0) {
        pr_err("failed to get \"reset\"\n");
        goto err;
    }
    sensor->compass_reset_gpio = ret;

err:
    return ret;
}

static int cwstm_power_on(bool on)
{
    int rc = 0;

    if (!on)
        goto power_off;

    rc = regulator_enable(sensor->vdd);
    if (rc) {
        dev_err(&sensor->client->dev, "Regulator vdd enable failed rc=%d\n", rc);
        return rc;
    }

    if (!IS_ERR(sensor->vcc_i2c)) {
        rc = regulator_enable(sensor->vcc_i2c);
        if (rc) {
            dev_err(&sensor->client->dev, "Regulator vcc_i2c enable failed rc=%d\n", rc);
            rc = regulator_disable(sensor->vcc_i2c);
        }
    }
    return rc;

power_off:
    rc = regulator_disable(sensor->vdd);
    if (rc) {
        dev_err(&sensor->client->dev,
                "Regulator vdd disable failed rc=%d\n", rc);
        return rc;
    }

    rc = regulator_disable(sensor->vcc_i2c);
    if (rc) {
        dev_err(&sensor->client->dev,
                "Regulator vcc_i2c disable failed rc=%d\n", rc);
        rc = regulator_enable(sensor->vcc_i2c);
    }

    return rc;
}

static int cwstm_power_init(bool on)
{
    int rc;

    if (!on)
        goto pwr_deinit;

    sensor->vdd = regulator_get(&sensor->client->dev, "vdd");
    if (IS_ERR(sensor->vdd)) {
        rc = PTR_ERR(sensor->vdd);
        dev_err(&sensor->client->dev, "Regulator get failed vdd rc=%d\n", rc);
        return rc;
    }

    if (regulator_count_voltages(sensor->vdd) > 0) {
        rc = regulator_set_voltage(sensor->vdd, FT_VTG_MIN_UV,
                FT_VTG_MAX_UV);
        if (rc) {
            dev_err(&sensor->client->dev, "Regulator set_vtg failed vdd rc=%d\n", rc);
            goto reg_vdd_put;
        }
    }

    sensor->vcc_i2c = regulator_get(&sensor->client->dev, "vcc_i2c");
    if (IS_ERR(sensor->vcc_i2c)) {
        rc = PTR_ERR(sensor->vcc_i2c);
        dev_err(&sensor->client->dev, "Regulator get failed vcc_i2c rc=%d\n", rc);
        goto reg_vdd_set_vtg;
    }

#ifdef CONFIG_SNR_HUB_VT
    sensor->mic = regulator_get(&sensor->client->dev, "mic");
    if (IS_ERR(sensor->mic)) {
        rc = PTR_ERR(sensor->mic);
        dev_err(&sensor->client->dev, "Regulator get failed mic rc=%d\n", rc);
        goto reg_mic_put;
    }
#endif

    if (regulator_count_voltages(sensor->vcc_i2c) > 0) {
        rc = regulator_set_voltage(sensor->vcc_i2c, FT_I2C_VTG_MIN_UV,
                FT_I2C_VTG_MAX_UV);
        if (rc) {
            dev_err(&sensor->client->dev, "Regulator set_vtg failed vcc_i2c rc=%d\n", rc);
            goto reg_vcc_i2c_put;
        }
    }

#ifdef CONFIG_SNR_HUB_VT
    if (regulator_count_voltages(sensor->mic) > 0) {
        rc = regulator_set_voltage(sensor->mic, 3075000, 3075000);
        if (rc) {
            dev_err(&sensor->client->dev, "Regulator set_vtg failed mic rc=%d\n", rc);
            goto reg_mic_put;
        }
    }
#endif
    return 0;
reg_vcc_i2c_put:
    regulator_put(sensor->vcc_i2c);
#ifdef CONFIG_SNR_HUB_VT
reg_mic_put:
    regulator_put(sensor->mic);
#endif
reg_vdd_set_vtg:
    if (regulator_count_voltages(sensor->vdd) > 0)
        regulator_set_voltage(sensor->vdd, 0, FT_VTG_MAX_UV);
reg_vdd_put:
    regulator_put(sensor->vdd);
    return rc;

pwr_deinit:
    if (regulator_count_voltages(sensor->vdd) > 0)
        regulator_set_voltage(sensor->vdd, 0, FT_VTG_MAX_UV);

    regulator_put(sensor->vdd);

    if (regulator_count_voltages(sensor->vcc_i2c) > 0)
        regulator_set_voltage(sensor->vcc_i2c, 0, FT_I2C_VTG_MAX_UV);

    regulator_put(sensor->vcc_i2c);
#ifdef CONFIG_SNR_HUB_VT
    regulator_put(sensor->mic);
#endif
    return 0;
}


static int CWMCU_i2c_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
    int error = 0;
    int i = 0;
    int ret = 0;

    printk(KERN_DEBUG "--CWMCU-- %s\n", __func__);

    printk(KERN_DEBUG "CWMCU SW CFG: "
#ifdef CWMCU_INTERRUPT
    "INT/" 
#else
    "POLL/"
#endif
#ifdef CONFIG_SNR_HUB_DVT1_COMPATIBLE
    "DVT1_OK/" 
#endif
#ifdef CONFIG_SNS_HUB_DEBUG
    "DEBUG/"  
#endif
#ifdef CONFIG_SNR_HUB_LG_HACK
    "LG_HACK/"
#endif
#ifdef CONFIG_SNR_HUB_MAG_LEAK_HACK
    "MAG_LEAK_HACK/"
#endif
#ifdef CONFIG_SNR_HUB_I2C_HACK
    "I2C_HACK/"
#endif
#ifdef CONFIG_SNR_HUB_VT
    "VT/"
#endif
    "\n");

    dev_dbg(&client->dev, "%s:\n", __func__);

    if (!i2c_check_functionality(client->adapter, I2C_FUNC_I2C)) {
        dev_err(&client->dev, "--CWMCU-- i2c_check_functionality error\n");
        return -EIO;
    }

    sensor = kzalloc(sizeof(struct CWMCU_data), GFP_KERNEL);
    if (!sensor) {
        printk(KERN_DEBUG "--CWMCU-- kzalloc error\n");
        return -ENOMEM;
    }

    sensor->client = client;
    i2c_set_clientdata(client, sensor);

    error = cwstm_parse_dt(&client->dev, sensor);
    if (error < 0) {
        pr_err("failed to parse device tree: %d\n", error);
        goto err_parse_dt;
    }
    printk("%s: irq = %d, wakeup = %d, reset = %d\n", __func__, sensor->irq_gpio, sensor->wakeup_gpio, sensor->reset_gpio);
#ifdef CONFIG_SNR_HUB_DVT1_COMPATIBLE
    printk("%s: reset-dvt1 = %d\n", __func__, sensor->reset_gpio_dvt1);
#endif
    gpio_request(sensor->wakeup_gpio, "cwstm,wakeup-gpio");
    gpio_direction_output(sensor->wakeup_gpio, 1);
    gpio_request(sensor->reset_gpio, "cwstm,reset-gpio");
    gpio_request(sensor->mic_gpio, "cwstm,mic-gpio");
#ifdef CONFIG_SNR_HUB_DVT1_COMPATIBLE
    gpio_request(sensor->reset_gpio_dvt1, "cwstm,reset-gpio-dvt1");
#endif
    gpio_request(sensor->irq_gpio, "cwstm,irq-gpio");
    gpio_request(sensor->compass_reset_gpio, "cwstm,compass-reset-gpio");

//    error = i2c_smbus_read_byte_data(client, CW_FWVERSION);
//    pr_info("%s:======fw version is 0x%x=======\n", __func__, error);
//    if (error < 0 )
//        return -1;
#ifdef CONFIG_SNR_HUB_MAG_LEAK_HACK
    gpio_direction_output(sensor->compass_reset_gpio, 0);
    udelay(5);
    gpio_set_value(sensor->compass_reset_gpio, 1);
    udelay(100);
#else
    gpio_direction_output(sensor->compass_reset_gpio, 1);
#endif

    error = cwstm_power_init(true);
    if (error) {
        dev_err(&client->dev, "power init failed");
    }

    error = cwstm_power_on(true);
    if (error) {
        dev_err(&client->dev, "power on failed");
    }

    CWMCU_reset_mcu(true); // Assure that we reset MCU during driver probe.

    CWMCU_init_input_device(sensor);

    error = sysfs_create_group(&sensor->input->dev.kobj,
                    &sysfs_attribute_group);
    if (error)
        goto exit_free_input;

    for (i = 0; i < CW_SENSORS_ID_END; i++) {
        sensor->report_period[i] = 200;
    }

    sensor->mcu_mode = CW_NORMAL;
    sensor->mcu_power = CW_NORMAL;
    sensor->mcu_screen = CW_NORMAL;
    sensor->mcu_reset = 0;

#ifdef CONFIG_SNR_HUB_VT
    sensor->pre_voice_trigger_value = 0;
    sensor->vt_enabled = false;
    /* Below two lines suspect that mic is not used by audio system on boot, and if audio wanna to use it,
     * sensor-hub will close clock output to mic and should have no side-effect. */
    sensor->dmic_req_codec = DMIC_REQ_PROC_RELEASE;
    sensor->dmic_req_snshub = DMIC_REQ_PROC_RELEASE;
    sensor->dmic_on = SNSHUB_DMIC_OFF;
    sensor->dmic_codec_release = true;
    INIT_WORK(&sensor->dmic_work, snshub_dmic_worker);
#endif
    mutex_init(&sensor->lock);

#ifdef CWMCU_INTERRUPT
    sensor->client->irq = gpio_to_irq(sensor->irq_gpio);

    printk( "--CWMCU--client->irq  =%d, irq_gpio %d~!!\n", sensor->client->irq, sensor->irq_gpio);
    //gpio_request(sensor->irq_gpio, "cwstm,irq-gpio");
    gpio_direction_output(sensor->irq_gpio, 1);
    usleep(20000);
    pr_info("%s:irq gpio vlaue %d\n", __func__, gpio_get_value(sensor->irq_gpio));
    gpio_direction_input(sensor->irq_gpio);
    usleep(5000);

    wake_lock_init(&psensor_wake_lock, WAKE_LOCK_SUSPEND, "psensor_wake_lock");
    wake_lock_init(&fwupdate_wake_lock, WAKE_LOCK_SUSPEND, "cwmcu_fwupdate_wake_lock");

    if (sensor->client->irq > 0) {

        error = request_threaded_irq(sensor->client->irq, NULL,
                           CWMCU_interrupt_thread,
                           IRQF_TRIGGER_FALLING | IRQF_ONESHOT,
                           "cwmcu", sensor);
        if (error < 0) {
                pr_err("request irq %d failed\n", sensor->client->irq);
                goto exit_destroy_mutex;
        }
        disable_irq(client->irq);
        INIT_WORK(&sensor->work, cwmcu_work_report);
        enable_irq(client->irq);
        enable_irq_wake(sensor->client->irq);
    }
#endif

#ifdef CONFIG_FB
    sensor->fb_notif.notifier_call = fb_notifier_callback;
    ret = fb_register_client(&sensor->fb_notif);
    if (ret)
        dev_err(&sensor->client->dev, "CWMCU:Unable to register fb_notifier: %d\n", ret);
#endif

    //i2c_set_clientdata(client, sensor);
    //pm_runtime_enable(&client->dev);
    usleep(10000);

    if (cwmcu_get_version())
       sensor->mcu_status = CW_INVAILD;
    else {
        sensor->mcu_status = CW_VAILD;
        get_mag_vendor();
    }

    sensor->sns_debug_flag = 0;
    sensor->sns_debug_log_list = DEBUG_SENSOR_TYPE;
#ifdef CONFIG_SNS_HUB_DEBUG
    sensor->power_test_on = false;
#endif
    sensor->is_suspend = false;
#ifdef CONFIG_SNR_HUB_LG_HACK
    sensor->light_keeped = false;
#endif
    cwmcu_init_timer();
    printk("--CWMCU-- CWMCU_i2c_probe success!\n");

    return 0;

exit_free_input:
    input_free_device(sensor->input);
err_parse_dt:
#ifdef CWMCU_INTERRUPT
exit_destroy_mutex:
    free_irq(sensor->irq_gpio, sensor);
#endif    
    kfree(sensor);
    return error;
}

static int CWMCU_i2c_remove(struct i2c_client *client)
{
    struct CWMCU_data *sensor = i2c_get_clientdata(client);
    wake_lock_destroy(&psensor_wake_lock);
    wake_lock_destroy(&fwupdate_wake_lock);
    kfree(sensor);
    return 0;
}

static struct of_device_id cwstm_match_table[] = {
    { .compatible = "cwstm,cwstm32",},
    { },
};

static const struct dev_pm_ops CWMCU_pm_ops = {
    .suspend = CWMCU_suspend,
    .resume = CWMCU_resume
};

static const struct i2c_device_id CWMCU_id[] = {
    { CWMCU_I2C_NAME, 0 },
    { }
};

MODULE_DEVICE_TABLE(i2c, CWMCU_id);

static struct i2c_driver CWMCU_driver = {
    .driver = {
        .name = CWMCU_I2C_NAME,
        .owner = THIS_MODULE,
        .pm = &CWMCU_pm_ops,
        .of_match_table = cwstm_match_table,
    },
    .probe    = CWMCU_i2c_probe,
    .remove   = CWMCU_i2c_remove,
    .id_table = CWMCU_id,
};

static int __init CWMCU_i2c_init(void){
    printk(KERN_DEBUG "CWMCU_i2c_init\n");
    return i2c_add_driver(&CWMCU_driver);
}

static void __exit CWMCU_i2c_exit(void){
    i2c_del_driver(&CWMCU_driver);
    cleanup_my_timer();
}

module_init(CWMCU_i2c_init);
module_exit(CWMCU_i2c_exit);

MODULE_DESCRIPTION("CWMCU I2C Bus Driver");
MODULE_AUTHOR("CyWee Group Ltd.");
MODULE_LICENSE("GPL");
