/*  drivers/input/misc/lpc5400x_update.c
*
*  Copyright (C) 2014 NXP semiconductors
*
*  Helper functions parts of LPC Sensor Hub driver to update firmware.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
*/
#define DEBUG
#include <linux/crc32.h>
#include <linux/i2c.h>
#include <linux/device.h>
#include <linux/firmware.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include "lpc5400x_update.h"

#ifndef GPIO_SENSORHUB
#define GPIO_SENSORHUB 49 /* TEGRA_GPIO_PX1, should align with board info */
#endif

#define FWUPDATE_TRANSFER_RETRY_MAX 5
#define BUS_PROBE_TIMEOUT_MS (5000)
#define BUS_COMMAND_TIMEOUT_MS (1000)

#define SL_FLASH_BLOCK_SZ        0x200    /*!< Block size */
#define SL_FLASH_SECT_SZ        0x8000    /*!< Size of a flash sector */
#define SL_FLASH_END            0x80000    /*!< Total size of the flash. */
#define SL_FLASH_PAGE_SZ        0x100    /*!< Size of the flash page. */

#define SL_MAX_I2C_TRANSPORT_SIZE (128)

/* Address of where the boot-able application is located in FLASH */
#define SL_BOOTAPP_ADDR                0x00008000
#define SL_BOOTAPP_START_BLOCK        (SL_BOOTAPP_ADDR/SL_FLASH_BLOCK_SZ)
#define SL_BOOTAPP_START_PAGE        (SL_BOOTAPP_ADDR/SL_FLASH_PAGE_SZ)

/* Offset in application where the IMG_HEADER_T is stored */
#define SL_BOOTAPP_IMGHDR_OFS        0x100
#define IMG_HEADER_MARKER            0xFEEDA5A5

/*!< Pin PIO0_11 has SPI0_SCK (1) function. */
#define PIN_SPI0_SCK_1        ((0 << 5) | 11)
/*!< Pin PIO1_3 has    SPI0_SCK (5) function. */
#define PIN_SPI0_SCK_2        ((1 << 5) | 3)
/*!< Pin PIO0_12 has SPI0_MOSI (1) function. */
#define PIN_SPI0_MOSI_1        ((0 << 5) | 12)
/*!< Pin PIO1_9 has    SPI0_MOSI (2) function. */
#define PIN_SPI0_MOSI_2        ((1 << 5) | 9)
/*!< Pin PIO0_13 has SPI0_MISO (1) function. */
#define PIN_SPI0_MISO_1        ((0 << 5) | 13)
/*!< Pin PIO1_4 has    SPI0_MISO (5) function. */
#define PIN_SPI0_MISO_2        ((1 << 5) | 4)
/*!< Pin PIO0_0 has    SPI0_SSEL0 (2) function. */
#define PIN_SPI0_SEL0_1        ((0 << 5) | 0)
/*!< Pin PIO0_9 has    SPI0_SSEL0 (5) function. */
#define PIN_SPI0_SEL0_2        ((0 << 5) | 9)
/*!< Pin PIO0_14 has SPI0_SSEL0 (1) function. */
#define PIN_SPI0_SEL0_3        ((0 << 5) | 14)

/*!< Pin PIO1_6 has    SPI1_SCK (2) function. */
#define PIN_SPI1_SCK_1        ((1 << 5) | 6)
/*!< Pin PIO1_12 has SPI1_SCK (4) function. */
#define PIN_SPI1_SCK_2        ((1 << 5) | 12)
/*!< Pin PIO1_7 has SPI1_MOSI (2) function. */
#define PIN_SPI1_MOSI_1        ((1 << 5) | 7)
/*!< Pin PIO1_13 has SPI1_MOSI (4) function. */
#define PIN_SPI1_MOSI_2        ((1 << 5) | 13)
/*!< Pin PIO1_8 has    SPI1_MISO (2) function. */
#define PIN_SPI1_MISO_1        ((1 << 5) | 8)
/*!< Pin PIO1_14 has SPI1_MISO (4) function. */
#define PIN_SPI1_MISO_2        ((1 << 5) | 14)
/*!< Pin PIO1_5 has    SPI1_SSEL0 (2) function. */
#define PIN_SPI1_SEL0_1        ((1 << 5) | 5)
/*!< Pin PIO1_15 has SPI1_SSEL0 (4) function. */
#define PIN_SPI1_SEL0_2        ((1 << 5) | 15)
#define SH_RESP_HDR_SOP            0x55

/* Bootloader specific commands */
#define SH_CMD_WHO_AM_I                0xA0
#define SH_CMD_GET_VERSION            0xA1
#define SH_CMD_RESET                0xA2
 /*!< Command to boot the flashed image */
#define SH_CMD_BOOT                    0xA3
/*!< Command to check image integrity.
    Return 0 if matches else computed CRC32 value
    (not include CRC field in mage).
*/
#define SH_CMD_CHECK_IMAGE            0xA4
#define SH_CMD_PROBE                0xA5
/*!< Command to write a block.
    Each block is 512 bytes.
    Check CmdRWBlockParam_t for message format.
*/
#define SH_CMD_WRITE_BLOCK            0xA6
/*!< Command to Read a block.
    Each block is 512 bytes.
    Check CmdRWBlockParam_t for message format.
*/
#define SH_CMD_READ_BLOCK            0xA7
#define SH_CMD_SECTOR_ERASE            0xA8
#define SH_CMD_PAGE_ERASE            0xA9
#define SH_CMD_PAGE_WRITE            0xAA
#define SH_CMD_PAGE_READ            0xAB
#define SH_CMD_WRITE_SUB_BLOCK        0xAC


#define SH_RW_PARAM_SKIP_CRC 1

/** Structure describing response packet format. */
struct _CmdResponse_t {
    /*!< Start of packet = 0x55 for boatloader */
    uint8_t sop;
    /*!< Response to the Command ID. For notification use 0xFF. */
    uint8_t cmd;
    /*!< Response data length not including the header. */
    uint16_t length;
};

/** Structure describing Read/Write block command packet format. */
struct _CmdRWBlockParam_t {
    uint8_t cmd; /*!< Command ID */
    /*!< specifies if we need to do CRC check before processing */
    uint8_t crc_check;
    /*!< Block/page number starting from 0x8000 offset.*/
    uint16_t block_page_nr;
    uint32_t data[SL_FLASH_BLOCK_SZ / 4]; /*!< Data */
    uint32_t crc32; /*!< CRC32 of command header and data */
};

/** Structure decribring Read/Write block command packet format. */
struct _CmdRWSubblockParam_t{
    uint8_t cmd;                            /*!< Command ID */
    uint8_t crc_check;                        /*!< specifies if we need to do CRC check before processing */
    uint16_t block_page_nr;                        /*!< Block number.*/
    uint32_t data[SL_MAX_I2C_TRANSPORT_SIZE/4];        /*!< Data */
    uint32_t crc32;                            /*!< CRC32 of command header and data */
};

/** Structure describing response packet with data. */
struct _CmdDataResp_t {
    struct _CmdResponse_t hdr; /*!< Response header. */
    uint32_t data[SL_FLASH_BLOCK_SZ / 4]; /*!< Data */
    uint32_t crc32; /*!< CRC32 of response packet. */
};

static uint8_t update_sendbuf[SL_FLASH_BLOCK_SZ * 2];
static int muc_irq;

static int SLBoot_xfer(struct i2c_client *client, u8 *txbuffer, int txlength,
        u8 *rxbuffer, int rxlength) {
    struct i2c_msg msg[2];
    int num = 0;

    if (txbuffer && txlength > 0) {
        msg[num].addr = client->addr;
        msg[num].flags = 0;
        msg[num].len = txlength;
        msg[num].buf = txbuffer;
        num++;
    }

    if (rxbuffer && rxlength > 0) {
        msg[num].addr = client->addr;
        msg[num].flags = I2C_M_RD;
        msg[num].len = rxlength;
        msg[num].buf = rxbuffer;
        num++;
    }

    return i2c_transfer(client->adapter, msg, num);
}

static int SLBoot_send_cmd(struct i2c_client *client, u8 *txbuffer,
        int txlength, u8 *rxbuffer, int rxlength) {
    unsigned long orig_jiffies;
    int ret = 0;

    if (!txbuffer || !rxbuffer) {
        dev_warn(&client->dev, "SLBoot_send_cmd, wrong parameter");
        return -EINVAL;
    }

    if ((txbuffer[0] != SH_CMD_WRITE_BLOCK) && (txbuffer[0] != SH_CMD_WRITE_SUB_BLOCK))
        dev_info(&client->dev, "SLBoot_send_cmd: 0x%x", txbuffer[0]);

    ret = SLBoot_xfer(client, txbuffer, txlength, NULL, 0);
    if (ret < 0) {
        dev_err(&client->dev, "SLBoot_send_cmd I2C error %d", ret);
        return ret;
    }

    orig_jiffies = jiffies;

    while (gpio_get_value_cansleep(muc_irq) == 1) {
        if (time_after(jiffies,
                orig_jiffies +
                msecs_to_jiffies(BUS_COMMAND_TIMEOUT_MS))) {
            dev_warn(&client->dev, "SLBoot_send_cmd, timeout");
            ret = -ETIMEDOUT;
            return ret;
        }
    }

    ret = SLBoot_xfer(client, NULL, 0, rxbuffer, rxlength);
    if (ret < 0) {
        dev_err(&client->dev, "SLBoot_send_cmd I2C error %d", ret);
        return ret;
    }

    return 0;
}

static bool lpc5400x_checkImage(struct i2c_client *client)
{
    uint32_t response[2];
    uint8_t cmd = SH_CMD_CHECK_IMAGE;
    int ret;
    struct _CmdResponse_t *pRhdr;

    ret = SLBoot_send_cmd(client, &cmd, 1, (uint8_t *) &response, 8);

    dev_dbg(&client->dev, "lpc5400x_checkImage: ret = %d, response: 0x%x, 0x%x",
            ret, response[0], response[1]);
    if (ret == 0) {
        pRhdr = (struct _CmdResponse_t *) &response[0];
        if ((pRhdr->sop == SH_RESP_HDR_SOP) && (pRhdr->cmd == cmd)
                && (response[1] == 0)) {
            return true;
        }
    }

    return false;
}

static bool lpc5400x_bootFirmware(struct i2c_client *client)
{
    uint32_t response;
    uint8_t cmd = SH_CMD_BOOT;
    int ret;

    ret = SLBoot_send_cmd(client, &cmd, 1, (uint8_t *) &response, 4);
    if (ret == 0)
        return true;

    return false;
}

bool lpc5400x_updateFirmware(struct i2c_client *client, char *fw_name)
{
    const struct firmware *fw_entry;
    size_t offset = 0;
    int ret, sendsize = 0, blocknum = 0, retry = 0, trannum = 0;
    bool bret = true;
    unsigned long crc = 0/* crc32(~0, NULL, 0) */;
    struct _CmdRWSubblockParam_t *datablock =
            (struct _CmdRWSubblockParam_t *) &update_sendbuf;
    uint32_t response[2];

    dev_info(&client->dev, "device name: %s", dev_name(&client->dev));
    ret = request_firmware(&fw_entry, fw_name, &client->dev);
    dev_dbg(&client->dev, "request firmware return %d", ret);
    if (ret == 0) {
        dev_dbg(&client->dev, "request_firmware firmwork size = %d",
                fw_entry->size);
        dev_dbg(&client->dev, "data: %X, %X, %X, %X",
                fw_entry->data[0], fw_entry->data[1],
                fw_entry->data[2], fw_entry->data[3]);
        dev_dbg(&client->dev, "last_data: %X, %X, %X, %X",
                fw_entry->data[fw_entry->size - 4],
                fw_entry->data[fw_entry->size - 3],
                fw_entry->data[fw_entry->size - 2],
                fw_entry->data[fw_entry->size - 1]);

        while (offset < fw_entry->size) {
            sendsize = fw_entry->size - offset;
            if (sendsize > SL_MAX_I2C_TRANSPORT_SIZE)
                sendsize = SL_MAX_I2C_TRANSPORT_SIZE;

            if(trannum == SL_FLASH_BLOCK_SZ/SL_MAX_I2C_TRANSPORT_SIZE)
            {
                trannum = 0;
                blocknum++;
            }

            datablock->cmd = SH_CMD_WRITE_SUB_BLOCK;
            // Each byte of empty block is initialized with 0xFF when shipped from nxp.
            memset(&datablock->data[0], 0xFF, SL_MAX_I2C_TRANSPORT_SIZE);
            memcpy(&datablock->data[0],
                    &fw_entry->data[offset], sendsize);
            datablock->crc_check = 0; /* CRC is checked */
            datablock->block_page_nr =
                blocknum + SL_BOOTAPP_START_BLOCK;

            /*
            * block_page_nr with highest bit = 1 per 4 blocks and the lastblock
            */
            /*
            if((trannum == (SL_FLASH_BLOCK_SZ/SL_MAX_I2C_TRANSPORT_SIZE - 1))
                || (offset + SL_MAX_I2C_TRANSPORT_SIZE >= fw_entry->size))
            {
                datablock->block_page_nr |= 0x8000;
            }
            */

            crc = crc32(0xFFFFFFFF,
                    (unsigned char const *) datablock,
                    SL_MAX_I2C_TRANSPORT_SIZE + 4);
            crc ^= 0xFFFFFFFF;
            datablock->crc32 = crc;
            ret = SLBoot_send_cmd(client, (uint8_t *) datablock,
                    sizeof(struct _CmdRWSubblockParam_t),
                    (uint8_t *) &response, 8);

            /* Need to check the ret and response from Niobe */
            if (ret < 0) {
                if (retry == FWUPDATE_TRANSFER_RETRY_MAX) {
                    dev_warn(&client->dev,
                            "Failed to upgrade firmware, I2C error");
                    bret = false;
                    break;
                } else {
                    retry++;
                    continue;
                }
            }
            offset += sendsize;
            retry = 0;
            trannum++;
        }

        if (bret) {
            bret = lpc5400x_checkImage(client);
            if (bret)
                lpc5400x_bootFirmware(client);
            else
                dev_warn(&client->dev, "Check firmware image failed");
        }

        release_firmware(fw_entry);
    } else {
        bret = false;
        dev_warn(&client->dev, "Failed to get firmware image from vendor path");
    }

    return bret;
}

bool lpc5400x_probebus(struct i2c_client *client, enum _SL_IFSEL_T busport, int irq_gpio)
{
    int val = 1;
    unsigned long orig_jiffies;
    uint8_t bus_probe_cmd[8] = { 0xA5, /* auto detect command */
    /* hostIRQ_pin: P0_30 used as host IRQ line on LPCXpresso board */
    0x55, (0 << 5) | 30,
    PIN_SPI0_MISO_2, /* MISO_pin: P1_4 */
    PIN_SPI0_MOSI_1, /* MOSI_pin: P0_12 */
    PIN_SPI0_SEL0_3, /* SEL_pin: P0_14 */
    PIN_SPI1_MISO_2, /* SCK_pin: P1_3 */
    0x55, };
    uint8_t rx_buff[8];
    bool ret = false;

    muc_irq = irq_gpio;

    bus_probe_cmd[1] = busport;
    /* XOR pin information */
    bus_probe_cmd[7] =
            bus_probe_cmd[0] ^ bus_probe_cmd[1] ^ bus_probe_cmd[2]
            ^ bus_probe_cmd[3] ^ bus_probe_cmd[4] ^ bus_probe_cmd[5]
            ^ bus_probe_cmd[6];

    orig_jiffies = jiffies;    
/*    
    val = gpio_get_value_cansleep(muc_irq);
    printk("%s: mcu irq %d, value = %d\n", __func__, muc_irq, val);
    dev_dbg(&client->dev,
            "Bus Probe: gpio_get_value_cansleep return %d", val);
    if (val == 0)
        return false;
*/
    while (val == 1) {
        /* send FW update command in loop */
        SLBoot_xfer(client, &bus_probe_cmd[0], 8, 0, 0);
        msleep(20);
        if (time_after(jiffies,
                orig_jiffies +
                msecs_to_jiffies(BUS_PROBE_TIMEOUT_MS)))
            break;
        val = gpio_get_value_cansleep(muc_irq);
        dev_dbg(&client->dev,
                "Bus Probe: gpio_get_value_cansleep return %d", val);
    }

    dev_dbg(&client->dev,
            "Bus Probe: gpio_get_value_cansleep2 return %d", val);

    if (val == 0) {
        SLBoot_xfer(client, NULL, 0, &rx_buff[0], 4);
        dev_dbg(&client->dev, "Bus Probe response: 0x%x, 0x%x, 0x%x, 0x%x",
                rx_buff[0], rx_buff[1], rx_buff[2], rx_buff[3]);
        ret = true;
    }

    return ret;
}

