/*
** $Id: //Department/DaVinci/BRANCHES/MT6620_WIFI_DRIVER_V2_3/mgmt/scan_fsm.c#2 $
*/

/*! \file   "scan_fsm.c"
    \brief  This file defines the state transition function for SCAN FSM.

    The SCAN FSM is part of SCAN MODULE and responsible for performing basic SCAN
    behavior as metioned in IEEE 802.11 2007 11.1.3.1 & 11.1.3.2 .
*/



/*
** $Log: scan_fsm.c $
**
** 08 15 2013 cp.wu
** [BORA00002253] [MT6630 Wi-Fi][Driver][Firmware] Add NLO and timeout mechanism to SCN module
** enlarge  match_ssid_num to 16 for PNO support
**
** 08 09 2013 cp.wu
** [BORA00002253] [MT6630 Wi-Fi][Driver][Firmware] Add NLO and timeout mechanism to SCN module
** 1. integrate scheduled scan functionality
** 2. condition compilation for linux-3.4 & linux-3.8 compatibility
** 3. correct CMD queue access to reduce lock scope
**
** 02 19 2013 cp.wu
** [BORA00002227] [MT6630 Wi-Fi][Driver] Update for Makefile and HIFSYS modifications
** take use of GET_BSS_INFO_BY_INDEX() and MAX_BSS_INDEX macros
** for correctly indexing of BSS-INFO pointers
**
** 02 05 2013 cp.wu
** [BORA00002227] [MT6630 Wi-Fi][Driver] Update for Makefile and HIFSYS modifications
** 1. change to use long format (FT=1) for initial command
** 2. fix a typo
**
** 01 22 2013 cp.wu
** [BORA00002253] [MT6630 Wi-Fi][Driver][Firmware] Add NLO and timeout mechanism to SCN module
** .add driver side NLO state machine
**
** 01 22 2013 cp.wu
** [BORA00002253] [MT6630 Wi-Fi][Driver][Firmware] Add NLO and timeout mechanism to SCN module
** modification for ucBssIndex migration
**
** 01 03 2013 cp.wu
** [BORA00002253] [MT6630 Wi-Fi][Driver][Firmware] Add NLO and timeout mechanism to SCN module
** carry timeout value and channel dwell time value to scan module
** 
** 09 17 2012 cm.chang
** [BORA00002149] [MT6630 Wi-Fi] Initial software development
** Duplicate source from MT6620 v2.3 driver branch
** (Davinci label: MT6620_WIFI_Driver_V2_3_120913_1942_As_MT6630_Base)
** 
** 08 31 2012 cp.wu
** [WCXRP00001269] [MT6620 Wi-Fi][Driver] cfg80211 porting merge back to DaVinci
** do not use fgIsP2PRegistered for checking but use network index
 *
 * 06 13 2012 yuche.tsai
 * NULL
 * Update maintrunk driver.
 * Add support for driver compose assoc request frame.
 *
 * 11 24 2011 wh.su
 * [WCXRP00001078] [MT6620 Wi-Fi][Driver] Adding the mediatek log improment support : XLOG
 * Adjust code for DBG and CONFIG_XLOG.
 *
 * 11 14 2011 yuche.tsai
 * [WCXRP00001095] [Volunteer Patch][Driver] Always Scan before enable Hot-Spot.
 * Fix bug when unregister P2P network..
 *
 * 11 11 2011 wh.su
 * [WCXRP00001078] [MT6620 Wi-Fi][Driver] Adding the mediatek log improment support : XLOG
 * modify the xlog related code.
 *
 * 11 02 2011 wh.su
 * [WCXRP00001078] [MT6620 Wi-Fi][Driver] Adding the mediatek log improment support : XLOG
 * adding the code for XLOG.
 *
 * 08 11 2011 cp.wu
 * [WCXRP00000830] [MT6620 Wi-Fi][Firmware] Use MDRDY counter to detect empty channel for shortening scan time
 * sparse channel detection:
 * driver: collect sparse channel information with scan-done event

 *
 * 07 18 2011 cp.wu
 * [WCXRP00000858] [MT5931][Driver][Firmware] Add support for scan to search for more than one SSID in a single scanning request
 * free mailbox message afte parsing is completed.
 *
 * 07 18 2011 cp.wu
 * [WCXRP00000858] [MT5931][Driver][Firmware] Add support for scan to search for more than one SSID in a single scanning request
 * add framework in driver domain for supporting new SCAN_REQ_V2 for more than 1 SSID support as well as uProbeDelay in NDIS 6.x driver model
 *
 * 04 18 2011 terry.wu
 * [WCXRP00000660] [MT6620 Wi-Fi][Driver] Remove flag CFG_WIFI_DIRECT_MOVED
 * Remove flag CFG_WIFI_DIRECT_MOVED.
 *
 * 03 29 2011 cp.wu
 * [WCXRP00000604] [MT6620 Wi-Fi][Driver] Surpress Klockwork Warning
 * surpress klock warning with code path rewritten
 *
 * 03 18 2011 cm.chang
 * [WCXRP00000576] [MT6620 Wi-Fi][Driver][FW] Remove P2P compile option in scan req/cancel command
 * As CR title
 *
 * 02 18 2011 yuche.tsai
 * [WCXRP00000478] [Volunteer Patch][MT6620][Driver] Probe request frame during search phase do not contain P2P wildcard SSID.
 * Take P2P wildcard SSID into consideration.
 *
 * 01 27 2011 yuche.tsai
 * [WCXRP00000399] [Volunteer Patch][MT6620/MT5931][Driver] Fix scan side effect after P2P module separate.
 * Fix scan channel extension issue when p2p module is not registered.
 *
 * 01 26 2011 yuche.tsai
 * [WCXRP00000388] [Volunteer Patch][MT6620][Driver/Fw] change Station Type in station record.
 * .
 *
 * 01 25 2011 yuche.tsai
 * [WCXRP00000388] [Volunteer Patch][MT6620][Driver/Fw] change Station Type in station record.
 * Fix Compile Error when DBG is disabled.
 *
 * 12 07 2010 cm.chang
 * [WCXRP00000238] MT6620 Wi-Fi][Driver][FW] Support regulation domain setting from NVRAM and supplicant
 * 1. Country code is from NVRAM or supplicant
 * 2. Change band definition in CMD/EVENT.
 *
 * 09 03 2010 kevin.huang
 * NULL
 * Refine #include sequence and solve recursive/nested #include issue
 *
 * 08 30 2010 cp.wu
 * NULL
 * eliminate klockwork errors
 *
 * 08 16 2010 cp.wu
 * NULL
 * add interface for RLM to trigger OBSS-SCAN.
 *
 * 08 16 2010 yuche.tsai
 * NULL
 * Fix bug for processing queued scan request.
 *
 * 08 11 2010 yuche.tsai
 * NULL
 * Add a function for returning channel.
 *
 * 08 05 2010 yuche.tsai
 * NULL
 * Update SCAN FSM for support P2P Device discovery scan.
 *
 * 08 03 2010 cp.wu
 * NULL
 * surpress compilation warning.
 *
 * 07 26 2010 yuche.tsai
 *
 * Add option of channel extension while cancelling scan request.
 *
 * 07 21 2010 yuche.tsai
 *
 * Add P2P Scan & Scan Result Parsing & Saving.
 *
 * 07 20 2010 cp.wu
 *
 * pass band information for scan in an efficient way by mapping ENUM_BAND_T into UINT_8..
 *
 * 07 19 2010 cp.wu
 *
 * due to FW/DRV won't be sync. precisely, some strict assertions should be eased.
 *
 * 07 19 2010 cp.wu
 *
 * [WPD00003833] [MT6620 and MT5931] Driver migration.
 * SCN module is now able to handle multiple concurrent scanning requests
 *
 * 07 16 2010 cp.wu
 *
 * [WPD00003833] [MT6620 and MT5931] Driver migration.
 * bugfix for SCN migration
 * 1) modify QUEUE_CONCATENATE_QUEUES() so it could be used to concatence with an empty queue
 * 2) before AIS issues scan request, network(BSS) needs to be activated first
 * 3) only invoke COPY_SSID when using specified SSID for scan
 *
 * 07 15 2010 cp.wu
 *
 * [WPD00003833] [MT6620 and MT5931] Driver migration.
 * driver no longer generates probe request frames
 *
 * 07 14 2010 cp.wu
 *
 * [WPD00003833] [MT6620 and MT5931] Driver migration.
 * pass band with channel number information as scan parameter
 *
 * 07 14 2010 cp.wu
 *
 * [WPD00003833] [MT6620 and MT5931] Driver migration.
 * remove timer in DRV-SCN.
 *
 * 07 09 2010 cp.wu
 *
 * 1) separate AIS_FSM state for two kinds of scanning. (OID triggered scan, and scan-for-connection)
 * 2) eliminate PRE_BSS_DESC_T, Beacon/PrebResp is now parsed in single pass
 * 3) implment DRV-SCN module, currently only accepts single scan request, other request will be directly dropped by returning BUSY
 *
 * 07 08 2010 cp.wu
 *
 * [WPD00003833] [MT6620 and MT5931] Driver migration - move to new repository.
 *
 * 07 08 2010 cp.wu
 * [WPD00003833][MT6620 and MT5931] Driver migration
 * take use of RLM module for parsing/generating HT IEs for 11n capability
 *
 * 07 02 2010 cp.wu
 * [WPD00003833][MT6620 and MT5931] Driver migration
 * when returning to SCAN_IDLE state, send a correct message to source FSM.
 *
 * 07 01 2010 cp.wu
 * [WPD00003833][MT6620 and MT5931] Driver migration
 * implementation of DRV-SCN and related mailbox message handling.
 *
 * 06 22 2010 cp.wu
 * [WPD00003833][MT6620 and MT5931] Driver migration
 * comment out RLM APIs by CFG_RLM_MIGRATION.
 *
 * 06 21 2010 cp.wu
 * [WPD00003833][MT6620 and MT5931] Driver migration
 * add scan_fsm into building.
 *
 * 05 14 2010 kevin.huang
 * [BORA00000794][WIFISYS][New Feature]Power Management Support
 * Refine the order of Stop TX Queue and Switch Channel
 *
 * 05 12 2010 kevin.huang
 * [BORA00000794][WIFISYS][New Feature]Power Management Support
 * Update pause/resume/flush API to new Bitmap API
 *
 * 05 12 2010 kevin.huang
 * [BORA00000794][WIFISYS][New Feature]Power Management Support
 * Add Power Management - Legacy PS-POLL support.
 *
 * 03 18 2010 kevin.huang
 * [BORA00000663][WIFISYS][New Feature] AdHoc Mode Support
 * Ignore the PROBE_DELAY state if the value of Probe Delay == 0
 *
 * 03 10 2010 kevin.huang
 * [BORA00000654][WIFISYS][New Feature] CNM Module - Ch Manager Support
 * Add Channel Manager for arbitration of JOIN and SCAN Req
 *
 * 02 23 2010 kevin.huang
 * [BORA00000603][WIFISYS] [New Feature] AAA Module Support
 * Add support scan channel 1~14 and update scan result's frequency infou1rwduu`wvpghlqg|n`slk+mpdkb
 *
 * 01 08 2010 kevin.huang
 * [BORA00000018]Integrate WIFI part into BORA for the 1st time
 * Add set RX Filter to receive BCN from different BSSID during SCAN
 *
 * 12 18 2009 cm.chang
 * [BORA00000018]Integrate WIFI part into BORA for the 1st time
 * .
 *
 * Nov 25 2009 mtk01461
 * [BORA00000018] Integrate WIFI part into BORA for the 1st time
 * Remove flag of CFG_TEST_MGMT_FSM
 *
 * Nov 20 2009 mtk01461
 * [BORA00000018] Integrate WIFI part into BORA for the 1st time
 * Change parameter of scanSendProbeReqFrames()
 *
 * Nov 16 2009 mtk01461
 * [BORA00000018] Integrate WIFI part into BORA for the 1st time
 * Update scnFsmSteps()
 *
 * Nov 5 2009 mtk01461
 * [BORA00000018] Integrate WIFI part into BORA for the 1st time
 * Fix typo
 *
 * Nov 5 2009 mtk01461
 * [BORA00000018] Integrate WIFI part into BORA for the 1st time
 *
*/

/*******************************************************************************
*                         C O M P I L E R   F L A G S
********************************************************************************
*/

/*******************************************************************************
*                    E X T E R N A L   R E F E R E N C E S
********************************************************************************
*/
#include "precomp.h"

/*******************************************************************************
*                              C O N S T A N T S
********************************************************************************
*/

/*******************************************************************************
*                             D A T A   T Y P E S
********************************************************************************
*/

/*******************************************************************************
*                            P U B L I C   D A T A
********************************************************************************
*/

/*******************************************************************************
*                           P R I V A T E   D A T A
********************************************************************************
*/
#if DBG
/*lint -save -e64 Type mismatch */
static PUINT_8 apucDebugScanState[SCAN_STATE_NUM] = {
    (PUINT_8)DISP_STRING("SCAN_STATE_IDLE"),
    (PUINT_8)DISP_STRING("SCAN_STATE_SCANNING"),
};
/*lint -restore */
#endif /* DBG */

/*******************************************************************************
*                                 M A C R O S
********************************************************************************
*/

/*******************************************************************************
*                   F U N C T I O N   D E C L A R A T I O N S
********************************************************************************
*/

/*******************************************************************************
*                              F U N C T I O N S
********************************************************************************
*/
/*----------------------------------------------------------------------------*/
/*!
* \brief
*
* \param[in]
*
* \return none
*/
/*----------------------------------------------------------------------------*/
VOID
scnFsmSteps (
    IN P_ADAPTER_T prAdapter,
    IN ENUM_SCAN_STATE_T eNextState
    )
{
    P_SCAN_INFO_T prScanInfo;
    P_SCAN_PARAM_T prScanParam;
    P_MSG_HDR_T prMsgHdr;

    BOOLEAN fgIsTransition = (BOOLEAN)FALSE;

    prScanInfo = &(prAdapter->rWifiVar.rScanInfo);
    prScanParam = &prScanInfo->rScanParam;

    do {

#if DBG
        DBGLOG(SCN, STATE, ("TRANSITION: [%s] -> [%s]\n",
                             apucDebugScanState[prScanInfo->eCurrentState],
                             apucDebugScanState[eNextState]));
#else
        DBGLOG(SCN, STATE, ("[%d] TRANSITION: [%d] -> [%d]\n",
                             DBG_SCN_IDX,
                             prScanInfo->eCurrentState,
                             eNextState));
#endif

        /* NOTE(Kevin): This is the only place to change the eCurrentState(except initial) */
        prScanInfo->eCurrentState = eNextState;

        fgIsTransition = (BOOLEAN)FALSE;

        switch (prScanInfo->eCurrentState) {
        case SCAN_STATE_IDLE:
            /* check for pending scanning requests */
            if(!LINK_IS_EMPTY(&(prScanInfo->rPendingMsgList))) {
                // load next message from pending list as scan parameters
                LINK_REMOVE_HEAD(&(prScanInfo->rPendingMsgList), prMsgHdr, P_MSG_HDR_T);

                if(prMsgHdr->eMsgId == MID_AIS_SCN_SCAN_REQ
                        || prMsgHdr->eMsgId == MID_BOW_SCN_SCAN_REQ
                        || prMsgHdr->eMsgId == MID_P2P_SCN_SCAN_REQ
                        || prMsgHdr->eMsgId == MID_RLM_SCN_SCAN_REQ) {
                    scnFsmHandleScanMsg(prAdapter, (P_MSG_SCN_SCAN_REQ)prMsgHdr);

                    eNextState = SCAN_STATE_SCANNING;
                    fgIsTransition = TRUE;
                }
                else if(prMsgHdr->eMsgId == MID_AIS_SCN_SCAN_REQ_V2
                        || prMsgHdr->eMsgId == MID_BOW_SCN_SCAN_REQ_V2
                        || prMsgHdr->eMsgId == MID_P2P_SCN_SCAN_REQ_V2
                        || prMsgHdr->eMsgId == MID_RLM_SCN_SCAN_REQ_V2) {
                    scnFsmHandleScanMsgV2(prAdapter, (P_MSG_SCN_SCAN_REQ_V2)prMsgHdr);

                    eNextState = SCAN_STATE_SCANNING;
                    fgIsTransition = TRUE;
                }
                else {
                    // should not happen
                    ASSERT(0);
                }

                /* switch to next state */
                cnmMemFree(prAdapter, prMsgHdr);
            }
            break;

        case SCAN_STATE_SCANNING:
            if(prScanParam->fgIsScanV2 == FALSE) {
                scnSendScanReq(prAdapter);
            }
            else {
                scnSendScanReqV2(prAdapter);
            }
            break;

        default:
            ASSERT(0);
            break;

        }
    }
    while (fgIsTransition);

    return;
}


/*----------------------------------------------------------------------------*/
/*!
* \brief        Generate CMD_ID_SCAN_REQ command
*
* \param[in]
*
* \return none
*/
/*----------------------------------------------------------------------------*/
VOID
scnSendScanReq (
    IN P_ADAPTER_T prAdapter
    )
{
    P_SCAN_INFO_T prScanInfo;
    P_SCAN_PARAM_T prScanParam;
    CMD_SCAN_REQ rCmdScanReq;
    UINT_32 i;

    ASSERT(prAdapter);

    prScanInfo = &(prAdapter->rWifiVar.rScanInfo);
    prScanParam = &prScanInfo->rScanParam;

    // send command packet for scan
    kalMemZero(&rCmdScanReq, sizeof(CMD_SCAN_REQ));

    rCmdScanReq.ucSeqNum        = prScanParam->ucSeqNum;
    rCmdScanReq.ucBssIndex      = prScanParam->ucBssIndex;
    rCmdScanReq.ucScanType      = (UINT_8)prScanParam->eScanType;
    rCmdScanReq.ucSSIDType      = prScanParam->ucSSIDType;

    if(prScanParam->ucSSIDNum == 1) {
        COPY_SSID(rCmdScanReq.aucSSID,
                rCmdScanReq.ucSSIDLength,
                prScanParam->aucSpecifiedSSID[0],
                prScanParam->ucSpecifiedSSIDLen[0]);
    }

    rCmdScanReq.ucChannelType       = (UINT_8)prScanParam->eScanChannel;

    if (prScanParam->eScanChannel == SCAN_CHANNEL_SPECIFIED) {
        /* P2P would use:
         * 1. Specified Listen Channel of passive scan for LISTEN state.
         * 2. Specified Listen Channel of Target Device of active scan for SEARCH state. (Target != NULL)
         */
        rCmdScanReq.ucChannelListNum    = prScanParam->ucChannelListNum;

        for(i = 0 ; i < rCmdScanReq.ucChannelListNum ; i++) {
            rCmdScanReq.arChannelList[i].ucBand =
                (UINT_8) prScanParam->arChnlInfoList[i].eBand;

            rCmdScanReq.arChannelList[i].ucChannelNum =
                (UINT_8)prScanParam->arChnlInfoList[i].ucChannelNum;
        }
    }

    rCmdScanReq.u2ChannelDwellTime = prScanParam->u2ChannelDwellTime;
    rCmdScanReq.u2TimeoutValue = prScanParam->u2TimeoutValue;

    if(prScanParam->u2IELen <= MAX_IE_LENGTH) {
        rCmdScanReq.u2IELen = prScanParam->u2IELen;
    }
    else {
        rCmdScanReq.u2IELen = MAX_IE_LENGTH;
    }

    if (prScanParam->u2IELen) {
        kalMemCopy(rCmdScanReq.aucIE,
                prScanParam->aucIE,
                sizeof(UINT_8) * rCmdScanReq.u2IELen);
    }

    wlanSendSetQueryCmd(prAdapter,
            CMD_ID_SCAN_REQ,
            TRUE,
            FALSE,
            FALSE,
            NULL,
            NULL,
            OFFSET_OF(CMD_SCAN_REQ, aucIE) + rCmdScanReq.u2IELen,
            (PUINT_8)&rCmdScanReq,
            NULL,
            0);
}


/*----------------------------------------------------------------------------*/
/*!
* \brief        Generate CMD_ID_SCAN_REQ_V2 command
*
* \param[in]
*
* \return none
*/
/*----------------------------------------------------------------------------*/
VOID
scnSendScanReqV2 (
    IN P_ADAPTER_T prAdapter
    )
{
    P_SCAN_INFO_T prScanInfo;
    P_SCAN_PARAM_T prScanParam;
    CMD_SCAN_REQ_V2 rCmdScanReq;
    UINT_32 i;

    ASSERT(prAdapter);

    prScanInfo = &(prAdapter->rWifiVar.rScanInfo);
    prScanParam = &prScanInfo->rScanParam;

    // send command packet for scan
    kalMemZero(&rCmdScanReq, sizeof(CMD_SCAN_REQ_V2));

    rCmdScanReq.ucSeqNum        = prScanParam->ucSeqNum;
    rCmdScanReq.ucBssIndex      = prScanParam->ucBssIndex;
    rCmdScanReq.ucScanType      = (UINT_8)prScanParam->eScanType;
    rCmdScanReq.ucSSIDType      = prScanParam->ucSSIDType;

    for (i = 0 ; i < prScanParam->ucSSIDNum; i++) {
        COPY_SSID(rCmdScanReq.arSSID[i].aucSsid,
                rCmdScanReq.arSSID[i].u4SsidLen,
                prScanParam->aucSpecifiedSSID[i],
                prScanParam->ucSpecifiedSSIDLen[i]);
    }

    rCmdScanReq.u2ProbeDelayTime    = (UINT_8)prScanParam->u2ProbeDelayTime;
    rCmdScanReq.ucChannelType       = (UINT_8)prScanParam->eScanChannel;

    if (prScanParam->eScanChannel == SCAN_CHANNEL_SPECIFIED) {
        /* P2P would use:
         * 1. Specified Listen Channel of passive scan for LISTEN state.
         * 2. Specified Listen Channel of Target Device of active scan for SEARCH state. (Target != NULL)
         */
        rCmdScanReq.ucChannelListNum    = prScanParam->ucChannelListNum;

        for(i = 0 ; i < rCmdScanReq.ucChannelListNum ; i++) {
            rCmdScanReq.arChannelList[i].ucBand =
                (UINT_8) prScanParam->arChnlInfoList[i].eBand;

            rCmdScanReq.arChannelList[i].ucChannelNum =
                (UINT_8)prScanParam->arChnlInfoList[i].ucChannelNum;
        }
    }

    rCmdScanReq.u2ChannelDwellTime = prScanParam->u2ChannelDwellTime;
    rCmdScanReq.u2TimeoutValue = prScanParam->u2TimeoutValue;

    if(prScanParam->u2IELen <= MAX_IE_LENGTH) {
        rCmdScanReq.u2IELen = prScanParam->u2IELen;
    }
    else {
        rCmdScanReq.u2IELen = MAX_IE_LENGTH;
    }

    if (prScanParam->u2IELen) {
        kalMemCopy(rCmdScanReq.aucIE,
                prScanParam->aucIE,
                sizeof(UINT_8) * rCmdScanReq.u2IELen);
    }

    wlanSendSetQueryCmd(prAdapter,
            CMD_ID_SCAN_REQ_V2,
            TRUE,
            FALSE,
            FALSE,
            NULL,
            NULL,
            OFFSET_OF(CMD_SCAN_REQ_V2, aucIE) + rCmdScanReq.u2IELen,
            (PUINT_8)&rCmdScanReq,
            NULL,
            0);

}


/*----------------------------------------------------------------------------*/
/*!
* \brief
*
* \param[in]
*
* \return none
*/
/*----------------------------------------------------------------------------*/
VOID
scnFsmMsgStart (
    IN P_ADAPTER_T prAdapter,
    IN P_MSG_HDR_T prMsgHdr
    )
{
    P_SCAN_INFO_T prScanInfo;
    P_SCAN_PARAM_T prScanParam;

    ASSERT(prMsgHdr);

    prScanInfo = &(prAdapter->rWifiVar.rScanInfo);
    prScanParam = &prScanInfo->rScanParam;


    if (prScanInfo->eCurrentState == SCAN_STATE_IDLE) {
        if(prMsgHdr->eMsgId == MID_AIS_SCN_SCAN_REQ
                || prMsgHdr->eMsgId == MID_BOW_SCN_SCAN_REQ
                || prMsgHdr->eMsgId == MID_P2P_SCN_SCAN_REQ
                || prMsgHdr->eMsgId == MID_RLM_SCN_SCAN_REQ) {
            scnFsmHandleScanMsg(prAdapter, (P_MSG_SCN_SCAN_REQ)prMsgHdr);
        }
        else if(prMsgHdr->eMsgId == MID_AIS_SCN_SCAN_REQ_V2
                || prMsgHdr->eMsgId == MID_BOW_SCN_SCAN_REQ_V2
                || prMsgHdr->eMsgId == MID_P2P_SCN_SCAN_REQ_V2
                || prMsgHdr->eMsgId == MID_RLM_SCN_SCAN_REQ_V2) {
            scnFsmHandleScanMsgV2(prAdapter, (P_MSG_SCN_SCAN_REQ_V2)prMsgHdr);
        }
        else {
            // should not deliver to this function
            ASSERT(0);
        }

        cnmMemFree(prAdapter, prMsgHdr);
        scnFsmSteps(prAdapter, SCAN_STATE_SCANNING);
    }
    else {
        LINK_INSERT_TAIL(&prScanInfo->rPendingMsgList, &prMsgHdr->rLinkEntry);
    }

    return;
}



/*----------------------------------------------------------------------------*/
/*!
* \brief
*
* \param[in]
*
* \return none
*/
/*----------------------------------------------------------------------------*/
VOID
scnFsmMsgAbort (
    IN P_ADAPTER_T prAdapter,
    IN P_MSG_HDR_T prMsgHdr
    )
{
    P_MSG_SCN_SCAN_CANCEL prScanCancel;
    P_SCAN_INFO_T prScanInfo;
    P_SCAN_PARAM_T prScanParam;
    CMD_SCAN_CANCEL rCmdScanCancel;

    ASSERT(prMsgHdr);

    prScanCancel = (P_MSG_SCN_SCAN_CANCEL)prMsgHdr;
    prScanInfo = &(prAdapter->rWifiVar.rScanInfo);
    prScanParam = &prScanInfo->rScanParam;

    if (prScanInfo->eCurrentState != SCAN_STATE_IDLE) {
        if(prScanCancel->ucSeqNum == prScanParam->ucSeqNum &&
                prScanCancel->ucBssIndex == prScanParam->ucBssIndex) {
            /* send cancel message to firmware domain */
            rCmdScanCancel.ucSeqNum = prScanParam->ucSeqNum;
            rCmdScanCancel.ucIsExtChannel = (UINT_8) prScanCancel->fgIsChannelExt;

            wlanSendSetQueryCmd(prAdapter,
                    CMD_ID_SCAN_CANCEL,
                    TRUE,
                    FALSE,
                    FALSE,
                    NULL,
                    NULL,
                    sizeof(CMD_SCAN_CANCEL),
                    (PUINT_8)&rCmdScanCancel,
                    NULL,
                    0);

            /* generate scan-done event for caller */
            scnFsmGenerateScanDoneMsg(prAdapter,
                    prScanParam->ucSeqNum,
                    prScanParam->ucBssIndex,
                    SCAN_STATUS_CANCELLED);

            /* switch to next pending scan */
            scnFsmSteps(prAdapter, SCAN_STATE_IDLE);
        }
        else {
            scnFsmRemovePendingMsg(prAdapter, prScanCancel->ucSeqNum, prScanCancel->ucBssIndex);
        }
    }

    cnmMemFree(prAdapter, prMsgHdr);

    return;
}


/*----------------------------------------------------------------------------*/
/*!
* \brief            Scan Message Parsing (Legacy)
*
* \param[in]
*
* \return none
*/
/*----------------------------------------------------------------------------*/
VOID
scnFsmHandleScanMsg (
    IN P_ADAPTER_T prAdapter,
    IN P_MSG_SCN_SCAN_REQ prScanReqMsg
    )
{
    P_SCAN_INFO_T prScanInfo;
    P_SCAN_PARAM_T prScanParam;
    UINT_32 i;

    ASSERT(prAdapter);
    ASSERT(prScanReqMsg);

    prScanInfo = &(prAdapter->rWifiVar.rScanInfo);
    prScanParam = &prScanInfo->rScanParam;

    prScanParam->eScanType      = prScanReqMsg->eScanType;
    prScanParam->ucBssIndex     = prScanReqMsg->ucBssIndex;
    prScanParam->ucSSIDType     = prScanReqMsg->ucSSIDType;
    if (prScanParam->ucSSIDType & (SCAN_REQ_SSID_SPECIFIED | SCAN_REQ_SSID_P2P_WILDCARD)) {
        prScanParam->ucSSIDNum = 1;

        COPY_SSID(prScanParam->aucSpecifiedSSID[0],
                prScanParam->ucSpecifiedSSIDLen[0],
                prScanReqMsg->aucSSID,
                prScanReqMsg->ucSSIDLength);

        // reset SSID length to zero for rest array entries
        for(i = 1 ; i < SCN_SSID_MAX_NUM ; i++) {
            prScanParam->ucSpecifiedSSIDLen[i] = 0;
        }
    }
    else {
        prScanParam->ucSSIDNum = 0;

        for(i = 0 ; i < SCN_SSID_MAX_NUM ; i++) {
            prScanParam->ucSpecifiedSSIDLen[i] = 0;
        }
    }

    prScanParam->u2ProbeDelayTime   = 0;
    prScanParam->eScanChannel   = prScanReqMsg->eScanChannel;
    if(prScanParam->eScanChannel == SCAN_CHANNEL_SPECIFIED) {
        if(prScanReqMsg->ucChannelListNum <= MAXIMUM_OPERATION_CHANNEL_LIST) {
            prScanParam->ucChannelListNum   = prScanReqMsg->ucChannelListNum;
        }
        else {
            prScanParam->ucChannelListNum   = MAXIMUM_OPERATION_CHANNEL_LIST;
        }

        kalMemCopy(prScanParam->arChnlInfoList,
                prScanReqMsg->arChnlInfoList,
                sizeof(RF_CHANNEL_INFO_T) * prScanParam->ucChannelListNum);
    }

    if(prScanReqMsg->u2IELen <= MAX_IE_LENGTH) {
        prScanParam->u2IELen    = prScanReqMsg->u2IELen;
    }
    else {
        prScanParam->u2IELen    = MAX_IE_LENGTH;
    }

    if(prScanParam->u2IELen) {
        kalMemCopy(prScanParam->aucIE, prScanReqMsg->aucIE, prScanParam->u2IELen);
    }

    prScanParam->u2ChannelDwellTime = prScanReqMsg->u2ChannelDwellTime;
    prScanParam->u2TimeoutValue = prScanReqMsg->u2TimeoutValue;
    prScanParam->ucSeqNum       = prScanReqMsg->ucSeqNum;

    if(prScanReqMsg->rMsgHdr.eMsgId == MID_RLM_SCN_SCAN_REQ) {
        prScanParam->fgIsObssScan   = TRUE;
    }
    else {
        prScanParam->fgIsObssScan   = FALSE;
    }

    prScanParam->fgIsScanV2 = FALSE;

    return;
}


/*----------------------------------------------------------------------------*/
/*!
* \brief            Scan Message Parsing - V2 with multiple SSID support
*
* \param[in]
*
* \return none
*/
/*----------------------------------------------------------------------------*/
VOID
scnFsmHandleScanMsgV2 (
    IN P_ADAPTER_T prAdapter,
    IN P_MSG_SCN_SCAN_REQ_V2 prScanReqMsg
    )
{
    P_SCAN_INFO_T prScanInfo;
    P_SCAN_PARAM_T prScanParam;
    UINT_32 i;

    ASSERT(prAdapter);
    ASSERT(prScanReqMsg);
    ASSERT(prScanReqMsg->ucSSIDNum <= SCN_SSID_MAX_NUM);

    prScanInfo = &(prAdapter->rWifiVar.rScanInfo);
    prScanParam = &prScanInfo->rScanParam;

    prScanParam->eScanType      = prScanReqMsg->eScanType;
    prScanParam->ucBssIndex     = prScanReqMsg->ucBssIndex;
    prScanParam->ucSSIDType     = prScanReqMsg->ucSSIDType;
    prScanParam->ucSSIDNum      = prScanReqMsg->ucSSIDNum;

    for(i = 0 ; i < prScanReqMsg->ucSSIDNum ; i++) {
        COPY_SSID(prScanParam->aucSpecifiedSSID[i],
                prScanParam->ucSpecifiedSSIDLen[i],
                prScanReqMsg->prSsid[i].aucSsid,
                (UINT_8)prScanReqMsg->prSsid[i].u4SsidLen);
    }

    prScanParam->u2ProbeDelayTime   = prScanReqMsg->u2ProbeDelay;
    prScanParam->eScanChannel       = prScanReqMsg->eScanChannel;
    if(prScanParam->eScanChannel == SCAN_CHANNEL_SPECIFIED) {
        if(prScanReqMsg->ucChannelListNum <= MAXIMUM_OPERATION_CHANNEL_LIST) {
            prScanParam->ucChannelListNum   = prScanReqMsg->ucChannelListNum;
        }
        else {
            prScanParam->ucChannelListNum   = MAXIMUM_OPERATION_CHANNEL_LIST;
        }

        kalMemCopy(prScanParam->arChnlInfoList,
                prScanReqMsg->arChnlInfoList,
                sizeof(RF_CHANNEL_INFO_T) * prScanParam->ucChannelListNum);
    }

    if(prScanReqMsg->u2IELen <= MAX_IE_LENGTH) {
        prScanParam->u2IELen    = prScanReqMsg->u2IELen;
    }
    else {
        prScanParam->u2IELen    = MAX_IE_LENGTH;
    }

    if(prScanParam->u2IELen) {
        kalMemCopy(prScanParam->aucIE, prScanReqMsg->aucIE, prScanParam->u2IELen);
    }

    prScanParam->u2ChannelDwellTime = prScanReqMsg->u2ChannelDwellTime;
    prScanParam->u2TimeoutValue = prScanReqMsg->u2TimeoutValue;
    prScanParam->ucSeqNum       = prScanReqMsg->ucSeqNum;

    if(prScanReqMsg->rMsgHdr.eMsgId == MID_RLM_SCN_SCAN_REQ) {
        prScanParam->fgIsObssScan   = TRUE;
    }
    else {
        prScanParam->fgIsObssScan   = FALSE;
    }

    prScanParam->fgIsScanV2 = TRUE;

    return;
}


/*----------------------------------------------------------------------------*/
/*!
* \brief            Remove pending scan request
*
* \param[in]
*
* \return none
*/
/*----------------------------------------------------------------------------*/
VOID
scnFsmRemovePendingMsg (
    IN P_ADAPTER_T  prAdapter,
    IN UINT_8       ucSeqNum,
    IN UINT_8       ucBssIndex
    )
{
    P_SCAN_INFO_T prScanInfo;
    P_SCAN_PARAM_T prScanParam;
    P_MSG_HDR_T prPendingMsgHdr, prPendingMsgHdrNext, prRemoveMsgHdr = NULL;
    P_LINK_ENTRY_T prRemoveLinkEntry = NULL;
    BOOLEAN fgIsRemovingScan = FALSE;

    ASSERT(prAdapter);

    prScanInfo = &(prAdapter->rWifiVar.rScanInfo);
    prScanParam = &prScanInfo->rScanParam;

    /* traverse through rPendingMsgList for removal */
    LINK_FOR_EACH_ENTRY_SAFE(prPendingMsgHdr,
            prPendingMsgHdrNext,
            &(prScanInfo->rPendingMsgList),
            rLinkEntry,
            MSG_HDR_T) {
        if(prPendingMsgHdr->eMsgId == MID_AIS_SCN_SCAN_REQ
                || prPendingMsgHdr->eMsgId == MID_BOW_SCN_SCAN_REQ
                || prPendingMsgHdr->eMsgId == MID_P2P_SCN_SCAN_REQ
                || prPendingMsgHdr->eMsgId == MID_RLM_SCN_SCAN_REQ) {
            P_MSG_SCN_SCAN_REQ prScanReqMsg = (P_MSG_SCN_SCAN_REQ)prPendingMsgHdr;

            if(ucSeqNum == prScanReqMsg->ucSeqNum &&
                    ucBssIndex == prScanReqMsg->ucBssIndex) {
                prRemoveLinkEntry = &(prScanReqMsg->rMsgHdr.rLinkEntry);
                prRemoveMsgHdr = prPendingMsgHdr;
                fgIsRemovingScan = TRUE;
            }
        }
        else if(prPendingMsgHdr->eMsgId == MID_AIS_SCN_SCAN_REQ_V2
                || prPendingMsgHdr->eMsgId == MID_BOW_SCN_SCAN_REQ_V2
                || prPendingMsgHdr->eMsgId == MID_P2P_SCN_SCAN_REQ_V2
                || prPendingMsgHdr->eMsgId == MID_RLM_SCN_SCAN_REQ_V2) {
            P_MSG_SCN_SCAN_REQ_V2 prScanReqMsgV2 = (P_MSG_SCN_SCAN_REQ_V2)prPendingMsgHdr;

            if(ucSeqNum == prScanReqMsgV2->ucSeqNum &&
                    ucBssIndex == prScanReqMsgV2->ucBssIndex) {
                prRemoveLinkEntry = &(prScanReqMsgV2->rMsgHdr.rLinkEntry);
                prRemoveMsgHdr = prPendingMsgHdr;
                fgIsRemovingScan = TRUE;
            }
        }

        if(prRemoveLinkEntry) {
            if(fgIsRemovingScan == TRUE) {
                /* generate scan-done event for caller */
                scnFsmGenerateScanDoneMsg(prAdapter,
                        ucSeqNum,
                        ucBssIndex,
                        SCAN_STATUS_CANCELLED);
            }

            /* remove from pending list */
            LINK_REMOVE_KNOWN_ENTRY(&(prScanInfo->rPendingMsgList), prRemoveLinkEntry);
            cnmMemFree(prAdapter, prRemoveMsgHdr);

            break;
        }
    }

    return;
}


/*----------------------------------------------------------------------------*/
/*!
* \brief
*
* \param[in]
*
* \return none
*/
/*----------------------------------------------------------------------------*/
VOID
scnEventScanDone (
    IN P_ADAPTER_T          prAdapter,
    IN P_EVENT_SCAN_DONE    prScanDone
    )
{
    P_SCAN_INFO_T prScanInfo;
    P_SCAN_PARAM_T prScanParam;

    prScanInfo = &(prAdapter->rWifiVar.rScanInfo);
    prScanParam = &prScanInfo->rScanParam;

    // buffer empty channel information
    if(prScanParam->eScanChannel == SCAN_CHANNEL_FULL
            || prScanParam->eScanChannel == SCAN_CHANNEL_2G4) {
        if(prScanDone->ucSparseChannelValid) {
            prScanInfo->fgIsSparseChannelValid      = TRUE;
            prScanInfo->rSparseChannel.eBand        = (ENUM_BAND_T)prScanDone->rSparseChannel.ucBand;
            prScanInfo->rSparseChannel.ucChannelNum = prScanDone->rSparseChannel.ucChannelNum;
        }
        else {
            prScanInfo->fgIsSparseChannelValid      = FALSE;
        }
    }

    if(prScanInfo->eCurrentState == SCAN_STATE_SCANNING &&
            prScanDone->ucSeqNum == prScanParam->ucSeqNum) {
        /* generate scan-done event for caller */
        scnFsmGenerateScanDoneMsg(prAdapter,
                prScanParam->ucSeqNum,
                prScanParam->ucBssIndex,
                SCAN_STATUS_DONE);

        /* switch to next pending scan */
        scnFsmSteps(prAdapter, SCAN_STATE_IDLE);
    }
    else {
        DBGLOG(SCN, LOUD, ("Unexpected SCAN-DONE event: SeqNum = %d, Current State = %d\n",
                 prScanDone->ucSeqNum,
                 prScanInfo->eCurrentState));
    }

    return;
} /* end of scnEventScanDone */


/*----------------------------------------------------------------------------*/
/*!
* \brief
*
* \param[in]
*
* \return none
*/
/*----------------------------------------------------------------------------*/
VOID
scnFsmGenerateScanDoneMsg (
    IN P_ADAPTER_T          prAdapter,
    IN UINT_8               ucSeqNum,
    IN UINT_8               ucBssIndex,
    IN ENUM_SCAN_STATUS     eScanStatus
    )
{
    P_SCAN_INFO_T prScanInfo;
    P_SCAN_PARAM_T prScanParam;
    P_MSG_SCN_SCAN_DONE prScanDoneMsg;

    ASSERT(prAdapter);

    prScanInfo = &(prAdapter->rWifiVar.rScanInfo);
    prScanParam = &prScanInfo->rScanParam;

    prScanDoneMsg = (P_MSG_SCN_SCAN_DONE)cnmMemAlloc(prAdapter, RAM_TYPE_MSG, sizeof(MSG_SCN_SCAN_DONE));
    if (!prScanDoneMsg) {
        ASSERT(0); // Can't indicate SCAN FSM Complete
        return;
    }

    if(prScanParam->fgIsObssScan == TRUE) {
        prScanDoneMsg->rMsgHdr.eMsgId = MID_SCN_RLM_SCAN_DONE;
    }
    else {
        switch(GET_BSS_INFO_BY_INDEX(prAdapter, ucBssIndex)->eNetworkType) {
        case NETWORK_TYPE_AIS:
            prScanDoneMsg->rMsgHdr.eMsgId = MID_SCN_AIS_SCAN_DONE;
            break;

        case NETWORK_TYPE_P2P:
            prScanDoneMsg->rMsgHdr.eMsgId = MID_SCN_P2P_SCAN_DONE;
            break;

        case NETWORK_TYPE_BOW:
            prScanDoneMsg->rMsgHdr.eMsgId = MID_SCN_BOW_SCAN_DONE;
            break;

        default:
            DBGLOG(SCN, LOUD, ("Unexpected Network Type: %d\n", GET_BSS_INFO_BY_INDEX(prAdapter, ucBssIndex)->eNetworkType));
            ASSERT(0);
            break;
        }
    }

    prScanDoneMsg->ucSeqNum         = ucSeqNum;
    prScanDoneMsg->ucBssIndex       = ucBssIndex;
    prScanDoneMsg->eScanStatus      = eScanStatus;

    mboxSendMsg(prAdapter,
            MBOX_ID_0,
            (P_MSG_HDR_T) prScanDoneMsg,
            MSG_SEND_METHOD_BUF);

} /* end of scnFsmGenerateScanDoneMsg() */


/*----------------------------------------------------------------------------*/
/*!
* \brief        Query for most sparse channel
*
* \param[in]
*
* \return none
*/
/*----------------------------------------------------------------------------*/
BOOLEAN
scnQuerySparseChannel (
    IN P_ADAPTER_T      prAdapter,
    P_ENUM_BAND_T       prSparseBand,
    PUINT_8             pucSparseChannel
    )
{
    P_SCAN_INFO_T prScanInfo;

    ASSERT(prAdapter);

    prScanInfo = &(prAdapter->rWifiVar.rScanInfo);

    if(prScanInfo->fgIsSparseChannelValid == TRUE) {
        if(prSparseBand) {
            *prSparseBand = prScanInfo->rSparseChannel.eBand;
        }

        if(pucSparseChannel) {
            *pucSparseChannel = prScanInfo->rSparseChannel.ucChannelNum;
        }

        return TRUE;
    }
    else {
        return FALSE;
    }
}


/*----------------------------------------------------------------------------*/
/*!
* \brief        Event handler for NLO done event
*
* \param[in]
*
* \return none
*/
/*----------------------------------------------------------------------------*/
VOID
scnEventNloDone(
    IN P_ADAPTER_T          prAdapter,
    IN P_EVENT_NLO_DONE_T   prNloDone
    )
{
    P_SCAN_INFO_T prScanInfo;
    P_NLO_PARAM_T prNloParam;
    P_SCAN_PARAM_T prScanParam;

    prScanInfo = &(prAdapter->rWifiVar.rScanInfo);
    prNloParam = &prScanInfo->rNloParam;
    prScanParam = &prNloParam->rScanParam;

    if(prScanInfo->fgNloScanning == TRUE &&
            prNloDone->ucSeqNum == prScanParam->ucSeqNum) {
        kalSchedScanResults(prAdapter->prGlueInfo);

        if(prNloParam->fgStopAfterIndication == TRUE) {
            prScanInfo->fgNloScanning = FALSE;
        }
    }
    else {
        DBGLOG(SCN, LOUD, ("Unexpected NLO-DONE event: SeqNum = %d, Current State = %d\n",
                 prNloDone->ucSeqNum,
                 prScanInfo->eCurrentState));
    }

    return;
}


/*----------------------------------------------------------------------------*/
/*!
* \brief        OID handler for starting scheduled scan
*
* \param[in]
*
* \return none
*/
/*----------------------------------------------------------------------------*/
BOOLEAN
scnFsmSchedScanRequest (
    IN P_ADAPTER_T prAdapter,
    IN UINT_8   ucSsidNum,
    IN P_PARAM_SSID_T prSsid,
    IN UINT_32  u4IeLength,
    IN PUINT_8  pucIe,
    IN UINT_16  u2Interval
    )
{
    P_SCAN_INFO_T prScanInfo;
    P_NLO_PARAM_T prNloParam;
    P_SCAN_PARAM_T prScanParam;
    P_CMD_NLO_REQ prCmdNloReq;
    UINT_32 i, j;

    ASSERT(prAdapter);

    prScanInfo = &(prAdapter->rWifiVar.rScanInfo);
    prNloParam = &prScanInfo->rNloParam;
    prScanParam = &prNloParam->rScanParam;

    ASSERT(prScanInfo->fgNloScanning == FALSE);

    /* 1. load parameters */
    prScanParam->ucSeqNum++;
    prScanParam->ucBssIndex             = prAdapter->prAisBssInfo->ucBssIndex;
    prNloParam->fgStopAfterIndication   = TRUE;
    prNloParam->ucFastScanIteration     = 0;
    prNloParam->u2FastScanPeriod        = u2Interval;
    prNloParam->u2SlowScanPeriod        = u2Interval;

    if(prScanParam->ucSSIDNum > CFG_SCAN_SSID_MAX_NUM) {
        prScanParam->ucSSIDNum = CFG_SCAN_SSID_MAX_NUM;
    }
    else {
        prScanParam->ucSSIDNum = ucSsidNum;
    }

    if(prNloParam->ucMatchSSIDNum > CFG_SCAN_SSID_MATCH_MAX_NUM) {
        prNloParam->ucMatchSSIDNum = CFG_SCAN_SSID_MATCH_MAX_NUM;
    }
    else {
        prNloParam->ucMatchSSIDNum = ucSsidNum;
    }

    for(i = 0 ; i < prNloParam->ucMatchSSIDNum ; i++) {
        if(i < CFG_SCAN_SSID_MAX_NUM) {
            COPY_SSID(prScanParam->aucSpecifiedSSID[i],
                    prScanParam->ucSpecifiedSSIDLen[i],
                    prSsid[i].aucSsid,
                    (UINT_8)prSsid[i].u4SsidLen);
        }

        COPY_SSID(prNloParam->aucMatchSSID[i],
                prNloParam->ucMatchSSIDLen[i],
                prSsid[i].aucSsid,
                (UINT_8)prSsid[i].u4SsidLen);

        prNloParam->aucCipherAlgo[i]   = 0;
        prNloParam->au2AuthAlgo[i]     = 0;

        for(j = 0 ; j < SCN_NLO_NETWORK_CHANNEL_NUM ; j++) {
            prNloParam->aucChannelHint[i][j] = 0;
        }
    }

    /* 2. prepare command for sending */
    prCmdNloReq = (P_CMD_NLO_REQ)cnmMemAlloc(prAdapter,
            RAM_TYPE_BUF,
            sizeof(CMD_NLO_REQ) + prScanParam->u2IELen);

    if(!prCmdNloReq) {
        ASSERT(0); // Can't initiate NLO operation
        return FALSE;
    }

    /* 3. send command packet for NLO operation */
    kalMemZero(prCmdNloReq, sizeof(CMD_NLO_REQ));

    prCmdNloReq->ucSeqNum               = prScanParam->ucSeqNum;
    prCmdNloReq->ucBssIndex             = prScanParam->ucBssIndex;
    prCmdNloReq->fgStopAfterIndication  = prNloParam->fgStopAfterIndication;
    prCmdNloReq->ucFastScanIteration    = prNloParam->ucFastScanIteration;
    prCmdNloReq->u2FastScanPeriod       = prNloParam->u2FastScanPeriod;
    prCmdNloReq->u2SlowScanPeriod       = prNloParam->u2SlowScanPeriod;
    prCmdNloReq->ucEntryNum             = prNloParam->ucMatchSSIDNum;
    for(i = 0 ; i < prNloParam->ucMatchSSIDNum ; i++) {
        COPY_SSID(prCmdNloReq->arNetworkList[i].aucSSID,
                prCmdNloReq->arNetworkList[i].ucSSIDLength,
                prNloParam->aucMatchSSID[i],
                prNloParam->ucMatchSSIDLen[i]);

        prCmdNloReq->arNetworkList[i].ucCipherAlgo  = prNloParam->aucCipherAlgo[i];
        prCmdNloReq->arNetworkList[i].u2AuthAlgo    = prNloParam->au2AuthAlgo[i];

        for(j = 0 ; j < SCN_NLO_NETWORK_CHANNEL_NUM ; j++) {
            prCmdNloReq->arNetworkList[i].ucNumChannelHint[j] = prNloParam->aucChannelHint[i][j];
        }
    }

    if(prScanParam->u2IELen <= MAX_IE_LENGTH) {
        prCmdNloReq->u2IELen = prScanParam->u2IELen;
    }
    else {
        prCmdNloReq->u2IELen = MAX_IE_LENGTH;
    }

    if (prScanParam->u2IELen) {
        kalMemCopy(prCmdNloReq->aucIE,
                prScanParam->aucIE,
                sizeof(UINT_8) * prCmdNloReq->u2IELen);
    }

    wlanSendSetQueryCmd(prAdapter,
            CMD_ID_SET_NLO_REQ,
            TRUE,
            FALSE,
            TRUE,
            nicCmdEventSetCommon,
            nicOidCmdTimeoutCommon,
            sizeof(CMD_NLO_REQ) + prCmdNloReq->u2IELen,
            (PUINT_8)prCmdNloReq,
            NULL,
            0);

    cnmMemFree(prAdapter, (PVOID)prCmdNloReq);

    return TRUE;
}


/*----------------------------------------------------------------------------*/
/*!
* \brief        OID handler for stopping scheduled scan
*
* \param[in]
*
* \return none
*/
/*----------------------------------------------------------------------------*/
BOOLEAN
scnFsmSchedScanStopRequest (
    IN P_ADAPTER_T prAdapter
    )
{
    P_SCAN_INFO_T prScanInfo;
    P_NLO_PARAM_T prNloParam;
    P_SCAN_PARAM_T prScanParam;
    CMD_NLO_CANCEL rCmdNloCancel;

    ASSERT(prAdapter);

    prScanInfo = &(prAdapter->rWifiVar.rScanInfo);
    prNloParam = &prScanInfo->rNloParam;
    prScanParam = &prNloParam->rScanParam;

    if(prScanInfo->fgNloScanning == TRUE) {
        /* send cancel message to firmware domain */
        rCmdNloCancel.ucSeqNum = prScanParam->ucSeqNum;

        wlanSendSetQueryCmd(prAdapter,
                CMD_ID_SET_NLO_CANCEL,
                TRUE,
                FALSE,
                TRUE,
                nicCmdEventSetStopSchedScan,
                nicOidCmdTimeoutCommon,
                sizeof(CMD_NLO_CANCEL),
                (PUINT_8)&rCmdNloCancel,
                NULL,
                0);

        prScanInfo->fgNloScanning = FALSE;

        return TRUE;
    }
    else {
        return FALSE;
    }
}


