/* 
 * 
 * Author: yucong xiong <yucong.xion@mediatek.com>
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */
#include <linux/interrupt.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/irq.h>
#include <linux/miscdevice.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <linux/input.h>
#include <linux/workqueue.h>
#include <linux/kobject.h>
#include <linux/earlysuspend.h>
#include <linux/platform_device.h>
#include <asm/atomic.h>

#include <mach/mt_typedefs.h>
#include <mach/mt_gpio.h>
#include <mach/mt_pm_ldo.h>

#define POWER_NONE_MACRO MT65XX_POWER_NONE

#include <linux/hwmsensor.h>
#include <linux/hwmsen_dev.h>
#include <linux/sensors_io.h>
#include <asm/io.h>
#include <cust_eint.h>
#include <cust_als_sub.h>
#include "al3320.h"
#include <linux/sched.h>
#include <als_sub.h>
#include <linux/batch.h>

#ifdef CUSTOM_KERNEL_SENSORHUB //need modify PS data length for the custom sensor hub
#include <SCP_sensorHub.h>
#endif
/******************************************************************************
 * configuration
 *******************************************************************************/
/*----------------------------------------------------------------------------*/

#define AL3320_DEV_NAME     "al3320"
/*----------------------------------------------------------------------------*/
#define APS_TAG                  "[ALS_SUB] "
#define APS_FUN(f)               printk(KERN_INFO 	APS_TAG"%s\n", __FUNCTION__)
#define APS_ERR(fmt, args...)    printk(KERN_ERR  	APS_TAG"%s %d : "fmt, __FUNCTION__, __LINE__, ##args)
#define APS_LOG(fmt, args...)    printk(KERN_ERR	APS_TAG fmt, ##args)
#define APS_DBG(fmt, args...)    printk(KERN_INFO 	APS_TAG fmt, ##args)    

#define I2C_FLAG_WRITE	0
#define I2C_FLAG_READ	1

/******************************************************************************
 * extern functions
 *******************************************************************************/

/*----------------------------------------------------------------------------*/
static int al3320_i2c_probe(struct i2c_client *client, const struct i2c_device_id *id); 
static int al3320_i2c_remove(struct i2c_client *client);
static int al3320_i2c_detect(struct i2c_client *client, struct i2c_board_info *info);
static int al3320_i2c_suspend(struct i2c_client *client, pm_message_t msg);
static int al3320_i2c_resume(struct i2c_client *client);
static int al3320_read_als(struct i2c_client *client, u16 *data);

/*----------------------------------------------------------------------------*/
static const struct i2c_device_id al3320_i2c_id[] = {{AL3320_DEV_NAME,0},{}};
static struct i2c_board_info __initdata i2c_al3320={ I2C_BOARD_INFO(AL3320_DEV_NAME, 0x1C)};
/*----------------------------------------------------------------------------*/
struct al3320_priv {
	struct als_sub_hw  *hw;
	struct i2c_client *client;

	/*misc*/
	u16 		als_modulus;
	atomic_t	i2c_retry;
	atomic_t	als_suspend;
	atomic_t	als_debounce;	/*debounce time after enabling als*/
	atomic_t	als_deb_on; 	/*indicates if the debounce is on*/
	atomic_t	als_deb_end;	/*the jiffies representing the end of debounce*/


	/*data*/
	u16	als;
	u8	_align;
	u16	als_level_num;
	u16 	als_value_num;
	u32	als_level[C_CUST_ALS_SUB_LEVEL-1];
	u32	als_value[C_CUST_ALS_SUB_LEVEL];

	u8 reg_cache[AL3320_NUM_CACHABLE_REGS];

	ulong	enable; 		/*enable mask*/

	/*early suspend*/
#if defined(CONFIG_HAS_EARLYSUSPEND)
	struct early_suspend	early_drv;
#endif     
};
/*----------------------------------------------------------------------------*/

static struct i2c_driver al3320_i2c_driver = {	
	.probe      = al3320_i2c_probe,
	.remove     = al3320_i2c_remove,
	.detect     = al3320_i2c_detect,
	.suspend    = al3320_i2c_suspend,
	.resume     = al3320_i2c_resume,
	.id_table   = al3320_i2c_id,
	.driver = {
		.name = AL3320_DEV_NAME,
	},
};

/*----------------------------------------------------------------------------*/
static struct i2c_client *al3320_i2c_client = NULL;
static struct al3320_priv *al3320_obj = NULL;

static int al3320_local_init(void);
static int al3320_remove(void);
static int al3320_init_flag =-1; // 0<==>OK -1 <==> fail
static struct als_sub_init_info al3320_init_info = {
	.name = "al3320",
	.init = al3320_local_init,
	.uninit = al3320_remove,

};

/*----------------------------------------------------------------------------*/

static DEFINE_MUTEX(al3320_mutex);


/*----------------------------------------------------------------------------*/
typedef enum {
	CMC_BIT_ALS    = 1,
}CMC_BIT;

/*-----------------------------------------------------------------------------*/

int AL3320_i2c_master_operate(struct i2c_client *client, const char *buf, int count, int i2c_flag)
{
	int res = 0;
	mutex_lock(&al3320_mutex);
	switch(i2c_flag){	
		case I2C_FLAG_WRITE:
			client->addr &=I2C_MASK_FLAG;
			res = i2c_master_send(client, buf, count);
			client->addr &=I2C_MASK_FLAG;
			break;

		case I2C_FLAG_READ:
			client->addr &=I2C_MASK_FLAG;
			client->addr |=I2C_WR_FLAG;
			client->addr |=I2C_RS_FLAG;
			res = i2c_master_send(client, buf, count);
			client->addr &=I2C_MASK_FLAG;
			break;
		default:
			APS_LOG("AL3320_i2c_master_operate i2c_flag command not support!\n");
			break;
	}
	if(res < 0)
	{
		goto EXIT_ERR;
	}
	mutex_unlock(&al3320_mutex);
	return res;
EXIT_ERR:
	mutex_unlock(&al3320_mutex);
	APS_ERR("AL3320_i2c_master_operate fail\n");
	return res;
}
/*----------------------------------------------------------------------------*/
static void al3320_power(struct als_sub_hw *hw, unsigned int on) 
{
	static unsigned int power_on = 0;

	APS_LOG("power %s\n", on ? "on" : "off");

	if(hw->power_id != POWER_NONE_MACRO)
	{
		if(power_on == on)
		{
			APS_LOG("ignore power control: %d\n", on);
		}
		else if(on)
		{
			if(!hwPowerOn(hw->power_id, hw->power_vol, "AL3320")) 
			{
				APS_ERR("power on fails!!\n");
			}
		}
		else
		{
			if(!hwPowerDown(hw->power_id, "AL3320")) 
			{
				APS_ERR("power off fail!!\n");   
			}
		}
	}
	power_on = on;
}
static int al3320_write_reg(struct i2c_client *client,
				u32 reg, u8 mask, u8 shift, u8 val)
{
	struct al3320_priv *data = i2c_get_clientdata(client);
	int ret = 0;
	u8 tmp;
	u8 idx = 0xff;
	u8 databuf[2];

	ADD_TO_IDX(reg,idx)
	if (idx >= AL3320_NUM_CACHABLE_REGS)
		return -EINVAL;


	tmp = data->reg_cache[idx];
	tmp &= ~mask;
	tmp |= val << shift;

	databuf[0] = reg;
	databuf[1] = tmp;

	ret = AL3320_i2c_master_operate(client, databuf,0x2,I2C_FLAG_WRITE);
	if (!ret)
		data->reg_cache[idx] = tmp;
	return ret;
}

static int al3320_read_reg(struct i2c_client *client,
			       u32 reg, u8 mask, u8 shift)
{
	struct al3320_priv *data = i2c_get_clientdata(client);
	u8 idx = 0xff;

	ADD_TO_IDX(reg,idx)
	return (data->reg_cache[idx] & mask) >> shift;
}

static int al3320_get_mode(struct i2c_client *client)
{
	int ret;

	ret = al3320_read_reg(client, AL3320_MODE_COMMAND,
			AL3320_MODE_MASK, AL3320_MODE_SHIFT);
	return ret;
}

static int al3320_set_mode(struct i2c_client *client, int mode)
{
  struct al3320_priv *data = i2c_get_clientdata(client);
  
  if (mode != al3320_get_mode(client))
  {
   	al3320_write_reg(client, AL3320_MODE_COMMAND,
				  AL3320_MODE_MASK, AL3320_MODE_SHIFT, mode);
  }
  return 0;
}

static int al3320_set_waiting_time(struct i2c_client *client, int wait_time)
{
	int ret = al3320_write_reg(client, AL3320_WAITING_TIME,
		AL3320_WAITING_MASK, AL3320_WAITING_SHIFT, wait_time);

	return ret;
}

static int al3320_set_int_enable(struct i2c_client *client, int flag)
{
	int ret = al3320_write_reg(client, AL3320_INT_ENABLE,
		AL3320_INT_ENABLE_MASK, AL3320_INT_ENABLE_SHIFT, flag);

	return ret;
}

static int al3320_set_sus_enable(struct i2c_client *client, int flag)
{
	int ret = al3320_write_reg(client, AL3320_INT_ENABLE,
		AL3320_SUS_ENABLE_MASK, AL3320_SUS_ENABLE_SHIFT, flag);

	return ret;
}

static int al3320_set_adummy(struct i2c_client *client, int adummy)
{
	return al3320_write_reg(client, AL3320_ALS_ADUMMY,
		AL3320_ADUMMY_MASK, AL3320_ADUMMY_SHIFT, adummy);;
}

static long al3320_get_range(struct i2c_client *client)
{
	u8 idx, exgain;
  
	exgain = al3320_read_reg(client, AL3320_RAN_COMMAND,
		AL3320_EXGAIN_MASK, AL3320_EXGAIN_SHIFT); 
    
	idx = al3320_read_reg(client, AL3320_RAN_COMMAND,
		AL3320_RAN_MASK, AL3320_RAN_SHIFT);
     
	return (exgain ? al3320_range[idx] : al3320_range[idx+4]);
}

static int al3320_set_range(struct i2c_client *client, int range)
{
	int adummy, ret;

	switch(range)
	{
		case ALS_RAN_0:	adummy = ALS_ADUMMY_0; break;
		case ALS_RAN_1:	adummy = ALS_ADUMMY_1; break;
		case ALS_RAN_2:	adummy = ALS_ADUMMY_2; break;
		case ALS_RAN_3:	adummy = ALS_ADUMMY_3; break;
		default: adummy = 0;
	}
	
	ret = al3320_set_adummy(client, adummy);

	if (ret)
		return ret;

	return al3320_write_reg(client, AL3320_RAN_COMMAND,
		AL3320_RAN_MASK, AL3320_RAN_SHIFT, range);;
}

static int al3320_get_exgain(struct i2c_client *client)
{
	u8 exgain = al3320_read_reg(client, AL3320_RAN_COMMAND,
		AL3320_EXGAIN_MASK, AL3320_EXGAIN_SHIFT); 

	return exgain;
}

static int al3320_set_exgain(struct i2c_client *client, int exgain)
{
	return al3320_write_reg(client, AL3320_RAN_COMMAND,
		AL3320_EXGAIN_MASK, AL3320_EXGAIN_SHIFT, exgain);;
}

static int al3320_set_persist(struct i2c_client *client, int persist)
{
	return al3320_write_reg(client, AL3320_ALS_PERSIST,
		AL3320_PERSIST_MASK, AL3320_PERSIST_SHIFT, persist);;
}

static int al3320_set_meantime(struct i2c_client *client, int meantime)
{
	return al3320_write_reg(client, AL3320_ALS_MEANTIME,
		AL3320_MEANTIME_MASK, AL3320_MEANTIME_SHIFT, meantime);;
}

/********************************************************************/
static int al3320_enable_als(struct i2c_client *client, int enable)
{
	struct al3320_priv *obj = i2c_get_clientdata(client);
	int res;
	APS_LOG("al3320_enable_als enable:%d\n", enable);
	
        res = al3320_set_mode(obj->client, enable);
	if(enable == 1)
	{
		atomic_set(&obj->als_deb_on, 1);
		atomic_set(&obj->als_deb_end, jiffies+atomic_read(&obj->als_debounce)/(1000/HZ));
		set_bit(CMC_BIT_ALS, &obj->enable);

		msleep(80);
		al3320_read_als(obj->client, &obj->als);
		als_sub_report_interrupt_data(obj->als);
		APS_ERR("al3320_enable_als first data report value:%d\n", obj->als);
	}
	else
	{
		atomic_set(&obj->als_deb_on, 0);
		clear_bit(CMC_BIT_ALS, &obj->enable);
	}
	return 0;
}

/********************************************************************/
int al3320_read_als(struct i2c_client *client, u16 *data)
{
	long res;
	u8 databuf[2];
	unsigned long tmp, range;
	struct al3320_priv *obj = i2c_get_clientdata(client);

	databuf[0] = AL3320_ADC_LSB;
	res = AL3320_i2c_master_operate(client, databuf, 0x201, I2C_FLAG_READ);
	if(res < 0)
	{
		APS_ERR("i2c_master_send function err\n");
		goto READ_ALS_EXIT_ERR;
	}

	range = al3320_get_range(client);

	tmp = (((databuf[1] << 8) |databuf[0]) * range) >> 16;
	tmp *= al3320_cali;
        *data = (tmp / 100);
	
	return 0;
READ_ALS_EXIT_ERR:
	return res;
}

/********************************************************************/
static int al3320_get_als_value(struct al3320_priv *obj, u16 als)
{
	int idx;
	int invalid = 0;
	for(idx = 0; idx < obj->als_level_num; idx++)
	{
		if(als < obj->hw->als_level[idx])
		{
			break;
		}
	}
	if(idx >= obj->als_value_num)
	{
		APS_ERR("exceed range\n"); 
		idx = obj->als_value_num - 1;
	}

	if(1 == atomic_read(&obj->als_deb_on))
	{
		unsigned long endt = atomic_read(&obj->als_deb_end);
		if(time_after(jiffies, endt))
		{
			atomic_set(&obj->als_deb_on, 0);
		}

		if(1 == atomic_read(&obj->als_deb_on))
		{
			invalid = 1;
		}
	}

	if(!invalid)
	{
		return obj->hw->als_value[idx];
	}
	else
	{
		return -1;
	}

}


/*-------------------------------attribute file for debugging----------------------------------*/

/******************************************************************************
 * Sysfs attributes
 *******************************************************************************/
static ssize_t al3320_show_reg(struct device_driver *ddri, char *buf)
{
	ssize_t len = 0;
	struct i2c_client *client = al3320_obj->client;
	u8 databuf[2];
	int i,res;	

	for (i = 0; i < AL3320_NUM_CACHABLE_REGS; i++)
	{
		databuf[0] =al3320_reg[i];
		res = AL3320_i2c_master_operate(client, databuf, 0x101, I2C_FLAG_READ);
		if(res<0)
		{
			APS_ERR("chip id REG 0x%x read error\n", i);
		}
		len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x%x value = 0x%x\n", i, ((databuf[1]<<8)|databuf[0]));
	}
	return len;
}


static ssize_t al3320_show_als_enable(struct device_driver *ddri, char *buf)
{
	struct al3320_priv *obj = al3320_obj;
	ssize_t len = 0;
	bool enable_als = test_bit(CMC_BIT_ALS, &obj->enable);
	APS_FUN();
	len += snprintf(buf + len, PAGE_SIZE - len, "%d\n", enable_als);
	return len;

}

static ssize_t al3320_store_als_enable(struct device_driver *ddri, const char *buf, size_t count)
{
	struct al3320_priv *obj = al3320_obj;
	uint16_t enable=0;
	int res;
	sscanf(buf, "%hu", &enable);
	APS_LOG("al3320_store_als_enable entry enable:%d !!! \n", enable);
	res = al3320_enable_als(obj->client, enable);
	if(res)
	{
		APS_ERR("als_enable_nodata is failed!!\n");
	}

	APS_LOG("al3320_store_als_enable finished !!! \n");
	return count;
}


static ssize_t al3320_show_als_data(struct device_driver *ddri, char *buf)
{
	struct al3320_priv *obj = al3320_obj;
	ssize_t len = 0;
	al3320_read_als(obj->client, &obj->als);
	len += snprintf(buf + len, PAGE_SIZE - len, "%d\n",  obj->als);
	return len;
}


static ssize_t al3320_store_reg_write(struct device_driver *ddri, const char *buf, size_t count)
{
	struct al3320_priv *obj = al3320_obj;
	int reg;
	int data;
	u8 databuf[2];
	int res;
	APS_FUN();

	sscanf(buf, "%x,%x",&reg, &data);

	APS_LOG("[%s]: reg=0x%x, data=0x%x", __func__, reg, data);
	databuf[1] = data & 0x00FF;
	databuf[0] = reg;
	res = AL3320_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
	if(res < 0)
	{
		APS_ERR("i2c_master_send function err\n");
	}
	return count;
}
/*---------------------------------------------------------------------------------------*/
static DRIVER_ATTR(reg,     S_IWUSR | S_IRUGO, al3320_show_reg, NULL);
static DRIVER_ATTR(als_enable,	S_IROTH  | S_IWOTH,	 al3320_show_als_enable, al3320_store_als_enable);
static DRIVER_ATTR(als_data,	S_IROTH  | S_IWOTH, 	al3320_show_als_data, NULL);
static DRIVER_ATTR(i2c_w,	S_IROTH  | S_IWOTH, 	NULL, al3320_store_reg_write );
/*----------------------------------------------------------------------------*/
static struct driver_attribute *al3320_attr_list[] = {
	&driver_attr_reg,
	&driver_attr_als_enable,
	&driver_attr_als_data,
	&driver_attr_i2c_w,
};

/*----------------------------------------------------------------------------*/
static int al3320_create_attr(struct device_driver *driver) 
{
	int idx, err = 0;
	int num = (int)(sizeof(al3320_attr_list)/sizeof(al3320_attr_list[0]));
	if (driver == NULL)
	{
		return -EINVAL;
	}

	for(idx = 0; idx < num; idx++)
	{
		if((err = driver_create_file(driver, al3320_attr_list[idx])))
		{            
			APS_ERR("driver_create_file (%s) = %d\n", al3320_attr_list[idx]->attr.name, err);
			break;
		}
	}    
	return err;
}
/*----------------------------------------------------------------------------*/
static int al3320_delete_attr(struct device_driver *driver)
{
	int idx ,err = 0;
	int num = (int)(sizeof(al3320_attr_list)/sizeof(al3320_attr_list[0]));

	if (!driver)
		return -EINVAL;

	for (idx = 0; idx < num; idx++) 
	{
		driver_remove_file(driver, al3320_attr_list[idx]);
	}

	return err;
}
/*----------------------------------------------------------------------------*/


/*-------------------------------MISC device related------------------------------------------*/
static int al3320_open(struct inode *inode, struct file *file)
{
	file->private_data = al3320_i2c_client;

	if (!file->private_data)
	{
		APS_ERR("null pointer!!\n");
		return -EINVAL;
	}
	return nonseekable_open(inode, file);
}
/************************************************************/
static int al3320_release(struct inode *inode, struct file *file)
{
	file->private_data = NULL;
	return 0;
}
/************************************************************/

/************************************************PS CALI*****************************************************************************/

static long al3320_unlocked_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct i2c_client *client = (struct i2c_client*)file->private_data;
	struct al3320_priv *obj = i2c_get_clientdata(client);  
	long err = 0;
	void __user *ptr = (void __user*) arg;
	int dat;
	uint32_t enable;
	int ps_result;
	int threshold[2];
	HWMON_PS_STRUCT ps_cali_temp;

	switch (cmd)
	{
		
		case ALSPS_SET_ALS_MODE:

			if(copy_from_user(&enable, ptr, sizeof(enable)))
			{
				err = -EFAULT;
				goto err_out;
			}
			if(enable)
			{
				if((err = al3320_enable_als(obj->client, 1)))
				{
					APS_ERR("enable als fail: %ld\n", err); 
					goto err_out;
				}
				set_bit(CMC_BIT_ALS, &obj->enable);
			}
			else
			{
				if((err = al3320_enable_als(obj->client, 0)))
				{
					APS_ERR("disable als fail: %ld\n", err); 
					goto err_out;
				}
				clear_bit(CMC_BIT_ALS, &obj->enable);
			}
			break;

		case ALSPS_GET_ALS_MODE:
			enable = test_bit(CMC_BIT_ALS, &obj->enable) ? (1) : (0);
			if(copy_to_user(ptr, &enable, sizeof(enable)))
			{
				err = -EFAULT;
				goto err_out;
			}
			break;

		case ALSPS_GET_ALS_DATA: 

			APS_ERR("al3320 get als data enter\n"); 
			if((err = al3320_read_als(obj->client, &obj->als)))
			{
				goto err_out;
			}			

			dat =obj->als;
			if(copy_to_user(ptr, &dat, sizeof(dat)))
			{
				err = -EFAULT;
				goto err_out;
			}			   
			break;

		case ALSPS_GET_ALS_RAW_DATA:	
			APS_ERR("al3320 get als raw data enter\n"); 
			if((err = al3320_read_als(obj->client, &obj->als)))
			{
				goto err_out;
			}

			dat = obj->als;
			if(copy_to_user(ptr, &dat, sizeof(dat)))
			{
				err = -EFAULT;
				goto err_out;
			}			   
			break;


		default:
			APS_ERR("%s not supported = 0x%04x", __FUNCTION__, cmd);
			err = -ENOIOCTLCMD;
			break;
	}

err_out:
	return err;    
}
/*------------------------------misc device related operation functions------------------------------------*/
static struct file_operations al3320_fops = {
	.owner = THIS_MODULE,
	.open = al3320_open,
	.release = al3320_release,
	.unlocked_ioctl = al3320_unlocked_ioctl,
};

static struct miscdevice al3320_device = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "als_sub",
	.fops = &al3320_fops,
};

/*--------------------------------------------------------------------------------------*/
static void al3320_early_suspend(struct early_suspend *h)
{
	struct al3320_priv *obj = container_of(h, struct al3320_priv, early_drv);	
	int err;
	APS_LOG("al3320_early_suspend entry!!!\n"); 
	if(!obj)
	{
		APS_ERR("null pointer!!\n");
		return;
	}
	if (1 == test_bit(CMC_BIT_ALS, &obj->enable))
	{
		if(err = al3320_enable_als(obj->client, 0))
		{
			APS_ERR("al3320_early_suspend disable als fail: %d\n", err); 
		}
		atomic_set(&obj->als_suspend, 1);
	}
	APS_LOG("al3320_early_suspend finished!!!\n"); 
}

static void al3320_late_resume(struct early_suspend *h) 
{
	struct al3320_priv *obj = container_of(h, struct al3320_priv, early_drv);		  
	int err;
	APS_LOG("al3320_late_resume entry!!!\n"); 
	if(!obj)
	{
		APS_ERR("null pointer!!\n");
		return;
	}
	if(1 == atomic_read(&obj->als_suspend))
	{
		if((err = al3320_enable_als(obj->client, 1)))
		{
			APS_ERR("enable als fail: %d\n", err);		  

		}
		atomic_set(&obj->als_suspend, 0);
	}
	APS_LOG("al3320_late_resume finished!!!\n"); 
}
/*--------------------------------------------------------------------------------*/
static int al3320_init_client(struct i2c_client *client)
{
	struct al3320_priv *obj = i2c_get_clientdata(client);  
	int res = 0;
	int i;
	APS_FUN();
        al3320_set_mode(client, ALS_RESET);
	mdelay(15);

	for (i = 0; i < AL3320_NUM_CACHABLE_REGS; i++) {
		int v = i2c_smbus_read_byte_data(client, al3320_reg[i]);
		if (v < 0)
			return -ENODEV;

		obj->reg_cache[i] = v;
	}
	al3320_set_waiting_time(client, ALS_NO_WAITING);
	mdelay(1);

	// ALS gain
	al3320_set_range(client, als_range);
	mdelay(1);
	al3320_set_exgain(client, als_exgain);
	mdelay(1);

	// ALS meantime
	al3320_set_meantime(client, als_meantime);
	mdelay(1);

	// interrupt, suspend settings
	al3320_set_sus_enable(client, DISABLE);
	mdelay(1);

	al3320_set_int_enable(client, DISABLE);
	mdelay(1);
	return AL3320_SUCCESS;

EXIT_ERR:
	APS_ERR("init dev: %d\n", res);
	return res;
}
/*--------------------------------------------------------------------------------*/

// if use  this typ of enable , Gsensor should report inputEvent(x, y, z ,stats, div) to HAL
static int als_open_report_data(int open)
{
	//should queuq work to report event if  is_report_input_direct=true
	return 0;
}

// if use  this typ of enable , Gsensor only enabled but not report inputEvent to HAL

static int als_enable_nodata(int en)
{
	struct al3320_priv *obj = al3320_obj;
	int res = 0;
	APS_LOG("al3320_obj als enable value = %d\n", en);

	if(!obj)
	{
		APS_ERR("al3320_obj is null!!\n");
		return -1;
	}
	res=	al3320_enable_als(obj->client, en);
	if(res)
	{
		APS_ERR("als_enable_nodata is failed!!\n");
		return -1;
	}
	return 0;
}

static int als_set_delay(u64 ns)
{
	return 0;
}

static int als_get_data(int* value, int* status)
{
	struct al3320_priv *obj = al3320_obj;
	int err = 0;

	if(!obj)
	{
		APS_ERR("al3320_obj is null!!\n");
		return -1;
	}
	if((err = al3320_read_als(obj->client, &obj->als)))
	{
		err = -1;
	}
	else
	{
		//APS_LOG("als_get_data value = %d\n", obj->als);
		*value = obj->als;//al3320_get_als_value(obj, obj->als);
		*status = SENSOR_STATUS_ACCURACY_MEDIUM;
	}
	return err;
}

// if use  this typ of enable , Gsensor only enabled but not report inputEvent to HAL

/*-----------------------------------i2c operations----------------------------------*/
static int al3320_i2c_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	struct al3320_priv *obj;
	int err = 0;
	struct als_sub_control_path als_ctl={0};
	struct als_sub_data_path als_data={0};
	int databuf[2];
	APS_LOG("al3320_i2c_probe entry!!!\n");
	databuf[0] = 0x00;
	err = AL3320_i2c_master_operate(client, databuf, 0x201, I2C_FLAG_READ);
	if(err<0)
	{
		APS_ERR("probe REG 0x00 read error\n");
		goto exit;
	}
	if(!(obj = kzalloc(sizeof(*obj), GFP_KERNEL)))
	{
		err = -ENOMEM;
		goto exit;
	}

	memset(obj, 0, sizeof(*obj));
	al3320_obj = obj;

	obj->hw = get_cust_als_sub_hw();//get custom file data struct

	obj->client = client;
	i2c_set_clientdata(client, obj);
	/*-----------------------------value need to be confirmed-----------------------------------------*/
	atomic_set(&obj->als_debounce, 200);
	atomic_set(&obj->als_deb_on, 0);
	atomic_set(&obj->als_deb_end, 0);
	atomic_set(&obj->als_suspend, 0);

	obj->enable = 0;
	obj->als_level_num = sizeof(obj->hw->als_level)/sizeof(obj->hw->als_level[0]);
	obj->als_value_num = sizeof(obj->hw->als_value)/sizeof(obj->hw->als_value[0]);
	/*-----------------------------value need to be confirmed-----------------------------------------*/

	BUG_ON(sizeof(obj->als_level) != sizeof(obj->hw->als_level));
	memcpy(obj->als_level, obj->hw->als_level, sizeof(obj->als_level));
	BUG_ON(sizeof(obj->als_value) != sizeof(obj->hw->als_value));
	memcpy(obj->als_value, obj->hw->als_value, sizeof(obj->als_value));
	atomic_set(&obj->i2c_retry, 3);
	al3320_i2c_client = client;

	if((err = al3320_init_client(client)))
	{
		goto exit_init_failed;
	}
	APS_LOG("al3320_init_client() OK!\n");

	if((err = misc_register(&al3320_device)))
	{
		APS_ERR("al3320_device register failed\n");
		goto exit_misc_device_register_failed;
	}
	APS_LOG("al3320_device misc_register OK!\n");

	/*------------------------al3320 attribute file for debug--------------------------------------*/
	if((err = al3320_create_attr(&(al3320_init_info.platform_diver_addr->driver))))
	{
		APS_ERR("create attribute err = %d\n", err);
		goto exit_create_attr_failed;
	}
	/*------------------------al3320 attribute file for debug--------------------------------------*/
	als_ctl.open_report_data= als_open_report_data;
	als_ctl.enable_nodata = als_enable_nodata;
	als_ctl.set_delay  = als_set_delay;
	als_ctl.is_report_input_direct = false;
#ifdef CUSTOM_KERNEL_SENSORHUB
	als_ctl.is_support_batch = obj->hw->is_batch_supported_als;
#else
	als_ctl.is_support_batch = false;
#endif

	err = als_sub_register_control_path(&als_ctl);
	if(err)
	{
		APS_ERR("register fail = %d\n", err);
		goto exit_sensor_obj_attach_fail;
	}

	als_data.get_data = als_get_data;
	als_data.vender_div = 100;
	err = als_sub_register_data_path(&als_data);	
	if(err)
	{
		APS_ERR("tregister fail = %d\n", err);
		goto exit_sensor_obj_attach_fail;
	}
#if defined(CONFIG_HAS_EARLYSUSPEND) && defined(CONFIG_EARLYSUSPEND)
	obj->early_drv.level  = EARLY_SUSPEND_LEVEL_STOP_DRAWING - 2,
		obj->early_drv.suspend  = al3320_early_suspend,
		obj->early_drv.resume  = al3320_late_resume,    
		register_early_suspend(&obj->early_drv);
#endif

	al3320_init_flag =0;
	return 0;

exit_create_attr_failed:
exit_sensor_obj_attach_fail:
exit_misc_device_register_failed:
	misc_deregister(&al3320_device);
exit_init_failed:
	kfree(obj);
exit:
	al3320_i2c_client = NULL;           
	APS_ERR("%s: err = %d\n", __func__, err);
	al3320_init_flag = -1;
	return err;
}

static int al3320_i2c_remove(struct i2c_client *client)
{
	int err;	
	/*------------------------al3320 attribute file for debug--------------------------------------*/	
	if((err = al3320_delete_attr(&(al3320_init_info.platform_diver_addr->driver))))
	{
		APS_ERR("al3320_delete_attr fail: %d\n", err);
	} 
	/*----------------------------------------------------------------------------------------*/

	if((err = misc_deregister(&al3320_device)))
	{
		APS_ERR("misc_deregister fail: %d\n", err);    
	}

	al3320_i2c_client = NULL;
	i2c_unregister_device(client);
	kfree(i2c_get_clientdata(client));
	return 0;

}

static int al3320_i2c_detect(struct i2c_client *client, struct i2c_board_info *info)
{
	strcpy(info->type, AL3320_DEV_NAME);
	return 0;

}

static int al3320_i2c_suspend(struct i2c_client *client, pm_message_t msg)
{
	APS_FUN();
	return 0;
}

static int al3320_i2c_resume(struct i2c_client *client)
{
	APS_FUN();
	return 0;
}
/*----------------------------------------------------------------------------*/
static int al3320_remove(void)
{
	struct als_sub_hw *hw = get_cust_als_sub_hw();
	al3320_power(hw, 0);
	i2c_del_driver(&al3320_i2c_driver);
	return 0;
}
/*----------------------------------------------------------------------------*/
static int  al3320_local_init(void)
{
	struct als_sub_hw *hw = get_cust_als_sub_hw();
	APS_LOG("al3320_local_init entry!!!\n");
	al3320_power(hw, 1);
	if(i2c_add_driver(&al3320_i2c_driver))
	{
		APS_ERR("add driver error\n");
		return -1;
	}
	if(-1 == al3320_init_flag)
	{
		APS_ERR("al3320_local_init failed!!!\n");
		return -1;
	}
	APS_LOG("al3320_local_init finished!!!\n");
	return 0;
}
/*----------------------------------------------------------------------------*/
static int __init al3320_init(void)
{
	struct als_sub_hw *hw = get_cust_als_sub_hw();
	APS_LOG("%s: i2c_number=%d\n", __func__, hw->i2c_num);
	i2c_register_board_info(hw->i2c_num, &i2c_al3320, 1);
	als_sub_driver_add(&al3320_init_info);
	APS_LOG("%s finished!!!\n", __func__);
	return 0;
}
/*----------------------------------------------------------------------------*/
static void __exit al3320_exit(void)
{
	APS_FUN();
}
/*----------------------------------------------------------------------------*/
module_init(al3320_init);
module_exit(al3320_exit);
/*----------------------------------------------------------------------------*/
MODULE_AUTHOR("yucong xiong");
MODULE_DESCRIPTION("al3320 driver");
MODULE_LICENSE("GPL");
