/* Copyright (c) 2011-2014, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#ifndef _IPC_ROUTER_PRIVATE_H
#define _IPC_ROUTER_PRIVATE_H

#include <linux/types.h>
#include <linux/socket.h>
#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/list.h>
#include <linux/platform_device.h>
#include <linux/msm_ipc.h>
#include <linux/ipc_router.h>
#include <linux/ipc_router_xprt.h>

#include <net/sock.h>

/* definitions for the R2R wire protcol */
#define IPC_ROUTER_V1		1
/*
 * Ambiguous definition but will enable multiplexing IPC_ROUTER_V2 packets
 * with an existing alternate transport in user-space, if needed.
 */
#define IPC_ROUTER_V2		3

#define IPC_ROUTER_ADDRESS			0x0000FFFF

#define IPC_ROUTER_NID_LOCAL			1
#define MAX_IPC_PKT_SIZE 66000

#define IPC_ROUTER_DEFAULT_RX_QUOTA	5

#define IPC_ROUTER_INFINITY -1
#define DEFAULT_RCV_TIMEO 0

#define ALIGN_SIZE(x) ((4 - ((x) & 3)) & 3)

#define ALL_SERVICE 0xFFFFFFFF
#define ALL_INSTANCE 0xFFFFFFFF

#define CONTROL_FLAG_CONFIRM_RX 0x1
#define CONTROL_FLAG_OPT_HDR 0x2

enum {
	CLIENT_PORT,
	SERVER_PORT,
	CONTROL_PORT,
	IRSC_PORT,
};

enum {
	NULL_MODE,
	SINGLE_LINK_MODE,
	MULTI_LINK_MODE,
};

struct msm_ipc_sock {
	struct sock sk;
	struct msm_ipc_port *port;
	void *default_node_vote_info;
};

/**
 * msm_ipc_router_create_raw_port() - Create an IPC Router port
 * @endpoint: User-space space socket information to be cached.
 * @notify: Function to notify incoming events on the port.
 *   @event: Event ID to be handled.
 *   @oob_data: Any out-of-band data associated with the event.
 *   @oob_data_len: Size of the out-of-band data, if valid.
 *   @priv: Private data registered during the port creation.
 * @priv: Private Data to be passed during the event notification.
 *
 * @return: Valid pointer to port on success, NULL on failure.
 *
 * This function is used to create an IPC Router port. The port is used for
 * communication locally or outside the subsystem.
 */
struct msm_ipc_port *msm_ipc_router_create_raw_port(void *endpoint,
	void (*notify)(unsigned event, void *oob_data,
		       size_t oob_data_len, void *priv),
	void *priv);
int msm_ipc_router_send_to(struct msm_ipc_port *src,
			   struct sk_buff_head *data,
			   struct msm_ipc_addr *dest);
int msm_ipc_router_read(struct msm_ipc_port *port_ptr,
			struct rr_packet **pkt,
			size_t buf_len);

int msm_ipc_router_recv_from(struct msm_ipc_port *port_ptr,
		      struct rr_packet **pkt,
		      struct msm_ipc_addr *src_addr,
		      long timeout);
int msm_ipc_router_register_server(struct msm_ipc_port *server_port,
			    struct msm_ipc_addr *name);
int msm_ipc_router_unregister_server(struct msm_ipc_port *server_port);

int msm_ipc_router_init_sockets(void);
void msm_ipc_router_exit_sockets(void);

void msm_ipc_sync_sec_rule(uint32_t service, uint32_t instance, void *rule);

void msm_ipc_sync_default_sec_rule(void *rule);

int msm_ipc_router_rx_data_wait(struct msm_ipc_port *port_ptr, long timeout);

void msm_ipc_router_free_skb(struct sk_buff_head *skb_head);
#endif
