/* 
 * 
 * Author: yucong xiong <yucong.xion@mediatek.com>
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */
#include <linux/interrupt.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/irq.h>
#include <linux/miscdevice.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <linux/input.h>
#include <linux/workqueue.h>
#include <linux/kobject.h>
#include <linux/earlysuspend.h>
#include <linux/platform_device.h>
#include <asm/atomic.h>

#include <mach/mt_typedefs.h>
//#include <mach/mt_gpio.h>
#include <mach/mt_pm_ldo.h>

#define POWER_NONE_MACRO MT65XX_POWER_NONE

#include <linux/hwmsensor.h>
#include <linux/hwmsen_dev.h>
#include <linux/sensors_io.h>
#include <asm/io.h>
#include <cust_eint.h>
#include <cust_als_sub.h>
#include "bh1745.h"
#include <linux/sched.h>
#include <als_sub.h>
#include <linux/batch.h>
//#include <math.h>

#ifdef CUSTOM_KERNEL_SENSORHUB //need modify PS data length for the custom sensor hub
#include <SCP_sensorHub.h>
#endif
/******************************************************************************
 * configuration
 *******************************************************************************/
/*----------------------------------------------------------------------------*/

#define BH1745_DEV_NAME     "bh1745"
/*----------------------------------------------------------------------------*/
#define APS_TAG                  "[ALS_SUB] "
#define APS_FUN(f)               printk(KERN_INFO 	APS_TAG"%s\n", __FUNCTION__)
#define APS_ERR(fmt, args...)    printk(KERN_ERR  	APS_TAG"%s %d : "fmt, __FUNCTION__, __LINE__, ##args)
#define APS_LOG(fmt, args...)    printk(KERN_ERR	APS_TAG fmt, ##args)
#define APS_DBG(fmt, args...)    printk(KERN_INFO 	APS_TAG fmt, ##args)    

#define I2C_FLAG_WRITE	0
#define I2C_FLAG_READ	1

#define JUDGE_FIXED_COEF  	1000
#define CUT_UNIT  	1000
#define TRANS 		1

static int judge = 161;
static int red[2] = {92, 98};
static int green[2]={525, 416};
static double ratio_judgeC= 0.27;
static double beff[3]={3.371, 12.143, 1.0};
static double color[4]={-2.909, 6.552, -2.832,  4.795};
static int cct_eff[4]= {11506, 1263, 18595, 1915};

unsigned short atime[6] = {160, 320, 640, 1280, 2560, 5120};
unsigned char again[3] = {1, 2, 16};

/******************************************************************************
 * extern functions
 *******************************************************************************/

/*----------------------------------------------------------------------------*/
static int bh1745_i2c_probe(struct i2c_client *client, const struct i2c_device_id *id); 
static int bh1745_i2c_remove(struct i2c_client *client);
static int bh1745_i2c_detect(struct i2c_client *client, struct i2c_board_info *info);
static int bh1745_i2c_suspend(struct i2c_client *client, pm_message_t msg);
static int bh1745_i2c_resume(struct i2c_client *client);
static int bh1745_read_als(struct i2c_client *client, u16 *data);

/*----------------------------------------------------------------------------*/
static const struct i2c_device_id bh1745_i2c_id[] = {{BH1745_DEV_NAME,0},{}};
static struct i2c_board_info __initdata i2c_bh1745={ I2C_BOARD_INFO(BH1745_DEV_NAME, 0x38)};
/*----------------------------------------------------------------------------*/
struct bh1745_priv {
	struct als_sub_hw  *hw;
	struct i2c_client *client;

	/*misc*/
	u16 		als_modulus;
	atomic_t	i2c_retry;
	atomic_t	als_suspend;
	atomic_t	als_debounce;	/*debounce time after enabling als*/
	atomic_t	als_deb_on; 	/*indicates if the debounce is on*/
	atomic_t	als_deb_end;	/*the jiffies representing the end of debounce*/


	/*data*/
	u16	als;
	u8	_align;
	u16	als_level_num;
	u16 	als_value_num;
	u32	als_level[C_CUST_ALS_SUB_LEVEL-1];
	u32	als_value[C_CUST_ALS_SUB_LEVEL];

	ulong	enable; 		/*enable mask*/

	/*early suspend*/
#if defined(CONFIG_HAS_EARLYSUSPEND)
	struct early_suspend	early_drv;
#endif     
};
/*----------------------------------------------------------------------------*/

static struct i2c_driver bh1745_i2c_driver = {	
	.probe      = bh1745_i2c_probe,
	.remove     = bh1745_i2c_remove,
	.detect     = bh1745_i2c_detect,
	.suspend    = bh1745_i2c_suspend,
	.resume     = bh1745_i2c_resume,
	.id_table   = bh1745_i2c_id,
	.driver = {
		.name = BH1745_DEV_NAME,
	},
};

/*----------------------------------------------------------------------------*/
static struct i2c_client *bh1745_i2c_client = NULL;
static struct bh1745_priv *bh1745_obj = NULL;

static int bh1745_local_init(void);
static int bh1745_remove(void);
static int bh1745_init_flag =-1; // 0<==>OK -1 <==> fail
static struct als_sub_init_info bh1745_init_info = {
	.name = "bh1745",
	.init = bh1745_local_init,
	.uninit = bh1745_remove,

};

/*----------------------------------------------------------------------------*/

static DEFINE_MUTEX(bh1745_mutex);


/*----------------------------------------------------------------------------*/
typedef enum {
	CMC_BIT_ALS    = 1,
}CMC_BIT;

/*-----------------------------------------------------------------------------*/

int BH1745_i2c_master_operate(struct i2c_client *client, const char *buf, int count, int i2c_flag)
{
	int res = 0;
	mutex_lock(&bh1745_mutex);
	switch(i2c_flag){	
		case I2C_FLAG_WRITE:
			client->addr &=I2C_MASK_FLAG;
			res = i2c_master_send(client, buf, count);
			client->addr &=I2C_MASK_FLAG;
			break;

		case I2C_FLAG_READ:
			client->addr &=I2C_MASK_FLAG;
			client->addr |=I2C_WR_FLAG;
			client->addr |=I2C_RS_FLAG;
			res = i2c_master_send(client, buf, count);
			client->addr &=I2C_MASK_FLAG;
			break;
		default:
			APS_LOG("BH1745_i2c_master_operate i2c_flag command not support!\n");
			break;
	}
	if(res < 0)
	{
		goto EXIT_ERR;
	}
	mutex_unlock(&bh1745_mutex);
	return res;
EXIT_ERR:
	mutex_unlock(&bh1745_mutex);
	APS_ERR("BH1745_i2c_master_operate fail\n");
	return res;
}
/*----------------------------------------------------------------------------*/
static void bh1745_power(struct als_sub_hw *hw, unsigned int on) 
{
	static unsigned int power_on = 0;

	APS_LOG("power %s\n", on ? "on" : "off");

	if(hw->power_id != POWER_NONE_MACRO)
	{
		if(power_on == on)
		{
			APS_LOG("ignore power control: %d\n", on);
		}
		else if(on)
		{
			if(!hwPowerOn(hw->power_id, hw->power_vol, "BH1745")) 
			{
				APS_ERR("power on fails!!\n");
			}
		}
		else
		{
			if(!hwPowerDown(hw->power_id, "BH1745")) 
			{
				APS_ERR("power off fail!!\n");   
			}
		}
	}
	power_on = on;
}


/********************************************************************/
static int bh1745_enable_als(struct i2c_client *client, int enable)
{
	struct bh1745_priv *obj = i2c_get_clientdata(client);
	u8 databuf[2];	  
	int res=0;
	APS_LOG("bh1745_enable_als enable:%d\n", enable);

	databuf[0]= REG_MODECONTROL2;
	res = BH1745_i2c_master_operate(client, databuf, 0x101, I2C_FLAG_READ);
	if(res <= 0)
	{
		return res;
	}
	
	if(enable == 1)
	{
		databuf[1] = databuf[0]|RGBC_EN_ON;
		databuf[0] = REG_MODECONTROL2;
		res = BH1745_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
		if(res <= 0)
		{
			return res;
		}
		atomic_set(&obj->als_deb_on, 1);
		atomic_set(&obj->als_deb_end, jiffies+atomic_read(&obj->als_debounce)/(1000/HZ));
		set_bit(CMC_BIT_ALS, &obj->enable);

		msleep(80);
		bh1745_read_als(obj->client, &obj->als);
		als_sub_report_interrupt_data(obj->als);
		APS_ERR("bh1745_enable_als first data report value:%d\n", obj->als);
	}
	else
	{
		databuf[1] = databuf[0]& (~RGBC_EN_ON);
		databuf[0] = REG_MODECONTROL2;
		res = BH1745_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
		if(res <= 0)
		{
			return res;
		}
		atomic_set(&obj->als_deb_on, 0);
		clear_bit(CMC_BIT_ALS, &obj->enable);
	}
	return 0;
}

/******************************************************************************
 * NAME       : calc_color_temp
 * FUNCTION   : Calculate color temperature
 * REMARKS    :
 * INPUT      : data : each data value from IC (Red, Green, Blue, Clear)
 * RETURN     : color temperature
 *****************************************************************************/
#if 0
#define CALC_E 2.71828183
static double calc_exp(double x)
{
	double t = 1.000000;
	int i ;
	if(0 == x)
	return t;
        for(i=0; i<x;i++)
        t*=CALC_E;
	return t;
}
int calc_color_temp(READ_DATA_ARG data)
{
	double R_adj, G_adj, B_adj, C_adj, RGB_adj;
	double R_ratio, B_ratio;
	double B_eff;
	double cct;
	int final=0;

	R_adj = data.red * 1.0 + 0.0;
	G_adj = data.green * 1.0 + 0.0;
	B_adj = data.blue * 1.0 + 0.0;
	C_adj = data.clear * 1.0 + 0.0;

	RGB_adj = R_adj + G_adj + B_adj;
	if ((G_adj < 1) || (RGB_adj < 1)) 
	{
		cct = 0;
	} 
	else 
	{
		R_ratio = (R_adj) / (RGB_adj);
		B_ratio = (B_adj) / (RGB_adj);

		if ((C_adj/G_adj) < ratio_judgeC) 
		{
			if ((B_ratio*beff[0]) > beff[2])
				B_eff = beff[2];
			else
				B_eff = B_ratio*beff[0];
			
			cct = (1 - B_eff) * cct_eff[0] * calc_exp(color[0] * R_ratio) + B_eff * cct_eff[1] * calc_exp(color[1] * B_ratio);
		} 
		else 
		{
			if ((B_ratio*beff[1]) > beff[2])
				B_eff = beff[2];
			else
				B_eff = B_ratio*beff[1];
			
			cct = (1 - B_eff) * cct_eff[2] * calc_exp(color[2] * R_ratio) + B_eff * cct_eff[3] * calc_exp(color[3] * B_ratio);
		}

		if (cct > 10000) 
		{
			cct = 10000;
		}
	}

	final = (int)cct;
	return final;
}
#endif

/******************************************************************************
 * NAME       : calc_lx
 * FUNCTION   : Calculate lux
 * REMARKS    :
 * INPUT      : data  : each data value from IC (Red, Green, Blue, Clear)
 *            : gain  : gain setting of the sensor
 *            : itime : measurement time of sensor (unit is ms)
 * RETURN     : Lux value
 *****************************************************************************/
int calc_lx(READ_DATA_ARG  data, unsigned char gain, unsigned short itime)
{
	unsigned long lx;
	unsigned long lx_tmp;

	if (data.green < 1) 
	{
		lx = 0;
	} 
	else 
	{
		if ((data.clear*JUDGE_FIXED_COEF) < (judge*data.green))
		{
			
			lx_tmp = red[0] * data.red + green[0] * data.green;
			
		} 
		else 
		{
		    lx_tmp = red[1] * data.red + green[1] * data.green;
		}

		lx = ((lx_tmp*160 *TRANS / gain) / itime) /CUT_UNIT;
	}

	return (lx);
}



/********************************************************************/
int bh1745_read_als(struct i2c_client *client, u16 *data)
{
	long res;
	u8 databuf0[2],databuf1[2],databuf2[2],databuf3[2];
	u8 buf[1];
	unsigned char  gain = 0;
	unsigned short time = 0; 
	unsigned char tmp;
	READ_DATA_ARG read_data;
	CALC_DATA_ARG light_data;
	//struct bh1745_priv *obj = i2c_get_clientdata(client);

	buf[0]= REG_MODECONTROL2;
	res = BH1745_i2c_master_operate(client, buf, 0x101, I2C_FLAG_READ);
	if ((buf[0] & RGBC_VALID_HIGH) == 0)
	{
		return -1; 
	}

	databuf0[0] = REG_RED_DATA;
	res = BH1745_i2c_master_operate(client, databuf0, 0x201, I2C_FLAG_READ);
	if(res < 0)
	{
		APS_ERR("i2c_master_send red function err\n");
		goto READ_ALS_EXIT_ERR;
	}

	databuf1[0] = REG_GREEN_DATA;
	res = BH1745_i2c_master_operate(client, databuf1, 0x201, I2C_FLAG_READ);
	if(res < 0)
	{
		APS_ERR("i2c_master_send green function err\n");
		goto READ_ALS_EXIT_ERR;
	}

	databuf2[0] = REG_BLUE_DATA;
	res = BH1745_i2c_master_operate(client, databuf2, 0x201, I2C_FLAG_READ);
	if(res < 0)
	{
		APS_ERR("i2c_master_send blue function err\n");
		goto READ_ALS_EXIT_ERR;
	}

	databuf3[0] = REG_CLEAR_DATA;
	res = BH1745_i2c_master_operate(client, databuf3, 0x201, I2C_FLAG_READ);
	if(res < 0)
	{
		APS_ERR("i2c_master_send clear function err\n");
		goto READ_ALS_EXIT_ERR;
	}

	read_data.red = (databuf0[1] <<8) |databuf0[0];
	read_data.green = (databuf1[1] <<8) |databuf1[0];
	read_data.blue= (databuf2[1] <<8) |databuf2[0];
	read_data.clear = (databuf3[1] <<8) |databuf3[0];

	buf[0]= REG_MODECONTROL2;
	res = BH1745_i2c_master_operate(client, buf, 0x101, I2C_FLAG_READ);
	if(res <= 0)
	{
		return res;
	}

	tmp = buf[0] & 0x3;
	gain = again[tmp];

	buf[0]= REG_MODECONTROL1;
	res = BH1745_i2c_master_operate(client, buf, 0x101, I2C_FLAG_READ);
	if(res <= 0)
	{
		return res;
	}

	tmp = buf[0] & 0x7;
	time = atime[tmp];
	light_data.lux = calc_lx(read_data, gain, time);
	//light_data.color_temp = calc_color_temp(read_data);
        *data = light_data.lux;

	printk(KERN_INFO "BH1745 als report: red = %d, green = %d, blue = %d, clear = %d, time = %d, gain = %d, lux = %d.\n", read_data.red, 
		read_data.green, read_data.blue, read_data.clear, time, gain, light_data.lux);
	
	return 0;
READ_ALS_EXIT_ERR:
	return res;
}
#if 0
/********************************************************************/
static int bh1745_get_als_value(struct bh1745_priv *obj, u16 als)
{
	int idx;
	int invalid = 0;
	for(idx = 0; idx < obj->als_level_num; idx++)
	{
		if(als < obj->hw->als_level[idx])
		{
			break;
		}
	}
	if(idx >= obj->als_value_num)
	{
		APS_ERR("exceed range\n"); 
		idx = obj->als_value_num - 1;
	}

	if(1 == atomic_read(&obj->als_deb_on))
	{
		unsigned long endt = atomic_read(&obj->als_deb_end);
		if(time_after(jiffies, endt))
		{
			atomic_set(&obj->als_deb_on, 0);
		}

		if(1 == atomic_read(&obj->als_deb_on))
		{
			invalid = 1;
		}
	}

	if(!invalid)
	{
		return obj->hw->als_value[idx];
	}
	else
	{
		return -1;
	}

}
#endif

/*-------------------------------attribute file for debugging----------------------------------*/

/******************************************************************************
 * Sysfs attributes
 *******************************************************************************/
static ssize_t bh1745_show_reg(struct device_driver *ddri, char *buf)
{
	ssize_t len = 0;
	struct i2c_client *client = bh1745_obj->client;
	u8 databuf[2];
	int i,res;	

	for (i = 0; i < BH1745_NUM_CACHABLE_REGS; i++)
	{
		databuf[0] =bh1745_reg[i];
		res = BH1745_i2c_master_operate(client, databuf, 0x101, I2C_FLAG_READ);
		if(res<0)
		{
			APS_ERR("chip id REG 0x%x read error\n", i);
		}
		len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x%x value = 0x%x\n", i, ((databuf[1]<<8)|databuf[0]));
	}
	return len;
}


static ssize_t bh1745_show_als_enable(struct device_driver *ddri, char *buf)
{
	struct bh1745_priv *obj = bh1745_obj;
	ssize_t len = 0;
	bool enable_als = test_bit(CMC_BIT_ALS, &obj->enable);
	APS_FUN();
	len += snprintf(buf + len, PAGE_SIZE - len, "%d\n", enable_als);
	return len;

}

static ssize_t bh1745_store_als_enable(struct device_driver *ddri, const char *buf, size_t count)
{
	struct bh1745_priv *obj = bh1745_obj;
	uint16_t enable=0;
	int res;
	sscanf(buf, "%hu", &enable);
	APS_LOG("bh1745_store_als_enable entry enable:%d !!! \n", enable);
	res = bh1745_enable_als(obj->client, enable);
	if(res)
	{
		APS_ERR("als_enable_nodata is failed!!\n");
	}

	APS_LOG("bh1745_store_als_enable finished !!! \n");
	return count;
}


static ssize_t bh1745_show_als_data(struct device_driver *ddri, char *buf)
{
	struct bh1745_priv *obj = bh1745_obj;
	ssize_t len = 0;
	bh1745_read_als(obj->client, &obj->als);
	len += snprintf(buf + len, PAGE_SIZE - len, "%d\n",  obj->als);
	return len;
}


static ssize_t bh1745_store_reg_write(struct device_driver *ddri, const char *buf, size_t count)
{
	struct bh1745_priv *obj = bh1745_obj;
	int reg;
	int data;
	u8 databuf[2];
	int res;
	APS_FUN();

	sscanf(buf, "%x,%x",&reg, &data);

	APS_LOG("[%s]: reg=0x%x, data=0x%x", __func__, reg, data);
	databuf[1] = data & 0x00FF;
	databuf[0] = reg;
	res = BH1745_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
	if(res < 0)
	{
		APS_ERR("i2c_master_send function err\n");
	}
	return count;
}
/*---------------------------------------------------------------------------------------*/
static DRIVER_ATTR(reg,     S_IWUSR | S_IRUGO, bh1745_show_reg, NULL);
static DRIVER_ATTR(als_enable,	S_IROTH  | S_IWOTH,	 bh1745_show_als_enable, bh1745_store_als_enable);
static DRIVER_ATTR(als_data,	S_IROTH  | S_IWOTH, 	bh1745_show_als_data, NULL);
static DRIVER_ATTR(i2c_w,	S_IROTH  | S_IWOTH, 	NULL, bh1745_store_reg_write );
/*----------------------------------------------------------------------------*/
static struct driver_attribute *bh1745_attr_list[] = {
	&driver_attr_reg,
	&driver_attr_als_enable,
	&driver_attr_als_data,
	&driver_attr_i2c_w,
};

/*----------------------------------------------------------------------------*/
static int bh1745_create_attr(struct device_driver *driver) 
{
	int idx, err = 0;
	int num = (int)(sizeof(bh1745_attr_list)/sizeof(bh1745_attr_list[0]));
	if (driver == NULL)
	{
		return -EINVAL;
	}

	for(idx = 0; idx < num; idx++)
	{
		if((err = driver_create_file(driver, bh1745_attr_list[idx])))
		{            
			APS_ERR("driver_create_file (%s) = %d\n", bh1745_attr_list[idx]->attr.name, err);
			break;
		}
	}    
	return err;
}
/*----------------------------------------------------------------------------*/
static int bh1745_delete_attr(struct device_driver *driver)
{
	int idx ,err = 0;
	int num = (int)(sizeof(bh1745_attr_list)/sizeof(bh1745_attr_list[0]));

	if (!driver)
		return -EINVAL;

	for (idx = 0; idx < num; idx++) 
	{
		driver_remove_file(driver, bh1745_attr_list[idx]);
	}

	return err;
}
/*----------------------------------------------------------------------------*/


/*-------------------------------MISC device related------------------------------------------*/
static int bh1745_open(struct inode *inode, struct file *file)
{
	file->private_data = bh1745_i2c_client;

	if (!file->private_data)
	{
		APS_ERR("null pointer!!\n");
		return -EINVAL;
	}
	return nonseekable_open(inode, file);
}
/************************************************************/
static int bh1745_release(struct inode *inode, struct file *file)
{
	file->private_data = NULL;
	return 0;
}
/************************************************************/

/************************************************PS CALI*****************************************************************************/

static long bh1745_unlocked_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct i2c_client *client = (struct i2c_client*)file->private_data;
	struct bh1745_priv *obj = i2c_get_clientdata(client);  
	long err = 0;
	void __user *ptr = (void __user*) arg;
	int dat;
	uint32_t enable;

	switch (cmd)
	{
		
		case ALSPS_SET_ALS_MODE:

			if(copy_from_user(&enable, ptr, sizeof(enable)))
			{
				err = -EFAULT;
				goto err_out;
			}
			if(enable)
			{
				if((err = bh1745_enable_als(obj->client, 1)))
				{
					APS_ERR("enable als fail: %ld\n", err); 
					goto err_out;
				}
				set_bit(CMC_BIT_ALS, &obj->enable);
			}
			else
			{
				if((err = bh1745_enable_als(obj->client, 0)))
				{
					APS_ERR("disable als fail: %ld\n", err); 
					goto err_out;
				}
				clear_bit(CMC_BIT_ALS, &obj->enable);
			}
			break;

		case ALSPS_GET_ALS_MODE:
			enable = test_bit(CMC_BIT_ALS, &obj->enable) ? (1) : (0);
			if(copy_to_user(ptr, &enable, sizeof(enable)))
			{
				err = -EFAULT;
				goto err_out;
			}
			break;

		case ALSPS_GET_ALS_DATA: 

			APS_ERR("bh1745 get als data enter\n"); 
			if((err = bh1745_read_als(obj->client, &obj->als)))
			{
				goto err_out;
			}			

			dat =obj->als;
			if(copy_to_user(ptr, &dat, sizeof(dat)))
			{
				err = -EFAULT;
				goto err_out;
			}			   
			break;

		case ALSPS_GET_ALS_RAW_DATA:	
			APS_ERR("bh1745 get als raw data enter\n"); 
			if((err = bh1745_read_als(obj->client, &obj->als)))
			{
				goto err_out;
			}

			dat = obj->als;
			if(copy_to_user(ptr, &dat, sizeof(dat)))
			{
				err = -EFAULT;
				goto err_out;
			}			   
			break;


		default:
			APS_ERR("%s not supported = 0x%04x", __FUNCTION__, cmd);
			err = -ENOIOCTLCMD;
			break;
	}

err_out:
	return err;    
}
/*------------------------------misc device related operation functions------------------------------------*/
static struct file_operations bh1745_fops = {
	.owner = THIS_MODULE,
	.open = bh1745_open,
	.release = bh1745_release,
	.unlocked_ioctl = bh1745_unlocked_ioctl,
};

static struct miscdevice bh1745_device = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "als_sub",
	.fops = &bh1745_fops,
};

/*--------------------------------------------------------------------------------------*/
static void bh1745_early_suspend(struct early_suspend *h)
{
	struct bh1745_priv *obj = container_of(h, struct bh1745_priv, early_drv);	
	int err;
	APS_LOG("bh1745_early_suspend entry!!!\n"); 
	if(!obj)
	{
		APS_ERR("null pointer!!\n");
		return;
	}
	if (1 == test_bit(CMC_BIT_ALS, &obj->enable))
	{
		if(err = bh1745_enable_als(obj->client, 0))
		{
			APS_ERR("bh1745_early_suspend disable als fail: %d\n", err); 
		}
		atomic_set(&obj->als_suspend, 1);
	}
	APS_LOG("bh1745_early_suspend finished!!!\n"); 
}

static void bh1745_late_resume(struct early_suspend *h) 
{
	struct bh1745_priv *obj = container_of(h, struct bh1745_priv, early_drv);		  
	int err;
	APS_LOG("bh1745_late_resume entry!!!\n"); 
	if(!obj)
	{
		APS_ERR("null pointer!!\n");
		return;
	}
	if(1 == atomic_read(&obj->als_suspend))
	{
		if((err = bh1745_enable_als(obj->client, 1)))
		{
			APS_ERR("enable als fail: %d\n", err);		  

		}
		atomic_set(&obj->als_suspend, 0);
	}
	APS_LOG("bh1745_late_resume finished!!!\n"); 
}
/*--------------------------------------------------------------------------------*/
static int bh1745_init_client(struct i2c_client *client)
{
	struct bh1745_priv *obj = i2c_get_clientdata(client);  
	int res = 0;
	u8 databuf[2];
	APS_FUN();


	databuf[1] = SW_RESET|INT_RESET;
	databuf[0] = REG_SYSTEMCONTROL;
	res = BH1745_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
	if(res < 0)
	{
		APS_ERR("i2c_master_send function err\n");
		goto EXIT_ERR;
	}

	databuf[1] = RGB_SET_MODE_CONTROL1;
	databuf[0] = REG_MODECONTROL1;
	res = BH1745_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
	if(res < 0)
	{
		APS_ERR("i2c_master_send function err\n");
		goto EXIT_ERR;
	}

	databuf[1] = RGB_SET_MODE_CONTROL2;
	databuf[0] = REG_MODECONTROL2;
	res = BH1745_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
	if(res < 0)
	{
		APS_ERR("i2c_master_send function err\n");
		goto EXIT_ERR;
	}

	databuf[1] = 0x3;
	databuf[0] = REG_MODECONTROL3;
	res = BH1745_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
	if(res < 0)
	{
		APS_ERR("i2c_master_send function err\n");
		goto EXIT_ERR;
	}

	return 0;

EXIT_ERR:
	APS_ERR("init dev: %d\n", res);
	return res;
}
/*--------------------------------------------------------------------------------*/

// if use  this typ of enable , Gsensor should report inputEvent(x, y, z ,stats, div) to HAL
static int als_open_report_data(int open)
{
	//should queuq work to report event if  is_report_input_direct=true
	return 0;
}

// if use  this typ of enable , Gsensor only enabled but not report inputEvent to HAL

static int als_enable_nodata(int en)
{
	struct bh1745_priv *obj = bh1745_obj;
	int res = 0;
	APS_LOG("bh1745_obj als enable value = %d\n", en);

	if(!obj)
	{
		APS_ERR("bh1745_obj is null!!\n");
		return -1;
	}
	res=	bh1745_enable_als(obj->client, en);
	if(res)
	{
		APS_ERR("als_enable_nodata is failed!!\n");
		return -1;
	}
	return 0;
}

static int als_set_delay(u64 ns)
{
	return 0;
}

static int als_get_data(int* value, int* status)
{
	struct bh1745_priv *obj = bh1745_obj;
	int err = 0;

	if(!obj)
	{
		APS_ERR("bh1745_obj is null!!\n");
		return -1;
	}
	if((err = bh1745_read_als(obj->client, &obj->als)))
	{
		err = -1;
	}
	else
	{
		//APS_LOG("als_get_data value = %d\n", obj->als);
		*value = obj->als;//bh1745_get_als_value(obj, obj->als);
		*status = SENSOR_STATUS_ACCURACY_MEDIUM;
	}
	return err;
}

// if use  this typ of enable , Gsensor only enabled but not report inputEvent to HAL

/*-----------------------------------i2c operations----------------------------------*/
static int bh1745_i2c_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	struct bh1745_priv *obj;
	int err = 0;
	struct als_sub_control_path als_ctl={0};
	struct als_sub_data_path als_data={0};
	int databuf[2];
	APS_LOG("bh1745_i2c_probe entry!!!\n");
	databuf[0] = REG_MANUFACT_ID;
	err = BH1745_i2c_master_operate(client, databuf, 0x101, I2C_FLAG_READ);
	if(databuf[0] != 0XE0)
	{
		APS_ERR("probe REG 0x00 read error\n");
		goto exit;
	}
	if(!(obj = kzalloc(sizeof(*obj), GFP_KERNEL)))
	{
		err = -ENOMEM;
		goto exit;
	}

	memset(obj, 0, sizeof(*obj));
	bh1745_obj = obj;

	obj->hw = get_cust_als_sub_hw();//get custom file data struct

	obj->client = client;
	i2c_set_clientdata(client, obj);
	/*-----------------------------value need to be confirmed-----------------------------------------*/
	atomic_set(&obj->als_debounce, 200);
	atomic_set(&obj->als_deb_on, 0);
	atomic_set(&obj->als_deb_end, 0);
	atomic_set(&obj->als_suspend, 0);

	obj->enable = 0;
	obj->als_level_num = sizeof(obj->hw->als_level)/sizeof(obj->hw->als_level[0]);
	obj->als_value_num = sizeof(obj->hw->als_value)/sizeof(obj->hw->als_value[0]);
	/*-----------------------------value need to be confirmed-----------------------------------------*/

	BUG_ON(sizeof(obj->als_level) != sizeof(obj->hw->als_level));
	memcpy(obj->als_level, obj->hw->als_level, sizeof(obj->als_level));
	BUG_ON(sizeof(obj->als_value) != sizeof(obj->hw->als_value));
	memcpy(obj->als_value, obj->hw->als_value, sizeof(obj->als_value));
	atomic_set(&obj->i2c_retry, 3);
	bh1745_i2c_client = client;

	if((err = bh1745_init_client(client)))
	{
		goto exit_init_failed;
	}
	APS_LOG("bh1745_init_client() OK!\n");

	if((err = misc_register(&bh1745_device)))
	{
		APS_ERR("bh1745_device register failed\n");
		goto exit_misc_device_register_failed;
	}
	APS_LOG("bh1745_device misc_register OK!\n");

	/*------------------------bh1745 attribute file for debug--------------------------------------*/
	if((err = bh1745_create_attr(&(bh1745_init_info.platform_diver_addr->driver))))
	{
		APS_ERR("create attribute err = %d\n", err);
		goto exit_create_attr_failed;
	}
	/*------------------------bh1745 attribute file for debug--------------------------------------*/
	als_ctl.open_report_data= als_open_report_data;
	als_ctl.enable_nodata = als_enable_nodata;
	als_ctl.set_delay  = als_set_delay;
	als_ctl.is_report_input_direct = false;
#ifdef CUSTOM_KERNEL_SENSORHUB
	als_ctl.is_support_batch = obj->hw->is_batch_supported_als;
#else
	als_ctl.is_support_batch = false;
#endif

	err = als_sub_register_control_path(&als_ctl);
	if(err)
	{
		APS_ERR("register fail = %d\n", err);
		goto exit_sensor_obj_attach_fail;
	}

	als_data.get_data = als_get_data;
	als_data.vender_div = 100;
	err = als_sub_register_data_path(&als_data);	
	if(err)
	{
		APS_ERR("tregister fail = %d\n", err);
		goto exit_sensor_obj_attach_fail;
	}
#if defined(CONFIG_HAS_EARLYSUSPEND) && defined(CONFIG_EARLYSUSPEND)
	obj->early_drv.level  = EARLY_SUSPEND_LEVEL_STOP_DRAWING - 2,
		obj->early_drv.suspend  = bh1745_early_suspend,
		obj->early_drv.resume  = bh1745_late_resume,    
		register_early_suspend(&obj->early_drv);
#endif

	bh1745_init_flag =0;
	return 0;

exit_create_attr_failed:
exit_sensor_obj_attach_fail:
exit_misc_device_register_failed:
	misc_deregister(&bh1745_device);
exit_init_failed:
	kfree(obj);
exit:
	bh1745_i2c_client = NULL;           
	APS_ERR("%s: err = %d\n", __func__, err);
	bh1745_init_flag = -1;
	return err;
}

static int bh1745_i2c_remove(struct i2c_client *client)
{
	int err;	
	/*------------------------bh1745 attribute file for debug--------------------------------------*/	
	if((err = bh1745_delete_attr(&(bh1745_init_info.platform_diver_addr->driver))))
	{
		APS_ERR("bh1745_delete_attr fail: %d\n", err);
	} 
	/*----------------------------------------------------------------------------------------*/

	if((err = misc_deregister(&bh1745_device)))
	{
		APS_ERR("misc_deregister fail: %d\n", err);    
	}

	bh1745_i2c_client = NULL;
	i2c_unregister_device(client);
	kfree(i2c_get_clientdata(client));
	return 0;

}

static int bh1745_i2c_detect(struct i2c_client *client, struct i2c_board_info *info)
{
	strcpy(info->type, BH1745_DEV_NAME);
	return 0;

}

static int bh1745_i2c_suspend(struct i2c_client *client, pm_message_t msg)
{
	APS_FUN();
	return 0;
}

static int bh1745_i2c_resume(struct i2c_client *client)
{
	APS_FUN();
	return 0;
}
/*----------------------------------------------------------------------------*/
static int bh1745_remove(void)
{
	struct als_sub_hw *hw = get_cust_als_sub_hw();
	bh1745_power(hw, 0);
	i2c_del_driver(&bh1745_i2c_driver);
	return 0;
}
/*----------------------------------------------------------------------------*/
static int  bh1745_local_init(void)
{
	struct als_sub_hw *hw = get_cust_als_sub_hw();
	APS_LOG("bh1745_local_init entry!!!\n");
	bh1745_power(hw, 1);
	if(i2c_add_driver(&bh1745_i2c_driver))
	{
		APS_ERR("add driver error\n");
		return -1;
	}
	if(-1 == bh1745_init_flag)
	{
		APS_ERR("bh1745_local_init failed!!!\n");
		return -1;
	}
	APS_LOG("bh1745_local_init finished!!!\n");
	return 0;
}
/*----------------------------------------------------------------------------*/
static int __init bh1745_init(void)
{
	struct als_sub_hw *hw = get_cust_als_sub_hw();
	APS_LOG("%s: i2c_number=%d\n", __func__, hw->i2c_num);
	i2c_register_board_info(hw->i2c_num, &i2c_bh1745, 1);
	als_sub_driver_add(&bh1745_init_info);
	APS_LOG("%s finished!!!\n", __func__);
	return 0;
}
/*----------------------------------------------------------------------------*/
static void __exit bh1745_exit(void)
{
	APS_FUN();
}
/*----------------------------------------------------------------------------*/
module_init(bh1745_init);
module_exit(bh1745_exit);
/*----------------------------------------------------------------------------*/
MODULE_AUTHOR("yucong xiong");
MODULE_DESCRIPTION("bh1745 driver");
MODULE_LICENSE("GPL");
