/* drivers/hwmon/mt6516/amit/epl252x.c - EPL252x ALS/PS driver
 *
 * Author: MingHsien Hsieh <minghsien.hsieh@mediatek.com>
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

/** VERSION: 1.2.0**/

#include <linux/interrupt.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/irq.h>
#include <linux/miscdevice.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <linux/input.h>
#include <linux/workqueue.h>
#include <linux/kobject.h>
#include <linux/earlysuspend.h>
#include <linux/platform_device.h>
#include <asm/atomic.h>

#include <linux/hwmsensor.h>
#include <linux/hwmsen_dev.h>
#include <linux/sensors_io.h>
#include <asm/io.h>
#include <cust_eint.h>
#include <cust_alsps.h>
#include <linux/hwmsen_helper.h>
#include "epl252x.h"

#include <mach/mt_typedefs.h>
#include <mach/mt_gpio.h>
#include <mach/mt_pm_ldo.h>
//add for fix resume issue
#include <linux/earlysuspend.h>
#include <linux/wakelock.h>
#include <linux/sched.h>
//add for fix resume issue end
#include <alsps.h>

#ifdef CUSTOM_KERNEL_SENSORHUB
#include <SCP_sensorHub.h>
#endif
/******************************************************************************
 * extern functions
 *******************************************************************************/
extern void mt_eint_mask(unsigned int eint_num);
extern void mt_eint_unmask(unsigned int eint_num);
extern void mt_eint_set_hw_debounce(unsigned int eint_num, unsigned int ms);
extern void mt_eint_set_polarity(unsigned int eint_num, unsigned int pol);
extern unsigned int mt_eint_set_sens(unsigned int eint_num, unsigned int sens);
extern void mt_eint_registration(unsigned int eint_num, unsigned int flow, void (EINT_FUNC_PTR)(void), unsigned int is_auto_umask);
extern void mt_eint_print_status(void);
/*-------------------------MT6516&MT6575 define-------------------------------*/
#define POWER_NONE_MACRO MT65XX_POWER_NONE

/******************************************************************************
 *  configuration
 ******************************************************************************/
#define NO_DATA         1 //0
#define LUX_PER_COUNT			400//700		// 0.7

#define COMMON_DEBUG    0
#define ALS_DEBUG       0
/*for als dync algo molg1 add 20150209 begin*/
#define ALS_DYN_INTT    1
#define ALS_DYN_INTT_FAST   1
/*for als dync algo molg1 add 20150209 end*/
/*******************************************************************/
#define TXBYTES 				       2
#define RXBYTES					2
#define PACKAGE_SIZE 			48
#define I2C_RETRY_COUNT 		2
#define EPL_DEV_NAME   		    "EPL252X"
#define DRIVER_VERSION          "1.2.0"

struct input_dev dev;
struct hwmsen_object *ps_hw, * als_hw;
//static struct epl_sensor_priv *epl_sensor_obj = NULL;
static struct platform_driver epl_sensor_alsps_driver;
static struct wake_lock ps_lock;
static struct mutex sensor_mutex;
static int intr_flag_value = 0;
#if ALS_DYN_INTT
//Dynamic INTT
int dynamic_intt_idx;
int dynamic_intt_init_idx = 1;	//initial dynamic_intt_idx
int c_gain;
int dynamic_intt_lux = 0;

uint16_t dynamic_intt_high_thr;
uint16_t dynamic_intt_low_thr;
uint32_t dynamic_intt_max_lux = 12000;
uint32_t dynamic_intt_min_lux = 0;
uint32_t dynamic_intt_min_unit = 1000;

static int als_dynamic_intt_intt[] = {EPL_ALS_INTT_8192, EPL_ALS_INTT_64, EPL_ALS_INTT_32};
static int als_dynamic_intt_value[] = {8192, 64, 32};
static int als_dynamic_intt_gain[] = {EPL_GAIN_MID, EPL_GAIN_MID, EPL_GAIN_MID};
static int als_dynamic_intt_high_thr[] = {60000, 53000, 50000};
static int als_dynamic_intt_low_thr[] = {200, 200, 2560};
static int als_dynamic_intt_intt_num =  sizeof(als_dynamic_intt_value)/sizeof(int);
#if ALS_DYN_INTT_FAST
bool dynamic_intt_fast_idx = 0;
#endif
#endif
/*lenovo-sw molg1 add 20150128 begin */
#define I2C_FLAG_WRITE	0
#define I2C_FLAG_READ	1
static DEFINE_MUTEX(epl252x_mutex);
/*lenovo-sw molg1 add 20150128 end */

/*----------------------------------------------------------------------------*/
#define APS_TAG                 	  	"[ALS/PS] "
#define APS_FUN(f)              	  	printk(KERN_INFO APS_TAG"%s\n", __FUNCTION__)
#define APS_ERR(fmt, args...)    	    printk(KERN_ERR  APS_TAG"%s %d : "fmt, __FUNCTION__, __LINE__, ##args)
#define APS_LOG(fmt, args...)    printk(KERN_DEBUG APS_TAG fmt, ##args)
#define APS_DBG(fmt, args...)    printk(KERN_DEBUG fmt, ##args)
/*----------------------------------------------------------------------------*/
static struct i2c_client *epl_sensor_i2c_client = NULL;
/*----------------------------------------------------------------------------*/
static const struct i2c_device_id epl_sensor_i2c_id[] = {{EPL_DEV_NAME,0},{}};
static struct i2c_board_info __initdata i2c_epl_sensor= { I2C_BOARD_INFO(EPL_DEV_NAME, (0x92>>1))};
/*----------------------------------------------------------------------------*/
static int epl_sensor_i2c_probe(struct i2c_client *client, const struct i2c_device_id *id);
static int epl_sensor_i2c_remove(struct i2c_client *client);
static int epl_sensor_i2c_suspend(struct i2c_client *client, pm_message_t msg);
static int epl_sensor_i2c_resume(struct i2c_client *client);
static int set_psensor_threshold(struct i2c_client *client);
#ifndef CUSTOM_KERNEL_SENSORHUB
static void epl_sensor_eint_func(void);
#endif
static int set_psensor_intr_threshold(uint16_t low_thd, uint16_t high_thd);
static int set_lsensor_intr_threshold(uint16_t low_thd, uint16_t high_thd);
static int ps_enable_nodata(int enable);
static int als_enable_nodata(int enable);
static int ps_sensing_time(int intt, int adc, int cycle);
static int als_sensing_time(int intt, int adc, int cycle);
static int alsps_local_init(void);
static int alsps_remove(void);
/*----------------------------------------------------------------------------*/
typedef enum
{
	CMC_TRC_ALS_DATA = 0x0001,
	CMC_TRC_PS_DATA = 0X0002,
	CMC_TRC_EINT    = 0x0004,
	CMC_TRC_IOCTL   = 0x0008,
	CMC_TRC_I2C     = 0x0010,
	CMC_TRC_CVT_ALS = 0x0020,
	CMC_TRC_CVT_PS  = 0x0040,
	CMC_TRC_DEBUG   = 0x0800,
} CMC_TRC;

/*----------------------------------------------------------------------------*/
typedef enum
{
	CMC_BIT_ALS   	= 1,
	CMC_BIT_PS     	= 2,
} CMC_BIT;
/*----------------------------------------------------------------------------*/
struct epl_sensor_i2c_addr      /*define a series of i2c slave address*/
{
	u8  write_addr;
	u8  ps_thd;     /*PS INT threshold*/
};

/*----------------------------------------------------------------------------*/
struct epl_sensor_priv
{
	struct alsps_hw  *hw;
	struct i2c_client *client;
	struct delayed_work  eint_work;
	struct input_dev *gs_input_dev;
	struct input_dev *hb_input_dev;

	/*i2c address group*/
	struct epl_sensor_i2c_addr  addr;

	/*misc*/
	atomic_t    als_suspend;
	atomic_t    ps_suspend;

	atomic_t    trace;
	atomic_t    i2c_retry;
	atomic_t    als_debounce;   /*debounce time after enabling als*/
	atomic_t    als_deb_on;     /*indicates if the debounce is on*/
	atomic_t    als_deb_end;    /*the jiffies representing the end of debounce*/
	atomic_t    ps_mask;        /*mask ps: always return far away*/
	atomic_t    ps_debounce;    /*debounce time after enabling ps*/
	atomic_t    ps_deb_on;      /*indicates if the debounce is on*/
	atomic_t    ps_deb_end;     /*the jiffies representing the end of debounce*/

	/*data*/
	u16		lux_per_count;
	bool   	als_enable;    /*record current als status*/
	bool    	ps_enable;     /*record current ps status*/
	ulong       enable;         	/*record HAL enalbe status*/
	ulong      	pending_intr;   	/*pending interrupt*/

	/*data*/
	u16         als;
	u16         ps;
	u16         als_level_num;
	u16         als_value_num;
	u32         als_level[C_CUST_ALS_LEVEL-1];
	u32         als_value[C_CUST_ALS_LEVEL];
	atomic_t	ps_thd_val_high;	 /*the cmd value can't be read, stored in ram*/
	atomic_t	ps_thd_val_low; 	/*the cmd value can't be read, stored in ram*/
	atomic_t	ps_ftm_cali_high;  /*used for backup factory value*/
	atomic_t	ps_ftm_cali_low;	/*used for backup factory value*/
	atomic_t	ps_eng_cali_high; /*the value is end of ps threshold high*/
	atomic_t	ps_eng_cali_low;  /*the value is end of ps threshold low*/
	atomic_t	ps_cali_high;   /*the value is Engineer Mode calibration  threshold high*/
	atomic_t	ps_cali_low; 	 /*the value is Engineer Mode calibration  threshold high*/
	atomic_t	ps_threshold_high;  /*the cmd value can't be read, stored in ram*/
	atomic_t	ps_threshold_low;  /*the cmd value can't be read, stored in ram*/
	/*early suspend*/
#if defined(CONFIG_HAS_EARLYSUSPEND)
	struct early_suspend    early_drv;
#endif
};

typedef struct _epl_raw_data
{
	u8 raw_bytes[PACKAGE_SIZE];
} epl_raw_data;

static epl_optical_sensor epl_sensor;
int i2c_max_count=8;

static struct epl_sensor_priv *epl_sensor_obj = NULL;
static epl_raw_data gRawData;
static int alsps_init_flag =-1; // 0<==>OK -1 <==> fail

static struct alsps_init_info epl_sensor_init_info = {
	.name = EPL_DEV_NAME,
	.init = alsps_local_init,
	.uninit = alsps_remove,
};

/*----------------------------------------------------------------------------*/
static struct i2c_driver epl_sensor_i2c_driver =
{
	.probe     	= epl_sensor_i2c_probe,
	.remove     = epl_sensor_i2c_remove,
	.suspend    = epl_sensor_i2c_suspend,
	.resume     = epl_sensor_i2c_resume,
	.id_table   = epl_sensor_i2c_id,
	.driver = {
		.name   = EPL_DEV_NAME,
	},
};

static int epl_sensor_I2C_Write_Cmd(struct i2c_client *client, uint8_t regaddr, uint8_t data, uint8_t txbyte)
{
	uint8_t buffer[2];
	int ret = 0;
	int retry;

	buffer[0] = regaddr;
	buffer[1] = data;

	for(retry = 0; retry < I2C_RETRY_COUNT; retry++)
	{
		ret = i2c_master_send(client, buffer, txbyte);
		if (ret == txbyte)
		{
			break;
		}
		APS_ERR("i2c write error,TXBYTES %d\n",ret);
		mdelay(10);
	}
	if(retry>=I2C_RETRY_COUNT)
	{
		APS_ERR("i2c write retry over %d\n", I2C_RETRY_COUNT);
		return -EINVAL;
	}
	return ret;
}

static int epl_sensor_I2C_Write(struct i2c_client *client, uint8_t regaddr, uint8_t data)
{
	int ret = 0;
	ret = epl_sensor_I2C_Write_Cmd(client, regaddr, data, 0x02);
	return ret;
}

int epl252x_i2c_master_operate(struct i2c_client *client, const char *buf, int count, int i2c_flag)
{
	int res = 0;
	mutex_lock(&epl252x_mutex);
	switch(i2c_flag){	
		case I2C_FLAG_WRITE:
			client->addr &=I2C_MASK_FLAG;
			res = i2c_master_send(client, buf, count);
			client->addr &=I2C_MASK_FLAG;
			break;

		case I2C_FLAG_READ:
			client->addr &=I2C_MASK_FLAG;
			client->addr |=I2C_WR_FLAG;
			client->addr |=I2C_RS_FLAG;
			res = i2c_master_send(client, buf, count);
			client->addr &=I2C_MASK_FLAG;
			break;
		default:
			APS_LOG("epl252x_i2c_master_operate  i2c_flag command not support!\n");
			break;
	}
	if(res <= 0)
	{
		goto EXIT_ERR;
	}
	mutex_unlock(&epl252x_mutex);
	return res;
EXIT_ERR:
	mutex_unlock(&epl252x_mutex);
	APS_ERR("epl252x_i2c_transfer fail\n");
	return res;
}

static int epl_sensor_I2C_Read(struct i2c_client *client, uint8_t regaddr, uint8_t bytecount)
{
	int ret = 0;
	int retry;
	int read_count=0, rx_count=0;

	while(bytecount>0)
	{
		epl_sensor_I2C_Write_Cmd(client, regaddr+read_count, 0x00, 0x01);

		for(retry = 0; retry < I2C_RETRY_COUNT; retry++)
		{
			rx_count = bytecount>i2c_max_count?i2c_max_count:bytecount;
			ret = i2c_master_recv(client, &gRawData.raw_bytes[read_count], rx_count);

			if (ret == rx_count)
				break;

			APS_ERR("i2c read error,RXBYTES %d\r\n",ret);
			mdelay(10);
		}

		if(retry>=I2C_RETRY_COUNT)
		{
			APS_ERR("i2c read retry over %d\n", I2C_RETRY_COUNT);
			return -EINVAL;
		}
		bytecount-=rx_count;
		read_count+=rx_count;
	}

	return ret;
}

#if ALS_DYN_INTT
long raw_convert_to_lux(u16 raw_data)
{
	long lux = 0;

	lux = raw_data * (c_gain / als_dynamic_intt_value[dynamic_intt_idx]);
#if ALS_DEBUG
	APS_LOG("[%s]:raw_data=%d, lux=%ld\r\n", __func__, raw_data, lux);
#endif

	if(lux >= (dynamic_intt_max_lux*dynamic_intt_min_unit)){
#if ALS_DEBUG
		APS_LOG("[%s]:raw_convert_to_lux: change max lux\r\n", __func__);
#endif
		lux = dynamic_intt_max_lux * dynamic_intt_min_unit;
	}
	else if(lux <= (dynamic_intt_min_lux*dynamic_intt_min_unit)){
#if ALS_DEBUG
		APS_LOG("[%s]:raw_convert_to_lux: change min lux\r\n", __func__);
#endif
		lux = dynamic_intt_min_lux * dynamic_intt_min_unit;
	}
	return lux;
}
static int epl252x_sensor_get_als_value(struct epl_sensor_priv *obj, u16 als)
{
	long now_lux=0, lux_tmp=0;
	bool change_flag = false;
#if ALS_DYN_INTT
	u8 databuf[2];
#endif
	if(als > dynamic_intt_high_thr)
	{
		if(dynamic_intt_idx == (als_dynamic_intt_intt_num - 1))
		{
			als = dynamic_intt_high_thr;
			lux_tmp = raw_convert_to_lux(als);
		}
		else
		{
			change_flag = true;
			als  = dynamic_intt_high_thr;
			lux_tmp = raw_convert_to_lux(als);
			dynamic_intt_idx++;
		}
	}
	else if(als < dynamic_intt_low_thr)
	{
		if(dynamic_intt_idx == 0)
		{
			lux_tmp = raw_convert_to_lux(als);
		}
		else
		{
			change_flag = true;
			als  = dynamic_intt_low_thr;
			lux_tmp = raw_convert_to_lux(als);
			dynamic_intt_idx--;
		}
	}
	else
	{
		lux_tmp = raw_convert_to_lux(als);
	}
	now_lux = lux_tmp;
	dynamic_intt_lux = now_lux/dynamic_intt_min_unit;
#if ALS_DYN_INTT_FAST
	if(dynamic_intt_fast_idx == (als_dynamic_intt_intt_num - 1))
	{
		epl_sensor.als.cycle = EPL_CYCLE_16;
		APS_LOG("[%s]: ALS normal mode \r\n", __func__);
		databuf[1] = epl_sensor.als.adc | epl_sensor.als.cycle;
		databuf[0] = 0x02;
		epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
	}
#endif

	if(change_flag == true)
	{
		int als_time = 0, ps_time = 0, total_time = 0;
		u8 databuf[2];
		bool enable_ps = test_bit(CMC_BIT_PS, &obj->enable);
		bool enable_als = test_bit(CMC_BIT_ALS, &obj->enable);
		als_time = als_sensing_time(epl_sensor.als.integration_time, epl_sensor.als.adc, epl_sensor.als.cycle);
		ps_time = ps_sensing_time(epl_sensor.ps.integration_time, epl_sensor.ps.adc, epl_sensor.ps.cycle);
		APS_LOG("[%s]: ps=%d, als=%d \r\n", __func__, enable_ps, enable_als);

		epl_sensor.als.integration_time = als_dynamic_intt_intt[dynamic_intt_idx];
		epl_sensor.als.gain = als_dynamic_intt_gain[dynamic_intt_idx];
		dynamic_intt_high_thr = als_dynamic_intt_high_thr[dynamic_intt_idx];
		dynamic_intt_low_thr = als_dynamic_intt_low_thr[dynamic_intt_idx];
		databuf[1] = 0x00;
		databuf[0] = 0x00;
		epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);

		databuf[1] = epl_sensor.als.integration_time | epl_sensor.als.gain;
		databuf[0] = 0x01;
		epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
		if(enable_ps == 0 && enable_als == 1)
		{
			databuf[1] = 0x01;
			total_time = als_time;
		}
		else if(enable_ps == 1 && enable_als == 0)
		{
			databuf[1] = 0x02;
			total_time = ps_time;
		}
		else if(enable_ps == 1 && enable_als == 1)
		{
			databuf[1] = 0x03;
			total_time = ps_time + als_time + wait_value[epl_sensor.wait>>4];
		}
		databuf[0] = 0x00;
		epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
		msleep(total_time);
		APS_LOG("[%s]: msleep total_time(%d) ps_time(%d) als_time(%d) \r\n", __func__, total_time, ps_time , als_time);
		dynamic_intt_lux = -1;
		change_flag = false;
	}
	return dynamic_intt_lux;
}
#endif
static int set_psensor_intr_threshold(uint16_t low_thd, uint16_t high_thd)
{
	struct epl_sensor_priv *epld = epl_sensor_obj;
	struct i2c_client *client = epld->client;
	uint8_t high_msb ,high_lsb, low_msb, low_lsb;


	high_msb = (uint8_t) (high_thd >> 8);
	high_lsb = (uint8_t) (high_thd & 0x00ff);
	low_msb  = (uint8_t) (low_thd >> 8);
	low_lsb  = (uint8_t) (low_thd & 0x00ff);

	APS_LOG("%s: low_thd = %d, high_thd = %d \n", __FUNCTION__, low_thd, high_thd);

	epl_sensor_I2C_Write(client, 0x0c, low_lsb);
	epl_sensor_I2C_Write(client, 0x0d, low_msb);
	epl_sensor_I2C_Write(client, 0x0e, high_lsb);
	epl_sensor_I2C_Write(client, 0x0f, high_msb);

	return 0;
}

static int set_lsensor_intr_threshold(uint16_t low_thd, uint16_t high_thd)
{
	struct epl_sensor_priv *epld = epl_sensor_obj;
	struct i2c_client *client = epld->client;
	uint8_t high_msb ,high_lsb, low_msb, low_lsb;

	high_msb = (uint8_t) (high_thd >> 8);
	high_lsb = (uint8_t) (high_thd & 0x00ff);
	low_msb  = (uint8_t) (low_thd >> 8);
	low_lsb  = (uint8_t) (low_thd & 0x00ff);

	epl_sensor_I2C_Write(client, 0x08, low_lsb);
	epl_sensor_I2C_Write(client, 0x09, low_msb);
	epl_sensor_I2C_Write(client, 0x0a, high_lsb);
	epl_sensor_I2C_Write(client, 0x0b, high_msb);

	APS_LOG("%s: low_thd = %d, high_thd = %d \n", __FUNCTION__, low_thd, high_thd);

	return 0;
}
/*----------------------------------------------------------------------------*/
int hw8k_init_device(struct i2c_client *client)
{
	APS_LOG("hw8k_init_device.........\r\n");

	epl_sensor_i2c_client = client;

	APS_LOG(" I2C Addr==[0x%x],line=%d\n", epl_sensor_i2c_client->addr,__LINE__);

	return 0;
}
/*----------------------------------------------------------------------------*/
static void epl_sensor_power(struct alsps_hw *hw, unsigned int on)
{
#ifndef FPGA_EARLY_PORTING
	static unsigned int power_on = 0;

	//APS_LOG("power %s\n", on ? "on" : "off");

	if(hw->power_id != POWER_NONE_MACRO)
	{
		if(power_on == on)
		{
			APS_LOG("ignore power control: %d\n", on);
		}
		else if(on)
		{
			if(!hwPowerOn(hw->power_id, hw->power_vol, EPL_DEV_NAME))
			{
				APS_ERR("power on fails!!\n");
			}
		}
		else
		{
			if(!hwPowerDown(hw->power_id, EPL_DEV_NAME))
			{
				APS_ERR("power off fail!!\n");
			}
		}
	}
	power_on = on;
#endif //#ifndef FPGA_EARLY_PORTING
}

static int als_sensing_time(int intt, int adc, int cycle)
{
	long sensing_us_time;
	int sensing_ms_time;
	int als_intt, als_adc, als_cycle;

	als_intt = als_intt_value[intt>>2];
	als_adc = adc_value[adc>>3];
	als_cycle = cycle_value[cycle];
#if COMMON_DEBUG
	APS_LOG("ALS: INTT=%d, ADC=%d, Cycle=%d \r\n", als_intt, als_adc, als_cycle);
#endif

	sensing_us_time = (als_intt + als_adc*2*2) * als_cycle;
	sensing_ms_time = sensing_us_time /1000;
#if COMMON_DEBUG
	APS_LOG("[%s]: sensing=%d ms \r\n", __func__, sensing_ms_time);
#endif
	return (sensing_ms_time + 10);
}

static int ps_sensing_time(int intt, int adc, int cycle)
{
	long sensing_us_time;
	int sensing_ms_time;
	int ps_intt, ps_adc, ps_cycle;

	ps_intt = ps_intt_value[intt>>2];
	ps_adc = adc_value[adc>>3];
	ps_cycle = cycle_value[cycle];
#if COMMON_DEBUG
	APS_LOG("PS: INTT=%d, ADC=%d, Cycle=%d \r\n", ps_intt, ps_adc, ps_cycle);
#endif

	sensing_us_time = (ps_intt*3 + ps_adc*2*3) * ps_cycle;
	sensing_ms_time = sensing_us_time /1000;
#if COMMON_DEBUG
	APS_LOG("[%s]: sensing=%d ms\r\n", __func__, sensing_ms_time);
#endif
	return (sensing_ms_time + 10);
}

void epl_sensor_eint_func(void)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	if(!obj)
	{
		return;
	}
	mt_eint_mask(CUST_EINT_ALS_NUM);
	schedule_delayed_work(&obj->eint_work, 0);
}

long epl252x_read_ps(struct i2c_client *client, u16 *data)
{
	u8 databuf[4];
	u16 temp_data;
	long res = 0;
	if(client == NULL)
	{
		APS_DBG("epl252x_read_ps client is null \n");
		return -1;
	}

	databuf[0]= 0x1C;
	res = epl252x_i2c_master_operate(client, databuf, 0x401, I2C_FLAG_READ);
	if(res <= 0)
	{
		goto EXIT_ERR;
	}

	temp_data = databuf[2] |(databuf[3]<<8);//databuf[0] | (databuf[1]<<8);
	*data =  temp_data;//databuf[2] | (databuf[3]<<8);
	return 0;	
EXIT_ERR:
	APS_LOG("epl252x_read_ps failed!!!\n");
	return res;
}

long epl252x_read_als_raw(struct i2c_client *client, u16 *data)
{
	u8 databuf[4];
	u16 temp_data;
	long res = 0;
	if(client == NULL)
	{
		APS_DBG("epl252x_read_als client is null \n");
		return -1;
	}
	databuf[0]= 0x13;
	res = epl252x_i2c_master_operate(client, databuf, 0x401, I2C_FLAG_READ);
	if(res <= 0)
	{
		goto EXIT_ERR;
	}
	temp_data = databuf[2] | (databuf[3]<<8);//channels[1] value
	*data =  temp_data;
#if ALS_DYN_INTT
	epl_sensor.als.data.channels[0] = (databuf[1]<<8) | databuf[0];
	epl_sensor.als.data.channels[1] = (databuf[3]<<8) | databuf[2];
#endif
	return 0;
EXIT_ERR:
	APS_LOG("epl252x_read_als failed!!!\n");
	return res;
}

/*----------------------------------------------------------------------------*/
static int epl252x_get_ps_value(struct epl_sensor_priv *obj, u16 ps)
{
	int val;// mask = atomic_read(&obj->ps_mask);
	int invalid = 0;
	static int val_temp=1;

	if((ps  > atomic_read(&obj->ps_eng_cali_high)))
	{
		val = 0;  /*close*/
		val_temp = 0;
		intr_flag_value = 1;
	}
	else if((ps  < atomic_read(&obj->ps_eng_cali_low)))
	{
		val = 1;  /*far away*/
		val_temp = 1;
		intr_flag_value = 0;
	}
	else
		val = val_temp;	


	if(atomic_read(&obj->ps_suspend))
	{
		invalid = 1;
	}
	else if(1 == atomic_read(&obj->ps_deb_on))
	{
		unsigned long endt = atomic_read(&obj->ps_deb_end);
		if(time_after(jiffies, endt))
		{
			atomic_set(&obj->ps_deb_on, 0);
		}

		if (1 == atomic_read(&obj->ps_deb_on))
		{
			invalid = 1;
		}
	}

	if(!invalid)
	{
		return val;
	}	
	else
	{
		return -1;
	}	
}

static int epl252x_check_intr(struct i2c_client *client) 
{
	int res,intp;
	u8 databuf[2];

	databuf[0] = 0x1B;
	res = epl252x_i2c_master_operate(client, databuf, 0x101, I2C_FLAG_READ);
	if(res <= 0)
	{
		APS_ERR("epl252x_check_intr fail\n");
		goto EXIT_ERR;
	}
	res = 0;
	intp = 0;
	if(0 != (databuf[0] & 0x04))
	{
		res = 0;
		intp = 1;
	}
	return res;

EXIT_ERR:
	APS_ERR("epl252x_check_intr fail\n");
	return 1;
}
static int epl252x_clear_intr(struct i2c_client *client) 
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	u8 databuf[3];

	epl_sensor.ps.compare_reset = EPL_CMP_RUN;
	epl_sensor.ps.lock = EPL_UN_LOCK;
	databuf[0] = 0x1B; 
	databuf[1] = epl_sensor.ps.compare_reset | epl_sensor.ps.lock;
	epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
	return 0;
}
/*----------------------------------------------------------------------------*/
static void epl_sensor_eint_work(struct work_struct *work)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	int res = 0;
	int err;
	u8 databuf[3];
	int ps_value;
	APS_LOG("epl_sensor_eint_work entry!!\n");
	if((err = epl252x_check_intr(obj->client)))
	{
		APS_ERR("epl252x_check_intr: %d\n", err);
	}
	else
	{
		mutex_lock(&sensor_mutex);
		epl252x_read_ps(obj->client, &obj->ps);
		ps_value = epl252x_get_ps_value(obj, obj->ps);
		APS_LOG("epl_sensor_eint_work rawdata ps=%d ps_value:%d ps_suspend:%d\n",obj->ps, ps_value, atomic_read(&obj->ps_suspend));
		err = ps_report_interrupt_data(ps_value);
		if(err != 0)
		{
			APS_ERR("ps_report_interrupt_data  err: %d\n", err);
		}
		if(intr_flag_value){
			databuf[0] = 0x0C;	
			databuf[1] = (u8)((atomic_read(&obj->ps_eng_cali_low)) & 0x00FF);
			res = epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
			if(res <= 0)
			{
				goto EXIT_ERR;
			}

			databuf[0] = 0x0D;	
			databuf[1] = (u8)(((atomic_read(&obj->ps_eng_cali_low)) & 0xFF00) >> 8);
			res = epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
			if(res <= 0)
			{
				goto EXIT_ERR;
			}
			databuf[0] = 0x0E;	
			databuf[1] = (u8)(0x00FF);
			res = epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
			if(res <= 0)
			{
				goto EXIT_ERR;
			}

			databuf[0] = 0x0F; 
			databuf[1] = (u8)((0xFF00) >> 8);
			res = epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
			if(res <= 0)
			{
				goto EXIT_ERR;
			}					
		}
		else
		{	
			databuf[0] = 0x0C;	
			databuf[1] = (u8)(0 & 0x00FF);
			res = epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
			if(res <= 0)
			{
				goto EXIT_ERR;
			}

			databuf[0] = 0x0D;	
			databuf[1] = (u8)((0 & 0xFF00) >> 8);
			res = epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
			if(res <= 0)
			{
				goto EXIT_ERR;
			}

			databuf[0] = 0x0E;	
			databuf[1] = (u8)((atomic_read(&obj->ps_eng_cali_high)) & 0x00FF);
			res = epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
			if(res <= 0)
			{
				goto EXIT_ERR;
			}

			databuf[0] = 0x0F; 
			databuf[1] = (u8)(((atomic_read(&obj->ps_eng_cali_high)) & 0xFF00) >> 8);
			res = epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
			res = i2c_master_send(obj->client, databuf, 0x2);
			if(res <= 0)
			{
				goto EXIT_ERR;
			}
		}
		mutex_unlock(&sensor_mutex);
	}
	epl252x_clear_intr(obj->client);
	mt_eint_unmask(CUST_EINT_ALS_NUM);
	APS_LOG("epl_sensor_eint_work exit!!\n");
	return;
EXIT_ERR:
	epl252x_clear_intr(obj->client);
	mt_eint_unmask(CUST_EINT_ALS_NUM); 
	APS_ERR("i2c_transfer error = %d\n", res);
	return;
}
/*----------------------------------------------------------------------------*/
int epl_sensor_setup_eint(struct i2c_client *client)
{
#ifdef CUSTOM_KERNEL_SENSORHUB
	int err = 0;
	err = SCP_sensorHub_rsp_registration(ID_PROXIMITY, alsps_irq_handler);
	return err;
#else //#ifdef CUSTOM_KERNEL_SENSORHUB
	APS_LOG("epl_sensor_setup_eint\n");

	/*configure to GPIO function, external interrupt*/
	mt_set_gpio_mode(GPIO_ALS_EINT_PIN, GPIO_ALS_EINT_PIN_M_EINT);
	mt_set_gpio_dir(GPIO_ALS_EINT_PIN, GPIO_DIR_IN);
	mt_set_gpio_pull_enable(GPIO_ALS_EINT_PIN, GPIO_PULL_ENABLE);
	mt_set_gpio_pull_select(GPIO_ALS_EINT_PIN, GPIO_PULL_UP);

	mt_eint_set_hw_debounce(CUST_EINT_ALS_NUM, CUST_EINT_ALS_DEBOUNCE_CN);
	mt_eint_registration(CUST_EINT_ALS_NUM, CUST_EINT_ALS_TYPE, epl_sensor_eint_func, 0);
	mt_eint_unmask(CUST_EINT_ALS_NUM);
	return 0;
#endif //#ifdef CUSTOM_KERNEL_SENSORHUB
}

/*----------------------------------------------------------------------------*/
static ssize_t epl_sensor_show_reg(struct device_driver *ddri, char *buf)
{
	ssize_t len = 0;
	struct i2c_client *client = epl_sensor_obj->client;

	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x00 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x00));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x01 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x01));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x02 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x02));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x03 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x03));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x04 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x04));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x05 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x05));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x06 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x06));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x07 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x07));
	if(epl_sensor.als.polling_mode == 0)
	{
		len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x08 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x08));
		len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x09 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x09));
		len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x0A value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x0A));
		len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x0B value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x0B));
	}
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x0C value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x0C));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x0D value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x0D));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x0E value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x0E));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x0F value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x0F));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x11 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x11));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x12 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x12));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x1B value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x1B));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x22 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x22));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x23 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x23));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x24 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x24));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0x25 value = 0x%x\n", i2c_smbus_read_byte_data(client, 0x25));
	len += snprintf(buf+len, PAGE_SIZE-len, "chip id REG 0xFC value = 0x%x\n", i2c_smbus_read_byte_data(client, 0xFC));

	return len;
}

static ssize_t epl_sensor_show_ps_enable(struct device_driver *ddri, char *buf)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	ssize_t len = 0;
	bool enable_ps = test_bit(CMC_BIT_PS, &obj->enable);
	APS_FUN();
	len += snprintf(buf + len, PAGE_SIZE - len, "%d\n", enable_ps);
	return len;

}static ssize_t epl_sensor_store_ps_enable(struct device_driver *ddri, const char *buf, size_t count)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	struct i2c_client *client = obj->client;
	uint16_t enable=0;
	u8 databuf[2];	  
	long res = 0;
	int  ps_time = 0;
	ps_time = ps_sensing_time(epl_sensor.ps.integration_time, epl_sensor.ps.adc, epl_sensor.ps.cycle);

	sscanf(buf, "%hu", &enable);
	APS_LOG("epl_sensor_store_ps_enable entry enable:%d !!! \n", enable);

	if(enable)
	{
		//unlock and reset interrupt state
		databuf[1] = 0x00;
		databuf[0] = 0x1b;
		res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
		if(res <= 0)
		{
			APS_LOG("epl_sensor_store_ps_enable i2 operation error buf:0x%d \n", databuf[0]);
		}

		databuf[1] = 0x00;		
		databuf[0] = 0x00;
		res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
		if(res <= 0)
		{
			APS_LOG("epl_sensor_store_ps_enable i2 operation error buf:0x%d\n", databuf[0]);
		}

		if(test_bit(CMC_BIT_ALS, &obj->enable))
			databuf[1] = 0x03;
		else
			databuf[1] = 0x02;		
		databuf[0] = 0x00;
		res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
		if(res <= 0)
		{
			APS_LOG("epl_sensor_store_ps_enable i2 operation error buf:0x%d \n", databuf[0]);
		}

		msleep(ps_time);

		//unlock and reset interrupt state
		databuf[1] = 0x02;
		databuf[0] = 0x1b;
		res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
		if(res <= 0)
		{
			APS_LOG("epl_sensor_store_ps_enable i2 operation error buf:0x%d\n", databuf[0]);
		}
		set_bit(CMC_BIT_PS, &obj->enable);
	}
	else
	{
		if(test_bit(CMC_BIT_ALS, &obj->enable))
			databuf[1] = 0x01;
		else
			databuf[1] = 0x00;

		databuf[0] = 0x00;
		res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
		if(res <= 0)
		{
			APS_LOG("epl_sensor_store_ps_enable i2 operation error buf:0x%d\n", databuf[0]);
		}
		clear_bit(CMC_BIT_PS, &obj->enable);
	}
	APS_LOG("epl_sensor_store_ps_enable finished !!! \n");
	return count;
}

static ssize_t epl_sensor_show_als_enable(struct device_driver *ddri, char *buf)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	ssize_t len = 0;
	bool enable_als = test_bit(CMC_BIT_ALS, &obj->enable);
	APS_FUN();
	len += snprintf(buf + len, PAGE_SIZE - len, "%d\n", enable_als);
	return len;

}

static ssize_t epl_sensor_store_als_enable(struct device_driver *ddri, const char *buf, size_t count)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	struct i2c_client *client = obj->client;
	uint16_t enable=0;
	u8 databuf[2];	  
	long res = 0;
	int als_time = 0;
	als_time = als_sensing_time(epl_sensor.als.integration_time, epl_sensor.als.adc, epl_sensor.als.cycle);

	sscanf(buf, "%hu", &enable);
	APS_LOG("epl_sensor_store_als_enable entry enable:%d !!! \n", enable);

	databuf[1] = 0x00;
	databuf[0] = 0x00;
	res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
	if(res <= 0)
	{
		APS_LOG("epl_sensor_store_als_enable i2 operation error buf:0x%d\n", databuf[0]);
	}

	if(enable)
	{
		if(test_bit(CMC_BIT_PS, &obj->enable))
			databuf[1] = 0x03;
		else
			databuf[1] = 0x01;

		databuf[0] = 0x00;
		res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
		if(res <= 0)
		{
			APS_LOG("epl_sensor_store_als_enable i2 operation error buf:0x%d\n", databuf[0]);
		}
		msleep(als_time);
		set_bit(CMC_BIT_ALS, &obj->enable);
	}
	else
	{
		if(test_bit(CMC_BIT_PS, &obj->enable))
		{
			databuf[1] = 0x02;
			databuf[0] = 0x00;
			res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
			if(res <= 0)
			{
				APS_LOG("epl_sensor_store_als_enable i2 operation error buf:0x%d\n", databuf[0]);
			}
		}
		clear_bit(CMC_BIT_ALS, &obj->enable);
	}
	APS_LOG("epl_sensor_store_als_enable finished !!! \n");
	return count;
}

static ssize_t epl_sensor_show_pdata(struct device_driver *ddri, char *buf)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	ssize_t len = 0;
	mutex_lock(&sensor_mutex);
	epl252x_read_ps(obj->client,  &obj->ps);
	mutex_unlock(&sensor_mutex);
	len += snprintf(buf + len, PAGE_SIZE - len, "%d\n", obj->ps);
	return len;
}

static ssize_t epl_sensor_show_als_data(struct device_driver *ddri, char *buf)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	ssize_t len = 0;
	mutex_lock(&sensor_mutex);
	epl252x_read_als_raw(obj->client, &obj->als);
	mutex_unlock(&sensor_mutex);
	len += snprintf(buf + len, PAGE_SIZE - len, "%d\n",  obj->als);
	return len;
}

static ssize_t epl_sensor_show_ps_threshold_high(struct device_driver *ddri, char *buf)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	ssize_t len = 0;
	int temp = atomic_read(&obj->ps_eng_cali_high);
	int data =atomic_read(&obj->ps_cali_high);
	len += snprintf(buf + len, PAGE_SIZE - len, "threshold high:%d %d\n", temp, data);
	return len;
}

static ssize_t epl_sensor_store_ps_threshold_high(struct device_driver *ddri, const char *buf, size_t count)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	int temp=0;
	sscanf(buf, "%d", &temp);
	atomic_set(&obj->ps_cali_high, temp);
	return count;
}

static ssize_t epl_sensor_show_ps_threshold_low(struct device_driver *ddri, char *buf)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	ssize_t len = 0;
	int temp = atomic_read(&obj->ps_eng_cali_low);
	int data =atomic_read(&obj->ps_cali_low);
	len += snprintf(buf + len, PAGE_SIZE - len, "threshold low:%d %d\n", temp, data);
	return len;
}

static ssize_t epl_sensor_store_ps_threshold_low(struct device_driver *ddri, const char *buf, size_t count)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	int temp=0;

	sscanf(buf, "%d", &temp);
	atomic_set(&obj->ps_cali_low, temp);
	return count;
}

static ssize_t epl_sensor_show_ps_threshold_offset_high(struct device_driver *ddri, char *buf)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	ssize_t len = 0;
	int tmp_hi = atomic_read(&obj->ps_threshold_high);
	int tmp_lo = atomic_read(&obj->ps_threshold_low);
	len += snprintf(buf + len, PAGE_SIZE - len, "high=%d low=%d\n", tmp_hi, tmp_lo);
	return len;
}

static ssize_t epl_sensor_store_ps_threshold_offset_high(struct device_driver *ddri, const char *buf, size_t count)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	int temp=0;

	sscanf(buf, "%d", &temp);
	atomic_set(&obj->ps_threshold_high, temp);
	return count;
}

static ssize_t epl_sensor_show_ps_threshold_offset_low(struct device_driver *ddri, char *buf)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	ssize_t len = 0;
	int tmp_hi = atomic_read(&obj->ps_threshold_high);
	int tmp_lo = atomic_read(&obj->ps_threshold_low);
	len += snprintf(buf + len, PAGE_SIZE - len, "high=%d low=%d\n", tmp_hi, tmp_lo);
	return len;
}

static ssize_t epl_sensor_store_ps_threshold_offset_low(struct device_driver *ddri, const char *buf, size_t count)
{
	struct epl_sensor_priv *obj = epl_sensor_obj;
	int temp=0;

	sscanf(buf, "%d", &temp);
	atomic_set(&obj->ps_threshold_low, temp);
	return count;
}

static ssize_t epl_sensor_store_reg_write(struct device_driver *ddri, const char *buf, size_t count)
{
	struct epl_sensor_priv *epld = epl_sensor_obj;
	int reg;
	int data;
	APS_FUN();

	sscanf(buf, "%x,%x",&reg, &data);

	APS_LOG("[%s]: reg=0x%x, data=0x%x", __func__, reg, data);
	epl_sensor_I2C_Write(epld->client, reg, data);

	return count;
}

/*----------------------------------------------------------------------------*/
static DRIVER_ATTR(elan_reg,	S_IROTH  | S_IWOTH, 	epl_sensor_show_reg, NULL);
static DRIVER_ATTR(ps_enable,	S_IROTH  | S_IWOTH, 	epl_sensor_show_ps_enable, epl_sensor_store_ps_enable);
static DRIVER_ATTR(als_enable,	S_IROTH  | S_IWOTH,	 epl_sensor_show_als_enable, epl_sensor_store_als_enable);
static DRIVER_ATTR(pdata,	S_IROTH  | S_IWOTH, 	epl_sensor_show_pdata, NULL);
static DRIVER_ATTR(als_data,	S_IROTH  | S_IWOTH, 	epl_sensor_show_als_data, NULL);
static DRIVER_ATTR(ps_threshold_high,	S_IROTH  | S_IWOTH, 	epl_sensor_show_ps_threshold_high, epl_sensor_store_ps_threshold_high);
static DRIVER_ATTR(ps_threshold_low,	S_IROTH  | S_IWOTH, 	epl_sensor_show_ps_threshold_low, epl_sensor_store_ps_threshold_low);
static DRIVER_ATTR(ps_threshold_offset_high,	S_IROTH  | S_IWOTH, epl_sensor_show_ps_threshold_offset_high, epl_sensor_store_ps_threshold_offset_high);
static DRIVER_ATTR(ps_threshold_offset_low,	S_IROTH  | S_IWOTH, 	epl_sensor_show_ps_threshold_offset_low, epl_sensor_store_ps_threshold_offset_low);
static DRIVER_ATTR(i2c_w,		S_IROTH  | S_IWOTH, 	NULL, epl_sensor_store_reg_write );
/*----------------------------------------------------------------------------*/
static struct driver_attribute * epl_sensor_attr_list[] =
{
	&driver_attr_elan_reg,
	&driver_attr_ps_enable,
	&driver_attr_als_enable,
	&driver_attr_pdata,
	&driver_attr_als_data,
	&driver_attr_ps_threshold_high,
	&driver_attr_ps_threshold_low,
	&driver_attr_ps_threshold_offset_high,
	&driver_attr_ps_threshold_offset_low,
	&driver_attr_i2c_w,
};
/*----------------------------------------------------------------------------*/
static int epl_sensor_create_attr(struct device_driver *driver)
{
	int idx, err = 0;
	int num = (int)(sizeof(epl_sensor_attr_list)/sizeof(epl_sensor_attr_list[0]));
	if (driver == NULL)
	{
		return -EINVAL;
	}

	for(idx = 0; idx < num; idx++)
	{
		if((err = driver_create_file(driver, epl_sensor_attr_list[idx])))
		{
			APS_ERR("driver_create_file (%s) = %d\n", epl_sensor_attr_list[idx]->attr.name, err);
			break;
		}
	}
	return err;
}
/*----------------------------------------------------------------------------*/
static int epl_sensor_delete_attr(struct device_driver *driver)
{
	int idx ,err = 0;
	int num = (int)(sizeof(epl_sensor_attr_list)/sizeof(epl_sensor_attr_list[0]));

	if (!driver)
		return -EINVAL;

	for (idx = 0; idx < num; idx++)
	{
		driver_remove_file(driver, epl_sensor_attr_list[idx]);
	}

	return err;
}
/******************************************************************************
 * Function Configuration
 ******************************************************************************/
static int epl_sensor_open(struct inode *inode, struct file *file)
{
	file->private_data = epl_sensor_i2c_client;

	APS_FUN();

	if (!file->private_data)
	{
		APS_ERR("null pointer!!\n");
		return -EINVAL;
	}

	return nonseekable_open(inode, file);
}
/*----------------------------------------------------------------------------*/
static int epl_sensor_release(struct inode *inode, struct file *file)
{
	APS_FUN();
	file->private_data = NULL;
	return 0;
}
static int epl252x_ps_average_val = 0;
static void epl252x_WriteCalibration(struct epl_sensor_priv *obj, HWMON_PS_STRUCT *data_cali)
{
	APS_LOG("le_WriteCalibration  1 %d," ,data_cali->close);
	APS_LOG("le_WriteCalibration  2 %d," ,data_cali->far_away);
	APS_LOG("le_WriteCalibration  3 %d,", data_cali->valid);

	if(data_cali->valid == 1)
	{
		atomic_set(&obj->ps_thd_val_high, data_cali->close);
		atomic_set(&obj->ps_thd_val_low, data_cali->far_away);
		atomic_set(&obj->ps_ftm_cali_high, data_cali->close);
		atomic_set(&obj->ps_ftm_cali_low, data_cali->far_away);

	}else{
		atomic_set(&obj->ps_thd_val_high, 10000);
		atomic_set(&obj->ps_thd_val_low, 8000);
	}
}
static int epl252x_read_data_for_cali(struct i2c_client *client, HWMON_PS_STRUCT *ps_data_cali)
{
	struct epl_sensor_priv *obj = i2c_get_clientdata(client);
	int i=0 ,res = 0;
	u16 data[32];
	u32 sum=0, data_cali=0;
	bool enable_ps = test_bit(CMC_BIT_PS, &obj->enable);

	if(enable_ps == 0)
		ps_enable_nodata(1);

	for(i = 0; i < 10; i++)
	{
		epl252x_read_ps(obj->client, &obj->ps);
		data[i] = obj->ps;
		if(res != 0)
		{
			APS_ERR("epl252x_read_data_for_cali fail: %d\n", i);
			break;
		}
		else
		{
			APS_LOG("[%d]sample = %d\n", i, data[i]);
			sum += data[i];
		}
		mdelay(40);
	}

	if(i < 10)
	{
		res=  -1;

		return res;
	}
	else
	{
		data_cali = sum / 10;
		epl252x_ps_average_val = data_cali;
		APS_LOG("epl252x_read_data_for_cali data = %d",data_cali);
		if( data_cali>50000)
		{
			APS_ERR("epl252x_read_data_for_cali fail value to high: %d\n", data_cali);
			return -2;
		}
		ps_data_cali->close =data_cali + atomic_read(&obj->ps_threshold_high);
		ps_data_cali->far_away =data_cali + atomic_read(&obj->ps_threshold_low);
		ps_data_cali->valid = 1;
		APS_LOG("epl252x_read_data_for_cali close = %d,far_away = %d,valid = %d\n",ps_data_cali->close,ps_data_cali->far_away,ps_data_cali->valid);
	}
	return res;
}
/*----------------------------------------------------------------------------*/
static long epl_sensor_unlocked_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	struct i2c_client *client = (struct i2c_client*)file->private_data;
	struct epl_sensor_priv *obj = i2c_get_clientdata(client);
	HWMON_PS_STRUCT ps_cali_temp;
	int err = 0;
	int ps_result;
	int threshold[2];
	void __user *ptr = (void __user*) arg;
	int dat;
	uint32_t enable;
	bool enable_ps = test_bit(CMC_BIT_PS, &obj->enable);
	bool enable_als = test_bit(CMC_BIT_ALS, &obj->enable);
	APS_LOG("%s cmd = 0x%04x", __FUNCTION__, cmd);
	switch (cmd)
	{
		case ALSPS_SET_PS_MODE:
			if(copy_from_user(&enable, ptr, sizeof(enable)))
			{
				err = -EFAULT;
				goto err_out;
			}
			ps_enable_nodata(enable);
			break;

		case ALSPS_GET_PS_MODE:
			enable=test_bit(CMC_BIT_PS, &obj->enable);
			if(copy_to_user(ptr, &enable, sizeof(enable)))
			{
				err = -EFAULT;
				goto err_out;
			}
			break;

		case ALSPS_GET_PS_DATA:
			if(enable_ps == 0)
			{
				ps_enable_nodata(1);
			}

			epl252x_read_ps(obj->client, &obj->ps);
			dat = epl252x_get_ps_value(obj, obj->ps);
			APS_LOG("ioctl epl252x_get_ps_value = %d \n", dat);
			if(copy_to_user(ptr, &dat, sizeof(dat)))
			{
				err = -EFAULT;
				goto err_out;
			}
			break;

		case ALSPS_GET_PS_RAW_DATA:
			if(enable_ps == 0)
			{
				set_bit(CMC_BIT_PS, &obj->enable);
				ps_enable_nodata(1);
			}
			mutex_lock(&sensor_mutex);
			epl252x_read_ps(obj->client, &obj->ps);
			mutex_unlock(&sensor_mutex);
			dat = obj->ps;
			APS_LOG("ioctl ps raw value = %d \n", dat);
			if(copy_to_user(ptr, &dat, sizeof(dat)))
			{
				err = -EFAULT;
				goto err_out;
			}
			break;

		case ALSPS_SET_ALS_MODE:
			if(copy_from_user(&enable, ptr, sizeof(enable)))
			{
				err = -EFAULT;
				goto err_out;
			}
			if(enable_als != enable)
			{
#if ALS_DYN_INTT
				dynamic_intt_idx = dynamic_intt_init_idx;
				epl_sensor.als.integration_time = als_dynamic_intt_intt[dynamic_intt_idx];
				epl_sensor.als.gain = als_dynamic_intt_gain[dynamic_intt_idx];
				dynamic_intt_high_thr = als_dynamic_intt_high_thr[dynamic_intt_idx];
				dynamic_intt_low_thr = als_dynamic_intt_low_thr[dynamic_intt_idx];
#endif
				als_enable_nodata(enable);
			}
			break;

		case ALSPS_GET_ALS_MODE:
			enable=test_bit(CMC_BIT_ALS, &obj->enable);
			if(copy_to_user(ptr, &enable, sizeof(enable)))
			{
				err = -EFAULT;
				goto err_out;
			}
			break;

		case ALSPS_GET_ALS_DATA:
			if(enable_als == 0)
			{
				als_enable_nodata(1);		
			}
			mutex_lock(&sensor_mutex);
			epl252x_read_als_raw(obj->client, &obj->als);
			mutex_unlock(&sensor_mutex);
#if ALS_DYN_INTT
			dat = epl252x_sensor_get_als_value(obj, obj->als);
#else
			dat = obj->als;
#endif
			APS_LOG("ioctl get als data = %d\n", dat);

			if(copy_to_user(ptr, &dat, sizeof(dat)))
			{
				err = -EFAULT;
				goto err_out;
			}
			break;

		case ALSPS_GET_ALS_RAW_DATA:
			if(enable_als == 0)
			{
				als_enable_nodata(1);	
			}
			mutex_lock(&sensor_mutex);
			epl252x_read_als_raw(obj->client, &obj->als);
			mutex_unlock(&sensor_mutex);
			dat = obj->als;
			APS_LOG("ioctl get als raw data = %d\n", dat);
			if(copy_to_user(ptr, &dat, sizeof(dat)))
			{
				err = -EFAULT;
				goto err_out;
			}
			break;

			/*----------------------------------for factory mode test---------------------------------------*/
		case ALSPS_GET_PS_TEST_RESULT:
			if(enable_ps == 0)
			{
				ps_enable_nodata(1);	
			}
			mutex_lock(&sensor_mutex);
			epl252x_read_ps(obj->client, &obj->ps);
			mutex_unlock(&sensor_mutex);

			if(obj->ps > obj->hw->ps_threshold_high)
			{
				ps_result = 0;
			}
			else
				ps_result = 1;

			APS_LOG("[%s] ps_result = %d \r\n", __func__, ps_result);

			if(copy_to_user(ptr, &ps_result, sizeof(ps_result)))
			{
				err = -EFAULT;
				goto err_out;
			}
			break;
		case ALSPS_GET_PS_THRESHOLD_HIGH:
			threshold[0] = obj->hw->ps_threshold_high;
			APS_LOG("[%s] get threshold high: %d\n", __func__, threshold[0]);
			if(copy_to_user(ptr, &threshold[0], sizeof(threshold[0])))
			{
				err = -EFAULT;
				goto err_out;
			}
			break;

		case ALSPS_GET_PS_THRESHOLD_LOW:
			threshold[0] = obj->hw->ps_threshold_low;
			APS_LOG("[%s] get threshold low: %d\n", __func__, threshold[0]);
			if(copy_to_user(ptr, &threshold[0], sizeof(threshold[0])))
			{
				err = -EFAULT;
				goto err_out;
			}
			break;

		case ALSPS_SET_PS_CALI:
			if(copy_from_user(&ps_cali_temp, ptr, sizeof(ps_cali_temp)))
			{
				APS_LOG("copy_from_user\n");
				err = -EFAULT;
				break;
			}
			epl252x_WriteCalibration(obj, &ps_cali_temp);
			APS_LOG(" ALSPS_SET_PS_CALI %d,%d,%d\t",ps_cali_temp.close,ps_cali_temp.far_away,ps_cali_temp.valid);
			break;
		case ALSPS_GET_PS_RAW_DATA_FOR_CALI:
			{
				//mt_eint_mask(CUST_EINT_ALS_NUM);
				err = epl252x_read_data_for_cali(obj->client,&ps_cali_temp);
				if(err < 0 ){
					goto err_out;
				}
				epl252x_WriteCalibration(obj, &ps_cali_temp);
				if(copy_to_user(ptr, &ps_cali_temp, sizeof(ps_cali_temp)))
				{
					err = -EFAULT;
					goto err_out;
				}
			}
			break;			
		case ALSPS_GET_PS_AVERAGE:
			enable = epl252x_ps_average_val;
			if(copy_to_user(ptr, &enable, sizeof(enable)))
			{
				err = -EFAULT;
				goto err_out;
			}
			break;
		case ALSPS_GET_PS_FAR_THRESHOLD:
			enable = atomic_read(&obj->ps_thd_val_low);
			if(copy_to_user(ptr, &enable, sizeof(enable)))
			{
				err = -EFAULT;
				goto err_out;
			}
			break;
		case ALSPS_GET_PS_CLOSE_THRESHOLD:
			enable = atomic_read(&obj->ps_thd_val_high);
			if(copy_to_user(ptr, &enable, sizeof(enable)))
			{
				err = -EFAULT;
				goto err_out;
			}
			break;

		default:
			APS_ERR("%s not supported = 0x%04x", __FUNCTION__, cmd);
			err = -ENOIOCTLCMD;
			break;
	}

err_out:
	return err;
}
/*----------------------------------------------------------------------------*/
static struct file_operations epl_sensor_fops =
{
	.owner = THIS_MODULE,
	.open = epl_sensor_open,
	.release = epl_sensor_release,
	.unlocked_ioctl = epl_sensor_unlocked_ioctl,
};
/*----------------------------------------------------------------------------*/
static struct miscdevice epl_sensor_device =
{
	.minor = MISC_DYNAMIC_MINOR,
	.name = "als_ps",
	.fops = &epl_sensor_fops,
};
/*----------------------------------------------------------------------------*/
static int epl_sensor_i2c_suspend(struct i2c_client *client, pm_message_t msg)
{
	APS_FUN();
	return 0;
}
/*----------------------------------------------------------------------------*/
static int epl_sensor_i2c_resume(struct i2c_client *client)
{
	APS_FUN();
	return 0;
}

/*----------------------------------------------------------------------------*/
static void epl_sensor_early_suspend(struct early_suspend *h)
{
	/*early_suspend is only applied for ALS*/
	struct epl_sensor_priv *obj = container_of(h, struct epl_sensor_priv, early_drv);
	bool enable_ps = test_bit(CMC_BIT_PS, &obj->enable);
	bool enable_als = test_bit(CMC_BIT_ALS, &obj->enable);

	APS_LOG("epl_sensor_early_suspend entry enable_ps:%d enable_als:%d\n", enable_ps, enable_als);
	if(!obj)
	{
		APS_ERR("null pointer!!\n");
		return;
	}
	if(enable_als)
	{
		als_enable_nodata(0);
		atomic_set(&obj->als_suspend, 1);
	}
	if(!enable_ps)
	{
		mt_eint_mask(CUST_EINT_ALS_NUM);
		atomic_set(&obj->ps_suspend, 1);
	}
	APS_LOG("epl_sensor_early_suspend finished!!! \n");
}
/*----------------------------------------------------------------------------*/
static void epl_sensor_late_resume(struct early_suspend *h)
{
	/*late_resume is only applied for ALS*/
	struct epl_sensor_priv *obj = container_of(h, struct epl_sensor_priv, early_drv);
	bool enable_ps = test_bit(CMC_BIT_PS, &obj->enable);
	bool enable_als = test_bit(CMC_BIT_ALS, &obj->enable);

	APS_LOG("epl_sensor_late_resume entry enable_ps:%d enable_als:%d\n", enable_ps, enable_als);

	if(!obj)
	{
		APS_ERR("null pointer!!\n");
		return;
	}

	if(1 == atomic_read(&obj->als_suspend))
	{
		als_enable_nodata(1);
		atomic_set(&obj->als_suspend, 0);
	}

	if(1 == atomic_read(&obj->ps_suspend))
	{
		mt_eint_unmask(CUST_EINT_ALS_NUM);
		atomic_set(&obj->ps_suspend, 0);
	}
	APS_LOG("epl_sensor_late_resume finished!!! \n");
}
/*--------------------------------------------------------------------------------*/
static int als_open_report_data(int open)
{
	//should queuq work to report event if  is_report_input_direct=true
	return 0;
}
/*--------------------------------------------------------------------------------*/
// if use  this typ of enable , Gsensor only enabled but not report inputEvent to HAL
static int als_enable_nodata(int enable)
{
	/*molg1 changed 20150128 begin*/
	struct epl_sensor_priv *obj = epl_sensor_obj;
	struct i2c_client *client = obj->client;
	int als_time = 0, ps_time = 0, total_time = 0;
	u8 databuf[2];	  
	long res = 0;
#if ALS_DYN_INTT
	dynamic_intt_idx = dynamic_intt_init_idx;
	epl_sensor.als.integration_time = als_dynamic_intt_intt[dynamic_intt_idx];
	epl_sensor.als.gain = als_dynamic_intt_gain[dynamic_intt_idx];
	dynamic_intt_high_thr = als_dynamic_intt_high_thr[dynamic_intt_idx];
	dynamic_intt_low_thr = als_dynamic_intt_low_thr[dynamic_intt_idx];
#if ALS_DYN_INTT
	epl_sensor.als.cycle = EPL_CYCLE_4;
#endif
#endif
	ps_time = ps_sensing_time(epl_sensor.ps.integration_time, epl_sensor.ps.adc, epl_sensor.ps.cycle);
	als_time = als_sensing_time(epl_sensor.als.integration_time, epl_sensor.als.adc, epl_sensor.als.cycle);
	APS_LOG("als_enable_nodata entry enable:%d ps_time:%d als_time:%d !!! \n", enable, ps_time, als_time);

	databuf[1] = 0x00;
	databuf[0] = 0x00;
	res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
	if(res <= 0)
	{
		goto EXIT_ERR;
	}

	if(enable)
	{
#if ALS_DYN_INTT
		databuf[1] = epl_sensor.als.integration_time | epl_sensor.als.gain;
		databuf[0] = 0x01;
		epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
#if ALS_DYN_INTT_FAST
		dynamic_intt_fast_idx = 0;

		APS_LOG("[%s]: ALS fast mode \r\n", __func__);
		databuf[1] = epl_sensor.als.adc | epl_sensor.als.cycle;
		databuf[0] = 0x02;
		epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);

#endif
#endif
		if(test_bit(CMC_BIT_PS, &obj->enable))
		{
			databuf[1] = 0x03;
			total_time = ps_time + als_time;
		}
		else
		{
			databuf[1] = 0x01;
			total_time = als_time;
		}
		databuf[0] = 0x00;
		res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
		if(res <= 0)
		{
			goto EXIT_ERR;
		}
		APS_LOG("als_enable_nodata total_time:%d!!!\n",total_time);
		msleep(total_time);
		set_bit(CMC_BIT_ALS, &obj->enable);
	}
	else
	{
		if(test_bit(CMC_BIT_PS, &obj->enable))
		{
			databuf[1] = 0x02;
			databuf[0] = 0x00;
			res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
			if(res <= 0)
			{
				goto EXIT_ERR;
			}
		}
		clear_bit(CMC_BIT_ALS, &obj->enable);
	}
	APS_LOG("als_enable_nodata finished !!! \n");
	return 0;	
EXIT_ERR:
	APS_ERR("als_enable_nodata fail\n");
	return res;
	/*molg1 changed 20150128 end*/
}
/*--------------------------------------------------------------------------------*/
static int als_set_delay(u64 ns)
{
	return 0;
}
/*--------------------------------------------------------------------------------*/
static int als_get_data(int* value, int* status)
{
	int err = 0;
	int als_value;
	u8 databuf[2];
	struct epl_sensor_priv *obj = epl_sensor_obj;
	if(!obj)
	{
		APS_ERR("obj is null!!\n");
		return -1;
	}

	do{
		mutex_lock(&sensor_mutex);
		epl252x_read_als_raw(obj->client, &obj->als);
		if(epl_sensor.wait == EPL_WAIT_SINGLE)
		{
			databuf[1] = epl_sensor.power | epl_sensor.reset;
			databuf[0] = 0x11;
			err = epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
			if (err <= 0)
			{
				APS_LOG("als_get_data epl252x_i2c_master_operate error!!!\n");
			}
		}
		mutex_unlock(&sensor_mutex);
#if ALS_DYN_INTT
		als_value = epl252x_sensor_get_als_value(obj, obj->als);
#else
		als_value = obj->als;
#endif
#if ALS_DYN_INTT_FAST
		dynamic_intt_fast_idx++;
		if(als_dynamic_intt_intt_num > (als_dynamic_intt_intt_num - 1))
		{
			dynamic_intt_fast_idx = als_dynamic_intt_intt_num;
		}
		//APS_LOG("[%s]: dynamic_intt_fast_idx=%d \r\n", __func__, dynamic_intt_fast_idx);
#endif
	}while(-1 == als_value);

	*value = als_value;
	*status = SENSOR_STATUS_ACCURACY_MEDIUM;
	return err;
}
/*--------------------------------------------------------------------------------*/
// if use  this typ of enable , Gsensor should report inputEvent(x, y, z ,stats, div) to HAL
static int ps_open_report_data(int open)
{
	//should queuq work to report event if  is_report_input_direct=true
	return 0;
}
static int set_psensor_threshold(struct i2c_client *client)
{
	struct epl_sensor_priv *obj = i2c_get_clientdata(client);
	int res = 0;
	u8 databuf[3];
	int ps_value = -1;
	epl252x_read_ps(obj->client, &obj->ps);
	APS_LOG("set_psensor_threshold epl252x_read_ps value:%d\n", obj->ps);
	if((obj->ps > 0)&& (obj->ps <  (atomic_read(&obj->ps_cali_high)+7000)))
	{
		atomic_set(&obj->ps_eng_cali_high, obj->ps + atomic_read(&obj->ps_threshold_high));
		atomic_set(&obj->ps_eng_cali_low,  obj->ps + atomic_read(&obj->ps_threshold_low));
	}
	else
	{
		atomic_set(&obj->ps_eng_cali_high, atomic_read(&obj->ps_cali_high)+100);
		atomic_set(&obj->ps_eng_cali_low,  atomic_read(&obj->ps_cali_low)+100);
	}   

	databuf[0] = 0x0C;	
	databuf[1] = (u8)((atomic_read(&obj->ps_eng_cali_low)) & 0x00FF);
	res = epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
	if(res <= 0)
	{
		APS_ERR("i2c_master_send function err\n");
		return -1;
	}
	databuf[0] = 0x0D;	
	databuf[1] = (u8)(((atomic_read(&obj->ps_eng_cali_low)) & 0xFF00) >> 8);
	res = epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
	if(res <= 0)
	{
		APS_ERR("i2c_master_send function err\n");
		return -1;
	}
	databuf[0] = 0x0E;	
	databuf[1] = (u8)((atomic_read(&obj->ps_eng_cali_high)) & 0x00FF);
	res = epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
	if(res <= 0)
	{
		APS_ERR("i2c_master_send function err\n");
		return -1;
	}
	databuf[0] = 0x0F; 
	databuf[1] =  (u8)(((atomic_read(&obj->ps_eng_cali_high)) & 0xFF00) >> 8);
	res = epl252x_i2c_master_operate(obj->client, databuf, 0x2, I2C_FLAG_WRITE);
	if(res <= 0)
	{
		APS_ERR("i2c_master_send function err\n");
		return -1;
	}
	APS_ERR("set_psensor_threshold function high: %d, low:%d\n",atomic_read(&obj->ps_eng_cali_high),atomic_read(&obj->ps_eng_cali_low));
#if 0 
	epl252x_read_ps(obj->client, &obj->ps);
	if (obj->ps < atomic_read(&obj->ps_eng_cali_low))
	{
		ps_value = 1;
	}
	else if(obj->ps >  atomic_read(&obj->ps_eng_cali_high))
	{
		ps_value = 0;
	}
	ps_report_interrupt_data(ps_value);
#endif
	return 0;
}

static int ps_enable_nodata(int enable)
{
	/*molg1 changed 20150128 begin*/
	struct epl_sensor_priv *obj = epl_sensor_obj;
	struct i2c_client *client = obj->client;
	int  ps_time = 0, als_time = 0, total_time = 0;
	u8 databuf[2];	  
	long res = 0;
	ps_time = ps_sensing_time(epl_sensor.ps.integration_time, epl_sensor.ps.adc, epl_sensor.ps.cycle);
	als_time = als_sensing_time(epl_sensor.als.integration_time, epl_sensor.als.adc, epl_sensor.als.cycle);
	APS_LOG("ps_enable_nodata entry enable:%d ps_time:%d als_time:%d !!! \n", enable, ps_time, als_time);

	if(enable)
	{
		//lock and reset interrupt state
		databuf[1] = 0x00;
		databuf[0] = 0x1b;
		res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
		if(res <= 0)
		{
			goto EXIT_ERR;
		}

		databuf[1] = 0x00;		
		databuf[0] = 0x00;
		res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
		if(res <= 0)
		{
			goto EXIT_ERR;
		}
		if(test_bit(CMC_BIT_ALS, &obj->enable))
		{
			databuf[1] = 0x03;
			total_time = ps_time + als_time;
		}
		else
		{
			databuf[1] = 0x02;
			total_time = ps_time;
		}
		databuf[0] = 0x00;
		res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
		if(res <= 0)
		{
			goto EXIT_ERR;
		}
		APS_LOG("ps_enable_nodata total_time:%d!!!\n",total_time);
		msleep(total_time);
		//add for first data report
		set_psensor_threshold(client);
		//unlock and reset interrupt state
		databuf[1] = 0x02;
		databuf[0] = 0x1b;
		res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
		if(res <= 0)
		{
			goto EXIT_ERR;
		}
		set_bit(CMC_BIT_PS, &obj->enable);
	}
	else
	{
		if(test_bit(CMC_BIT_ALS, &obj->enable))
			databuf[1] =0x01;
		else
			databuf[1] = 0x00;

		databuf[0] = 0x00;
		res = epl252x_i2c_master_operate(client, databuf, 0x2, I2C_FLAG_WRITE);
		if(res <= 0)
		{
			goto EXIT_ERR;
		}
		clear_bit(CMC_BIT_PS, &obj->enable);
	}
	APS_LOG("ps_enable_nodata finished !!! \n");
	return 0;	
EXIT_ERR:
	APS_ERR("ps_enable_nodata fail\n");
	return res;
	/*molg1 changed 20150128 end*/
}
/*--------------------------------------------------------------------------------*/
static int ps_set_delay(u64 ns)
{
	return 0;
}
/*--------------------------------------------------------------------------------*/
static int ps_get_data(int* value, int* status)
{

	int err = 0;
	struct epl_sensor_priv *obj = epl_sensor_obj;
	int ps_value;
	APS_LOG("ps_get_data entry!!!\n");
	mutex_lock(&sensor_mutex);
	epl252x_read_ps(obj->client, &obj->ps);
	APS_LOG("ps_get_data rawdata ps=%d \n",obj->ps);
	ps_value = epl252x_get_ps_value(obj, obj->ps);
	mutex_unlock(&sensor_mutex);
	*value = ps_value;
	*status = SENSOR_STATUS_ACCURACY_MEDIUM;

	APS_LOG("[%s]:*value = %d\n", __func__, *value);
	return err;
}
/*
//====================initial global variable===============//
 */
static void write_global_variable(struct i2c_client *client)
{
	u8 buf;
	//wake up chip
	buf = epl_sensor.reset | epl_sensor.power;
	epl_sensor_I2C_Write(client, 0x11, buf);

	/* read revno*/
	epl_sensor_I2C_Read(client, 0x20, 2);
	epl_sensor.revno = gRawData.raw_bytes[0] | gRawData.raw_bytes[1] << 8;

	/*chip refrash*/
	epl_sensor_I2C_Write(client, 0xfd, 0x8e);
	epl_sensor_I2C_Write(client, 0xfe, 0x22);
	epl_sensor_I2C_Write(client, 0xfe, 0x02);
	epl_sensor_I2C_Write(client, 0xfd, 0x00);
	/*ps setting*/
	buf = epl_sensor.ps.integration_time | epl_sensor.ps.gain;
	epl_sensor_I2C_Write(client, 0x03, buf);

	buf = epl_sensor.ps.adc | epl_sensor.ps.cycle;
	epl_sensor_I2C_Write(client, 0x04, buf);

	buf = epl_sensor.ps.ir_on_control | epl_sensor.ps.ir_mode | epl_sensor.ps.ir_drive;
	epl_sensor_I2C_Write(client, 0x05, buf);

	buf = epl_sensor.interrupt_control | epl_sensor.ps.persist |epl_sensor.ps.interrupt_type;
	epl_sensor_I2C_Write(client, 0x06, buf);

	buf = epl_sensor.ps.compare_reset | epl_sensor.ps.lock;
	epl_sensor_I2C_Write(client, 0x1b, buf);

	epl_sensor_I2C_Write(client, 0x22, (u8)(epl_sensor.ps.cancelation& 0xff));
	epl_sensor_I2C_Write(client, 0x23, (u8)((epl_sensor.ps.cancelation & 0xff00) >> 8));
	set_psensor_intr_threshold(epl_sensor.ps.low_threshold, epl_sensor.ps.high_threshold);

	/*als setting*/
	buf = epl_sensor.als.integration_time | epl_sensor.als.gain;
	epl_sensor_I2C_Write(client, 0x01, buf);

	buf = epl_sensor.als.adc | epl_sensor.als.cycle;
	epl_sensor_I2C_Write(client, 0x02, buf);

	buf = epl_sensor.als.interrupt_channel_select | epl_sensor.als.persist | epl_sensor.als.interrupt_type;
	epl_sensor_I2C_Write(client, 0x07, buf);

	buf = epl_sensor.als.compare_reset | epl_sensor.als.lock;
	epl_sensor_I2C_Write(client, 0x12, buf);

	set_lsensor_intr_threshold(epl_sensor.als.low_threshold, epl_sensor.als.high_threshold);
	//set mode and wait
	buf = epl_sensor.wait | epl_sensor.mode;
	epl_sensor_I2C_Write(client, 0x00, buf);

}

static void initial_global_variable(struct i2c_client *client, struct epl_sensor_priv *obj)
{
	//general setting
	epl_sensor.power = EPL_POWER_ON;
	epl_sensor.reset = EPL_RESETN_RUN;
	epl_sensor.mode = EPL_MODE_IDLE;
	epl_sensor.wait = EPL_WAIT_20_MS;
	epl_sensor.osc_sel = EPL_OSC_SEL_1MHZ;
	//als setting
	epl_sensor.als.polling_mode = obj->hw->polling_mode_als;
	epl_sensor.als.integration_time = EPL_ALS_INTT_64;
	epl_sensor.als.gain = EPL_GAIN_MID;
	epl_sensor.als.adc = EPL_PSALS_ADC_13;
	epl_sensor.als.cycle = EPL_CYCLE_16;
	epl_sensor.als.interrupt_channel_select = EPL_ALS_INT_CHSEL_1;
	epl_sensor.als.persist = EPL_PERIST_1;
	epl_sensor.als.compare_reset = EPL_CMP_RESET;
	epl_sensor.als.lock = EPL_UN_LOCK;
	//wait to solve molg1
	epl_sensor.als.high_threshold = obj->hw->als_threshold_high;
	epl_sensor.als.low_threshold = obj->hw->als_threshold_low;

	//als factory
	epl_sensor.als.factory.calibration_enable =  false;
	epl_sensor.als.factory.calibrated = false;
	epl_sensor.als.factory.lux_per_count = LUX_PER_COUNT;
#if ALS_DYN_INTT
	dynamic_intt_idx = dynamic_intt_init_idx;
	epl_sensor.als.integration_time = als_dynamic_intt_intt[dynamic_intt_idx];
	epl_sensor.als.gain = als_dynamic_intt_gain[dynamic_intt_idx];
	dynamic_intt_high_thr = als_dynamic_intt_high_thr[dynamic_intt_idx];
	dynamic_intt_low_thr = als_dynamic_intt_low_thr[dynamic_intt_idx];
	//c_gain = 21000; // 21000 /1000 = 21
	c_gain = 11670; // 11670 /1000 = 11.67
#endif
	//ps setting
	epl_sensor.ps.polling_mode = obj->hw->polling_mode_ps;
	epl_sensor.ps.integration_time = EPL_PS_INTT_144;
	epl_sensor.ps.gain = EPL_GAIN_LOW;
	epl_sensor.ps.adc = EPL_PSALS_ADC_12;
	epl_sensor.ps.cycle = EPL_CYCLE_8;
	epl_sensor.ps.persist = EPL_PERIST_1;
	epl_sensor.ps.ir_on_control = EPL_IR_ON_CTRL_ON;
	epl_sensor.ps.ir_mode = EPL_IR_MODE_CURRENT;
	epl_sensor.ps.ir_drive = EPL_IR_DRIVE_10;
	epl_sensor.ps.compare_reset = EPL_CMP_RESET;
	epl_sensor.ps.lock = EPL_UN_LOCK;
	epl_sensor.ps.high_threshold = obj->hw->ps_threshold_high;
	epl_sensor.ps.low_threshold = obj->hw->ps_threshold_low;

	//ps factory
	epl_sensor.ps.factory.calibration_enable =  false;
	epl_sensor.ps.factory.calibrated = false;
	epl_sensor.ps.factory.cancelation= 0;

	//set als / ps interrupt control mode and trigger type
	epl_sensor.interrupt_control = 	EPL_INT_CTRL_PS;
	epl_sensor.als.interrupt_type = EPL_INTTY_DISABLE;
	epl_sensor.ps.interrupt_type = EPL_INTTY_ACTIVE;
	//write setting to sensor
	write_global_variable(client);
}

static int epl_sensor_init_client(struct i2c_client *client)
{
#ifndef CUSTOM_KERNEL_SENSORHUB
	struct epl_sensor_priv *obj = i2c_get_clientdata(client);
#endif
	int err=0;
	APS_LOG("epl_sensor_init_client I2C Addr==[0x%x],line=%d\n",epl_sensor_i2c_client->addr,__LINE__);
#ifdef CUSTOM_KERNEL_SENSORHUB
	epl_sensor_setup_eint(client);
#else //#ifdef CUSTOM_KERNEL_SENSORHUB
	if(obj->hw->polling_mode_ps == 0)
	{
		if((err = epl_sensor_setup_eint(client)))
		{
			APS_ERR("setup eint: %d\n", err);
			return err;
		}
		APS_LOG("epl_sensor interrupt setup\n");
	}
#endif //#ifdef CUSTOM_KERNEL_SENSORHUB

	if((err = hw8k_init_device(client)) != 0)
	{
		APS_ERR("init dev: %d\n", err);
		return err;
	}
	return err;
}

static int epl_sensor_i2c_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	struct epl_sensor_priv *obj;
#if NO_DATA
	struct als_control_path als_ctl={0};
	struct als_data_path als_data={0};
	struct ps_control_path ps_ctl={0};
	struct ps_data_path ps_data={0};
#endif
	int err = 0;

	APS_ERR("epl_sensor_i2c_probe entry!!!\n");	
	if(i2c_smbus_read_byte_data(client, 0x00) < 0)
	{
		APS_ERR("epl_sensor_device failed addr:%d\n", client->addr);
		err = -ENOMEM;
		goto exit;
	}	
	if(!(obj = kzalloc(sizeof(*obj), GFP_KERNEL)))
	{
		err = -ENOMEM;
		goto exit;
	}
	memset(obj, 0, sizeof(*obj));
	epl_sensor_obj = obj;
	obj->hw = get_cust_alsps_hw();

	epl_sensor_obj->als_level_num = sizeof(epl_sensor_obj->hw->als_level)/sizeof(epl_sensor_obj->hw->als_level[0]);
	epl_sensor_obj->als_value_num = sizeof(epl_sensor_obj->hw->als_value)/sizeof(epl_sensor_obj->hw->als_value[0]);
	BUG_ON(sizeof(epl_sensor_obj->als_level) != sizeof(epl_sensor_obj->hw->als_level));
	memcpy(epl_sensor_obj->als_level, epl_sensor_obj->hw->als_level, sizeof(epl_sensor_obj->als_level));
	BUG_ON(sizeof(epl_sensor_obj->als_value) != sizeof(epl_sensor_obj->hw->als_value));
	memcpy(epl_sensor_obj->als_value, epl_sensor_obj->hw->als_value, sizeof(epl_sensor_obj->als_value));

	INIT_DELAYED_WORK(&obj->eint_work, epl_sensor_eint_work);

	obj->client = client;
#ifdef FPGA_EARLY_PORTING
	obj->client->timing = 100;
#else
	obj->client->timing = 400;
#endif
	mutex_init(&sensor_mutex);
	wake_lock_init(&ps_lock, WAKE_LOCK_SUSPEND, "ps wakelock");

	i2c_set_clientdata(client, obj);

	atomic_set(&obj->als_debounce, 50);//2000);
	atomic_set(&obj->als_deb_on, 0);
	atomic_set(&obj->als_deb_end, 0);
	atomic_set(&obj->ps_debounce, 10);//1000);
	atomic_set(&obj->ps_deb_on, 0);
	atomic_set(&obj->ps_deb_end, 0);
	atomic_set(&obj->ps_mask, 0);
	atomic_set(&obj->trace, 0x00);
	atomic_set(&obj->ps_suspend, 0);
	atomic_set(&obj->als_suspend, 0);
	atomic_set(&obj->ps_eng_cali_high, 10000);//obj->hw ->ps_threshold_high);
	atomic_set(&obj->ps_eng_cali_low, 8000);//obj->hw ->ps_threshold_low);
	atomic_set(&obj->ps_thd_val_high,   10000);//obj->hw ->ps_threshold_high);
	atomic_set(&obj->ps_thd_val_low,  8000);//obj->hw ->ps_threshold_low);
	atomic_set(&obj->ps_ftm_cali_high, 10000);//obj->hw ->ps_threshold_high);
	atomic_set(&obj->ps_ftm_cali_low, 8000);//obj->hw ->ps_threshold_low);
	atomic_set(&obj->ps_cali_high,  10000);
	atomic_set(&obj->ps_cali_low,  8000);
	atomic_set(&obj->ps_threshold_high,  450);//used for set high threshold
	atomic_set(&obj->ps_threshold_low,  370);//used for set low threshold
	obj->ps_enable = 0;
	obj->als_enable = 0;
	obj->lux_per_count = LUX_PER_COUNT;
	obj->enable = 0;
	obj->pending_intr = 0;
	atomic_set(&obj->i2c_retry, 3);
	epl_sensor_i2c_client = client;

	//initial global variable and write to senosr
	initial_global_variable(client, obj);

	if((err =  epl_sensor_init_client(client)))
	{
		goto exit_init_failed;
	}
	if((err = misc_register(&epl_sensor_device)))
	{
		APS_ERR("epl_sensor_device register failed\n");
		goto exit_misc_device_register_failed;
	}
#if NO_DATA
	if((err = epl_sensor_create_attr(&epl_sensor_init_info.platform_diver_addr->driver)))
	{
		APS_ERR("create attribute err = %d\n", err);
		goto exit_create_attr_failed;
	}
#else
	if(err = epl_sensor_create_attr(&epl_sensor_alsps_driver.driver))
	{
		APS_ERR("create attribute err = %d\n", err);
		goto exit_create_attr_failed;
	}
#endif
#if NO_DATA
	als_ctl.open_report_data= als_open_report_data;
	als_ctl.enable_nodata = als_enable_nodata;
	als_ctl.set_delay  = als_set_delay;
	als_ctl.is_report_input_direct = false;
#ifdef CUSTOM_KERNEL_SENSORHUB
	als_ctl.is_support_batch = true;
#else
	als_ctl.is_support_batch = false;
#endif

	err = als_register_control_path(&als_ctl);
	if(err)
	{
		APS_ERR("register fail = %d\n", err);
		goto exit_sensor_obj_attach_fail;
	}

	als_data.get_data = als_get_data;
	als_data.vender_div = 100;
	err = als_register_data_path(&als_data);	
	if(err)
	{
		APS_ERR("tregister fail = %d\n", err);
		goto exit_sensor_obj_attach_fail;
	}

	ps_ctl.open_report_data= ps_open_report_data;
	ps_ctl.enable_nodata = ps_enable_nodata;
	ps_ctl.set_delay  = ps_set_delay;
	ps_ctl.is_report_input_direct = true;
#ifdef CUSTOM_KERNEL_SENSORHUB
	ps_ctl.is_support_batch = true;
#else
	ps_ctl.is_support_batch = false;
#endif

	err = ps_register_control_path(&ps_ctl);
	if(err)
	{
		APS_ERR("register fail = %d\n", err);
		goto exit_sensor_obj_attach_fail;
	}

	ps_data.get_data = ps_get_data;
	ps_data.vender_div = 100;
	err = ps_register_data_path(&ps_data);	
	if(err)
	{
		APS_ERR("tregister fail = %d\n", err);
		goto exit_sensor_obj_attach_fail;
	}
#endif
#ifndef FPGA_EARLY_PORTING
#if defined(CONFIG_HAS_EARLYSUSPEND)
	obj->early_drv.level    = EARLY_SUSPEND_LEVEL_STOP_DRAWING - 2,
		obj->early_drv.suspend  = epl_sensor_early_suspend,
		obj->early_drv.resume   = epl_sensor_late_resume,
		register_early_suspend(&obj->early_drv);
#endif
#endif
	alsps_init_flag = 0;
	APS_LOG("%s: OK\n", __FUNCTION__);
	return 0;
exit_create_attr_failed:
exit_sensor_obj_attach_fail:
	misc_deregister(&epl_sensor_device);
exit_misc_device_register_failed:
exit_init_failed:
	kfree(obj);
exit:
	epl_sensor_i2c_client = NULL;
	APS_ERR("%s: err = %d\n", __FUNCTION__, err);
	alsps_init_flag = -1;
	return err;
}

static int epl_sensor_i2c_remove(struct i2c_client *client)
{
	int err;
#if NO_DATA
	if((err = epl_sensor_delete_attr(&epl_sensor_init_info.platform_diver_addr->driver)))
	{
		APS_ERR("epl_sensor_delete_attr fail: %d\n", err);
	}
#endif
	if((err = misc_deregister(&epl_sensor_device)))
	{
		APS_ERR("misc_deregister fail: %d\n", err);
	}
	epl_sensor_i2c_client = NULL;
	i2c_unregister_device(client);
	kfree(i2c_get_clientdata(client));
	return 0;
}

static int alsps_local_init(void)
{
	struct alsps_hw *hw = get_cust_alsps_hw();

	APS_ERR("alsps_local_init entry\n");
	epl_sensor_power(hw, 1);
	if(i2c_add_driver(&epl_sensor_i2c_driver))
	{
		APS_ERR("add driver error\n");
		return -1;
	}
	if(-1 == alsps_init_flag)
	{
		APS_ERR("alsps_local_init failed\n");
		return -1;
	}
	APS_ERR("alsps_local_init finished!!!\n");
	return 0;
}
/*----------------------------------------------------------------------------*/
static int alsps_remove()
{
	struct alsps_hw *hw = get_cust_alsps_hw();
	APS_FUN();
	epl_sensor_power(hw, 0);

	APS_ERR("EPL2521 remove \n");
	i2c_del_driver(&epl_sensor_i2c_driver);
	return 0;
}
/*----------------------------------------------------------------------------*/
static int __init epl_sensor_init(void)
{
	struct alsps_hw *hw = get_cust_alsps_hw();
	APS_LOG("%s: i2c_number=%d\n", __func__,hw->i2c_num);
	i2c_register_board_info(hw->i2c_num, &i2c_epl_sensor, 1);
	alsps_driver_add(&epl_sensor_init_info);
	return 0;
}
/*----------------------------------------------------------------------------*/
static void __exit epl_sensor_exit(void)
{
	APS_FUN();
	//platform_driver_unregister(&epl_sensor_alsps_driver);
}
/*----------------------------------------------------------------------------*/
module_init(epl_sensor_init);
module_exit(epl_sensor_exit);
/*----------------------------------------------------------------------------*/
MODULE_AUTHOR("renato.pan@eminent-tek.com");
MODULE_DESCRIPTION("EPL252x ALPsr driver");
MODULE_LICENSE("GPL");
