/*
 * Copyright (C) 2008 Google, Inc.
 * Author: Nick Pelly <npelly@google.com>
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

/* Control bluetooth power for QSD platform */

#include <linux/platform_device.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/rfkill.h>
#include <linux/delay.h>
#include <asm/gpio.h>
#include <mach/vreg.h>
#include <linux/gpio/gpio_def.h>


#define DEBUG
#ifdef DEBUG
#define BTRFKILL_INFO(fmt, arg...) printk(KERN_INFO "bt_rfkill: " fmt "\n" , ## arg)
#define BTRFKILL_DBG(fmt, arg...)  printk(KERN_INFO "%s: " fmt "\n" , __FUNCTION__ , ## arg)
#define BTRFKILL_ERR(fmt, arg...)  printk(KERN_ERR  "%s: " fmt "\n" , __FUNCTION__ , ## arg)
#else
#define BTRFKILL_INFO(fmt, arg...)
#define BTRFKILL_DBG(fmt, arg...)
#define BTRFKILL_ERR(fmt, arg...)
#endif

static struct rfkill *bt_rfk;
static const char bt_name[] = "bcm4329";


#if 0
void rfkill_switch_all(enum rfkill_type type, enum rfkill_state state);
static int bluetooth_set_power(void *data, enum rfkill_state state)
{
#ifdef BOARD_RUBY_EVB
	struct vreg *vreg_rftx;
	int rc = 0;
	//Enable Class 1.5
	vreg_rftx = vreg_get(NULL, "rftx");
#endif
	BTRFKILL_DBG(" state = %d", state);
	//dump_stack();
	switch (state) {
	case RFKILL_STATE_UNBLOCKED:

#ifdef BOARD_RUBY_EVB
		//Enable Class 1.5
		rc = vreg_set_level(vreg_rftx, 2500);
		if (rc) {
			printk(KERN_ERR "%s: vreg set level failed (%d)\n",
			       __func__, rc);
			return -EIO;
		}
		rc = vreg_enable(vreg_rftx);
		if (rc) {
			printk(KERN_ERR "%s: vreg enable failed (%d)\n",
			       __func__, rc);
			return -EIO;
		}
		printk(KERN_DEBUG "bluetooth_power: Turn on BT power Class 1.5 DONE\n");
		//Turn on BT_REG_ON
		printk(KERN_DEBUG "bluetooth_power: Turn on chip \n");
#endif
		printk(KERN_DEBUG "bluetooth_power: Pull high for BT_EN\n");
		gpio_set_value(BT_EN, RFKILL_STATE_ON);
		mdelay(100);
		printk(KERN_DEBUG "bluetooth_power: Pull high for BT_RST_N\n");
		//Turn on BT_RST_N
		gpio_set_value(BT_RST_N, RFKILL_STATE_ON);

		printk(KERN_DEBUG "bluetooth_power: Pull low for BT_WAKE\n");
		//Turn on BT_WAKEUP
		gpio_set_value(BT_WAKE, 0);
		break;

	case RFKILL_STATE_SOFT_BLOCKED:

		//Turn off BT_RST_N
		gpio_set_value(BT_RST_N, RFKILL_STATE_OFF);
		//Turn off BT_REG_ON
		printk(KERN_DEBUG "bluetooth_power: Turn off chip \n" );
		gpio_set_value(BT_EN, RFKILL_STATE_OFF);

#ifdef BOARD_RUBY_EVB
		//Enable Class 1.5
		rc = vreg_disable(vreg_rftx);
		printk(KERN_DEBUG "bluetooth_power: Turn off BT power Class 1.5 \n");
		if (rc) {
			printk(KERN_ERR "%s: vreg disable failed (%d)\n",
			       __func__, rc);
			return -EIO;

		}
#endif
		printk(KERN_DEBUG "bluetooth_power: Pull high for BT_WAKE\n");
		//Turn off BT_WAKE
		gpio_set_value(BT_WAKE, 1);
		break;
	default: 
		printk(KERN_ERR "bad bluetooth rfkill state %d\n", state);
		
	}
	return 0;
}

static int bt_rfkill_probe(struct platform_device *pdev)
{
	int rc = 0;
	BTRFKILL_DBG();
    //Gilbert mod for kernel update
	/* default to bluetooth off */
	//rfkill_switch_all(RFKILL_TYPE_BLUETOOTH, RFKILL_STATE_OFF);
    rfkill_set_default(RFKILL_TYPE_BLUETOOTH, RFKILL_STATE_OFF); 
	bluetooth_set_power(NULL, RFKILL_STATE_OFF);

	bt_rfk = rfkill_allocate(&pdev->dev, RFKILL_TYPE_BLUETOOTH);
	if (!bt_rfk)
		return -ENOMEM;
	
	bt_rfk->name = bt_name;
	bt_rfk->state = RFKILL_STATE_SOFT_BLOCKED;
	/* userspace cannot take exclusive control */
	bt_rfk->user_claim_unsupported = 1;
	bt_rfk->user_claim = 0;
	bt_rfk->data = NULL;  // user data
	bt_rfk->toggle_radio = bluetooth_set_power;
	
	rc = rfkill_register(bt_rfk);

	if (rc)
		rfkill_free(bt_rfk);
	return rc;
}
#endif 


static int bluetooth_set_power(void *data, bool blocked)
{

	BTRFKILL_DBG(" block = %d", blocked);
	//dump_stack();
	if (!blocked)
	{
		printk(KERN_DEBUG "bluetooth_power: Pull high for BT_EN\n");
		gpio_direction_output(BT_EN, 1);
		mdelay(100);
		
		printk(KERN_DEBUG "bluetooth_power: Pull high for BT_RST_N\n");
		//Turn on BT_RST_N
		gpio_direction_output(BT_RST_N, 1);
		
		}
		
   else 
   {
		//Turn off BT_RST_N
		gpio_direction_output(BT_RST_N, 0);
		
		//Turn off BT_REG_ON
		printk(KERN_DEBUG "bluetooth_power: Turn off chip \n" );
		gpio_direction_output(BT_EN, 0);

		}
	
	  return 0;
}

static const struct rfkill_ops bluetooth_power_rfkill_ops = {
	.set_block = bluetooth_set_power,
};

static int bt_rfkill_probe(struct platform_device *pdev)
{
	int rc = 0;
	bool default_state = true;  /* off */

	rc = gpio_request(BT_RST_N, "bt_reset");
	if (rc)
		goto err_gpio_reset;
		
	rc = gpio_request(BT_EN, "bt_enable");
	if (rc)
		goto err_gpio_shutdown;
		

	bluetooth_set_power(NULL, default_state);

	bt_rfk = rfkill_alloc(bt_name, &pdev->dev, RFKILL_TYPE_BLUETOOTH,
				&bluetooth_power_rfkill_ops, NULL);
	if (!bt_rfk) {
		rc = -ENOMEM;
		goto err_rfkill_alloc;
	}

	rfkill_set_states(bt_rfk, default_state, false);

	/* userspace cannot take exclusive control */

	rc = rfkill_register(bt_rfk);
	if (rc)
		goto err_rfkill_reg;

	return 0;

err_rfkill_reg:
	rfkill_destroy(bt_rfk);
err_rfkill_alloc:
	gpio_free(BT_EN);
err_gpio_shutdown:
	gpio_free(BT_RST_N);
err_gpio_reset:
	return rc;
}

static int bt_rfkill_remove(struct platform_device *dev)
{
	rfkill_unregister(bt_rfk);
	rfkill_destroy(bt_rfk);
	gpio_free(BT_EN);
	gpio_free(BT_RST_N);

	return 0;
}

static struct platform_driver bt_rfkill_driver = {
	.probe = bt_rfkill_probe,
	.remove = bt_rfkill_remove,
	.driver = {
		.name = "bt_rfkill",
		.owner = THIS_MODULE,
	},
};

static int __init bt_rfkill_init(void)
{
	int ret;
	printk(KERN_INFO "BootLog, +%s\n", __func__);
	ret = platform_driver_register(&bt_rfkill_driver);
	printk(KERN_INFO "BootLog, -%s, ret=%d\n", __func__, ret); 
	return ret;
}

static void __exit bt_rfkill_exit(void)
{
	printk(KERN_INFO "BootLog, +%s\n", __func__);
	platform_driver_unregister(&bt_rfkill_driver);
	//printk(KERN_INFO "BootLog, -%s, ret=%d\n", __func__, ret); 

}

module_init(bt_rfkill_init);
module_exit(bt_rfkill_exit);
MODULE_DESCRIPTION("bcm 4329 rfkill");
MODULE_AUTHOR("Nick Pelly <npelly@google.com>");
MODULE_LICENSE("GPL");
