/* Copyright (c) 2010-2011, Code Aurora Forum. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 */
#include <linux/bitmap.h>
#include <linux/bitops.h>
#include <linux/gpio.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/platform_device.h>


/**
 * struct invalid_gpio_dev: invalid GPIO device structure
 *
 * Used to give warning if anyone uses these gpio numbers
 * by invoking exported gpio interface
 */
struct invalid_gpio_dev {
	struct gpio_chip gpio_chip;
};


static int invalid_gpio_get(struct gpio_chip *chip, unsigned offset)
{
	WARN_ON(1);
	return 0;
}

static void invalid_gpio_set(struct gpio_chip *chip, unsigned offset, int val)
{
	WARN_ON(1);
}

static int invalid_gpio_direction_input(struct gpio_chip *chip, unsigned offset)
{
	WARN_ON(1);
	return 0;
}

static int invalid_gpio_direction_output(struct gpio_chip *chip,
				unsigned offset,
				int val)
{
	WARN_ON(1);
	return 0;
}

static int invalid_gpio_to_irq(struct gpio_chip *chip, unsigned offset)
{
	WARN_ON(1);
	return 0;
}

static int invalid_gpio_request(struct gpio_chip *chip, unsigned offset)
{
	WARN_ON(1);
	return 0;
}

static void invalid_gpio_free(struct gpio_chip *chip, unsigned offset)
{
	WARN_ON(1);
}

static struct invalid_gpio_dev invalid_gpio = {
	.gpio_chip = {
		.base             = INVALID_GPIO_NO,
		.ngpio            = 2,
		.direction_input  = invalid_gpio_direction_input,
		.direction_output = invalid_gpio_direction_output,
		.get              = invalid_gpio_get,
		.set              = invalid_gpio_set,
		.to_irq           = invalid_gpio_to_irq,
		.request          = invalid_gpio_request,
		.free             = invalid_gpio_free,
	},
};


static int __devinit invalid_gpio_probe(struct platform_device *dev)
{
	int ret;

	invalid_gpio.gpio_chip.label = dev->name;
	ret = gpiochip_add(&invalid_gpio.gpio_chip);
	if (ret < 0)
		return ret;

	return 0;
}

static int __devexit invalid_gpio_remove(struct platform_device *dev)
{
	int ret = gpiochip_remove(&invalid_gpio.gpio_chip);

	if (ret < 0)
		return ret;

	return 0;
}


static struct platform_driver invalid_gpio_driver = {
	.probe = invalid_gpio_probe,
	.remove = __devexit_p(invalid_gpio_remove),
	.driver = {
		.name = "invalidgpio",
		.owner = THIS_MODULE,
	},
};

static struct platform_device msm_device_gpio = {
	.name = "invalidgpio",
	.id   = 0,
};

static int __init invalid_gpio_init(void)
{
	int rc;

	rc = platform_driver_register(&invalid_gpio_driver);
	if (rc == 0)
		rc = platform_device_register(&msm_device_gpio);

	return rc;
}

static void __exit invalid_gpio_exit(void)
{
	platform_device_unregister(&msm_device_gpio);
	platform_driver_unregister(&invalid_gpio_driver);
}

postcore_initcall(invalid_gpio_init);
module_exit(invalid_gpio_exit);


MODULE_AUTHOR("Leo.Qin <gbean@codeaurora.org>");
MODULE_DESCRIPTION("Driver for Invalid Gpio");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("platform:invalgpio");
