/* Copyright (c) 2010-2011, Code Aurora Forum. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 */
#include <linux/bitmap.h>
#include <linux/bitops.h>
#include <linux/gpio.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/sysdev.h>
#include <mach/msm_iomap.h>
#include "gpiomux.h"
#include "mpm.h"
#include <linux/debugfs.h>
#include <linux/seq_file.h>

/* Bits of interest in the GPIO_IN_OUT register.
 */
enum {
	GPIO_IN_BIT  = 0,
	GPIO_OUT_BIT = 1
};

/* Bits of interest in the GPIO_INTR_STATUS register.
 */
enum {
	INTR_STATUS_BIT = 0,
};

/* Bits of interest in the GPIO_CFG register.
 */
enum {
	GPIO_OE_BIT = 9,
};

/* Bits of interest in the GPIO_INTR_CFG register.
 */
enum {
	INTR_ENABLE_BIT        = 0,
	INTR_POL_CTL_BIT       = 1,
	INTR_DECT_CTL_BIT      = 2,
	INTR_RAW_STATUS_EN_BIT = 3,
};

/* Codes of interest in GPIO_INTR_CFG_SU.
 */
enum {
	TARGET_PROC_SCORPION = 4,
	TARGET_PROC_NONE     = 7,
};

/*
 * There is no 'DC_POLARITY_LO' because the GIC is incapable
 * of asserting on falling edge or level-low conditions.  Even though
 * the registers allow for low-polarity inputs, the case can never arise.
 */
enum {
	DC_POLARITY_HI	= BIT(11),
	DC_IRQ_ENABLE	= BIT(3),
};

enum msm_tlmm_register {
	SDC4_HDRV_PULL_CTL = 0x20a0,
	SDC3_HDRV_PULL_CTL = 0x20a4,
};

struct tlmm_field_cfg {
	enum msm_tlmm_register reg;
	u8                     off;
};

static const struct tlmm_field_cfg tlmm_hdrv_cfgs[] = {
	{SDC4_HDRV_PULL_CTL, 6}, /* TLMM_HDRV_SDC4_CLK  */
	{SDC4_HDRV_PULL_CTL, 3}, /* TLMM_HDRV_SDC4_CMD  */
	{SDC4_HDRV_PULL_CTL, 0}, /* TLMM_HDRV_SDC4_DATA */
	{SDC3_HDRV_PULL_CTL, 6}, /* TLMM_HDRV_SDC3_CLK  */
	{SDC3_HDRV_PULL_CTL, 3}, /* TLMM_HDRV_SDC3_CMD  */
	{SDC3_HDRV_PULL_CTL, 0}, /* TLMM_HDRV_SDC3_DATA */
};

static const struct tlmm_field_cfg tlmm_pull_cfgs[] = {
	{SDC4_HDRV_PULL_CTL, 11}, /* TLMM_PULL_SDC4_CMD  */
	{SDC4_HDRV_PULL_CTL, 9},  /* TLMM_PULL_SDC4_DATA */
	{SDC3_HDRV_PULL_CTL, 11}, /* TLMM_PULL_SDC3_CMD  */
	{SDC3_HDRV_PULL_CTL, 9},  /* TLMM_PULL_SDC3_DATA */
};

/*
 * When a GPIO triggers, two separate decisions are made, controlled
 * by two separate flags.
 *
 * - First, INTR_RAW_STATUS_EN controls whether or not the GPIO_INTR_STATUS
 * register for that GPIO will be updated to reflect the triggering of that
 * gpio.  If this bit is 0, this register will not be updated.
 * - Second, INTR_ENABLE controls whether an interrupt is triggered.
 *
 * If INTR_ENABLE is set and INTR_RAW_STATUS_EN is NOT set, an interrupt
 * can be triggered but the status register will not reflect it.
 */
#define INTR_RAW_STATUS_EN BIT(INTR_RAW_STATUS_EN_BIT)
#define INTR_ENABLE        BIT(INTR_ENABLE_BIT)
#define INTR_DECT_CTL_EDGE BIT(INTR_DECT_CTL_BIT)
#define INTR_POL_CTL_HI    BIT(INTR_POL_CTL_BIT)

#define GPIO_INTR_CFG_SU(gpio)    (MSM_TLMM_BASE + 0x0400 + (0x04 * (gpio)))
#define DIR_CONN_INTR_CFG_SU(irq) (MSM_TLMM_BASE + 0x0700 + (0x04 * (irq)))
#define GPIO_CONFIG(gpio)         (MSM_TLMM_BASE + 0x1000 + (0x10 * (gpio)))
#define GPIO_IN_OUT(gpio)         (MSM_TLMM_BASE + 0x1004 + (0x10 * (gpio)))
#define GPIO_INTR_CFG(gpio)       (MSM_TLMM_BASE + 0x1008 + (0x10 * (gpio)))
#define GPIO_INTR_STATUS(gpio)    (MSM_TLMM_BASE + 0x100c + (0x10 * (gpio)))

/**
 * struct msm_gpio_dev: the MSM8660 SoC GPIO device structure
 *
 * @enabled_irqs: a bitmap used to optimize the summary-irq handler.  By
 * keeping track of which gpios are unmasked as irq sources, we avoid
 * having to do readl calls on hundreds of iomapped registers each time
 * the summary interrupt fires in order to locate the active interrupts.
 *
 * @wake_irqs: a bitmap for tracking which interrupt lines are enabled
 * as wakeup sources.  When the device is suspended, interrupts which are
 * not wakeup sources are disabled.
 *
 * @dual_edge_irqs: a bitmap used to track which irqs are configured
 * as dual-edge, as this is not supported by the hardware and requires
 * some special handling in the driver.
 */
struct msm_gpio_dev {
	struct gpio_chip gpio_chip;
	DECLARE_BITMAP(enabled_irqs, NR_MSM_GPIOS);
	DECLARE_BITMAP(wake_irqs, NR_MSM_GPIOS);
	DECLARE_BITMAP(dual_edge_irqs, NR_MSM_GPIOS);
};

static DEFINE_SPINLOCK(tlmm_lock);

static inline struct msm_gpio_dev *to_msm_gpio_dev(struct gpio_chip *chip)
{
	return container_of(chip, struct msm_gpio_dev, gpio_chip);
}

static inline void set_gpio_bits(unsigned n, void __iomem *reg)
{
	writel(readl(reg) | n, reg);
}

static inline void clr_gpio_bits(unsigned n, void __iomem *reg)
{
	writel(readl(reg) & ~n, reg);
}

static int msm_gpio_get(struct gpio_chip *chip, unsigned offset)
{
	int rc;
	rc = readl(GPIO_IN_OUT(offset)) & BIT(GPIO_IN_BIT);
	dsb();
	return rc;
}

static void msm_gpio_set(struct gpio_chip *chip, unsigned offset, int val)
{
	writel(val ? BIT(GPIO_OUT_BIT) : 0, GPIO_IN_OUT(offset));
	dsb();
}

static int msm_gpio_direction_input(struct gpio_chip *chip, unsigned offset)
{
	unsigned long irq_flags;

	spin_lock_irqsave(&tlmm_lock, irq_flags);
	clr_gpio_bits(BIT(GPIO_OE_BIT), GPIO_CONFIG(offset));
	dsb();
	spin_unlock_irqrestore(&tlmm_lock, irq_flags);
	return 0;
}

static int msm_gpio_direction_output(struct gpio_chip *chip,
				unsigned offset,
				int val)
{
	unsigned long irq_flags;

	spin_lock_irqsave(&tlmm_lock, irq_flags);
	msm_gpio_set(chip, offset, val);
	set_gpio_bits(BIT(GPIO_OE_BIT), GPIO_CONFIG(offset));
	dsb();
	spin_unlock_irqrestore(&tlmm_lock, irq_flags);
	return 0;
}

static int msm_gpio_to_irq(struct gpio_chip *chip, unsigned offset)
{
	return MSM_GPIO_TO_INT(offset - chip->base);
}

static inline int msm_irq_to_gpio(struct gpio_chip *chip, unsigned irq)
{
	return irq - MSM_GPIO_TO_INT(chip->base);
}

static int msm_gpio_request(struct gpio_chip *chip, unsigned offset)
{
	return msm_gpiomux_get(chip->base + offset);
}

static void msm_gpio_free(struct gpio_chip *chip, unsigned offset)
{
	msm_gpiomux_put(chip->base + offset);
}

/* GORAB1A_CR #xxxxx, Jey Zhou, 20110614, show more and real time gpio information in debugfs. {*/
enum gpio_drv {
	GPIO_DRV_2MA = 0,
	GPIO_DRV_4MA,
	GPIO_DRV_6MA,
	GPIO_DRV_8MA,
	GPIO_DRV_10MA,
	GPIO_DRV_12MA,
	GPIO_DRV_14MA,
	GPIO_DRV_16MA,
};

enum gpio_func {
	GPIO_FUNC_GPIO = 0,
	GPIO_FUNC_1,
	GPIO_FUNC_2,
	GPIO_FUNC_3,
	GPIO_FUNC_4,
	GPIO_FUNC_5,
	GPIO_FUNC_6,
	GPIO_FUNC_7,
	GPIO_FUNC_8,
	GPIO_FUNC_9,
	GPIO_FUNC_A,
	GPIO_FUNC_B,
	GPIO_FUNC_C,
	GPIO_FUNC_D,
	GPIO_FUNC_E,
	GPIO_FUNC_F,
};

enum gpio_pull {
	GPIO_PULL_NONE = 0,
	GPIO_PULL_DOWN,
	GPIO_PULL_KEEPER,
	GPIO_PULL_UP,
};

enum gpio_dir {
	GPIO_IN = 0,
	GPIO_OUT_HIGH,
	GPIO_OUT_LOW,
};

static void msm_gpio_dbg_show(struct seq_file *s, struct gpio_chip *chip)
{
	unsigned		i;
	unsigned		gpio = chip->base;
	unsigned int	config;
	unsigned 	func, in_out, pull, strength;
	char		pull_str[16];

	seq_printf(s, " GPIO-num  FuncId  Direction  Level  Pull    Strength\n");
	for (i = 0; i < chip->ngpio; i++, gpio++) {
		config = readl(GPIO_CONFIG(i));
		func = (config & (0xf << 2)) >> 2;
		in_out = ((config & (BIT(GPIO_OE_BIT))) >> GPIO_OE_BIT);
		pull = config & (0x3);
		strength = (config & (0x7 << 6)) >> 6;

		switch (pull) {
		case GPIO_PULL_NONE:
			strcpy(pull_str, "no");
			break;
		case GPIO_PULL_DOWN:
			strcpy(pull_str, "down");
			break;
		case GPIO_PULL_KEEPER:
			strcpy(pull_str, "keeper");
			break;
		case GPIO_PULL_UP:
			strcpy(pull_str, "up");
			break;
		default: 
			strcpy(pull_str, "unknown");
			break;
		}
		seq_printf(s, " gpio-%-3d  %-6d  %-9.9s  %-5.5s  %-6.6s  %-3dmA",
			gpio, func,
			in_out ? "out" : "in",
			chip->get ? (chip->get(chip, i) ? "high" : "low ") : "?  ",
			pull_str, (strength+1)*2);
		seq_printf(s, "\n");
	}
}
/* GORAB1A_CR #xxxxx, Jey Zhou, 20110614, show more and real time gpio information in debugfs. }*/

/* GORAB1A_CR #xxxxx, Jey Zhou, 20110620, add interface for gpio debug. {*/
void msm_gpio_dbg_show_trace(void)
{
	unsigned		i,gpio=0;
	unsigned int	config;
	unsigned 	func, in_out, pull, strength,level;
	char		pull_str[16];

	printk("GPIO-num  FuncId  Direction  Level  Pull    Strength\n");
	for (i = 0; i < NR_MSM_GPIOS; i++, gpio++) {
		config = readl(GPIO_CONFIG(i));
		func = (config & (0xf << 2)) >> 2;
		in_out = ((config & (BIT(GPIO_OE_BIT))) >> GPIO_OE_BIT);
		pull = config & (0x3);
		strength = (config & (0x7 << 6)) >> 6;
		level = readl(GPIO_IN_OUT(i)) & BIT(GPIO_IN_BIT);

		switch (pull) {
		case GPIO_PULL_NONE:
			strcpy(pull_str, "no");
			break;
		case GPIO_PULL_DOWN:
			strcpy(pull_str, "down");
			break;
		case GPIO_PULL_KEEPER:
			strcpy(pull_str, "keeper");
			break;
		case GPIO_PULL_UP:
			strcpy(pull_str, "up");
			break;
		default: 
			strcpy(pull_str, "unknown");
			break;
		}
		printk("gpio-%-3d  %-6d  %-9.9s  %-5.5s  %-6.6s  %-3dmA",
			gpio, func,
			in_out ? "out" : "in",
			level ? "high" : "low ",
			pull_str, (strength+1)*2);
		printk("\n");
	}
}
EXPORT_SYMBOL(msm_gpio_dbg_show_trace);
/* GORAB1A_CR #xxxxx, Jey Zhou, 20110620, add interface for gpio debug. }*/

static struct msm_gpio_dev msm_gpio = {
	.gpio_chip = {
		.base             = 0,
		.ngpio            = NR_MSM_GPIOS,
		.direction_input  = msm_gpio_direction_input,
		.direction_output = msm_gpio_direction_output,
		.get              = msm_gpio_get,
		.set              = msm_gpio_set,
		.to_irq           = msm_gpio_to_irq,
		.request          = msm_gpio_request,
		.free             = msm_gpio_free,
/* GORAB1A_CR #xxxxx, Jey Zhou, 20110614, show more and real time gpio information in debugfs. {*/
		.dbg_show     = msm_gpio_dbg_show,
/* GORAB1A_CR #xxxxx, Jey Zhou, 20110614, show more and real time gpio information in debugfs. }*/
	},
};

/* For dual-edge interrupts in software, since the hardware has no
 * such support:
 *
 * At appropriate moments, this function may be called to flip the polarity
 * settings of both-edge irq lines to try and catch the next edge.
 *
 * The attempt is considered successful if:
 * - the status bit goes high, indicating that an edge was caught, or
 * - the input value of the gpio doesn't change during the attempt.
 * If the value changes twice during the process, that would cause the first
 * test to fail but would force the second, as two opposite
 * transitions would cause a detection no matter the polarity setting.
 *
 * The do-loop tries to sledge-hammer closed the timing hole between
 * the initial value-read and the polarity-write - if the line value changes
 * during that window, an interrupt is lost, the new polarity setting is
 * incorrect, and the first success test will fail, causing a retry.
 *
 * Algorithm comes from Google's msmgpio driver, see mach-msm/gpio.c.
 */
static void msm_gpio_update_dual_edge_pos(unsigned gpio)
{
	int loop_limit = 100;
	unsigned val, val2, intstat;

	do {
		val = readl(GPIO_IN_OUT(gpio)) & BIT(GPIO_IN_BIT);
		if (val)
			clr_gpio_bits(INTR_POL_CTL_HI, GPIO_INTR_CFG(gpio));
		else
			set_gpio_bits(INTR_POL_CTL_HI, GPIO_INTR_CFG(gpio));
		val2 = readl(GPIO_IN_OUT(gpio)) & BIT(GPIO_IN_BIT);
		intstat = readl(GPIO_INTR_STATUS(gpio)) & BIT(INTR_STATUS_BIT);
		if (intstat || val == val2) {
			/* switch the configuration in the mpm as well*/
#ifdef CONFIG_MSM_RPM
			if (val)
				msm_mpm_set_irq_type(msm_gpio_to_irq(
						&msm_gpio.gpio_chip, gpio),
						IRQF_TRIGGER_FALLING);
			else
				msm_mpm_set_irq_type(msm_gpio_to_irq(
						&msm_gpio.gpio_chip, gpio),
						IRQF_TRIGGER_RISING);
#endif
			return;
		}
	} while (loop_limit-- > 0);
	pr_err("%s: dual-edge irq failed to stabilize, "
	       "interrupts dropped. %#08x != %#08x\n",
	       __func__, val, val2);
}

static void msm_gpio_irq_ack(unsigned int irq)
{
	int gpio = msm_irq_to_gpio(&msm_gpio.gpio_chip, irq);

	writel(BIT(INTR_STATUS_BIT), GPIO_INTR_STATUS(gpio));
	if (test_bit(gpio, msm_gpio.dual_edge_irqs))
		msm_gpio_update_dual_edge_pos(gpio);
	dsb();
}

static void __msm_gpio_irq_mask(unsigned int gpio)
{
	__raw_writel(TARGET_PROC_NONE, GPIO_INTR_CFG_SU(gpio));
	clr_gpio_bits(INTR_RAW_STATUS_EN | INTR_ENABLE, GPIO_INTR_CFG(gpio));
}

static void msm_gpio_irq_mask(unsigned int irq)
{
	int gpio = msm_irq_to_gpio(&msm_gpio.gpio_chip, irq);
	unsigned long irq_flags;

	spin_lock_irqsave(&tlmm_lock, irq_flags);
	__msm_gpio_irq_mask(gpio);
	__clear_bit(gpio, msm_gpio.enabled_irqs);
	dsb();
	spin_unlock_irqrestore(&tlmm_lock, irq_flags);

#ifdef CONFIG_MSM_RPM
	msm_mpm_enable_irq(irq, 0);
#endif
}

static void __msm_gpio_irq_unmask(unsigned int gpio)
{
	set_gpio_bits(INTR_RAW_STATUS_EN | INTR_ENABLE, GPIO_INTR_CFG(gpio));
	__raw_writel(TARGET_PROC_SCORPION, GPIO_INTR_CFG_SU(gpio));
}

static void msm_gpio_irq_unmask(unsigned int irq)
{
	int gpio = msm_irq_to_gpio(&msm_gpio.gpio_chip, irq);
	unsigned long irq_flags;

	spin_lock_irqsave(&tlmm_lock, irq_flags);
	__set_bit(gpio, msm_gpio.enabled_irqs);
	__msm_gpio_irq_unmask(gpio);
	dsb();
	spin_unlock_irqrestore(&tlmm_lock, irq_flags);

#ifdef CONFIG_MSM_RPM
	msm_mpm_enable_irq(irq, 1);
#endif
}

static void msm_gpio_irq_disable(unsigned int irq)
{
#ifdef CONFIG_MSM_RPM
	msm_mpm_enable_irq(irq, 0);
#endif
}

static int msm_gpio_irq_set_type(unsigned int irq, unsigned int flow_type)
{
	int gpio = msm_irq_to_gpio(&msm_gpio.gpio_chip, irq);
	unsigned long irq_flags;
	uint32_t bits;

	spin_lock_irqsave(&tlmm_lock, irq_flags);

	bits = readl(GPIO_INTR_CFG(gpio));

	if (flow_type & IRQ_TYPE_EDGE_BOTH) {
		bits |= INTR_DECT_CTL_EDGE;
		irq_desc[irq].handle_irq = handle_edge_irq;
		if ((flow_type & IRQ_TYPE_EDGE_BOTH) == IRQ_TYPE_EDGE_BOTH)
			__set_bit(gpio, msm_gpio.dual_edge_irqs);
		else
			__clear_bit(gpio, msm_gpio.dual_edge_irqs);
	} else {
		bits &= ~INTR_DECT_CTL_EDGE;
		irq_desc[irq].handle_irq = handle_level_irq;
		__clear_bit(gpio, msm_gpio.dual_edge_irqs);
	}

	if (flow_type & (IRQ_TYPE_EDGE_RISING | IRQ_TYPE_LEVEL_HIGH))
		bits |= INTR_POL_CTL_HI;
	else
		bits &= ~INTR_POL_CTL_HI;

	writel(bits, GPIO_INTR_CFG(gpio));

	if ((flow_type & IRQ_TYPE_EDGE_BOTH) == IRQ_TYPE_EDGE_BOTH)
		msm_gpio_update_dual_edge_pos(gpio);

	dsb();
	spin_unlock_irqrestore(&tlmm_lock, irq_flags);
#ifdef CONFIG_MSM_RPM
	msm_mpm_set_irq_type(irq, flow_type);
#endif

	return 0;
}

/*
 * When the summary IRQ is raised, any number of GPIO lines may be high.
 * It is the job of the summary handler to find all those GPIO lines
 * which have been set as summary IRQ lines and which are triggered,
 * and to call their interrupt handlers.
 */
static irqreturn_t msm_summary_irq_handler(int irq, void *data)
{
	unsigned long i;
	struct irq_desc *desc = irq_to_desc(irq);

	for (i = find_first_bit(msm_gpio.enabled_irqs, NR_MSM_GPIOS);
	     i < NR_MSM_GPIOS;
	     i = find_next_bit(msm_gpio.enabled_irqs, NR_MSM_GPIOS, i + 1)) {
		if (readl(GPIO_INTR_STATUS(i)) & BIT(INTR_STATUS_BIT))
			generic_handle_irq(msm_gpio_to_irq(&msm_gpio.gpio_chip,
							   i));
	}

	desc->chip->ack(irq);
	return IRQ_HANDLED;
}

static int msm_gpio_irq_set_wake(unsigned int irq, unsigned int on)
{
	int gpio = msm_irq_to_gpio(&msm_gpio.gpio_chip, irq);

	if (on) {
		if (bitmap_empty(msm_gpio.wake_irqs, NR_MSM_GPIOS))
			set_irq_wake(TLMM_MSM_SUMMARY_IRQ, 1);
		set_bit(gpio, msm_gpio.wake_irqs);
	} else {
		clear_bit(gpio, msm_gpio.wake_irqs);
		if (bitmap_empty(msm_gpio.wake_irqs, NR_MSM_GPIOS))
			set_irq_wake(TLMM_MSM_SUMMARY_IRQ, 0);
	}

#ifdef CONFIG_MSM_RPM
	msm_mpm_set_irq_wake(irq, on);
#endif
	return 0;
}

static struct irq_chip msm_gpio_irq_chip = {
	.name		= "msmgpio",
	.mask		= msm_gpio_irq_mask,
	.unmask		= msm_gpio_irq_unmask,
	.ack		= msm_gpio_irq_ack,
	.set_type	= msm_gpio_irq_set_type,
	.set_wake	= msm_gpio_irq_set_wake,
	.disable	= msm_gpio_irq_disable,
};

static int __devinit msm_gpio_probe(struct sysdev_class *sysclass)
{
	int i, irq, ret;

	spin_lock_init(&tlmm_lock);
	bitmap_zero(msm_gpio.enabled_irqs, NR_MSM_GPIOS);
	bitmap_zero(msm_gpio.wake_irqs, NR_MSM_GPIOS);
	bitmap_zero(msm_gpio.dual_edge_irqs, NR_MSM_GPIOS);
	msm_gpio.gpio_chip.label = sysclass->name;
	ret = gpiochip_add(&msm_gpio.gpio_chip);
	if (ret < 0)
		return ret;

	for (i = 0; i < msm_gpio.gpio_chip.ngpio; ++i) {
		irq = msm_gpio_to_irq(&msm_gpio.gpio_chip, i);
		set_irq_chip(irq, &msm_gpio_irq_chip);
		set_irq_handler(irq, handle_level_irq);
		set_irq_flags(irq, IRQF_VALID);
	}

	ret = request_irq(TLMM_MSM_SUMMARY_IRQ, msm_summary_irq_handler,
			IRQF_TRIGGER_HIGH, "msmgpio", NULL);
	if (ret) {
		pr_err("Request_irq failed for TLMM_MSM_SUMMARY_IRQ - %d\n",
				ret);
		return ret;
	}
	return 0;
}

static int __devexit msm_gpio_remove(struct sysdev_class *sysclass)
{
	int ret = gpiochip_remove(&msm_gpio.gpio_chip);

	if (ret < 0)
		return ret;

	set_irq_handler(TLMM_MSM_SUMMARY_IRQ, NULL);

	return 0;
}

#ifdef CONFIG_PM
static int msm_gpio_suspend(struct sys_device *sys_dev,
					pm_message_t state)
{
	unsigned long irq_flags;
	unsigned long i;

	spin_lock_irqsave(&tlmm_lock, irq_flags);
	for_each_set_bit(i, msm_gpio.enabled_irqs, NR_MSM_GPIOS)
			__msm_gpio_irq_mask(i);

	for_each_set_bit(i, msm_gpio.wake_irqs, NR_MSM_GPIOS)
		__msm_gpio_irq_unmask(i);
	dsb();
	spin_unlock_irqrestore(&tlmm_lock, irq_flags);
	return 0;
}

void msm_gpio_show_resume_irq(void)
{
	unsigned long irq_flags;
	int i, irq, intstat;

	spin_lock_irqsave(&tlmm_lock, irq_flags);
	for_each_set_bit(i, msm_gpio.wake_irqs, NR_MSM_GPIOS) {
		intstat = readl(GPIO_INTR_STATUS(i)) & BIT(INTR_STATUS_BIT);
		if (intstat) {
			irq = msm_gpio_to_irq(&msm_gpio.gpio_chip, i);
			pr_warning("%s: %d triggered\n",
				__func__, irq);
		}
	}
	spin_unlock_irqrestore(&tlmm_lock, irq_flags);
}

static int msm_gpio_resume(struct sys_device *sys_dev)
{
	unsigned long irq_flags;
	unsigned long i;

	spin_lock_irqsave(&tlmm_lock, irq_flags);
	for_each_set_bit(i, msm_gpio.wake_irqs, NR_MSM_GPIOS)
		__msm_gpio_irq_mask(i);

	for_each_set_bit(i, msm_gpio.enabled_irqs, NR_MSM_GPIOS)
		__msm_gpio_irq_unmask(i);
	dsb();
	spin_unlock_irqrestore(&tlmm_lock, irq_flags);
	return 0;
}
#else
#define msm_gpio_suspend NULL
#define msm_gpio_resume NULL
#endif

static struct sysdev_class msm_gpio_sysdev_class = {
	.name = "msmgpio",
	.suspend = msm_gpio_suspend,
	.resume = msm_gpio_resume,
};

static struct sys_device msm_gpio_sys_device = {
	.id = 0,
	.cls = &msm_gpio_sysdev_class,
};

static int __init msm_gpio_init(void)
{
	int rc;

	rc = sysdev_class_register(&msm_gpio_sysdev_class);
	if (!rc) {
		rc = sysdev_register(&msm_gpio_sys_device);
		if (!rc) {
			rc = msm_gpio_probe(&msm_gpio_sysdev_class);
			if (rc)
				pr_err("msm_gpio_probe failed err = %d\n", rc);
		} else
			pr_err("sysdev_register failed for msm_gpio_sys_device \
					error = %d\n", rc);
	}

	return rc;
}

static void __exit msm_gpio_exit(void)
{
	msm_gpio_remove(&msm_gpio_sysdev_class);
	sysdev_unregister(&msm_gpio_sys_device);
	sysdev_class_unregister(&msm_gpio_sysdev_class);
}

postcore_initcall(msm_gpio_init);
module_exit(msm_gpio_exit);

static void msm_tlmm_set_field(const struct tlmm_field_cfg *configs,
			       unsigned id, unsigned width, unsigned val)
{
	unsigned long irqflags;
	u32 mask = (1 << width) - 1;
	u32 __iomem *reg = MSM_TLMM_BASE + configs[id].reg;
	u32 reg_val;

	spin_lock_irqsave(&tlmm_lock, irqflags);
	reg_val = readl(reg);
	reg_val &= ~(mask << configs[id].off);
	reg_val |= (val & mask) << configs[id].off;
	writel(reg_val, reg);
	dsb();
	spin_unlock_irqrestore(&tlmm_lock, irqflags);
}

void msm_tlmm_set_hdrive(enum msm_tlmm_hdrive_tgt tgt, int drv_str)
{
	msm_tlmm_set_field(tlmm_hdrv_cfgs, tgt, 3, drv_str);
}
EXPORT_SYMBOL(msm_tlmm_set_hdrive);

void msm_tlmm_set_pull(enum msm_tlmm_pull_tgt tgt, int pull)
{
	msm_tlmm_set_field(tlmm_pull_cfgs, tgt, 2, pull);
}
EXPORT_SYMBOL(msm_tlmm_set_pull);

int gpio_tlmm_config(unsigned config, unsigned disable)
{
	uint32_t flags;
	unsigned gpio = GPIO_PIN(config);

	if (gpio > NR_MSM_GPIOS)
		return -EINVAL;

	flags = ((GPIO_DIR(config) << 9) & (0x1 << 9)) |
		((GPIO_DRVSTR(config) << 6) & (0x7 << 6)) |
		((GPIO_FUNC(config) << 2) & (0xf << 2)) |
		((GPIO_PULL(config) & 0x3));
	writel(flags, GPIO_CONFIG(gpio));
	dsb();

	return 0;
}
EXPORT_SYMBOL(gpio_tlmm_config);

int msm_gpio_install_direct_irq(unsigned gpio, unsigned irq,
					unsigned int input_polarity)
{
	unsigned long irq_flags;
	uint32_t bits;

	if (gpio >= NR_MSM_GPIOS || irq >= NR_TLMM_MSM_DIR_CONN_IRQ)
		return -EINVAL;

	spin_lock_irqsave(&tlmm_lock, irq_flags);

	writel(readl(GPIO_CONFIG(gpio)) | BIT(GPIO_OE_BIT),
		GPIO_CONFIG(gpio));
	writel(readl(GPIO_INTR_CFG(gpio)) &
		~(INTR_RAW_STATUS_EN | INTR_ENABLE),
		GPIO_INTR_CFG(gpio));
	writel(DC_IRQ_ENABLE | TARGET_PROC_NONE,
		GPIO_INTR_CFG_SU(gpio));

	bits = TARGET_PROC_SCORPION | (gpio << 3);
	if (input_polarity)
		bits |= DC_POLARITY_HI;
	writel(bits, DIR_CONN_INTR_CFG_SU(irq));

	dsb();
	spin_unlock_irqrestore(&tlmm_lock, irq_flags);

	return 0;
}
EXPORT_SYMBOL(msm_gpio_install_direct_irq);

MODULE_AUTHOR("Gregory Bean <gbean@codeaurora.org>");
MODULE_DESCRIPTION("Driver for Qualcomm MSM TLMMv2 SoC GPIOs");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("sysdev:msmgpio");
