/* Copyright (c) 2008-2009, Code Aurora Forum. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 */

#include <linux/device.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/types.h>
#include "smd_private.h"


typedef struct
{
	/*Manufacturer ID*/
	uint32_t       manufacture_id;
	/* DDR Size in Mega Bytes */  
	uint32_t       ramsize;    
}ddr_info_type;

typedef struct
{
	uint32_t id;
	char name[20];
}ddr_manufacture_name;

static ssize_t show_external_ddr_info(struct device *dev, struct device_attribute *attr, char *buf)
{
	ddr_info_type *ddr_info;
	uint32_t size;
	uint32_t i = 0;
	
	ddr_manufacture_name ddr_vendor_table[] = 
	{
		{0,"RESERVED_0"},
		{1,"SAMSUNG"},
		{2,"QIMONDA"},
		{3,"ELPIDA"},
		{4,"ETRON"},
		{5,"NANYA"},
		{6,"HYNIX"},
		{7,"MOSEL"},
		{8,"WINBOND"},
		{9,"ESMT"},
		{10,"RESERVED_1"},
		{11,"SPANSION"},
		{12,"SST"},
		{13,"ZMOS"},
		{14,"INTEL"},
		{254,"NUMONYX"},
		{255,"MICRON"},
		{0x7FFFFFFF,"DDR_MANUFACTURES_MAX"},
	};

	ddr_info = (ddr_info_type*)smem_get_entry(SMEM_ID_VENDOR0,&size);
	
	if(ddr_info != NULL)
	{
		while( ddr_vendor_table[i].id != 0x7FFFFFFF )
		{
			if( ddr_vendor_table[i].id == ddr_info-> manufacture_id )
				break; 
			i++;
		}
		
		return sprintf(buf,"Manufacture:%s Size:%dMB\n",
					   ddr_vendor_table[i].name,ddr_info-> ramsize);
	}
	else
	{
		printk(KERN_ERR "%s,read ddr smd entry error\n",__func__);
		return sprintf(buf, "Read ddr smd entry error\n");
	}
}

static DEVICE_ATTR(ex_lpddr2_info, S_IRUGO, show_external_ddr_info, NULL);


static int lpddr2_test_open(struct inode *inode, struct file *file)
{
	/*To be extended*/
	printk(KERN_DEBUG "%s\n", __func__);
	
	return 0;
}

static int lpddr2_test_release(struct inode *inode, struct file *file)
{
	/*To be extended*/
	printk(KERN_DEBUG "%s\n", __func__);
	
	return 0;
}

static int lpddr2_test_ioctl(struct inode *inode, struct file *file,
			  unsigned cmd, unsigned long arg)
{
	/*To be extended*/
	printk(KERN_DEBUG "%s\n", __func__);
	
	return 0;
}

/**********************************************************************
 * Register ourselves as a misc device to be able to get DDR device info and
   DDR controller info.*/

static const struct file_operations lpddr2_test_fops = {
	.owner = THIS_MODULE,
	.ioctl = lpddr2_test_ioctl,
	.open = lpddr2_test_open,
	.release = lpddr2_test_release,
};

static struct miscdevice lpddr2_test_dev = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "lpddr2_test",
	.fops = &lpddr2_test_fops,
};

static int lpddr2_test_init(void)
{
	int ret = 0;

	ret = misc_register(&lpddr2_test_dev);
	if (ret < 0)
		return ret;
		
	ret = device_create_file(lpddr2_test_dev.this_device, &dev_attr_ex_lpddr2_info);
	if ( ret < 0 ) goto error_devfs;
	
	printk(KERN_INFO "%s, minor number %d\n", __func__, lpddr2_test_dev.minor);
    
	return 0;
	
error_devfs:
	misc_deregister(&lpddr2_test_dev);
	return ret ; 
}

static void lpddr2_test_exit(void)
{
	device_remove_file(lpddr2_test_dev.this_device, &dev_attr_ex_lpddr2_info);
	misc_deregister(&lpddr2_test_dev);
	printk(KERN_INFO "%s\n", __func__);
}

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("Test for lpddr2");
MODULE_VERSION("0.01");

module_init(lpddr2_test_init);
module_exit(lpddr2_test_exit);
