/* Copyright (c) 2010-2011, Code Aurora Forum. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/reboot.h>
#include <linux/io.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/cpu.h>
#include <linux/interrupt.h>
#include <linux/mfd/pmic8058.h>
#include <linux/mfd/pmic8901.h>

#include <mach/msm_iomap.h>
#include <mach/restart.h>
#include <mach/scm-io.h>
#include <asm/mach-types.h>
#include <mach/irqs.h>
#include <mach/scm.h>
#include <mach/msm_watchdog.h>

#include <mach/restart_flag.h>

#define TCSR_WDT_CFG 0x30

#define WDT0_RST       (MSM_TMR0_BASE + 0x38)
#define WDT0_EN        (MSM_TMR0_BASE + 0x40)
#define WDT0_BARK_TIME (MSM_TMR0_BASE + 0x4C)
#define WDT0_BITE_TIME (MSM_TMR0_BASE + 0x5C)

#define PSHOLD_CTL_SU (MSM_TLMM_BASE + 0x820)



static int  panic_mode;

#define IMEM_BASE           0x2A05F000

#define RESTART_REASON_ADDR 0x65C
#define DLOAD_MODE_ADDR     0x0

#define SCM_IO_DISABLE_PMIC_ARBITER	1

static int restart_mode;
void *restart_reason;

int pmic_reset_irq;

extern void write_OemRebootFlag(unsigned int oemValue);
static int get_ramdump_mode(void);

#ifdef CONFIG_MSM_DLOAD_MODE
static int in_panic;
static void *dload_mode_addr;

/* Download mode master kill-switch */
static int dload_set(const char *val, struct kernel_param *kp);
static int download_mode = 1;
module_param_call(download_mode, dload_set, param_get_int,
			&download_mode, 0644);

static int panic_prep_restart(struct notifier_block *this,
			      unsigned long event, void *ptr)
{
	in_panic = 1;
	return NOTIFY_DONE;
}

static struct notifier_block panic_blk = {
	.notifier_call	= panic_prep_restart,
};

static void set_dload_mode(int on)
{
	

	
	if (dload_mode_addr) {
		writel(on ? 0xE47B337D : 0, dload_mode_addr);
		writel(on ? 0xCE14091A : 0,
		       dload_mode_addr + sizeof(unsigned int));
		mb();
	}
}

static void my_set_dload_mode(int on)
{
	
	if (dload_mode_addr) {
		writel(on ? 0xE47B337D : 0, dload_mode_addr);
		writel(on ? 0xCE14091A : 0,
		       dload_mode_addr + sizeof(unsigned int));
		mb();
	}
}

static int dload_set(const char *val, struct kernel_param *kp)
{
	int ret;
	int old_val = download_mode;

	ret = param_set_int(val, kp);

	if (ret)
		return ret;

	/* If download_mode is not zero or one, ignore. */
	if (download_mode >> 1) {
		download_mode = old_val;
		return -EINVAL;
	}

	set_dload_mode(download_mode);

	return 0;
}

#else
#define set_dload_mode(x) do {} while (0)
#endif

void msm_set_restart_mode(int mode)
{
	restart_mode = mode;
}
EXPORT_SYMBOL(msm_set_restart_mode);

static int s_ramdump_mode = -1;

static int get_ramdump_mode(void)
{
	return s_ramdump_mode;
}

void set_ramdump_mode(int mode)
{
	s_ramdump_mode = mode;
}
EXPORT_SYMBOL(set_ramdump_mode);

void msm_set_panic_mode(int mode)
{
	panic_mode = mode;
}
EXPORT_SYMBOL(msm_set_panic_mode);

static void __msm_power_off(int lower_pshold)
{
	printk(KERN_CRIT "Powering off the SoC\n");
#ifdef CONFIG_MSM_DLOAD_MODE
	set_dload_mode(0);
#endif

	
	writel(0x0, restart_reason);
	write_OemRebootFlag(0x0);
	mb();
	preempt_disable();


	pm8058_reset_pwr_off(0);
	pm8901_reset_pwr_off(0);
	
	
	mdelay(1);  
	
	
	if (lower_pshold) {
		writel(0, PSHOLD_CTL_SU);
		mdelay(10000);
		printk(KERN_ERR "Powering off has failed\n");
	}

	return;
}

static void msm_power_off(void)
{
	/* MSM initiated power off, lower ps_hold */
	__msm_power_off(1);
}

void arch_power_off(void)
{
	
	__msm_power_off(1);
}
EXPORT_SYMBOL(arch_power_off);

static void cpu_power_off(void *data)
{
	int rc;

	pr_err("PMIC Initiated shutdown %s cpu=%d\n", __func__,
						smp_processor_id());
	if (smp_processor_id() == 0) {
		/*
		 * PMIC initiated power off, do not lower ps_hold, pmic will
		 * shut msm down
		 */
		__msm_power_off(0);

		pet_watchdog();
		pr_err("Calling scm to disable arbiter\n");
		/* call secure manager to disable arbiter and never return */
		rc = scm_call_atomic1(SCM_SVC_PWR,
						SCM_IO_DISABLE_PMIC_ARBITER, 1);

		pr_err("SCM returned even when asked to busy loop rc=%d\n", rc);
		pr_err("waiting on pmic to shut msm down\n");
	}

	preempt_disable();
	while (1)
		;
}

static irqreturn_t resout_irq_handler(int irq, void *dev_id)
{
	pr_warn("%s PMIC Initiated shutdown\n", __func__);
	oops_in_progress = 1;
	smp_call_function_many(cpu_online_mask, cpu_power_off, NULL, 0);
	if (smp_processor_id() == 0)
		cpu_power_off(NULL);
	preempt_disable();
	while (1)
		;
	return IRQ_HANDLED;
}

#ifdef CONFIG_RAM_DUMP_ENABLE
static int hexstr_to_integer(char *hex, int len)
{
	int value = 0;
	int i;

	for (i=0; i<len; i++) {
		int num;
		switch(hex[i]){
			case 'a':
			case 'A':
				num=10;
				break;
			case 'b':
			case 'B':
				num=11;
				break;
			case 'c':
			case 'C':
				num=12;
				break;
			case 'd':
			case 'D':
				num=13;
				break;
			case 'e':
			case 'E':
				num=14;
				break;
			case 'f':
			case 'F':
				num=15;
				break;
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9': 
				num = hex[i]-'0';
				break;
			default:
				return 0;
		}
		value += num << 4*(len -i-1);
	}
	
	return value;
}
#endif

void arch_reset(char mode, const char *cmd)
{
	int dload_flag = 0;


#ifdef CONFIG_MSM_DLOAD_MODE

	/* This looks like a normal reboot at this point. */
	set_dload_mode(0);

	/* Write download mode flags if we're panic'ing */
	set_dload_mode(in_panic);

	/* Write download mode flags if restart_mode says so */
	if (restart_mode == RESTART_DLOAD)
		set_dload_mode(1);

	/* Kill download mode if master-kill switch is set */
	if (!download_mode)
		set_dload_mode(0);
#endif

	if (in_panic){
		set_dload_mode(1);
		
		switch(restart_mode){
			case RESTART_DLOAD:				
				break;
			case RESTART_MODEM_RAMDUMP:
				break;
			case RESTART_Q6_RAMDUMP:
				break;
			case RESTART_KERNEL_RAMDUMP:
				break;
			case RESTART_NORMAL:
				set_dload_mode(0);
				break;
				
			default:
				BUG();
				break;
		}
	}

	printk(KERN_NOTICE "Going down for restart now\n");
	printk(KERN_NOTICE "cmd is %s, panic_mode is %d, in_panic is %d.\n",cmd, panic_mode, in_panic);

	pm8058_reset_pwr_off(1);

	if (cmd != NULL) {
		if (!strncmp(cmd, "bootloader", 10)) {
			writel(FASTBOOT_MODE, restart_reason);
			write_OemRebootFlag(FASTBOOT_MODE);
		} else if (!strncmp(cmd, "recovery", 8)) {
			writel(RECOVERY_MODE, restart_reason);
			write_OemRebootFlag(RECOVERY_MODE);
		
		
		} else if (!strncmp(cmd, "emergency", 9)) {
			writel(REBOOT_EMERGENCY_DLOAD_MODE, restart_reason);
			write_OemRebootFlag(REBOOT_EMERGENCY_DLOAD_MODE);
			
#ifdef CONFIG_RAM_DUMP_ENABLE
		} else if ( (!strncmp(cmd, "modem-ramdump", 13)) 
			
			|| ( (1 == panic_mode)|| (2 == panic_mode)|| (3 == panic_mode)) ) {
			writel(REBOOT_MODEM_RAMDUMPS_MODE, restart_reason);
			write_OemRebootFlag(REBOOT_MODEM_RAMDUMPS_MODE);
			dload_flag = 0;
			if(3 == get_ramdump_mode()) {
				dload_flag = 1;
				writel(REBOOT_DLOAD_MODE, restart_reason);
				write_OemRebootFlag(REBOOT_DLOAD_MODE);
			}
			
		} else if ( (!strncmp(cmd, "kernel-ramdump", 14)) 
			|| (5 == panic_mode)
			) {
			
			writel(REBOOT_KERNEL_RAMDUMPS_MODE, restart_reason);
			write_OemRebootFlag(REBOOT_KERNEL_RAMDUMPS_MODE);
			dload_flag = 0;
			if(3 == get_ramdump_mode()) {
				dload_flag = 1;
				writel(REBOOT_DLOAD_MODE, restart_reason);
				write_OemRebootFlag(REBOOT_DLOAD_MODE);
			}

		} else if (!strncmp(cmd, "ramdumps", 8)) {
			/*force into rumdumps mode, for manufacture*/
			writel(REBOOT_RAMDUMPS_MODE_PRE, restart_reason);
			write_OemRebootFlag(REBOOT_RAMDUMPS_MODE_PRE);
			dload_flag = 1;

		} else if (!strncmp(cmd, "ramdumped", 9)) {
			/*force into rumdumps mode, for test*/
			writel(REBOOT_RAMDUMPED_MODE, restart_reason);
			write_OemRebootFlag(REBOOT_RAMDUMPED_MODE);

		} else if (!strncmp(cmd, "modesel", 7)) {
			/*force into modesel mode*/
			writel(REBOOT_MODESEL_MODE1, restart_reason);
			write_OemRebootFlag(REBOOT_MODESEL_MODE1);

		} else if (!strncmp(cmd, "debug", 5)) {
			/*debugxxx, ramdump and go into fastboot*/
			writel(( FASTBOOT_DBG_MODE + hexstr_to_integer((char*)(cmd+5), 2) ), restart_reason);	
			write_OemRebootFlag(( FASTBOOT_DBG_MODE + hexstr_to_integer((char*)(cmd+5), 2) ));
			dload_flag = 1;		
#endif
		
		} else if (!strncmp(cmd, "alarm-on", 8)) {
			/*force into rtc-alarm power-on mode, for off-charging*/
			writel(REBOOT_ALARM_MODE, restart_reason);
			write_OemRebootFlag(REBOOT_ALARM_MODE);
			
		} else if (!strncmp(cmd, "pwrkey-on", 9)) {
			/*force into power-key power-on mode, for of-charging*/
			writel(REBOOT_PWRKEY_MODE, restart_reason);
			write_OemRebootFlag(REBOOT_PWRKEY_MODE);
			
		} else if (!strncmp(cmd, "charge", 6)) {
			/*force into charge power-on mode, for of-charging*/
			writel(REBOOT_CHARGE_MODE, restart_reason);
			write_OemRebootFlag(REBOOT_CHARGE_MODE);
			
		} else if (!strncmp(cmd, "ftd", 3)) {
			/*force into ftd test mode, for manufacture*/
			writel(REBOOT_FT_TEST_MODE, restart_reason);
			write_OemRebootFlag(REBOOT_FT_TEST_MODE);
			
		} else if (!strncmp(cmd, "dload-mode", 10)) {
			/*force into dload mode*/
			writel(REBOOT_DLOAD_MODE, restart_reason);	
			write_OemRebootFlag(REBOOT_DLOAD_MODE);
			my_set_dload_mode(1);

			dload_flag = 1;
		
		} else if (!strncmp(cmd, "oem-", 4)) {
			unsigned long code;
			strict_strtoul(cmd + 4, 16, &code);
			code = code & 0xff;
			writel(0x6f656d00 | code, restart_reason);
			write_OemRebootFlag(0x6f656d00 | code);
		} else {
			writel(NONE_MODE, restart_reason);		
			write_OemRebootFlag(NONE_MODE);	
		}
	} else {
			
		writel(NONE_MODE, restart_reason);		
		write_OemRebootFlag(NONE_MODE);	
	}	
	
	set_dload_mode(dload_flag);
	mb();

	writel(0, WDT0_EN);
	if (!(machine_is_msm8x60_charm_surf() ||
	      machine_is_msm8x60_charm_ffa())) {
		dsb();
		writel(0, PSHOLD_CTL_SU); /* Actually reset the chip */
		mdelay(5000);
		pr_notice("PS_HOLD didn't work, falling back to watchdog\n");
	}

	__raw_writel(1, WDT0_RST);
	__raw_writel(5*0x31F3, WDT0_BARK_TIME);
	__raw_writel(0x31F3, WDT0_BITE_TIME);
	__raw_writel(3, WDT0_EN);
	secure_writel(3, MSM_TCSR_BASE + TCSR_WDT_CFG);

	mdelay(10000);
	printk(KERN_ERR "Restarting has failed\n");
}

extern unsigned int get_mfg_mode(void);

static int __init msm_restart_init(void)
{
	void *imem = ioremap_nocache(IMEM_BASE, SZ_4K);
	int rc;

	dload_mode_addr = imem + DLOAD_MODE_ADDR;

#ifdef CONFIG_MSM_DLOAD_MODE
	atomic_notifier_chain_register(&panic_notifier_list, &panic_blk);

	/* Reset detection is switched on below.*/
	set_dload_mode(1);
#endif

	restart_reason = imem + RESTART_REASON_ADDR;
	pm_power_off = msm_power_off;

	if (pmic_reset_irq != 0) {
		rc = request_any_context_irq(pmic_reset_irq,
					resout_irq_handler, IRQF_TRIGGER_HIGH,
					"restart_from_pmic", NULL);
		if (rc < 0)
			pr_err("pmic restart irq fail rc = %d\n", rc);
	} else {
		pr_warn("no pmic restart interrupt specified\n");
	}

#ifdef CONFIG_RAM_DUMP_ENABLE
	{			
		/*force into rumdumps mode*/
		writel(REBOOT_RAMDUMPS_MODE, restart_reason);
		write_OemRebootFlag(REBOOT_RAMDUMPS_MODE);
		set_dload_mode(0);
		/*not enable ramdump in off-charge / factory mode*/
		if( (3 == get_mfg_mode()) 
			||(1 == get_mfg_mode())
			||(2 == get_ramdump_mode()) ){		
			writel(0, restart_reason);
			write_OemRebootFlag(0);
		}	
		else if(3 == get_ramdump_mode()) {
			writel(REBOOT_DLOAD_MODE, restart_reason);
			write_OemRebootFlag(REBOOT_DLOAD_MODE);
			set_dload_mode(1);
		}
		mb();
	}
#else
	{
		/*force into rumdumps mode*/
		writel(0, restart_reason);
		write_OemRebootFlag(0);
		set_dload_mode(0);
		mb();
	}
#endif	

	return 0;
}

void disable_default_ramdump(void)
{
	if(!restart_reason)
		writel(0, restart_reason);
		
	write_OemRebootFlag(0);
	set_dload_mode(0);
	mb();
}

EXPORT_SYMBOL(disable_default_ramdump);

late_initcall(msm_restart_init);
