/* drivers/input/misc/eadset_det.c
 *
 * Copyright (C) 2008-2011 Qisda Corporation
 * Author: Leo Qin <leo.qin@qisda.com>
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */


#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/gpio/gpio_def.h>
#include <linux/cdev.h>
#include <linux/interrupt.h>
#include <linux/uaccess.h>
#include <linux/fs.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <linux/irq.h>
#include <linux/input.h>
#include <mach/gpio.h>
#include <linux/device.h>
#include <mach/system.h>

MODULE_VERSION(DRIVER_VERSION);
MODULE_AUTHOR("Leo.Qin");
MODULE_DESCRIPTION("Orange gpio volume key detection");
MODULE_LICENSE("GPL v2");

#define DRIVER_VERSION "v1.0"

#define DEBUG

#ifdef DEBUG
#define VOL_KEY_INFO(fmt, arg...) printk(KERN_INFO "volKey: " fmt "\n" , ## arg)
#define VOL_KEY_DBG(fmt, arg...)  printk(KERN_INFO "%s: " fmt "\n" , __FUNCTION__ , ## arg)
#define VOL_KEY_ERR(fmt, arg...)  printk(KERN_ERR  "%s: " fmt "\n" , __FUNCTION__ , ## arg)
#else
#define VOL_KEY_INFO(fmt, arg...)
#define VOL_KEY_DBG(fmt, arg...)
#define VOL_KEY_ERR(fmt, arg...)
#endif

#ifdef DEBUG
#define assert(expr) \
        if(!(expr)) {				\
        do {} while (0);    \
        }
#else
#define assert(expr) \
        if(!(expr)) {					\
        printk( "Assertion failed! %s,%s,%s,line=%d\n",	\
        #expr,__FILE__,__func__,__LINE__);		\
        }
#endif

/* work function to volKey event */
void volKey_work_func(struct work_struct *work);

/* work function to monitor hook key status */
DECLARE_DELAYED_WORK(volKey_work, volKey_work_func);

#define MAX_KEY_NUM 4

struct gpio_vol_key_cfg{
	int	keyNum;
	int	hotKeyCombine[2];
	int	keyGpio[MAX_KEY_NUM];
	int	keyIrq[MAX_KEY_NUM];
	int	keyCode[MAX_KEY_NUM];
	int	keyWake[MAX_KEY_NUM];
};

struct gpio_vol_key_info{
	struct gpio_vol_key_cfg	*pVolCfg;
	int	keyState[MAX_KEY_NUM];
	int	keyValid[MAX_KEY_NUM];
	int	gpioState[MAX_KEY_NUM];
	int pressDebounce;
	int releaseDebounce;
	int nHotKeyPressed;
	int startCkeckHotKey;
	struct input_dev *volInput;
};

static struct gpio_vol_key_info	vol_key_data = {
	.keyState = {1,1,1,1},
	.pressDebounce = 30,
	.releaseDebounce = 30,
	.nHotKeyPressed = 1,
	.nHotKeyPressed = 1,
	/*Leo Qin: "volup+voldown" is used for emergency dload detect now, 
		so this mechanism is not necessary. */
	.startCkeckHotKey = 1/*0*/
};

const static struct gpio_vol_key_cfg volKeyCfg = {
	2,
	{0,1},
	{KEY_VOLUME_UP, KEY_VOLUME_DOWN, -1, -1},
	{MSM_GPIO_TO_INT(KEY_VOLUME_UP),MSM_GPIO_TO_INT(KEY_VOLUME_DOWN), -1, -1},
	{KEY_VOLUMEUP, KEY_VOLUMEDOWN, -1, -1},
/*	{KEY_BACK, KEY_MENU, -1, -1},*/
	{1,1,0,0}
};

struct gpio_vol_key_cfg * getVolKeyCFG(void)
{
	return (struct gpio_vol_key_cfg *)&volKeyCfg;
}



static void key_changed(unsigned int	keyIndex, unsigned int keyLevel)
{
	assert(keyIndex < vol_key_data.pVolCfg->keyNum);
	
	if(vol_key_data.keyState[keyIndex] != keyLevel)
	{
		input_report_key(vol_key_data.volInput, vol_key_data.pVolCfg->keyCode[keyIndex], !keyLevel);
		input_sync(vol_key_data.volInput);
		
		vol_key_data.keyState[keyIndex] = keyLevel;
		
		VOL_KEY_DBG("press");
	}
}

void volKey_work_func(struct work_struct *work)
{	
	int i;
	int keyLevel;
	
	for(i = 0; i < vol_key_data.pVolCfg->keyNum ; i++)
	{
		keyLevel = gpio_get_value(vol_key_data.pVolCfg->keyGpio[i]);
		if( keyLevel != vol_key_data.keyState[i])
			key_changed(i, keyLevel);
	}
}

#define HOTKEY_DELAY 0x100000
extern void pet_watchdog(void);

static irqreturn_t volIrq_handler(int irq, void *dev_id)
{
	int i, j;
	int level = -1;
	
	for(i = 0; i < vol_key_data.pVolCfg->keyNum ; i++)
	{
		vol_key_data.gpioState[i] = gpio_get_value(vol_key_data.pVolCfg->keyGpio[i]);
		if(irq == vol_key_data.pVolCfg->keyIrq[i])
			level = vol_key_data.gpioState[i];
	}
	
	/*the combination may be used for boot mode control, so if these keys were 
	  pressed from powering-on, skip the check.*/
	if(vol_key_data.startCkeckHotKey)
	{
		if( !(vol_key_data.gpioState[vol_key_data.pVolCfg->hotKeyCombine[0]] 
			  + vol_key_data.gpioState[vol_key_data.pVolCfg->hotKeyCombine[1]]))
		{
//			pet_watchdog();
			
			for(j = 0; j < HOTKEY_DELAY; j++)
			{
				for(i = 0; i < vol_key_data.pVolCfg->keyNum ; i++)
				{
					vol_key_data.gpioState[i] = gpio_get_value(vol_key_data.pVolCfg->keyGpio[i]);
					if(irq == vol_key_data.pVolCfg->keyIrq[i])
						level = vol_key_data.gpioState[i];
				}
				
				if(vol_key_data.gpioState[vol_key_data.pVolCfg->hotKeyCombine[0]] 
					  + vol_key_data.gpioState[vol_key_data.pVolCfg->hotKeyCombine[1]])
				{
					VOL_KEY_DBG("hotkey pressed, but released soon.");
					break;
				}
			}
			
			if( HOTKEY_DELAY == j)
			{
				vol_key_data.nHotKeyPressed = 0;
				
				/*!!!!!Leo Qin: invoke hotkey handler here !!!!*/
				VOL_KEY_DBG("hotkey pressed");
				arch_reset(0x5a, "kernel-ramdump");
			}
		}
	}
	else if( vol_key_data.gpioState[vol_key_data.pVolCfg->hotKeyCombine[0]] 
			  + vol_key_data.gpioState[vol_key_data.pVolCfg->hotKeyCombine[1]])
	{
		vol_key_data.startCkeckHotKey = 1;
	}
	
	if( -1 == level)
	{
		VOL_KEY_ERR("unsupported Irq??");
	}
	else
	{
		if(!level)
			schedule_delayed_work(&volKey_work, vol_key_data.pressDebounce);
		else
			schedule_delayed_work(&volKey_work, vol_key_data.releaseDebounce);
	}
				
	return IRQ_HANDLED;
}

static int __init gpio_vol_key_init(void)
{
	#define NAME_LEN 16
	int ret;
	char name[NAME_LEN];
	int i;

	printk(KERN_INFO "BootLog, +%s\n", __func__);
	
	memset(&vol_key_data, 0, sizeof(vol_key_data));
	
	/*Get config data*/
	vol_key_data.pVolCfg = getVolKeyCFG();
	if(!vol_key_data.pVolCfg){
		VOL_KEY_ERR("Fail to init .pVolCfg");
		return 0;
	}
	
	/*check and correct config*/
	if(1 > vol_key_data.pVolCfg->keyNum){
		VOL_KEY_ERR("Failed, keyNum is %d, abnormal\n!",vol_key_data.pVolCfg->keyNum);
		return 0;
	}
	for(i = 0; i < vol_key_data.pVolCfg->keyNum; i ++){
		if( (0 > vol_key_data.pVolCfg->keyGpio)
			||(0 > vol_key_data.pVolCfg->keyIrq)
			||(0 > vol_key_data.pVolCfg->keyCode) ){
			
			return 0;
		}
	}
	
	/*input device init*/
	vol_key_data.volInput = input_allocate_device();
	if (!vol_key_data.volInput) {
		ret = -ENOMEM;
		return 0;
	}

	vol_key_data.volInput->name = "Vol_key";

	for(i = 0; i < vol_key_data.pVolCfg->keyNum; i ++){
		input_set_capability(vol_key_data.volInput, EV_KEY, vol_key_data.pVolCfg->keyCode[i]);
	}
	
	ret = input_register_device(vol_key_data.volInput);
	if (ret < 0)
	{
		VOL_KEY_ERR("Fail to register input device");
		return 0;
	}
	
	/*Irq init and enable*/
	/*!!!!!Leo Qin: the concern of hotkey, 
	  use direct(non-gpio) interrupt and high priority interrupt!!!!!!!!!
	  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!*/	
	for(i = 0; i < vol_key_data.pVolCfg->keyNum; i ++){		
		memset(name,0,sizeof(name));
		sprintf(name,"VolKeyIO%d",i);
		ret = gpio_request(vol_key_data.pVolCfg->keyGpio[i],name);
		if(ret)
			break;
		
		ret = request_irq(vol_key_data.pVolCfg->keyIrq[i], volIrq_handler, 
											IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING, name, &vol_key_data);
		if(ret)
			break;
	}

	printk(KERN_INFO "BootLog, -%s, ret=%d\n", __func__, ret);

	/*Check key initial status*/
	volIrq_handler(vol_key_data.pVolCfg->keyIrq[0],&vol_key_data);
	
	return ret;
}


static void __exit gpio_vol_key_exit(void)
{
	VOL_KEY_DBG("Poweroff, +");
}

module_init(gpio_vol_key_init);
module_exit(gpio_vol_key_exit);

