/* drivers/input/misc/eadset_det.c
 *
 * Copyright (C) 2008-2011 Qisda Corporation
 * Author:Terry Cheng <terry.cheng@qisda.com>
 *        Leo Qin <leo.qin@qisda.com>
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/cdev.h>
#include <linux/interrupt.h>
#include <linux/uaccess.h>
#include <linux/fs.h>
#include <linux/platform_device.h>
#include <linux/delay.h>
#include <linux/irq.h>
#include <linux/switch.h>
#include <linux/input.h>
#include <mach/gpio.h>
#include <linux/device.h>
#include <mach/pmic.h>
#include <mach/oem_rapi_client.h>
#include <linux/pmic8058-othc.h>

#define DRIVER_VERSION "v2.0"

#define DEBUG
#ifdef DEBUG
#define HEADSET_INFO(fmt, arg...) printk(KERN_INFO "headset: " fmt "\n" , ## arg)
#define HEADSET_DBG(fmt, arg...)  printk(KERN_INFO "%s: " fmt "\n" , __FUNCTION__ , ## arg)
#define HEADSET_ERR(fmt, arg...)  printk(KERN_ERR  "%s: " fmt "\n" , __FUNCTION__ , ## arg)
#else
#define HEADSET_INFO(fmt, arg...)
#define HEADSET_DBG(fmt, arg...)
#define HEADSET_ERR(fmt, arg...)
#endif

#ifdef DEBUG
#define assert(expr) \
        if(!(expr)) {				\
        do {} while (0);    \
        }
#else
#define assert(expr) \
        if(!(expr)) {					\
        printk( "Assertion failed! %s,%s,%s,line=%d\n",	\
        #expr,__FILE__,__func__,__LINE__);		\
        }
#endif


void cable_ins_det_work_func(struct work_struct *work);


void check_hook_key_state_work_func(struct work_struct *work);



DECLARE_DELAYED_WORK(cable_ins_det_work, cable_ins_det_work_func);

DECLARE_DELAYED_WORK(check_hook_key_state_work, check_hook_key_state_work_func);



typedef enum  {
	HSD_NONE,
	BIT_HEADSET = (1 <<0),
	BIT_HEADSET_SPEAKER_ONLY = (1 << 1),
}headset_type;

typedef enum  {
	HSD_IRQ_DISABLE,
	HSD_IRQ_ENABLE
}irq_status;

struct headset_info{
	unsigned int	jack_gpio;
	unsigned int	hook_gpio;
	unsigned int	jack_irq;
	unsigned int	hook_irq;
	unsigned int	jack_status;
	unsigned int	button_state;
	headset_type	hs_type;
	unsigned int 	debounceDelay1;
	unsigned int 	debounceDelay2;
	unsigned int 	debounceDelayIns;
	unsigned int 	debounceDelayHookKey;
	int 	hookStateInHandler;		
	struct switch_dev sdev;
	struct input_dev *input;
} ;

static struct headset_info	headset_data;


MODULE_VERSION(DRIVER_VERSION);
MODULE_AUTHOR("Leo.Qin");
MODULE_DESCRIPTION("Orange Headset detection");
MODULE_LICENSE("GPL v2");


static void button_pressed(void)
{
	HEADSET_DBG("headset_data.button_state = %d", headset_data.button_state);
	if(headset_data.button_state)
	{
		headset_data.button_state = 0;
		input_report_key(headset_data.input, KEY_F24, 1);
		input_sync(headset_data.input);
		HEADSET_DBG("press");
	}
}

static void button_released(void)
{

	HEADSET_DBG("headset_data.button_state = %d", headset_data.button_state);
	if(!headset_data.button_state)
	{
		headset_data.button_state = 1;
		input_report_key(headset_data.input, KEY_F24, 0);
		input_sync(headset_data.input);
		HEADSET_DBG("release");
	}
}

int msm_micbias_en(unsigned char enable)
{
	int ret = -1;

#ifdef CONFIG_PMIC8058_OTHC
	if(enable)
		ret = pm8058_othc_svideo_enable(OTHC_MICBIAS_1,OTHC_SIGNAL_ALWAYS_ON);
	else
		ret = pm8058_othc_svideo_enable(OTHC_MICBIAS_1,	OTHC_SIGNAL_OFF);
#endif
		
	return ret;
}

void cable_ins_det_work_func(struct work_struct *work)
{
	int hook_key_status = 0;
	
	headset_data.jack_status  = gpio_get_value(headset_data.jack_gpio);	
	HEADSET_DBG("jack_gpio = %d", headset_data.jack_status);

	if( 0 == headset_data.jack_status)
	{
		
		msm_micbias_en(1);
		
		mdelay(headset_data.debounceDelay2);

		hook_key_status = gpio_get_value(headset_data.hook_gpio);
		HEADSET_DBG("hook_key_status = %d", hook_key_status);

#if 1
		if ( 1 == hook_key_status )
		{
			
			headset_data.hs_type = BIT_HEADSET_SPEAKER_ONLY;
			
			msm_micbias_en(0);
			
		}
		else
#endif
		{
			
			headset_data.hs_type = BIT_HEADSET;
		}
		headset_data.button_state = 1;
	}
	else
	{
		
		if (BIT_HEADSET == headset_data.hs_type)
		{
			
			msm_micbias_en(0);
			
			
		}
		headset_data.hs_type = HSD_NONE;
		
	}
	
	switch_set_state(&headset_data.sdev, headset_data.hs_type);
	HEADSET_DBG("hs_type = %d and notify android headset observer", headset_data.hs_type);
}

void check_hook_key_state_work_func(struct work_struct *work)
{
	int hook_key_status = 0;

	if ( HSD_NONE == headset_data.hs_type )
	{
		HEADSET_DBG("Headset removed!!");
		if(!headset_data.button_state)
		{
			button_released();
		}
		return;
	}
	
	if (BIT_HEADSET != headset_data.hs_type)
		return ;

	hook_key_status = gpio_get_value(headset_data.hook_gpio);	
	HEADSET_DBG("hook_gpio = %d,", hook_key_status);
	if ( 1 == hook_key_status )
	{
		
		if(headset_data.hookStateInHandler == hook_key_status)
		{
			button_pressed();
		}
	}
	else
	{
		
		if(headset_data.hookStateInHandler == hook_key_status)
		{
			button_released();
		}
	}
}

static irqreturn_t hook_irqhandler(int irq, void *dev_id)
{
	headset_data.hookStateInHandler = gpio_get_value(headset_data.hook_gpio);
	HEADSET_DBG("hook_gpio = %d", headset_data.hookStateInHandler);

	

	
	schedule_delayed_work(&check_hook_key_state_work, headset_data.debounceDelayHookKey);
	return IRQ_HANDLED;
}

static irqreturn_t jack_irqhandler(int irq, void *dev_id)
{
	HEADSET_DBG("jack_gpio = %d", gpio_get_value(headset_data.jack_gpio));

	
	schedule_delayed_work(&cable_ins_det_work, headset_data.debounceDelayIns);
	return IRQ_HANDLED;
}

static ssize_t switch_hds_print_state(struct switch_dev *sdev, char *buf)
{

	switch (switch_get_state(&headset_data.sdev)) {
		case HSD_NONE:
			return sprintf(buf, "NoDevice\n");
		case BIT_HEADSET:
			return sprintf(buf, "Headset\n");
		case BIT_HEADSET_SPEAKER_ONLY:
			return sprintf(buf, "Headset\n");
	}

	return -EINVAL;
}


static int __init_or_module headset_probe(struct platform_device *pdev)
{

	int ret;
	struct resource *res;
	unsigned int	irq;

	
	memset(&headset_data, 0, sizeof( headset_data));

	
	headset_data.sdev.name = pdev->name;
	headset_data.sdev.print_name = switch_hds_print_state;

	ret = switch_dev_register(&headset_data.sdev);
	if (ret < 0)
	{
		goto err_switch_dev_register;
	}
	headset_data.debounceDelay1 = 300;
	headset_data.debounceDelay2 = 200;
	headset_data.debounceDelayHookKey = 60;	
	headset_data.debounceDelayIns	= 15;
	headset_data.jack_status = 0;
	headset_data.hs_type = HSD_NONE;

	res = platform_get_resource_byname(pdev, IORESOURCE_IRQ,
				"hook_int");
	if (!res) {
		HEADSET_ERR("couldn't find hook int\n");
		ret = -ENODEV;
		goto err_get_hook_gpio_resource;
	}
	headset_data.hook_gpio = res->start;
	HEADSET_INFO("headset_probe headset_data.hook_gpio = %d\n", headset_data.hook_gpio);

	ret = gpio_request(headset_data.hook_gpio, "headset_hook");
	if (ret < 0)
		goto err_request_hook_gpio;

	ret = gpio_direction_input(headset_data.hook_gpio);
	if (ret < 0)
		goto err_set_hook_gpio;

	
	irq = MSM_GPIO_TO_INT(headset_data.hook_gpio);

	if (irq <0)
	{
		ret = irq;
		goto err_get_hook_detect_irq_num_failed;
	}
	headset_data.hook_irq = irq;
	HEADSET_INFO("headset_probe headset_data.hook_irq = %d\n", headset_data.hook_irq);

	ret = request_irq(headset_data.hook_irq, hook_irqhandler, IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING, "headset_hook", &headset_data);
	if( ret < 0)
	{
		HEADSET_ERR("Fail to request hook irq");
		goto err_request_austin_headset_hook_irq;
	}

	res = platform_get_resource_byname(pdev, IORESOURCE_IRQ,
				"jack_int");
	if (!res) {
		HEADSET_ERR("couldn't find jack int\n");
		ret = -ENODEV;
		goto err_get_jack_int_resource;
	}
	headset_data.jack_gpio = res->start;
	HEADSET_INFO("headset_probe headset_data.jack_int = %d\n", headset_data.jack_gpio);

	ret = gpio_request(headset_data.jack_gpio, "headset_jack");
	if (ret < 0)
		goto err_request_jack_gpio;

	ret = gpio_direction_input(headset_data.jack_gpio);
	if (ret < 0)
		goto err_set_jack_gpio;

	
	mdelay(300);
	
	irq = MSM_GPIO_TO_INT(headset_data.jack_gpio);
	if (irq <0)
	{
		ret = irq;
		goto err_get_jack_detect_irq_num_failed;
	}
	headset_data.jack_irq = irq;

	HEADSET_INFO("headset_probe headset_data.jack_irq = %d\n", headset_data.jack_irq);

	ret = request_irq(headset_data.jack_irq, jack_irqhandler, IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING, "headset_jack", &headset_data);
	if (ret < 0)
	{
		HEADSET_ERR("Fail to request jack irq");
		goto err_request_austin_headset_jack_irq;
	}

	ret = set_irq_wake(headset_data.jack_irq, 1);
	if (ret < 0)
		goto err_request_input_dev;

#if 0		
	ret = set_irq_wake(headset_data.hook_irq, 1);
	if (ret < 0)
		goto err_request_input_dev;
#endif

	headset_data.input = input_allocate_device();
	if (!headset_data.input) {
		ret = -ENOMEM;
		goto err_request_input_dev;
	}
	headset_data.input->name = "Headset_key";
	headset_data.input->evbit[0] = BIT_MASK(EV_KEY);
	set_bit(KEY_F24, headset_data.input->keybit);
	ret = input_register_device(headset_data.input);
	if (ret < 0)
	{
		HEADSET_ERR("Fail to register input device");
		goto err_register_input_dev;
	}

	
	schedule_delayed_work(&cable_ins_det_work, 800);
	
	return 0;
err_register_input_dev:
	input_free_device(headset_data.input);
err_request_input_dev:
	free_irq(headset_data.jack_irq, 0);
err_request_austin_headset_jack_irq:
err_get_jack_detect_irq_num_failed:
err_set_jack_gpio:
	gpio_free(headset_data.jack_gpio);
err_request_jack_gpio:
err_get_jack_int_resource:
	free_irq(headset_data.hook_irq, 0);
err_request_austin_headset_hook_irq:
err_get_hook_detect_irq_num_failed:
err_set_hook_gpio:
	 gpio_free(headset_data.hook_gpio);
err_request_hook_gpio:
err_get_hook_gpio_resource:
	switch_dev_unregister(&headset_data.sdev);
err_switch_dev_register:
	HEADSET_ERR(" Failed to register driver");

	return ret;

}

static int headset_remove(struct platform_device *pdev)
{

	HEADSET_DBG("");

	input_unregister_device(headset_data.input);
	free_irq(headset_data.hook_irq, 0);
	free_irq(headset_data.jack_irq, 0);
	switch_dev_unregister(&headset_data.sdev);
	return 0;

}

static struct platform_driver headset_driver = {
	.probe = headset_probe,
	.remove = headset_remove,
	.driver = {
		.name = "h2w",
		.owner = THIS_MODULE,
	},
};


static int __init headset_init(void)
{
	int ret;

	printk(KERN_INFO "BootLog, +%s\n", __func__);
	
	ret = platform_driver_register(&headset_driver);
	printk(KERN_INFO "BootLog, -%s, ret=%d\n", __func__, ret);

	return ret;
}


static void __exit headset_exit(void)
{

	HEADSET_DBG("Poweroff, +");
	platform_driver_unregister(&headset_driver);

}

module_init(headset_init);
module_exit(headset_exit);

