/* Copyright (c) 2010-2011, Qisda Corporation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

#define DEBUG

#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/input.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/mfd/pmic8058.h>
#include <linux/pmic8058-cblpwrkey.h>
#include <linux/log2.h>
#include <linux/spinlock.h>
#include <linux/hrtimer.h>
#include <linux/pm.h>
#include <linux/slab.h>
#include <linux/pm_runtime.h>

#define PON_CNTL_1	0x1C
#define PON_CNTL_PULL_UP BIT(7)
#define PON_CNTL_TRIG_DELAY_MASK (0x7)

struct pmic8058_cblpwrkey {
	struct input_dev *cblpwr;
	int cbl_det_irq;
	struct pm8058_chip	*pm_chip;
	struct hrtimer timer;
	bool key_pressed;
	bool pressed_first;
	struct pmic8058_cblpwrkey_pdata *pdata;
	spinlock_t lock;
};

#ifndef DEBUG
#define my_printk(fmt, args...)
#else
#define my_printk printk
#endif

static enum hrtimer_restart pmic8058_cblpwrkey_timer(struct hrtimer *timer)
{
	unsigned long flags;
	struct pmic8058_cblpwrkey *cblpwrkey = container_of(timer,
						struct pmic8058_cblpwrkey,	timer);

	my_printk(KERN_DEBUG  "pmic8058_cblpwrkey_timer in\n");

	spin_lock_irqsave(&cblpwrkey->lock, flags);
	cblpwrkey->key_pressed = true;

	input_report_key(cblpwrkey->cblpwr, KEY_POWER, 1);
	my_printk(KERN_DEBUG  "pmic8058_cblpwrkey_timer PWR press\n");
	input_sync(cblpwrkey->cblpwr);
	spin_unlock_irqrestore(&cblpwrkey->lock, flags);

	return HRTIMER_NORESTART;
}

static irqreturn_t cblpwrkey_det_irq(int irq, void *_cblpwrkey)
{
	struct pmic8058_cblpwrkey *cblpwrkey = _cblpwrkey;
	struct pmic8058_cblpwrkey_pdata *pdata = cblpwrkey->pdata;
	unsigned long flags;
	
	int re_state;
	
	re_state = pm8058_irq_get_rt_status(cblpwrkey->pm_chip, irq);

	my_printk(KERN_DEBUG "cblpwrkey_det_irq in, re_state is %d\n",re_state);
  
	spin_lock_irqsave(&cblpwrkey->lock, flags);
	if (re_state) {
		cblpwrkey->pressed_first = true;
		
		if (!cblpwrkey->pdata->cblpwrkey_time_ms) {
			input_report_key(cblpwrkey->cblpwr, KEY_POWER, 1);
			input_sync(cblpwrkey->cblpwr);
			spin_unlock_irqrestore(&cblpwrkey->lock, flags);
			return IRQ_HANDLED;
		}

		input_report_key(cblpwrkey->cblpwr, KEY_END, 1);
		my_printk(KERN_DEBUG "cblpwrkey_det_irq END press\n");
		input_sync(cblpwrkey->cblpwr);

		hrtimer_start(&cblpwrkey->timer,
				ktime_set(pdata->cblpwrkey_time_ms / 1000,
				(pdata->cblpwrkey_time_ms % 1000) * 1000000),
				HRTIMER_MODE_REL);
	}
	else{
		cblpwrkey->pressed_first = false;
		
		if (!cblpwrkey->pdata->cblpwrkey_time_ms) {
			input_report_key(cblpwrkey->cblpwr, KEY_POWER, 0);
			input_sync(cblpwrkey->cblpwr);
			spin_unlock_irqrestore(&cblpwrkey->lock, flags);
			return IRQ_HANDLED;
		}

		my_printk(KERN_DEBUG "cblpwrkey_det_irq,hrtimer_cancel\n");
		hrtimer_cancel(&cblpwrkey->timer);

		if (cblpwrkey->key_pressed) {
			cblpwrkey->key_pressed = false;
			input_report_key(cblpwrkey->cblpwr, KEY_POWER, 0);
			my_printk(KERN_DEBUG "cblpwrkey_det_irq PWR release\n");
			input_sync(cblpwrkey->cblpwr);
		}

		input_report_key(cblpwrkey->cblpwr, KEY_END, 0);
		my_printk(KERN_DEBUG "cblpwrkey_det_irq END release\n");
		input_sync(cblpwrkey->cblpwr);
	}
		
	spin_unlock_irqrestore(&cblpwrkey->lock, flags);

	return IRQ_HANDLED;
}

#ifdef CONFIG_PM
static int pmic8058_cblpwrkey_suspend(struct device *dev)
{
	struct pmic8058_cblpwrkey *cblpwrkey = dev_get_drvdata(dev);

	if (device_may_wakeup(dev)) {
		enable_irq_wake(cblpwrkey->cbl_det_irq);
	}

	return 0;
}

static int pmic8058_cblpwrkey_resume(struct device *dev)
{
	struct pmic8058_cblpwrkey *cblpwrkey = dev_get_drvdata(dev);

	if (device_may_wakeup(dev)) {
		disable_irq_wake(cblpwrkey->cbl_det_irq);
	}

	return 0;
}

static struct dev_pm_ops pm8058_cblpwr_key_pm_ops = {
	.suspend	= pmic8058_cblpwrkey_suspend,
	.resume		= pmic8058_cblpwrkey_resume,
};
#endif

static int __devinit pmic8058_cblpwrkey_probe(struct platform_device *pdev)
{
	struct input_dev *cblpwr;
	int cbl_det_irq = platform_get_irq(pdev, 0);
	int err;
	unsigned int delay;
	struct pmic8058_cblpwrkey *cblpwrkey;
	struct pmic8058_cblpwrkey_pdata *pdata = pdev->dev.platform_data;
	struct pm8058_chip	*pm_chip;

	pm_chip = platform_get_drvdata(pdev);
	if (pm_chip == NULL) {
		dev_err(&pdev->dev, "no parent data passed in\n");
		return -EFAULT;
	}

	if (!pdata) {
		dev_err(&pdev->dev, "cbl power key platform data not supplied\n");
		return -EINVAL;
	}

	if (pdata->kpd_trigger_delay_us > 62500) {
		dev_err(&pdev->dev, "invalid cblpwr key trigger delay\n");
		return -EINVAL;
	}

	if (pdata->cblpwrkey_time_ms &&
	     (pdata->cblpwrkey_time_ms < 500 || pdata->cblpwrkey_time_ms > 1000)) {
		dev_err(&pdev->dev, "invalid cblpwr key time supplied\n");
		return -EINVAL;
	}

	cblpwrkey = kzalloc(sizeof(*cblpwrkey), GFP_KERNEL);
	if (!cblpwrkey)
		return -ENOMEM;

	cblpwrkey->pm_chip = pm_chip;
	cblpwrkey->pdata   = pdata;
	cblpwrkey->pressed_first = false;
	
	err = pm_runtime_set_active(&pdev->dev);
	if (err < 0)
		dev_dbg(&pdev->dev, "unable to set runtime pm state\n");
	pm_runtime_enable(&pdev->dev);

	cblpwr = input_allocate_device();
	if (!cblpwr) {
		dev_dbg(&pdev->dev, "Can't allocate power button\n");
		err = -ENOMEM;
		goto free_cblpwrkey;
	}

	input_set_capability(cblpwr, EV_KEY, KEY_POWER);
	input_set_capability(cblpwr, EV_KEY, KEY_END);

	cblpwr->name = "pmic8058_cblpwrkey";
	cblpwr->phys = "pmic8058_cblpwrkey/input0";
	cblpwr->dev.parent = &pdev->dev;

	delay = (pdata->kpd_trigger_delay_us << 10) / USEC_PER_SEC;
	delay = 1 + ilog2(delay);

	spin_lock_init(&cblpwrkey->lock);

	hrtimer_init(&cblpwrkey->timer, CLOCK_MONOTONIC, HRTIMER_MODE_REL);
	cblpwrkey->timer.function = pmic8058_cblpwrkey_timer;

	err = input_register_device(cblpwr);
	if (err) {
		dev_dbg(&pdev->dev, "Can't register power key: %d\n", err);
		goto free_input_dev;
	}

	cblpwrkey->cbl_det_irq = cbl_det_irq;
	cblpwrkey->cblpwr = cblpwr;

	platform_set_drvdata(pdev, cblpwrkey);

	err = request_threaded_irq(cbl_det_irq, NULL, cblpwrkey_det_irq,
			 IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING, "pmic8058_cblpwrkey_det", cblpwrkey);
	if (err < 0) {
		dev_dbg(&pdev->dev, "Can't get %d IRQ for cblpwrkey: %d\n",
				 cbl_det_irq, err);
		goto unreg_input_dev;
	}

	device_init_wakeup(&pdev->dev, pdata->wakeup);

	return 0;

unreg_input_dev:
	input_unregister_device(cblpwr);
	cblpwr = NULL;
free_input_dev:
	input_free_device(cblpwr);
free_cblpwrkey:
	pm_runtime_set_suspended(&pdev->dev);
	pm_runtime_disable(&pdev->dev);
	kfree(cblpwrkey);
	return err;
}

static int __devexit pmic8058_cblpwrkey_remove(struct platform_device *pdev)
{
	struct pmic8058_cblpwrkey *cblpwrkey = platform_get_drvdata(pdev);
	int cbl_det_irq = platform_get_irq(pdev, 0);

	pm_runtime_set_suspended(&pdev->dev);
	pm_runtime_disable(&pdev->dev);
	device_init_wakeup(&pdev->dev, 0);

	free_irq(cbl_det_irq, cblpwrkey);
	input_unregister_device(cblpwrkey->cblpwr);
	kfree(cblpwrkey);

	return 0;
}

static struct platform_driver pmic8058_cblpwrkey_driver = {
	.probe		= pmic8058_cblpwrkey_probe,
	.remove		= __devexit_p(pmic8058_cblpwrkey_remove),
	.driver		= {
		.name	= "pm8058-cblpwrkey",
		.owner	= THIS_MODULE,
#ifdef CONFIG_PM
		.pm	= &pm8058_cblpwr_key_pm_ops,
#endif
	},
};

static int __init pmic8058_cblpwrkey_init(void)
{
	return platform_driver_register(&pmic8058_cblpwrkey_driver);
}
module_init(pmic8058_cblpwrkey_init);

static void __exit pmic8058_cblpwrkey_exit(void)
{
	platform_driver_unregister(&pmic8058_cblpwrkey_driver);
}
module_exit(pmic8058_cblpwrkey_exit);

MODULE_ALIAS("platform:pmic8058_cblpwrkey");
MODULE_DESCRIPTION("PMIC8058 cable-detect Power Key");
MODULE_LICENSE("GPL v2");
