/* drivers/input/touchscreen/auo_touch.c
 *
 * Copyright (c) 2008 QUALCOMM Incorporated. 
 * Copyright (c) 2008 QUALCOMM USA, INC.
 *
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/input.h>
#include <linux/platform_device.h>
#include <linux/jiffies.h>
#include <linux/io.h>
#include <linux/uaccess.h>
#include <linux/miscdevice.h>
#include <linux/i2c.h>
#include <linux/semaphore.h>
#include <linux/delay.h>
#include <linux/slab.h>

#ifdef CONFIG_HAS_EARLYSUSPEND
#include <linux/earlysuspend.h>
#endif
#include <linux/time.h>
#include <linux/mutex.h>

#include <mach/gpio.h>
#include <linux/gpio/gpio_def.h>
#include <mach/vreg.h>
#include <mach/auo_touch.h>
#include <linux/pmic8058-othc.h>
#include <linux/mfd/pmic8901.h>
#include <linux/mfd/pmic8058.h>
#include <linux/regulator/pmic8058-regulator.h>
#include <linux/regulator/pmic8901-regulator.h>
#include <linux/regulator/consumer.h>
#include <linux/regulator/machine.h>


#define SPEED_DISABLETOUCH_WHEN_PSENSORACTIVE 1

#if SPEED_DISABLETOUCH_WHEN_PSENSORACTIVE
// 20100521 sean, psensor status indication for key reporting.
atomic_t psensor_approached;
#endif

#define GETTOUCHLOG_AFTER_SCREENOFF 1

#if GETTOUCHLOG_AFTER_SCREENOFF
atomic_t GetTouchLog1AfterLaterResume = ATOMIC_INIT(0);
atomic_t GetTouchLog2AfterLaterResume = ATOMIC_INIT(0);
#endif


#define TCH_DBG(fmt, args...) printk(KERN_INFO "AUO_TOUCH: " fmt, ##args)

static uint printCoord = 0;
module_param(printCoord, uint, 0644);

static uint printDebug = 0;
module_param(printDebug, uint, 0644);

static uint printReport = 0;
module_param(printReport, uint, 0644);

struct ts_t {
	struct i2c_client        *client_4_i2c;
	struct input_dev         *input;
	int                      irq;
	int                      gpio_num;
	int                      gpio_rst;
/*Orange EVT2 CR XXXX Jack W Lu 2011-09-03 Add for L1B vote, Power Consumption issue {*/
	int (*pm8901_l1b_init)(int on);
	int (*pm8901_l1b_vote_set)(int on);
/*Orange EVT2 CR XXXX Jack W Lu 2011-09-03 Add for L1B vote, Power Consumption issue }*/
	int                      open_count;
	struct delayed_work      touch_work;
	struct workqueue_struct  *touch_wqueue;
	struct auo_reg_map_t     reg_map;
	struct auo_mt_t          mtMsg;
#ifdef CONFIG_HAS_EARLYSUSPEND
	struct early_suspend     touch_early_suspend;
#endif
};

#ifdef USE_PIXCIR
struct ts_power_t {
	char     *id;
	unsigned mv;
};
struct ts_power_t const auo_power = {"gp2",  3000};
#endif

static struct ts_t         *g_ts;

#if TOUCH_INDI_INT_MODE
static uint8_t             g_touch_ind_mode = 1;
#else
static uint8_t             g_touch_ind_mode = 0;
#endif
DECLARE_MUTEX( ts_sem );

static uint8_t             g_touch_suspended = 0;
DECLARE_MUTEX( auo_ts_suspend_sem );

static uint8_t             g_pixcir_detected = 0;
static uint8_t             g_pixcir_freeze = 0;


static int __devinit auo_ts_probe(struct i2c_client *,  const struct i2c_device_id *);

static int auo_ts_suspend_ic(struct ts_t *, struct i2c_client *);
static int auo_ts_resume_ic(struct ts_t *, struct i2c_client *);
static ssize_t auo_touch_dump_property(struct device *dev, struct device_attribute *attr, char *buf);

#define TOUCH_RETRY_COUNT 5

static int gpio_get_value_for_touch(unsigned gpio)
{
#if defined(CONFIG_MACH_EVT2)
	return gpio_get_value(gpio);
#elif defined (CONFIG_MACH_EVT1)
	return gpio_get_value_cansleep(gpio);
#endif

}
static void gpio_set_value_for_touch(unsigned gpio,int value)
{
#if defined(CONFIG_MACH_EVT2)
	gpio_set_value(gpio,value);
#elif defined (CONFIG_MACH_EVT1)
	gpio_set_value_cansleep(gpio,value);
#endif
}

static int ts_write_i2c( struct i2c_client *client,
                         uint8_t           regBuf,
                         uint8_t           *dataBuf,
                         uint8_t           dataLen )
{
	int     result = 0;
	uint8_t *buf = NULL;
	int     retryCnt = TOUCH_RETRY_COUNT;

	struct  i2c_msg msgs[] = {
		[0] = {
		.addr   = client->addr,
		.flags  = 0,
		.buf    = (void *)buf,
 		.len    = 0
		}
	};

	buf = kzalloc( dataLen+1, GFP_KERNEL );
	if( NULL == buf ){
		TCH_DBG("ts_write_i2c: alloc memory failed\n");
		return -EFAULT;
	}

	buf[0] = regBuf;
	memcpy( &buf[1], dataBuf, dataLen );
	msgs[0].buf = buf;
	msgs[0].len = dataLen+1;

	while( retryCnt ){
		result = i2c_transfer( client->adapter, msgs, 1 );
		if( result != ARRAY_SIZE(msgs) ){
			TCH_DBG("ts_write_i2c: write %Xh %d bytes return failure, %d\n", buf[0], dataLen, result);;
			if( -ETIMEDOUT == result ) msleep(10);
				retryCnt--;
			}else {
				result = 0;
				break;
			         }
		}

	if( (result == 0) && (retryCnt < TOUCH_RETRY_COUNT) )
		TCH_DBG("ts_write_i2c: write %Xh %d bytes retry at %d\n", buf[0], dataLen, TOUCH_RETRY_COUNT-retryCnt);

	kfree( buf );

	return result;
}

static int ts_read_i2c( struct i2c_client *client,
                        uint8_t           regBuf,
                        uint8_t           *dataBuf,
                        uint8_t           dataLen )
{
	int     result = 0;
	int     retryCnt = TOUCH_RETRY_COUNT;

	struct  i2c_msg msgs[] = {
		[0] = {
		.addr   = client->addr,
		.flags  = 0,
		.buf    = (void *)&regBuf,
		.len    = 1
		},
		 [1] = {
		.addr   = client->addr,
		.flags  = I2C_M_RD,
		.buf    = (void *)dataBuf,
		.len    = dataLen
		}
	};

	
	while(retryCnt){
		result = i2c_transfer( client->adapter, msgs, 2 );

		if (result != ARRAY_SIZE(msgs)){
			TCH_DBG("ts_read_i2c: read %Xh %d bytes return failure, %d\n", regBuf, dataLen, result );
			if(-ETIMEDOUT == result) 
				msleep(10);
				retryCnt--;
		} else{
			result = 0;
			break;
		      }
	}

	return result;
}


static int boot_read_mem(struct i2c_client *client,uint16_t start, uint8_t size, uint8_t *mem)
{
	int     result;
	struct  i2c_msg msgs[] = { 
		[0] = {                     
		.addr   = 0x5D,
		.flags  = I2C_M_RD,
		.buf    = (void *)mem,
		.len    = size
		}
	};
    

	result = i2c_transfer(client->adapter, msgs, ARRAY_SIZE(msgs) );
	if( result != ARRAY_SIZE(msgs) ){
		TCH_DBG( "ERROR_LEVEL:""%s fail\n", __func__);
		return -1;
	}

	return 0;
}

static int boot_ts_write_i2c( struct i2c_client *client,
                         uint8_t           regBuf,
                         uint8_t           *dataBuf,
                         uint8_t           dataLen )
{
	int     result = 0;
	uint8_t *buf = NULL;
	int     retryCnt = TOUCH_RETRY_COUNT;

	struct  i2c_msg msgs[] = {
		[0] = {
		.addr   = 0x5d,
		.flags  = 0,
		.buf    = (void *)buf,
		.len    = 0
		}
	};

	buf = kzalloc( dataLen+1, GFP_KERNEL );
	if( NULL == buf ){
		TCH_DBG("ts_write_i2c: alloc memory failed\n");
		return -EFAULT;
	}

	memcpy( &buf[0], dataBuf, dataLen );
	msgs[0].buf = buf;
	msgs[0].len = dataLen;

	while( retryCnt ){
		result = i2c_transfer( client->adapter, msgs, 1 );
		if( result != ARRAY_SIZE(msgs) ){
			TCH_DBG("ts_write_i2c: write %Xh %d bytes return failure, %d\n", buf[0], dataLen, result);
			if( -ETIMEDOUT == result ) msleep(10);
				retryCnt--;
		}else {
			result = 0;
			break;
		      }
	}

	if( (result == 0) && (retryCnt < TOUCH_RETRY_COUNT) )
		TCH_DBG("ts_write_i2c: write %Xh %d bytes retry at %d\n", buf[0], dataLen, TOUCH_RETRY_COUNT-retryCnt);

	kfree( buf );
	return result;
}

static void ts_report_coord_via_mt_protocol(struct ts_t *pTS, struct auo_point_info_t *pInfo)
{
	int i;

	for(i=0;i<AUO_REPORT_POINTS;i++){
		if((pInfo+i)->state == AUO_RELEASE){
			input_report_abs(pTS->input, ABS_MT_TOUCH_MAJOR, 0);
			input_report_abs(pTS->input, ABS_MT_WIDTH_MAJOR, 0);
			input_report_abs(pTS->input, ABS_MT_POSITION_X, 0);
			input_report_abs(pTS->input, ABS_MT_POSITION_Y, 0);
		} else {
			input_report_abs(pTS->input, ABS_MT_TOUCH_MAJOR, 10);
			input_report_abs(pTS->input, ABS_MT_WIDTH_MAJOR, 10);
			input_report_abs(pTS->input, ABS_MT_POSITION_X, (pInfo+i)->coord.x);
			input_report_abs(pTS->input, ABS_MT_POSITION_Y, (pInfo+i)->coord.y);
		       }
		 
		input_mt_sync(pTS->input);
	}
	input_sync(pTS->input);
}

static void ts_fill_mt_info(struct auo_point_info_t *pInfo, struct auo_point_t *pPoint)
{
	int i, x, y;

	for(i=0; i<AUO_REPORT_POINTS; i++){
		x = (pPoint+i)->x;
		y = (pPoint+i)->y;
		if(x > 0 && y > 0){
			(pInfo+i)->coord.x = AUO_X_MAX + 1 - x;
			(pInfo+i)->coord.y = AUO_Y_MAX + 1 - y;			
			//(pInfo+i)->coord.x = x;
			//(pInfo+i)->coord.y = y;
			(pInfo+i)->state = AUO_PRESS;
		} else {
			(pInfo+i)->coord.x = 0;
			(pInfo+i)->coord.y = 0;
			(pInfo+i)->state = AUO_RELEASE;
		       }
		}
}

static void ts_get_touch_state(enum auo_mt_state *touch,
                                struct auo_point_t *point)
{
	if(point[0].x == 0 || point[0].y == 0){
		*touch = AUO_NOT_TOUCH_STATE;
		memset( point, 0, sizeof(struct auo_point_t)*AUO_REPORT_POINTS );
	} else if (point[1].x == 0 || point[1].y == 0){
		*touch = AUO_ONE_TOUCH_STATE;
		memset(&point[1], 0, sizeof(struct auo_point_t));
		} else {
			*touch = AUO_TWO_TOUCH_STATE;
		       }

	return;
}

static void ts_report_coord(struct ts_t *pTS,
                             struct auo_point_t *point)
{

	enum auo_mt_state        mt_state;
	struct auo_point_info_t  mt_points[AUO_REPORT_POINTS];

	
	ts_get_touch_state(&mt_state, point);
	ts_fill_mt_info(mt_points, point);
// report via mt protocol
	ts_report_coord_via_mt_protocol(pTS, mt_points);

	return;
}

static void ts_irqWorkHandler(struct work_struct *work)
{
	struct ts_t *pTS = container_of(work,
                                    struct ts_t,
                                    touch_work.work);
	struct i2c_client           *client = pTS->client_4_i2c;
	struct auo_point_t          point[AUO_REPORT_POINTS];
	uint8_t                     rawCoord[8];
	uint8_t                     i;

	struct sched_param s = { .sched_priority = 1 };

	
	if(!rt_task(current)){
		if( sched_setscheduler_nocheck(current, SCHED_FIFO, &s) != 0 )
			TCH_DBG("%s: unable to set rt priority\n", __func__);
	}
    
    
	if( !ts_read_i2c( client, pTS->reg_map.x1_lsb, rawCoord, 8 ) ){
		for( i=0; i<ARRAY_SIZE(point); i++ ){
#if USE_AUO_5INCH
			if( 1 == g_pixcir_detected ){
#if defined(TOUCH_FB_PORTRAIT)
				point[i].x = rawCoord[4*i+1]<<8 | rawCoord[4*i];
				point[i].y = rawCoord[4*i+3]<<8 | rawCoord[4*i+2];			
#else
				point[i].y = rawCoord[4*i+1]<<8 | rawCoord[4*i];
				point[i].x = rawCoord[4*i+3]<<8 | rawCoord[4*i+2];

				if( 0 != point[i].x ){
					point[i].x = AUO_X_MAX + 1 - point[i].x;
				}
#endif
			} else {
#if defined(TOUCH_FB_PORTRAIT)
				point[i].x = (rawCoord[2*i+4]<<8) | rawCoord[2*i];
				point[i].y = (rawCoord[2*i+5]<<8) | rawCoord[2*i+1];
				if( point[i].y != 0 )
					point[i].y = AUO_Y_MAX + 1 - point[i].y;
#else
				point[i].y = (rawCoord[2*i+4]<<8) | rawCoord[2*i];
				point[i].x = (rawCoord[2*i+5]<<8) | rawCoord[2*i+1];
#endif
			       }
#else
			point[i].x = (rawCoord[2*i+4]<<8) | rawCoord[2*i];
			point[i].y = (rawCoord[2*i+5]<<8) | rawCoord[2*i+1];
#endif
			if( point[i].x > AUO_X_MAX || point[i].y > AUO_Y_MAX ){
				point[i].x = point[i].y = 0;
			}

		}

		ts_report_coord( pTS, point );
	} 
	
	enable_irq( pTS->irq );

	return;
}

static irqreturn_t ts_irqHandler(int irq, void *dev_id)
{
	struct ts_t *ts = dev_id;

#if GETTOUCHLOG_AFTER_SCREENOFF
	if (atomic_read(&GetTouchLog2AfterLaterResume) >0 ){
		atomic_dec(&GetTouchLog2AfterLaterResume);
	}
#endif
	disable_irq_nosync( irq );
	queue_delayed_work(ts->touch_wqueue, &ts->touch_work, 0);
	return IRQ_HANDLED;
}

static int ts_release_gpio(int gpio_num , int gpio_rst)
{
	TCH_DBG( "ts_release_gpio: releasing gpio %d\n", gpio_num );
	gpio_free( gpio_num );
	return 0;
}

/*Qsida,Alan P Xu,2011.07.27, GPIO of touch be changed{*/
#if defined(CONFIG_MACH_EVT2)
static int ts_setup_gpio(int gpio_num,int gpio_rst)
{
	int rc;
/*Qisda, Alan.P.Xu, 2012.02.21, mach free gpio{*/	
	rc = gpio_request( gpio_num, "auo_touchscreen_irq" );
	if( rc )	TCH_DBG("[Leo], AUO Touch Screen irq request gpio fail \n");
/*Qisda, Alan.P.Xu, 2012.02.21, mach free gpio}*/
	
	rc = gpio_tlmm_config(GPIO_CFG(gpio_num,0,GPIO_CFG_INPUT,GPIO_CFG_NO_PULL,GPIO_CFG_2MA), GPIO_CFG_ENABLE);//touch irq
	if( rc )	TCH_DBG("[Leo], AUO Touch Screen TOUCH_IRQ fail \n");
    
	rc = gpio_tlmm_config(GPIO_CFG(gpio_rst, 0, GPIO_CFG_OUTPUT, GPIO_CFG_PULL_UP, GPIO_CFG_10MA), GPIO_CFG_ENABLE);//touch rst
	if( rc )	TCH_DBG("[Leo], AUO Touch Screen TOUCH_RST fail \n");

	return rc;
}
#elif defined (CONFIG_MACH_EVT1)
static int ts_setup_gpio(int gpio_num, int gpio_rst)
{
	int rc;
	int i;
		
	struct pm8058_gpio_cfg {
		int 	gpio;
		struct pm8058_gpio cfg;
	};
	
	struct pm8058_gpio_cfg gpio_cfgs[] = {

		{ /* Touch RST */
			22, // Orange EVT1 from 27 to 23
			{
			.direction	= PM_GPIO_DIR_OUT,
			.output_value	= 1,
			.output_buffer	= PM_GPIO_OUT_BUF_CMOS,
			.pull		= PM_GPIO_PULL_DN,
			.out_strength	= PM_GPIO_STRENGTH_HIGH,
			.function	= PM_GPIO_FUNC_NORMAL,
			.vin_sel	= PM_GPIO_VIN_L5,
			.inv_int_pol	= 0,
			}
		},

    		{ /* Touch INT */
        			24,
			{
			.direction	= PM_GPIO_DIR_IN,
			.pull		= PM_GPIO_PULL_DN,
			.function	= PM_GPIO_FUNC_NORMAL,
			.vin_sel	=PM_GPIO_VIN_L5,
			.inv_int_pol	= 0,
			}
		}
	};

		    		
	rc = gpio_request( gpio_num, "auo_touchscreen_irq" );
	rc = gpio_request( gpio_rst, "auo_touchscreen_rst" );

	for (i = 0; i < ARRAY_SIZE(gpio_cfgs); ++i) {
		rc = pm8058_gpio_config(gpio_cfgs[i].gpio,&gpio_cfgs[i].cfg);
		if (rc < 0) {
			TCH_DBG("[Leo], AUO Touch Screen ts_setup_gpio fail \n");
			goto exit_setup_gpio;
		}
	}
		
// Leo Cheng Level Shift
	rc = gpio_request(72, "auo_touchscreen_le_en");
	rc = gpio_direction_output(72, 1);
	return rc;

exit_setup_gpio:
	ts_release_gpio(gpio_num, gpio_rst);

	
	return rc;
}
#endif

/*Qsida,Alan P Xu,2011.07.27, GPIO of touch be changed}*/
static int ts_config_gpio(int gpio_num, int gpio_rst)
{
	int rc=0;

	TCH_DBG("[Leo], AUO Touch Screen ts_config_gpio \n");

	gpio_set_value_for_touch(gpio_rst, 0);
	msleep(1);
	gpio_set_value_for_touch(gpio_rst, 1);
/*Qsida,Alan P Xu,2011.07.27, GPIO of touch be changed{*/
	gpio_tlmm_config(GPIO_CFG(TOUCH_I2C_DATA, 1, GPIO_CFG_OUTPUT, GPIO_CFG_PULL_UP, GPIO_CFG_10MA), GPIO_CFG_ENABLE);// TOUCH_I2C_Data
	gpio_tlmm_config(GPIO_CFG(TOUCH_I2C_CLK, 1, GPIO_CFG_OUTPUT, GPIO_CFG_PULL_UP, GPIO_CFG_10MA), GPIO_CFG_ENABLE);// TOUCH_I2C_Clk
/*Qsida,Alan P Xu,2011.07.27, GPIO of touch be changed}*/
	msleep(60);

/*Qsida,Alan P Xu,2011.07.27, GPIO of touch be changed{*/
#if defined(CONFIG_MACH_EVT2)

#elif defined (CONFIG_MACH_EVT1)
	rc = gpio_tlmm_config(GPIO_CFG(72, 0, GPIO_CFG_OUTPUT, GPIO_CFG_PULL_UP, GPIO_CFG_10MA), GPIO_CFG_ENABLE);// TOUCH_LE_EN
	if( rc )	TCH_DBG("[Leo], AUO Touch Screen TOUCH_LE_EN fail \n");
#endif
/*Qsida,Alan P Xu,2011.07.27, GPIO of touch be changed}*/
	
//exit_config_gpio:
	return rc;
}

static int ts_config_panel(struct ts_t *pTS)
{
#if CONFIG_PANEL
	int result;
	struct i2c_client *client = pTS->client_4_i2c;
	uint8_t value;
/*Qisda, Alan.P.Xu, 2012.02.21, touch invalid{*/
	//uint8_t value2;
	//uint8_t value3;

	//value2 = 0x22;
	//value3 = 0x22;	
	
	//result = ts_write_i2c(client, 0x6F, &value2, 1);	// X Sensitivity
	//result = ts_write_i2c(client, 0x70, &value3, 1);	// Y Sensitivity
/*Qisda, Alan.P.Xu, 2012.02.21, touch invalid}*/
	result = ts_read_i2c(client, pTS->reg_map.int_setting, &value, 1);
	if( result ){
	TCH_DBG("[Leo], AUO Touch Screen unable to read reg %Xh, return %d\n", pTS->reg_map.int_setting, result);
	return result;
	}
	
	if( (value & AUO_INT_SETTING_MASK) == AUO_INT_SETTING_VAL ){
		goto config_sens;
	}

	value &= (~AUO_INT_SETTING_MASK);
	value |= AUO_INT_SETTING_VAL;

	result = ts_write_i2c( client, pTS->reg_map.int_setting, &value, 1 );
	if( result ){
		TCH_DBG("[Leo], AUO Touch Screen unable to write reg %Xh, return %d\n", pTS->reg_map.int_setting, result);
		return result;
	}
config_sens:
	down( &ts_sem );
  
	if( (AUO_INT_SETTING_VAL & AUO_INT_MODE_MASK) == (AUO_TOUCH_IND_ENA & AUO_INT_MODE_MASK) )
		g_touch_ind_mode = 1;
	else
		g_touch_ind_mode = 0;
  
	up( &ts_sem );

#endif
	return 0;
}

static void ts_reset_panel(struct ts_t *pTS)
{
#ifdef USE_PIXCIR
	gpio_set_value_for_touch(pTS->gpio_rst, 0);
	msleep(1);
	gpio_set_value_for_touch(pTS->gpio_rst, 1);
	msleep(60);
#endif
	return;
}

static int ts_detect_pixcir(struct ts_t *pTS)
{
#ifdef USE_PIXCIR
	int result;
	uint8_t value;

	value = 0;
	result = ts_read_i2c(pTS->client_4_i2c, 0x70, &value, 1);
	if(result)
		return result;
	else
		g_pixcir_detected = (!!value)? 1: 0;

//detect freeze mode
	value = 0;
	result = ts_read_i2c(pTS->client_4_i2c, 0x77, &value, 1);
	if(result)
		return result;
	else {
		TCH_DBG("Firmware Version is 0x%X\n", value);
		g_pixcir_freeze = (value > 0x97)? 1: 0;
	     }
#endif
	return 0;
}

static int ts_setup_reg_map(struct ts_t *pTS)
{
	if( NULL == pTS ) {
		return -1;
	}

	TCH_DBG("[Leo], AUO Touch Screen ts_setup_reg_map g_pixcir_detected = %d \n", g_pixcir_detected);
	
	if( 0 == g_pixcir_detected ) {
// Cypress IC
		pTS->reg_map.x1_lsb = 0x40;
		pTS->reg_map.x_sensitivity = 0x65;
		pTS->reg_map.y_sensitivity = 0x64;
		pTS->reg_map.int_setting = 0x66;
		pTS->reg_map.int_width = 0x67;
		pTS->reg_map.power_mode = 0x69;
	} else {
		pTS->reg_map.x1_lsb = AUO_X1_LSB;
		pTS->reg_map.x_sensitivity = AUO_X_SENSITIVITY;
		pTS->reg_map.y_sensitivity = AUO_Y_SENSITIVITY;
		pTS->reg_map.int_setting = AUO_INT_SETTING;
		pTS->reg_map.int_width = AUO_INT_WIDTH;
		pTS->reg_map.power_mode = AUO_POWER_MODE;
	       }

	TCH_DBG("ts_setup_reg_map: map is 0x%X 0x%X 0x%X 0x%X 0x%X 0x%X\n",
	pTS->reg_map.x1_lsb, pTS->reg_map.x_sensitivity, pTS->reg_map.y_sensitivity,
	pTS->reg_map.int_setting, pTS->reg_map.int_width, pTS->reg_map.power_mode);

	TCH_DBG("ts_setup_reg_map<-\n");

	return 0;
}

#ifdef USE_PIXCIR
static int ts_read_eeprom_calib(char *buf_x, char *buf_y)
{
	int rc = 0;
	char addr[4] = { 0x78, 0x01, 0x00, 0x00 };
	struct i2c_client *client = g_ts->client_4_i2c;

	struct i2c_msg msgs[] = {
		[0] = {
			.addr   = client->addr,
			.flags	= 0,
			.buf	= (void *)addr,
			.len	= 4
		},
		[1] = {
			.addr   = client->addr,
			.flags	= I2C_M_RD,
			//.buf	= (void *)buf_x,
			//.len	= 1
		}
	};

	TCH_DBG("ts_read_eeprom_calib->\n");
	addr[2] = AUO_EEPROM_CALIB_X;
	msgs[1].buf = (void *)buf_x;
	msgs[1].len = AUO_EEPROM_CALIB_X_LEN;
	rc = i2c_transfer(client->adapter, msgs, 2);
	if(rc != 2){
		TCH_DBG("ts_read_eeprom_calib: unable to read reg %Xh %Xh %Xh %Xh, return %d\n",
		msgs[0].buf[0], msgs[0].buf[1], msgs[0].buf[2], msgs[0].buf[3], rc);
		return rc;
	}

	addr[2] = AUO_EEPROM_CALIB_Y;
	msgs[1].buf = (void *)buf_y;
	msgs[1].len = AUO_EEPROM_CALIB_Y_LEN;
	rc = i2c_transfer(client->adapter, msgs, 2);
	if(rc != 2){
		TCH_DBG("ts_read_eeprom_calib: unable to read reg %Xh %Xh %Xh %Xh, return %d\n",
		msgs[0].buf[0], msgs[0].buf[1], msgs[0].buf[2], msgs[0].buf[3], rc);
		return rc;
	}

	TCH_DBG("ts_read_eeprom_calib<-\n");
	return 0;
}


static int pm8058_s3_vote_init(int init)
{
	static struct  regulator *vreg_auo_s3;	//1.8V
	int     rc = 0;

	
	if (init) {
		vreg_auo_s3 = regulator_get(NULL, "8058_s3");
		rc = regulator_set_voltage(vreg_auo_s3, 1800000, 1800000);
		regulator_enable(vreg_auo_s3);
	} else {
		if (regulator_disable(vreg_auo_s3))
			TCH_DBG("[Leo], AUO Touch Screen unable to disable regulator S3 \n");
		regulator_put(vreg_auo_s3);
	       }

	return rc;
}

static int ts_power_switch(int on,struct ts_t *ts)
{ 
	int  rc = 0;
     
	rc = pm8058_s3_vote_init(on);
	rc = ts->pm8901_l1b_init(on);
     
	if (rc) {
		TCH_DBG("[Leo], AUO Touch Screen set_voltage fail \n");
	}
     
	return rc;
}


static int ts_get_eeprom_calib(struct auo_eeprom_calib_t *pCalib)
{
	int rc = 0;
	char *buf_x, *buf_y;

	TCH_DBG("ts_get_eeprom_calib->\n");

	if( NULL == pCalib->raw_buf_x || NULL == pCalib->raw_buf_y ){
		TCH_DBG("ts_get_eeprom_calib: invalid args\n");
		return -1;
	}
	if( pCalib->raw_len_x < 0 || pCalib->raw_len_x > AUO_EEPROM_CALIB_X_LEN ||
	   pCalib->raw_len_y < 0 || pCalib->raw_len_y > AUO_EEPROM_CALIB_Y_LEN ){
		TCH_DBG("ts_get_eeprom_calib: invalid length %d %d\n",
		pCalib->raw_len_x, pCalib->raw_len_y);
		return -1;
	}

	buf_x = buf_y = NULL;
	buf_x = kzalloc( pCalib->raw_len_x, GFP_KERNEL );
	buf_y = kzalloc( pCalib->raw_len_y, GFP_KERNEL );
	if(NULL == buf_x || NULL == buf_y){
		TCH_DBG("ts_get_eeprom_calib: alloc failed\n");
		rc = -1;
		goto exit_get_calib;
	}

	rc = ts_read_eeprom_calib(buf_x, buf_y);

	if(rc)
		goto exit_get_calib;

	if (copy_to_user((void *)pCalib->raw_buf_x,
	   (void *)buf_x, pCalib->raw_len_x)){
		TCH_DBG("ts_get_eeprom_calib: copy to user failed!\n");
		rc = -EFAULT;
	}

	if (copy_to_user((void *)pCalib->raw_buf_y,
	   (void *)buf_y, pCalib->raw_len_y)){
		TCH_DBG("ts_get_eeprom_calib: copy to user failed!\n");
		rc = -EFAULT;
	}

exit_get_calib:
	if( buf_x )
		kfree(buf_x);
	if( buf_y )
		kfree(buf_y);

	TCH_DBG("ts_get_eeprom_calib<-\n");

	return rc;
}

static int ts_get_raw_data(struct auo_raw_data_t *pRawData)
{
	int rc;
	struct i2c_client *client = g_ts->client_4_i2c;
	uint8_t *rawBuf = NULL;
	int     *buf = NULL;
	int     i;

//TCH_DBG("ts_get_raw_data()+\n");

	if(NULL == pRawData->raw_buf_x || NULL == pRawData->raw_buf_y){
		TCH_DBG("ts_get_raw_data: invalid arg\n");
		rc = -1;
		goto exit_raw_data;
	}

	if(pRawData->raw_len_x < AUO_RAW_DATA_X_LEN ||
	   pRawData->raw_len_y < AUO_RAW_DATA_Y_LEN){
		TCH_DBG("ts_get_raw_data: invalid buffer length %d %d\n",
		pRawData->raw_len_x, pRawData->raw_len_y);
		rc = -1;
		goto exit_raw_data;
	}

	rawBuf = kzalloc((AUO_RAW_DATA_X_LEN+AUO_RAW_DATA_Y_LEN)*2, GFP_KERNEL);
	buf = kzalloc( (AUO_RAW_DATA_X_LEN+AUO_RAW_DATA_Y_LEN)*sizeof(int), GFP_KERNEL );
	if(NULL == rawBuf || NULL == buf){
		TCH_DBG("ts_get_raw_data: alloc failed\n");
		rc = -1;
		goto exit_raw_data;
	}

	if(pRawData->mode != AUO_KEEP_READ){
		uint8_t enableAddr, enableData;

		enableAddr = AUO_TOUCH_STRENGTH_ENA;
		if(pRawData->mode == AUO_ENABLE_READ)
			enableData = 1;
		else
			enableData = 0;

		rc = ts_write_i2c( client, enableAddr, &enableData, 1 );
		if(rc)
			goto exit_raw_data;
	}

	rc = ts_read_i2c(client, AUO_RAW_DATA_X, rawBuf,
		         (AUO_RAW_DATA_X_LEN+AUO_RAW_DATA_Y_LEN)*2);
	if(rc)
		goto exit_raw_data;

	for(i=0; i<(AUO_RAW_DATA_X_LEN+AUO_RAW_DATA_Y_LEN); i++){
		//buf[i] = (rawBuf[2*i+1] << 8) | rawBuf[2*i];
		//*(buf+i) = (*(rawBuf+2*i+1) << 8) | *(rawBuf+2*i);
		*(buf+i) = *(rawBuf+2*i+1);
		*(buf+i) = (*(buf+i)) << 8;
		*(buf+i) |= *(rawBuf+2*i);
		//TCH_DBG("ts_get_raw_data: %X = %X %X\n",
		//        *(buf+i), *(rawBuf+2*i+1), *(rawBuf+2*i));
	}

	if (copy_to_user((void *)pRawData->raw_buf_x,
    	                 (void *)buf, AUO_RAW_DATA_X_LEN * sizeof(int))){
		TCH_DBG("ts_get_raw_data: copy to user failed!\n");
		rc = -EFAULT;
	}

	if (copy_to_user((void *)pRawData->raw_buf_y,
		         (void *)(buf+AUO_RAW_DATA_X_LEN),
		         AUO_RAW_DATA_Y_LEN * sizeof(int))){
		TCH_DBG("ts_get_raw_data: copy to user failed!\n");
		rc = -EFAULT;
	}

exit_raw_data:
	if( rawBuf )
		kfree(rawBuf);
	if( buf )
		kfree(buf);

	//TCH_DBG("ts_get_raw_data()-, rc=%d\n", rc);

	return rc;
}
#endif //#ifdef USE_PIXCIR

static int ts_open(struct input_dev *dev)
{
	int rc = 0;
	struct ts_t *pTS = input_get_drvdata(dev);

	TCH_DBG("[Leo], AUO Touch Screen ts_open \n");

// request irq
	if(0 == pTS->open_count){
		TCH_DBG("[Leo], AUO Touch Screen pTS->open_count \n");

	} else {
		TCH_DBG("ts_open: opened %d times previously\n", pTS->open_count);
	       }
	if(!rc)
		pTS->open_count++;

	return rc;
}

static void ts_close(struct input_dev *dev)
{
	struct ts_t *pTS = input_get_drvdata(dev);

	TCH_DBG("[Leo], AUO Touch Screen ts_close \n");
	
	if(pTS->open_count > 0){
		pTS->open_count--;
		TCH_DBG("ts_close: still opened %d times\n", pTS->open_count);
#if 0//Leo Guo disable for request and free are matched 
		if(0 == pTS->open_count) {
			TCH_DBG("ts_close: irq %d will be freed\n", pTS->irq);
			free_irq(pTS->irq, pTS);
		}
#endif
	}

	return;
}

static int ts_event(struct input_dev *dev, unsigned int type, unsigned int code, int value)
{
	TCH_DBG("[Leo], AUO Touch Screen ts_event \n");

	return 0;
}

static int ts_register_input(struct input_dev **input,
                              struct auo_touchscreen_platform_data *pdata,
                              struct i2c_client *client)
{
	int rc;
	struct input_dev *input_dev;
	unsigned int x_max, y_max;

	input_dev = input_allocate_device();
	if (!input_dev) {
		rc = -ENOMEM;
	return rc;
	}

	TCH_DBG("[Leo], AUO Touch Screen ts_register_input \n");
	
	input_dev->name = AUO_TS_DRIVER_NAME;
	input_dev->phys = "auo_touchscreen/input0";
	input_dev->id.bustype = BUS_I2C;
	input_dev->id.vendor = 0x0001;
	input_dev->id.product = 0x0002;
	input_dev->id.version = 0x0100;
	input_dev->dev.parent = &client->dev;

	input_dev->open = ts_open;
	input_dev->close = ts_close;
	input_dev->event = ts_event;

	input_dev->evbit[0] = BIT_MASK(EV_KEY) | BIT_MASK(EV_ABS);
	input_dev->absbit[0] = BIT(ABS_X) | BIT(ABS_Y);
#if 0
 // Menghan Cheng, 20090204, Report the second point to input class
	input_dev->absbit[BIT_WORD(ABS_HAT0X)] |= BIT(ABS_HAT0X) | BIT(ABS_HAT0Y);
	input_dev->keybit[BIT_WORD(BTN_BASE2)] |= BIT_MASK(BTN_BASE2);
//~Menghan
#endif
	//input_dev->absbit[BIT_WORD(ABS_MISC)] |= BIT_MASK(ABS_MISC);
	input_dev->keybit[BIT_WORD(BTN_TOUCH)] |= BIT_MASK(BTN_TOUCH);

	//pdata = NULL;
	if (pdata) {
		x_max = pdata->x_max ? : AUO_X_MAX;
		y_max = pdata->y_max ? : AUO_Y_MAX;
	} else {
		x_max = AUO_X_MAX;
		y_max = AUO_Y_MAX;
	       }

	input_set_abs_params(input_dev, ABS_X, 0, x_max, 0, 0);
	input_set_abs_params(input_dev, ABS_Y, 0, y_max, 0, 0);
//for multi touch protocol
	input_set_abs_params(input_dev, ABS_MT_POSITION_X, 0, AUO_X_MAX, 0, 0);
	input_set_abs_params(input_dev, ABS_MT_POSITION_Y, 0, AUO_Y_MAX, 0, 0);
	input_set_abs_params(input_dev, ABS_MT_TOUCH_MAJOR,  0, AUO_X_MAX, 0, 0);
	input_set_abs_params(input_dev, ABS_MT_WIDTH_MAJOR,  0, AUO_X_MAX, 0, 0);

	rc = input_register_device(input_dev);

	if (rc){
		TCH_DBG("[Leo], AUO Touch Screen input register fail \n");
		input_free_device( input_dev );
	}else{
		TCH_DBG("[Leo], AUO Touch Screen input register ok \n");
		*input = input_dev;
	     }

	return rc;
}

/*Qisda,Alan P Xu,2011.08.09,add touch ftd{*/
int ts_Calibration(void)
{
	struct i2c_client *client = g_ts->client_4_i2c;
	int   ret = 0;
	short x;
	char data[60];
	char  calib_data;
	char calib_data13;
	int i;
	int PF = 0;
	int Count = 0;
   
	TCH_DBG("Touch Panel Calibration Starting...\n");
Run:
	if(Count >= 5){
		return PF;
	}      
      
	calib_data = 0x03;
	ret = ts_write_i2c(client, 0x78, &calib_data, 1);
	if(ret == 0){
		TCH_DBG("write Calibration command Success\n");
	}else{
		TCH_DBG("write Calibration command Fail\n");
		return -1;
	     } 
            
	mdelay(1000);
	calib_data13 = 1; 
	ret = ts_write_i2c(client, 13, &calib_data13, 1);
	if(0 != ret)
		TCH_DBG("ts_write_i2c(client, 13, &calib_data13, 1) failed\n");
      
	PF = 0;
  
	mdelay(100);
	ret = ts_read_i2c(client, 0x2B, &data[0],58);
	if(ret != 0)
	TCH_DBG("ts_Calibration read i2c 58 data failed!\n");
     
	for(i=0;i<18;i++){
		x = (((short)data[(i*2)+1]) << 8) | (short)data[(i)*2];  
		if(x<-5 || x>5){   
			calib_data13 = 0;
			ret = ts_write_i2c(client, 13, &calib_data13, 1);  
			TCH_DBG("Touch Fail, %d %d Count = %d\n", i,x,Count); 
			Count++;
			PF = -1;
			goto Run;
              
		}       
		TCH_DBG("Touch  %d %d\r\n", i,x);        
	}  
     
	for(i=0;i<11;i++){
		x = ((short)data[(i*2)+37] << 8) | (short)data[(i*2)+36];
		if(x<-5 || x>5){
			calib_data13 = 0;
			ret = ts_write_i2c(client, 13, &calib_data13, 1);   
			TCH_DBG("Touch Fail, %d %d Count = %d\n", i,x,Count); 
			Count++;
			PF = -1;
			goto Run;
              
		}     
		TCH_DBG("Touch  %d %d\r\n", i,x);     
	} 
	calib_data13 = 0;
	ret = ts_write_i2c(client, 13, &calib_data13, 1);    
	return PF;
}

int ts_Firmware(int EnterDownload, struct i2c_client *client, unsigned int size,\
                  unsigned char *touch_hex_data, unsigned int Version)
{   
	unsigned int i,j,k = 0;
	char buf_r[4];
	char buf_w[143];
	char dat;
	int ret = 0;
	char data[8];
    
	char TouchVer;

	TCH_DBG("client->address=%d\n",client->addr);

  ret = ts_read_i2c(client, 0x77, &TouchVer, 1);
	if(0 != ret){
			EnterDownload = 0;
	}
	if(ret == 0){
	    EnterDownload = 1;
			TCH_DBG("i2c read sucess touch version is 0x%x\n",TouchVer);
	}
	if(EnterDownload){
		if(TouchVer == Version){
			TCH_DBG("version is new!\n");
			return 0;
		}
		if(TouchVer>=0x90 && TouchVer<=0x9F){
			msleep(10);
#if 1
			for(i=0; i<8; i++){ 
				ret = ts_read_i2c(client, i, &data[i],1);
				if (ret ==0)
					TCH_DBG("data[%d]=%d\n",i,data[i]);
				else {
					TCH_DBG("read data[%d]fail\n",i);
					return -4;
				     }
			}
#endif
			dat = 0x05;
			ret = ts_write_i2c(client, 0x78, &dat, 1);
			if(0 != ret){
				TCH_DBG("i2c write failed:\n");
				return -5;
			} else
				TCH_DBG("i2c write sucess:\n");
		} else {
			msleep(10);
		       }
	}
    	
    	
	client->addr = 0x5D;
	ret = boot_read_mem( client,0xFF,5, buf_r);
	if(0 != ret){
		TCH_DBG("i2c read failed:\n");
		return -4;
	} else
		TCH_DBG("client->addr read sucess buf data is 0x%x,0x%x,0x%x,0x%x\n",buf_r[0],buf_r[1],buf_r[2],buf_r[3]);
          
	for(i=0; i<size/143; i++){
		TCH_DBG("write number is %d\n",i);
		for(j=0; j<143; j++){
			buf_w[j] = *touch_hex_data;
			touch_hex_data ++;
		}
		ret = boot_ts_write_i2c(client, 0,buf_w, 143);
		if(0 != ret){
			TCH_DBG("i2c write failed:\n");
			return -5;
		}
		k = k+143;
		for(j=0; j<100; j++){
			mdelay(50);//add 50
			if(gpio_get_value_for_touch(g_ts->gpio_num) == 0)
				break;
			if(buf_w[0]==0x01 && buf_r[1]==0xA5) 
				break;
		}  
	    
		if(buf_w[0]==0x01 && buf_r[1]==0xA5){  
			break;
		}
	    
		ret = boot_read_mem( client,0xFF,5, buf_r);
		if(0 != ret){
			TCH_DBG("i2c read failed:\n");
			return -4;
		}
        
		msleep(10);
	    
		if((buf_w[0]==0x02 && buf_r[0]& 0x80) == true){
			TCH_DBG("CRC Error\n");
			return -1;
		}
		if((buf_w[0]==0x02 && buf_r[1]==0xFF) == true){
			TCH_DBG("EEPROM Erased\n!");
			return -2;
		}
	    
	}
	
	mdelay(500);
	ts_reset_panel(g_ts);
	return 0;
    
}
#if 0
int StartTouch_Firmware(void)
{
	struct i2c_client *client = g_ts->client_4_i2c;
	unsigned char *data_firmware,*data;
	int size;
	int i;
	int ret;
	size = sizeof(DownFirmware);
  
	data_firmware = (unsigned char *)kzalloc(size, GFP_KERNEL);
	if(!data_firmware){
		ret = -ENOMEM;
		TCH_DBG("download firmware alloc memory failed\n");
		return ret;
	}
	memset(data_firmware, 0, size);
	data = data_firmware;
  
	i = 0;
	while(i<size){
		*data = DownFirmware[i];
		data++;
		i++;
	}
  
	ret = ts_Firmware(0, client, size, data_firmware, 0);
	if(ret == 0){
		TCH_DBG("download firmware sucess\n");
		kfree(data_firmware);
		ret = ts_Calibration();
		if(0==ret)
			TCH_DBG("calibration touch\n");
		else {
			TCH_DBG("can not calibration touch\n");
			return -1;
		     }
	} else {
		TCH_DBG("download firmware failed\n");
		return -1;
	       }
	return 0;
  }
/*Qisda,Alan P Xu,2011.08.09,add touch ftd}*/
#endif
#if USE_AUO_5INCH
static long ts_misc_ioctl (struct file *fp,
                            unsigned int cmd,
                            unsigned long arg)
{
	struct i2c_client *client = g_ts->client_4_i2c;
	struct auo_sensitivity_t sensitivity;
	struct auo_interrupt_t   interrupt;
	struct auo_power_mode_t  power_mode;
#ifdef USE_PIXCIR
	struct auo_touch_area_t touch_area;
	struct auo_touch_strength_t touch_strength;
#endif
	uint8_t value = 0;
	uint8_t addr = 0;
	uint8_t *pData = NULL;
	uint    length = 0;
	int     rc = 0;
	uint8_t debug_info[2];
	TCH_DBG("ts_misc_ioctl()+, cmd number=%d\n", _IOC_NR(cmd));

	switch(cmd){
		case  AUO_TOUCH_IOCTL_FOR_DEBUG:
			rc = ts_read_i2c(client, 0x70, &value, 1);	
			if(rc)
				debug_info[1]=0;
			else
				debug_info[1]=1;
			rc = ts_read_i2c(client, 0x77, &value, 1);
			if(rc)
				debug_info[1]=0;
			else
				debug_info[1]=1;
			debug_info[0]=gpio_get_value_for_touch(g_ts->gpio_num);
			rc = copy_to_user((void *)arg,&debug_info[0],2);
		break;
	
		case AUO_TOUCH_IOCTL_GET_INFO:{
			uint8_t value_temp[3];
			rc = ts_read_i2c(client, 0x77, &value, 1);
			value_temp[2]=value;
			rc = copy_to_user( (void *)arg,&value_temp[0],3 );
			if (rc){
				TCH_DBG( "ERROR_LEVEL:" "copy GET_VERSION to user failed!\n" );
				rc = -EFAULT;
				return rc;
			}
			TCH_DBG("INFO_LEVEL:" "value[0]: %x\n",value_temp[2]);
		}
		break;
		case AUO_TOUCH_IOCTL_GET_X_SENSITIVITY:
			addr = g_ts->reg_map.x_sensitivity;
			pData = (void *)&sensitivity;
			length = sizeof(sensitivity);
			rc = ts_read_i2c(client, addr, &value, 1);
			sensitivity.value = value;
		break;
		case AUO_TOUCH_IOCTL_SET_X_SENSITIVITY:
			addr = g_ts->reg_map.x_sensitivity;
			pData = (void *)&sensitivity;
			length = sizeof(sensitivity);
			if(copy_from_user((void *)pData,
			   (void *)arg, length)){
				TCH_DBG("ts_misc_ioctl: copy from user failed!\n");
				rc = -EFAULT;
			} else {
				value = sensitivity.value;
			       }
		break;
		case AUO_TOUCH_IOCTL_SET_Y_SENSITIVITY:
			addr = g_ts->reg_map.y_sensitivity;
			pData = (void *)&sensitivity;
			length = sizeof(sensitivity);
			if( copy_from_user((void *)pData,
			   (void *)arg, length)){
				TCH_DBG("ts_misc_ioctl: copy from user failed!\n");
				rc = -EFAULT;
			} else {
				value = sensitivity.value;
			       }
		break;
		case AUO_TOUCH_IOCTL_GET_Y_SENSITIVITY:
			addr = g_ts->reg_map.y_sensitivity;
			pData = (void *)&sensitivity;
			length = sizeof(sensitivity);
			rc = ts_read_i2c(client, addr, &value, 1);
			sensitivity.value = value;
		break;
		case AUO_TOUCH_IOCTL_SET_INT_SETTING:
			addr = g_ts->reg_map.int_setting;
			pData = (void *)&interrupt;
			length = sizeof(interrupt);
			if(copy_from_user((void *)pData,
			   (void *)arg, length)){
				TCH_DBG("ts_misc_ioctl: copy from user failed!\n");
				rc = -EFAULT;
			} else {
				value = interrupt.int_setting;
			       }
		break;
		case AUO_TOUCH_IOCTL_GET_INT_SETTING:
			addr = g_ts->reg_map.int_setting;
			pData = (void *)&interrupt;
			length = sizeof(interrupt);
			rc = ts_read_i2c(client, addr, &value, 1);
			interrupt.int_setting = value;
		break;
		case AUO_TOUCH_IOCTL_SET_INT_WIDTH:
			addr = g_ts->reg_map.int_width;
			pData = (void *)&interrupt;
			length = sizeof(interrupt);
			if(copy_from_user((void *)pData,
			   (void *)arg, length)){
				TCH_DBG("ts_misc_ioctl: copy from user failed!\n");
				rc = -EFAULT;
			} else {
				value = interrupt.int_width;
			       }
		break;
		case AUO_TOUCH_IOCTL_GET_INT_WIDTH:
			addr = g_ts->reg_map.int_width;
			pData = (void *)&interrupt;
			length = sizeof(interrupt);
			rc = ts_read_i2c(client, addr, &value, 1);
			interrupt.int_width = value;
		break;
		case AUO_TOUCH_IOCTL_SET_POWER_MODE:
			addr = g_ts->reg_map.power_mode;
			pData = (void *)&power_mode;
			length = sizeof(power_mode);
			if(copy_from_user((void *)pData,
			   (void *)arg, length)){
				TCH_DBG("ts_misc_ioctl: copy from user failed!\n");
				rc = -EFAULT;
				break;
			}
			if(power_mode.mode <= AUO_UNKNOW_POWER_MODE ||
			   power_mode.mode >= AUO_MAX_POWER_MODE){
				TCH_DBG("ts_misc_ioctl: unknown powr mode %d!\n", power_mode.mode);
				rc = -EFAULT;
				break;
			}
			rc = ts_read_i2c( client, addr, &value, 1 );
			if(!rc){
				power_mode.mode -= AUO_ACTIVE_POWER_MODE;
				power_mode.mode &= AUO_POWER_MODE_MASK;
				value &= ~AUO_POWER_MODE_MASK;
				value |= power_mode.mode;
			}
		break;
		case AUO_TOUCH_IOCTL_GET_POWER_MODE:
			addr = g_ts->reg_map.power_mode;
			pData = (void *)&power_mode;
			length = sizeof(power_mode);
			rc = ts_read_i2c( client, addr, &value, 1 );
			value &= AUO_POWER_MODE_MASK;
			value += AUO_ACTIVE_POWER_MODE;
			power_mode.mode = (enum auo_touch_power_mode)value;
		break;
/*Qisda, Alan P Xu, 2011.09.18, touch debug info{*/
		case AUO_TOUCH_RESET_PANEL:{
/*Qisda, Alan.P.Xu, 2012.02.21, touch invalid{*/
			rc = 0;
	    disable_irq(g_ts->irq);
			ts_reset_panel(g_ts);
			msleep(200);
			enable_irq(g_ts->irq );
			return rc;
/*Qisda, Alan.P.Xu, 2012.02.21, touch invalid}*/
		}
		break;
		case AUO_GET_POWER_MODE:{
			addr = g_ts->reg_map.power_mode;
			rc = ts_read_i2c( client, addr, &value, 1 );
			if(copy_to_user((void *)arg, (void *)&value, sizeof(value))){
				TCH_DBG("ts_misc_ioctl: copy to user failed!\n");
				rc = -EFAULT;
			} 
			return rc;
		}
		break;
		case AUO_GET_IRQ_VALUE:{
			struct auo_Irqvalue irq_state;
     
			addr = g_ts->reg_map.int_setting;
			rc = ts_read_i2c(client, addr, &value, 1);
			irq_state.int_setting = value;
     
			addr = g_ts->reg_map.int_width;
			rc = ts_read_i2c(client, addr, &value, 1);
			irq_state.int_width = value;
     
			irq_state.irq_value = gpio_get_value_for_touch(g_ts->gpio_num);
     
			if(copy_to_user((void *)arg, (void *)&irq_state, sizeof(irq_state))){
				TCH_DBG("ts_misc_ioctl: copy to user failed!\n");
				rc = -EFAULT;
			} 
			return rc;
		}
		break;
		case AUO_TOUCH_GET_RAW_DATA:{
			struct Raw_Inter_data raw_inter;
			char raw_data13;
			char data[58];
			
			if(copy_from_user( (void *)(&raw_inter),(void *)arg, sizeof(raw_inter))){
				TCH_DBG("ts_misc_ioctl: copy to user failed!\n");
				rc = -EFAULT;
			} 
			
			rc = ts_read_i2c(client, 0x2B, &data[0],58);
			if(rc != 0)
				TCH_DBG("read i2c 58 raw data failed!\n");
    
			if(copy_to_user((void *)raw_inter.Rawdata, (void *)data, 58)){
				TCH_DBG("ts_misc_ioctl: copy to user failed!\n");
				rc = -EFAULT;
			} 
			raw_data13 = 1; 
			rc = ts_write_i2c(client, 13, &raw_data13, 1);
			if(0 != rc)
				TCH_DBG("ts_write_i2c(client, 13, &calib_data13, 1) failed\n");
  
			mdelay(100);
			rc = ts_read_i2c(client, 0x2B, &data[0],58);
			if(rc != 0)
				TCH_DBG("read i2c 58 internal data failed!\n");
    
			if(copy_to_user((void *)raw_inter.Interdata, (void *)data, 58)){
				TCH_DBG("ts_misc_ioctl: copy to user failed!\n");
				rc = -EFAULT;
			} 
    
			raw_data13 = 0;
			rc = ts_write_i2c(client, 13, &raw_data13, 1); 
			return rc;
		}
		break;
		case AUO_TOUCH_GET_POINT:{
			struct auo_point_t  point[AUO_REPORT_POINTS];
			uint8_t             rawCoord[8];
			uint8_t             i;
    
			rc = ts_read_i2c( client, g_ts->reg_map.x1_lsb, rawCoord, 8);
			if(!rc){
				for( i=0; i<ARRAY_SIZE(point); i++ ){
#if USE_AUO_5INCH
					if( 1 == g_pixcir_detected ){
#if defined(TOUCH_FB_PORTRAIT)
						point[i].x = rawCoord[4*i+1]<<8 | rawCoord[4*i];
						point[i].y = rawCoord[4*i+3]<<8 | rawCoord[4*i+2];			
#else
						point[i].y = rawCoord[4*i+1]<<8 | rawCoord[4*i];
						point[i].x = rawCoord[4*i+3]<<8 | rawCoord[4*i+2];

						if( 0 != point[i].x ){
							point[i].x = AUO_X_MAX + 1 - point[i].x;
						}
#endif
					}else {
#if defined(TOUCH_FB_PORTRAIT)
						point[i].x = (rawCoord[2*i+4]<<8) | rawCoord[2*i];
						point[i].y = (rawCoord[2*i+5]<<8) | rawCoord[2*i+1];
						if( point[i].y != 0 )
							point[i].y = AUO_Y_MAX + 1 - point[i].y;
#else
						point[i].y = (rawCoord[2*i+4]<<8) | rawCoord[2*i];
						point[i].x = (rawCoord[2*i+5]<<8) | rawCoord[2*i+1];
#endif
					      }
#else
					point[i].x = (rawCoord[2*i+4]<<8) | rawCoord[2*i];
					point[i].y = (rawCoord[2*i+5]<<8) | rawCoord[2*i+1];
#endif
					if( point[i].x > AUO_X_MAX || point[i].y > AUO_Y_MAX ){
						point[i].x = point[i].y = 0;
					}

				} 
      
				if(copy_to_user((void *)arg, (void *)point, sizeof(point))){
					TCH_DBG("ts_misc_ioctl: copy to user failed!\n");
					rc = -EFAULT;
				} 
			}
			return rc;
		}   
		break;
/*Qisda, Alan P Xu, 2011.09.18, touch debug info}*/

/*Qisda, Alan P Xu, 2011.09.04, add performance test{*/
		case AUO_TOUCH_SET_POWER_MODE:{
			struct auo_power_switch_t power;
			addr = g_ts->reg_map.power_mode;
			if(copy_from_user((void *)&power,
			   (void *)arg, sizeof(power))){
				TCH_DBG("ts_misc_ioctl: copy from user failed!\n");
				rc = -EFAULT;
				break;
			}
			if(1 == g_touch_suspended)
			rc = auo_ts_resume_ic(g_ts, client); //resume touch panel
  
			rc = ts_read_i2c( client, addr, &value, 1 );
			if(!rc){
				value = value | power.on;
				rc = ts_write_i2c( client, addr, &value, 1 );
			}
    
			 return rc;
		}
		break;
		case AUO_TOUCH_POWER_SWITCH:{
			struct auo_power_switch_t power;
			if(copy_from_user((void *)&power,
			   (void *)arg, sizeof(power))){
				rc = -EFAULT;
				TCH_DBG("ts_misc_ioctl: copy from user failed!\n");
				goto exit_ioctl;
			}

			if(power.on)
				rc = auo_ts_resume_ic(g_ts, client);
			else
				rc = auo_ts_suspend_ic(g_ts, client);
   
			return rc;
			}
		break;
/*Qisda, Alan P Xu, 2011.09.04, add performance test}*/

/*Qisda, Alan P Xu, 2011.08.04, add touch ftd{*/
		case AUO_TOUCH_CALIBRATION:{
			rc = ts_Calibration();
			if(rc < 0){
				TCH_DBG("can not calibration touch\n");
				goto exit_ioctl;
			}
			 return 0;
		}
		break;
		case AUO_TOUCH_DOWNLOAD_FIRWARE:{
			int EnterDownload = 1;
			struct auo_download_firware_info download_firware_info;
			unsigned char *download_firware_user_addr;
  
			if(copy_from_user((void *)(&download_firware_info), (void *)arg, sizeof(download_firware_info))){ 
				TCH_DBG("ts_misc_ioctl: copy size from user failed!\n");
				rc = -EFAULT;
				goto exit_ioctl;
			}
  
			download_firware_user_addr = download_firware_info.touch_hex_data;
			download_firware_info.touch_hex_data = (unsigned char *)kmalloc(download_firware_info.size_kernel, GFP_KERNEL);
			if(copy_from_user((void *)download_firware_info.touch_hex_data,\
			   (void *)download_firware_user_addr,\
			   download_firware_info.size_kernel)){     
				TCH_DBG("ts_misc_ioctl: copy size from user failed!\n");
				rc = -EFAULT;
				goto exit_ioctl;
			}
			disable_irq(g_ts->irq);
			rc = ts_Firmware(EnterDownload, client, download_firware_info.size_kernel, \
				         download_firware_info.touch_hex_data, download_firware_info.Version);
			if (rc < 0) {
				TCH_DBG("touch download firware fail\n");
				kfree(download_firware_info.touch_hex_data);
				client->addr = 0x5C;
				enable_irq(g_ts->irq);
				goto exit_ioctl;
			}
     
			kfree(download_firware_info.touch_hex_data);
			client->addr = 0x5C;
  
			rc = ts_Calibration();
			if (rc < 0){
				TCH_DBG("touch calibration fail\n");
				enable_irq(g_ts->irq);
				goto exit_ioctl;
			}
			mdelay(50);
			ts_config_panel(g_ts);
			enable_irq(g_ts->irq);
/*Qisda, Alan P Xu, 2011.12.27, test performance after suspending{*/
			rc = auo_ts_suspend_ic(g_ts, client);
			if(rc)
			{
        TCH_DBG("touch suspend fail\n");
				goto exit_ioctl;
      }
      mdelay(1000);
			rc = auo_ts_resume_ic(g_ts, client);
			if(rc)
			{
        TCH_DBG("touch resume fail\n");
				goto exit_ioctl;
      }
/*Qisda, Alan P Xu, 2011.12.27, test performance after suspending}*/
			return 0;
		}
		break;
/*Qisda, Alan P Xu, 2011.08.04, add touch ftd}*/
#ifdef USE_PIXCIR
		case AUO_TOUCH_IOCTL_START_CALIB:{
			addr = 0x78;
			value = 0x03;
			break;
		}
		case AUO_TOUCH_IOCTL_GET_EEPROM_CALIB:{
			struct auo_eeprom_calib_t eeprom_calib;
			if(copy_from_user((void *)&eeprom_calib,
			   (void *)arg, sizeof(eeprom_calib))){
				rc = -EFAULT;
				TCH_DBG("ts_misc_ioctl: copy from user failed!\n");
				break;
			}
			rc = ts_get_eeprom_calib(&eeprom_calib);
			break;
		}
		case AUO_TOUCH_IOCTL_POWER_SWITCH:{
			struct auo_power_switch_t power;
			if(copy_from_user((void *)&power,
			   (void *)arg, sizeof(power))){
				rc = -EFAULT;
				TCH_DBG("ts_misc_ioctl: copy from user failed!\n");
				goto exit_ioctl;
			}

			if(power.on)
				rc = auo_ts_resume_ic(g_ts, client);
			else
				rc = auo_ts_suspend_ic(g_ts, client);

		break;
		}
		case AUO_TOUCH_IOCTL_GET_RAW_DATA:{
			struct auo_raw_data_t raw_data;
			if(copy_from_user((void *)&raw_data,
			   (void *)arg, sizeof(raw_data))){
				rc = -EFAULT;
				TCH_DBG("ts_misc_ioctl: copy from user failed!\n");
				goto exit_ioctl;
			}
			rc = ts_get_raw_data( &raw_data );
		break;
		}
		case AUO_TOUCH_IOCTL_RESET_PANEL:{
			ts_reset_panel(g_ts);
			rc = ts_config_panel(g_ts);
		break;
		}
		case AUO_TOUCH_IOCTL_GET_TOUCH_AREA:{
			uint8_t area[2] = {0};

			addr = AUO_TOUCH_AREA_X;
			pData = (void *)&touch_area;
			length = sizeof(touch_area);
			rc = ts_read_i2c(client, addr, &area[0], 2);
			touch_area.area_x = area[0];
			touch_area.area_y = area[1];
		break;
		}
		case AUO_TOUCH_IOCTL_GET_TOUCH_STRENGTH:{
			uint8_t enableAddr, enableData;
			uint8_t strength[4] = {0};

			if(copy_from_user((void *)&touch_strength,
			   (void *)arg, sizeof(touch_strength))){
				rc = -EFAULT;
				TCH_DBG("ts_misc_ioctl: copy from user failed!\n");
				goto exit_ioctl;
			}

			if(touch_strength.mode == AUO_ENABLE_READ){
				enableAddr = AUO_TOUCH_STRENGTH_ENA;
				enableData = 1;
				rc = ts_write_i2c(client, enableAddr, &enableData, 1);
				if(rc)
					break;
			}

			addr = AUO_TOUCH_STRENGTH_X;
			pData = (void *)&touch_strength;
			length = sizeof(touch_strength);
			rc = ts_read_i2c(client, addr, strength, 4);
			if(rc)
				break;

			touch_strength.strength_x = touch_strength.strength_y = 0;
//strength enginner mode is landscape
			touch_strength.strength_y = (strength[1] << 8) | strength[0];
			touch_strength.strength_x = (strength[3] << 8) | strength[2];
			if(strength[1] & 0x80)
				touch_strength.strength_y |= (~0xFFFF);
			if(strength[3] & 0x80)
				touch_strength.strength_x |= (~0xFFFF);

			TCH_DBG("ts_misc_ioctl: strength_x=%d strength_y=%d\n",
			touch_strength.strength_x, touch_strength.strength_y);

			if(touch_strength.mode == AUO_DISABLE_READ){
				enableAddr = AUO_TOUCH_STRENGTH_ENA;
				enableData = 0;
				rc = ts_write_i2c(client, enableAddr, &enableData, 1);
			}
		break;
		}
#endif //USE_PIXCEL
		default:
			TCH_DBG("ts_misc_ioctl: unsupported ioctl cmd 0x%X\n", cmd);
			rc = -EFAULT;
			break;
	}

	//if(rc < 0)
	if(rc)
		goto exit_ioctl;

#ifdef USE_PIXCIR
	if(AUO_TOUCH_IOCTL_POWER_SWITCH == cmd ||
	   AUO_TOUCH_IOCTL_GET_EEPROM_CALIB == cmd ||
	   AUO_TOUCH_IOCTL_GET_RAW_DATA == cmd ||
	   AUO_TOUCH_IOCTL_RESET_PANEL == cmd)
		goto exit_ioctl;
#endif

	if (_IOC_READ == _IOC_DIR(cmd)){
		if (!rc)
			if (copy_to_user((void *)arg,
			   (void *)pData, length)){
				TCH_DBG("ts_misc_ioctl: copy to user failed!\n");
				rc = -EFAULT;
			}
	} else {
		if (g_ts->reg_map.int_setting == addr)
			down(&ts_sem);

		rc = ts_write_i2c( client, addr, &value, 1);
		if (!rc){
			if (g_ts->reg_map.int_setting == addr){
				if((value & AUO_TOUCH_IND_ENA) == AUO_TOUCH_IND_ENA)
					g_touch_ind_mode = 1;
				else
					g_touch_ind_mode = 0;
				TCH_DBG("%s: touch indicate mode? %s\n", __func__, (g_touch_ind_mode? "YES": "NO"));
			}
		}

		if (g_ts->reg_map.int_setting == addr)
			up(&ts_sem);
	}

exit_ioctl:
	TCH_DBG("ts_misc_ioctl()-, rc=%d\n", rc );
  
	return rc;
}
#endif

static int ts_misc_release(struct inode *inode, struct file *fp)
{
	TCH_DBG("ts_misc_release()\n");

	return 0;
}

static int ts_misc_open(struct inode *inode, struct file *fp)
{
	TCH_DBG("ts_misc_open()\n");

	return 0;
}

static struct file_operations ts_misc_fops = {
	.owner 	= THIS_MODULE,
	.open 	= ts_misc_open,
	.release = ts_misc_release,
#if USE_AUO_5INCH
	.unlocked_ioctl = ts_misc_ioctl,
#endif
};

static struct miscdevice ts_misc_device = {
	.minor 	= MISC_DYNAMIC_MINOR,
	.name 	= "auo_misc_touch",
	.fops 	= &ts_misc_fops,
};

static int auo_ts_remove(struct i2c_client *client)
{
	struct ts_t *ts = (struct ts_t *)i2c_get_clientdata( client );

	TCH_DBG("auo_ts_remove()\n");

 // destroy touch work queue
	cancel_rearming_delayed_workqueue(ts->touch_wqueue, &ts->touch_work);
	destroy_workqueue(ts->touch_wqueue);
	misc_deregister(&ts_misc_device);

	input_unregister_device(ts->input);
	input_free_device(ts->input);
	ts->input = NULL;

	ts_release_gpio(ts->gpio_num, ts->gpio_rst);
	i2c_set_clientdata(client, NULL);

	g_ts = NULL;
	kfree(ts);

	return 0;
}

static int auo_ts_suspend_ic(struct ts_t *ts, struct i2c_client *client)
{
	int rc = 0;
	uint8_t addr, data;

	down(&auo_ts_suspend_sem);

	if(g_touch_suspended)
		goto exit_suspend;

	disable_irq(ts->irq);
	g_touch_suspended = 1;

	rc = cancel_work_sync(&ts->touch_work.work);
	if (rc) /* if work was pending disable-count is now 2 */
		enable_irq(ts->irq);
	//flush_workqueue(ts->touch_wqueue);

  
	addr = ts->reg_map.power_mode;
	rc = ts_read_i2c(client, addr, &data, 1);
	if( rc ) goto exit_suspend;

	data &= (~AUO_POWER_MODE_MASK);
	if( g_pixcir_freeze ){
		data |= (AUO_REAL_SLEEP_POWER_MODE - AUO_ACTIVE_POWER_MODE);
	}else{
		data |= (AUO_DEEP_SLEEP_POWER_MODE - AUO_ACTIVE_POWER_MODE);
	     }
	rc = ts_write_i2c( client, addr, &data, 1);
	if(rc)
		TCH_DBG("auo_ts_suspend_ic write i2c failed!!!\n");
    
	rc = ts->pm8901_l1b_vote_set(0);

exit_suspend:
	if(rc) 
		TCH_DBG("%s: touch suspend failed\n", __func__);
	up(&auo_ts_suspend_sem);

	return rc;
}

static int auo_ts_resume_ic(struct ts_t *ts, struct i2c_client *client)
{
	int rc = 0;
	uint8_t addr, data;
	addr = ts->reg_map.power_mode;
   
	down(&auo_ts_suspend_sem);

	if(0 == g_touch_suspended)
		goto exit_resume;
  
	rc = ts->pm8901_l1b_vote_set(1);
    
	if( g_pixcir_freeze ){
		ts_reset_panel(ts);
	}else {
		addr = ts->reg_map.power_mode;
		rc = ts_read_i2c( client, addr, &data, 1);
		if( rc ) 
			goto exit_resume;
		data &= (~AUO_POWER_MODE_MASK);
		data |= (AUO_ACTIVE_POWER_MODE - AUO_ACTIVE_POWER_MODE);
		rc = ts_write_i2c( client, addr, &data, 1);
		if( rc )
			goto exit_resume;
	}
	rc = ts_config_panel(ts);
	if( rc )
		goto exit_resume;

	if(!rc){
		enable_irq( ts->irq );
		g_touch_suspended = 0;
	} 

exit_resume:
	if( rc ) 
		TCH_DBG( "auo_ts_resume_ic: touch resume failed\n");
	up( &auo_ts_suspend_sem );

	return rc;
}

#ifdef CONFIG_HAS_EARLYSUSPEND
static void auo_ts_suspend(struct early_suspend *h)
#else
static int auo_ts_suspend(struct i2c_client *client, pm_message_t mseg)
#endif
{
	int rc = 0;
#ifdef CONFIG_HAS_EARLYSUSPEND
	struct ts_t *ts = container_of(h,
                                  struct ts_t,
                                  touch_early_suspend);
	struct i2c_client *client = ts->client_4_i2c;
#else
	struct ts_t *ts = (struct ts_t *)i2c_get_clientdata(client);
#endif

	rc = auo_ts_suspend_ic(ts, client);

	TCH_DBG( "auo_ts_suspend: X\n");

#if GETTOUCHLOG_AFTER_SCREENOFF
	atomic_set(&GetTouchLog1AfterLaterResume,0);
	atomic_set(&GetTouchLog2AfterLaterResume,0);
#endif

#ifndef CONFIG_HAS_EARLYSUSPEND
	return rc;
#endif
}

#ifdef CONFIG_HAS_EARLYSUSPEND
static void auo_ts_resume(struct early_suspend *h)
#else
static int auo_ts_resume(struct i2c_client *client)
#endif
{
	int rc = 0;
#ifdef CONFIG_HAS_EARLYSUSPEND
	struct ts_t *ts = container_of(h,
                                struct ts_t,
                                touch_early_suspend);
	struct i2c_client *client = ts->client_4_i2c;
#else
	struct ts_t *ts = (struct ts_t *)i2c_get_clientdata(client);
#endif


#if GETTOUCHLOG_AFTER_SCREENOFF
	atomic_set(&GetTouchLog1AfterLaterResume,5);
	atomic_set(&GetTouchLog2AfterLaterResume,5);
#endif

	rc = auo_ts_resume_ic(ts, client);

	TCH_DBG("auo_ts_resume: X\n");

#ifndef CONFIG_HAS_EARLYSUSPEND
	return rc;
#endif
}

//### DO NOT MODIFY THE ORDER ###
static struct device_attribute auo_touch_ctrl_attrs[] = {
	__ATTR(dump_reg, 0444, auo_touch_dump_property, NULL),
	__ATTR(dump_var, 0444, auo_touch_dump_property, NULL),
	__ATTR(dump_touch, 0444, auo_touch_dump_property, NULL),
};

static ssize_t auo_touch_dump_property(struct device *dev, struct device_attribute *attr, char *buf)
{
	const ptrdiff_t offset = attr - auo_touch_ctrl_attrs;
	struct i2c_client *client = container_of(dev,
                                           struct i2c_client,
                                           dev);
	uint8_t data[5];
	int strLen = 0;
	
	strLen = sprintf(buf+strLen,"----- DUMP START -----\n");

	switch(offset){
		case 0: //dump_reg
		{   
//TODO: check ic power if needed
			if(!ts_read_i2c(client, 0x77, data, 1))
				strLen += sprintf(buf+strLen, "VERSION(0x%02X):\n", data[0]);

			if(!ts_read_i2c(client, 0x6F, data, 5)){
				strLen += sprintf(buf+strLen, "X_SENSITIVITY=0x%02X\n", data[0]);
				strLen += sprintf(buf+strLen, "Y_SENSITIVITY=0x%02X\n", data[1]);
				strLen += sprintf(buf+strLen, "INT_SETTING=0x%02X\n", data[2]);
				strLen += sprintf(buf+strLen, "INT_WIDTH=0x%02X\n", data[3]);
				strLen += sprintf(buf+strLen, "POWER_MODE=0x%02X\n", data[4]);
			}
		break;
		}
		case 1: //dump_var
		{
			down(&ts_sem);
			strLen += sprintf(buf+strLen, "GLOBAL VARS:\n");
			strLen += sprintf(buf+strLen, "psensor approached=%d\n", atomic_read(&psensor_approached));
			strLen += sprintf(buf+strLen, "touch indicate int=%d\n", g_touch_ind_mode);
			strLen += sprintf(buf+strLen, "pixcir detected=%d\n", g_pixcir_detected);
			strLen += sprintf(buf+strLen, "pixcir freeze=%d\n", g_pixcir_freeze);
			up(&ts_sem);
		break;
		}
		case 2: //dump_touch
		{
			struct auo_mt_t *pMsg = &g_ts->mtMsg;
			struct auo_point_info_t *pPoint;
			int i;

			strLen += sprintf(buf+strLen, "TOUCH INFO:\n");
			strLen += sprintf(buf+strLen, "mt state=%d\n", pMsg->state);
			for(i=0; i<AUO_REPORT_POINTS; i++){
				pPoint = &pMsg->points[i];
				strLen += sprintf(buf+strLen, "point[%d] state =%d\n", i, pPoint->state);
				strLen += sprintf(buf+strLen, "point[%d] coord =(%d,%d)\n", i, pPoint->coord.x, pPoint->coord.y);
			}
		break;
		}
	}

	strLen += sprintf(buf+strLen, "----- DUMP END   -----\n");

	return strLen;
}

static const struct i2c_device_id i2cAuoTouch_idtable[] = {
	{ AUO_TS_DRIVER_NAME, 0 },
	{ }
};

MODULE_DEVICE_TABLE(i2c, i2cAuoTouch_idtable);

static struct i2c_driver i2c_ts_driver = {
	.driver = {
		.owner = THIS_MODULE,
		.name  = AUO_TS_DRIVER_NAME,
	},
	.probe	 = auo_ts_probe,
	.remove	 = auo_ts_remove,
#ifndef CONFIG_HAS_EARLYSUSPEND
	.suspend = auo_ts_suspend,
	.resume  = auo_ts_resume,
#endif
	.id_table = i2cAuoTouch_idtable,
};

//static int auo_ts_probe(struct i2c_client *client)
static int __devinit auo_ts_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	int    result;
	struct ts_t *ts;
	struct auo_touchscreen_platform_data *pdata;
	int i;

	TCH_DBG("[Leo], AUO Touch Screen auo_ts_probe init \n");	

// allc ts context
	ts = kzalloc(sizeof(struct ts_t), GFP_KERNEL);
	if(!ts){
		result = -ENOMEM;
		return result;
	}

 // assign context
	memset(ts, 0, sizeof(struct ts_t));
	pdata = client->dev.platform_data;
	ts->gpio_num = pdata->gpioirq;
	ts->gpio_rst = pdata->gpiorst;
/*Orange EVT2 CR XXXX Jack W Lu 2011-09-03 Add for L1B vote, Power Consumption issue {*/
	ts->pm8901_l1b_init = pdata->pm8901_l1b_init;
	ts->pm8901_l1b_vote_set = pdata->pm8901_l1b_vote_set;
/*Orange EVT2 CR XXXX Jack W Lu 2011-09-03 Add for L1B vote, Power Consumption issue }*/
	//ts->irq = MSM_GPIO_TO_INT(ts->gpio_num);
	ts->irq = gpio_to_irq(ts->gpio_num);
	ts->client_4_i2c = client;
 
	TCH_DBG("[Leo], AUO Touch Screen ts->gpio_rst is %d \n", ts->gpio_rst);
	TCH_DBG("[Leo], AUO Touch Screen ts->gpio_num is %d \n", ts->gpio_num);	
	TCH_DBG("[Leo], AUO Touch Screen ts->irq is %d \n", ts->irq);	
	
	result = misc_register( &ts_misc_device );
	if(result){
		TCH_DBG("[Leo], AUO Touch Screen misc_register fail \n");
		goto fail_alloc_mem;
	}

  // register input device
	result = ts_register_input(&ts->input, pdata, client);
	if(result)
		goto fail_misc_register;
	input_set_drvdata(ts->input, ts);

  // setup gpio
	result = ts_setup_gpio( ts->gpio_num, ts->gpio_rst);	
	if(result)
		goto fail_config_gpio;
        		
  // enalbe power
	result = ts_power_switch(1,ts);
	if(result){
		TCH_DBG("[Leo], AUO Touch Screen power switch fail \n");
		goto fail_setup_gpio;
	}
	ts->pm8901_l1b_vote_set(1);
  
  //mdelay(100);

  // config gpio
	result = ts_config_gpio(ts->gpio_num, ts->gpio_rst);
	if(result){
		TCH_DBG("[Leo], AUO Touch Screen config_gpio fail \n");
		goto fail_power_on;
	}

  // detect pixcir IC
	result = ts_detect_pixcir(ts);
	if(result){
		int ret;
		char buf_r[5];
		TCH_DBG("[Leo], AUO Touch Screen detect fail \n");
		ts->client_4_i2c->addr = 0x5D;
		ret = boot_read_mem( client,0xFF,5, buf_r);
		ts->client_4_i2c->addr = 0x5C;
		 if(0 != ret)
			goto fail_power_on;
		else
			TCH_DBG("[Leo], AUO Touch Screen is in bootloader mode \n");
	}

  // set up register map
	result = ts_setup_reg_map(ts);
	if(result){
		TCH_DBG("[Leo], AUO Touch Screen setup reg map fail \n");
		goto fail_power_on;
	}

  // optional; depend on panel
	result = ts_config_panel(ts);
	if(result){
		TCH_DBG("[Leo], AUO Touch Screen config panel fail \n");
	}

  // create touch work queue
	INIT_DELAYED_WORK( &ts->touch_work, ts_irqWorkHandler);
	ts->touch_wqueue = create_singlethread_workqueue(AUO_TS_DRIVER_NAME);
	if (!ts->touch_wqueue) {
		result = -ESRCH;
		TCH_DBG("[Leo], AUO Touch Screen create touch_wqueue fail \n");
		goto fail_power_on;
	}
	client->driver = &i2c_ts_driver;
	i2c_set_clientdata(client, ts);
	g_ts = ts;

	TCH_DBG("[Leo], AUO Touch Screen ts->irq=%d \n", ts->irq);
/*Qsida,Alan P Xu,2011.07.27, GPIO of touch be changed{*/
#if defined(CONFIG_MACH_EVT2)
	result = request_irq( ts->irq, ts_irqHandler, IRQF_TRIGGER_RISING, "AUO_touchscreen", ts );
#elif defined (CONFIG_MACH_EVT1)
	result = request_any_context_irq( ts->irq, ts_irqHandler, IRQF_TRIGGER_RISING,"AUO_touchscreen", ts);  
#endif
/*Qsida,Alan P Xu,2011.07.27, GPIO of touch be changed}*/                    	
	TCH_DBG("[Leo], AUO Touch Screen request_irq result is %d \n", result);
	

// register early suspend/resume
#ifdef CONFIG_HAS_EARLYSUSPEND
	ts->touch_early_suspend.level = 150; //EARLY_SUSPEND_LEVEL_DISABLE_FB;
	ts->touch_early_suspend.suspend = auo_ts_suspend;
	ts->touch_early_suspend.resume = auo_ts_resume;
	register_early_suspend(&ts->touch_early_suspend);
#endif

// create device attribute
	for (i=0; i<ARRAY_SIZE(auo_touch_ctrl_attrs); i++){
		result = device_create_file(&client->dev, &auo_touch_ctrl_attrs[i]);
		if (result) 
			TCH_DBG("%s: create file err, %d\n", __func__, result);
	}

	return 0;

fail_power_on:
#ifdef USE_PIXCIR
/*Orange Jack W Lu 2011-12-20 fix the L1B voltage error {*/
	if(ts->pm8901_l1b_vote_set != NULL)
		ts->pm8901_l1b_vote_set(0);
/*Orange Jack W Lu 2011-12-20 fix the L1B voltage error }*/
	ts_power_switch(0,ts);
fail_setup_gpio:
#endif
	ts_release_gpio(ts->gpio_num, ts->gpio_rst);
fail_config_gpio:
	input_unregister_device(ts->input);
	input_free_device(ts->input);
	ts->input = NULL;

fail_misc_register:
	misc_deregister(&ts_misc_device);

fail_alloc_mem:
	kfree(ts);
	g_ts = NULL;
  
	return result;
}


static int __init auo_ts_init(void)
{
	int rc;
	TCH_DBG("[Leo], AUO_TouchScreen auo_ts_init init \n");
	i2c_ts_driver.driver.name = AUO_TS_DRIVER_NAME;
	rc = i2c_add_driver( &i2c_ts_driver );
	TCH_DBG("[Leo], AUO_TouchScreen init result is %d \n",rc);

	return rc;
}
module_init(auo_ts_init);


static void __exit ts_exit(void)
{
	TCH_DBG("ts_exit()\n");

	i2c_del_driver(&i2c_ts_driver);
}
module_exit(ts_exit);


MODULE_DESCRIPTION("AUO touchscreen driver");
MODULE_LICENSE("GPL v2");
MODULE_AUTHOR("MENG-HAN CHENG");
MODULE_ALIAS("platform:auo_touch");
