
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>			/* Initdata			*/
#include <linux/videodev2.h>		/* kernel radio structs		*/
#include <media/v4l2-common.h>
#include <media/v4l2-ioctl.h>
#include <linux/version.h>      	/* for KERNEL_VERSION MACRO     */
#include <linux/delay.h>
/*ICE,20100305,Snail Qian,implement VIDIOC_DQBUF{*/
#include <linux/kfifo.h>        /* lock free circular buffer    */
#include <linux/irq.h>
#include <linux/interrupt.h>

#include <net/bluetooth/bluetooth.h>
#include <net/bluetooth/hci_core.h>
#include <media/radio-bcm4329.h>
#include <net/bluetooth/hci.h>
#include <linux/completion.h>
#include <linux/platform_device.h>


#define DRIVER_VERSION	"v0.01"
#define RADIO_VERSION	KERNEL_VERSION(0, 0, 1)

#define DRIVER_AUTHOR	"Snail.Qian <snail.qian@qisda.com>"
#define DRIVER_DESC	"A driver for the BCM4325 radio chip."

#define PINFO(format, ...)\
	printk(KERN_INFO KBUILD_MODNAME ": "\
		DRIVER_VERSION ": " format "\n", ## __VA_ARGS__)
#define PWARN(format, ...)\
	printk(KERN_WARNING KBUILD_MODNAME ": "\
		DRIVER_VERSION ": " format "\n", ## __VA_ARGS__)
#define PDEBUG(format, ...)\
	printk(KERN_DEBUG KBUILD_MODNAME ": "\
		DRIVER_VERSION ": " format "\n", ## __VA_ARGS__)


/* Frequency limits in MHz -- these are European values.  For Japanese
devices, that would be 76000 and 91000.  */

#define FREQ_MIN_E  87500
#define FREQ_MAX_E 108000
#define FREQ_MIN_J  76000
#define FREQ_MAX_J 91000
#define FREQ_MUL 16

int fm_standard=0; //0 means E and 1 means J


#define BCM_HCI_ERROR -1
#define BCM_HCI_READ  0x01
#define BCM_HCI_WRITE 0x00

#ifndef RADIO_BCM4329_XTAL
#define RADIO_BCM4329_XTAL 1
#endif

#define V4L2_AUDCAP_ANALOG		0x00003
#define V4L2_AUDCAP_I2S			0x00004


static int radio_nr = -1;

/* RDS buffer blocks */
static unsigned int rds_buf = 100;

struct mutex			ioctl_mutex;

int g_signal_th=0x64;
bool is_lp=false;
//int FM_MODE=0;  // OFF=0; RX=1; TX=2;
int FM_MODE=1;


struct bcm4329_device {
	struct video_device		*videodev;
	struct hci_dev            *hdev;
	struct mutex			mutex;	
	int				user;
	struct completion hci_confirmed;	
	uint8_t i2c_addr;	
	int     hci_result;	
	uint8_t read_len;
	uint8_t *read_buf;
	struct kfifo data_buf[BCM4329_BUF_MAX];	
	spinlock_t buf_lock[BCM4329_BUF_MAX];
	/* wait queue for blocking event read */
	wait_queue_head_t event_queue;
	/* wait queue for raw rds read */
	wait_queue_head_t read_queue;
};

struct bcm4329_device *bcm4329_device_data;

#ifdef FM_RDS
static void rds_timer( unsigned long data);
static DEFINE_TIMER(fm_rds, rds_timer, 0, 0);
#endif

/* I2C code related */
int bcm4329_read(uint8_t regaddr,uint8_t *buf, uint32_t rdlen)
{
	 /*i2c address + read flag(0x01) + read length*/
	uint8_t para[3];
	int ret=0;
	PINFO("bcm4329_read" );
	/*ICE,201000325,Snail Qian, add for ioctl block*/
	mutex_lock(&ioctl_mutex);

	para[0] = regaddr;
	para[1] = BCM_HCI_READ;
	para[2] = rdlen;
	
	bcm4329_device_data->read_buf = buf;	
	bcm4329_device_data->read_len = rdlen;
	bcm4329_device_data->i2c_addr = regaddr;
	
	ret = hci_send_cmd(bcm4329_device_data->hdev,VSC_HCI_WR_CMD,3,para);	

	if (!ret && !wait_for_completion_timeout(&bcm4329_device_data->hci_confirmed, msecs_to_jiffies(1000))){
		ret = BCM_HCI_ERROR;
	}	
	else{
		ret = bcm4329_device_data->hci_result;
	}
	mutex_unlock(&ioctl_mutex);
	
	return ret;
}

int bcm4329_write(uint8_t regaddr, uint8_t *data, int wlen)
{
	uint8_t* para;
	int ret=0;
	
	PINFO("bcm4329_write" );

	mutex_lock(&ioctl_mutex);
	
	/*i2c address + read flag(0x00) + write length*/
	para =(uint8_t*)kmalloc(wlen+2, GFP_KERNEL);
    
	if(!para){
		return BCM_HCI_ERROR;
	}
  
	para[0] = regaddr;
	para[1] = BCM_HCI_WRITE;
	memcpy(para+2,data,wlen);

	ret = hci_send_cmd(bcm4329_device_data->hdev,VSC_HCI_WR_CMD,wlen+2,para);
	
	bcm4329_device_data->i2c_addr = regaddr;
	
	if (!ret && !wait_for_completion_timeout(&bcm4329_device_data->hci_confirmed, msecs_to_jiffies(1000))){
		ret = BCM_HCI_ERROR;
	}	
	else{
		ret = bcm4329_device_data->hci_result;
	}
	mutex_unlock(&ioctl_mutex);
	PINFO("ioctl_mutex unlock" );
	kfree(para);
	return ret;
}

static void bcm4329_q_event(struct bcm4329_device *radio,
				enum bcm4329_evt_t event)
{
	struct kfifo *data_b = &radio->data_buf[BCM4329_BUF_EVENTS];
	unsigned char evt = event;
	PINFO("updating event_q with event %x\n", event);
	if (kfifo_in_locked(data_b, &evt, 1, &radio->buf_lock[BCM4329_BUF_EVENTS]))
		wake_up_interruptible(&radio->event_queue);
}


/* V4L2 code related */
static struct v4l2_queryctrl radio_qctrl[] = {
	{
		.id            = V4L2_CID_AUDIO_MUTE,
		.name          = "Mute",
		.minimum       = 0,
		.maximum       = 1,
		.default_value = 1,
		.type          = V4L2_CTRL_TYPE_BOOLEAN,
	}
};


static void bcm4329_power_up(struct bcm4329_device *radio)
{
  uint8_t fm_mode = 0;
  uint8_t fm_power = 0;
  uint8_t data =0;
  int ret=0;

	//RX MODE
	if (FM_MODE==1)
	{
		fm_mode=BCOMFM_RX_PAGE;
		ret = bcm4329_write(I2C_PAGE_CONTEXT, &fm_mode, sizeof(fm_mode));
		PINFO("set I2C_PAGE_CONTEXT ret=%d \n",ret );
		
		fm_power |= BCOMFM_POWER_PWF;
		ret = bcm4329_write(I2C_FM_RDS_SYSTEM, &fm_power, sizeof(fm_power));
		PINFO("BCOMFM_POWER_ON_IO:I2C_FM_RDS_SYSTEM ret=%d \n",ret );

		//For stable power on
		mdelay(10);
		//I2C_FM_CTRL
		data= BCOMFM_BAND_EUUS;	//auto stereo/mono select
		data |=BCOMFM_AUTO_SELECT_STEREO_MONO;
		//data |=BCOMFM_STEREO_MONO_BLEND;
		PINFO("BCOMFM_POWER_ON_IO:I2C_FM_CTRL data = 0x%x \n",data );
		ret = bcm4329_write(I2C_FM_CTRL, &data, sizeof(data));
		PINFO("BCOMFM_POWER_ON_IO:I2C_FM_CTRL ret=%d \n",ret );

		//I2C_FM_AUDIO_CTRL0
		data= (BCOMFM_DAC_OUTPUT_LEFT|BCOMFM_DAC_OUTPUT_RIGHT|
		BCOMFM_DE_EMPHASIS_75|BCOMFM_AUDIO_ROUTE_DAC);	//auto stereo/mono select
		PINFO("BCOMFM_POWER_ON_IO:I2C_FM_AUDIO_CTRL0 data = 0x%x \n",data );
		ret = bcm4329_write(I2C_FM_AUDIO_CTRL0, &data, sizeof(data));
		PINFO("BCOMFM_POWER_ON_IO:I2C_FM_AUDIO_CTRL0 ret=%d \n",ret );
	}

	//TX MODE
	else if (FM_MODE==2)
	{
		fm_mode=BCOMFM_TX_PAGE;
		ret = bcm4329_write(I2C_PAGE_CONTEXT, &fm_mode, sizeof(fm_mode));
		PINFO("set I2C_PAGE_CONTEXT ret=%d \n",ret );
		
		data = BCOMFM_POWER_TXON;
		data |= BCOMFM_FMTX_BAND_UE;	
		data |=BCOMFM_FMTX_AUD_ADC;
		data |=BCOMFM_FMTX_AUD_STR;
		data |=BCOMFM_FMTX_PREEMPH_75;
		ret = bcm4329_write(I2C_FMTX_CTRL_SET, &data, sizeof(data));
		PINFO("I2C_FMTX_CTRL_SET ret=%d \n",ret );

	}
	
	
}


static void bcm4329_power_down(struct bcm4329_device *radio)
{
	uint8_t data =0;
	uint8_t fm_mode;
	data = 0x00;

	PINFO( "bcm4329 power down ");
	if (FM_MODE==1) 
	{
		fm_mode=BCOMFM_RX_PAGE;
		bcm4329_write(I2C_PAGE_CONTEXT, &fm_mode, sizeof(fm_mode));
		mdelay(10);
		bcm4329_write(I2C_FM_RDS_SYSTEM, &data, sizeof(data));
	}
	else if (FM_MODE==2) 
	{
		fm_mode=BCOMFM_TX_PAGE;
		bcm4329_write(I2C_PAGE_CONTEXT, &fm_mode, sizeof(fm_mode));
		mdelay(10);
		bcm4329_write(I2C_FMTX_CTRL_SET, &data, sizeof(data));
	}
	
}

static void bcm4329_set_freq(struct bcm4329_device *radio, int *freq)
{
	*freq = (*freq * 100) - 64000;
}


static int bcm4329_get_freq(struct bcm4329_device *radio)
{
	uint8_t  freqdata[2];
	uint32_t freq = 0;
	
	if(!bcm4329_read(I2C_FM_FREQ0, freqdata, 2))
	{
	  PINFO( "BCOMFM_GET_FREQ result: freqdata %x,%x \n", freqdata[0],freqdata[1]);
    //Transfer frequency
    freq = (freqdata[1] << 8) | freqdata[0];
    freq = (freq + 64000) /100;
    
    PINFO( "BCOMFM_GET_FREQ: freq = %d\n", freq);
	}
		
	return freq; 
}

static void bcm4329_tune(struct bcm4329_device *radio, int freq)
{
	int temp = freq;
	uint8_t data;
	uint8_t freqdata[2] = {0};
	uint16_t count =0;
	uint8_t flag_status[2]= {0};
	
	bcm4329_set_freq(radio, &temp);	 		
	freqdata[0] = (uint8_t) ( temp & 0xff );    		
	freqdata[1] = (uint8_t) ( temp >> 8 );		  
	PINFO("BCOMFM_SPECIFIED_FREQ_SET: freq =0x%x 0x%x \n", freqdata[0], freqdata[1]);		
	bcm4329_write(I2C_FM_FREQ0, freqdata, 2);
	if (FM_MODE==1) 
	{
		bcm4329_write(I2C_FM_FREQ0, freqdata, 2);
		
		data = BCOMFM_PRE_SET;	
		bcm4329_write(I2C_FM_SEARCH_TUNE_MODE,&data,1);	
		
		do {
		mdelay(50);
		bcm4329_read(I2C_FM_RDS_FLAG0,  flag_status,  sizeof(flag_status));
		PINFO("I2C_FM_RDS_FLAG0 :flag_status[0] = %x, flag_status[1] = %x count = %d\n", flag_status[0], flag_status[1], count);
		count++;
		}while( (0 == (flag_status[0] & 0x01) ) && (count < BCOMFM_MAX_TUNING_TRIALS));
	
	}
	else if (FM_MODE==2) 
	{
		bcm4329_write(I2C_FMTX_CHANL_SET0, freqdata, 2);

		do {
		mdelay(50);
		bcm4329_read(I2C_FMTX_IRQ_FLAG0,  flag_status,  sizeof(flag_status));
		PINFO("I2C_FMTX_IRQ_FLAG0 :flag_status[3] = %x,  count = %d\n", flag_status[0],count);
		count++;
	}while( (0 == (flag_status[3] & 0x01) ) && (count < BCOMFM_MAX_TUNING_TRIALS));
	}
		
	//Check Tuning process if finish
	
	
}

static void bcm4329_set_audout_mode(struct bcm4329_device *radio, int audmode)
{
	uint8_t data =0;
	if (FM_MODE==1) 
	{
		bcm4329_read(I2C_FM_CTRL, &data, 2);
		if (fm_standard==0)
		{
			data|=BCOMFM_BAND_EUUS;
		}
		else
		{
			data|=BCOMFM_BAND_JP;
		}
		
		if(audmode==V4L2_TUNER_MODE_STEREO)
		{
			data|=BCOMFM_AUTO_SELECT_STEREO_MONO|BCOMFM_STEREO_MONO_SWITCH;
		
		} 
		else
		{
			data|=BCOMFM_AUTO_SELECT_STEREO_MONO|BCOMFM_STEREO_MONO_BLEND; 
		}
		
		bcm4329_write(I2C_FM_CTRL, &data, sizeof(data));
	}
	else if (FM_MODE==2) 
	{	
		bcm4329_read(I2C_FMTX_CTRL_SET, &data, 2);
		if (fm_standard==0)
		{
			data|=BCOMFM_FMTX_BAND_UE;
		}
		else
		{
			data|=BCOMFM_FMTX_BAND_JA;
		}
		
		if(audmode==V4L2_TUNER_MODE_STEREO)
		{
			data|=BCOMFM_FMTX_AUD_STR;
		
		}  //infact it's better to read ,then change
		else
		{
			data|=BCOMFM_FMTX_AUD_MONO; 
		}
		
		bcm4329_write(I2C_FMTX_CTRL_SET, &data, sizeof(data));
	}

}

static void bcm4329_set_audout_path(struct bcm4329_device *radio, int audpath)
{
	uint8_t data =0;
	PINFO("bcm4329_set_audout_path\n");
	
	if (FM_MODE==1) 
	{	
		bcm4329_read(I2C_FM_AUDIO_CTRL0, &data, 2);
		PINFO("pre data= %d\n", data);
		if (audpath==V4L2_AUDCAP_ANALOG) // 0x0003: Analog; 0x0004: I2S
		{
			data|= BCOMFM_AUDIO_ROUTE_DAC;
			data&=0xdf;
			PINFO("A : new data= %d\n", data);
		}
		else
		{
			data|= BCOMFM_AUDIO_ROUTE_I2S;
			data&=0xef;
			PINFO("D : new data= %d\n", data);
		}
		bcm4329_write(I2C_FM_AUDIO_CTRL0, &data, sizeof(data));
	}
	else if (FM_MODE==2) 
	{
		bcm4329_read(I2C_FMTX_CTRL_SET, &data, 2);
		if (audpath==V4L2_AUDCAP_ANALOG) // 0: Analog; 1: I2S
		{
			data|= BCOMFM_FMTX_AUD_ADC;
		}
		else
		{
			data|= BCOMFM_FMTX_AUD_I2S;
		}		
		bcm4329_write(I2C_FMTX_CTRL_SET, &data, sizeof(data));
	}

}


static int bcm4329_get_audout_mode(struct bcm4329_device *radio)
{
	uint8_t data=0;
	int mode;
	if (FM_MODE==1)
	{
		if(!bcm4329_read(I2C_FM_CTRL, &data, 2))
		{     
    		PINFO( "aud mode : %d\n", data);
			if ((data&0x08)==0)
			{
				mode=V4L2_TUNER_MODE_MONO;
			}	
			else
			{
				mode=V4L2_TUNER_MODE_STEREO;
			}
		}
	}
	else if (FM_MODE==2)
	{
		if(!bcm4329_read(I2C_FMTX_CTRL_SET, &data, 2))
		{     
    		PINFO( "aud mode : %d\n", data);
			if ((data&0x02)==0)
			{
				mode=V4L2_TUNER_MODE_MONO;
			}	
			else
			{
				mode=V4L2_TUNER_MODE_STEREO;
			}
		}
	}
	return mode; 
	
}


static void bcm4329_mute(struct bcm4329_device *radio, int on)
{
	uint8_t data =0;
	if(on==1)
		{
			data=BCOMFM_ENABLE_RF_MUTE|BCOMFM_MANUAL_MUTE_ON;
		}
	else
		{
			data=BCOMFM_DISABLE_RF_MUTE|BCOMFM_MANUAL_MUTE_OFF;
		}
	bcm4329_write(I2C_FM_AUDIO_CTRL0, &data, sizeof(data));  
}

static int bcm4329_is_muted(struct bcm4329_device *radio)
{
	uint8_t data =0;
	int is_muted=0;
	bcm4329_read(I2C_FM_AUDIO_CTRL0, &data, sizeof(data));
	
	if((data&0x03)==0x03)
	{
		is_muted=1;
	}
	else
	{
		is_muted=0;
	}
	return is_muted;
}


static int bcm4329_set_signal_th(int value)
{
	g_signal_th =value;
	return 0;
}

static int bcm4329_get_signal_th(void)
{
	return g_signal_th ;
}

static int bcm4329_enter_lp(struct bcm4329_device *radio)
{
	if(is_lp==false)
		{
			bcm4329_power_down(radio);
			is_lp=true;
		}
	return 0;		
	
}

static int bcm4329_exit_lp(struct bcm4329_device *radio)
{
		if(is_lp==true)
			{
				bcm4329_power_up(radio);				
				is_lp=false;
			}
					
		return 0;		
		

}


/* V4L2 vidioc */
static int vidioc_querycap(struct file *file, void  *priv,
					struct v4l2_capability *v)
{
	struct bcm4329_device *radio = video_drvdata(file);
	struct video_device *dev = radio->videodev;

	strlcpy(v->driver, dev->dev.driver->name, sizeof(v->driver));
	strlcpy(v->card, dev->name, sizeof(v->card));
	//snprintf(v->bus_info, sizeof(v->bus_info), "I2C:%s", dev->dev.bus_id);
	v->version = RADIO_VERSION;
#ifdef FM_RDS
	v->capabilities = V4L2_CAP_TUNER | V4L2_CAP_RADIO ;
#endif
	v->capabilities = V4L2_CAP_TUNER | V4L2_CAP_RADIO | V4L2_CAP_RDS_CAPTURE;//snail ,RDS is added
	return 0;
}

static int vidioc_g_tuner(struct file *file, void *priv,
				struct v4l2_tuner *v)
{
	struct bcm4329_device *radio = video_drvdata(file);
	
	if (v->index > 0)
		return -EINVAL;

	memset(v, 0, sizeof(v));  //snail need to confirm what's in v
	strcpy(v->name, "FM");
	v->type = V4L2_TUNER_RADIO;
	if (fm_standard ==0)
	{
		v->rangelow   = FREQ_MIN_E * FREQ_MUL;
		v->rangehigh  = FREQ_MAX_E * FREQ_MUL;
	}
	else
	{
		v->rangelow   = FREQ_MIN_J * FREQ_MUL;
		v->rangehigh  = FREQ_MAX_J * FREQ_MUL;
	}
	v->capability = V4L2_TUNER_CAP_LOW | V4L2_TUNER_CAP_STEREO;
	v->rxsubchans = V4L2_TUNER_SUB_STEREO;
	v->audmode = bcm4329_get_audout_mode(radio);
	return 0;
}


static int vidioc_s_tuner(struct file *file, void *priv,
				struct v4l2_tuner *v)
{
	struct bcm4329_device *radio = video_drvdata(file);

	if (v->index > 0)
		return -EINVAL;
    if ((v->rangehigh==FREQ_MAX_E)&&(v->rangelow==FREQ_MIN_E))
    	{
			fm_standard=0;
		}
	else if ((v->rangehigh==FREQ_MAX_J)&&(v->rangelow==FREQ_MIN_J))
		{
			fm_standard=1;
		}
	else
		{
			PINFO( "invalid rang");
		}
	bcm4329_set_audout_mode(radio, v->audmode);  
	return 0;
}


static int vidioc_s_frequency(struct file *file, void *priv,
				struct v4l2_frequency *f)
{
	struct bcm4329_device *radio = video_drvdata(file);

	if (f->tuner != 0)
		return -EINVAL;
	if (f->frequency == 0) {
		/* We special case this as a power down control. */
		bcm4329_power_down(radio);
	}
	
	bcm4329_tune(radio, f->frequency);  

	PINFO("bcm4329_q_event:BCM4329_EVT_TUNE_SUCC");
	bcm4329_q_event(radio,BCM4329_EVT_TUNE_SUCC);

	return 0;
}


static int vidioc_g_frequency(struct file *file, void *priv,
				struct v4l2_frequency *f)
{
	struct bcm4329_device *radio = video_drvdata(file);
	
	PINFO("vidioc_g_frequency");

	memset(f, 0, sizeof(f));
	f->type = V4L2_TUNER_RADIO;
	
	f->frequency = bcm4329_get_freq(radio);

	return 0;
}


static int vidioc_queryctrl(struct file *file, void *priv,
			    struct v4l2_queryctrl *qc)
{
	int i;

	for (i = 0; i < ARRAY_SIZE(radio_qctrl); i++) {
		if (qc->id && qc->id == radio_qctrl[i].id) {
			memcpy(qc, &(radio_qctrl[i]), sizeof(*qc));
			return 0;
		}
	}
	return -EINVAL;
}


static uint8_t bcm4329_get_rssi(struct bcm4329_device *radio)
{
  uint8_t rssi = 0;
	bcm4329_read(I2C_FM_RSSI,  &rssi,  sizeof(rssi));
	PINFO("bcm4329_get_rssi value is %x\n",rssi);
	
	return rssi;
}

static int vidioc_g_ctrl(struct file *file, void *priv,
			    struct v4l2_control *ctrl)
{
	struct bcm4329_device *radio = video_drvdata(file);

	switch (ctrl->id) {
	case V4L2_CID_AUDIO_MUTE:
		 ctrl->value = bcm4329_is_muted(radio); 
		 break;
	case BCOMFM_PRIVATE_GET_RSSI:
	     ctrl->value = bcm4329_get_rssi(radio);
		 break;
	case V4L2_CID_PRIVATE_BCM4329_LP_MODE:
	     ctrl->value=is_lp;
		 break;
	case V4L2_CID_PRIVATE_BCM4329_SIGNAL_TH:
		 ctrl->value=g_signal_th;
		 break;
	default:
	  PINFO("bcom fm get no handle control %d\n", ctrl->id);
	  break;
	}
	return 0;
}

static int vidioc_s_ctrl(struct file *file, void *priv,
			    struct v4l2_control *ctrl)
{
	struct bcm4329_device *radio = video_drvdata(file);
	int ret=-1;
	uint8_t data =0;

	switch (ctrl->id) {
	case V4L2_CID_AUDIO_MUTE:
		bcm4329_mute(radio, ctrl->value);
		ret=0;
		break;
	case V4L2_CID_PRIVATE_BCM4329_SIGNAL_TH:
		ret=bcm4329_set_signal_th(ctrl->value);
		break;
	case V4L2_CID_PRIVATE_BCM4329_RDSON:
		{
			#ifdef FM_RDS
			if(ctrl->value==TRUE)
				{
					data=BCOMFM_POWER_PWF|BCOMFM_POWER_PWR;
					ret = bcm4329_write(I2C_FM_RDS_SYSTEM, &data, sizeof(data));
					if (ret==0)
					{
					   data = RDS_WATER_LEVEL;
					   bcm4329_write(I2C_RDS_WLINE, &data, sizeof(data));
					}
					mod_timer(&fm_rds, jiffies + msecs_to_jiffies(1000));
				}
			else
				{
					data=BCOMFM_POWER_PWF;
					PINFO("STOP test timer" );
					del_timer(&fm_rds);
					ret = bcm4329_write(I2C_FM_RDS_SYSTEM, &data, sizeof(data));

				}
			#endif
		}
		break;
	case V4L2_CID_PRIVATE_BCM4329_LP_MODE:
		{
			if(ctrl->value==1)
			{
					ret=bcm4329_enter_lp(radio);
			}
			else
			{
					ret=bcm4329_exit_lp(radio);
			}
		}
		break;
	case V4L2_CID_PRIVATE_BCM4329_STATE:
		{
			if(ctrl->value==0)
			{
					bcm4329_power_down(radio);
					bcm4329_q_event(radio, BCM4329_EVT_ERROR);
					ret=0;
			}
			else 
			{
				    FM_MODE= ctrl->value;
					bcm4329_power_up(radio);
					bcm4329_q_event(radio, BCM4329_EVT_RADIO_READY);
					PINFO("bcm4329_q_event:BCM4329_EVT_RADIO_READY" );
					ret=0;
			}
		}
		break;
	case V4L2_CID_PRIVATE_BCM4329_SRCHON:
		{
			if((ctrl->value==0) && (FM_MODE==1))
			{
					data = BCOMFM_STOP_TUNE;	
					ret=bcm4329_write(I2C_FM_SEARCH_TUNE_MODE,&data,1);
			}
			else
			{
					PINFO("bcom fm set no handle ");
			}
		}
		break;
		
	default:
	  PINFO("bcom fm set no handle control %d\n", ctrl->id);
	  //ret=-1;
	  ret=0;
	  break;
	}
	return ret;
}


static int vidioc_g_audio(struct file *file, void *priv,
			   struct v4l2_audio *a)
{
	if (a->index > 1)
		return -EINVAL;

	strcpy(a->name, "Radio");
	a->capability = V4L2_AUDCAP_STEREO;
	return 0;
}

static int vidioc_s_audio(struct file *file, void *priv,
			   struct v4l2_audio *a)
{
	struct bcm4329_device *radio = video_drvdata(file);
	if (a->index != 0)
		return -EINVAL; 
	
	bcm4329_set_audout_path(radio,a->capability); //temp 
	return 0;
}

/*ICE,20100408,Snail Qian,for set_control{*/
static int vidioc_s_hw_freq_seek(struct file *file, void *priv,
					struct v4l2_hw_freq_seek *seek)
{  
	uint8_t data = 0;
	uint8_t flag_status[2]= {0};
	uint16_t count = 0;
	int start_freq;
	uint8_t freqdata[2] = {0};
	struct bcm4329_device *radio = video_drvdata(file);

	if( seek->seek_upward)
	{
		start_freq=bcm4329_get_freq(radio)-1;
	}
	else
	{
		start_freq=bcm4329_get_freq(radio)+1;
	}
	if (fm_standard==0)
	{
		if (start_freq>FREQ_MAX_E/100)
			start_freq=FREQ_MIN_E/100;
	}
	else 
	{
		if (start_freq>FREQ_MAX_J/100)
			start_freq=FREQ_MIN_J/100;
	}
	bcm4329_set_freq(radio, &start_freq);	
	PINFO("start freq set: %d \n",start_freq);
	freqdata[0] = (uint8_t) ( start_freq & 0xff );    		
	freqdata[1] = (uint8_t) ( start_freq >> 8 );		  
	PINFO("start freq set: freq =0x%x 0x%x \n", freqdata[0], freqdata[1]);		
	bcm4329_write(I2C_FM_FREQ0, freqdata, 2);	
	/*ICE,20100413,Snail Qian,adjust start freq of seek{*/
	
	PINFO("BCOMFM_FRE_SCAN: up or down = %d\n", seek->seek_upward);

	//Set scan direction and RSSI threshold, RSSI threshold = -100dBm
	//data = 0x64;		//-100dBm
	data=bcm4329_get_signal_th();
	if( seek->seek_upward){
		data |= BCOMFM_SERACH_UP;
	}
	
	bcm4329_write(I2C_FM_SEARCH_CTRL0, &data, sizeof(data));

	data = 0x00;
	//Set tuning mode as Auto search mode
	data = BCOMFM_AUTO_SERCH;
	bcm4329_write(I2C_FM_SEARCH_TUNE_MODE, &data, sizeof(data));	

	//Check Tuning process if finish
	do {
		mdelay(50);
		bcm4329_read(I2C_FM_RDS_FLAG0,  flag_status,  sizeof(flag_status));
		count++;
		PINFO("I2C_FM_RDS_FLAG0 :flag_status[0] = %x, flag_status[1] = %x count = %d\n", flag_status[0], flag_status[1],count);
	}while( ((0 == (flag_status[0] & 0x01) ) || (1 == (flag_status[0] & 0x02) )) && (count < BCOMFM_MAX_TUNING_TRIALS));

	bcm4329_q_event(radio, BCM4329_EVT_SEEK_COMPLETE);
  return 0;
}

static int vidioc_d_qbuf(struct file *file, void *priv,
			struct v4l2_buffer *buffer)
{
					
	struct bcm4329_device  *radio = video_get_drvdata(video_devdata(file));
	enum bcm4329_buf_t buf_type = buffer->index;
	struct kfifo *data_fifo;
	unsigned char *buf = (unsigned char *)buffer->m.userptr;
	unsigned int len = buffer->length;

	PINFO("vidioc_d_qbuf");
	/* check if we can access the user buffer */
	if (!access_ok(VERIFY_WRITE, buf, len))
		{
			PINFO("can't access!!!");
			return -EFAULT;
		}
	
	data_fifo = &radio->data_buf[buf_type];
	
	if (wait_event_interruptible(radio->event_queue,
	kfifo_len(data_fifo)) < 0) {
		PINFO("wait_event_interruptible fail!!!");
		return -EINTR;
								}
						
    
	buffer->bytesused = kfifo_out_locked(data_fifo, buf, len, 
					&radio->buf_lock[buf_type]);
    
	PINFO("buffer->bytesused= %d",buffer->bytesused);
					
		return 0;
}

/* Snail Qian 20110411, change to follow V4L2 interface {*/
static int vidioc_g_fmt_type_private(struct file *file, void *priv,
						struct v4l2_format *f)
{
	return 0;

}
/* Snail Qian 20110411, change to follow V4L2 interface }*/

static int bcm4329_open(struct file *file)
{
	struct bcm4329_device *radio = video_drvdata(file);

	PINFO("bcm4329_open");
	
	mutex_lock(&radio->mutex);
	/*now we only support one client*/
	if (radio->user > 0)
		return -ENODEV;		

	radio->user++;
	mutex_unlock(&radio->mutex);
	
	/* initialize and power off the chip */	
	radio->hdev = hci_dev_get(0);
	
	bcm4329_set_audout_mode(radio, V4L2_TUNER_MODE_STEREO);
	
	bcm4329_mute(radio, 1);	
	
	
	return 0;
}

static int bcm4329_close(struct file *file)
{
	struct bcm4329_device *radio = video_drvdata(file);
	
	PINFO("bcm4329_close");

	if (!radio)
		return -ENODEV;
	mutex_lock(&radio->mutex);
	radio->user--;
	mutex_unlock(&radio->mutex);

	return 0;
}

#ifdef FM_RDS
static ssize_t bcm4329_fop_read(struct file *file, char __user *buf,
				size_t count, loff_t *ppos)
{
	struct bcm4329_device *radio = video_get_drvdata(video_devdata(file));
	struct kfifo *rds_buf = &radio->data_buf[BCM4329_BUF_RAW_RDS];

	/* block if no new data available */
	while (!kfifo_len(rds_buf)) {
		if (file->f_flags & O_NONBLOCK)
			return -EWOULDBLOCK;
		if (wait_event_interruptible(radio->read_queue,
			kfifo_len(rds_buf)) < 0)
			return -EINTR;
	}

	count /= 3;

	/* check if we can write to the user buffer */
	if (!access_ok(VERIFY_WRITE, buf, count*3))
		return -EFAULT;

	/* copy RDS block out of internal buffer and to user buffer */
	return kfifo_out_locked(rds_buf, buf, count*3, 
				&radio->buf_lock[BCM4329_BUF_RAW_RDS]);
}

#endif

/* File system interface */
static const struct v4l2_file_operations bcm4329_fops = {
   .owner	    = THIS_MODULE,
   .open      = bcm4329_open,
   .release   = bcm4329_close,
   .ioctl	    = video_ioctl2,
   #ifdef FM_RDS
   .read	=   bcm4329_fop_read,
   #endif
};

static const struct v4l2_ioctl_ops bcm4329_ioctl_ops = {
	.vidioc_querycap    = vidioc_querycap,
	.vidioc_g_tuner     = vidioc_g_tuner,
	.vidioc_s_tuner     = vidioc_s_tuner,
	.vidioc_g_audio     = vidioc_g_audio,
	.vidioc_s_audio     = vidioc_s_audio,
	.vidioc_g_frequency = vidioc_g_frequency,  //get freq
	.vidioc_s_frequency = vidioc_s_frequency,  //set freq
	.vidioc_s_hw_freq_seek    = vidioc_s_hw_freq_seek, //scan auto
	.vidioc_queryctrl   = vidioc_queryctrl,
	.vidioc_g_ctrl      = vidioc_g_ctrl,
	.vidioc_s_ctrl      = vidioc_s_ctrl,
	.vidioc_dqbuf    = vidioc_d_qbuf,
	/* Snail Qian 20110411, change to follow V4L2 interface {*/
	.vidioc_g_fmt_type_private  = vidioc_g_fmt_type_private,
	/* Snail Qian 20110411, change to follow V4L2 interface }*/
};

/* V4L2 interface */
static struct video_device bcm4329_radio_template = {
	.name		     = "bcm4329",
	.fops        = &bcm4329_fops,
	.ioctl_ops 	 = &bcm4329_ioctl_ops,
	.release	   = video_device_release,
};   

static int bcm4329_probe(struct platform_device *pdev)
{
  struct bcm4329_device *radio;
	int ret=0;
	unsigned int buf_size;
	int i;

	PINFO("bcm4329_probe");
	radio = kmalloc(sizeof(struct bcm4329_device), GFP_KERNEL);
	if (!radio)
		return -ENOMEM;

	mutex_init(&radio->mutex);	
	
	mutex_init(&ioctl_mutex);
	
	init_completion(&radio->hci_confirmed);
	
	init_waitqueue_head(&radio->event_queue);
	
	radio->user = 0;

	radio->videodev = video_device_alloc();
	if (!(radio->videodev)) {
		ret = -ENOMEM;
		goto errfr;
	}
	memcpy(radio->videodev, &bcm4329_radio_template,
		sizeof(bcm4329_radio_template));

		/*allocate internal buffers for decoded rds and event buffer*/
	buf_size = 64;
	for (i = 0; i < BCM4329_BUF_MAX; i++) 
		{
			int kfifo_alloc_rc=0;
			spin_lock_init(&radio->buf_lock[i]);
	
			if (i == BCM4329_BUF_RAW_RDS)
				kfifo_alloc_rc = kfifo_alloc(&radio->data_buf[i], 
					rds_buf*3, GFP_KERNEL);
			else
				kfifo_alloc_rc = kfifo_alloc(&radio->data_buf[i], 
					buf_size, GFP_KERNEL);
	
			if (kfifo_alloc_rc!=0) {
				printk(KERN_ERR "%s: failed allocating buffers %d\n",
					__func__, kfifo_alloc_rc);
				goto err_bufs;
			}
		}

	video_set_drvdata(radio->videodev, radio);

	ret = video_register_device(radio->videodev, VFL_TYPE_RADIO, radio_nr);
	if (ret < 0) {
		PWARN("Could not register video device!");
		goto errrel;
	}
	else{
	  PWARN("bcm4329: register video device!");
	}
	bcm4329_device_data = radio;	
	
	return 0;
	   
errrel:
	video_device_release(radio->videodev);
	
err_bufs:
	for (; i > -1; i--)
		kfifo_free(&radio->data_buf[i]);
errfr:
	kfree(radio);
	return ret;
}

static int __devexit bcm4329_remove(struct platform_device *pdev)
{
  video_unregister_device(&bcm4329_radio_template);
  
  if(bcm4329_device_data != NULL){
    kfree(bcm4329_device_data);
  }
  
  bcm4329_device_data = NULL;
  
  return 0;
}

static int bcm4329_suspend(struct platform_device *pdev, pm_message_t state)
{
  return 0;
}

static int bcm4329_resume(struct platform_device *pdev)
{
  return 0;
}


static struct platform_driver bcm4329_driver = {
  .probe    = bcm4329_probe,
  .remove   = bcm4329_remove,
  .suspend  = bcm4329_suspend,
  .resume   = bcm4329_resume,
  .driver   = {
   .name    = "bcm4329_device",
   .owner   = THIS_MODULE,
  },
};

static int __init bcm4329_init(void)
{
  int ret;
  
  PINFO("bcm4329_init");
  ret = platform_driver_register(&bcm4329_driver);

  return ret;
}
  
/* cleanup the driver */
static void __exit bcm4329_exit(void)
{
  PINFO("bcm4329_exit");
  platform_driver_unregister(&bcm4329_driver);		
}


int bcm4329_receive_evt(struct hci_dev *hdev, struct sk_buff *skb)
{
	struct bcom_fm_event_pack *rp = (void *) skb->data;
	
	PINFO("bcm4329_receive_evt================");	
	PINFO("status :%x", rp->status);
	PINFO("i2c cmd :%x", rp->i2caddr);
	PINFO("i2c read or write :%d", rp->r_w);

	//just an  architecture ,will update. 
	//point is :how to pass the value data to where the sent cmd funtion is called.
	//and to fit different payload in read completed event, better to define more structures in hci.h or other head file
	
	if((rp->r_w == 0) && (rp->i2caddr == bcm4329_device_data->i2c_addr)){ 
	  bcm4329_device_data->hci_result = rp->status;
	  bcm4329_device_data->i2c_addr = 0;
	  complete(&bcm4329_device_data->hci_confirmed);
	  PINFO("bcm4329 hci_confirmed  \n");
	}
	else if((rp->r_w == 1) && (rp->i2caddr == bcm4329_device_data->i2c_addr)){ 
	  bcm4329_device_data->hci_result = rp->status;
	  bcm4329_device_data->i2c_addr = 0;
	  
	  if(rp->data)
	  {
	    memcpy(bcm4329_device_data->read_buf, rp->data, bcm4329_device_data->read_len);
	  }	  
	  
	  complete(&bcm4329_device_data->hci_confirmed);
	  PINFO("bcm4329 hci_confirmed  \n");
	}
	else{
	  PINFO("bcm4329 receive a wrong response \n");
	}
	
	return 0;
}

EXPORT_SYMBOL(bcm4329_receive_evt);


#ifdef FM_RDS

int current_group;
struct kfifo *rds_buf_parse ;
static void rds_parse_data(uint8_t id,uint16_t data)
{
	struct bcm4329_device *radio= bcm4329_device_data;
	switch(id)
	{
		case 0x00:  //block A : PI info
			{
				PINFO("Block A \n");
				rds_buf_parse= &bcm4329_device_data->data_buf[BCM4329_BUF_RAW_RDS];
				kfifo_in_locked(rds_buf_parse, &data, 3, &bcm4329_device_data->buf_lock[BCM4329_BUF_RAW_RDS]);
				bcm4329_q_event(radio, BCM4329_EVT_NEW_RAW_RDS);
				break;
			}
		
		case 0x01: //blcok B :need parse group
			{
				current_group =(data&0xf800)>>11 ;
				switch (current_group)
				{
					case 0x00:
					case 0x01:  //group 0 :basic info, get AF
					{
						data=data&0x1f;
						rds_buf_parse= &bcm4329_device_data->data_buf[BCM4329_BUF_AF_LIST];
						kfifo_in_locked(rds_buf_parse, &data, 3, &bcm4329_device_data->buf_lock[BCM4329_BUF_AF_LIST]);
						bcm4329_q_event(radio, BCM4329_EVT_NEW_AF_LIST);
						break;
					}
					default: break;
				}
			}
			break;
		case 0x02: //block C : parse follow group define
			{
				  if (current_group==0x00||current_group==0x01)
				  {
					rds_buf_parse= & bcm4329_device_data->data_buf[BCM4329_BUF_PS_RDS];
					kfifo_in_locked(rds_buf_parse, &data, 3, &bcm4329_device_data->buf_lock[BCM4329_BUF_PS_RDS]);
					bcm4329_q_event(radio, BCM4329_EVT_NEW_PS_RDS);
				   }
				  else if ((current_group==0x20||current_group==0x21))
				  {
					rds_buf_parse= &bcm4329_device_data->data_buf[BCM4329_BUF_RT_RDS];
					kfifo_in_locked(rds_buf_parse, &data, 3, &bcm4329_device_data->buf_lock[BCM4329_BUF_PS_RDS]);
					bcm4329_q_event(radio, BCM4329_EVT_NEW_RT_RDS);
				  }
			}
			break;
					
		case 0x03: //block D: 			
			{
				 if ((current_group==0x20||current_group==0x21))
				  {
					rds_buf_parse= & bcm4329_device_data->data_buf[BCM4329_BUF_RT_RDS];
					kfifo_in_locked(rds_buf_parse, &data, 3, &bcm4329_device_data->buf_lock[BCM4329_BUF_PS_RDS]);
					bcm4329_q_event(radio, BCM4329_EVT_NEW_RT_RDS);
				  }
			}
			break;
	    default:
				rds_buf_parse= & bcm4329_device_data->data_buf[BCM4329_BUF_RAW_RDS];
				kfifo_in_locked(rds_buf_parse, &data, 3, &bcm4329_device_data->buf_lock[BCM4329_BUF_RAW_RDS]);
				bcm4329_q_event(radio, BCM4329_EVT_NEW_RAW_RDS);
				break;
		}
			
}

static void rds_timer( unsigned long data)
{
	uint8_t rds_data[3] = {0};		
	bcom_fm_rds_parameters rds_parsed_data;
	struct kfifo *rds_buf = &bcm4329_device_data->data_buf[BCM4329_BUF_RAW_RDS];//MOVE
	uint16_t i = 0;

	memset(&rds_parsed_data, 0, sizeof(bcom_fm_rds_parameters));		
	for (i = 0; i <RDS_WATER_LEVEL ; i++)
	{
		bcm4329_read(I2C_RDS_DATA,rds_data,sizeof(rds_data));
		rds_parsed_data.rds_block_id[i] = ( (rds_data[0] & 0xf0)>>4);
		rds_parsed_data.rds_block_errors[i] = ((rds_data[0] & 0x0c)>>2);
		rds_parsed_data.rds_data_block[i] = ((rds_data[1] & 0xff)<<8) | (rds_data[2]& 0xff);
		rds_parse_data(rds_parsed_data.rds_block_id[i],rds_parsed_data.rds_data_block[i]);
	}

	/* copy RDS block to internal buffer */
	//kfifo_in_locked(rds_buf, rds_data, 3, &bcm4329_device_data->buf_lock[BCM4329_BUF_RAW_RDS]);
	
	/* wake up read queue */
	if (kfifo_len(rds_buf))
		wake_up_interruptible(&bcm4329_device_data->read_queue);
	
	mod_timer(&fm_rds, jiffies + msecs_to_jiffies(5000));
}

#endif


MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL");

module_init(bcm4329_init);
module_exit(bcm4329_exit);

