
/* Copyright (c) 2009-2010, Code Aurora Forum. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 */
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/leds-pmic8058.h>
#include <linux/pwm.h>
#include <linux/pmic8058-pwm.h>
#include <linux/hrtimer.h>
#include <mach/pmic.h>
#include <mach/camera.h>
#include <mach/gpio.h>
#include <linux/i2c.h>
#include <linux/slab.h>

struct timer_list timer_flash;

enum msm_cam_flash_stat{
	MSM_CAM_FLASH_OFF,
	MSM_CAM_FLASH_ON,
};


struct qsd_flash_work_t {
	struct work_struct work;
};

static struct qsd_flash_work_t *qsd_flash_w;
static struct i2c_client *qsd_flash_client;
static DECLARE_WAIT_QUEUE_HEAD(qsd_flash_wait_queue);


#define PM8058_GPIO_BASE			NR_MSM_GPIOS
#define PM8058_GPIO_PM_TO_SYS(pm_gpio)		(pm_gpio + PM8058_GPIO_BASE)
#define PM8058_GPIO_SYS_TO_PM(sys_gpio)		(sys_gpio - PM8058_GPIO_BASE)

#if 1
static int qsd_flash_i2c_rxdata(unsigned short saddr,
	unsigned char *rxdata, int length)
{
	struct i2c_msg msgs[] = {
		{
			.addr  = saddr,
			.flags = 0,
			.len   = 1,
			.buf   = rxdata,
		},
		{
			.addr  = saddr,
			.flags = I2C_M_RD,
			.len   = 1,
			.buf   = rxdata,
		},
	};
	if (i2c_transfer(qsd_flash_client->adapter, msgs, 2) < 0) {
		CDBG("qsd_flash_i2c_rxdata failed!\n");
		return -EIO;
	}
	return 0;
}
static int32_t qsd_flash_i2c_txdata(unsigned short saddr,
				unsigned char *txdata, int length)
{
	struct i2c_msg msg[] = {
		{
			.addr = saddr,
			.flags = 0,
			.len = length,
			.buf = txdata,
		 },
	};
	if (i2c_transfer(qsd_flash_client->adapter, msg, 1) < 0) {
		CDBG("qsd_flash_i2c_txdata faild 0x%x\n", qsd_flash_client->addr);
		return -EIO;
	}

	return 0;
}

static int32_t qsd_flash_i2c_read(uint8_t raddr,
	unsigned short *rdata, int rlen)
{
	int32_t rc = 0;
	unsigned char buf[2];
	if (!rdata)
		return -EIO;
	memset(buf, 0, sizeof(buf));
	buf[0] = raddr;
	
	

	rc = qsd_flash_i2c_rxdata(qsd_flash_client->addr, buf, rlen);
	if (rc < 0) {
		printk("qsd_flash_i2c_read 0x%x failed!\n", raddr);
		return rc;
	}
	*rdata = buf[0];
	
	
	return rc;
}


static int32_t qsd_flash_i2c_write_b_sensor(uint8_t waddr, uint8_t bdata)
{
	int32_t rc = -EFAULT;
	unsigned char buf[3];
	memset(buf, 0, sizeof(buf));
	
	buf[0] = waddr;
	buf[1] = bdata;

	


	

	rc = qsd_flash_i2c_txdata(qsd_flash_client->addr, buf, 2);
	if (rc < 0) 
		printk("qsd_flash_i2c_write_b failed, addr = 0x%x, val = 0x%x!\n",
			waddr, bdata);
	
	

	return rc;
}

static int qsd_flash_adp_1650_read_id(uint16_t *flash_id)
{
	int rc = 0;
	uint16_t chipid = 0;

	rc = qsd_flash_i2c_read(0x00, &chipid, 1);
	*flash_id = chipid;
	
	return rc;
}

static int qsd_flash_adp_1650(unsigned char led_state)
{
	int rc = 0;
	uint16_t chipid = 0;

	if((MSM_CAMERA_LED_LOW == led_state) || (MSM_CAMERA_LED_HIGH == led_state))
	{
		printk("qsd_flash_adp_1650===>MSM_CAMERA_LED_ON\n");

	
	

		rc = qsd_flash_i2c_read(0x00, &chipid, 1);
		printk("qsd_flash_adp_1650=>ID(0x00): %x\n", chipid);
	        if((rc < 0) || (chipid != 0x22))
		       return -1;

		qsd_flash_i2c_write_b_sensor(0x02,0xAF);	
		
		rc = qsd_flash_i2c_read(0x02, &chipid, 1);
		printk("qsd_flash_adp_1650=>ID(0x02): %x\n", chipid);

		qsd_flash_i2c_write_b_sensor(0x03,0x4D);	
		
		rc = qsd_flash_i2c_read(0x03, &chipid, 1);
		printk("qsd_flash_adp_1650=>ID(0x03): %x\n", chipid);

		qsd_flash_i2c_write_b_sensor(0x04,0x2B);	

		rc = qsd_flash_i2c_read(0x04, &chipid, 1);
		printk("qsd_flash_adp_1650=>ID(0x04): %x\n", chipid);

		#if 0   

		qsd_flash_i2c_write_b_sensor(0x04,0x2A);	

		rc = qsd_flash_i2c_read(0x04, &chipid, 1);
		printk("qsd_flash_adp_1650=>ID(0x04): %x\n", chipid);
		#endif

		rc = qsd_flash_i2c_read(0x05, &chipid, 1);
		printk("qsd_flash_adp_1650=>ID(0x05): %x\n", chipid);

		if((rc < 0) || (chipid != 0x00))
			return -1;



	}
    
    else if(MSM_CAMERA_LED_VIDEO_TORCH == led_state)
	{
		printk("qsd_flash_adp_1650===>MSM_CAMERA_LED_ON\n");

	
		rc = qsd_flash_i2c_read(0x00, &chipid, 1);
		printk("qsd_flash_adp_1650=>ID(0x00): %x\n", chipid);
	        if((rc < 0) || (chipid != 0x22))
		       return -1;

		qsd_flash_i2c_write_b_sensor(0x02,0xAF);	
		
		rc = qsd_flash_i2c_read(0x02, &chipid, 1);
		printk("qsd_flash_adp_1650=>ID(0x02): %x\n", chipid);

		qsd_flash_i2c_write_b_sensor(0x03,0x4A);	
		
		rc = qsd_flash_i2c_read(0x03, &chipid, 1);
		printk("qsd_flash_adp_1650=>ID(0x03): %x\n", chipid);

		qsd_flash_i2c_write_b_sensor(0x04,0x2A);	

		rc = qsd_flash_i2c_read(0x04, &chipid, 1);
		printk("qsd_flash_adp_1650=>ID(0x04): %x\n", chipid);

		rc = qsd_flash_i2c_read(0x05, &chipid, 1);
		printk("qsd_flash_adp_1650=>ID(0x05): %x\n", chipid);

		if((rc < 0) || (chipid != 0x00))
			return -1;



	}  
    
	else
	{
		printk("qsd_flash_adp_1650===>MSM_CAMERA_LED_OFF\n");
		
	#if 0	
		qsd_flash_i2c_write_b_sensor(0x04,0x26);	
	#endif
		qsd_flash_i2c_write_b_sensor(0x04,0x23);	
	
		rc = qsd_flash_i2c_read(0x04, &chipid, 1);
		printk("qsd_flash_adp_1650=>ID(0x04): %x\n", chipid);
		
		qsd_flash_i2c_write_b_sensor(0x04,0x20);	
	
		rc = qsd_flash_i2c_read(0x04, &chipid, 1);
		printk("qsd_flash_adp_1650=>ID(0x04): %x\n", chipid);
		
		rc = qsd_flash_i2c_read(0x05, &chipid, 1);
		printk("qsd_flash_adp_1650=>ID(0x05): %x\n", chipid);

		if((rc < 0) || (chipid != 0x00))
			return -1;
	}

	return rc;
}


static int qsd_flash_5v (unsigned char led_state)
{
	int rc = 0;
	int PM_PWM_BANK = 7;
	int PM_PWM_ID = PM_PWM_LED_FLASH1;
	int PM_PWM_MODE = PM_PWM_CONF_PWM1;

	static struct pwm_device *flash_5v_pwm;

	if (!flash_5v_pwm)
	{
		
		flash_5v_pwm = pwm_request (PM_PWM_BANK, "camera-flash");
		if (flash_5v_pwm == NULL || IS_ERR (flash_5v_pwm))
		{
			pr_err ("%s: FAIL pwm_request(): flash_5v_pwm=%p\n",
				__func__, flash_5v_pwm);
			flash_5v_pwm = NULL;
			return -ENXIO;
		}
	}

	if ((MSM_CAMERA_LED_LOW == led_state)
	    || (MSM_CAMERA_LED_HIGH == led_state))
	{


		pm8058_pwm_config_led (flash_5v_pwm, PM_PWM_ID, PM_PWM_MODE,
				       300);
    
		rc = pwm_config (flash_5v_pwm, 999, 1000);
	
		if (rc >= 0)
			rc = pwm_enable (flash_5v_pwm);



	}
	else if (MSM_CAMERA_LED_VIDEO_TORCH == led_state)
	{

		pm8058_pwm_config_led (flash_5v_pwm, PM_PWM_ID, PM_PWM_MODE,
				       60);
    
		rc = pwm_config (flash_5v_pwm, 999, 1000);

		if (rc >= 0)
			rc = pwm_enable (flash_5v_pwm);


	}
	else
	{


		pwm_disable (flash_5v_pwm);

	}

	return rc;
}

#endif
static int qsd_flash_init_client(struct i2c_client *client)
{
	
	init_waitqueue_head(&qsd_flash_wait_queue);
	return 0;
}

static const struct i2c_device_id qsd_flash_i2c_id[] = {
  { "qsd_flash_i2c", 0 },
  { }
};

static int qsd_flash_i2c_probe(struct i2c_client *client,
	const struct i2c_device_id *id)
{
	int rc = 0;
	
	printk("qsd_flash_i2c_probe called!\n");

	if (!i2c_check_functionality(client->adapter, I2C_FUNC_I2C)) {
		printk("i2c_check_functionality failed\n");
		goto probe_failure;
	}

	qsd_flash_w = kzalloc(sizeof(struct qsd_flash_work_t), GFP_KERNEL);
	if (!qsd_flash_w) {
		printk("kzalloc failed.\n");
		rc = -ENOMEM;
		goto probe_failure;
	}

	i2c_set_clientdata(client, qsd_flash_w);
	qsd_flash_init_client(client);
	qsd_flash_client = client;

#if 0
<<<<<<< HEAD
	mdelay(50);
=======
#endif


	printk("qsd_flash_probe successed! rc = %d\n", rc);
	return 0;

probe_failure:
	printk("qsd_flash_probe failed! rc = %d\n", rc);
	return rc;
}

static struct i2c_driver qsd_flash_i2c_driver = {
  .driver = {
    .owner = THIS_MODULE,
    .name = "qsd_flash_i2c"
  },
  .id_table = qsd_flash_i2c_id,
  .probe    = qsd_flash_i2c_probe,
  
  
  
  
};

static int config_flash_gpio_table(enum msm_cam_flash_stat stat,
			struct msm_camera_sensor_strobe_flash_data *sfdata)
{
	int rc = 0, i = 0;
	int msm_cam_flash_gpio_tbl[][2] = {
		{sfdata->flash_trigger, 1},
		{sfdata->flash_charge, 1},
		{sfdata->flash_charge_done, 0}
	};

	if (stat == MSM_CAM_FLASH_ON) {
		for (i = 0; i < ARRAY_SIZE(msm_cam_flash_gpio_tbl); i++) {
			rc = gpio_request(msm_cam_flash_gpio_tbl[i][0],
							  "CAM_FLASH_GPIO");
			if (unlikely(rc < 0)) {
				pr_err("%s not able to get gpio\n", __func__);
				for (i--; i >= 0; i--)
					gpio_free(msm_cam_flash_gpio_tbl[i][0]);
				break;
			}
			if (msm_cam_flash_gpio_tbl[i][1])
				gpio_direction_output(
					msm_cam_flash_gpio_tbl[i][0], 0);
			else
				gpio_direction_input(
					msm_cam_flash_gpio_tbl[i][0]);
		}
	} else {
		for (i = 0; i < ARRAY_SIZE(msm_cam_flash_gpio_tbl); i++) {
			gpio_direction_input(msm_cam_flash_gpio_tbl[i][0]);
			gpio_free(msm_cam_flash_gpio_tbl[i][0]);
		}
	}
	return rc;
}

int msm_camera_flash_current_driver(
	struct msm_camera_sensor_flash_current_driver *current_driver,
	unsigned led_state)
{
	int rc = 0;
#if defined CONFIG_LEDS_PMIC8058
	int idx;
	const struct pmic8058_leds_platform_data *driver_channel =
		current_driver->driver_channel;
	int num_leds = driver_channel->num_leds;

	CDBG("%s: led_state = %d\n", __func__, led_state);

	/* Evenly distribute current across all channels */
	switch (led_state) {
	case MSM_CAMERA_LED_OFF:
		for (idx = 0; idx < num_leds; ++idx) {
			rc = pm8058_set_led_current(
				driver_channel->leds[idx].id, 0);
			if (rc < 0)
				pr_err(
					"%s: FAIL name = %s, rc = %d\n",
					__func__,
					driver_channel->leds[idx].name,
					rc);
		}
		break;

	case MSM_CAMERA_LED_LOW:
		for (idx = 0; idx < num_leds; ++idx) {
			rc = pm8058_set_led_current(
				driver_channel->leds[idx].id,
				current_driver->low_current/num_leds);
			if (rc < 0)
				pr_err(
					"%s: FAIL name = %s, rc = %d\n",
					__func__,
					driver_channel->leds[idx].name,
					rc);
		}
		break;

	case MSM_CAMERA_LED_HIGH:
		for (idx = 0; idx < num_leds; ++idx) {
			rc = pm8058_set_led_current(
				driver_channel->leds[idx].id,
				current_driver->high_current/num_leds);
			if (rc < 0)
				pr_err(
					"%s: FAIL name = %s, rc = %d\n",
					__func__,
					driver_channel->leds[idx].name,
					rc);
		}
		break;

	default:
		rc = -EFAULT;
		break;
	}
#endif /* CONFIG_LEDS_PMIC8058 */

	CDBG("msm_camera_flash_led_pmic8058: return %d\n", rc);

	return rc;
}


static int msm_camera_flash_pwm(
	struct msm_camera_sensor_flash_pwm *pwm,
	unsigned led_state)
{
	int rc = 0;
	int PWM_PERIOD = USEC_PER_SEC / pwm->freq;

	static struct pwm_device *flash_pwm;

	if (!flash_pwm) {
		flash_pwm = pwm_request(pwm->channel, "camera-flash");
		if (flash_pwm == NULL || IS_ERR(flash_pwm)) {
			pr_err("%s: FAIL pwm_request(): flash_pwm=%p\n",
			       __func__, flash_pwm);
			flash_pwm = NULL;
			return -ENXIO;
		}
	}

	switch (led_state) {
	case MSM_CAMERA_LED_LOW:
		rc = pwm_config(flash_pwm,
			(PWM_PERIOD/pwm->max_load)*pwm->low_load,
			PWM_PERIOD);
		if (rc >= 0)
			rc = pwm_enable(flash_pwm);
		break;

	case MSM_CAMERA_LED_HIGH:
		rc = pwm_config(flash_pwm,
			(PWM_PERIOD/pwm->max_load)*pwm->high_load,
			PWM_PERIOD);
		if (rc >= 0)
			rc = pwm_enable(flash_pwm);
		break;

	case MSM_CAMERA_LED_OFF:
		pwm_disable(flash_pwm);
		break;

	default:
		rc = -EFAULT;
		break;
	}

	return rc;
}


int msm_camera_flash_i2c_driver(unsigned led_state)
{
	int rc = 0;


	
	printk("msm_camera_flash_i2c_driver: led_state:(%d)\n", led_state);


	switch (led_state) {
	case MSM_CAMERA_LED_OFF:
		rc = qsd_flash_adp_1650(MSM_CAMERA_LED_OFF);
		break;

	case MSM_CAMERA_LED_LOW:
		rc = qsd_flash_adp_1650(MSM_CAMERA_LED_LOW);
		break;

	case MSM_CAMERA_LED_HIGH:
		rc = qsd_flash_adp_1650(MSM_CAMERA_LED_HIGH);
		break;

	case MSM_CAMERA_LED_CHIP_ID:
		rc = qsd_flash_adp_1650(MSM_CAMERA_LED_CHIP_ID);
		break;


	case MSM_CAMERA_LED_VIDEO_TORCH:
		rc = qsd_flash_adp_1650(MSM_CAMERA_LED_VIDEO_TORCH);
		break;        


	default:
		rc = -EFAULT;
		break;
	}
	return rc;
}


int msm_camera_flash_5v_driver (unsigned led_state)
{
	int rc = 0;

	printk ("msm_camera_flash_i2c_driver: led_state:(%d)\n",
		led_state);

	switch (led_state)
	{
	case MSM_CAMERA_LED_OFF:
		rc = qsd_flash_5v (MSM_CAMERA_LED_OFF);
		break;

	case MSM_CAMERA_LED_LOW:
		rc = qsd_flash_5v (MSM_CAMERA_LED_LOW);
		break;

	case MSM_CAMERA_LED_HIGH:
		rc = qsd_flash_5v (MSM_CAMERA_LED_HIGH);
		break;

	case MSM_CAMERA_LED_CHIP_ID:
		rc = qsd_flash_5v (MSM_CAMERA_LED_CHIP_ID);
		break;

	case MSM_CAMERA_LED_VIDEO_TORCH:
		rc = qsd_flash_5v (MSM_CAMERA_LED_VIDEO_TORCH);
		break;

	default:
		rc = -EFAULT;
		break;
	}
	return rc;
}


int msm_camera_flash_pmic(
	struct msm_camera_sensor_flash_pmic *pmic,
	unsigned led_state)
{
	int rc = 0;

	switch (led_state) {
	case MSM_CAMERA_LED_OFF:
		rc = pmic->pmic_set_current(pmic->led_src_1, 0);
		if (pmic->num_of_src > 1)
			rc = pmic->pmic_set_current(pmic->led_src_2, 0);
		break;

	case MSM_CAMERA_LED_LOW:
		rc = pmic->pmic_set_current(pmic->led_src_1,
				pmic->low_current);
		if (pmic->num_of_src > 1)
			rc = pmic->pmic_set_current(pmic->led_src_2, 0);
		break;

	case MSM_CAMERA_LED_HIGH:
		rc = pmic->pmic_set_current(pmic->led_src_1,
			pmic->high_current);
		if (pmic->num_of_src > 1)
			rc = pmic->pmic_set_current(pmic->led_src_2,
				pmic->high_current);
		break;

	default:
		rc = -EFAULT;
		break;
	}
	CDBG("flash_set_led_state: return %d\n", rc);

	return rc;
}

int32_t msm_camera_flash_read_led_id(uint16_t *flashid)
{
	uint16_t flash_id;
	int32_t rc = 0;
	
	rc = qsd_flash_adp_1650_read_id(&flash_id);
	*flashid = flash_id;
	
	

	return rc;
}

int32_t msm_camera_flash_set_led_state(
	struct msm_camera_sensor_flash_data *fdata, unsigned led_state)
{
	int32_t rc;

	CDBG("flash_set_led_state: %d flash_sr_type=%d\n", led_state,
	    fdata->flash_src->flash_sr_type);
	printk("flash_set_led_state: %d flash_sr_type=%d\n", led_state,
	    fdata->flash_src->flash_sr_type);

	if (fdata->flash_type != MSM_CAMERA_FLASH_LED)
		return -ENODEV;

	switch (fdata->flash_src->flash_sr_type) {
	case MSM_CAMERA_FLASH_SRC_PMIC:
		rc = msm_camera_flash_pmic(&fdata->flash_src->_fsrc.pmic_src,
			led_state);
		break;

	case MSM_CAMERA_FLASH_SRC_PWM:
		rc = msm_camera_flash_pwm(&fdata->flash_src->_fsrc.pwm_src,
			led_state);
		break;

	case MSM_CAMERA_FLASH_SRC_CURRENT_DRIVER:
		rc = msm_camera_flash_current_driver(
			&fdata->flash_src->_fsrc.current_driver_src,
			led_state);
		break;
	
	case MSM_CAMERA_FLASH_SRC_I2C_DRIVER:
		rc = msm_camera_flash_i2c_driver(led_state);
		break;


	case MSM_CAMERA_FLASH_SRC_5V_DRIVER:
		rc = msm_camera_flash_5v_driver (led_state);
		break;

        
	default:
		rc = -ENODEV;
		break;
	}

	return rc;
}

static int msm_strobe_flash_xenon_charge(int32_t flash_charge,
		int32_t charge_enable, uint32_t flash_recharge_duration)
{
	gpio_set_value_cansleep(flash_charge, charge_enable);
	if (charge_enable) {
		timer_flash.expires = jiffies +
			msecs_to_jiffies(flash_recharge_duration);
		/* add timer for the recharge */
		if (!timer_pending(&timer_flash))
			add_timer(&timer_flash);
	} else
		del_timer_sync(&timer_flash);
	return 0;
}

static void strobe_flash_xenon_recharge_handler(unsigned long data)
{
	unsigned long flags;
	struct msm_camera_sensor_strobe_flash_data *sfdata =
		(struct msm_camera_sensor_strobe_flash_data *)data;

	spin_lock_irqsave(&sfdata->timer_lock, flags);
	msm_strobe_flash_xenon_charge(sfdata->flash_charge, 1,
		sfdata->flash_recharge_duration);
	spin_unlock_irqrestore(&sfdata->timer_lock, flags);

	return;
}

static irqreturn_t strobe_flash_charge_ready_irq(int irq_num, void *data)
{
	struct msm_camera_sensor_strobe_flash_data *sfdata =
		(struct msm_camera_sensor_strobe_flash_data *)data;

	/* put the charge signal to low */
	gpio_set_value_cansleep(sfdata->flash_charge, 0);

	return IRQ_HANDLED;
}

static int msm_strobe_flash_xenon_init(
	struct msm_camera_sensor_strobe_flash_data *sfdata)
{
	unsigned long flags;
	int rc = 0;

	spin_lock_irqsave(&sfdata->spin_lock, flags);
	if (!sfdata->state) {

		rc = config_flash_gpio_table(MSM_CAM_FLASH_ON, sfdata);
		if (rc < 0) {
			pr_err("%s: gpio_request failed\n", __func__);
			goto go_out;
		}
		rc = request_irq(sfdata->irq, strobe_flash_charge_ready_irq,
			IRQF_TRIGGER_RISING, "charge_ready", sfdata);
		if (rc < 0) {
			pr_err("%s: request_irq failed %d\n", __func__, rc);
			goto go_out;
		}

		spin_lock_init(&sfdata->timer_lock);
		/* setup timer */
		init_timer(&timer_flash);
		timer_flash.function = strobe_flash_xenon_recharge_handler;
		timer_flash.data = (unsigned long)sfdata;
	}
	sfdata->state++;
go_out:
	spin_unlock_irqrestore(&sfdata->spin_lock, flags);

	return rc;
}

static int msm_strobe_flash_xenon_release
(struct msm_camera_sensor_strobe_flash_data *sfdata, int32_t final_release)
{
	unsigned long flags;

	spin_lock_irqsave(&sfdata->spin_lock, flags);
	if (sfdata->state > 0) {
		if (final_release)
			sfdata->state = 0;
		else
			sfdata->state--;

		if (!sfdata->state) {
			free_irq(sfdata->irq, sfdata);
			config_flash_gpio_table(MSM_CAM_FLASH_OFF, sfdata);
			if (timer_pending(&timer_flash))
				del_timer_sync(&timer_flash);
		}
	}
	spin_unlock_irqrestore(&sfdata->spin_lock, flags);
	return 0;
}

static void msm_strobe_flash_xenon_fn_init
	(struct msm_strobe_flash_ctrl *strobe_flash_ptr)
{
	strobe_flash_ptr->strobe_flash_init =
				msm_strobe_flash_xenon_init;
	strobe_flash_ptr->strobe_flash_charge =
				msm_strobe_flash_xenon_charge;
	strobe_flash_ptr->strobe_flash_release =
				msm_strobe_flash_xenon_release;
}

int msm_strobe_flash_init(struct msm_sync *sync, uint32_t sftype)
{
	int rc = 0;
	switch (sftype) {
	case MSM_CAMERA_STROBE_FLASH_XENON:
		if (sync->sdata->strobe_flash_data) {
			msm_strobe_flash_xenon_fn_init(&sync->sfctrl);
			rc = sync->sfctrl.strobe_flash_init(
			sync->sdata->strobe_flash_data);
		} else
			return -ENODEV;
		break;
	default:
		rc = -ENODEV;
	}
	return rc;
}

int msm_strobe_flash_ctrl(struct msm_camera_sensor_strobe_flash_data *sfdata,
	struct strobe_flash_ctrl_data *strobe_ctrl)
{
	int rc = 0;
	switch (strobe_ctrl->type) {
	case STROBE_FLASH_CTRL_INIT:
		if (!sfdata)
			return -ENODEV;
		rc = msm_strobe_flash_xenon_init(sfdata);
		break;
	case STROBE_FLASH_CTRL_CHARGE:
		rc = msm_strobe_flash_xenon_charge(sfdata->flash_charge,
			strobe_ctrl->charge_en,
			sfdata->flash_recharge_duration);
		break;
	case STROBE_FLASH_CTRL_RELEASE:
		if (sfdata)
			rc = msm_strobe_flash_xenon_release(sfdata, 0);
		break;
	default:
		pr_err("Invalid Strobe Flash State\n");
		rc = -EINVAL;
	}
	return rc;
}

static int flash_adp1650_id = 0;

int msm_flash_ctrl(struct msm_camera_sensor_info *sdata,
	struct flash_ctrl_data *flash_info)
{
	int rc = 0;
	
	
	uint16_t flash_id;

	printk(" msm_flash_ctrl:flash type:(%d) \n",flash_info->flashtype);

	switch (flash_info->flashtype) {
	case LED_FLASH:
		if(MSM_CAMERA_LED_CHIP_ID==flash_info->ctrl_data.led_state)
		{
			rc = msm_camera_flash_read_led_id(&flash_id);
			
			if(rc == 0)
				flash_adp1650_id = (int)flash_id;
				
			printk(" msm_flash_ctrl()=>Flash ID:(%d) \n",flash_adp1650_id);
		}
		else
		rc = msm_camera_flash_set_led_state(sdata->flash_data,
			flash_info->ctrl_data.led_state);
			break;
	case STROBE_FLASH:
		#if 0
		rc = msm_strobe_flash_ctrl(sdata->strobe_flash_data,
			&(flash_info->ctrl_data.strobe_ctrl));
		#endif
		break;
	default:
		pr_err("Invalid Flash MODE\n");
		rc = -EINVAL;
	}
	return rc;
}


    
static int test_value = 0;

static int
module_setTest (const char *val, struct kernel_param *kp)
{
	int rc;
    int gpio_data_rc;
    int gpio_clk_rc;
    int i;
    int cur;

	rc = param_set_int (val, kp);

	gpio_data_rc = gpio_request(47,"FL_DATA");
    if(gpio_data_rc == 0)
    {
	    gpio_tlmm_config(GPIO_CFG(47, 1, GPIO_CFG_OUTPUT, GPIO_CFG_PULL_UP, GPIO_CFG_10MA), GPIO_CFG_ENABLE);
    }   
	gpio_clk_rc = gpio_request(48,"FL_CLK");
    if(gpio_clk_rc == 0)
    {
    	gpio_tlmm_config(GPIO_CFG(48, 1, GPIO_CFG_OUTPUT, GPIO_CFG_PULL_UP, GPIO_CFG_10MA), GPIO_CFG_ENABLE);
    }

    
	if (test_value == 0)
	{
        
        qsd_flash_adp_1650(MSM_CAMERA_LED_OFF);
	}
        
	else if (10000 < test_value && test_value < 20000 )
	{
        
        switch(test_value-10000)
    	{
            case 50:
                cur = 0x01;
                break;
            case 100:
                cur = 0x03;                
                break;
            case 150:
                cur = 0x05;                
                break;
            case 200:
                cur = 0x07;                
                break;
                 
            default:
                cur = 0x00;                                
                break;
                    
        }

    	qsd_flash_i2c_write_b_sensor(0x02,0xA4);	
	    qsd_flash_i2c_write_b_sensor(0x03,cur);	    
		qsd_flash_i2c_write_b_sensor(0x04,0x2A);	
        
	}
	else if (20000 < test_value && test_value < 30000 )
	{
        

        switch(test_value-20000)
    	{
            case 500:
                cur = 0x20;
                break;
            case 600:
                cur = 0x30;                
                break;
            case 700:
                cur = 0x40;                
                break;
            case 800:
                cur = 0x50;                
                break;
            case 900:
                cur = 0x60;                
                break;
            case 1000:
                cur = 0x70;                
                break;                
            case 1100:
                cur = 0x80;                
                break;
            case 1200:
                cur = 0x90;                
                break;            
                
            default:
                cur = 0x00;                                
                break;
                    
        }

        for(i=0;i<5;i++)
        {
    		qsd_flash_i2c_write_b_sensor(0x02,0xA1);	
	    	qsd_flash_i2c_write_b_sensor(0x03,cur);	    
		    qsd_flash_i2c_write_b_sensor(0x04,0x2B);	
		    msleep(3000);
        }

	}    
	    
	else if (test_value == 3)
	{
        

		qsd_flash_i2c_write_b_sensor(0x02,0xA4);	
		qsd_flash_i2c_write_b_sensor(0x03,0x8D);	
		qsd_flash_i2c_write_b_sensor(0x04,0x2B);	

	}
	else if (test_value == 4)
	{
        
        
		qsd_flash_i2c_write_b_sensor(0x02,0xA4);	
		qsd_flash_i2c_write_b_sensor(0x03,0x8D);	
		qsd_flash_i2c_write_b_sensor(0x04,0x2F);	

	}     
    else if (test_value == 5)
	{
		
		
		
		

        
		rc = qsd_flash_i2c_write_b_sensor (0x04, 0x29);	

	}    
	else
	{
		printk (KERN_ERR "unknown command = 0x%x\n", test_value);
	}


    if(gpio_data_rc == 0)
	    gpio_free(47);
    if(gpio_clk_rc == 0)
	    gpio_free(48);
    
	return rc;
}

module_param_call (testFile, module_setTest, param_get_long,
		   &test_value, S_IWUSR | S_IRUGO);
	


    
static int test_value_5v = 0;

static int
module_setTest_5v (const char *val, struct kernel_param *kp)
{
	int rc;

	rc = param_set_int (val, kp);
    
	if (test_value_5v == 0)
	{
        
        qsd_flash_5v(MSM_CAMERA_LED_OFF);
	}      
	else if (test_value_5v == 1 )
	{
        qsd_flash_5v(MSM_CAMERA_LED_HIGH);
        msleep(300);
        qsd_flash_5v(MSM_CAMERA_LED_OFF);           

	}    
	else if (test_value_5v == 2 )
	{
        qsd_flash_5v(MSM_CAMERA_LED_VIDEO_TORCH);     

	}       
	else
	{
		printk (KERN_ERR "unknown command = 0x%x\n", test_value_5v);
	}
    
	return rc;
}

module_param_call (testFile_5v, module_setTest_5v, param_get_long,
		   &test_value_5v, S_IWUSR | S_IRUGO);
	

static int __init qsd_flash_init(void)
{





       int rc = 0;
	uint16_t chipid = 0;
	uint16_t flash_id;
	
	
	printk("QisdaRay: __qsd_flash_probe E  \n");
	
	printk(" qsd_flash_init \n");
#if 1	
	
	rc = i2c_add_driver(&qsd_flash_i2c_driver);
	if (rc < 0 || qsd_flash_client == NULL) {
		printk(" i2c_add_driver fail \n");
		rc = -EINVAL;
	}
#endif


	rc = gpio_request(31, "FL_EN"); 

	gpio_direction_output(31, 1);
	
	rc = gpio_request(47,"FL_DATA");
	gpio_tlmm_config(GPIO_CFG(47, 1, GPIO_CFG_OUTPUT, GPIO_CFG_PULL_UP, GPIO_CFG_10MA), GPIO_CFG_ENABLE);
	printk(" I2C DATA = %d \n",rc);

	rc = gpio_request(48,"FL_CLK");
	gpio_tlmm_config(GPIO_CFG(48, 1, GPIO_CFG_OUTPUT, GPIO_CFG_PULL_UP, GPIO_CFG_10MA), GPIO_CFG_ENABLE);
	printk(" I2C CLK = %d \n",rc);
	
	mdelay(1000);
	
	rc = msm_camera_flash_read_led_id(&flash_id);
	
	if(rc == 0)
		flash_adp1650_id = (int)flash_id;
				
	printk(" msm_flash_ctrl()=>Flash ID:(%d) \n",flash_adp1650_id);
			
	
	rc = qsd_flash_i2c_read(0x00, &chipid, 1);
	printk("qsd_flash_adp_1650=>ID(0x00): %x\n", chipid);
	if((rc < 0) || (chipid != 0x22))
	{
		gpio_free(47);
		gpio_free(48);
		return -1;
	}
		
	mdelay(500);
	
	rc = qsd_flash_i2c_write_b_sensor(0x04,0x20);	

	rc = qsd_flash_i2c_read(0x04, &chipid, 1);
	printk("qsd_flash_adp_1650=>ID(0x04): %x\n", chipid);
	
	gpio_free(47);
	gpio_free(48);
		
	return rc;
	
	
}

module_init(qsd_flash_init);


module_param(flash_adp1650_id, int, 0664);

void qsd_flash_exit(void)
{
	i2c_del_driver(&qsd_flash_i2c_driver);
}

