/* Copyright (c) 2010, Code Aurora Forum. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 */

#include <linux/delay.h>
#include <linux/types.h>
#include <linux/i2c.h>
#include <linux/uaccess.h>
#include <linux/miscdevice.h>
#include <media/msm_camera.h>
#include <mach/gpio.h>
#include <mach/camera.h>
#include <linux/slab.h>
#include "mt9m114.h"
#include "mt9m114_reg.h"

#undef CDBG
#define LOCAL_DEBUG
#ifdef LOCAL_DEBUG
#define CDBG(fmt, args...) pr_err(fmt, ##args)
#else
#define CDBG(fmt, args...) do { } while (0)
#endif


#define REG_GROUPED_PARAMETER_HOLD			0x0104
#define GROUPED_PARAMETER_HOLD_OFF			0x00
#define GROUPED_PARAMETER_HOLD				0x01

#define REG_COARSE_INTEGRATION_TIME			0x3012

#define REG_GLOBAL_GAIN		0x305E

#define REG_FRAME_LENGTH_LINES			0x0340

#define REG_TEST_PATTERN_MODE				0x0601




#define	Q8		0x00000100
#define	Q10		0x00000400
#define	MT9M114_MASTER_CLK_RATE	24000000


struct mt9m114_work_t {
	struct work_struct work;
};

static struct mt9m114_work_t *mt9m114_sensorw;
static struct i2c_client *mt9m114_client;


static int mt9m114_id = 0;


struct mt9m114_ctrl_t {
	const struct  msm_camera_sensor_info *sensordata;

	uint32_t sensormode;
	uint32_t fps_divider;   	
	uint32_t pict_fps_divider;  
	uint16_t fps;

	int16_t curr_lens_pos;
	uint16_t curr_step_pos;
	uint16_t my_reg_gain;
	uint32_t my_reg_line_count;
	uint16_t total_lines_per_frame;

	enum mt9m114_resolution_t prev_res;
	enum mt9m114_resolution_t pict_res;
	enum mt9m114_resolution_t curr_res;
	enum mt9m114_test_mode_t  set_test;
};


static bool CSI_CONFIG;
static struct mt9m114_ctrl_t *mt9m114_ctrl;
static DECLARE_WAIT_QUEUE_HEAD(mt9m114_wait_queue);

static int mt9m114_effect = CAMERA_EFFECT_MAX;


#define MT9M114_WB_AUTO          1
#define MT9M114_WB_INCANDESCENT  3
#define MT9M114_WB_FLUORESCENT   4
#define MT9M114_WB_DAYLIGHT      5
#define MT9M114_WB_CLOUDY        6
#define MT9M114_WB_MAX           15
static int mt9m114_WB_mode = MT9M114_WB_MAX;


#define MT9M114_SCENE_MODE_OFF        0
#define MT9M114_SCENE_MODE_LANDSCAPE  1
#define MT9M114_SCENE_MODE_SNOW       2
#define MT9M114_SCENE_MODE_NIGHT      5
#define MT9M114_SCENE_MODE_PORTRAIT   6
#define MT9M114_SCENE_MODE_MAX        25
static int mt9m114_scene_mode = MT9M114_SCENE_MODE_MAX;

DEFINE_MUTEX(mt9m114_mut);



static int mt9m114_i2c_rxdata(unsigned short saddr,
	unsigned char *rxdata, int length)
{
	struct i2c_msg msgs[] = {
		{
			.addr  = saddr,
			.flags = 0,
			.len   = 2,
			.buf   = rxdata,
		},
		{
			.addr  = saddr,
			.flags = I2C_M_RD,
			.len   = 2,
			.buf   = rxdata,
		},
	};
	if (i2c_transfer(mt9m114_client->adapter, msgs, 2) < 0) {
		pr_err("mt9m114_i2c_rxdata failed!\n");
		return -EIO;
	}
	return 0;
}
static int32_t mt9m114_i2c_txdata(unsigned short saddr,
				unsigned char *txdata, int length)
{
	struct i2c_msg msg[] = {
		{
			.addr = saddr,
			.flags = 0,
			.len = length,
			.buf = txdata,
		 },
	};
	if (i2c_transfer(mt9m114_client->adapter, msg, 1) < 0) {
		pr_err("mt9m114_i2c_txdata faild 0x%x\n", mt9m114_client->addr);
		return -EIO;
	}

	return 0;
}


int32_t mt9m114_i2c_read(unsigned short raddr,
	unsigned short *rdata, int rlen)
{
	int32_t rc = 0;
	unsigned char buf[2];
	if (!rdata)
		return -EIO;
	memset(buf, 0, sizeof(buf));
	buf[0] = (raddr & 0xFF00) >> 8;
	buf[1] = (raddr & 0x00FF);
	rc = mt9m114_i2c_rxdata(mt9m114_client->addr, buf, rlen);
	if (unlikely(rc < 0)) {
		pr_err("mt9m114_i2c_read 0x%x failed!\n", raddr);
		return rc;
	}
	*rdata = (rlen == 2 ? buf[0] << 8 | buf[1] : buf[0]);
	
	return rc;
}

static int32_t mt9m114_i2c_write_w_sensor(unsigned short waddr, uint16_t wdata)
{
	int32_t rc = -EFAULT;
	unsigned char buf[4];
	memset(buf, 0, sizeof(buf));
	buf[0] = (waddr & 0xFF00) >> 8;
	buf[1] = (waddr & 0x00FF);
	buf[2] = (wdata & 0xFF00) >> 8;
	buf[3] = (wdata & 0x00FF);

	rc = mt9m114_i2c_txdata(mt9m114_client->addr, buf, 4);
	if (unlikely(rc < 0)) {
		pr_err("mt9m114_i2c_write_w_sensor failed, addr = 0x%x, val = 0x%x!\n",
			waddr, wdata);
	}
	return rc;
}

static int32_t mt9m114_i2c_write_b_sensor(unsigned short waddr, uint8_t bdata)
{
	int32_t rc = -EFAULT;
	unsigned char buf[3];
	memset(buf, 0, sizeof(buf));
	buf[0] = (waddr & 0xFF00) >> 8;
	buf[1] = (waddr & 0x00FF);
	buf[2] = bdata;

	rc = mt9m114_i2c_txdata(mt9m114_client->addr, buf, 3);
	if (unlikely(rc < 0)) {
		pr_err("mt9m114_i2c_write_b_sensor failed, addr = 0x%x, val = 0x%x!\n",
			waddr, bdata);
	}
	return rc;
}


static int32_t mt9m114_i2c_write_table(struct mt9m114_reg_tuning 
					 *reg_conf_tbl, int num)
{
	int i;
	int32_t rc = -EIO;
       if (reg_conf_tbl->next == NULL)
       {
        	for (i = 0; i < num; i++) {
                
        		if (reg_conf_tbl->is_byte) {
        			rc = mt9m114_i2c_write_b_sensor(reg_conf_tbl->waddr,
        				(uint8_t)(reg_conf_tbl->wdata));
        		}else{
        			rc = mt9m114_i2c_write_w_sensor(reg_conf_tbl->waddr,
        				reg_conf_tbl->wdata);
        		}
        		if (unlikely(rc < 0)) {
        			pr_err("%s failed!\n",__func__);
        			break;
        		}
        		reg_conf_tbl++;
        	}
       }
       else
       {
        	for (i = 0; i < num; i++) {
                
        		if (reg_conf_tbl->is_byte) {
        			rc = mt9m114_i2c_write_b_sensor(reg_conf_tbl->waddr,
        				(uint8_t)(reg_conf_tbl->wdata));
        		}else{
        			rc = mt9m114_i2c_write_w_sensor(reg_conf_tbl->waddr,
        				reg_conf_tbl->wdata);
        		}
        		if (unlikely(rc < 0)) {
        			pr_err("%s failed!\n",__func__);
        			break;
        		}
        		reg_conf_tbl = reg_conf_tbl->next;
        	}
       }
	return rc;
}


static int32_t mt9m114_exit_standby(void)
{
	uint16_t chipid = 0;
	int rc = 0;
	int count = 0;

	CDBG("%s\n", __func__);
	mt9m114_i2c_write_b_sensor(0xDC00, 0x54);
	mt9m114_i2c_write_w_sensor(0x0080, 0x8002);
	mdelay(50);
	while(1)
	{
		rc = mt9m114_i2c_read(0x0080, &chipid, 2);
		if (unlikely(rc < 0))
		{
			pr_err("%s() mt9m114_i2c_read failed!\n",__func__);
			return rc;
		}

		if ( (chipid & 0x8002) == 0x8000 ) break;
		if (count == 3)
		{
			pr_err("%s() Set State cmd failed! I2C readvalue=0x%x.\n", __func__, chipid);
			return -EPERM;
		}
		count++;
		mdelay(20);
	}

	rc = mt9m114_i2c_read(0xDC01, &chipid, 1);
	if (unlikely(rc < 0))
	{
		pr_err("%s() mt9m114_i2c_read failed!\n",__func__);
		return rc;
	}
	if ( chipid != 0x31 )
	{
		pr_err("%s() System state is not STREAMING 0x%x!\n",__func__, chipid);
		return -EPERM;
	}

	return rc;
}

static int32_t mt9m114_enter_standby(void)
{
	uint16_t chipid = 0;
	int rc;
	int count = 0;

	CDBG("%s\n", __func__);
	mt9m114_i2c_write_b_sensor(0xDC00, 0x50);
	mt9m114_i2c_write_w_sensor(0x0080, 0x8002);
	mdelay(50);
	while(1)
	{
		rc = mt9m114_i2c_read(0x0080, &chipid, 2);
		if (unlikely(rc < 0))
		{
			pr_err("%s() mt9m114_i2c_read failed!\n",__func__);
			return rc;
		}
		if ( (chipid & 0x8002) == 0x8000 ) break;
		if (count == 3)
		{
			pr_err("%s() Set State cmd failed! I2C readvalue=0x%x.\n", __func__, chipid);
			return -EPERM;
		}
		count++;
		mdelay(20);
	}

	rc = mt9m114_i2c_read(0xDC01, &chipid, 1);
	if (unlikely(rc < 0))
	{
		pr_err("%s() mt9m114_i2c_read failed!\n",__func__);
		return rc;
	}
	if ( chipid != 0x52 )
	{
		pr_err("%s() System state is not STANDBY 0x%x!\n",__func__, chipid);
		return -EPERM;
	}

	return rc;
}




static int32_t mt9m114_poll_cmd_reg(uint16_t hostCommandX)
{
	uint16_t chipid = 0;
	int rc = 0;
	int count = 0;
	uint16_t value = hostCommandX | 0x8000;

	CDBG("%s\n", __func__);

	mdelay(20);
	while(1)
	{
		rc = mt9m114_i2c_read(0x0080, &chipid, 2);
		if (unlikely(rc < 0))
		{
			pr_err("%s() mt9m114_i2c_read failed!\n",__func__);
			return rc;
		}
		if ( (chipid & value) == 0x8000 ) break;
		if (count == 6)
		{
			pr_err("%s() Poll timeout. HOST_COMMAND=0x%x, I2C readvalue=0x%x.\n", __func__, hostCommandX, chipid);
			return -EPERM;
		}
		count++;
		mdelay(20);
	}

	return rc;
}

static int32_t mt9m114_preview_setting_abicon(void)
{
	int32_t rc = 0;

	
	mt9m114_i2c_write_w_sensor(0x001A, 0x0001);	
	mdelay(10);
	mt9m114_i2c_write_w_sensor(0x001A, 0x0000);	
	mdelay(50);

	mt9m114_i2c_write_w_sensor(0x301A, 0x0234);	

	
	
	mt9m114_effect = CAMERA_EFFECT_MAX;
	
	
	mt9m114_WB_mode = MT9M114_WB_MAX;
	
	
	mt9m114_scene_mode = MT9M114_SCENE_MODE_MAX;
	

	rc = mt9m114_i2c_write_table(mt9m114_regs.reg_pll_timing,
		mt9m114_regs.reg_pll_timing_size);
	if (unlikely(rc < 0))
	{
		pr_err("%s() mt9m114_i2c_write_table(mt9m114_regs.reg_pll_timing) failed!\n",__func__);
		return rc;
	}

	rc = mt9m114_i2c_write_table(mt9m114_regs.reg_optimization,
		mt9m114_regs.reg_optimization_size);
	if (unlikely(rc < 0))
	{
		pr_err("%s() mt9m114_i2c_write_table(mt9m114_regs.reg_optimization) failed!\n",__func__);
		return rc;
	}

	rc = mt9m114_i2c_write_table(mt9m114_regs.reg_patch0202,
		mt9m114_regs.reg_patch0202_size);
	if (unlikely(rc < 0))
	{
		pr_err("%s() mt9m114_i2c_write_table(mt9m114_regs.reg_patch0202) failed!\n",__func__);
		return rc;
	}
	mt9m114_i2c_write_w_sensor(0x0080, 0xFFF0);
	
	rc = mt9m114_poll_cmd_reg(0x1);
	if (unlikely(rc < 0)){
		pr_err("%s mt9m114_poll_cmd_reg failed! Line%d\n", __func__, __LINE__);
		return rc;
	}
	mt9m114_i2c_write_w_sensor(0x0080, 0xFFF1);
	
	rc = mt9m114_poll_cmd_reg(0x1);
	if (unlikely(rc < 0)){
		pr_err("%s mt9m114_poll_cmd_reg failed! Line%d\n", __func__, __LINE__);
		return rc;
	}

	rc = mt9m114_i2c_write_table(mt9m114_regs.reg_patch0302,
		mt9m114_regs.reg_patch0302_size);
	if (unlikely(rc < 0))
	{
		pr_err("%s() mt9m114_i2c_write_table(mt9m114_regs.reg_patch0302) failed!\n",__func__);
		return rc;
	}
	mt9m114_i2c_write_w_sensor(0x0080, 0xFFF0);	
	
	rc = mt9m114_poll_cmd_reg(0x1);
	if (unlikely(rc < 0)){
		pr_err("%s mt9m114_poll_cmd_reg failed! Line%d\n", __func__, __LINE__);
		return rc;
	}
	mt9m114_i2c_write_w_sensor(0x0080, 0xFFF1);	
	
	rc = mt9m114_poll_cmd_reg(0x1);
	if (unlikely(rc < 0)){
		pr_err("%s mt9m114_poll_cmd_reg failed! Line%d\n", __func__, __LINE__);
		return rc;
	}

	rc = mt9m114_i2c_write_table(mt9m114_regs.reg_APGA,
		mt9m114_regs.reg_APGA_size);
	if (unlikely(rc < 0))
	{
		pr_err("%s() mt9m114_i2c_write_table(mt9m114_regs.reg_APGA) failed!\n",__func__);
		return rc;
	}

	rc = mt9m114_i2c_write_table(mt9m114_regs.reg_AWB_CCM,
		mt9m114_regs.reg_AWB_CCM_size);
	if (unlikely(rc < 0))
	{
		pr_err("%s() mt9m114_i2c_write_table(mt9m114_regs.reg_AWB_CCM) failed!\n",__func__);
		return rc;
	}

	rc = mt9m114_i2c_write_table(mt9m114_regs.reg_MIPI_setting,
		mt9m114_regs.reg_MIPI_setting_size);
	if (unlikely(rc < 0))
	{
		pr_err("%s() mt9m114_i2c_write_table(mt9m114_regs.reg_MIPI_setting) failed!\n",__func__);
		return rc;
	}

	
	
	mt9m114_i2c_write_b_sensor(0xDC00, 0x40);
	mt9m114_i2c_write_w_sensor(0x0080, 0x8002);
	
	rc = mt9m114_poll_cmd_reg(0x2);
	if (unlikely(rc < 0)){
		pr_err("%s mt9m114_poll_cmd_reg failed! Line%d\n", __func__, __LINE__);
		return rc;
	}

	mt9m114_i2c_write_b_sensor(0xDC00, 0x34);
	mt9m114_i2c_write_w_sensor(0x0080, 0x8002);
	
	rc = mt9m114_poll_cmd_reg(0x2);
	if (unlikely(rc < 0)){
		pr_err("%s mt9m114_poll_cmd_reg failed! Line%d\n", __func__, __LINE__);
		return rc;
	}
	
	rc = mt9m114_i2c_write_table(mt9m114_regs.reg_Dual_Lane_Mipi_Receiver_Init,
		mt9m114_regs.reg_Dual_Lane_Mipi_Receiver_Init_size);
	if (unlikely(rc < 0))
	{
		pr_err("%s() mt9m114_i2c_write_table(mt9m114_regs.reg_Dual_Lane_Mipi_Receiver_Init) failed!\n",__func__);
		return rc;
	}
	mt9m114_i2c_write_b_sensor(0xDC00, 0x28);
	mt9m114_i2c_write_w_sensor(0x0080, 0x8002);
	
	rc = mt9m114_poll_cmd_reg(0x2);
	if (unlikely(rc < 0)){
		pr_err("%s mt9m114_poll_cmd_reg failed! Line%d\n", __func__, __LINE__);
		return rc;
	}









	return rc;
}

static int32_t mt9m114_sensor_setting(int update_type, int rt)
{

	int32_t rc = 0;
	
	struct msm_camera_csi_params mt9m114_csi_params;
	CDBG("%s(%d, %d)\n", __func__, update_type, rt);
	
	if(update_type == REG_INIT) {
		CSI_CONFIG = 0;
	}else if (update_type == UPDATE_PERIODIC) {
		if (rt == RES_PREVIEW) {
            
		}else{
            
		}
		if (!CSI_CONFIG) {
			
			rc = mt9m114_preview_setting_abicon();
			if (unlikely(rc < 0)){
				pr_err("%s mt9m114_preview_setting_abicon failed!\n", __func__);
				return rc;
			}
			msleep(10);
			rc = mt9m114_enter_standby();
			if (unlikely(rc < 0)){
				pr_err("%s mt9m114_enter_standby failed!\n", __func__);
				return rc;
			}

			mt9m114_csi_params.data_format = CSI_8BIT;
			mt9m114_csi_params.lane_cnt = 1;
			mt9m114_csi_params.lane_assign = 0xe4;
			mt9m114_csi_params.dpcm_scheme = 0;
			mt9m114_csi_params.settle_cnt = 14;
			rc = msm_camio_csi_config(&mt9m114_csi_params);
			msleep(10);
			CSI_CONFIG = 1;
			rc = mt9m114_exit_standby();
			if (unlikely(rc < 0)){
				pr_err("%s mt9m114_exit_standby failed!\n", __func__);
				return rc;
			}
		}
	}
	return rc;
}


static int32_t mt9m114_video_config(int mode)
{

	int32_t	rc = 0;
	int	rt;
	CDBG("%s\n",__func__);
	
	if (mt9m114_ctrl->prev_res == QTR_SIZE) {
		rt = RES_PREVIEW;
	} else {
		rt = RES_CAPTURE;
	}
	if (mt9m114_sensor_setting(UPDATE_PERIODIC, rt) < 0)
		return rc;

	mt9m114_ctrl->curr_res = mt9m114_ctrl->prev_res;
	mt9m114_ctrl->sensormode = mode;
	return rc;
}

static int32_t mt9m114_snapshot_config(int mode)
{
	int32_t rc = 0;
	int rt;

	CDBG("%s\n",__func__);
	
	if (mt9m114_ctrl->curr_res != mt9m114_ctrl->pict_res) {
		if (mt9m114_ctrl->pict_res == QTR_SIZE) {
			rt = RES_PREVIEW;
		} else {
			rt = RES_CAPTURE;
		}
	if (mt9m114_sensor_setting(UPDATE_PERIODIC, rt) < 0)
		return rc;
	}

	mt9m114_ctrl->curr_res = mt9m114_ctrl->pict_res;
	mt9m114_ctrl->sensormode = mode;
	return rc;
} 

static int32_t mt9m114_raw_snapshot_config(int mode)
{
	int32_t rc = 0;
	int rt;

	CDBG("%s\n",__func__);
	
	if (mt9m114_ctrl->curr_res != mt9m114_ctrl->pict_res) {
		if (mt9m114_ctrl->pict_res == QTR_SIZE) {
			rt = RES_PREVIEW;
		} else {
			rt = RES_CAPTURE;
		}
		if (mt9m114_sensor_setting(UPDATE_PERIODIC, rt) < 0)
			return rc;
	}

	mt9m114_ctrl->curr_res = mt9m114_ctrl->pict_res;
	mt9m114_ctrl->sensormode = mode;
	return rc;
} 

static int32_t mt9m114_set_sensor_mode(int mode,
	int res)
{
	int32_t rc = 0;
	switch (mode) {
	case SENSOR_PREVIEW_MODE:
		rc = mt9m114_video_config(mode);
		break;
	case SENSOR_SNAPSHOT_MODE:
		rc = mt9m114_snapshot_config(mode);
		break;
	case SENSOR_RAW_SNAPSHOT_MODE:
		rc = mt9m114_raw_snapshot_config(mode);
		break;
	default:
		rc = -EINVAL;
		break;
	}
	return rc;
}
static int32_t mt9m114_power_down(void)
{
	return 0;
}
 

static int mt9m114_probe_init_done(const struct msm_camera_sensor_info *data)
{
	CDBG("%s\n",__func__);




	return 0;
}

static int mt9m114_probe_init_sensor(const struct msm_camera_sensor_info *data)
{
	int32_t rc = 0; 
	uint16_t chipid = 0;
	CDBG("%s\n",__func__);

	rc = mt9m114_i2c_read(0x0000, &chipid, 2);
	if (unlikely(rc < 0))
	{
		pr_err("%s() mt9m114_i2c_read failed!\n",__func__);
		goto init_probe_fail;
	}
	CDBG("ID: %d\n", chipid);
	

	mt9m114_id = chipid;
	

	
	if (chipid != 0x2481) {
		rc = -ENODEV;
		pr_err("mt9m114_probe_init_sensor fail chip id doesnot match\n");
		goto init_probe_fail;
	}

	mt9m114_ctrl = kzalloc(sizeof(struct mt9m114_ctrl_t), GFP_KERNEL);
	if (!mt9m114_ctrl) {
		pr_err("%s memory error!\n", __func__);
		rc = -ENOMEM;
	}
	mt9m114_ctrl->fps_divider = 1 * 0x00000400;
	mt9m114_ctrl->pict_fps_divider = 1 * 0x00000400;
	mt9m114_ctrl->set_test = TEST_OFF;
	mt9m114_ctrl->prev_res = QTR_SIZE;
	mt9m114_ctrl->pict_res = FULL_SIZE;
	

	if (data)
		mt9m114_ctrl->sensordata = data;

	goto init_probe_done;
init_probe_fail:
	gpio_direction_output(data->sensor_reset, 0);
	mt9m114_probe_init_done(data);
init_probe_done:
	return rc;
}


int mt9m114_sensor_open_init(const struct msm_camera_sensor_info *data)
{
	int32_t rc = 0;

	CDBG("%s\n", __func__);

	mt9m114_ctrl = kzalloc(sizeof(struct mt9m114_ctrl_t), GFP_KERNEL);
	if (!mt9m114_ctrl) {
		pr_err("%s memory error!\n", __func__);
		rc = -ENOMEM;
		goto init_done;
	}
	mt9m114_ctrl->fps_divider = 1 * 0x00000400;
	mt9m114_ctrl->pict_fps_divider = 1 * 0x00000400;
	mt9m114_ctrl->set_test = TEST_OFF;
	mt9m114_ctrl->prev_res = QTR_SIZE;
	mt9m114_ctrl->pict_res = FULL_SIZE;
	

	if (data)
		mt9m114_ctrl->sensordata = data;

	
	msm_camio_clk_rate_set(MT9M114_MASTER_CLK_RATE);
	rc = mt9m114_probe_init_sensor(data);
	if (unlikely(rc < 0)) {
		pr_err("%s mt9m114_probe_init_sensor failed\n", __func__);
		return rc;
	}
	CDBG("%s init settings\n", __func__);
	if (mt9m114_ctrl->prev_res == QTR_SIZE) {
		if (mt9m114_sensor_setting(REG_INIT, RES_PREVIEW) < 0)
			goto init_fail;
	} else {
		if (mt9m114_sensor_setting(REG_INIT, RES_CAPTURE) < 0)
			goto init_fail;
	}
	mt9m114_ctrl->fps = 30*Q8;
	if (unlikely(rc < 0))
		goto init_fail;
	else
		goto init_done;
init_fail:
	pr_err("%s init_fail\n", __func__);
	gpio_direction_output(data->sensor_reset, 0);
	mt9m114_probe_init_done(data);
init_done:
	CDBG("%s init_done\n", __func__);
	return rc;
} 

static int mt9m114_init_client(struct i2c_client *client)
{
	
	init_waitqueue_head(&mt9m114_wait_queue);
	return 0;
}

static const struct i2c_device_id mt9m114_i2c_id[] = {
	{"mt9m114", 0},
	{ }
};

static int mt9m114_i2c_probe(struct i2c_client *client,
	const struct i2c_device_id *id)
{
	int rc = 0;
	CDBG("%s\n", __func__);

	if (!i2c_check_functionality(client->adapter, I2C_FUNC_I2C)) {
		pr_err("%s i2c_check_functionality failed!\n", __func__);
		goto probe_failure;
	}

	mt9m114_sensorw = kzalloc(sizeof(struct mt9m114_work_t), GFP_KERNEL);
	if (!mt9m114_sensorw) {
		pr_err("%s memory error!c failed!\n", __func__);
		rc = -ENOMEM;
		goto probe_failure;
	}

	i2c_set_clientdata(client, mt9m114_sensorw);
	mt9m114_init_client(client);
	mt9m114_client = client;

	mdelay(50);

	return 0;

probe_failure:
	pr_err("%s failed!\n", __func__);
	return rc;
}
static int __exit mt9m114_remove(struct i2c_client *client)
{
	struct mt9m114_work_t_t *sensorw = i2c_get_clientdata(client);
	free_irq(client->irq, sensorw);
	mt9m114_client = NULL;
	kfree(sensorw);
	return 0;
}

static struct i2c_driver mt9m114_i2c_driver = {
	.id_table = mt9m114_i2c_id,
	.probe  = mt9m114_i2c_probe,
	.remove = __exit_p(mt9m114_i2c_remove),
	.driver = {
		.name = "mt9m114",
	},
};



static int mt9m114_change_config(void)
{
	int rc = 0;
	uint16_t chipid = 0;

	rc = mt9m114_i2c_write_b_sensor(0xDC00, 0x28);
	if (unlikely(rc < 0)){
		pr_err("%s() I2C operation failed!\n", __func__);
		return rc;
	}
	rc = mt9m114_i2c_write_w_sensor(0x0080, 0x8002);
	if (unlikely(rc < 0)){
		pr_err("%s() I2C operation failed!\n", __func__);
		return rc;
	}

    
	rc = mt9m114_poll_cmd_reg(0x2);
	if (unlikely(rc < 0)){
		pr_err("%s mt9m114_poll_cmd_reg failed!\n", __func__);
		return rc;
	}

	
	rc = mt9m114_i2c_read(0xDC01, &chipid, 1);
	if (unlikely(rc < 0))
	{
		pr_err("%s() mt9m114_i2c_read failed!\n",__func__);
		return rc;
	}
	if ( chipid != 0x31 )
	{
		pr_err("%s() System state is not STREAMING 0x%x!\n",__func__, chipid);
		return -EPERM;
	}

	return rc;
}

static int mt9m114_set_effect(int effect)
{
	int rc = 0;
    uint8_t reg_val;

	CDBG("%s(%d), current effect(%d)\n", __func__, effect, mt9m114_effect);

	if (mt9m114_effect == effect){
		return 0;
	};

	switch (effect) {
	case CAMERA_EFFECT_OFF: {
		reg_val = 0x00;
	}
		break;

	case CAMERA_EFFECT_MONO: {
		reg_val = 0x01;
	}
		break;

	case CAMERA_EFFECT_NEGATIVE: {
		reg_val = 0x03;
	}
		break;

	case CAMERA_EFFECT_SOLARIZE: {
		reg_val = 0x05;
	}
		break;

	case CAMERA_EFFECT_SEPIA: {
		reg_val = 0x02;
	}
		break;

	case CAMERA_EFFECT_POSTERIZE: {
		reg_val = 0x04;
	}
		break;

	default: {
		pr_err("%s() Invalid effect %d!\n", __func__, effect);
		return 0;
	}
	}

	rc = mt9m114_i2c_write_w_sensor(0x098E, 0xC874);
	if (unlikely(rc < 0)){
		pr_err("%s() I2C operation failed!\n", __func__);
		return rc;
	}
	rc = mt9m114_i2c_write_b_sensor(0xC874, reg_val);
	if (unlikely(rc < 0)){
		pr_err("%s() I2C operation failed!\n", __func__);
		return rc;
	}

	
	rc = mt9m114_change_config();
	if (unlikely(rc < 0)){
		pr_err("%s() mt9m114_change_config failed!\n", __func__);
		return rc;
	}
	
	mt9m114_effect = effect;

	return 0;
}



static int mt9m114_set_wb(int WBmode)
{
	int rc = 0;
	uint16_t reg_C8D0, reg_C8D2, reg_CC18;

	CDBG("%s(%d), current WBmode(%d)\n", __func__, WBmode, mt9m114_WB_mode);

	if (mt9m114_WB_mode == WBmode){
		return 0;
	};

	switch (WBmode) {
	case MT9M114_WB_AUTO: {
		rc = mt9m114_i2c_write_w_sensor(0xC8D0, 0x00A4);
		if (unlikely(rc < 0)){
			pr_err("%s() I2C operation failed!\n", __func__);
			return rc;
		}
		rc = mt9m114_i2c_write_w_sensor(0xC8D2, 0x00AC);
		if (unlikely(rc < 0)){
			pr_err("%s() I2C operation failed!\n", __func__);
			return rc;
		}
		rc = mt9m114_i2c_write_w_sensor(0x098E, 0xCC01);
		if (unlikely(rc < 0)){
			pr_err("%s() I2C operation failed!\n", __func__);
			return rc;
		}
		rc = mt9m114_i2c_write_b_sensor(0xCC01, 0x01);
		if (unlikely(rc < 0)){
			pr_err("%s() I2C operation failed!\n", __func__);
			return rc;
		}

		mt9m114_WB_mode = MT9M114_WB_AUTO;

		return 0;
	}
		break;

	case MT9M114_WB_INCANDESCENT: {
		reg_C8D0 = 0x00A4;
		reg_C8D2 = 0x00AC;
		reg_CC18 = 0x0B28;
	}
		break;

	case MT9M114_WB_FLUORESCENT: {
		reg_C8D0 = 0x00A4;
		reg_C8D2 = 0x00AC;
		reg_CC18 = 0x0F0A;
	}
		break;

	case MT9M114_WB_DAYLIGHT: {
		reg_C8D0 = 0x00C8;
		reg_C8D2 = 0x008C;       
		reg_CC18 = 0x1964;
	}
		break;

	case MT9M114_WB_CLOUDY: {
		reg_C8D0 = 0x00A4;
		reg_C8D2 = 0x00AC;
		reg_CC18 = 0x1964;
	}
		break;

	default: {
		pr_err("%s() Invalid WBmode %d!\n", __func__, WBmode);
		return 0;
	}
	}

	rc = mt9m114_i2c_write_w_sensor(0x098E, 0xCC01);
	if (unlikely(rc < 0)){
		pr_err("%s() I2C operation failed!\n", __func__);
		return rc;
	}
	rc = mt9m114_i2c_write_b_sensor(0xCC01, 0x00);
	if (unlikely(rc < 0)){
		pr_err("%s() I2C operation failed!\n", __func__);
		return rc;
	}
	mdelay(25);
	rc = mt9m114_i2c_write_w_sensor(0xC8D0, reg_C8D0);
	if (unlikely(rc < 0)){
		pr_err("%s() I2C operation failed!\n", __func__);
		return rc;
	}
	rc = mt9m114_i2c_write_w_sensor(0xC8D2, reg_C8D2);
	if (unlikely(rc < 0)){
		pr_err("%s() I2C operation failed!\n", __func__);
		return rc;
	}
 
	mdelay(20);
 
	rc = mt9m114_i2c_write_w_sensor(0xCC18, reg_CC18);
	if (unlikely(rc < 0)){
		pr_err("%s() I2C operation failed!\n", __func__);
		return rc;
	}

	mt9m114_WB_mode = WBmode;

	return 0;
}



static int mt9m114_set_scene_mode(int scene_mode)
{
	int rc = 0;
	struct mt9m114_reg_tuning  *reg_scene_mode_conf1, *reg_scene_mode_conf2;
	int reg_scene_mode_size1, reg_scene_mode_size2;

	CDBG("%s(%d), current scene mode(%d)\n", __func__, scene_mode, mt9m114_scene_mode);

    if (mt9m114_scene_mode == scene_mode){
		return 0;
	};

	switch (scene_mode) {
	case MT9M114_SCENE_MODE_OFF: {
		reg_scene_mode_conf1 = mt9m114_reg_settings.reg_scene_mode_off1;
		reg_scene_mode_conf2 = mt9m114_reg_settings.reg_scene_mode_off2;
		reg_scene_mode_size1 = mt9m114_reg_settings.reg_scene_mode_off1_size;
		reg_scene_mode_size2 = mt9m114_reg_settings.reg_scene_mode_off2_size;

	}
		break;

	case MT9M114_SCENE_MODE_LANDSCAPE: {
		reg_scene_mode_conf1 = mt9m114_reg_settings.reg_scene_mode_landscape1;
		reg_scene_mode_conf2 = mt9m114_reg_settings.reg_scene_mode_landscape2;
		reg_scene_mode_size1 = mt9m114_reg_settings.reg_scene_mode_landscape1_size;
		reg_scene_mode_size2 = mt9m114_reg_settings.reg_scene_mode_landscape2_size;
	}
		break;

	case MT9M114_SCENE_MODE_SNOW: {
		reg_scene_mode_conf1 = mt9m114_reg_settings.reg_scene_mode_snow1;
		reg_scene_mode_conf2 = mt9m114_reg_settings.reg_scene_mode_snow2;
		reg_scene_mode_size1 = mt9m114_reg_settings.reg_scene_mode_snow1_size;
		reg_scene_mode_size2 = mt9m114_reg_settings.reg_scene_mode_snow2_size;
	}
		break;

	case MT9M114_SCENE_MODE_NIGHT: {
		reg_scene_mode_conf1 = mt9m114_reg_settings.reg_scene_mode_night1;
		reg_scene_mode_conf2 = mt9m114_reg_settings.reg_scene_mode_night2;
		reg_scene_mode_size1 = mt9m114_reg_settings.reg_scene_mode_night1_size;
		reg_scene_mode_size2 = mt9m114_reg_settings.reg_scene_mode_night2_size;
	}
		break;

	case MT9M114_SCENE_MODE_PORTRAIT: {
		reg_scene_mode_conf1 = mt9m114_reg_settings.reg_scene_mode_portrain1;
		reg_scene_mode_conf2 = mt9m114_reg_settings.reg_scene_mode_portrain2;
		reg_scene_mode_size1 = mt9m114_reg_settings.reg_scene_mode_portrain1_size;
		reg_scene_mode_size2 = mt9m114_reg_settings.reg_scene_mode_portrain2_size;
	}
		break;

	default: {
		pr_err("%s() Invalid scene mode %d!\n", __func__, scene_mode);
		return 0;
	}
	}

	rc = mt9m114_i2c_write_table(reg_scene_mode_conf1,
		reg_scene_mode_size1);
	if (unlikely(rc < 0))
	{
		pr_err("%s() mt9m114_i2c_write_table failed!\n",__func__);
		return rc;
	}

	
	rc = mt9m114_change_config();
	if (unlikely(rc < 0)){
		pr_err("%s() mt9m114_change_config failed!\n", __func__);
		return rc;
	}

	rc = mt9m114_i2c_write_table(reg_scene_mode_conf2,
		reg_scene_mode_size2);
	if (unlikely(rc < 0))
	{
		pr_err("%s() mt9m114_i2c_write_table failed!\n",__func__);
		return rc;
	}


	mt9m114_scene_mode = scene_mode;

	return 0;
}


int mt9m114_sensor_config(void __user *argp)
{
	struct sensor_cfg_data cdata;
	long   rc = 0;
	if (copy_from_user(&cdata,
		(void *)argp,
		sizeof(struct sensor_cfg_data)))
		return -EFAULT;

	mutex_lock(&mt9m114_mut);
	CDBG("mt9m114_sensor_config: cfgtype=%d mode=%d rs=%d\n", cdata.cfgtype, cdata.mode, cdata.rs);

	switch (cdata.cfgtype) {
		case CFG_SET_MODE:
			rc = mt9m114_set_sensor_mode(cdata.mode,
					cdata.rs);
			break;

		case CFG_PWR_DOWN:
			rc = mt9m114_power_down();
			break;

		
		case CFG_SET_EFFECT:
			rc = mt9m114_set_effect(cdata.cfg.effect);
			break;
		

		
		case CFG_SET_WB:
			rc = mt9m114_set_wb(cdata.cfg.effect);
			break;
		

		
		case CFG_SET_SCENE_MODE:
			rc = mt9m114_set_scene_mode(cdata.cfg.effect);
			break;
		

		default:
			rc = -EFAULT;
			break;
	}

	mutex_unlock(&mt9m114_mut);

	return rc;
}




static int mt9m114_sensor_release(void)
{
	int rc = -EBADF;
	CDBG("%s\n", __func__);
	mutex_lock(&mt9m114_mut);
	mt9m114_power_down();
    
#if 0    
	gpio_free(mt9m114_ctrl->sensordata->sensor_reset);
#endif

	kfree(mt9m114_ctrl);
	mt9m114_ctrl = NULL;
	mutex_unlock(&mt9m114_mut);

	return rc;
}

static int mt9m114_sensor_probe(const struct msm_camera_sensor_info *info,
		struct msm_sensor_ctrl *s)
{
	int rc = 0;
	CDBG("%s\n", __func__);
	rc = i2c_add_driver(&mt9m114_i2c_driver);
	if (rc < 0 || mt9m114_client == NULL){
		pr_err("%s i2c_add_driver failed %d!\n", __func__, rc);
		rc = -ENOTSUPP;
		goto probe_fail;
	}

	msm_camio_clk_rate_set(MT9M114_MASTER_CLK_RATE);

	rc = mt9m114_probe_init_sensor(info);
	if (unlikely(rc < 0)){
		pr_err("%s mt9m114_probe_init_sensor failed %d!\n", __func__, rc);
		rc = -ENOTSUPP;
		goto probe_fail;
	}

	s->s_init = mt9m114_sensor_open_init;
	s->s_release = mt9m114_sensor_release;
	s->s_config  = mt9m114_sensor_config;
	s->s_camera_type = FRONT_CAMERA_2D;
    
	s->s_mount_angle = 90;
    
	mt9m114_probe_init_done(info);
	return rc;

probe_fail:
	pr_err("%s probe_fail %d!\n",__func__, rc);
	return rc;
}

static int __mt9m114_probe(struct platform_device *pdev)
{
	CDBG("%s\n", __func__);
	return msm_camera_drv_start(pdev, mt9m114_sensor_probe);
}

static struct platform_driver msm_camera_driver = {
	.probe = __mt9m114_probe,
	.driver = {
		.name = "msm_camera_mt9m114",
		.owner = THIS_MODULE,
	},
};

static int __init mt9m114_init(void)
{
	CDBG("%s\n", __func__);
	return platform_driver_register(&msm_camera_driver);
}

module_init(mt9m114_init);


module_param(mt9m114_id, int, 0664);


void mt9m114_exit(void)
{
	CDBG("%s\n", __func__);
	i2c_del_driver(&mt9m114_i2c_driver);
}
