
#include "ov5642.h"

static int ov5642_current_effect = CAMERA_EFFECT_MAX;

#define OV5642_WB_AUTO          1
#define OV5642_WB_INCANDESCENT  3
#define OV5642_WB_FLUORESCENT   4
#define OV5642_WB_DAYLIGHT      5
#define OV5642_WB_CLOUDY        6
#define OV5642_WB_MAX           15
static int ov5642_current_wb = OV5642_WB_MAX;

#define OV5642_SCENE_MODE_OFF        0
#define OV5642_SCENE_MODE_LANDSCAPE  1
#define OV5642_SCENE_MODE_SNOW       2
#define OV5642_SCENE_MODE_NIGHT      5
#define OV5642_SCENE_MODE_PORTRAIT   6
#define OV5642_SCENE_MODE_MAX        25
static int ov5642_current_scene = OV5642_SCENE_MODE_MAX;


#ifdef USE_AF
extern struct ov5642_af_func_array ov5642_af_func;
#endif


#define Q8    0x00000100


#define REG_OV5642_MODEL_ID_MSB                       0x300A
#define REG_OV5642_MODEL_ID_LSB                       0x300B


#define OV5642_MODEL_ID                       0x5642


#define OV5642_MASTER_CLK_RATE   24000000

#ifdef USE_AE_TRANS
extern struct ov5642_ae_func_array ov5642_1D_ae_func;
#endif



struct ov5642_work_t
{
    struct work_struct work;
};

static struct ov5642_work_t *ov5642_sensorw;
static struct i2c_client *ov5642_client;

#ifdef WB_REWRITE
#define REWRITE_DELAY			4000    
static struct hrtimer reg_timer;
static int reg_timer_value = REWRITE_DELAY;
struct work_struct reg_work;
#endif

static int ov5642_id = 0;
struct ov5642_ctrl_t
{
    const struct msm_camera_sensor_info *sensordata;
    uint32_t sensormode;
    uint32_t fps_divider;       
    uint32_t pict_fps_divider;  
    uint32_t fps;
    int32_t curr_lens_pos;
    uint32_t curr_step_pos;
    uint32_t my_reg_gain;
    uint32_t my_reg_line_count;
    uint32_t total_lines_per_frame;
    enum ov5642_test_mode_t set_test;
    unsigned short imgaddr;
};
static struct ov5642_ctrl_t *ov5642_ctrl;
static int32_t config_csi;

static DECLARE_WAIT_QUEUE_HEAD (ov5642_wait_queue);
DEFINE_MUTEX (ov5642_mut);


static long ov5642_set_white_balance (int8_t whiteBalance, bool redo);
static int ov5642_set_color_effect (int effect, bool redo);
static long ov5642_set_scene_mode (int8_t mode, bool redo);



static int
ov5642_i2c_rxdata (unsigned short saddr, unsigned char *rxdata, int length)
{
    struct i2c_msg msgs[] = {
        {
         .addr = saddr,
         .flags = 0,
         .len = 2,
         .buf = rxdata,
         },
        {
         .addr = saddr,
         .flags = I2C_M_RD,
         .len = 1,
         .buf = rxdata,
         },
    };
    if (i2c_transfer (ov5642_client->adapter, msgs, 2) < 0)
    {
        CDBG ("ov5642_i2c_rxdata failed!\n");
        return -EIO;
    }
    return 0;
}

static int32_t
ov5642_i2c_txdata (unsigned short saddr, unsigned char *txdata, int length)
{
    struct i2c_msg msg[] = {
        {
         .addr = saddr,
         .flags = 0,
         .len = 3,
         .buf = txdata,
         },
    };
    if (i2c_transfer (ov5642_client->adapter, msg, 1) < 0)
    {
        CDBG ("ov5642_i2c_txdata faild 0x%x\n", ov5642_client->addr);
        return -EIO;
    }

    return 0;
}

int32_t
ov5642_i2c_read (unsigned short raddr, uint8_t * rdata, int rlen)
{
    int32_t rc = 0;
    unsigned char buf[2];
    if (!rdata)
        return -EIO;
    memset (buf, 0, sizeof (buf));
    buf[0] = (raddr & 0xFF00) >> 8;
    buf[1] = (raddr & 0x00FF);
    rc = ov5642_i2c_rxdata (ov5642_client->addr, buf, rlen);
    if (rc < 0)
    {
        CDBG ("ov5642_i2c_read 0x%x failed!\n", raddr);
        return rc;
    }
    *rdata = buf[0];
    return rc;
}

int32_t
ov5642_i2c_write_b_sensor (unsigned short waddr, uint8_t bdata)
{
    int32_t rc = -EFAULT;
    unsigned char buf[3];

    static uint8_t reg_mask = 0;
    uint8_t reg_val = 0;

    if (0xFFFF == waddr)
    {
        msleep (bdata);
        return 0;
    }

    if (0xFFFE == waddr)
    {
        reg_mask = bdata;
        return 0;
    }


    if (0 != reg_mask)
    {

        ov5642_i2c_read (waddr, &reg_val, 1);
        reg_val &= (~reg_mask);
        reg_val |= bdata;
        bdata = reg_val;

        reg_mask = 0;
        reg_val = 0;

    }

    memset (buf, 0, sizeof (buf));
    buf[0] = (waddr & 0xFF00) >> 8;
    buf[1] = (waddr & 0x00FF);
    buf[2] = bdata;


    rc = ov5642_i2c_txdata (ov5642_client->addr, buf, 3);
    if (rc < 0)
        CDBG ("i2c_write_b failed, addr = 0x%x, val = 0x%x!\n", waddr, bdata);

    return rc;
}

int32_t
ov5642_i2c_write_b_table (struct ov5642_reg * reg_conf_tbl, int num)
{
    int i;
    int32_t rc = -EIO;
    
    if (reg_conf_tbl->next == NULL)
    {
        for (i = 0; i < num; i++)
        {
            rc = ov5642_i2c_write_b_sensor (reg_conf_tbl->addr,
                                            reg_conf_tbl->val);
            if (rc < 0)
            {
                printk ("i2c_write fail !!!\n");
                break;
            }

            reg_conf_tbl++;
        }
    }
    else
    {
        for (i = 0; i < num; i++)
        {
            rc = ov5642_i2c_write_b_sensor (reg_conf_tbl->addr,
                                            reg_conf_tbl->val);

            if (rc < 0)
            {
                printk ("i2c_write fail !!!\n");
                break;
            }

            reg_conf_tbl = reg_conf_tbl->next;
        }

    }


    
    return rc;
}


#define MAX_RETRY_COUNT 5

static inline int
isValidClient (void *client)
{
    return ((client != NULL) ? 0 : -1);
}

int32_t
ov5642_i2c_burst_write (char const *data, uint32_t size)
{
    int retryCnt;
    int rc;

    struct i2c_msg msg[] = {
        [0] = {
               .addr = OV5642_SLAVE_ADDR,
               .flags = 0,
               .buf = (void *) data,
               .len = size,
               },
    };

    if (isValidClient (ov5642_client))
    {
        CDBG ("ov5642_i2c_burst_write: invalid client\n");
        return -EFAULT;
    }

#ifndef MAX_RETRY_COUNT
    if (i2c_transfer (ov5642_client->adapter, msg, ARRAY_SIZE (msg)) !=
        ARRAY_SIZE (msg))
    {
        CDBG ("ov5642_i2c_burst_write: write %02x%02xh %d bytes failed\n",
              *data, *(data + 1), size);
        return -EIO;
    }
#else
    for (retryCnt = 0; retryCnt < MAX_RETRY_COUNT; retryCnt++)
    {
        rc = i2c_transfer (ov5642_client->adapter, msg, ARRAY_SIZE (msg));
        if (ARRAY_SIZE (msg) == rc)
        {
            if (retryCnt > 0)
                CDBG ("ov5642_i2c_burst_write: write %Xh %d bytes successfully at retryCnt=%d\n", *data, size, retryCnt);
            break;
        }
    }

    if ((rc != ARRAY_SIZE (msg)) && (retryCnt == MAX_RETRY_COUNT))
    {
        CDBG ("ov5642_i2c_burst_write: write %Xh %d bytes failed\n",
              *data, size);
        return -EIO;
    }
#endif

    return 0;
}




#ifdef USE_AE_TRANS
int32_t
ov5642_i2c_read_array (struct ov5642_reg * regs, uint32_t size)
{
    uint32_t i;
    int retryCnt;
    int rc;
    unsigned char buf[2];
    
    struct i2c_msg msg[] = {
        [0] = {
               .addr = OV5642_SLAVE_ADDR,
               
               .flags = 0,
               .buf = (void *) buf,
               .len = 2,
               },
        [1] = {
               .addr = OV5642_SLAVE_ADDR,
               .flags = I2C_M_RD,
               
               .len = 1,
               },
    };

    if (isValidClient (ov5642_client))
    {
        CDBG ("ov5642_i2c_read: invalid client\n");
        return -EFAULT;
    }

    for (i = 0; i < size; i++)
    {
        buf[0] = (regs->addr & 0xFF00) >> 8;
        buf[1] = (regs->addr & 0x00FF);
        msg[1].buf = (void *) &regs->val;

#ifndef MAX_RETRY_COUNT
        if (ARRAY_SIZE (msg) != i2c_transfer (ov5642_client->adapter, msg, 2))
        {
            CDBG ("ov5642_i2c_read: read %Xh failed\n",
                  (buf[0] << 8) | buf[1]);
            return -EIO;
        }
#else
        for (retryCnt = 0; retryCnt < MAX_RETRY_COUNT; retryCnt++)
        {
            rc = i2c_transfer (ov5642_client->adapter, msg, 2);
            if (ARRAY_SIZE (msg) == rc)
            {
                if (retryCnt > 0)
                    CDBG ("ov5642_i2c_read: read %Xh=0x%X successfull at retryCnt=%d\n", (buf[0] << 8) | buf[1], regs->val, retryCnt);
                break;
            }
        }

        if ((rc != ARRAY_SIZE (msg)) && (retryCnt == MAX_RETRY_COUNT))
        {
            CDBG ("ov5642_i2c_read: read %Xh failed\n",
                  (buf[0] << 8) | buf[1]);
            return -EIO;
        }
#endif
        regs++;
    }


    return 0;
}
#endif



extern void msm_camio_csi_core_ahb_reset (void);

static void
reset_and_config_mipi (void)
{
	int32_t rc = 0;
	struct msm_camera_csi_params ov5642_csi_params;
	CDBG ("reset mipi \n");
	msm_camio_csi_core_ahb_reset ();
	msleep (10);
	ov5642_csi_params.lane_cnt = 2;
	ov5642_csi_params.data_format = CSI_8BIT;
	ov5642_csi_params.lane_assign = 0xe4;
	ov5642_csi_params.dpcm_scheme = 0;
	ov5642_csi_params.settle_cnt = 0x18;
       printk("msm_camio_csi_core_ahb_reset end\n");
	rc = msm_camio_csi_config (&ov5642_csi_params);
       printk("msm_camio_csi_config end\n");
}



#ifdef WB_REWRITE
static void
reg_work_func (struct work_struct *work)
{
    printk (KERN_INFO "%s:rewrite register.\n", __func__);

    ov5642_i2c_write_b_sensor (0x5183, 0x14);
}

static enum hrtimer_restart
reg_timer_func (struct hrtimer *timer)
{
    printk (KERN_INFO "%s:rewrite register.\n", __func__);

    schedule_work (&reg_work);

    return HRTIMER_NORESTART;
}
#endif

static int ftd_mode = 0;


static int32_t
ov5642_sensor_setting (int update_type, int rt)
{

    int32_t rc = 0;


    

    if (update_type == REG_INIT)
    {
        printk ("OV5642 Initial ... \n");

        rc = ov5642_i2c_write_b_table
            (ov5642_regs.default_settings, ov5642_regs.default_settings_size);
        if (rc < 0)
            return rc;


        rc = ov5642_i2c_write_b_table
            (ov5642_regs.iq_settings, ov5642_regs.iq_settings_size);
        if (rc < 0)
            return rc;

        msleep (10);
        if (ftd_mode == 1)
        {
            printk ("In Camera FTD Mode \n");
            ov5642_i2c_write_b_sensor (0x5183, 0x94);
        }
        else
        {
            printk ("In Camera normal Mode \n");
        }


#ifdef USE_AF
        ov5642_af_func.init_auto_focus ();
#endif
    }
    else if (update_type == UPDATE_PERIODIC)
    {

        if (rt == RES_PREVIEW)
        {
            printk (" OV5642 Preview !!!!!!!!!!!!!!!!!... \n");
            ov5642_i2c_write_b_sensor (0x4201, 0x01);
            ov5642_i2c_write_b_sensor (0x4202, 0x00);
            msleep (120);
            reset_and_config_mipi ();
            msleep (200);

#ifdef USE_AE_TRANS
            rc = ov5642_1D_ae_func.snapshot_to_preview (ov5642_current_wb);
            if (rc)
                return rc;
#endif

            rc = ov5642_i2c_write_b_table
                (ov5642_regs.preview_settings,
                 ov5642_regs.preview_settings_size);
            if (rc < 0)
                return rc;
	     msleep (10);
	     ov5642_i2c_write_b_sensor (0x4202, 0x01);
	     ov5642_i2c_write_b_sensor (0x4201, 0x00);
	     msleep (50);
	     ov5642_i2c_write_b_sensor (0x4202, 0x00);
	     


            ov5642_set_white_balance (ov5642_current_wb, TRUE);

#ifdef USE_AF
            rc = ov5642_af_func.prepare_auto_focus ();

            if (rc)
                return rc;
#endif
            msleep(150);
#ifdef WB_REWRITE
            hrtimer_start (&reg_timer,
                           ktime_set (reg_timer_value / 1000,
                                      (reg_timer_value % 1000) * 1000000),
                           HRTIMER_MODE_REL);
#endif

        }
        else if (rt == RES_1080P)
        {
            printk (" OV5642 1080P Preview !!!!!!!!!!!!!!!!!... \n");

            rc = ov5642_i2c_write_b_table
                (ov5642_regs.default_settings,
                 ov5642_regs.default_settings_size);
            if (rc < 0)
                return rc;

            rc = ov5642_i2c_write_b_table
                (ov5642_regs.iq_settings, ov5642_regs.iq_settings_size);
            if (rc < 0)
                return rc;

#ifdef USE_AF
            ov5642_af_func.init_auto_focus ();
#endif

            ov5642_i2c_write_b_sensor (0x4201, 0x01);
            ov5642_i2c_write_b_sensor (0x4202, 0x00);
            msleep (50);
            reset_and_config_mipi ();
            msleep (40);

            rc = ov5642_i2c_write_b_table
                (ov5642_regs.preview_settings_1080p,
                 ov5642_regs.preview_settings_1080p_size);
            if (rc < 0)
                return rc;
            msleep (33);
            ov5642_i2c_write_b_sensor (0x4202, 0x01);
            ov5642_i2c_write_b_sensor (0x4201, 0x00);
            msleep (66);
            ov5642_i2c_write_b_sensor (0x4202, 0x00);
            msleep (33);


            ov5642_set_color_effect (ov5642_current_effect, TRUE);
            ov5642_set_white_balance (ov5642_current_wb, TRUE);

#ifdef USE_AF                           
            rc = ov5642_af_func.prepare_auto_focus ();

            if (rc)
                return rc;
#endif
        }
        else
        {
            printk ("OV5642 Snapshot !!!!!!!!!!!!!!!!!... \n");            
            ov5642_i2c_write_b_sensor (0x4201, 0x01);
            msleep (34);
            reset_and_config_mipi ();
            msleep (40);


#ifdef USE_AE_TRANS
            printk ("OV5642 111... \n");
            rc = ov5642_1D_ae_func.preview_to_snapshot ();
            if (rc)
                return rc;
            printk ("OV5642 222... \n");
#endif


            ov5642_i2c_write_b_sensor (0x4202, 0x01);
            ov5642_i2c_write_b_sensor (0x4201, 0x00);
            msleep (66);

        }

    }

    

    return rc;
}

static int32_t
ov5642_video_config (int mode)
{
    int32_t rc = 0;
    int rt;
    

    if (ov5642_ctrl->sensormode != mode)
    {
        switch (mode)
        {
        case SENSOR_VIDEO_1080P_MODE:
            rt = RES_1080P;
            break;
        case SENSOR_PREVIEW_MODE:
            rt = RES_PREVIEW;
            break;
        case SENSOR_SNAPSHOT_MODE:
            rt = RES_CAPTURE;
            break;
        case SENSOR_RAW_SNAPSHOT_MODE:
            rt = RES_CAPTURE;
            break;
        default:
            CDBG ("rt error\n");
            break;
        }
        
        if (ov5642_sensor_setting (UPDATE_PERIODIC, rt) < 0)
            return rc;

        ov5642_ctrl->sensormode = mode;
    }

    return rc;
}

static int32_t
ov5642_set_sensor_mode (int mode, int res)
{
    int32_t rc = 0;

    

    rc = ov5642_video_config (mode);

    return rc;
}

static int32_t
ov5642_power_down (void)
{
    return 0;
}


static long
ov5642_set_scene_mode (int8_t scene, bool redo)
{
    long rc = 0;

    

    

    if (redo == FALSE && ov5642_current_scene == scene)
    {
        return 0;
    };

    switch (scene)
    {
    case OV5642_SCENE_MODE_OFF:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_scene_mode_regs.auto_sm,
                 ov5642_scene_mode_regs.auto_sm_size);
            if (rc < 0)
                return rc;
        }
        break;
    case OV5642_SCENE_MODE_NIGHT:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_scene_mode_regs.night_sm,
                 ov5642_scene_mode_regs.night_sm_size);
            if (rc < 0)
                return rc;
        }
        break;
    case OV5642_SCENE_MODE_PORTRAIT:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_scene_mode_regs.portrait_sm,
                 ov5642_scene_mode_regs.portrait_sm_size);
            if (rc < 0)
                return rc;
        }
        break;
    case OV5642_SCENE_MODE_LANDSCAPE:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_scene_mode_regs.landscape_sm,
                 ov5642_scene_mode_regs.landscape_sm_size);
            if (rc < 0)
                return rc;
        }
        break;
#if 0
    case OV5642_SCENE_MODE_SPORTS:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_scene_mode_regs.sport_sm,
                 ov5642_scene_mode_regs.sport_sm_size);
            if (rc < 0)
                return rc;
        }
        break;
#endif
    case OV5642_SCENE_MODE_SNOW:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_scene_mode_regs.snow_sm,
                 ov5642_scene_mode_regs.snow_sm_size);
            if (rc < 0)
                return rc;
        }
        break;
    default:
        {
            CDBG ("ov5642_set_scene: unsupported scene %d\n", scene);
            rc = -EFAULT;
        }
        break;
    }


    
    ov5642_current_scene = scene;

    
    return rc;
}



static long
ov5642_set_white_balance (int8_t whiteBalance, bool redo)
{
    long rc = 0;

    
    

    if (redo == FALSE && ov5642_current_wb == whiteBalance)
    {
        return 0;
    };

    switch (whiteBalance)
    {
    case OV5642_WB_AUTO:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_white_balance_regs.auto_wb,
                 ov5642_white_balance_regs.auto_wb_size);
            if (rc < 0)
                return rc;
        }
        break;
    case OV5642_WB_DAYLIGHT:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_white_balance_regs.daylight_wb,
                 ov5642_white_balance_regs.daylight_wb_size);
            if (rc < 0)
                return rc;
        }
        break;
    case OV5642_WB_FLUORESCENT:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_white_balance_regs.fluorescent_wb,
                 ov5642_white_balance_regs.fluorescent_wb_size);
            if (rc < 0)
                return rc;
        }
        break;
    case OV5642_WB_INCANDESCENT:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_white_balance_regs.incandescent_wb,
                 ov5642_white_balance_regs.incandescent_wb_size);
            if (rc < 0)
                return rc;
        }
        break;
    case OV5642_WB_CLOUDY:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_white_balance_regs.cloudy_wb,
                 ov5642_white_balance_regs.cloudy_wb_size);
            if (rc < 0)
                return rc;
        }
        break;
    default:
        {
            CDBG ("ov5642_set_wb: unsupported wb %d\n", whiteBalance);
            rc = -EFAULT;
        }
        break;
    }

    ov5642_current_wb = whiteBalance;
    
    return rc;
}




static int
ov5642_set_color_effect (int effect, bool redo)
{

    int32_t rc = 0;
    


    if (redo == FALSE && ov5642_current_effect == effect)
    {
        return 0;
    };

    switch (effect)
    {
    case CAMERA_EFFECT_OFF:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_color_effect_regs.normal_sde,
                 ov5642_color_effect_regs.normal_sde_size);
            if (rc < 0)
                return rc;

        }
        break;

    case CAMERA_EFFECT_MONO:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_color_effect_regs.mono_sde,
                 ov5642_color_effect_regs.mono_sde_size);
            if (rc < 0)
                return rc;
        }
        break;

    case CAMERA_EFFECT_NEGATIVE:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_color_effect_regs.negative_sde,
                 ov5642_color_effect_regs.negative_sde_size);
            if (rc < 0)
                return rc;

        }
        break;
    case CAMERA_EFFECT_SOLARIZE:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_color_effect_regs.solarize_sde,
                 ov5642_color_effect_regs.solarize_sde_size);
            if (rc < 0)
                return rc;

        }
        break;

    case CAMERA_EFFECT_SEPIA:
        {
            rc = ov5642_i2c_write_b_table
                (ov5642_color_effect_regs.sepia_sde,
                 ov5642_color_effect_regs.sepia_sde_size);
            if (rc < 0)
                return rc;

        }
        break;

    default:
        {
            pr_err ("%s() Invalid effect %d!\n", __func__, effect);
            return -EINVAL;
        }
    }

    ov5642_current_effect = effect;
    
    return 0;
}


static int
ov5642_probe_init_sensor (const struct msm_camera_sensor_info *data)
{
    uint8_t model_id_msb, model_id_lsb = 0;
    uint16_t model_id;
    int32_t rc = 0;
    static int firsttime=1;
    FUN_START ();

    if (!rc)
    {
        CDBG ("sensor_reset = %d\n", rc);
	 gpio_set_value (data->sensor_reset, 0);
	 gpio_direction_output (data->sensor_pwd, 0);
	 msleep (5);
	 gpio_set_value (data->sensor_reset, 1);
	 msleep (5);
    }
    if(firsttime)
    {
        rc = ov5642_i2c_write_b_sensor (0x3017, 0x7F);
        if (rc < 0)
            goto init_probe_fail;
        rc = ov5642_i2c_write_b_sensor (0x3018, 0xFC);
        if (rc < 0)
            goto init_probe_fail;

        
        rc = ov5642_i2c_read (REG_OV5642_MODEL_ID_MSB, &model_id_msb, 1);
        if (rc < 0)
            goto init_probe_fail;
        rc = ov5642_i2c_read (REG_OV5642_MODEL_ID_LSB, &model_id_lsb, 1);
        if (rc < 0)
            goto init_probe_fail;

        model_id = (model_id_msb << 8) | ((model_id_lsb & 0x00FF));
        CDBG ("ov5642 model_id = 0x%x, 0x%x, 0x%x\n",
              model_id, model_id_msb, model_id_lsb);

        ov5642_id = model_id;
        
        if (model_id != OV5642_MODEL_ID)
        {
            rc = -ENODEV;
            goto init_probe_fail;
        }


        ov5642_ctrl = kzalloc (sizeof (struct ov5642_ctrl_t), GFP_KERNEL);
        if (!ov5642_ctrl)
        {
            CDBG ("ov5642_init failed!\n");
            rc = -ENOMEM;
        }
        ov5642_ctrl->fps_divider = 1 * 0x00000400;
        ov5642_ctrl->pict_fps_divider = 1 * 0x00000400;
        ov5642_ctrl->set_test = TEST_OFF;
        ov5642_ctrl->sensormode = RES_MAX;

        if (data)
            ov5642_ctrl->sensordata = data;
        firsttime=0;
    
    }
    printk (KERN_INFO " ov5642_probe_init_sensor finishes\n");
    goto init_probe_done;

  init_probe_fail:
    pr_warning (" ov5642_probe_init_sensor fails\n");
  init_probe_done:

    FUN_END ();

    return rc;
}

int
ov5642_sensor_open_init (const struct msm_camera_sensor_info *data)
{
    int32_t rc = 0;
    FUN_START ();
    ov5642_ctrl = kzalloc (sizeof (struct ov5642_ctrl_t), GFP_KERNEL);
    if (!ov5642_ctrl)
    {
        CDBG ("ov5642_init failed!\n");
        rc = -ENOMEM;
        goto init_done;
    }
    ov5642_ctrl->fps_divider = 1 * 0x00000400;
    ov5642_ctrl->pict_fps_divider = 1 * 0x00000400;
    ov5642_ctrl->set_test = TEST_OFF;
    ov5642_ctrl->sensormode = RES_MAX;
    config_csi = 0;

    if (data)
        ov5642_ctrl->sensordata = data;

    

    msm_camio_clk_rate_set (OV5642_MASTER_CLK_RATE);

    rc = ov5642_probe_init_sensor (data);
    if (rc < 0)
    {
        CDBG ("Calling ov5642_sensor_open_init fail\n");
        goto init_fail;
    }

    if (ov5642_sensor_setting (REG_INIT, RES_PREVIEW) < 0)
        goto init_fail;

    ov5642_ctrl->fps = 30 * Q8;

    if (rc < 0)
        goto init_fail;
    else
        goto init_done;

  init_fail:
    CDBG (" ov5642_sensor_open_init fail\n");
    kfree (ov5642_ctrl);
  init_done:
    CDBG ("ov5642_sensor_open_init done\n");
    FUN_END ();
    return rc;
}

static int
ov5642_init_client (struct i2c_client *client)
{
    
    init_waitqueue_head (&ov5642_wait_queue);
    return 0;
}

static const struct i2c_device_id ov5642_i2c_id[] = {
    {"ov5642", 0},
    {}
};

static int
ov5642_i2c_probe (struct i2c_client *client, const struct i2c_device_id *id)
{
    int rc = 0;
    CDBG ("ov5642_i2c_probe called!\n");

    if (!i2c_check_functionality (client->adapter, I2C_FUNC_I2C))
    {
        CDBG ("i2c_check_functionality failed\n");
        goto probe_failure;
    }

    ov5642_sensorw = kzalloc (sizeof (struct ov5642_work_t), GFP_KERNEL);
    if (!ov5642_sensorw)
    {
        CDBG ("kzalloc failed.\n");
        rc = -ENOMEM;
        goto probe_failure;
    }

    i2c_set_clientdata (client, ov5642_sensorw);
    ov5642_init_client (client);
    ov5642_client = client;
    mdelay (50);
    CDBG ("ov5642_i2c_probe success! rc = %d\n", rc);
    return 0;

  probe_failure:
    CDBG ("ov5642_i2c_probe failed! rc = %d\n", rc);
    return rc;
}

static int __exit
ov5642_remove (struct i2c_client *client)
{

    struct ov5642_work_t_t *sensorw = i2c_get_clientdata (client);

    FUN_START ();

#ifdef WB_REWRITE
    hrtimer_cancel (&reg_timer);
    cancel_work_sync (&reg_work);
#endif

    free_irq (client->irq, sensorw);
    ov5642_client = NULL;
    kfree (sensorw);

    FUN_END ();
    return 0;
}

static struct i2c_driver ov5642_i2c_driver = {
    .id_table = ov5642_i2c_id,
    .probe = ov5642_i2c_probe,
    .remove = __exit_p (ov5642_i2c_remove),
    .driver = {
               .name = "ov5642",
               },
};

int
ov5642_sensor_config (void __user * argp)
{
    struct sensor_cfg_data cdata;
    long rc = 0;
    

    if (copy_from_user (&cdata,
                        (void *) argp, sizeof (struct sensor_cfg_data)))
        return -EFAULT;
    mutex_lock (&ov5642_mut);
    CDBG ("ov5642_sensor_config: cfgtype = %d\n", cdata.cfgtype);
    switch (cdata.cfgtype)
    {
    case CFG_SET_MODE:
        rc = ov5642_set_sensor_mode (cdata.mode, cdata.rs);
        break;
    case CFG_PWR_DOWN:
        rc = ov5642_power_down ();
        break;

    case CFG_SET_WB:
        rc = ov5642_set_white_balance (cdata.cfg.effect, FALSE);
        break;

    case CFG_SET_SCENE_MODE:
        rc = ov5642_set_scene_mode (cdata.cfg.effect, FALSE);
        break;

    case CFG_SET_EFFECT:
        rc = ov5642_set_color_effect (cdata.cfg.effect, FALSE);
        break;

    case CFG_PREPARE_SNAPSHOT:
        rc = ov5642_1D_ae_func.prepare_snapshot ();
        break;

#ifdef USE_AF

    case CFG_SET_AUTO_FOCUS:
        rc = ov5642_af_func.detect_auto_focus ();
        CDBG ("CFG_SET_AUTO_FOCUS AAAAA !!!");
        if (rc)
        {
            CDBG ("sorry AF die !");
            break;
        }

        rc = ov5642_af_func.set_auto_focus (&cdata.cfg.af_status);
        CDBG ("CFG_SET_AUTO_FOCUS BBBBB !!!");
        if (!rc)
        {
            if (copy_to_user ((void *) argp,
                              &cdata, sizeof (struct sensor_cfg_data)))
            {
                CDBG ("ov5642_ioctl: copy to user failed\n");
                rc = -EFAULT;
            }
        }


        break;

    case CFG_SET_DEFAULT_FOCUS:

        rc = ov5642_af_func.set_default_focus ();
        CDBG ("set cancel  ccccc !!!");
        break;


#endif

    default:
        rc = -EFAULT;
        break;
    }

    mutex_unlock (&ov5642_mut);
    

    return rc;
}
static int
ov5642_sensor_release (void)
{
    int rc = -EBADF;
    FUN_START ();

#ifdef WB_REWRITE
    hrtimer_cancel (&reg_timer);
    cancel_work_sync (&reg_work);
#endif

    mutex_lock (&ov5642_mut);

#ifdef USE_AF
    ov5642_af_func.deinit_auto_focus ();
#endif

    ov5642_power_down ();

#if 0
    gpio_free (ov5642_ctrl->sensordata->sensor_reset);
    gpio_free (ov5642_ctrl->sensordata->sensor_pwd);
#endif

    

    kfree (ov5642_ctrl);
    ov5642_ctrl = NULL;

    CDBG ("ov5642_release completed\n");
    mutex_unlock (&ov5642_mut);

    FUN_END ();
    return rc;
}

static int
ov5642_sensor_probe (const struct msm_camera_sensor_info *info,
                     struct msm_sensor_ctrl *s)
{
    int rc = 0;

    FUN_START ();

    rc = i2c_add_driver (&ov5642_i2c_driver);
    if (rc < 0 || ov5642_client == NULL)
    {
        rc = -ENOTSUPP;
        goto probe_fail;
    }
    msm_camio_clk_rate_set (OV5642_MASTER_CLK_RATE);
    rc = ov5642_probe_init_sensor (info);
    if (rc < 0)
        goto probe_fail;
    s->s_init = ov5642_sensor_open_init;
    s->s_release = ov5642_sensor_release;
    s->s_config = ov5642_sensor_config;
    s->s_camera_type = BACK_CAMERA_2D;

    s->s_mount_angle = 90;

#ifdef WB_REWRITE
    INIT_WORK (&reg_work, reg_work_func);
    hrtimer_init (&reg_timer, CLOCK_MONOTONIC, HRTIMER_MODE_REL);
    reg_timer.function = reg_timer_func;
#endif

    FUN_END ();
    return rc;

  probe_fail:
    CDBG ("ov5642_sensor_probe: SENSOR PROBE FAILS!\n");
    i2c_del_driver (&ov5642_i2c_driver);

    FUN_END ();
    return rc;
}


static int test_value = 0;

static int
module_setTest (const char *val, struct kernel_param *kp)
{
    int rc;

    rc = param_set_int (val, kp);

    if (test_value == 0)
    {
        ftd_mode = 0;
    }
    else if (test_value == 1)
    {

        ftd_mode = 1;
    }
    else
    {
        printk (KERN_ERR "unknown command = 0x%x\n", test_value);
    }

    return rc;
}

module_param_call (testFile, module_setTest, param_get_long,
                   &test_value, S_IWUSR | S_IRUGO);



static int
__ov5642_probe (struct platform_device *pdev)
{

    int rc;
    FUN_START ();
    rc = msm_camera_drv_start (pdev, ov5642_sensor_probe);
    FUN_END ();
    return rc;
}

static struct platform_driver msm_camera_driver = {
    .probe = __ov5642_probe,
    .driver = {
               .name = "msm_camera_ov5642",
               .owner = THIS_MODULE,
               },
};


static int __init
ov5642_init (void)
{
    int rc;
    FUN_START ();
    rc = platform_driver_register (&msm_camera_driver);
    FUN_END ();

    return rc;
}

module_init (ov5642_init);

module_param (ov5642_id, int, 0664);

void
ov5642_exit (void)
{
    i2c_del_driver (&ov5642_i2c_driver);
}


MODULE_DESCRIPTION ("OMNI VGA YUV sensor driver");
MODULE_LICENSE ("GPL v2");
