
#include "ov5642.h"

#ifdef USE_AF
#include "ov5642_af.h"
#endif


#ifdef USE_AF

static char doAutoFocus;
static wait_queue_head_t waitAutoFocus;
static enum ov5642_af_status_t afStatus;
static struct work_struct af_download_work;
DECLARE_MUTEX (ov5642_af_sem);

static long
ov5642_enable_AF (void)
{
	long int rc;
	FUN_START ();
	rc = ov5642_i2c_write_b_table ((struct ov5642_reg *)
				       ov5642_enable_af,
				       (sizeof (ov5642_enable_af) /
					sizeof (ov5642_enable_af[0])));
	FUN_END ();
	return rc;
}

static long
ov5642_reset_AF (void)
{
	long int rc;
	FUN_START ();
	rc = ov5642_i2c_write_b_table ((struct ov5642_reg *)
				       ov5642_reset_af,
				       ARRAY_SIZE (ov5642_reset_af));
	FUN_END ();
	return rc;
}

static long
ov5642_start_single_AF (void)
{
	long int rc;
	FUN_START ();
	rc = ov5642_i2c_write_b_table ((struct ov5642_reg *)
				       ov5642_single_af,
				       ARRAY_SIZE (ov5642_single_af));
	FUN_END ();
	return rc;
}

static long
ov5642_load_AF (void)
{
	long rc;
	int i;
	uint32_t writeCnt;
	FUN_START ();
	rc = ov5642_i2c_write_b_table ((struct ov5642_reg *)
				       ov5642_download_af,
				       sizeof (ov5642_download_af) /
				       sizeof (ov5642_download_af[0]));
	if (rc)
	{
		CDBG ("Sorry Load AF error!");
		goto exit_load_AF;

	}

	writeCnt = 258;
	for (i = 0; (i + writeCnt - 1) < ARRAY_SIZE (ov5642_autofocus_reg);
	     i += writeCnt)
	{
		CDBG ("ov5642_load_AF: af %x%xh %d bytes\n",
		      ov5642_autofocus_reg[i], ov5642_autofocus_reg[i + 1],
		      writeCnt);
		rc = ov5642_i2c_burst_write (&ov5642_autofocus_reg[i],
					     writeCnt);
	}
	if (i < sizeof (ov5642_autofocus_reg))
	{
		writeCnt = sizeof (ov5642_autofocus_reg) - i;
		CDBG ("ov5642_load_AF: af %d bytes\n", writeCnt);
		rc = ov5642_i2c_burst_write (&ov5642_autofocus_reg[i],
					     writeCnt);
	}

      exit_load_AF:
	FUN_END ();
	return rc;
}


static long
ov5642_config_af (void)
{
	long rc = 0;
	struct ov5642_reg command_reg;
	FUN_START ();
	
	command_reg.addr = OV5642_AF_CMD_TAG;
	command_reg.val = CMD_TAG_VALUE_NOT_ZERO;
	rc |= ov5642_i2c_write_b_table (&command_reg, 1);

	command_reg.addr = OV5642_AF_CMD_MAIN;
	command_reg.val = CMD_MAIN_RETURN_TO_IDLE;
	rc |= ov5642_i2c_write_b_table (&command_reg, 1);


	command_reg.addr = OV5642_AF_STA_FOCUS;
	ov5642_i2c_read (command_reg.addr, &command_reg.val, 1);
	CDBG ("%s: read %04Xh=0x%02X\n", __func__, command_reg.addr,
	      command_reg.val);
	FUN_END ();
	return rc;
}

static long
ov5642_af_ready (void)
{
	long rc;
	struct ov5642_reg status_reg;
	FUN_START ();
	status_reg.addr = OV5642_AF_STA_FOCUS;
	rc = ov5642_i2c_read (status_reg.addr, &status_reg.val, 1);
	if (rc)
		return rc;

	if ((status_reg.val != STA_FOCUS_S_IDLE) &&
	    (status_reg.val != STA_FOCUS_S_FOCUSED))
	{
		CDBG ("%s: status(0x%02X) is not idle or focused, reset it\n",
		      __func__, status_reg.val);
		ov5642_reset_AF ();
		ov5642_enable_AF ();
		rc = -1;
	}
	FUN_END ();
	return rc;
}

static long
ov5642_set_focus (struct auto_focus_status *pStatus)
{
	long rc = 0;

	struct ov5642_reg status_reg;
	int i;
	
	struct timespec focusStart;
	FUN_START ();

	status_reg.addr = OV5642_AF_STA_FOCUS;

	down (&ov5642_af_sem);

	
	doAutoFocus = 1;
	ov5642_config_af ();
	CDBG ("ov5642_set_focus: starting single focus\n");
	rc = ov5642_start_single_AF ();
	if (rc)
		goto not_focused;

	
	focusStart = CURRENT_TIME;

	for (i = 0; i < 20; i++)
	{
		up (&ov5642_af_sem);
		rc = wait_event_interruptible_timeout (waitAutoFocus,
						       !doAutoFocus,
						       msecs_to_jiffies
						       (100));
		down (&ov5642_af_sem);

		if (!doAutoFocus)
		{
			CDBG ("ov5642_set_focus: canceled\n");
			break;
		}

		if (rc < 0)
		{
			CDBG ("ov5642_set_focus: signaled\n");
			break;
		}

		rc = ov5642_i2c_read (status_reg.addr, &status_reg.val, 1);
		if (!rc)
		{
			CDBG ("ov5642_set_focus: status=0x%X(%d)\n",
			      status_reg.val, i);
			
			if (status_reg.val == STA_FOCUS_S_FOCUSING)
				continue;
			
			else if (status_reg.val == STA_FOCUS_S_FOCUSED)
			{

				
				focusStart =
					timespec_sub (CURRENT_TIME,
						      focusStart);
				pStatus->duration =
					(focusStart.tv_sec * 1000) +
					(focusStart.tv_nsec / 1000000);

				
				
				
				pStatus->steps = 0;
				
				

				CDBG ("ov5642_set_focus: focused with %d steps, %d ms, status is %X\n", pStatus->steps, pStatus->duration, status_reg.val);
				rc = 0;
				goto exit_focus;
			}
			
			else
			{
				CDBG ("ov5642_set_focus: exit due to %X\n",
				      status_reg.val);
				break;
			}
		}
		CDBG ("ov5642_set_focus: (i=%d)status is %x\n", i,
		      status_reg.val);
	}

      not_focused:
	ov5642_config_af ();
	rc = -1;
	CDBG ("ov5642_set_focus: not focused\n");

      exit_focus:
	doAutoFocus = 0;
	up (&ov5642_af_sem);
	CDBG ("ov5642_set_focus()<-, rc=%ld\n", rc);
	FUN_END ();
	return rc;
}

static long
ov5642_set_default_focus (void)
{
	long rc = 0;
	FUN_START ();
	down (&ov5642_af_sem);

	
	if (doAutoFocus)
	{
		doAutoFocus = 0;
		wake_up (&waitAutoFocus);
	}
	else
	{
		rc = ov5642_config_af ();
	}

	up (&ov5642_af_sem);

	CDBG ("ov5642_set_default_focus(), rc=%ld\n", rc);
	FUN_END ();
	return rc;
}

static void
ov5642_af_download_func (struct work_struct *work)
{
#ifdef CHECK_PERF
	struct timespec start;

	start = CURRENT_TIME;
#endif
	FUN_START ();
	ov5642_load_AF ();
	ov5642_enable_AF ();

	down (&ov5642_af_sem);
	afStatus = IDLE_STATUS;
	up (&ov5642_af_sem);
	FUN_END ();
#ifdef CHECK_PERF
	start = timespec_sub (CURRENT_TIME, start);
	CDBG ("ov5642_af_download_func: %ld ms\n",
	      ((long) start.tv_sec * 1000) +
	      ((long) start.tv_nsec / 1000000));
#endif
}

static long
ov5642_detect_AF (void)
{
	long rc = 0;
	FUN_START ();
	down (&ov5642_af_sem);

	if (IDLE_STATUS == afStatus)
	{
		rc = ov5642_af_ready ();
	}
	else
	{
		rc = -1;
	}

	up (&ov5642_af_sem);
	FUN_END ();
	return rc;
}

static long
ov5642_prepare_AF (void)
{
	long rc = 0;
	FUN_START ();
	down (&ov5642_af_sem);

	if (UNKNOWN_STATUS == afStatus)
	{
		afStatus = DOWNLOADING_STATUS;
		schedule_work (&af_download_work);
	}
	else if (IDLE_STATUS == afStatus)
	{
		rc |= ov5642_reset_AF ();
		rc |= ov5642_enable_AF ();
	}

	up (&ov5642_af_sem);
	FUN_END ();
	return rc;
}

static void
ov5642_init_AF (void)
{
	FUN_START ();
	down (&ov5642_af_sem);
	afStatus = UNKNOWN_STATUS;
	INIT_WORK (&af_download_work, ov5642_af_download_func);
	init_waitqueue_head (&waitAutoFocus);
	up (&ov5642_af_sem);
	FUN_END ();
}

static void
ov5642_deinit_AF (void)
{
	FUN_START ();
	cancel_work_sync (&af_download_work);
	down (&ov5642_af_sem);
	if (doAutoFocus)
	{
		doAutoFocus = 0;
		wake_up (&waitAutoFocus);
	}
	up (&ov5642_af_sem);
	FUN_END ();
}

struct ov5642_af_func_array ov5642_af_func = {
	.set_default_focus = ov5642_set_default_focus,
	.set_auto_focus = ov5642_set_focus,
	.init_auto_focus = ov5642_init_AF,
	.prepare_auto_focus = ov5642_prepare_AF,
	.detect_auto_focus = ov5642_detect_AF,
	.deinit_auto_focus = ov5642_deinit_AF,
};
#endif 
