#include "parse_header.h"

extern int32_t ov5642_i2c_read (unsigned short raddr, uint8_t * rdata,
                                int rlen);
extern int32_t mt9m114_i2c_read (unsigned short raddr,
                                 unsigned short *rdata, int rlen);

#define TUNING_SIZE 0x50000     

#define MY_MAGIC 'm'
#define DEVICE_IOC_READ _IOR(MY_MAGIC, 1, char *)
#define DEVICE_IOC_WRITE _IOW(MY_MAGIC, 2, char *)
#define DEVICE_IOC_PARSE_OV5642 _IO(MY_MAGIC, 3)
#define DEVICE_IOC_PARSE_MT9M114 _IO(MY_MAGIC, 4)
#define DEVICE_IOC_READ_OV5642_I2C _IOR(MY_MAGIC, 5, char *)
#define DEVICE_IOC_WRITE_I2C _IOW(MY_MAGIC, 6, char *)

#define DEVICE_IOC_READ_MT9M114_W_I2C _IOR(MY_MAGIC, 7,char *)
#define DEVICE_IOC_READ_MT9M114_B_I2C _IOR(MY_MAGIC, 8,char *)

#define TUNING_MAJOR 210


#define TUNING_NAME "tuning"

static int tuning_major = TUNING_MAJOR;

static int tuning_open (struct inode *inode, struct file *filp);
static int tuning_release (struct inode *inode, struct file *filp);
static int tuning_ioctl (struct inode *inodep, struct file *filp,
                         unsigned int cmd, unsigned long arg);

static const struct file_operations my_miscdevice_fops = {
    .owner = THIS_MODULE,
    .ioctl = tuning_ioctl,
    .open = tuning_open,
    .release = tuning_release,
};


static struct miscdevice miscdev = {
    .name = TUNING_NAME,
    .fops = &my_miscdevice_fops,
};


struct tuning_dev
{
    struct miscdevice *misc_dev;
    unsigned char mem[TUNING_SIZE];
    unsigned char reg_array[10];
};

struct tuning_dev *tuning_devp;

static struct tuning_dev my_misc_dev = {
    .misc_dev = &miscdev,
};




int
tuning_open (struct inode *inode, struct file *filp)
{

    filp->private_data = tuning_devp;
    printk (KERN_INFO "Can open!\n");
    return 0;
}


int
tuning_release (struct inode *inode, struct file *filp)
{
    printk (KERN_INFO "Can release!\n");
    return 0;
}


static int
tuning_ioctl (struct inode *inodep, struct file *filp, unsigned int cmd,
              unsigned long arg)
{
    struct tuning_dev *dev = filp->private_data;

    int ret = 2;

    printk (KERN_INFO "You are here now, welcome!\n");

    switch (cmd)
    {
    case DEVICE_IOC_WRITE:
        printk (KERN_INFO "We will write user buf to kernel memory.\n");
        if (filp->f_pos == TUNING_SIZE)
        {
            printk (KERN_INFO "No space for write!\n");
            return -1;
        }
        memset (dev->mem, 0, TUNING_SIZE);
        if (copy_from_user (dev->mem, (void __user *) arg, TUNING_SIZE))
        {
            printk (KERN_INFO "Failed to copy data to kernel memory!\n");
            ret = -EFAULT;
        }

        break;

    case DEVICE_IOC_WRITE_I2C:
        printk (KERN_INFO "We will write user buf to kernel memory.\n");

        memset (dev->reg_array, 0, 10);
        if (copy_from_user (dev->reg_array, (void __user *) arg, 10))
        {
            printk (KERN_INFO "Failed to copy data to kernel memory!\n");
            ret = -EFAULT;
        }
        printk (KERN_INFO "%s\n", dev->reg_array);
        break;

    case DEVICE_IOC_PARSE_OV5642:

        printk (KERN_INFO "OV5642 === length 0x%X\n", strlen (dev->mem));
        parse_tuning_file_ov5642 (dev->mem);

        break;

#if 1
    case DEVICE_IOC_PARSE_MT9M114:

        printk (KERN_INFO "MT9M114 === length 0x%X\n", strlen (dev->mem));
        parse_tuning_file_mt9m114 (dev->mem);

        break;
#endif

    case DEVICE_IOC_READ:
        printk (KERN_INFO "We will read kernel memory to user buf.\n");

        if (copy_to_user ((void __user *) arg, dev->mem, TUNING_SIZE))
        {
            printk (KERN_INFO "Failed to copy data to user space!\n");
            ret = -EFAULT;
        }
        printk (KERN_INFO "%s\n", dev->mem);
        break;

    case DEVICE_IOC_READ_OV5642_I2C:
        {
            unsigned short reg_addr = 0xFFFF;
            uint8_t reg_val = 0xFF;

            reg_addr = simple_strtoul (trim (dev->reg_array), 0, 16);
            ov5642_i2c_read (reg_addr, &reg_val, 1);
            sprintf (dev->reg_array, "%02X", reg_val);
            if (copy_to_user ((void __user *) arg, dev->reg_array, 10))
            {
                printk (KERN_INFO "Failed to copy data to user space!\n");
                ret = -EFAULT;
            }

            printk (KERN_INFO "\t READ_TEST \t 0X%s = 0X%X\n", dev->reg_array,
                    reg_val);

        }
        break;

    case DEVICE_IOC_READ_MT9M114_W_I2C:
        {
            unsigned short reg_addr = 0xFFFF;
            unsigned short reg_val = 0xFFFF;

            reg_addr = simple_strtoul (trim (dev->reg_array), 0, 16);
            mt9m114_i2c_read (reg_addr, &reg_val, 2);
            sprintf (dev->reg_array, "%04X", reg_val);
            if (copy_to_user ((void __user *) arg, dev->reg_array, 10))
            {
                printk (KERN_INFO "Failed to copy data to user space!\n");
                ret = -EFAULT;
            }

            printk (KERN_INFO "\t READ_TEST \t 0X%s = 0X%X\n", dev->reg_array,
                    reg_val);

        }
        break;

    case DEVICE_IOC_READ_MT9M114_B_I2C:
        {
            unsigned short reg_addr = 0xFFFF;
            unsigned short reg_val = 0xFF;

            reg_addr = simple_strtoul (trim (dev->reg_array), 0, 16);
            mt9m114_i2c_read (reg_addr, &reg_val, 1);
            sprintf (dev->reg_array, "%02X", reg_val);
            if (copy_to_user ((void __user *) arg, dev->reg_array, 10))
            {
                printk (KERN_INFO "Failed to copy data to user space!\n");
                ret = -EFAULT;
            }

            printk (KERN_INFO "\t READ_TEST \t 0X%s = 0X%X\n", dev->reg_array,
                    reg_val);

        }
        break;

    default:
        printk (KERN_INFO "Error command!\n");
        return -EINVAL;
    }

    printk (KERN_INFO "Goodbye, see you next time!\n");
    ret = 0;
    return ret;
}





int
tuning_init (void)
{

    int result = -1, ret = -1;
    tuning_devp = kmalloc (sizeof (struct tuning_dev), GFP_KERNEL);
    if (!tuning_devp)
    {

        printk (KERN_INFO "kmalloc tuning_devp error!!!");
        result = -ENOMEM;
    }
    else
        printk (KERN_INFO "kmalloc tuning_devp sucessfully!!!");

    memset (my_misc_dev.mem, 0, sizeof (my_misc_dev.mem));
    memset (my_misc_dev.reg_array, 0, sizeof (my_misc_dev.reg_array));

    ret = misc_register (my_misc_dev.misc_dev);
    return ret;
}


void
tuning_exit (void)
{
    int ret;

    ret = misc_deregister (my_misc_dev.misc_dev);
    if (ret)
    {
        printk (KERN_DEBUG "I am going out now \n");
    }
}

MODULE_AUTHOR ("Adil.Zhu");
MODULE_LICENSE ("Dual BSD/GPL");

module_param (tuning_major, int, S_IRUGO);

module_init (tuning_init);
module_exit (tuning_exit);
