/*
 $License:
    Copyright (C) 2010 InvenSense Corporation, All Rights Reserved.
 $
 */

/*
 *  @defgroup   ACCELDL (Motion Library - Accelerometer Driver Layer)
 *  @brief      Provides the interface to setup and handle an accelerometers
 *              connected to the secondary I2C interface of the gyroscope.
 *
 *  @{
 *      @file   bma222.c
 *      @brief  Accelerometer setup and handling methods.
 */

/* ------------------ */
/* - Include Files. - */
/* ------------------ */
#include <string.h>
#include <linux/slab.h>

#ifdef __KERNEL__
#include <linux/module.h>
#endif

#include "mpu.h"
#include "mlsl.h"
#include "mlos.h"

#define ACCEL_BMA222_RANGE_REG          (0x0F)
#define ACCEL_BMA222_BW_REG             (0x10)
#define ACCEL_BMA222_SUSPEND_REG        (0x11)
#define ACCEL_BMA222_SFT_RST_REG        (0x14)
#define BMA250_SELF_TEST_REG            (0x32)

#ifndef ABS
#define ABS(x) (((x)>=0)?(x):-(x))
#endif


/*********************************************
    Accelerometer Initialization Functions
**********************************************/

static int bma222_suspend(void *mlsl_handle,
			  struct ext_slave_descr *slave,
			  struct ext_slave_platform_data *pdata)
{
	int result = ML_SUCCESS;

	result =
	    MLSLSerialWriteSingle(mlsl_handle, pdata->address,
				  ACCEL_BMA222_SUSPEND_REG, 0x80);
	ERROR_CHECK(result);

	return result;
}

static int bma222_resume(void *mlsl_handle,
			 struct ext_slave_descr *slave,
			 struct ext_slave_platform_data *pdata)
{
	int result = ML_SUCCESS;
	unsigned char reg = 0;

	/* Soft reset */
	result =
	    MLSLSerialWriteSingle(mlsl_handle, pdata->address,
				  ACCEL_BMA222_SFT_RST_REG, 0xB6);
	ERROR_CHECK(result);
	MLOSSleep(10);

	/*Bandwidth */
	result =
	    MLSLSerialWriteSingle(mlsl_handle, pdata->address,
				  ACCEL_BMA222_BW_REG, 0x0C);
	ERROR_CHECK(result);

	/* Full Scale */
	if (slave->range.mantissa == 4)
		reg |= 0x05;
	else if (slave->range.mantissa == 8)
		reg |= 0x08;
	else if (slave->range.mantissa == 16)
		reg |= 0x0C;
	else {
		slave->range.mantissa = 2;
		reg |= 0x03;
	}
	slave->range.fraction = 0;

	result =
	    MLSLSerialWriteSingle(mlsl_handle, pdata->address,
				  ACCEL_BMA222_RANGE_REG, reg);
	ERROR_CHECK(result);

	return result;
}

int bma222_read_memory(void *mlsl_handle,
               struct ext_slave_descr *slave, 
               struct ext_slave_platform_data *pdata,
               struct ext_slave_config *config)
{
    int result = ML_SUCCESS;

    result = MLSLSerialRead(mlsl_handle,
							pdata->address, 
                            (unsigned char)config->key,
                            (unsigned short)config->len,
                          	(unsigned char*)config->data );  
    ERROR_CHECK(result);

	return result;
}

int bma222_config(void *mlsl_handle,
		       struct ext_slave_descr *slave,
		       struct ext_slave_platform_data *pdata,
		       struct ext_slave_config *config)
{
    int result = ML_SUCCESS;
	unsigned char *tmp_data;

	tmp_data = kzalloc(config->len, GFP_KERNEL);
	if (!tmp_data)
		return ML_ERROR;
	
	memcpy(tmp_data, config->data, config->len);		
	memcpy(config->data, &config->key, 1);
	memcpy(config->data+1, tmp_data, config->len);	

	result =MLSLSerialWrite( mlsl_handle, pdata->address,
                          (unsigned short)config->len + 1,
                          (unsigned char*)config->data );                          
                            
    ERROR_CHECK(result);
	kfree(tmp_data);

	return result;
}

int gsensor_test_x(void *mlsl_handle,
               struct ext_slave_descr *slave, 
               struct ext_slave_platform_data *pdata)
{
    int result = ML_SUCCESS;
	unsigned char data[2];
	unsigned short result_x_1 = 0, result_x_2 = 0;	
	unsigned short temp = 0;	
	int i = 0;
	
	/*printk("Eric: ========gsensor_test_x+========\n");*/
	result =
	    MLSLSerialWriteSingle(mlsl_handle, pdata->address,
				  BMA250_SELF_TEST_REG, 0x01);
	ERROR_CHECK(result);
	MLOSSleep(10);	

	for(i=0; i<20; i++)
	{
    	result = MLSLSerialRead(mlsl_handle,
							pdata->address, 
                            0x02,
                            2,
                          	data);  
		ERROR_CHECK(result);		
		/*printk("Eric: data0 = %x, data1 = %x\n", data[0], data[1]);*/
		temp = (data[1]* 4) + (data[0]>>6);	
		/*printk("Eric: data = %x\n", temp);		*/
		result_x_1 += temp;
	}
	result_x_1 = result_x_1 / 20;
	/*printk("Eric: result_x_1 = %x\n", result_x_1);	*/

	/*printk("Eric: ========gsensor_test_x-========\n");*/
	result =
	    MLSLSerialWriteSingle(mlsl_handle, pdata->address,
				  BMA250_SELF_TEST_REG, 0x05);
	ERROR_CHECK(result);
	MLOSSleep(10);	

	for(i=0; i<20; i++)
	{
    	result = MLSLSerialRead(mlsl_handle,
							pdata->address, 
                            0x02,
                            2,
                          	data);  
		ERROR_CHECK(result);		
		/*printk("Eric: data0 = %x, data1 = %x\n", data[0], data[1]);*/
		temp = (data[1]* 4) + (data[0]>>6);	
		/*printk("Eric: data = %x\n", temp);		*/
		result_x_2 += temp;

	}
	result_x_2 = result_x_2 / 20;	
	/*printk("Eric: result_x_2 = %x\n", result_x_2);*/

	if(ABS(result_x_1-result_x_2) >= 200)
		return 1;
	else 
		return 0;
}

int gsensor_test_y(void *mlsl_handle,
               struct ext_slave_descr *slave, 
               struct ext_slave_platform_data *pdata)
{
    int result = ML_SUCCESS;
	unsigned char data[2];
	unsigned short result_y_1 = 0, result_y_2 = 0;	
	unsigned short temp = 0;
	int i = 0;

	/*printk("Eric: ========gsensor_test_y+========\n");*/

	result =
	    MLSLSerialWriteSingle(mlsl_handle, pdata->address,
				  BMA250_SELF_TEST_REG, 0x02);
	ERROR_CHECK(result);
	MLOSSleep(10);	

	for(i=0; i<20; i++)
	{
    	result = MLSLSerialRead(mlsl_handle,
							pdata->address, 
                            0x04,
                            2,
                          	data);  
		ERROR_CHECK(result);		
		/*printk("Eric: data0 = %x, data1 = %x\n", data[0], data[1]);*/
		temp = (data[1]* 4) + (data[0]>>6);	
		/*printk("Eric: data = %x\n", temp);		*/
		result_y_1 += temp;

	}
	result_y_1 = result_y_1 / 20;
	/*printk("Eric: result_y_1 = %x\n", result_y_1);	*/

	/*printk("Eric: ========gsensor_test_y-========\n");*/
	result =
	    MLSLSerialWriteSingle(mlsl_handle, pdata->address,
				  BMA250_SELF_TEST_REG, 0x06);
	ERROR_CHECK(result);
	MLOSSleep(10);	

	for(i=0; i<20; i++)
	{
    	result = MLSLSerialRead(mlsl_handle,
							pdata->address, 
                            0x04,
                            2,
                          	data);  
		ERROR_CHECK(result);		
		/*printk("Eric: data0 = %x, data1 = %x\n", data[0], data[1]);*/
		temp = (data[1]* 4) + (data[0]>>6);	
		/*printk("Eric: data = %x\n", temp);		*/
		result_y_2 += temp;

	}
	result_y_2 = result_y_2 / 20;	
	/*printk("Eric: result_y_2 = %x\n", result_y_2);	*/

	if(ABS(result_y_1-result_y_2) >= 200)
		return 2;
	else 
		return 0;
}

int gsensor_test_z(void *mlsl_handle,
               struct ext_slave_descr *slave, 
               struct ext_slave_platform_data *pdata)
{
    int result = ML_SUCCESS;
	unsigned char data[2];
	unsigned short result_z_1 = 0, result_z_2 = 0;	
	unsigned short temp = 0;

	int i = 0;

	/*printk("Eric: ========gsensor_test_z+========\n");*/
	result =
	    MLSLSerialWriteSingle(mlsl_handle, pdata->address,
				  BMA250_SELF_TEST_REG, 0x03);
	ERROR_CHECK(result);
	MLOSSleep(10);	

	for(i=0; i<20; i++)
	{
    	result = MLSLSerialRead(mlsl_handle,
							pdata->address, 
                            0x06,
                            2,
                          	data);  
		ERROR_CHECK(result);		
		/*printk("Eric: data0 = %x, data1 = %x\n", data[0], data[1]);*/
		temp = (data[1]* 4) + (data[0]>>6);	
		/*printk("Eric: data = %x\n", temp);		*/
		result_z_1 += temp;

	}
	result_z_1 = result_z_1 / 20;
	/*printk("Eric: result_z_1 = %x\n", result_z_1);	*/

	/*printk("Eric: ========gsensor_test_z-========\n");*/
	result =
	    MLSLSerialWriteSingle(mlsl_handle, pdata->address,
				  BMA250_SELF_TEST_REG, 0x07);
	ERROR_CHECK(result);
	MLOSSleep(10);	

	for(i=0; i<20; i++)
	{
    	result = MLSLSerialRead(mlsl_handle,
							pdata->address, 
                            0x06,
                            2,
                          	data);  
		ERROR_CHECK(result);		
		/*printk("Eric: data0 = %x, data1 = %x\n", data[0], data[1]);*/
		temp = (data[1]* 4) + (data[0]>>6);	
		/*printk("Eric: data = %x\n", temp);		*/
		result_z_2 += temp;

	}
	result_z_2 = result_z_2 / 20;	
	/*printk("Eric: result_z_2 = %x\n", result_z_2);	*/

	if(ABS(result_z_1-result_z_2) >= 100)
		return 4;
	else 
		return 0;
}

int bma222_selftest(void *mlsl_handle,
               struct ext_slave_descr *slave, 
               struct ext_slave_platform_data *pdata,
			   unsigned char *pass)

{
    int result = ML_SUCCESS;
	unsigned char test_result;
	unsigned char x_pass, y_pass, z_pass;

	/* Soft reset */
	result =
	    MLSLSerialWriteSingle(mlsl_handle, pdata->address,
				  ACCEL_BMA222_SFT_RST_REG, 0xB6);
	ERROR_CHECK(result);
	MLOSSleep(10);

	/*Range */
	result =
	    MLSLSerialWriteSingle(mlsl_handle, pdata->address,
				  ACCEL_BMA222_RANGE_REG, 0x03);
	ERROR_CHECK(result);

	/*BandWidth */
	result =
	    MLSLSerialWriteSingle(mlsl_handle, pdata->address,
				  ACCEL_BMA222_BW_REG, 0x1f);
	ERROR_CHECK(result);

	x_pass = gsensor_test_x(mlsl_handle, slave, pdata);
	y_pass = gsensor_test_y(mlsl_handle, slave, pdata);
	z_pass = gsensor_test_z(mlsl_handle, slave, pdata);	
	printk("x_pass: %d | y_pass: %d | z_pass: %d\n", x_pass, y_pass, z_pass);
	test_result = x_pass | y_pass | z_pass;
	memcpy(pass, &test_result, 1);
	/*printk("Eric: test_result = %x\n", test_result);*/
	
	result =
	    MLSLSerialWriteSingle(mlsl_handle, pdata->address,
				  BMA250_SELF_TEST_REG, 0x00);
    ERROR_CHECK(result);
	/* Soft reset */
	result =
	    MLSLSerialWriteSingle(mlsl_handle, pdata->address,
				  ACCEL_BMA222_SFT_RST_REG, 0xB6);
	ERROR_CHECK(result);

	return result;
}


static int bma222_read(void *mlsl_handle,
		       struct ext_slave_descr *slave,
		       struct ext_slave_platform_data *pdata,
		       unsigned char *data)
{
	return ML_ERROR_FEATURE_NOT_IMPLEMENTED;
}
static struct ext_slave_descr bma222_descr = {
	/*.init             = */ NULL,
	/*.exit             = */ NULL,
	/*.suspend          = */ bma222_suspend,
	/*.resume           = */ bma222_resume,
	/*.read             = */ bma222_read,
	/*.config           = */ bma222_config,
	/*.name             = */ "bma222",
	/*.type             = */ EXT_SLAVE_TYPE_ACCELEROMETER,
	/*.id               = */ ACCEL_ID_BMA222,
	/*.reg              = */ 0x02,
	/*.len              = */ 6,
	/*.endian           = */ EXT_SLAVE_LITTLE_ENDIAN,
	/*.range            = */ {2, 0},
  	/*.read_memory		= */bma222_read_memory,
   	/*.selftest		= */bma222_selftest
};

struct ext_slave_descr *bma222_get_slave_descr(void)
{
	return &bma222_descr;
}
EXPORT_SYMBOL(bma222_get_slave_descr);

/*
 *  @}
 */
