/* Copyright (c) 2010-2011, Code Aurora Forum. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/pmic8058-vibrator.h>
#include <linux/mfd/pmic8058.h>
#include <linux/pm.h>
#include <linux/pm_runtime.h>
#include <linux/slab.h>

#include <linux/fs.h>
#include <linux/uaccess.h>
#include <linux/miscdevice.h>

#include <linux/regulator/consumer.h>
#include <linux/regulator/pmic8058-regulator.h>
#include <linux/regulator/pmic8901-regulator.h>

#include <linux/smb136_charge.h>
#include <mach/mpp.h>
#include <linux/gpio/gpio_def.h>
#ifdef CONFIG_BATTERY_QSD
#include <mach/qsd_battery.h>
#endif



extern unsigned int get_mfg_mode(void);



#include "../staging/android/timed_output.h"

#define VIB_DRV			0x4A

#define VIB_DRV_SEL_MASK	0xf8
#define VIB_DRV_SEL_SHIFT	0x03
#define VIB_DRV_EN_MANUAL_MASK	0xfc

#define VIB_MAX_LEVEL_mV	3100
#define VIB_MIN_LEVEL_mV	1200

struct pmic8058_vib {
	struct hrtimer vib_timer;
	struct timed_output_dev timed_dev;
	spinlock_t lock;
	struct work_struct work;

	struct device *dev;
	struct pmic8058_vibrator_pdata *pdata;
	int state;
	int level;
	u8  reg_vib_drv;

	struct pm8058_chip	*pm_chip;
};

/* REVISIT: just for debugging, will be removed in final working version */
static void __dump_vib_regs(struct pmic8058_vib *vib, char *msg)
{
	u8 temp;

	dev_dbg(vib->dev, "%s\n", msg);

	pm8058_read(vib->pm_chip, VIB_DRV, &temp, 1);
	dev_dbg(vib->dev, "VIB_DRV - %X\n", temp);
}

static int pmic8058_vib_read_u8(struct pmic8058_vib *vib,
				 u8 *data, u16 reg)
{
	int rc;

	rc = pm8058_read(vib->pm_chip, reg, data, 1);
	if (rc < 0)
		dev_warn(vib->dev, "Error reading pmic8058: %X - ret %X\n",
				reg, rc);

	return rc;
}

static int pmic8058_vib_write_u8(struct pmic8058_vib *vib,
				 u8 data, u16 reg)
{
	int rc;

	rc = pm8058_write(vib->pm_chip, reg, &data, 1);
	if (rc < 0)
		dev_warn(vib->dev, "Error writing pmic8058: %X - ret %X\n",
				reg, rc);
	return rc;
}

extern int get_Battery_valid(void);
static int pmic8058_vib_set(struct pmic8058_vib *vib, int on)
{
	int rc;
	u8 val;


	if( (3 == get_mfg_mode()) || (!get_Battery_valid()) )
		return 0;

	
	if (on) {
		rc = pm_runtime_resume(vib->dev);
		if (rc < 0)
			dev_dbg(vib->dev, "pm_runtime_resume failed\n");

		val = vib->reg_vib_drv;
		val |= ((vib->level << VIB_DRV_SEL_SHIFT) & VIB_DRV_SEL_MASK);
		rc = pmic8058_vib_write_u8(vib, val, VIB_DRV);
		if (rc < 0)
			return rc;
		vib->reg_vib_drv = val;
	} else {
		val = vib->reg_vib_drv;
		val &= ~VIB_DRV_SEL_MASK;
		rc = pmic8058_vib_write_u8(vib, val, VIB_DRV);
		if (rc < 0)
			return rc;
		vib->reg_vib_drv = val;

		rc = pm_runtime_suspend(vib->dev);
		if (rc < 0)
			dev_dbg(vib->dev, "pm_runtime_suspend failed\n");
	}
	__dump_vib_regs(vib, "vib_set_end");

	return rc;
}

static void pmic8058_vib_enable(struct timed_output_dev *dev, int value)
{
	struct pmic8058_vib *vib = container_of(dev, struct pmic8058_vib,
					 timed_dev);
	unsigned long flags;

retry:
	spin_lock_irqsave(&vib->lock, flags);
	if (hrtimer_try_to_cancel(&vib->vib_timer) < 0) {
		spin_unlock_irqrestore(&vib->lock, flags);
		cpu_relax();
		goto retry;
	}

	if (value == 0)
		vib->state = 0;
	else {
		value = (value > vib->pdata->max_timeout_ms ?
				 vib->pdata->max_timeout_ms : value);
		vib->state = 1;
		hrtimer_start(&vib->vib_timer,
			      ktime_set(value / 1000, (value % 1000) * 1000000),
			      HRTIMER_MODE_REL);
	}
	spin_unlock_irqrestore(&vib->lock, flags);
	schedule_work(&vib->work);
}

static void pmic8058_vib_update(struct work_struct *work)
{
	struct pmic8058_vib *vib = container_of(work, struct pmic8058_vib,
					 work);

	pmic8058_vib_set(vib, vib->state);
}

static int pmic8058_vib_get_time(struct timed_output_dev *dev)
{
	struct pmic8058_vib *vib = container_of(dev, struct pmic8058_vib,
					 timed_dev);

	if (hrtimer_active(&vib->vib_timer)) {
		ktime_t r = hrtimer_get_remaining(&vib->vib_timer);
		return r.tv.sec * 1000 + r.tv.nsec / 1000000;
	} else
		return 0;
}

static enum hrtimer_restart pmic8058_vib_timer_func(struct hrtimer *timer)
{
	struct pmic8058_vib *vib = container_of(timer, struct pmic8058_vib,
					 vib_timer);
	vib->state = 0;
	schedule_work(&vib->work);
	return HRTIMER_NORESTART;
}

#ifdef CONFIG_PM
static int pmic8058_vib_suspend(struct device *dev)
{
	struct pmic8058_vib *vib = dev_get_drvdata(dev);

	hrtimer_cancel(&vib->vib_timer);
	cancel_work_sync(&vib->work);
	/* turn-off vibrator */
	pmic8058_vib_set(vib, 0);
	return 0;
}

static struct dev_pm_ops pmic8058_vib_pm_ops = {
	.suspend = pmic8058_vib_suspend,
};
#endif




#define VIB_KER_TEST
#define VREG_KER_TEST

#ifdef VREG_KER_TEST
struct name_id_table{
char name[32];
int vreg_id;
};

#define VREG_DESCRIB(_id, _name) \
	[_id] = { \
		.vreg_id = _id, \
		.name = _name, \
	}

struct name_id_table pm8901_DESCRIB[] =
{
	VREG_DESCRIB(PM8901_VREG_ID_L0, "8901_l0"),
	VREG_DESCRIB(PM8901_VREG_ID_L1, "8901_l1"),
	VREG_DESCRIB(PM8901_VREG_ID_L2, "8901_l2"),
	VREG_DESCRIB(PM8901_VREG_ID_L3, "8901_l3"),
	VREG_DESCRIB(PM8901_VREG_ID_L4, "8901_l4"),
	VREG_DESCRIB(PM8901_VREG_ID_L5, "8901_l5"),
	VREG_DESCRIB(PM8901_VREG_ID_L6, "8901_l6"),

	VREG_DESCRIB(PM8901_VREG_ID_S0, "8901_s0"),
	VREG_DESCRIB(PM8901_VREG_ID_S1, "8901_s1"),
	VREG_DESCRIB(PM8901_VREG_ID_S2, "8901_s2"),
	VREG_DESCRIB(PM8901_VREG_ID_S3, "8901_s3"),
	VREG_DESCRIB(PM8901_VREG_ID_S4, "8901_s4"),
	
	VREG_DESCRIB(PM8901_VREG_ID_MPP0,     "8901_mpp0"),

	VREG_DESCRIB(PM8901_VREG_ID_LVS0,     "8901_lvs0"),
	VREG_DESCRIB(PM8901_VREG_ID_LVS1,     "8901_lvs1"),
	VREG_DESCRIB(PM8901_VREG_ID_LVS2,     "8901_lvs2"),
	VREG_DESCRIB(PM8901_VREG_ID_LVS3,     "8901_lvs3"),
	
	VREG_DESCRIB(PM8901_VREG_ID_MVS0,     "8901_mvs0"),
	
	VREG_DESCRIB(PM8901_VREG_ID_USB_OTG,  "8901_usb_otg"),
	VREG_DESCRIB(PM8901_VREG_ID_HDMI_MVS, "8901_hdmi_mvs"),
};



static struct name_id_table pm8058_DESCRIB[] = {
	VREG_DESCRIB(PM8058_VREG_ID_L0,  "8058_l0"),
	VREG_DESCRIB(PM8058_VREG_ID_L1,  "8058_l1"),
	VREG_DESCRIB(PM8058_VREG_ID_L2,  "8058_l2"),
	VREG_DESCRIB(PM8058_VREG_ID_L3,  "8058_l3"),
	VREG_DESCRIB(PM8058_VREG_ID_L4,  "8058_l4"),
	VREG_DESCRIB(PM8058_VREG_ID_L5,  "8058_l5"),
	VREG_DESCRIB(PM8058_VREG_ID_L6,  "8058_l6"),
	VREG_DESCRIB(PM8058_VREG_ID_L7,  "8058_l7"),
	VREG_DESCRIB(PM8058_VREG_ID_L8,  "8058_l8"),
	VREG_DESCRIB(PM8058_VREG_ID_L9,  "8058_l9"),
	VREG_DESCRIB(PM8058_VREG_ID_L10, "8058_l10"),
	VREG_DESCRIB(PM8058_VREG_ID_L11, "8058_l11"),
	VREG_DESCRIB(PM8058_VREG_ID_L12, "8058_l12"),
	VREG_DESCRIB(PM8058_VREG_ID_L13, "8058_l13"),
	VREG_DESCRIB(PM8058_VREG_ID_L14, "8058_l14"),
	VREG_DESCRIB(PM8058_VREG_ID_L15, "8058_l15"),
	VREG_DESCRIB(PM8058_VREG_ID_L16, "8058_l16"),
	VREG_DESCRIB(PM8058_VREG_ID_L17, "8058_l17"),
	VREG_DESCRIB(PM8058_VREG_ID_L18, "8058_l18"),
	VREG_DESCRIB(PM8058_VREG_ID_L19, "8058_l19"),
	VREG_DESCRIB(PM8058_VREG_ID_L20, "8058_l20"),
	VREG_DESCRIB(PM8058_VREG_ID_L21, "8058_l21"),
	VREG_DESCRIB(PM8058_VREG_ID_L22, "8058_l22"),
	VREG_DESCRIB(PM8058_VREG_ID_L23, "8058_l23"),
	VREG_DESCRIB(PM8058_VREG_ID_L24, "8058_l24"),
	VREG_DESCRIB(PM8058_VREG_ID_L25, "8058_l25"),

	VREG_DESCRIB(PM8058_VREG_ID_S0, "8058_s0"),
	VREG_DESCRIB(PM8058_VREG_ID_S1, "8058_s1"),
	VREG_DESCRIB(PM8058_VREG_ID_S2, "8058_s2"),
	VREG_DESCRIB(PM8058_VREG_ID_S3, "8058_s3"),
	VREG_DESCRIB(PM8058_VREG_ID_S4, "8058_s4"),

	VREG_DESCRIB(PM8058_VREG_ID_LVS0, "8058_lvs0"),
	VREG_DESCRIB(PM8058_VREG_ID_LVS1, "8058_lvs1"),

	VREG_DESCRIB(PM8058_VREG_ID_NCP, "8058_ncp"),
};
static struct regulator * pm_vreg8058_data[PM8058_VREG_MAX];
static struct regulator * pm_vreg8901_data[PM8901_VREG_MAX];
#endif

#ifdef VIB_KER_TEST
static 	struct pmic8058_vib pmic8058_vib_ktest;

static long pm8058_vib_misc_ioctl(struct file *filep, unsigned int cmd, unsigned long arg)
{
#ifdef DO_BY_SCH_WORK_TEST
  unsigned long flags;
  int value;
#endif
  int rc = 0;
  struct vib_pm8058_set vibrator_data[1];
  struct pmic8058_vib *vib = &pmic8058_vib_ktest;

  switch(cmd)
  {
    case VIB_PM8058_CTRL:
    {
      if (copy_from_user(vibrator_data, (struct vib_pm8058_set __user *) arg, sizeof(struct vib_pm8058_set)))
      {
        pr_err("VREG_PM8058_CTRL copy_from_user error .\n");
        rc = -EFAULT;
        break;
      }
      vib->level = vibrator_data[0].vib_data.level_mV/100;
      vib->pdata->max_timeout_ms = vibrator_data[0].vib_data.max_timeout_ms;
      vib->pdata->initial_vibrate_ms = vibrator_data[0].vib_data.initial_vibrate_ms;
      vib->pdata->level_mV = vibrator_data[0].vib_data.level_mV;
      if(vibrator_data[0].enable<2)
      {
#ifndef DO_BY_SCH_WORK_TEST
        rc = pmic8058_vib_set(vib, (int)vibrator_data[0].enable);
        if (rc)
          pr_err("pmic8058_vib_set error.\n");
#else
        spin_lock_irqsave(&vib->lock, flags);
        hrtimer_cancel(&vib->vib_timer);
        value = vibrator_data[0].enable;
        if (value == 0)
          vib->state = 0;
        else {
          value = (value > vib->pdata->max_timeout_ms ?
          		 vib->pdata->max_timeout_ms : value);
          vib->state = 1;
          hrtimer_start(&vib->vib_timer,
          	      ktime_set(value / 1000, (value % 1000) * 1000000),
          	      HRTIMER_MODE_REL);
        }
        spin_unlock_irqrestore(&vib->lock, flags);
        schedule_work(&vib->work);      
#endif
      }
#ifdef VREG_KER_TEST
      else
      {
        int vreg_id;
        int vreg_volt;
        int enable_vreg;

        switch(vibrator_data[0].enable)
        {
          case 2:
          {
            
            vreg_id = vibrator_data[0].vib_data.max_timeout_ms;
            enable_vreg = vibrator_data[0].vib_data.initial_vibrate_ms;
            vreg_volt = vibrator_data[0].vib_data.level_mV;
            
            if(enable_vreg == 1){
              pm_vreg8058_data[vreg_id] = regulator_get(NULL, pm8058_DESCRIB[vreg_id].name);
              if (IS_ERR(pm_vreg8058_data[vreg_id])){
                pr_err("regulator_get 8058 error.\n");
                return -ENODEV;					
              }
              if(vreg_id < PM8058_VREG_ID_LVS0)
              {
                rc = regulator_set_voltage(pm_vreg8058_data[vreg_id], vreg_volt,vreg_volt);
                if (rc) {
                  pr_err("regulator_set_voltage 8058 error.\n");
                  return rc;
                }
              }

              rc = regulator_enable(pm_vreg8058_data[vreg_id]);
              if (rc) {
                pr_err("regulator_enable 8058 error.\n");
                return rc;
              }
            }
            else
            {
              if (!pm_vreg8058_data[vreg_id])
                return -ENODEV;					

              rc = regulator_disable(pm_vreg8058_data[vreg_id]);
              if (rc < 0)
                pr_err("regulator_disable 8058 error.\n");
              regulator_put(pm_vreg8058_data[vreg_id]);
              pm_vreg8058_data[vreg_id] = NULL;
            }
            break;
          }
	case 3:
	{
		
		int VREG_voltage;
		unsigned int VREG_mode;
		int need_to_release = 0;
		vreg_id = vibrator_data[0].vib_data.max_timeout_ms;
		enable_vreg = vibrator_data[0].vib_data.initial_vibrate_ms;
		if(enable_vreg == 0)
		{
			printk(KERN_INFO "%s: 8058 VREG ID = " "%d\n", __func__, vreg_id);
			if (!pm_vreg8058_data[vreg_id])
			{
				need_to_release = 1;
			              pm_vreg8058_data[vreg_id] = 
			              	regulator_get(NULL, pm8058_DESCRIB[vreg_id].name);
			}
			VREG_voltage = regulator_get_voltage(pm_vreg8058_data[vreg_id]);
			VREG_mode = regulator_get_mode(pm_vreg8058_data[vreg_id]);
			if(need_to_release)
				regulator_put(pm_vreg8058_data[vreg_id]);
		}
		else
		{
			printk(KERN_INFO "%s: 8901 VREG ID = " "%d\n", __func__, vreg_id);
			if (!pm_vreg8901_data[vreg_id])
			{
				need_to_release = 1;
			              pm_vreg8901_data[vreg_id] = 
			              	regulator_get(NULL, pm8901_DESCRIB[vreg_id].name);
			}
			VREG_voltage = regulator_get_voltage(pm_vreg8901_data[vreg_id]);
			VREG_mode = regulator_get_mode(pm_vreg8901_data[vreg_id]);
			if(need_to_release)
				regulator_put(pm_vreg8901_data[vreg_id]);
		}
		printk(KERN_INFO "%s: regulator_get_voltage = " "%d\n", __func__, VREG_voltage);
		printk(KERN_INFO "%s: regulator_get_mode = " "0x%x\n", __func__, VREG_mode);
		break;
	}
          
          case 4:
          {
            
            vreg_id = vibrator_data[0].vib_data.max_timeout_ms;
            enable_vreg = vibrator_data[0].vib_data.initial_vibrate_ms;
            vreg_volt = vibrator_data[0].vib_data.level_mV;
            if(enable_vreg == 1){
              pm_vreg8901_data[vreg_id] = regulator_get(NULL, pm8901_DESCRIB[vreg_id].name);
              if (IS_ERR(pm_vreg8901_data[vreg_id])){
                pr_err("regulator_get 8901 error.\n");
                return -ENODEV;					
              }
              if(vreg_id < PM8901_VREG_ID_LVS0){
                rc = regulator_set_voltage(pm_vreg8901_data[vreg_id], vreg_volt,vreg_volt);
                if (rc) {
                  pr_err("regulator_set_voltage 8901 error.\n");
                  return rc;
                }
              }

              rc = regulator_enable(pm_vreg8901_data[vreg_id]);
              if (rc) {
                pr_err("regulator_enable 8901 error.\n");
                return rc;
              }
            }
            else
            {
              if (!pm_vreg8901_data[vreg_id])
                return -ENODEV;					

              rc = regulator_disable(pm_vreg8901_data[vreg_id]);
              if (rc < 0)
                pr_err("regulator_disable 8901 error.\n");
              regulator_put(pm_vreg8901_data[vreg_id]);
              pm_vreg8901_data[vreg_id] = NULL;
            }
            break;
          }

#if 0
          case 5:
          {
            
            unsigned char chg_reg_data[1];
            unsigned char chg_reg;
            chg_reg = vibrator_data[0].vib_data.max_timeout_ms;
            chg_reg_data[0] = vibrator_data[0].vib_data.initial_vibrate_ms;
            
            rc = smb136_write(chg_reg, &chg_reg_data[0], 1);

            break;
          }


          case 6:
          {
            
            unsigned char chg_reg_data[1];
            unsigned char chg_reg;
            chg_reg = vibrator_data[0].vib_data.max_timeout_ms;

            rc = smb136_read(chg_reg,&chg_reg_data[0],1);
            
            vibrator_data[0].vib_data.initial_vibrate_ms = chg_reg_data[0];

            if (copy_to_user((void __user *)arg, vibrator_data, sizeof(struct vib_pm8058_set)))
            {
              pr_err("RTC_PM8058_READ_TIME copy_to_user error.\n");
              rc = -EFAULT;
              break;
            }
            break;
          }
#endif
          case 7:
          {
            
            
            unsigned int level, ctrl;
            level = vibrator_data[0].vib_data.max_timeout_ms;
            ctrl = vibrator_data[0].vib_data.initial_vibrate_ms;
            if(level == 3)
            {
              level = PM8058_MPP_DIG_LEVEL_L2;
              pm8058_mpp_config_digital_out(2, level, ctrl);
            }
            else
            {
              level = PM8058_MPP_DIG_LEVEL_L3;
              pm8058_mpp_config_digital_out(2, level, ctrl);
            }
            break;
          }
#if 0
          case 8:
          {
            int val;
            val = vibrator_data[0].vib_data.max_timeout_ms;
            if(val == 1)
              qsd_bat_update_usb_status(USB_STATUS_USB_IN|USB_STATUS_USB_500);
            else if(val == 2)
              qsd_bat_update_usb_status(USB_STATUS_USB_IN | USB_STATUS_USB_1500);
            else if(val == 3)
              qsd_bat_update_usb_status(USB_STATUS_USB_IN | USB_STATUS_USB_100);
            else
              qsd_bat_update_usb_status(USB_STATUS_USB_NONE | USB_STATUS_USB_0);
            break;
          }
#endif
          case 9:
          {



            
            unsigned int mpp_id, level, ctrl;
            mpp_id = vibrator_data[0].vib_data.max_timeout_ms;
            level = vibrator_data[0].vib_data.initial_vibrate_ms;
            ctrl = vibrator_data[0].vib_data.level_mV;
            rc = pm8901_mpp_config_digital_out(mpp_id, level, ctrl);
            break;
          }

	case 8:
	{
		
		int VREG_voltage;
		unsigned int VREG_mode;
		int is_enabled_t;
		int need_to_release = 0;
		
		for(vreg_id =0; vreg_id < PM8058_VREG_MAX; vreg_id++)
		{

			if (!pm_vreg8058_data[vreg_id])
			{
				need_to_release = 1;
			              pm_vreg8058_data[vreg_id] = 
			              	regulator_get(NULL, pm8058_DESCRIB[vreg_id].name);
			}
			VREG_voltage = regulator_get_voltage(pm_vreg8058_data[vreg_id]);
			is_enabled_t = regulator_is_enabled(pm_vreg8058_data[vreg_id]);
			VREG_mode = regulator_get_mode(pm_vreg8058_data[vreg_id]);
			if(need_to_release)
			{
				regulator_put(pm_vreg8058_data[vreg_id]);
				pm_vreg8058_data[vreg_id] = NULL;
				need_to_release = 0;
			}
			printk(KERN_INFO "8058 VREG name = %s, uV = %d, enable = %d, mode = 0x%x\n", 
				pm8058_DESCRIB[vreg_id].name, VREG_voltage, is_enabled_t, VREG_mode);
		}
		
		for(vreg_id =0; vreg_id < PM8901_VREG_MAX; vreg_id++)
		{
			if(vreg_id == 12)
				continue; 
			if (!pm_vreg8901_data[vreg_id])
			{
				need_to_release = 1;
			              pm_vreg8901_data[vreg_id] = 
			              	regulator_get(NULL, pm8901_DESCRIB[vreg_id].name);
			}
			VREG_voltage = regulator_get_voltage(pm_vreg8901_data[vreg_id]);
			is_enabled_t = regulator_is_enabled(pm_vreg8901_data[vreg_id]);
			VREG_mode = regulator_get_mode(pm_vreg8901_data[vreg_id]);
			if(need_to_release)
			{
				regulator_put(pm_vreg8901_data[vreg_id]);
				pm_vreg8901_data[vreg_id] = NULL;
				need_to_release = 0;
			}
			printk(KERN_INFO "8901 VREG name = %s, uV = %d, enable = %d, mode = 0x%x\n", 
				pm8901_DESCRIB[vreg_id].name, VREG_voltage, is_enabled_t, VREG_mode);
		}
		break;
	}
          default:
            pr_err("pm8058_vib_misc_ioctl unknown command.\n");
          break;
        }

      }
#endif
      break;
    }
    
    default:
      pr_err("pm8058_vib_misc_ioctl unknown command.\n");
    break;
  }
  return rc;
}
static int pm8058_vib_misc_release(struct inode *inode, struct file *fp)
{
    return 0;
}

static int pm8058_vib_misc_open(struct inode *inode, struct file *fp)
{
  return 0;
}

static struct file_operations pm8058_vib_misc_fops = {
  .owner    = THIS_MODULE,
  .open     = pm8058_vib_misc_open,
  .release  = pm8058_vib_misc_release,
  .unlocked_ioctl     = pm8058_vib_misc_ioctl,
};



static struct miscdevice pm8058_vib_misc_device = {
  .minor  = MISC_DYNAMIC_MINOR,
  .name   = "pm8058_vib_ktest",
  .fops   = &pm8058_vib_misc_fops,
};
#endif


static int __devinit pmic8058_vib_probe(struct platform_device *pdev)

{
	struct pmic8058_vibrator_pdata *pdata = pdev->dev.platform_data;
	struct pmic8058_vib *vib;
	u8 val;
	int rc;

	struct pm8058_chip	*pm_chip;

	pm_chip = platform_get_drvdata(pdev);
	if (pm_chip == NULL) {
		dev_err(&pdev->dev, "no parent data passed in\n");
		return -EFAULT;
	}

	if (!pdata)
		return -EINVAL;

	if (pdata->level_mV < VIB_MIN_LEVEL_mV ||
			 pdata->level_mV > VIB_MAX_LEVEL_mV)
		return -EINVAL;

	vib = kzalloc(sizeof(*vib), GFP_KERNEL);
	if (!vib)
		return -ENOMEM;

	/* Enable runtime PM ops, start in ACTIVE mode */
	rc = pm_runtime_set_active(&pdev->dev);
	if (rc < 0)
		dev_dbg(&pdev->dev, "unable to set runtime pm state\n");
	pm_runtime_enable(&pdev->dev);

	vib->pm_chip	= pm_chip;
	vib->pdata	= pdata;
	vib->level	= pdata->level_mV / 100;
	vib->dev	= &pdev->dev;

	spin_lock_init(&vib->lock);
	INIT_WORK(&vib->work, pmic8058_vib_update);

	hrtimer_init(&vib->vib_timer, CLOCK_MONOTONIC, HRTIMER_MODE_REL);
	vib->vib_timer.function = pmic8058_vib_timer_func;


#if (defined(CONFIG_BOARD_IS_LA) || defined(CONFIG_BOARD_IS_SONG))
	vib->timed_dev.name = "vibrator_pmic8058";
#else 
	vib->timed_dev.name = "vibrator";
#endif

	vib->timed_dev.get_time = pmic8058_vib_get_time;
	vib->timed_dev.enable = pmic8058_vib_enable;

	__dump_vib_regs(vib, "boot_vib_default");

	/* operate in manual mode */
	rc = pmic8058_vib_read_u8(vib, &val, VIB_DRV);
	if (rc < 0)
		goto err_read_vib;
	val &= ~VIB_DRV_EN_MANUAL_MASK;
	rc = pmic8058_vib_write_u8(vib, val, VIB_DRV);
	if (rc < 0)
		goto err_read_vib;

	vib->reg_vib_drv = val;


#ifdef VIB_KER_TEST

   memcpy(&pmic8058_vib_ktest, vib, sizeof(struct pmic8058_vib));
#endif


	rc = timed_output_dev_register(&vib->timed_dev);
	if (rc < 0)
		goto err_read_vib;

	
	if(4 != get_mfg_mode())
		pmic8058_vib_enable(&vib->timed_dev, pdata->initial_vibrate_ms);

	platform_set_drvdata(pdev, vib);

	pm_runtime_set_suspended(&pdev->dev);

  
#ifdef VIB_KER_TEST
   misc_register(&pm8058_vib_misc_device);
#endif
  

	return 0;

err_read_vib:
	pm_runtime_set_suspended(&pdev->dev);
	pm_runtime_disable(&pdev->dev);
	kfree(vib);
	return rc;
}

static int __devexit pmic8058_vib_remove(struct platform_device *pdev)
{
	struct pmic8058_vib *vib = platform_get_drvdata(pdev);

	pm_runtime_disable(&pdev->dev);
	cancel_work_sync(&vib->work);
	hrtimer_cancel(&vib->vib_timer);
	timed_output_dev_unregister(&vib->timed_dev);
	kfree(vib);

	return 0;
}

static struct platform_driver pmic8058_vib_driver = {
	.probe		= pmic8058_vib_probe,
	.remove		= __devexit_p(pmic8058_vib_remove),
	.driver		= {
		.name	= "pm8058-vib",
		.owner	= THIS_MODULE,
#ifdef CONFIG_PM
		.pm	= &pmic8058_vib_pm_ops,
#endif
	},
};

static int __init pmic8058_vib_init(void)
{
	return platform_driver_register(&pmic8058_vib_driver);
}
module_init(pmic8058_vib_init);

static void __exit pmic8058_vib_exit(void)
{
  
#ifdef VIB_KER_TEST
   misc_deregister(&pm8058_vib_misc_device);
#endif
  
	platform_driver_unregister(&pmic8058_vib_driver);
}
module_exit(pmic8058_vib_exit);

MODULE_ALIAS("platform:pmic8058_vib");
MODULE_DESCRIPTION("PMIC8058 vibrator driver");
MODULE_LICENSE("GPL v2");
