

#include <linux/module.h>
#include <linux/init.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/delay.h>
#include <linux/err.h>
#include <linux/pwm.h>
#include <linux/workqueue.h>
#include <linux/slab.h>
#include <linux/i2c/q_isa1200.h>
#include "../staging/android/timed_output.h"


#include <linux/regulator/pmic8058-regulator.h>
#include <linux/regulator/consumer.h>
#include <linux/kernel.h>


#ifdef CONFIG_PMIC8058_PWM
# include <linux/mfd/pmic8058.h>
# include <linux/pmic8058-pwm.h>
#endif



#define q_isa1200_SCTRL		0x00


#define q_isa1200_HCTRL0	    0x30
#define q_isa1200_HCTRL1		0x31


#define q_isa1200_HCTRL2		0x32
#define q_isa1200_HCTRL3		0x33
#define q_isa1200_HCTRL4		0x34


#define q_isa1200_HCTRL5		0x35


#define q_isa1200_HCTRL6		0x36
#define q_isa1200_HCTRL7		0x37
#define q_isa1200_HCTRL8		0x38
#define q_isa1200_HCTRL9		0x39
#define q_isa1200_HCTRLA		0x3A
#define q_isa1200_HCTRLB		0x3B
#define q_isa1200_HCTRLC		0x3C
#define q_isa1200_HCTRLD		0x3D
#define q_isa1200_HCTRLE		0x3E
#define q_isa1200_HCTRLF		0x3F


#define q_isa1200_HCTRL0_RESET	0x01
#define q_isa1200_HCTRL1_RESET	0x4B

#define q_isa1200_HCTRL5_VIB_STRT	0xD5
#define q_isa1200_HCTRL5_VIB_STOP	0x6B

#define LEVEL_LOW		0
#define LEVEL_HIGH		1



#define q_isa1200_ERROR   0
#define q_isa1200_SUCCESS 1


#ifdef q_isa1200_DEBUG
# define DBGPRINTK printk
#else
# define DBGPRINTK(a...)
#endif



struct q_isa1200_chip {
	struct i2c_client *client;
	struct q_isa1200_platform_data *pdata;
	struct pwm_device *pwm;
	struct hrtimer timer;
	struct timed_output_dev dev;
	struct work_struct work;
	spinlock_t lock;
	unsigned int enable;
	unsigned int period_ns;
};


static void set_regisetr_bit(int *original_value, unsigned int bit)
{
	*original_value |= (0x01 << bit);
}

static void clear_regisetr_bit(int *original_value, unsigned int bit)
{
	*original_value &= ~(0x01 << bit);
}



static int q_isa1200_read_reg(struct i2c_client *client, int reg)
{
	int ret;

	ret = i2c_smbus_read_byte_data(client, reg);
	if (ret < 0)
		dev_err(&client->dev, "%s: err %d\n", __func__, ret);

	return ret;
}

static int
q_isa1200_write_reg(struct i2c_client *client, int reg, u8 value)
{
	int ret;

	ret = i2c_smbus_write_byte_data(client, reg, value);
	if (ret < 0)
		dev_err(&client->dev, "%s: err %d\n", __func__, ret);

	return ret;
}

static void q_isa1200_vib_set(struct q_isa1200_chip *haptic, int enable)
{

	int period_us;
	int rc = 0;

	DBGPRINTK(KERN_ERR "+ %s\n", __func__);

	if (enable) {

		gpio_set_value(haptic->pdata->hap_en_gpio, LEVEL_HIGH);




		if (haptic->pdata->mode_ctrl == Q_PWM_INPUT_MODE) {

			period_us = haptic->period_ns / 1000;
			rc = pwm_config(haptic->pwm,
					(period_us *
					 haptic->pdata->duty) / 100,
					period_us);
			if (rc < 0)
				pr_err("%s: pwm_config fail\n", __func__);
			rc = pwm_enable(haptic->pwm);
			if (rc < 0)
				pr_err("%s: pwm_enable fail\n", __func__);
		} else if (haptic->pdata->mode_ctrl == Q_PWM_GEN_MODE) {
			DBGPRINTK(KERN_ERR " start Q_PWM_GEN_MODE \n");

			rc = q_isa1200_write_reg(haptic->client,
						 q_isa1200_HCTRL5,
						 q_isa1200_HCTRL5_VIB_STRT);
			if (rc < 0)
				pr_err("%s: start vibartion fail\n",
				       __func__);
		}
	} else {
		if (haptic->pdata->mode_ctrl == Q_PWM_INPUT_MODE) {
			pwm_disable(haptic->pwm);

		} else if (haptic->pdata->mode_ctrl == Q_PWM_GEN_MODE) {
			DBGPRINTK(KERN_ERR " stop Q_PWM_GEN_MODE \n");

			rc = q_isa1200_write_reg(haptic->client,
						 q_isa1200_HCTRL5,
						 q_isa1200_HCTRL5_VIB_STOP);
			if (rc < 0)
				pr_err("%s: stop vibartion fail\n",
				       __func__);
		}

		gpio_set_value(haptic->pdata->hap_en_gpio, LEVEL_LOW);

	}
	DBGPRINTK(KERN_ERR "- %s\n", __func__);
}

static void q_isa1200_chip_work(struct work_struct *work)
{
	struct q_isa1200_chip *haptic;
	DBGPRINTK(KERN_ERR "+ %s\n", __func__);
	haptic = container_of(work, struct q_isa1200_chip, work);
	q_isa1200_vib_set(haptic, haptic->enable);
	DBGPRINTK(KERN_ERR "- %s\n", __func__);
}

static void q_isa1200_chip_enable(struct timed_output_dev *dev, int value)
{
	struct q_isa1200_chip *haptic =
	    container_of(dev, struct q_isa1200_chip,
			 dev);
	unsigned long flags;
	DBGPRINTK(KERN_ERR "+ %s\n", __func__);
	spin_lock_irqsave(&haptic->lock, flags);
	hrtimer_cancel(&haptic->timer);
	if (value == 0)
		haptic->enable = 0;
	else {
		value = (value > haptic->pdata->max_timeout ?
			 haptic->pdata->max_timeout : value);
		haptic->enable = 1;
		hrtimer_start(&haptic->timer,
			      ktime_set(value / 1000,
					(value % 1000) * 1000000),
			      HRTIMER_MODE_REL);
		DBGPRINTK(KERN_ERR "timer = %dms\n", value);
	}
	spin_unlock_irqrestore(&haptic->lock, flags);
	schedule_work(&haptic->work);
	DBGPRINTK(KERN_ERR "- %s\n", __func__);
}

static int q_isa1200_chip_get_time(struct timed_output_dev *dev)
{
	struct q_isa1200_chip *haptic =
	    container_of(dev, struct q_isa1200_chip,
			 dev);

	DBGPRINTK(KERN_ERR "+ %s\n", __func__);
	DBGPRINTK(KERN_ERR "- %s\n", __func__);

	if (hrtimer_active(&haptic->timer)) {
		ktime_t r = hrtimer_get_remaining(&haptic->timer);
		return r.tv.sec * 1000 + r.tv.nsec / 1000000;
	} else
		return 0;
}

static enum hrtimer_restart q_isa1200_vib_timer_func(struct hrtimer *timer)
{
	struct q_isa1200_chip *haptic =
	    container_of(timer, struct q_isa1200_chip,
			 timer);
	DBGPRINTK(KERN_ERR "+ %s\n", __func__);

	haptic->enable = 0;
	schedule_work(&haptic->work);

	DBGPRINTK(KERN_ERR "- %s\n", __func__);
	return HRTIMER_NORESTART;
}

static void dump_q_isa1200_reg(char *str, struct i2c_client *client)
{


	printk(KERN_ERR
	       "%s\n reg0x%02X=0x%02X\n reg0x%02X=0x%02X\n reg0x%02X=0x%02X\n"
	       " reg0x%02X=0x%02X\n reg0x%02X=0x%02X\n reg0x%02X=0x%02X\n"
	       " reg0x%02X=0x%02X\n reg0x%02X=0x%02X\n reg0x%02X=0x%02X\n"
	       " reg0x%02X=0x%02X\n reg0x%02X=0x%02X\n reg0x%02X=0x%02X\n"
	       " reg0x%02X=0x%02X\n reg0x%02X=0x%02X\n reg0x%02X=0x%02X\n"
	       " reg0x%02X=0x%02X\n reg0x%02X=0x%02X\n",
	       str, q_isa1200_SCTRL, q_isa1200_read_reg(client,
							q_isa1200_SCTRL),
	       q_isa1200_HCTRL0, q_isa1200_read_reg(client,
						    q_isa1200_HCTRL0),
	       q_isa1200_HCTRL1, q_isa1200_read_reg(client,
						    q_isa1200_HCTRL1),
	       q_isa1200_HCTRL2, q_isa1200_read_reg(client,
						    q_isa1200_HCTRL2),
	       q_isa1200_HCTRL3, q_isa1200_read_reg(client,
						    q_isa1200_HCTRL3),
	       q_isa1200_HCTRL4, q_isa1200_read_reg(client,
						    q_isa1200_HCTRL4),
	       q_isa1200_HCTRL5, q_isa1200_read_reg(client,
						    q_isa1200_HCTRL5),
	       q_isa1200_HCTRL6, q_isa1200_read_reg(client,
						    q_isa1200_HCTRL6),
	       q_isa1200_HCTRL7, q_isa1200_read_reg(client,
						    q_isa1200_HCTRL7),
	       q_isa1200_HCTRL8, q_isa1200_read_reg(client,
						    q_isa1200_HCTRL8),
	       q_isa1200_HCTRL9, q_isa1200_read_reg(client,
						    q_isa1200_HCTRL9),
	       q_isa1200_HCTRLA, q_isa1200_read_reg(client,
						    q_isa1200_HCTRLA),
	       q_isa1200_HCTRLB, q_isa1200_read_reg(client,
						    q_isa1200_HCTRLB),
	       q_isa1200_HCTRLC, q_isa1200_read_reg(client,
						    q_isa1200_HCTRLC),
	       q_isa1200_HCTRLD, q_isa1200_read_reg(client,
						    q_isa1200_HCTRLD),
	       q_isa1200_HCTRLE, q_isa1200_read_reg(client,
						    q_isa1200_HCTRLE),
	       q_isa1200_HCTRLF, q_isa1200_read_reg(client,
						    q_isa1200_HCTRLF));

}



static int q_isa1200_input_mode_config(struct q_isa1200_chip *haptic)
{
	struct i2c_client *hapClient = haptic->client;
	int value_HCTRL0, value_HCTRL1;
	int rc;

	rc = 0;

	
	value_HCTRL0 = 0x01;
	value_HCTRL1 = 0x4B;

	DBGPRINTK(KERN_ERR "+ %s\n", __func__);

	
	rc += q_isa1200_write_reg(hapClient, q_isa1200_HCTRL2, 0x80);
	rc += q_isa1200_write_reg(hapClient, q_isa1200_HCTRL2, 0x00);

	
	if (true == haptic->pdata->is_erm) {
		set_regisetr_bit(&value_HCTRL1, 5);
	} else {
		clear_regisetr_bit(&value_HCTRL1, 5);
	}
	rc +=
	    q_isa1200_write_reg(hapClient, q_isa1200_HCTRL1, value_HCTRL1);

	
	clear_regisetr_bit(&value_HCTRL0, 4);
	set_regisetr_bit(&value_HCTRL0, 3);

	
	switch (haptic->pdata->pwm_fd.pwm_freq) {
	case 22400:
		clear_regisetr_bit(&value_HCTRL0, 0);
		clear_regisetr_bit(&value_HCTRL0, 1);
		break;
	case 44800:
		
		
		set_regisetr_bit(&value_HCTRL0, 0);
		clear_regisetr_bit(&value_HCTRL0, 1);
		break;
	case 89600:
		set_regisetr_bit(&value_HCTRL0, 1);
		clear_regisetr_bit(&value_HCTRL0, 0);
		break;
	case 179200:
		set_regisetr_bit(&value_HCTRL0, 1);
		clear_regisetr_bit(&value_HCTRL0, 1);
		break;

	default:
		pr_err("%s: Invalid divider\n", __func__);
		goto reset_hctrl1;
		break;
	}

	
	if (1 == haptic->pdata->chip_en) {
		set_regisetr_bit(&value_HCTRL0, 7);
	} else {
		clear_regisetr_bit(&value_HCTRL0, 7);
	}


	
	rc +=
	    q_isa1200_write_reg(hapClient, q_isa1200_HCTRL0, value_HCTRL0);

	DBGPRINTK(KERN_ERR "- %s\n", __func__);
	return rc;
      reset_hctrl1:
	i2c_smbus_write_byte_data(hapClient, q_isa1200_HCTRL1,
				  q_isa1200_HCTRL1_RESET);
	return -1;
}



static int q_isa1200_generation_mode_config(struct q_isa1200_chip *haptic)
{
	struct i2c_client *hapClient = haptic->client;
	int value_HCTRL0, value_HCTRL1;
	int rc;
	rc = 0;

	DBGPRINTK(KERN_ERR "+ %s\n", __func__);

	
	rc += q_isa1200_write_reg(hapClient, q_isa1200_HCTRL0, 0x09);

	
	value_HCTRL0 = 0x01;
	value_HCTRL1 = 0x4B;

	
	rc += q_isa1200_write_reg(hapClient, q_isa1200_HCTRL2, 0x80);
	rc += q_isa1200_write_reg(hapClient, q_isa1200_HCTRL2, 0x00);

	
	if (true == haptic->pdata->is_erm) {
		set_regisetr_bit(&value_HCTRL1, 5);
	} else {
		clear_regisetr_bit(&value_HCTRL1, 5);
	}


	
	
	set_regisetr_bit(&value_HCTRL0, 4);
	clear_regisetr_bit(&value_HCTRL0, 3);

	
	if (true == haptic->pdata->ext_clk_en) {
		
		
		set_regisetr_bit(&value_HCTRL1, 7);
		
		clear_regisetr_bit(&value_HCTRL1, 4);
	} else {
		

		
		clear_regisetr_bit(&value_HCTRL1, 7);

		
		set_regisetr_bit(&value_HCTRL1, 4);

		q_isa1200_write_reg(hapClient, q_isa1200_HCTRL9, 0x40);
		q_isa1200_write_reg(hapClient, q_isa1200_HCTRLA, 0x2C);

		q_isa1200_write_reg(hapClient, q_isa1200_HCTRL3, 0x13);

	}

	
	rc += q_isa1200_write_reg(hapClient, q_isa1200_HCTRL6, 0xD6);

	
	switch (haptic->pdata->pwm_fd.pwm_div) {
	case 128:
		clear_regisetr_bit(&value_HCTRL0, 0);
		clear_regisetr_bit(&value_HCTRL0, 1);
		break;
	case 256:
		
		
		set_regisetr_bit(&value_HCTRL0, 0);
		clear_regisetr_bit(&value_HCTRL0, 1);
		break;
	case 512:
		set_regisetr_bit(&value_HCTRL0, 1);
		clear_regisetr_bit(&value_HCTRL0, 0);
		break;
	case 1024:
		set_regisetr_bit(&value_HCTRL0, 1);
		clear_regisetr_bit(&value_HCTRL0, 1);
		break;

	default:
		pr_err("%s: Invalid divider\n", __func__);
		goto reset_hctrl1;
		break;
	}

	
	if (1 == haptic->pdata->chip_en) {
		set_regisetr_bit(&value_HCTRL0, 7);
	} else {
		clear_regisetr_bit(&value_HCTRL0, 7);
	}


	
	rc += q_isa1200_write_reg(hapClient, q_isa1200_HCTRL5,
				  q_isa1200_HCTRL5_VIB_STOP);

	rc +=
	    q_isa1200_write_reg(hapClient, q_isa1200_HCTRL1, value_HCTRL1);
	rc +=
	    q_isa1200_write_reg(hapClient, q_isa1200_HCTRL0, value_HCTRL0);

	dump_q_isa1200_reg("Generation Mode Registers:", hapClient);

	DBGPRINTK(KERN_ERR "- %s\n", __func__);

	return rc;

      reset_hctrl1:
	i2c_smbus_write_byte_data(hapClient, q_isa1200_HCTRL1,
				  q_isa1200_HCTRL1_RESET);
	return -1;
}



static int q_isa1200_setup(struct i2c_client *client)
{
	struct q_isa1200_chip *haptic = i2c_get_clientdata(client);
	int rc;

	DBGPRINTK(KERN_ERR "+ %s\n", __func__);

	gpio_set_value(haptic->pdata->hap_ldo_en_gpio, LEVEL_HIGH);


#if 0
	gpio_set_value_cansleep(haptic->pdata->hap_en_gpio, 0);
	udelay(250);
	gpio_set_value_cansleep(haptic->pdata->hap_en_gpio, 1);
#endif

	if (haptic->pdata->mode_ctrl == Q_PWM_GEN_MODE) {

		rc = q_isa1200_generation_mode_config(haptic);
		if (rc < 0) {
			pr_err("%s: generation_mode_config failure\n",
			       __func__);
			return rc;
		}

	} else if (haptic->pdata->mode_ctrl == Q_PWM_INPUT_MODE) {

		haptic->period_ns =
		    NSEC_PER_SEC / haptic->pdata->pwm_fd.pwm_freq;

		rc = q_isa1200_input_mode_config(haptic);
		if (rc < 0) {
			pr_err("%s: input_mode_config failure\n",
			       __func__);
			return rc;
		}
	}

	dump_q_isa1200_reg("new:", client);
	DBGPRINTK(KERN_ERR "- %s\n", __func__);
	return 0;

}


static void
q_isa1200_timed_output_dev_config(struct q_isa1200_chip *haptic)
{
	const char *name;
	name = haptic->pdata->name;
	DBGPRINTK(KERN_ERR "+ %s\n", __func__);
	spin_lock_init(&haptic->lock);
	INIT_WORK(&haptic->work, q_isa1200_chip_work);

	hrtimer_init(&haptic->timer, CLOCK_MONOTONIC, HRTIMER_MODE_REL);
	haptic->timer.function = q_isa1200_vib_timer_func;

	
	haptic->dev.name = name;
	haptic->dev.get_time = q_isa1200_chip_get_time;
	haptic->dev.enable = q_isa1200_chip_enable;
	DBGPRINTK(KERN_ERR "- %s\n", __func__);
}

static int q_isa1200_gpio_config(struct q_isa1200_chip *haptic)
{
	struct q_isa1200_platform_data *pdata;
	struct i2c_client *client;
	int ret;

	DBGPRINTK(KERN_ERR "+ %s\n", __func__);

	pdata = haptic->pdata;
	client = haptic->client;

	
	gpio_tlmm_config(GPIO_CFG
			 (pdata->hap_en_gpio, 0, GPIO_CFG_OUTPUT,
			  GPIO_CFG_NO_PULL, GPIO_CFG_2MA),
			 GPIO_CFG_ENABLE);

	ret = gpio_is_valid(pdata->hap_en_gpio);
	if (ret) {
		ret = gpio_request(pdata->hap_en_gpio, "haptic_gpio");
		if (ret) {
			dev_err(&client->dev,
				"%s: gpio %d request failed\n", __func__,
				pdata->hap_en_gpio);
			return q_isa1200_ERROR;
		}
	} else {
		dev_err(&client->dev, "%s: Invalid gpio %d\n", __func__,
			pdata->hap_en_gpio);
		return q_isa1200_ERROR;
	}
	gpio_set_value(haptic->pdata->hap_en_gpio, LEVEL_LOW);

	
	gpio_tlmm_config(GPIO_CFG
			 (pdata->hap_ldo_en_gpio, 0, GPIO_CFG_OUTPUT,
			  GPIO_CFG_NO_PULL, GPIO_CFG_2MA),
			 GPIO_CFG_ENABLE);

	ret = gpio_is_valid(pdata->hap_ldo_en_gpio);
	if (ret) {
		ret =
		    gpio_request(pdata->hap_ldo_en_gpio,
				 "haptic_ldo_gpio");
		if (ret) {
			dev_err(&client->dev,
				"%s: gpio %d request failed\n", __func__,
				pdata->hap_ldo_en_gpio);
			return q_isa1200_ERROR;
		}
	} else {
		dev_err(&client->dev, "%s: Invalid gpio %d\n", __func__,
			pdata->hap_ldo_en_gpio);
		return q_isa1200_ERROR;
	}
	gpio_set_value(haptic->pdata->hap_ldo_en_gpio, LEVEL_LOW);

	DBGPRINTK(KERN_ERR "- %s\n", __func__);
	return q_isa1200_SUCCESS;

}





static int __devinit
q_isa1200_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	struct q_isa1200_chip *haptic;
	struct q_isa1200_platform_data *pdata;
	int ret;

	DBGPRINTK(KERN_ERR "+ %s\n", __func__);

	if (!i2c_check_functionality
	    (client->adapter, I2C_FUNC_SMBUS_BYTE_DATA)) {
		dev_err(&client->dev, "%s: no support for i2c read/write"
			"byte data\n", __func__);
		return -EIO;
	}

	pdata = client->dev.platform_data;
	if (!pdata) {
		dev_err(&client->dev, "%s: no platform data\n", __func__);
		return -EINVAL;
	}

	if (pdata->dev_setup) {
		ret = pdata->dev_setup(true);
		if (ret < 0) {
			dev_err(&client->dev, "dev setup failed\n");
			return -EINVAL;
		}
	}

	haptic = kzalloc(sizeof(struct q_isa1200_chip), GFP_KERNEL);
	if (!haptic) {
		ret = -ENOMEM;
		goto mem_alloc_fail;
	}
	haptic->client = client;
	haptic->enable = 0;
	haptic->pdata = pdata;

	if (pdata->power_on) {
		ret = pdata->power_on(1);
		if (ret) {
			dev_err(&client->dev, "%s: power-up failed\n",
				__func__);
			goto pwr_up_fail;
		}
	}

	q_isa1200_timed_output_dev_config(haptic);

	ret = timed_output_dev_register(&haptic->dev);
	if (ret < 0)
		goto timed_reg_fail;

	i2c_set_clientdata(client, haptic);

	ret = q_isa1200_gpio_config(haptic);

	if (q_isa1200_ERROR == ret) {
		goto gpio_fail;
	}



	ret = q_isa1200_setup(client);
	if (ret) {
		dev_err(&client->dev, "%s: setup fail %d\n", __func__,
			ret);
		goto gpio_fail;
	}

	if (haptic->pdata->mode_ctrl == Q_PWM_INPUT_MODE) {
		haptic->pwm = pwm_request(pdata->pwm_ch_id, id->name);
		if (IS_ERR(haptic->pwm)) {
			dev_err(&client->dev, "%s: pwm request failed\n",
				__func__);
			ret = PTR_ERR(haptic->pwm);
			goto reset_hctrl0;
		}
	}

	printk(KERN_INFO "%s: %s registered\n", __func__, id->name);





	DBGPRINTK(KERN_ERR "- %s\n", __func__);
	return 0;

      reset_hctrl0:
	i2c_smbus_write_byte_data(client, q_isa1200_HCTRL0,
				  q_isa1200_HCTRL0_RESET);
      gpio_fail:
	timed_output_dev_unregister(&haptic->dev);
      timed_reg_fail:
	if (pdata->power_on)
		pdata->power_on(0);
      pwr_up_fail:
	kfree(haptic);
      mem_alloc_fail:
	if (pdata->dev_setup)
		pdata->dev_setup(false);
	return ret;
}

static int __devexit q_isa1200_remove(struct i2c_client *client)
{
	struct q_isa1200_chip *haptic = i2c_get_clientdata(client);

	hrtimer_cancel(&haptic->timer);
	cancel_work_sync(&haptic->work);

	
	q_isa1200_vib_set(haptic, 0);

	if (haptic->pdata->mode_ctrl == Q_PWM_INPUT_MODE)
		pwm_free(haptic->pwm);

	timed_output_dev_unregister(&haptic->dev);
	gpio_free(haptic->pdata->hap_en_gpio);

	gpio_free(haptic->pdata->hap_ldo_en_gpio);


	
	i2c_smbus_write_byte_data(client, q_isa1200_HCTRL0,
				  q_isa1200_HCTRL0_RESET);
	i2c_smbus_write_byte_data(client, q_isa1200_HCTRL1,
				  q_isa1200_HCTRL1_RESET);


	
	if (haptic->pdata->power_on)
		haptic->pdata->power_on(0);

	if (haptic->pdata->dev_setup)
		haptic->pdata->dev_setup(false);


	kfree(haptic);
	return 0;
}

#ifdef CONFIG_PM
static int q_isa1200_suspend(struct i2c_client *client, pm_message_t mesg)
{
	struct q_isa1200_chip *haptic = i2c_get_clientdata(client);
#if 1
	int ret;
#endif
	DBGPRINTK(KERN_ERR "+ %s\n", __func__);
	hrtimer_cancel(&haptic->timer);
	cancel_work_sync(&haptic->work);
	
	q_isa1200_vib_set(haptic, 0);
#if 1
	if (haptic->pdata->power_on) {
		ret = haptic->pdata->power_on(0);
		if (ret) {
			dev_err(&client->dev, "power-down failed\n");
			return ret;
		}
	}
#endif

	gpio_set_value(haptic->pdata->hap_ldo_en_gpio, LEVEL_LOW);

	DBGPRINTK(KERN_ERR "- %s\n", __func__);
	return 0;
}

static int q_isa1200_resume(struct i2c_client *client)
{
	struct q_isa1200_chip *haptic = i2c_get_clientdata(client);
	int ret;
	DBGPRINTK(KERN_ERR "+ %s\n", __func__);
	if (haptic->pdata->power_on) {
		ret = haptic->pdata->power_on(1);
		if (ret) {
			dev_err(&client->dev, "power-up failed\n");
			return ret;
		}
	}

	q_isa1200_setup(client);
	DBGPRINTK(KERN_ERR "- %s\n", __func__);
	return 0;
}
#else
#define q_isa1200_suspend		NULL
#define q_isa1200_resume		NULL
#endif

static const struct i2c_device_id q_isa1200_id[] = {
	{"q_isa1200_1", 0},
	{},
};
MODULE_DEVICE_TABLE(i2c, q_isa1200_id);

static struct i2c_driver q_isa1200_driver = {
	.driver = {
		   .name = "q_isa1200",

		   .owner = THIS_MODULE,

		   },
	.probe = q_isa1200_probe,
	.remove = __devexit_p(q_isa1200_remove),
	.suspend = q_isa1200_suspend,
	.resume = q_isa1200_resume,
	.id_table = q_isa1200_id,
};

static int __init q_isa1200_init(void)
{
	return i2c_add_driver(&q_isa1200_driver);
}

static void __exit q_isa1200_exit(void)
{
	i2c_del_driver(&q_isa1200_driver);
}

module_init(q_isa1200_init);
module_exit(q_isa1200_exit);

MODULE_AUTHOR("Kyungmin Park <kyungmin.park@samsung.com>");
MODULE_DESCRIPTION("ISA1200 Haptic Motor driver");
MODULE_LICENSE("GPL");
