/*
 * drivers/i2c/chips/tsl2583.c
 *
 * Copyright (C) 2008 Nokia Corporation
 *
 * Written by Timo O. Karjalainen <timo.o.karjalainen@nokia.com>
 * Contact: Amit Kucheria <amit.kucheria@verdurent.com>
 *
 * Converted to IIO driver
 * Amit Kucheria <amit.kucheria@verdurent.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/input.h>

#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/mutex.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/miscdevice.h>
#include <asm/uaccess.h>
#include <asm/mach-types.h>

#include <linux/pm.h>
#include <linux/hwmon.h>
#include <linux/err.h>
#include <linux/slab.h>
#include <linux/lightsensor.h>
#include <linux/tsl2583.h>
#include <linux/regulator/pmic8058-regulator.h>
#include <linux/regulator/pmic8901-regulator.h>


#define ADC_FRAC_BITS		(14)


#define FRAC10K(f)		(((f) * (1L << (ADC_FRAC_BITS))) / (10000))


#define CALIB_FRAC_BITS		(10)

#define CALIB_FRAC_HALF		(1 << (CALIB_FRAC_BITS - 1))

#define CALIB_FRAC(n, b)	(((n) << CALIB_FRAC_BITS) / (b))

#define CALIB_BASE_SYSFS	(1000)

#define TSL2583_CMD		(0x80)
#define TSL2583_CLEARINT	(0x40)
#define TAOS_REG_MAX	(0x08)

#define TSL2583_REG_CTRL	(0x00)
#define TSL2583_REG_TIMING	(0x01)
#define TSL2583_REG_INT		(0x02)
#define TSL2583_REG_LOWLOW	(0x03) 
#define TSL2583_REG_LOWHIGH	(0x04)
#define TSL2583_REG_HIGHLOW	(0x05) 
#define TSL2583_REG_HIGHHIGH	(0x06)
#define TSL2583_REG_ANALOG		(0x07)
#define TSL2583_REG_ID		(0x12)
#define TSL2583_REG_CONSTANT		(0x13) 
#define TSL2583_REG_DATA0LOW	(0x14) 
#define TSL2583_REG_DATA0HIGH	(0x15)
#define TSL2583_REG_DATA1LOW	(0x16) 
#define TSL2583_REG_DATA1HIGH	(0x17)
#define TSL2583_REG_TIMERLOW	(0x18) 
#define TSL2583_REG_TIMERHIGH	(0x19)

#define TSL2583_CMD_POWER_ON	(0x01)
#define TSL2583_CMD_POWER_OFF	(0x00)
#define TSL2583_CTRL_POWER_MASK	(0x01)

#define TSL2583_TIMING_MANUAL	(0x00)
#define TSL2583_TIMING_3MS		(0xFF)
#define TSL2583_TIMING_5MS		(0xFE)
#define TSL2583_TIMING_51MS		(0xED)
#define TSL2583_TIMING_100MS	(0xDB)
#define TSL2583_TIMING_200MS	(0xB4)
#define TSL2583_TIMING_400MS	(0x6C)
#define TSL2583_TIMING_688MS	(0x01)

#define TSL2583_TIMING_GAIN1	(0x00)
#define TSL2583_TIMING_GAIN8	(0x01)
#define TSL2583_TIMING_GAIN16	(0x10)
#define TSL2583_TIMING_GAIN111	(0x11)

#define TSL2583_INT_DISBLED	(0x00)
#define TSL2583_INT_LEVEL	(0x10)
#define TSL2583_INT_PERSIST(n)	((n) & 0x0F)

static unsigned int adc_to_lux(u32 adc0, u32 adc1);
static u32 normalize_adc(u16 adc, u8 gain, u8 ch);


struct tsl2583_gainlevel_coeff {
	u8 gain;
	u8 time;
	u16 min;
	u16 max;
};

struct tsl2583_chip {
	struct mutex		lock;
	struct i2c_client	*client;
	struct input_dev *ls_input_dev;	
	struct delayed_work	poweroff_work;

	
	pm_message_t		state;

	u8 gain;
	u8 time;
	
	u16			low_thres;
	u16			high_thres;
	u8			intr;

	
	u32			calib0;
	u32			calib1;
	int			cover_comp_gain;

	
	u32			data0;
	u32			data1;
};

struct tsl2583_chip *chip;


static struct regulator *vreg_s3a_1p8;		
static struct regulator *vreg_l10a_2p85;	

static int tsl2583_power(int on)
{
	int rc = 0;
#define _GET_REGULATOR(var, name) do {				\
		var = regulator_get(NULL, name);			\
		if (IS_ERR(var)) {					\
			pr_info("'%s' regulator not found, rc=%ld\n",	\
				name, IS_ERR(var)); 		\
			var = NULL; 				\
			return -ENODEV; 				\
		}							\
	} while (0)

	if (!vreg_l10a_2p85)
		_GET_REGULATOR(vreg_l10a_2p85, "8058_l10");
	if (!vreg_s3a_1p8)
		_GET_REGULATOR(vreg_s3a_1p8, "8058_s3");
#undef _GET_REGULATOR

	if (on) {
		rc = regulator_set_voltage(vreg_l10a_2p85, 2850000, 2850000);
		if (rc) {
			pr_info("%s: '%s' regulator set voltage failed,\
				rc=%d\n", __func__, "8058_l10", rc);
			return rc;
		}

		rc = regulator_enable(vreg_l10a_2p85);
		if (rc) {
			pr_err("%s: '%s' regulator enable failed,\
				rc=%d\n", __func__, "8058_l10", rc);
			return rc;
		}
	} else {
		rc = regulator_disable(vreg_l10a_2p85);
		if (rc)
			pr_warning("%s: '%s' regulator disable failed, rc=%d\n",
				__func__, "8058_l10", rc);
	}

	if (on) {
		rc = regulator_set_voltage(vreg_s3a_1p8, 1800000, 1800000);
		if (rc) {
			pr_info("%s: '%s' regulator set voltage failed,\
				rc=%d\n", __func__, "8058_s3", rc);
			return rc;
		}

		rc = regulator_enable(vreg_s3a_1p8);
		if (rc) {
			pr_err("%s: '%s' regulator enable failed,\
				rc=%d\n", __func__, "8058_s3", rc);
			return rc;
		}
	} else {
		rc = regulator_disable(vreg_s3a_1p8);
		if (rc)
			pr_warning("%s: '%s' regulator disable failed, rc=%d\n",
				__func__, "8058_s3", rc);
	}

	return rc;
}


static int tsl2583_write(struct i2c_client *client, u8 reg, u8 value)
{
	int ret;
	u8 buf[2];

	buf[0] = TSL2583_CMD | reg;
	buf[1] = value;

	ret = i2c_master_send(client, buf, sizeof(buf));
	return (ret == sizeof(buf)) ? 0 : ret;
}

static int tsl2583_read(struct i2c_client *client, u8 reg, void *buf, int len)
{
	int ret;
	u8 cmd = TSL2583_CMD | reg;

	ret = i2c_master_send(client, &cmd, sizeof(cmd));
	if (ret != sizeof(cmd))
		return ret;

	return i2c_master_recv(client, buf, len);
}

static int tsl2583_set_power(int on)
{
	struct i2c_client *client = chip->client;
	u8 cmd;

	cmd = on ? TSL2583_CMD_POWER_ON : TSL2583_CMD_POWER_OFF;
	return tsl2583_write(client, TSL2583_REG_CTRL, cmd);
}


static int tsl2583_get_power(void)
{
	struct i2c_client *client = chip->client;
	int ret;
	u8 val;

	ret = tsl2583_read(client, TSL2583_REG_CTRL, &val, sizeof(val));
	if (ret != sizeof(val))
		return ret;

	return (val & TSL2583_CTRL_POWER_MASK) == TSL2583_CMD_POWER_ON;
}

static int tsl2583_set_adc(int enable)
{
	struct i2c_client *client = chip->client;
	u8 cmd;
	int ret = 0;

	ret = tsl2583_read(client, TSL2583_REG_CTRL, &cmd, sizeof(cmd));
	if (ret != sizeof(cmd))
		return ret;

	cmd = enable ? (cmd | 0x2) : (cmd & 0xfd);

	return tsl2583_write(client, TSL2583_REG_CTRL, cmd);
}

static int tsl2583_configure(void)
{
	struct i2c_client *client = chip->client;
	int ret;

	ret = tsl2583_write(client, TSL2583_REG_LOWLOW,
			chip->low_thres && 0xff);	
	ret = tsl2583_write(client, TSL2583_REG_LOWHIGH,
			(chip->low_thres && 0xff00 )>> 8);
	ret = tsl2583_write(client, TSL2583_REG_HIGHLOW,
			chip->high_thres && 0xff);	
	ret = tsl2583_write(client, TSL2583_REG_HIGHHIGH,
			(chip->high_thres && 0xff00 )>> 8);
	ret = tsl2583_write(client, TSL2583_REG_TIMING,
			chip->time);	
	ret = tsl2583_write(client, TSL2583_REG_ANALOG,
			chip->gain);	
	if (ret)
		goto out;

	ret = tsl2583_write(client, TSL2583_REG_INT, chip->intr);

out:
	return ret;
}

static void tsl2583_poweroff_work(struct work_struct *work)
{
	tsl2583_set_power(0);
}

static int tsl2583_detect(void)
{
	int ret;

	ret = tsl2583_set_power(1);
	if (ret)
		return ret;

	mdelay(2);
	ret = tsl2583_get_power();
	if (ret < 0)
		return ret;

	return ret ? 0 : -ENODEV;
}

static int tsl2583_read_id(u8 *id)
{
	struct i2c_client *client = chip->client;
	int ret;

	ret = tsl2583_read(client, TSL2583_REG_ID, id, sizeof(*id));
	if (ret != sizeof(*id))
		return ret;

	return 0;
}

#define CH0GAIN128X 107 
#define CH1GAIN128X 115 


static u32 normalize_adc(u16 adc, u8 gain, u8 ch)
{
	switch (gain) {
	case TSL2583_TIMING_GAIN1:
		
		break;
	case TSL2583_TIMING_GAIN8:
		adc = adc >>3;
		break;
	case TSL2583_TIMING_GAIN16:
		adc = adc >>4;
		break;
	case TSL2583_TIMING_GAIN111:
		if( ch == 0 )
			adc = adc / CH0GAIN128X;
		else if( ch == 1 )
			adc = adc / CH1GAIN128X;		
		break;		
	}

	return adc;
}

static void tsl2583_wait_adc(void)
{
	unsigned int delay;

	switch (chip->time) {
	case TSL2583_TIMING_3MS:
		delay = 3;
		break;
	case TSL2583_TIMING_5MS:
		delay = 5;
		break;
	case TSL2583_TIMING_51MS:
		delay = 51;
		break;
	case TSL2583_TIMING_100MS:
		delay = 100;
		break;			
	case TSL2583_TIMING_200MS:
		delay = 200;
		break;
	case TSL2583_TIMING_400MS:
		delay = 400;
		break;	
	case TSL2583_TIMING_688MS:
		delay = 688;
		break;				
	default:
		delay = 402;
	}
	
	schedule_timeout_interruptible(msecs_to_jiffies(delay) + 2);
}

static int tsl2583_get_adc(void)
{
	struct i2c_client *client = chip->client;
	u8 buf0[2], buf1[2];
	u8 val;
	u16 adc0, adc1;
	int ret = 0;

	if (chip->state.event != PM_EVENT_ON)
		goto out;

	cancel_delayed_work(&chip->poweroff_work);

	if (!tsl2583_get_power()) {
		ret = tsl2583_set_power(1);
		if (ret)
			goto out;
		ret = tsl2583_configure();
		if (ret)
			goto out;
		tsl2583_wait_adc();
	}

	tsl2583_set_adc(1);
	tsl2583_wait_adc();
	ret = tsl2583_read(client, TSL2583_REG_CTRL, &val, sizeof(val));
	if (ret != sizeof(val))
		goto out;	
	if(!(val & 0x1<<0x4))
	{
		printk(KERN_INFO "tsl2583 Data not valid, so return LAST VALUE\n");
		return -1;
	}

	ret = tsl2583_read(client,
			   TSL2583_REG_DATA0LOW,
			   buf0, sizeof(buf0));
	if (ret != sizeof(buf0))
		goto out;

	ret = tsl2583_read(client, TSL2583_REG_DATA1LOW,
			   buf1, sizeof(buf1));
	if (ret != sizeof(buf1))
		goto out;

	adc0 = (buf0[1] << 8) + buf0[0];
	adc1 = (buf1[1] << 8) + buf1[0];

	chip->data0 = normalize_adc(adc0, chip->gain, 0);
	chip->data1 = normalize_adc(adc1, chip->gain, 1);

	schedule_delayed_work(&chip->poweroff_work, 5 * HZ);

	ret = 0;
out:
	return ret;
}

static inline int calib_to_sysfs(u32 calib)
{
	return (int) (((calib * CALIB_BASE_SYSFS) +
		       CALIB_FRAC_HALF) >> CALIB_FRAC_BITS);
}

static inline u32 calib_from_sysfs(int value)
{
	return (((u32) value) << CALIB_FRAC_BITS) / CALIB_BASE_SYSFS;
}



struct tsl2583_lux_coeff {
	unsigned long ch_ratio;
	unsigned long ch0_coeff;
	unsigned long ch1_coeff;
};



















static const struct tsl2583_lux_coeff lux_table[] = {
	{
		.ch_ratio	= FRAC10K(3000),
		.ch0_coeff	= FRAC10K(1300),
		.ch1_coeff	= FRAC10K(2400),
	}, {
		.ch_ratio	= FRAC10K(3800),
		.ch0_coeff	= FRAC10K(1649),
		.ch1_coeff	= FRAC10K(3562),
	}, {
		.ch_ratio	= FRAC10K(4500),
		.ch0_coeff	= FRAC10K(974),
		.ch1_coeff	= FRAC10K(1786),
	}, {
		.ch_ratio	= FRAC10K(5400),
		.ch0_coeff	= FRAC10K(620),
		.ch1_coeff	= FRAC10K(1000),
	}, {
		.ch_ratio	= ULONG_MAX,
		.ch0_coeff	= 0,
		.ch1_coeff	= 0,
	},

};


static unsigned int adc_to_lux(u32 adc0, u32 adc1)
{
	const struct tsl2583_lux_coeff *lp = lux_table;
	unsigned long ratio, lux, ch0 = adc0, ch1 = adc1;

	ratio = ch0 ? ((ch1 << ADC_FRAC_BITS) / ch0) : ULONG_MAX;

	while (lp->ch_ratio < ratio)
		lp++;

	lux = ch0 * lp->ch0_coeff - ch1 * lp->ch1_coeff;

	return (unsigned int) (lux >> ADC_FRAC_BITS);
}










static ssize_t tsl2583_adc0_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	int ret;

	mutex_lock(&chip->lock);

	ret = tsl2583_get_adc();
	if (ret)
		goto out;

	ret = snprintf(buf, PAGE_SIZE, "0x%X\n", chip->data0);
out:
	mutex_unlock(&chip->lock);
	return ret;
}

static ssize_t tsl2583_adc1_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	int ret;

	mutex_lock(&chip->lock);

	ret = tsl2583_get_adc();
	if (ret)
		goto out;

	ret = snprintf(buf, PAGE_SIZE, "0x%X\n", chip->data1);
out:
	mutex_unlock(&chip->lock);
	return ret;
}


static u32 calib_adc(u32 adc, u32 calib)
{
	unsigned long scaled = adc;

	scaled *= calib;
	scaled >>= CALIB_FRAC_BITS;

	return (u32) scaled;
}

static ssize_t tsl2583_lux_show(struct device *dev,
				struct device_attribute *attr, char *buf)
{
	u32 calib0, calib1;
	int ret;

	mutex_lock(&chip->lock);

	ret = tsl2583_get_adc();
	if (ret)
		goto out;

	calib0 = calib_adc(chip->data0, chip->calib0) * chip->cover_comp_gain;
	calib1 = calib_adc(chip->data1, chip->calib1) * chip->cover_comp_gain;

	ret = snprintf(buf, PAGE_SIZE, "0x%X\n", adc_to_lux(calib0, calib1));

out:
	mutex_unlock(&chip->lock);
	return ret;
}

static ssize_t format_calib(char *buf, int len, u32 calib)
{
	return snprintf(buf, PAGE_SIZE, "%d\n", calib_to_sysfs(calib));
}

static ssize_t tsl2583_calib0_show(struct device *dev,
				   struct device_attribute *attr, char *buf)
{
	int ret;

	mutex_lock(&chip->lock);
	ret = format_calib(buf, PAGE_SIZE, chip->calib0);
	mutex_unlock(&chip->lock);
	return ret;
}

static ssize_t tsl2583_calib1_show(struct device *dev,
				   struct device_attribute *attr, char *buf)
{
	int ret;

	mutex_lock(&chip->lock);
	ret = format_calib(buf, PAGE_SIZE, chip->calib1);
	mutex_unlock(&chip->lock);
	return ret;
}

static int do_calib_store(struct device *dev, const char *buf, size_t len,
			  int ch)
{
	int value;
	u32 calib;

	if (1 != sscanf(buf, "%d", &value))
		return -EINVAL;

	calib = calib_from_sysfs(value);

	if (ch)
		chip->calib1 = calib;
	else
		chip->calib0 = calib;

	return len;
}

static ssize_t tsl2583_calib0_store(struct device *dev,
				    struct device_attribute *attr,
				    const char *buf, size_t len)
{
	return do_calib_store(dev, buf, len, 0);
}

static ssize_t tsl2583_calib1_store(struct device *dev,
				    struct device_attribute *attr,
				    const char *buf, size_t len)
{
	return do_calib_store(dev, buf, len, 1);
}


static DEVICE_ATTR(adc0, S_IRUGO, tsl2583_adc0_show, NULL);
static DEVICE_ATTR(adc1, S_IRUGO, tsl2583_adc1_show, NULL);
static DEVICE_ATTR(lux, S_IRUGO, tsl2583_lux_show, NULL);
static DEVICE_ATTR(calib0, S_IRUGO | S_IWUSR,
		   tsl2583_calib0_show, tsl2583_calib0_store);
static DEVICE_ATTR(calib1, S_IRUGO | S_IWUSR,
		   tsl2583_calib1_show, tsl2583_calib1_store);

static ssize_t tsl2583_show_name(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	return sprintf(buf, "%s\n", chip->client->name);
}

static DEVICE_ATTR(name, S_IRUGO, tsl2583_show_name, NULL);

static struct attribute *tsl2583_attributes[] = {
	&dev_attr_adc0.attr,
	&dev_attr_adc1.attr,
	&dev_attr_lux.attr,
	&dev_attr_calib0.attr,
	&dev_attr_calib1.attr,
	&dev_attr_name.attr,
	NULL
};

static const struct attribute_group tsl2583_group = {
	.attrs = tsl2583_attributes,
};

static int lightsensor_open(struct inode *inode, struct file *file)
{
	int rc = 0;
	pr_debug("%s\n", __func__);

	return rc;
}

static int lightsensor_release(struct inode *inode, struct file *file)
{
	pr_debug("%s\n", __func__);

	return 0;
}

static long lightsensor_ioctl(struct file *file, unsigned int cmd,
unsigned long arg)
{
	int ret=0, val=0;
	u32 calib0, calib1;
	unsigned char chip_id;	
	struct i2c_client *client = chip->client;	
	mutex_lock(&chip->lock);

	switch (cmd) {
	case LIGHTSENSOR_IOCTL_GET_CHIPID:
		ret = tsl2583_read(client, TSL2583_REG_ID, &chip_id, sizeof(chip_id));
		if (ret != sizeof(chip_id)) return ret;
		ret = put_user(chip_id, (unsigned char __user *)arg);
		break;
	case LIGHTSENSOR_IOCTL_ENABLE:
		if (get_user(val, (unsigned long __user *)arg)) {
			ret= -EFAULT;
			printk(KERN_INFO "%s get_user ERROR!\n", __func__);
			break;
		}
		
		ret = val ? 	tsl2583_set_power(1) : 	tsl2583_set_power(0);
		break;
	case LIGHTSENSOR_IOCTL_GET_ENABLED:
		val = tsl2583_get_power();
		
		ret = put_user(val, (unsigned long __user *)arg);
		break;
	case LIGHTSENSOR_IOCTL_GET_ADC:
		tsl2583_get_adc();
		calib0 = calib_adc(chip->data0, chip->calib0) * chip->cover_comp_gain;
		calib1 = calib_adc(chip->data1, chip->calib1) * chip->cover_comp_gain;
		val = adc_to_lux(calib0, calib1);
		ret= put_user(val, (unsigned long __user *)arg);
		break;
	default:
		pr_err("%s: invalid cmd %d\n", __func__, _IOC_NR(cmd));
		ret = -EINVAL;
	}

	mutex_unlock(&chip->lock);
	return ret;
}


static struct file_operations lightsensor_fops = {
	.owner = THIS_MODULE,
	.open = lightsensor_open,
	.release = lightsensor_release,
	.unlocked_ioctl = lightsensor_ioctl
};

static struct miscdevice lightsensor_misc = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "light-sensor",
	.fops = &lightsensor_fops
};




static struct i2c_driver tsl2583_i2c_driver;

static int __devinit tsl2583_probe(struct i2c_client *client,
				const struct i2c_device_id *device_id)
{	
	int err = 0;
	int ret;
	u8 id;

	struct kobject *module_kobj; 
	struct kobject *tsl2583_kobj;
	struct tsl2583_platform_data *pdata = client->dev.platform_data;

	chip = kzalloc(sizeof(*chip), GFP_KERNEL);
	if (!chip)
		return -ENOMEM;

	i2c_set_clientdata(client, chip);
	chip->client = client;

	tsl2583_power(1);
	err = tsl2583_detect();
	if (err) {
		dev_err(&client->dev, "device not found, error %d\n", -err);
		goto fail1;
	}

	err = tsl2583_read_id(&id);
	if (err)
		goto fail1;

	mutex_init(&chip->lock);

	
	chip->low_thres = 0x0;
	chip->high_thres = 0xffff;
	chip->gain = TSL2583_TIMING_GAIN8;
	chip->time = TSL2583_TIMING_400MS;
	chip->intr = TSL2583_INT_PERSIST(4);
	chip->calib0 = calib_from_sysfs(CALIB_BASE_SYSFS);
	chip->calib1 = calib_from_sysfs(CALIB_BASE_SYSFS);

	if (pdata)
		chip->cover_comp_gain = pdata->cover_comp_gain;
	else
		chip->cover_comp_gain = 1;

	dev_info(&client->dev, "tsl2583 model 0x%X, rev. 0x%X\n", id >> 4, id & 0x0f);

	chip->ls_input_dev = input_allocate_device();
	if (!chip->ls_input_dev) {
		pr_err("%s: could not allocate ls input device\n", __func__);
		return -ENOMEM;
	}

	chip->ls_input_dev->name = "lightsensor-level";
	set_bit(EV_ABS, chip->ls_input_dev->evbit);
	input_set_abs_params(chip->ls_input_dev, ABS_MISC, 0, 9, 0, 0);

	ret = input_register_device(chip->ls_input_dev);
	if (ret < 0) {
		pr_err("%s: can not register ls input device\n",
		__func__);
		return ret;
	}

	ret = misc_register(&lightsensor_misc);
	if (ret < 0) {
		pr_err("%s: can not register ls misc device\n",
		__func__);
		return ret;
	}

	module_kobj = &(lightsensor_misc.this_device->kobj); 
	tsl2583_kobj = kobject_create_and_add("sensors_detail", module_kobj);
	ret = sysfs_create_group(tsl2583_kobj, &tsl2583_group);
	
	err = tsl2583_configure();
	if (err)
		goto fail2;

	INIT_DELAYED_WORK(&chip->poweroff_work, tsl2583_poweroff_work);
	schedule_delayed_work(&chip->poweroff_work, 5 * HZ);

	return 0;
fail2:
	input_unregister_device(chip->ls_input_dev);
	misc_deregister(&lightsensor_misc);
	tsl2583_power(0);

fail1:
	kfree(chip);
	return err;
}

static int tsl2583_remove(struct i2c_client *client)
{
	struct tsl2583_chip *chip = i2c_get_clientdata(client);

	free_irq(client->irq, NULL);
	input_unregister_device(chip->ls_input_dev);
	misc_deregister(&lightsensor_misc);
	tsl2583_power(0);

	kfree(chip);
	return 0;
}

static int tsl2583_suspend(struct i2c_client *client, pm_message_t state)
{
	struct tsl2583_chip *chip = i2c_get_clientdata(client);
	int ret;

	mutex_lock(&chip->lock);

	ret = tsl2583_set_power(0);
	if (ret)
		goto out;

	chip->state = state;

out:
	mutex_unlock(&chip->lock);
	return ret;
}

static int tsl2583_resume(struct i2c_client *client)
{
	struct tsl2583_chip *chip = i2c_get_clientdata(client);
	int ret;

	mutex_lock(&chip->lock);

	ret = tsl2583_set_power(1);
	if (ret)
		goto out;

	ret = tsl2583_configure();
	if (ret)
		goto out;

	chip->state.event = PM_EVENT_ON;

out:
	mutex_unlock(&chip->lock);
	return ret;
}

static const struct i2c_device_id tsl2583_id[] = {
	{ "tsl2580", 0 },
	{ "tsl2581", 1 },
	{ "tsl2582", 2 },
	{ "tsl2583", 3 },
	{}
};
MODULE_DEVICE_TABLE(i2c, tsl2583_id);

static struct i2c_driver tsl2583_i2c_driver = {
	.driver = {
		.name	 = "tsl2583",
	},
	.suspend	= tsl2583_suspend,
	.resume		= tsl2583_resume,
	.probe		= tsl2583_probe,
	.remove		= __devexit_p(tsl2583_remove),
	.id_table	= tsl2583_id,
};

static int __init tsl2583_init(void)
{
	return i2c_add_driver(&tsl2583_i2c_driver);
}

static void __exit tsl2583_exit(void)
{
	i2c_del_driver(&tsl2583_i2c_driver);
}

MODULE_AUTHOR("Qisda Corporation");
MODULE_DESCRIPTION("tsl2583 light sensor driver");
MODULE_LICENSE("GPL");

module_init(tsl2583_init);
module_exit(tsl2583_exit);
