/*
 *  linux/drivers/mmc/core/mmc.c
 *
 *  Copyright (C) 2003-2004 Russell King, All Rights Reserved.
 *  Copyright (C) 2005-2007 Pierre Ossman, All Rights Reserved.
 *  MMCv4 support Copyright (C) 2006 Philip Langdale, All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/err.h>
#include <linux/slab.h>

#include <linux/mmc/host.h>
#include <linux/mmc/card.h>
#include <linux/mmc/mmc.h>

#include "core.h"
#include "bus.h"
#include "mmc_ops.h"

static const unsigned int tran_exp[] = {
	10000,		100000,		1000000,	10000000,
	0,		0,		0,		0
};

static const unsigned char tran_mant[] = {
	0,	10,	12,	13,	15,	20,	25,	30,
	35,	40,	45,	50,	55,	60,	70,	80,
};

static const unsigned int tacc_exp[] = {
	1,	10,	100,	1000,	10000,	100000,	1000000, 10000000,
};

static const unsigned int tacc_mant[] = {
	0,	10,	12,	13,	15,	20,	25,	30,
	35,	40,	45,	50,	55,	60,	70,	80,
};

#define UNSTUFF_BITS(resp,start,size)					\
	({								\
		const int __size = size;				\
		const u32 __mask = (__size < 32 ? 1 << __size : 0) - 1;	\
		const int __off = 3 - ((start) / 32);			\
		const int __shft = (start) & 31;			\
		u32 __res;						\
									\
		__res = resp[__off] >> __shft;				\
		if (__size + __shft > 32)				\
			__res |= resp[__off-1] << ((32 - __shft) % 32);	\
		__res & __mask;						\
	})

/*
 * Given the decoded CSD structure, decode the raw CID to our CID structure.
 */
static int mmc_decode_cid(struct mmc_card *card)
{
	u32 *resp = card->raw_cid;

	/*
	 * The selection of the format here is based upon published
	 * specs from sandisk and from what people have reported.
	 */
	switch (card->csd.mmca_vsn) {
	case 0: /* MMC v1.0 - v1.2 */
	case 1: /* MMC v1.4 */
		card->cid.manfid	= UNSTUFF_BITS(resp, 104, 24);
		card->cid.prod_name[0]	= UNSTUFF_BITS(resp, 96, 8);
		card->cid.prod_name[1]	= UNSTUFF_BITS(resp, 88, 8);
		card->cid.prod_name[2]	= UNSTUFF_BITS(resp, 80, 8);
		card->cid.prod_name[3]	= UNSTUFF_BITS(resp, 72, 8);
		card->cid.prod_name[4]	= UNSTUFF_BITS(resp, 64, 8);
		card->cid.prod_name[5]	= UNSTUFF_BITS(resp, 56, 8);
		card->cid.prod_name[6]	= UNSTUFF_BITS(resp, 48, 8);
		card->cid.hwrev		= UNSTUFF_BITS(resp, 44, 4);
		card->cid.fwrev		= UNSTUFF_BITS(resp, 40, 4);
		card->cid.serial	= UNSTUFF_BITS(resp, 16, 24);
		card->cid.month		= UNSTUFF_BITS(resp, 12, 4);
		card->cid.year		= UNSTUFF_BITS(resp, 8, 4) + 1997;
		break;

	case 2: /* MMC v2.0 - v2.2 */
	case 3: /* MMC v3.1 - v3.3 */
	case 4: /* MMC v4 */
		card->cid.manfid	= UNSTUFF_BITS(resp, 120, 8);
		card->cid.oemid		= UNSTUFF_BITS(resp, 104, 16);
		card->cid.prod_name[0]	= UNSTUFF_BITS(resp, 96, 8);
		card->cid.prod_name[1]	= UNSTUFF_BITS(resp, 88, 8);
		card->cid.prod_name[2]	= UNSTUFF_BITS(resp, 80, 8);
		card->cid.prod_name[3]	= UNSTUFF_BITS(resp, 72, 8);
		card->cid.prod_name[4]	= UNSTUFF_BITS(resp, 64, 8);
		card->cid.prod_name[5]	= UNSTUFF_BITS(resp, 56, 8);
		card->cid.serial	= UNSTUFF_BITS(resp, 16, 32);
		card->cid.month		= UNSTUFF_BITS(resp, 12, 4);
		card->cid.year		= UNSTUFF_BITS(resp, 8, 4) + 1997;
		break;

	default:
		printk(KERN_ERR "%s: card has unknown MMCA version %d\n",
			mmc_hostname(card->host), card->csd.mmca_vsn);
		return -EINVAL;
	}

	return 0;
}

/*
 * Given a 128-bit response, decode to our card CSD structure.
 */
static int mmc_decode_csd(struct mmc_card *card)
{
	struct mmc_csd *csd = &card->csd;
	unsigned int e, m;
	u32 *resp = card->raw_csd;

	/*
	 * We only understand CSD structure v1.1 and v1.2.
	 * v1.2 has extra information in bits 15, 11 and 10.
	 * We also support eMMC v4.4 & v4.41.
	 */
	csd->structure = UNSTUFF_BITS(resp, 126, 2);
	if (csd->structure == 0) {
		printk(KERN_ERR "%s: unrecognised CSD structure version %d\n",
			mmc_hostname(card->host), csd->structure);
		return -EINVAL;
	}

	csd->mmca_vsn	 = UNSTUFF_BITS(resp, 122, 4);
	m = UNSTUFF_BITS(resp, 115, 4);
	e = UNSTUFF_BITS(resp, 112, 3);
	csd->tacc_ns	 = (tacc_exp[e] * tacc_mant[m] + 9) / 10;
	csd->tacc_clks	 = UNSTUFF_BITS(resp, 104, 8) * 100;

	m = UNSTUFF_BITS(resp, 99, 4);
	e = UNSTUFF_BITS(resp, 96, 3);
	csd->max_dtr	  = tran_exp[e] * tran_mant[m];
	csd->cmdclass	  = UNSTUFF_BITS(resp, 84, 12);

	e = UNSTUFF_BITS(resp, 47, 3);
	m = UNSTUFF_BITS(resp, 62, 12);
	csd->capacity	  = (1 + m) << (e + 2);

	csd->read_blkbits = UNSTUFF_BITS(resp, 80, 4);
	csd->read_partial = UNSTUFF_BITS(resp, 79, 1);
	csd->write_misalign = UNSTUFF_BITS(resp, 78, 1);
	csd->read_misalign = UNSTUFF_BITS(resp, 77, 1);
	csd->r2w_factor = UNSTUFF_BITS(resp, 26, 3);
	csd->write_blkbits = UNSTUFF_BITS(resp, 22, 4);
	csd->write_partial = UNSTUFF_BITS(resp, 21, 1);

	return 0;
}

/*
 * Read and decode extended CSD.
 */
static int mmc_read_ext_csd(struct mmc_card *card)
{
	int err;
	u8 *ext_csd;

	BUG_ON(!card);

	if (card->csd.mmca_vsn < CSD_SPEC_VER_4)
		return 0;

	/*
	 * As the ext_csd is so large and mostly unused, we don't store the
	 * raw block in mmc_card.
	 */
	ext_csd = kmalloc(512, GFP_KERNEL);
	if (!ext_csd) {
		printk(KERN_ERR "%s: could not allocate a buffer to "
			"receive the ext_csd.\n", mmc_hostname(card->host));
		return -ENOMEM;
	}

	err = mmc_send_ext_csd(card, ext_csd);
	if (err) {
		/* If the host or the card can't do the switch,
		 * fail more gracefully. */
		if ((err != -EINVAL)
		 && (err != -ENOSYS)
		 && (err != -EFAULT))
			goto out;

		/*
		 * High capacity cards should have this "magic" size
		 * stored in their CSD.
		 */
		if (card->csd.capacity == (4096 * 512)) {
			printk(KERN_ERR "%s: unable to read EXT_CSD "
				"on a possible high capacity card. "
				"Card will be ignored.\n",
				mmc_hostname(card->host));
		} else {
			printk(KERN_WARNING "%s: unable to read "
				"EXT_CSD, performance might "
				"suffer.\n",
				mmc_hostname(card->host));
			err = 0;
		}

		goto out;
	}

	/* Version is coded in the CSD_STRUCTURE byte in the EXT_CSD register */
	if (card->csd.structure == 3) {
		int ext_csd_struct = ext_csd[EXT_CSD_STRUCTURE];
		if (ext_csd_struct > 2) {
			printk(KERN_ERR "%s: unrecognised EXT_CSD structure "
				"version %d\n", mmc_hostname(card->host),
					ext_csd_struct);
			err = -EINVAL;
			goto out;
		}
	}

	card->ext_csd.rev = ext_csd[EXT_CSD_REV];
	if (card->ext_csd.rev > 5) {
		printk(KERN_ERR "%s: unrecognised EXT_CSD revision %d\n",
			mmc_hostname(card->host), card->ext_csd.rev);
		err = -EINVAL;
		goto out;
	}

	if (card->ext_csd.rev >= 2) {
		card->ext_csd.sectors =
			ext_csd[EXT_CSD_SEC_CNT + 0] << 0 |
			ext_csd[EXT_CSD_SEC_CNT + 1] << 8 |
			ext_csd[EXT_CSD_SEC_CNT + 2] << 16 |
			ext_csd[EXT_CSD_SEC_CNT + 3] << 24;
		if (card->ext_csd.sectors) {
			unsigned boot_sectors;
			/* size is in 256K chunks, i.e. 512 sectors each */
			boot_sectors = ext_csd[EXT_CSD_BOOT_SIZE_MULTI] * 512;
			card->ext_csd.sectors -= boot_sectors;
		}
	}

	switch (ext_csd[EXT_CSD_CARD_TYPE] & EXT_CSD_CARD_TYPE_MASK) {
	case EXT_CSD_CARD_TYPE_52 | EXT_CSD_CARD_TYPE_26:
		card->ext_csd.hs_max_dtr = 52000000;
		break;
	case EXT_CSD_CARD_TYPE_26:
		card->ext_csd.hs_max_dtr = 26000000;
		break;
	default:
		/* MMC v4 spec says this cannot happen */
		printk(KERN_WARNING "%s: card is mmc v4 but doesn't "
			"support any high-speed modes.\n",
			mmc_hostname(card->host));
	}

	if (card->ext_csd.rev >= 3) {
		u8 sa_shift = ext_csd[EXT_CSD_S_A_TIMEOUT];

		/* Sleep / awake timeout in 100ns units */
		if (sa_shift > 0 && sa_shift <= 0x17)
			card->ext_csd.sa_timeout =
					1 << ext_csd[EXT_CSD_S_A_TIMEOUT];
	}

	/* set ERASE_GROUP_DEF since we use WP feature */
	if (card->ext_csd.rev > 3) {
		if (ext_csd[EXT_CSD_ERASE_GROUP_DEF] == 0) {
			err = mmc_switch(card, 0, EXT_CSD_ERASE_GROUP_DEF, 1);

			if (err && err != -EBADMSG)
				goto out;

			if (err) {
				printk(KERN_ERR "%s: faile to enable ERASE_GROUP_DEF.\n",
						mmc_hostname(card->host));
				err = 0;
			} 
		}
	}

	card->ext_csd.rst_n_func = ext_csd[EXT_CSD_RST_n_FUNCTION];
	card->ext_csd.fw_config = ext_csd[EXT_CSD_FW_CONFIG];

out:
	kfree(ext_csd);

	return err;
}


#define MMC_TRAN_STATE              4
#define MMC_PROG_STATE              7

#define MMC_E_SUCCESS                0
#define MMC_E_FAILURE                1

typedef enum {
	MMC_WRITE_PROTECT_NONE = 0,              // No protection is enabled.
	MMC_WRITE_PROTECT_PERMANENT,             // Permanent protection.
	MMC_WRITE_PROTECT_POWER_ON,              // Turn on power on protection.
	MMC_WRITE_PROTECT_TEMPORARY_ON,          // Turn on temporary protection.
	MMC_WRITE_PROTECT_TEMPORARY_OFF,          // Turn off temporary protection. Only this mode is to turn off protection.
	MMC_WRITE_PROTECT_MAX,
} MMC_WRITE_PROTECT_MODE; 


static int mmc_set_wp_mode_user(struct mmc_card *card,
	unsigned int addr, unsigned int size, unsigned char wp_mode)
{
	struct mmc_command cmd;
	unsigned int wp_group_size, loop_count;
	u32 status;
	u8 ext_csd[512], old_user_wp;
	int i;

	int err = MMC_E_FAILURE;

	printk(KERN_ERR  "%s+++\n", __func__);

	BUG_ON(!card);

	if (card->csd.mmca_vsn < CSD_SPEC_VER_4)
		goto wp_out;

	err = mmc_send_ext_csd(card, ext_csd);
	if (err) {
		printk(KERN_ERR "%s: faile to read "
				"EXT_CSD.\n",
				mmc_hostname(card->host));

		goto wp_out;
	}
	old_user_wp = ext_csd[EXT_CSD_USER_WP];
	printk(KERN_ERR  "1EXT_USER_WP: %x, ERASE_GROUP_DEF:%d\n", \
		ext_csd[EXT_CSD_USER_WP], ext_csd[EXT_CSD_ERASE_GROUP_DEF]);

	if(ext_csd[EXT_CSD_ERASE_GROUP_DEF] == 0)
	{
		err = mmc_switch(card, 0, EXT_CSD_ERASE_GROUP_DEF, 1);
		if (err) {
			printk(KERN_ERR "%s: faile to enable ERASE_GROUP_DEF.\n",
					mmc_hostname(card->host));
			goto wp_out;
		}
		do
		{
			err = mmc_send_status(card, &status);
			if(R1_CURRENT_STATE(status) == MMC_TRAN_STATE)
				break;
		} while( (err == MMC_E_SUCCESS) && (R1_CURRENT_STATE(status) == MMC_PROG_STATE));

		if( err != MMC_E_SUCCESS )
		{
			printk(KERN_ERR  "Sending CMD13 to check card status fail: %x\n", err);
			goto wp_out;
		}

		ext_csd[EXT_CSD_ERASE_GROUP_DEF] = 1;
	}

	memset( (struct mmc_command *)&cmd, 0,
			sizeof(struct mmc_command) );

#if 0
	mmc_ret = mmc_boot_switch_cmd(card, MMC_BOOT_ACCESS_WRITE,
								  MMC_BOOT_EXT_USER_WP,
								  MMC_BOOT_US_PERM_WP_DIS);

	if(mmc_ret != MMC_E_SUCCESS)
	{
		printk(KERN_ERR  "Disable PERM_WP for USER AREA fail: %x\n", mmc_ret);
		return mmc_ret;
	}


	do
	{
		err = mmc_send_status(card, &status);
		if(R1_CURRENT_STATE(status) == MMC_TRAN_STATE)
			break;
	} while( (err == MMC_E_SUCCESS) && (R1_CURRENT_STATE(status) == MMC_PROG_STATE));

	if( err != MMC_E_SUCCESS )
	{
		printk(KERN_ERR  "Sending CMD13 to check card status fail: %x\n", err);
		goto wp_out;
	}
#endif




	if(ext_csd[EXT_CSD_ERASE_GROUP_DEF])
	{
		printk(KERN_ERR  "HC_WP_GRP_SIZE: %d, HC_ERASE_GRP_SIZE: %d\n", \
					ext_csd[EXT_CSD_HC_WP_GRP_SIZE], \
					ext_csd[EXT_CSD_HC_ERASE_GRP_SIZE]);

		wp_group_size = (512*1024) * ext_csd[EXT_CSD_HC_WP_GRP_SIZE] *
					ext_csd[EXT_CSD_HC_ERASE_GRP_SIZE] / 512;
	}
	else
	{
		printk(KERN_ERR  "card->csd.erase_grp_size: %d, erase_grp_mult: %d, wp_grp_size: %d\n", \
					UNSTUFF_BITS(card->raw_csd, 42, 5), UNSTUFF_BITS(card->raw_csd, 37, 5), \
					UNSTUFF_BITS(card->raw_csd, 32, 5));

		wp_group_size = (UNSTUFF_BITS(card->raw_csd, 42, 5) + 1) *
					(UNSTUFF_BITS(card->raw_csd, 37, 5) + 1) *
					(UNSTUFF_BITS(card->raw_csd, 32, 5) + 1);
	}

	printk(KERN_ERR  "wp_group_size: %d\n", wp_group_size);

	if(wp_group_size == 0)
	{
		return MMC_E_FAILURE;
	}

	switch (wp_mode) {
		case MMC_WRITE_PROTECT_PERMANENT:
			if ((ext_csd[EXT_CSD_USER_WP] & EXT_CSD_US_PERM_WP_DIS) ||
				(ext_csd[EXT_CSD_USER_WP] & EXT_CSD_CD_PERM_WP_DIS))
			{
			   printk(KERN_ERR "User permanent WP has been disabled");
			   return MMC_E_FAILURE;
			}
			err = mmc_switch(card, 0, EXT_CSD_USER_WP, EXT_CSD_US_PERM_WP_EN);
			break;

		case MMC_WRITE_PROTECT_POWER_ON:
			if (ext_csd[EXT_CSD_USER_WP] & EXT_CSD_US_PWR_WP_DIS)
			{
			   printk(KERN_ERR "Use of user power-on WP has been disabled");
			   return MMC_E_FAILURE;
			}
			err = mmc_switch(card, 0, EXT_CSD_USER_WP, EXT_CSD_US_PWR_WP_EN);
			break;		
				
		case MMC_WRITE_PROTECT_TEMPORARY_ON:
		case MMC_WRITE_PROTECT_TEMPORARY_OFF:
			printk(KERN_ERR "TEMPORARY wp");
			err = mmc_switch(card, 0, EXT_CSD_USER_WP, 0);
			break;

		default:
			return MMC_E_FAILURE;
	}

	if(err != MMC_E_SUCCESS)
	{
		printk(KERN_ERR  "Setting WP for USER AREA fail: %x\n", err);
		goto restore_wp_mode;
	}

	do
	{
		err = mmc_send_status(card, &status);
		if(R1_CURRENT_STATE(status) == MMC_TRAN_STATE)
			break;
	} while( (err == MMC_E_SUCCESS) && (R1_CURRENT_STATE(status) == MMC_PROG_STATE));

	if( err != MMC_E_SUCCESS )
	{
		printk(KERN_ERR  "Sending CMD13 to check card status fail: %x\n", err);
		goto restore_wp_mode;
	}

	err = mmc_send_ext_csd(card, ext_csd);
	if (err) {
		printk(KERN_ERR "%s: faile to read "
				"EXT_CSD.\n",
				mmc_hostname(card->host));

		goto restore_wp_mode;
	}
	printk(KERN_ERR  "3EXT_USER_WP: %x, ERASE_GROUP_DEF:%d\n", \
		ext_csd[EXT_CSD_USER_WP], ext_csd[EXT_CSD_ERASE_GROUP_DEF]);


	if(size % wp_group_size)
	{
		loop_count = (size / wp_group_size) + 1;
	}
	else
	{
		loop_count = (size / wp_group_size);
	}
	printk(KERN_ERR  "wp size=%d * %d\n", loop_count, wp_group_size);

	if(MMC_WRITE_PROTECT_TEMPORARY_OFF != wp_mode)
		cmd.opcode = MMC_SET_WRITE_PROT;	/* CMD28 */
	else
		cmd.opcode = MMC_CLR_WRITE_PROT;	/* CMD29 */

	cmd.flags= MMC_RSP_R1B | MMC_CMD_AC;

	for(i=0;i<loop_count;i++)
	{
		cmd.arg = (addr + (i * wp_group_size));

		printk(KERN_ERR "i=%d,cmd.arg:%x\n", i, cmd.arg);
		err = mmc_wait_for_cmd(card->host, &cmd, MMC_CMD_RETRIES);

		if(err != MMC_E_SUCCESS)
		{
			goto restore_wp_mode;
		}

		printk(KERN_ERR "resp[0]:%x\n", cmd.resp[0]);
		if(cmd.resp[0] & R1_OUT_OF_RANGE)
		{
			printk(KERN_ERR "address argument was out of the allowed range."\
				"start sector:%d, size:%d\n", addr, size);
			err = MMC_E_FAILURE;
			goto restore_wp_mode;
		}

		do
		{
			err = mmc_send_status(card, &status);
			if(R1_CURRENT_STATE(status) == MMC_TRAN_STATE)
				break;
		} while( (err == MMC_E_SUCCESS) && (R1_CURRENT_STATE(status) == MMC_PROG_STATE));

		if( err != MMC_E_SUCCESS )
		{
			printk(KERN_ERR  "Sending CMD13 to check card status fail: %x\n", err);
			goto restore_wp_mode;
		}
	}
	return MMC_E_SUCCESS;

restore_wp_mode:
	mmc_switch(card, 0, EXT_CSD_USER_WP, old_user_wp);

wp_out:
	printk(KERN_ERR  "%s---\n", __func__);
	return err;
}

static unsigned int mmc_get_wp_status (struct mmc_card *card,
	unsigned int sector, unsigned char *p_mode)
{
	unsigned int rc = MMC_E_FAILURE, resp_len = 8;
	unsigned char wp_status_buf[512];
	memset(wp_status_buf,0, sizeof(wp_status_buf));

	if(card->cid.manfid == 0x02)
	{
		resp_len = 512;
	}
	rc = mmc_send_data(card, MMC_SEND_WRITE_PROT_TYPE, sector, wp_status_buf, resp_len);

	printk(KERN_ERR "CMD31 resp:%x, rc:%d, resp_len:%d\n",wp_status_buf[7], rc, resp_len);
	if(rc != MMC_E_SUCCESS)
	{
		printk(KERN_ERR "Fail to get wp status, rc:%d\n", rc);
		return rc;
	}
	*p_mode = wp_status_buf[7] & 0x3;
	return rc;

}

static unsigned int g_start_sector = 0, g_num_of_sector = 0;
static unsigned int g_result = MMC_E_FAILURE;
ssize_t mmc_export_show(struct device *dev,
	struct device_attribute *attr, char *buf)
{

	return sprintf(buf, "0x%x %d\n", \
		g_start_sector, g_num_of_sector);
}

ssize_t mmc_export_store(struct device *dev,
	struct device_attribute *attr, const char *buf, size_t count)
{
	printk(KERN_ERR "mmc_export_store+++++++\n");
	sscanf(buf, "%x %d", &g_start_sector, &g_num_of_sector);

	printk(KERN_ERR "0x%x(%d) %d\n",g_start_sector, g_start_sector, g_num_of_sector);
	return count;
}

ssize_t mmc_user_wp_show(struct device *dev,
	struct device_attribute *attr, char *buf)
{
	struct mmc_card *card = container_of(dev, struct mmc_card, dev);
	u8 mode = 0;
	u8 ext_csd[512];
	int err = MMC_E_FAILURE;

	mmc_claim_host(card->host);

	err = mmc_send_ext_csd(card, ext_csd);
	if (err) {
		printk(KERN_ERR "%s: faile to read "
				"EXT_CSD.\n",
				mmc_hostname(card->host));

	}
	printk(KERN_ERR  "1EXT_USER_WP: %x, ERASE_GROUP_DEF:%d,0x%x\n", \
		ext_csd[EXT_CSD_USER_WP], ext_csd[EXT_CSD_ERASE_GROUP_DEF], ext_csd[179]);

	mmc_get_wp_status(card, g_start_sector, &mode);
	mmc_release_host(card->host);

	return sprintf(buf, "%d %d\n", mode, g_result);
}

ssize_t mmc_user_wp_store(struct device *dev,
	struct device_attribute *attr, const char *buf, size_t count)
{
	unsigned int set;
	struct mmc_card *card = container_of(dev, struct mmc_card, dev);

	printk(KERN_ERR "mmc_user_wp_store+++++++\n");
	sscanf(buf, "%d", &set);

	g_result = MMC_E_FAILURE;
	if((g_start_sector == 0) || (g_num_of_sector == 0)) {
		printk(KERN_ERR  "start:%x or num:%d is invalid\n", \
			g_start_sector, g_num_of_sector);
		return count;
	}
	if((set == 0) || (set >= MMC_WRITE_PROTECT_MAX)) {
		printk(KERN_ERR  "Unknown wp mode:%x\n", set);
		return count;
	}
	
	mmc_claim_host(card->host);
	g_result = mmc_set_wp_mode_user(card, g_start_sector, g_num_of_sector, set);
	mmc_release_host(card->host);

	return count;
}

static DEVICE_ATTR(export, 0660, mmc_export_show, mmc_export_store);
static DEVICE_ATTR(user_wp, 0660, mmc_user_wp_show, mmc_user_wp_store);

MMC_DEV_ATTR(cid, "%08x%08x%08x%08x\n", card->raw_cid[0], card->raw_cid[1],
	card->raw_cid[2], card->raw_cid[3]);
MMC_DEV_ATTR(csd, "%08x%08x%08x%08x\n", card->raw_csd[0], card->raw_csd[1],
	card->raw_csd[2], card->raw_csd[3]);
MMC_DEV_ATTR(date, "%02d/%04d\n", card->cid.month, card->cid.year);
MMC_DEV_ATTR(fwrev, "0x%x\n", card->cid.fwrev);
MMC_DEV_ATTR(hwrev, "0x%x\n", card->cid.hwrev);
MMC_DEV_ATTR(manfid, "0x%06x\n", card->cid.manfid);
MMC_DEV_ATTR(name, "%s\n", card->cid.prod_name);
MMC_DEV_ATTR(oemid, "0x%04x\n", card->cid.oemid);
MMC_DEV_ATTR(serial, "0x%08x\n", card->cid.serial);
MMC_DEV_ATTR(ext_csd, "%02x%02x\n", (u32)(card->ext_csd.rst_n_func),
	(u32)(card->ext_csd.fw_config));

static struct attribute *mmc_std_attrs[] = {
	&dev_attr_cid.attr,
	&dev_attr_csd.attr,
	&dev_attr_date.attr,
	&dev_attr_fwrev.attr,
	&dev_attr_hwrev.attr,
	&dev_attr_manfid.attr,
	&dev_attr_name.attr,
	&dev_attr_oemid.attr,
	&dev_attr_serial.attr,
	&dev_attr_ext_csd.attr,
	&dev_attr_export.attr,
	&dev_attr_user_wp.attr,
	NULL,
};

static struct attribute_group mmc_std_attr_group = {
	.attrs = mmc_std_attrs,
};

static const struct attribute_group *mmc_attr_groups[] = {
	&mmc_std_attr_group,
	NULL,
};

static struct device_type mmc_type = {
	.groups = mmc_attr_groups,
};

/*
 * Handle the detection and initialisation of a card.
 *
 * In the case of a resume, "oldcard" will contain the card
 * we're trying to reinitialise.
 */
static int mmc_init_card(struct mmc_host *host, u32 ocr,
	struct mmc_card *oldcard)
{
	struct mmc_card *card;
	int err;
	u32 cid[4];
	unsigned int max_dtr;
	u32 rocr;

	BUG_ON(!host);
	WARN_ON(!host->claimed);

	/*
	 * Since we're changing the OCR value, we seem to
	 * need to tell some cards to go back to the idle
	 * state.  We wait 1ms to give cards time to
	 * respond.
	 */
	mmc_go_idle(host);

	/* The extra bit indicates that we support high capacity */
	err = mmc_send_op_cond(host, ocr | MMC_CARD_SECTOR_ADDR, &rocr);
	if (err)
		goto err;

	/*
	 * For SPI, enable CRC as appropriate.
	 */
	if (mmc_host_is_spi(host)) {
		err = mmc_spi_set_crc(host, use_spi_crc);
		if (err)
			goto err;
	}

	/*
	 * Fetch CID from card.
	 */
	if (mmc_host_is_spi(host))
		err = mmc_send_cid(host, cid);
	else
		err = mmc_all_send_cid(host, cid);
	if (err)
		goto err;

	if (oldcard) {
		if (memcmp(cid, oldcard->raw_cid, sizeof(cid)) != 0) {
			err = -ENOENT;
			goto err;
		}

		card = oldcard;
	} else {
		/*
		 * Allocate card structure.
		 */
		card = mmc_alloc_card(host, &mmc_type);
		if (IS_ERR(card)) {
			err = PTR_ERR(card);
			goto err;
		}

		card->type = MMC_TYPE_MMC;
		card->rca = 1;
		memcpy(card->raw_cid, cid, sizeof(card->raw_cid));
	}

	/*
	 * For native busses:  set card RCA and quit open drain mode.
	 */
	if (!mmc_host_is_spi(host)) {
		err = mmc_set_relative_addr(card);
		if (err)
			goto free_card;

		mmc_set_bus_mode(host, MMC_BUSMODE_PUSHPULL);
	}

	if (!oldcard) {
		/*
		 * Fetch CSD from card.
		 */
		err = mmc_send_csd(card, card->raw_csd);
		if (err)
			goto free_card;

		err = mmc_decode_csd(card);
		if (err)
			goto free_card;
		err = mmc_decode_cid(card);
		if (err)
			goto free_card;
	}

	/*
	 * Select card, as all following commands rely on that.
	 */
	if (!mmc_host_is_spi(host)) {
		err = mmc_select_card(card);
		if (err)
			goto free_card;
	}

	if (!oldcard) {
		/*
		 * Fetch and process extended CSD.
		 */
		err = mmc_read_ext_csd(card);
		if (err)
			goto free_card;

		if (card->ext_csd.sectors && (rocr & MMC_CARD_SECTOR_ADDR))
			mmc_card_set_blockaddr(card);
	}

	/*
	 * Activate high speed (if supported)
	 */
	if ((card->ext_csd.hs_max_dtr != 0) &&
		(host->caps & MMC_CAP_MMC_HIGHSPEED)) {
		err = mmc_switch(card, EXT_CSD_CMD_SET_NORMAL,
			EXT_CSD_HS_TIMING, 1);
		if (err && err != -EBADMSG)
			goto free_card;

		if (err) {
			printk(KERN_WARNING "%s: switch to highspeed failed\n",
			       mmc_hostname(card->host));
			err = 0;
		} else {
			mmc_card_set_highspeed(card);
			mmc_set_timing(card->host, MMC_TIMING_MMC_HS);
		}
	}

	/*
	 * Compute bus speed.
	 */
	max_dtr = (unsigned int)-1;

	if (mmc_card_highspeed(card)) {
		if (max_dtr > card->ext_csd.hs_max_dtr)
			max_dtr = card->ext_csd.hs_max_dtr;
	} else if (max_dtr > card->csd.max_dtr) {
		max_dtr = card->csd.max_dtr;
	}

	mmc_set_clock(host, max_dtr);

	/*
	 * Activate wide bus (if supported).
	 */
	if ((card->csd.mmca_vsn >= CSD_SPEC_VER_4) &&
	    (host->caps & (MMC_CAP_4_BIT_DATA | MMC_CAP_8_BIT_DATA))) {
		unsigned ext_csd_bit, bus_width;

		if (host->caps & MMC_CAP_8_BIT_DATA) {
			pr_debug("Setting the bus width to 8 bit\n");
			ext_csd_bit = EXT_CSD_BUS_WIDTH_8;
			bus_width = MMC_BUS_WIDTH_8;
		} else if (host->caps & MMC_CAP_4_BIT_DATA) {
			pr_debug("Setting the bus width to 4 bit\n");
			ext_csd_bit = EXT_CSD_BUS_WIDTH_4;
			bus_width = MMC_BUS_WIDTH_4;
		} else {
			pr_debug("Setting the bus width to 1 bit\n");
			ext_csd_bit = EXT_CSD_BUS_WIDTH_1;
			bus_width = MMC_BUS_WIDTH_1;
		}
		err = mmc_switch(card, EXT_CSD_CMD_SET_NORMAL,
				 EXT_CSD_BUS_WIDTH, ext_csd_bit);

		if (err && err != -EBADMSG)
			goto free_card;

		if (err) {
			printk(KERN_WARNING "%s: switch to bus width %d "
			       "failed\n", mmc_hostname(card->host),
			       1 << bus_width);
			err = 0;
		} else {
			mmc_set_bus_width(card->host, bus_width);
		}
	}

	if (!oldcard)
		host->card = card;

	return 0;

free_card:
	if (!oldcard)
		mmc_remove_card(card);
err:

	return err;
}

/*
 * Host is being removed. Free up the current card.
 */
static void mmc_remove(struct mmc_host *host)
{
	BUG_ON(!host);
	BUG_ON(!host->card);

	mmc_remove_card(host->card);

	mmc_claim_host(host);
	host->card = NULL;
	mmc_release_host(host);
}

/*
 * Card detection callback from host.
 */
static void mmc_detect(struct mmc_host *host)
{
	int err;

	BUG_ON(!host);
	BUG_ON(!host->card);

	mmc_claim_host(host);

	/*
	 * Just check if our card has been removed.
	 */
	err = mmc_send_status(host->card, NULL);

	mmc_release_host(host);

	if (err) {
		mmc_remove(host);

		mmc_claim_host(host);
		mmc_detach_bus(host);
		mmc_release_host(host);
	}
}

/*
 * Suspend callback from host.
 */
static int mmc_suspend(struct mmc_host *host)
{
	int err = 0;

	BUG_ON(!host);
	BUG_ON(!host->card);

	mmc_claim_host(host);
	if (mmc_card_can_sleep(host))
		err = mmc_card_sleep(host);
	else if (!mmc_host_is_spi(host))
		mmc_deselect_cards(host);
	host->card->state &= ~MMC_STATE_HIGHSPEED;
	mmc_release_host(host);

	return err;
}

/*
 * Resume callback from host.
 *
 * This function tries to determine if the same card is still present
 * and, if so, restore all state to it.
 */
static int mmc_resume(struct mmc_host *host)
{
	int err;

	BUG_ON(!host);
	BUG_ON(!host->card);

	mmc_claim_host(host);
	err = mmc_init_card(host, host->ocr, host->card);
	mmc_release_host(host);

	return err;
}

static void mmc_power_restore(struct mmc_host *host)
{
	host->card->state &= ~MMC_STATE_HIGHSPEED;
	mmc_claim_host(host);
	mmc_init_card(host, host->ocr, host->card);
	mmc_release_host(host);
}

static int mmc_sleep(struct mmc_host *host)
{
	struct mmc_card *card = host->card;
	int err = -ENOSYS;

	if (card && card->ext_csd.rev >= 3) {
		err = mmc_card_sleepawake(host, 1);
		if (err < 0)
			pr_debug("%s: Error %d while putting card into sleep",
				 mmc_hostname(host), err);
	}

	return err;
}

static int mmc_awake(struct mmc_host *host)
{
	struct mmc_card *card = host->card;
	int err = -ENOSYS;

	if (card && card->ext_csd.rev >= 3) {
		err = mmc_card_sleepawake(host, 0);
		if (err < 0)
			pr_debug("%s: Error %d while awaking sleeping card",
				 mmc_hostname(host), err);
	}

	return err;
}

static const struct mmc_bus_ops mmc_ops = {
	.awake = mmc_awake,
	.sleep = mmc_sleep,
	.remove = mmc_remove,
	.detect = mmc_detect,
	.suspend = NULL,
	.resume = NULL,
	.power_restore = mmc_power_restore,
};

static const struct mmc_bus_ops mmc_ops_unsafe = {
	.awake = mmc_awake,
	.sleep = mmc_sleep,
	.remove = mmc_remove,
	.detect = mmc_detect,
	.suspend = mmc_suspend,
	.resume = mmc_resume,
	.power_restore = mmc_power_restore,
};

static void mmc_attach_bus_ops(struct mmc_host *host)
{
	const struct mmc_bus_ops *bus_ops;

	if (!mmc_card_is_removable(host))
		bus_ops = &mmc_ops_unsafe;
	else
		bus_ops = &mmc_ops;
	mmc_attach_bus(host, bus_ops);
}

/*
 * Starting point for MMC card init.
 */
int mmc_attach_mmc(struct mmc_host *host, u32 ocr)
{
	int err;

	BUG_ON(!host);
	WARN_ON(!host->claimed);

	mmc_attach_bus_ops(host);

	/*
	 * We need to get OCR a different way for SPI.
	 */
	if (mmc_host_is_spi(host)) {
		err = mmc_spi_read_ocr(host, 1, &ocr);
		if (err)
			goto err;
	}

	/*
	 * Sanity check the voltages that the card claims to
	 * support.
	 */
	if (ocr & 0x7F) {
		printk(KERN_WARNING "%s: card claims to support voltages "
		       "below the defined range. These will be ignored.\n",
		       mmc_hostname(host));
		ocr &= ~0x7F;
	}

	host->ocr = mmc_select_voltage(host, ocr);

	/*
	 * Can we support the voltage of the card?
	 */
	if (!host->ocr) {
		err = -EINVAL;
		goto err;
	}

	/*
	 * Detect and init the card.
	 */
	err = mmc_init_card(host, host->ocr, NULL);
	if (err)
		goto err;

	mmc_release_host(host);

	err = mmc_add_card(host->card);
	if (err)
		goto remove_card;

	return 0;

remove_card:
	mmc_remove_card(host->card);
	host->card = NULL;
	mmc_claim_host(host);
err:
	mmc_detach_bus(host);
	mmc_release_host(host);

	printk(KERN_ERR "%s: error %d whilst initialising MMC card\n",
		mmc_hostname(host), err);

	return err;
}
