/*
 * BQ27x00 battery driver
 *
 * Copyright (C) 2008 Rodolfo Giometti <giometti@linux.it>
 * Copyright (C) 2008 Eurotech S.p.A. <info@eurotech.it>
 *
 * Based on a previous work by Copyright (C) 2008 Texas Instruments, Inc.
 *
 * This package is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * THIS PACKAGE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
#include <linux/module.h>
#include <linux/param.h>
#include <linux/jiffies.h>
#include <linux/workqueue.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/power_supply.h>
#include <linux/idr.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <asm/unaligned.h>
#include <linux/power_supply.h>

#include <linux/msm-charger.h>
#include <linux/gpio.h>
#include <linux/gpio/gpio_def.h>
#include <linux/i2c/bq27520_gauge.h>
#include <linux/time.h>
#include <linux/rtc.h>
#define DRIVER_VERSION			"1.1.0"

#define BQ27x00_REG_TEMP		0x06
#define BQ27x00_REG_VOLT		0x08
#define BQ27x00_REG_AI			0x14
#define BQ27x00_REG_FLAGS		0x0A
#define BQ27x00_REG_TTE			0x16
#define BQ27x00_REG_TTF			0x18
#define BQ27x00_REG_TTECP		0x26

#define BQ27000_REG_RSOC		0x0B 
#define BQ27000_FLAG_CHGS		BIT(7)

#define BQ27500_REG_SOC			0x2c
#define BQ27500_FLAG_DSC		BIT(0)
#define BQ27500_FLAG_FC			BIT(9)
#define BQ27520_REG_REMAIN_CAPACITY		0x10
#define BQ27520_REG_FULL_CHARGE_CAPACITY	0X12


#define BQ27520_REGISTER_TO_MSM				1
#ifdef BQ27520_REGISTER_TO_MSM
#define BQ27520_BATT_CHG_THERM_OPERATIONAL_MAX_CELCIUS 		480
#define BQ27520_BATT_CHG_THERM_OPERATIONAL_MIN_CELCIUS 		-20
#define BQ27520_BATT_DISCHG_THERM_OPERATIONAL_MAX_CELCIUS 	600
#define BQ27520_BATT_DISCHG_THERM_OPERATIONAL_MIN_CELCIUS 	-200
#define BQ27520_CAPACITY_IS_ZERO_VOLTAGE_THRESHOLD	3200000

#define BQ27520_DUMMY_BATTERY_SUPPLY			1
#define BQ27520_CAPACITY_NOT_ZERO_VOLTAGE_THRESHOLD	3400000
#define BQ27520_CAPACITY_DUMMY_BATTERY_MONITOR		15
#define BQ27520_OLD_DFI_VERSION				0X04

#endif


 static int battery_low_flag=0;
 struct delayed_work battery_low_work;
 struct delayed_work soc_int_work;

#ifndef CHECK_POWERON_CONDITION
	#define CHECK_POWERON_CONDITION
#endif
#ifdef CHECK_POWERON_CONDITION
	struct delayed_work poweron_condition_check_work_work;
#endif
 
 
 static int update_dfi_status=0;
 int in_rom_mode = 0;
 
 
 
static DEFINE_IDR(battery_id);
static DEFINE_MUTEX(battery_mutex);
static DEFINE_MUTEX(gauge_mutex);
static DEFINE_MUTEX(dfi_mutex);

struct bq27x00_device_info;
struct bq27x00_access_methods {
	int (*read)(u8 reg, int *rt_value, int b_single,
		struct bq27x00_device_info *di);
};

enum bq27x00_chip { BQ27000, BQ27500 };

struct bq27x00_device_info {
	struct device 		*dev;
	int			id;
	struct bq27x00_access_methods	*bus;
	struct power_supply	bat;
	enum bq27x00_chip	chip;

	struct i2c_client	*client;
};


#ifndef	TST_BIT
#define  TST_BIT(x,b)    ((x & (1<<b)) ? 1 : 0)
#endif

#define	BQ275250_DFI_SUPPORT		1
#define	BQ27520_DFI_SIZE		1024+1
#define 	boolean 				uint8_t
#define 	I2C_RETRY_MAX			5
#define	I2C_ADDR_NORMAL_MODE	0x55
#define	I2C_ADDR_ROM_MODE		0X0b
#define	BQ27520_MAX_RETRY		5
#define	DEBUG_BQ27520_DFI		1
#define	BQ27520_ID			0x0520
#define	BQ27520_FIRMWARE		0x0302
#define	BQ27520_HARDWARE		0x00a3
#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE 
#define FALSE 0
#endif



extern unsigned int get_mfg_mode(void);
extern int in_pre_charge_stage;
#define KNOWN_SPECIAL_CONDITION ( ( (3 == get_mfg_mode() ) && in_pre_charge_stage )||(1==in_rom_mode))
static int Battery_valid = 1;

struct bq27x00_device_info *di_global;

int stop_charging_flag=0;
struct bq27x00_gauge_data
{	
	int bat_status;
	int bat_health;
	int bat_present;
	int bat_capacity;
	int bat_vol;
	int bat_temp;
	int bat_curr;

	int bat_over_voltage;     
	int bat_temp_high;
	int bat_temp_low; 
	int bat_temp_good;
	int bat_full;     

};
struct bq27x00_gauge_data bq27x00_bat_data; 

int batery_current=0;


static struct bq27520_platform_data *bq27520_data;	
static enum power_supply_property bq27x00_battery_props[] = {
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_PRESENT,
	POWER_SUPPLY_PROP_VOLTAGE_NOW,
	POWER_SUPPLY_PROP_CURRENT_NOW,
	POWER_SUPPLY_PROP_CAPACITY,
	POWER_SUPPLY_PROP_TEMP,
	POWER_SUPPLY_PROP_TIME_TO_EMPTY_NOW,
	POWER_SUPPLY_PROP_TIME_TO_EMPTY_AVG,
	POWER_SUPPLY_PROP_TIME_TO_FULL_NOW,
	POWER_SUPPLY_PROP_ENERGY_NOW,
	
	POWER_SUPPLY_PROP_DFI_BUF,
	
};

 

	
#ifdef BQ275250_DFI_SUPPORT
static char yIFRowData_recovery[2][96]=
	{
		{
			0xea,0xff,0x33,0x8e,0xf5,0x33,0x35,0xf6,0x33,0x24,0xfe,0x33,0x2c,0xfe,0x33,0x54,
			0x54,0x15,0xff,0xff,0x3f,0xff,0xff,0x3f,0xff,0xff,0x3f,0xff,0xff,0x3f,0xff,0xff,
			0x3f,0xff,0xff,0x3f,0xff,0xff,0x3f,0xff,0xff,0x3f,0xff,0xff,0x3f,0xff,0xff,0x3f,
			0x02,0x03,0x3f,0x58,0xcb,0x33,0x76,0x00,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0xff,
			0xaa,0x0e,0xfb,0xa7,0x0e,0xff,0xa6,0x0e,0xff,0xa1,0x0e,0xff,0xa0,0x0e,0xfe,0xa3,
			0x0e,0x26,0xa2,0x0e,0xd2,0xff,0x3a,0xff,0xa1,0x0e,0x66,0xa0,0x0e,0xff,0xa3,0x0e
		},
		{
			0x63,0xa2,0x0e,0xbd,0xa5,0x0e,0x03,0xa4,0x0e,0xca,0xff,0x3a,0xfe,0xa1,0x0e,0x26,
			0xa0,0x0e,0xfe,0xa3,0x0e,0x26,0xa2,0x0e,0xca,0xff,0x3a,0xc2,0xff,0x3a,0xd6,0xff,
			0x33,0xff,0xaf,0x0e,0x01,0x4f,0x03,0x3f,0x11,0x0c,0xcd,0xff,0x30,0xd4,0xff,0x35,
			0x2f,0x10,0x0c,0xd4,0xff,0x35,0xff,0xff,0x23,0x01,0xaf,0x14,0x01,0x4f,0x03,0x3f,
			0x11,0x0c,0xc5,0xff,0x30,0xcc,0xff,0x35,0x2f,0x10,0x0c,0xcc,0xff,0x35,0xff,0xff,
			0x23,0xff,0xdf,0x0b,0xc4,0xff,0x33,0x05,0xf1,0x39,0xc1,0xab,0x0e,0xe2,0x7f,0x3a
		}
	};
static uint8_t *yDataFlashImage;
static int bq27x00_read(u8 reg, int *rt_value, int b_single,

		struct bq27x00_device_info *di);


int bq27520_write(uint8_t  reg, uint8_t* buf, uint8_t  len,
				struct bq27x00_device_info *di)
{
	int32_t i;
	
	uint8_t buf_w[100];
	int err;
	struct i2c_client *client = di->client;
	struct i2c_msg msgs[] = {
		[0] = {
			.addr   = client->addr,
			.flags  = 0,
			.buf    = (void *)buf_w,
			.len    = len+1
		}
	};

	if(len >= sizeof(buf_w))  
		return -ENOMEM;
	
	memset(buf_w,0,sizeof(buf_w));
	buf_w[0] = reg;
	for(i=0; i<len; i++){
		buf_w[i+1] = buf[i];  
		
	}
	err= i2c_transfer(client->adapter, msgs, 1);

	return err;
}

boolean bq27520_write_retry(uint8_t addr,uint8_t  reg, uint8_t* buf, uint8_t  len)
{

	int i,ret;
	boolean result = FALSE;
	(di_global->client)->addr=addr;	  
	for(i=0; i<I2C_RETRY_MAX; i++){
		ret = bq27520_write(reg,buf,len,di_global);
		if(ret<0){
			msleep(5);
		}else{
			result=TRUE;
			break;
		}		
	}
	msleep(2);
	
	return result;
}
boolean bq27520_write_loop(uint8_t addr,int num,uint8_t array[])
{
	int i=1;
	int j;
	int ret=0;
	boolean result = FALSE;
	(di_global->client)->addr=addr;	
	while(i<=num){		
		j=2*(i-1);
		ret = bq27520_write_retry(addr,array[j],&array[j+1],sizeof(array[j+1]));
		if (!ret)
		{
			result=FALSE;
			printk("bq27520 write data %x with command %x fail\n",array[j],array[j+1]);
			return result;
		}
		else{
			result=TRUE;
			printk("bq27520 write data %x with command %x ok\n",array[j],array[j+1]);
		}
		i++;
	}
	return result;
}

int bq27520_read(uint8_t reg, uint8_t  *buf, uint8_t  len,struct bq27x00_device_info *di)
{
	int err;
	struct i2c_client *client = di->client;
	struct i2c_msg msgs[] = {
		[0] = {
			.addr   = client->addr,
			.flags  = 0,
			.buf    = (void *)&reg,
			.len    = 1
		},
		[1] = {
			.addr   = client->addr,
			.flags  = I2C_M_RD,
			.buf    = (void *)buf,
			.len    = len
		}
	};
	if(!client)
		return -ENODEV;
	err=i2c_transfer(client->adapter, msgs, 2);
	return err;
}
boolean bq27520_read_retry(uint8_t addr,uint8_t reg, uint8_t  *buf, uint8_t  len)
{
	int i,ret;
	boolean result = FALSE;
	(di_global->client)->addr=addr;	
	for(i=0; i<I2C_RETRY_MAX; i++){
		ret = bq27520_read(reg,buf,len,di_global);
		if(ret<0){
			msleep(5);
		}else{
			result=TRUE;
			break;
		}
	}
	msleep(2);	
	return result;
}
static int bq27520_gauge_IT_enable(struct bq27x00_device_info *di, union power_supply_propval *val)
{

	char buf[2];	
	char temp[4];	
	int retval;
	
		
	buf[0]=0x21;
	buf[1]=0x00;
	memset(temp,0,sizeof(temp));
	
	retval=bq27520_write(0x00,buf,sizeof(buf),di);
	if (retval<0){
		printk("bq27520 write i2c error,can not IT enable\n");	
		return -ENODATA;
	}
	
	return retval;
}

static int bq27520_gauge_IT_disable(struct bq27x00_device_info *di, union power_supply_propval *val)
{

	char buf[2];	
	char temp[4];	
	int retval;
	
		
	buf[0]=0x23;
	buf[1]=0x00;
	memset(temp,0,sizeof(temp));
	
	retval=bq27520_write(0x00,buf,sizeof(buf),di);
	if (retval<0){
		printk("bq27520 write i2c error,can not IT enable\n");	
		return -ENODATA;
	}
	
	return retval;
}

static int bq27520_gauge_reset(struct bq27x00_device_info *di, union power_supply_propval *val)
{

	char buf[2];	
	char temp[4];	
	int retval;
	
		
	buf[0]=0x41;
	buf[1]=0x00;
	memset(temp,0,sizeof(temp));
	
	retval=bq27520_write(0x00,buf,sizeof(buf),di);
	if (retval<0){
		printk("bq27520 write i2c error,can not IT enable\n");	
		return -ENODATA;
	}
	
	return retval;
}
static int bq27520_gauge_id(struct bq27x00_device_info *di, union power_supply_propval *val)
{

	char buf[2];	
	char temp[4];	
	int retval;
	int ret=0;
		
	buf[0]=0x01;
	buf[1]=0x00;
	memset(temp,0,sizeof(temp));
	retval=bq27520_write(0x00,buf,sizeof(buf),di);
	if (retval<0){
		printk("bq27520 write i2c error\n");	
		return -ENODATA;
	}
	msleep(20);
	retval = bq27520_read(0x00, temp, sizeof(temp), di);
	if(retval<0)
		printk("bq27520 read control register 0X00 error\n");
	else{
		ret=(ret)|temp[3];
		ret=(ret<<24)|temp[2];
		ret=(ret<<16)|temp[1];
		ret=(ret<<8)|temp[0];
	}
	val->intval = ret;
	return retval;
}

static int bq27520_gauge_firmware_version(struct bq27x00_device_info *di, union power_supply_propval *val)
{

	char buf[2];	
	char temp[4];	
	int retval;
	int ret=0;
		
	buf[0]=0x02;
	buf[1]=0x00;
	memset(temp,0,sizeof(temp));
	retval=bq27520_write(0x00,buf,sizeof(buf),di);
	if (retval<0){
		printk("bq27520 write i2c error\n");	
		return -ENODATA;
	}
	msleep(20);
	retval = bq27520_read(0x00, temp, sizeof(temp), di);
	if(retval<0)
		printk("bq27520 read control 0x00 register error\n");
	else{
		ret=(ret)|temp[3];
		ret=(ret<<24)|temp[2];
		ret=(ret<<16)|temp[1];
		ret=(ret<<8)|temp[0];
	}
	val->intval = ret;
	return retval;
}

static int bq27520_gauge_hardware_version(struct bq27x00_device_info *di, union power_supply_propval *val)
{

	char buf[2];	
	char temp[4];	
	int retval;
	int ret=0;
		
	buf[0]=0x03;
	buf[1]=0x00;
	memset(temp,0,sizeof(temp));
	retval=bq27520_write(0x00,buf,sizeof(buf),di);
	if (retval<0){
		printk("bq27520 write i2c error\n");	
		return -ENODATA;
	}
	msleep(20);
	retval = bq27520_read(0x00, temp, sizeof(temp), di);
	if(retval<0)
		printk("bq27520 read control register 0x00 error\n");
	else{
		ret=(ret)|temp[3];
		ret=(ret<<24)|temp[2];
		ret=(ret<<16)|temp[1];
		ret=(ret<<8)|temp[0];
	}
	val->intval = ret;
	return retval;
}

boolean bq27520_gauge_IC_check(void)
{
	struct bq27x00_device_info *di;
	union power_supply_propval val;
	int ret;
	boolean result=FALSE;
	di=di_global;
	
	ret=bq27520_gauge_id(di,&val);
	if(ret<0){
		printk("bq27520 read gauge id fail\n");
		return result;
	}else if(val.intval!=BQ27520_ID){
		printk("bq27520 check gauge id=0x%x\n,error!",val.intval);
		return result;
	}else{
		printk("bq27520 check gauge id=0x%x,success!\n",val.intval);
	}
	
	ret=bq27520_gauge_firmware_version(di,&val);
	if(ret<0){
		printk("bq27520 read gauge firmware version fail\n");
		return result;
	}else if(val.intval!=BQ27520_FIRMWARE){
		printk("bq27520 check firmware version=0x%x\n,error!",val.intval);
		return result;
	}else{	
		printk("bq27520 check firmware version=0x%x,success!\n",val.intval);
	}
	
	ret=bq27520_gauge_hardware_version(di,&val);
	if(ret<0){
		printk("bq27520 read gauge hardware version fail\n");
		return result;
	}else if(val.intval!=BQ27520_HARDWARE){
		printk("bq27520 check hardware version=0x%x\n,error!",val.intval);
		return result;
	}else{
		printk("bq27520 check gauge hardware version=0x%x,success!\n",val.intval);		
	}
	result=TRUE;
	return result;
}

uint16_t bq27520_get_fw_version(void)
{
	uint16_t fw_ver = 0x0000;
	uint16_t cmd = 0x0002;

	if(bq27520_write_retry(I2C_ADDR_NORMAL_MODE,0x00, (uint8_t*)&cmd, 2)){
		bq27520_read_retry(I2C_ADDR_NORMAL_MODE,0x00,(uint8_t*)&fw_ver, 2);
	}

	return fw_ver;
}

boolean bq27520_enable_rom_mode(boolean enable)
{
	uint8_t exit_rom[2] = {0x0F, 0x00};
	boolean result = FALSE;
	uint8_t gauge_data[2];
	int ret;
	gauge_data[0]=0x00;
	gauge_data[1]=0x0F;

	if(enable){		
		ret=bq27520_write_retry(I2C_ADDR_NORMAL_MODE,0x00,gauge_data, sizeof(gauge_data));
		if(!ret){
			printk("bq27520 write 0x0f00 with command 0x00 fail\n");
			return result;
		}else{
			printk("bq27520 enter rom mode successfully\n");			
		}

	}
	else{
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE,0x00,&exit_rom[0], 1);
		if(!ret){
			printk("bq27520 write 0x0f with command 0x00 fail\n");
			return result;
		}
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE,0x64,&exit_rom[0], 1);
		if(!ret){
			printk("bq27520 write 0x0f with command 0x64 fail\n");
			return result;
		}
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE,0x65,&exit_rom[1], 1);
		if(!ret){
			printk("bq27520 write 0x00 with command 0x64 fail\n");
			return result;
		}else{
			printk("bq27520 exit rom mode successfully\n");
		}	
	}
	result=TRUE;
	return result;
}

boolean bq27520_erase_instruction_flash(void)
{
	uint8_t i, write_buf[2] = {0x03, 0x00};  	
	uint8_t *write_buf_ptr = write_buf;
	boolean ret;
	uint8_t  read_buf = 0xFF;
	uint8_t *read_buf_ptr = &read_buf;
	boolean success = FALSE;

	success = FALSE;
	printk("bq27520_erase_instruction_flash---start\n");
	for(i = 0; i < 3; i++){
		
		 ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x00, write_buf_ptr, 1);
		if (!ret){
			printk("****erase instruction flash***bq27520 write  data 0x00 with command 0x03 fail\n");		
			return ret;
		}
		
		 ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x64, write_buf_ptr, 2);
		if (!ret){
			printk("****erase instruction flash***bq27520 write  data 0x03,0x00 with command 0x64 fail\n");		
			return ret;
		}
		
		msleep(20);
		 ret=bq27520_read_retry(I2C_ADDR_ROM_MODE, 0x66, read_buf_ptr, 1);
		if (!ret){
			printk("****erase instruction flash***bq27520 read  data with command 0x66 fail\n");		
			return ret;
		}

		if(*read_buf_ptr == 0x00){
			success = TRUE;
			break;
		}

	}
	
	return success;
}

boolean bq27520_erase_data_flash(void)
{
	uint8_t i, write_buf[2];
	uint8_t *write_buf_ptr = write_buf;
	boolean success = FALSE;
	boolean ret;
	uint8_t read_buf = 0xFF;
	uint8_t *read_buf_ptr = &read_buf;
	success = FALSE;
	printk("bq27520_erase_data_flash ---start\n");
	for(i = 0; i < 3; i++){
		
		*write_buf_ptr = 0x0C;
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x00, write_buf_ptr, 1);
		if (!ret){
			printk("****erase data flash***bq27520 write  data 0x0c with command 0x00 fail\n");		
			return ret;
		}
		
		*write_buf_ptr = 0x83;
		*(write_buf_ptr + 1) = 0xDE;
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x04, write_buf_ptr, 2);
		if (!ret){
			printk("****erase data flash***bq27520 write  data 0x83,0xde with command 0x04 fail\n");		
			return ret;
		}
		
		*write_buf_ptr = 0x6D;
		
		*(write_buf_ptr + 1) = 0x01;
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x64, write_buf_ptr, 2);
		if (!ret){
			printk("****erase data flash***bq27520 write  data 0x6d,0x01 with command 0x64 fail\n");		
			return ret;
		}
		
		msleep(200);
		
		ret=bq27520_read_retry(I2C_ADDR_ROM_MODE, 0x66, read_buf_ptr, 1);
		if (!ret){
			printk("****erase data flash***bq27520 read  data  with command 0x66 fail\n");		
			return ret;
		}
		if(read_buf == 0x00){
			success = TRUE;
			break;
		}
	}
	printk("bq27520_erase_data_flash ---end\n");
	return success;
}

boolean bq27520_dfi_write_data_flash(uint16_t *DFI_checksum)
{
	uint8_t row, i, read_buf, write_buf[2], num;
	uint8_t *read_buf_ptr = &read_buf, *write_buf_ptr = write_buf, *dfi_ptr = yDataFlashImage;
	uint16_t checksum;
	uint32_t sum_of_dfi_buf;	
	boolean ret;
	printk("bq27520_dfi_write_data_flash---start\n");
	for(i = 0; i < 3; i++){
		for(row = 0; row < 0x400 / 32; row++){

			
			*write_buf_ptr = 0x0A;
			*(write_buf_ptr + 1) = row;
			ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x00, write_buf_ptr, 2);
			if (!ret){
				printk("****write data flash***bq27520 write  data 0x0a,row with command 0x00 fail\n");		
				return ret;
			}
			
			ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x04,  (dfi_ptr + row * 32), 32);
			if (!ret){
				printk("****write data flash***bq27520 write  data DFI with command 0x04 fail\n");		
				return ret;
			}
			sum_of_dfi_buf = 0;

			for(num = 0; num < 32; num++){
				sum_of_dfi_buf += *(dfi_ptr + row * 32 + num);
			}
			
			checksum = (uint16_t)((0x0A + row + sum_of_dfi_buf) % 0x10000);
			*DFI_checksum = (uint16_t)((*DFI_checksum + sum_of_dfi_buf) % 0x10000);

			
			*write_buf_ptr = checksum & 0x00FF;
			*(write_buf_ptr + 1) = (checksum & 0xFF00) >> 8;
			ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x64,  write_buf_ptr, 2);
			if (!ret){
				printk("****write data flash***bq27520 write  data LSB,MSB,row with command 0x64 fail\n");		
				return ret;
			}
	
			
			
			msleep(2);
			*read_buf_ptr = 0xFF;
			ret=bq27520_read_retry(I2C_ADDR_ROM_MODE, 0x66, read_buf_ptr, 1);
			if (!ret){
				printk("****write data flash***bq27520 read data with command 0x66 fail\n");		
				return ret;
			}
			if(*read_buf_ptr != 0x00){
				printk("start writing image,read 0x66\n");
				break;
			}
		}

		if(row < 0x400 / 32){
			if(!bq27520_erase_data_flash()){
				return FALSE;
			}
		}
		else{
			return TRUE;
		}
	}
	printk("bq27520_dfi_write_data_flash---end\n");

	return FALSE;
}
uint8_t bq27520_rom_update_DFI(void)
{
	uint8_t  instruction_bak[2][96], read_buf[2] = {0xFF, 0xFF}, write_buf[2];
	uint16_t DFI_checksum, DFI_checksum_RB, checksum;
	uint32_t sum_of_dfi_bak;
	int row,i,num,j;
	boolean ret;
	uint8_t *instruction_bak_ptr = (uint8_t *)instruction_bak, *read_buf_ptr = read_buf, *write_buf_ptr = write_buf;
	j=0;
	
	printk(KERN_INFO "%s:read and erase instruction flash in rom mode!\n", __func__);
	
 	 
	for(row = 0; row < 2; row++){
		*write_buf_ptr = 0x00;
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x00, write_buf_ptr, 1);
		if (!ret)	{
			printk("bq27520 write  data 0x00 with command 0x00 fail\n");		
			return ret;
		}
		*write_buf_ptr = row;  
		*(write_buf_ptr + 1) = 0x00; 

		
		
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x01, write_buf_ptr, 2);
		if (!ret)	{
			printk("bq27520 write  data row,0x00 with command 0x01 fail\n");		
			return ret;
		}
	   	
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x64, write_buf_ptr, 2);
		if (!ret)	{
			printk("bq27520 write  data row,0x00 with command 0x64 fail\n");		
			return ret;
		}
	    	
		printk("bq27520 read instruction flash into instruction_bak\n");
		ret=bq27520_read_retry(I2C_ADDR_ROM_MODE, 0x04, instruction_bak_ptr + row * sizeof(instruction_bak[row]), sizeof(instruction_bak[row]));
		if (!ret)	{
			printk("bq27520 read instruction flash fail\n");		
			return ret;
		}
		
		msleep(20);
	}

	if(!bq27520_erase_instruction_flash()){
		return FALSE;
	}

	printk("bq27520_calibration_DFI---start writing image\n");
	for(i = 0; i < 3; i++){
		DFI_checksum = 0;

		if(!bq27520_erase_data_flash() ||
		!bq27520_dfi_write_data_flash(&DFI_checksum)){
			return FALSE;
		}

		
		*write_buf_ptr = 0x08;
		*(write_buf_ptr + 1) = 0x00;
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x00, write_buf_ptr, 1);
		if (!ret)	{
			printk("***data flash checksum***bq27520 write  data 0x08 with command 0x00 fail\n");		
			return ret;
		}
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x64, write_buf_ptr, 2);
		if (!ret)	{
			printk("***data flash checksum***bq27520 write  data 0x08,0x00 with command 0x64 fail\n");		
			return ret;
		}
		
		msleep(20);

		
		ret=bq27520_read_retry(I2C_ADDR_ROM_MODE, 0x04, read_buf_ptr, 2);
		if (!ret)	{
			printk("***data flash checksum***bq27520 read data  with command 0x04 fail\n");		
			return ret;
		}
		DFI_checksum_RB = ((read_buf[1] << 8) & 0xFF00) | read_buf[0];

		
		if(DFI_checksum == DFI_checksum_RB){
			break;
		}
	}
	if(i >= 3)	{
		return FALSE;
	}
	
	memset(instruction_bak,0,sizeof(instruction_bak));
	for(i=0;i<2;i++){
		for(j=0;j<sizeof(instruction_bak[i]);j++){
			instruction_bak[i][j]=yIFRowData_recovery[i][j];
			
		}
	}	
	  
	printk("bq27520_calibration_DFI---end writing image---start\n");
	for(row = 1; row >= 0; row--){
		for(i = 0; i < 3; i++){
			
			*write_buf_ptr = 0x02;
			ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x00, write_buf_ptr, 1);
			if (!ret)	{
				printk("***restore instruction flash***bq27520 write  data 0x02 with command 0x00 fail\n");		
				return ret;
			}
			
			*write_buf_ptr = row;
			*(write_buf_ptr + 1) = 0x00;
			ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x01, write_buf_ptr, 2);
			if (!ret)	{
				printk("***restore instruction flash***bq27520 write  data row,0x00 with command 0x01 fail\n");		
				return ret;
			}
			
			ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x04, instruction_bak_ptr + row * sizeof(instruction_bak[row]), sizeof(instruction_bak[row]));
			if (!ret)	{
				printk("***restore instruction flash***bq27520 write data IFrowdata with command 0x04 fail\n");		
				return ret;
			}
			
			sum_of_dfi_bak = 0;
			for(num = 0; num < 96; num++){
				sum_of_dfi_bak += *(instruction_bak_ptr + row * sizeof(instruction_bak[row]) + num);
			}

			
			checksum = (uint16_t)((0x02 + row + sum_of_dfi_bak) % 0x10000);

			*write_buf_ptr = checksum & 0x00FF;
			*(write_buf_ptr + 1) = (checksum & 0xFF00) >> 8;

			ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x64, write_buf_ptr, 2);
			if (!ret)	{
				printk("***restore instruction flash***bq27520 write  data LSB,MSB with command 0x64 fail\n");		
				return ret;
			}
			
			msleep(20);

			
			*read_buf_ptr = 0xFF;
			ret=bq27520_read_retry(I2C_ADDR_ROM_MODE, 0x66, read_buf_ptr, 1);
			if (!ret)	{
				printk("***restore instruction flash***bq27520 read data  with command 0x66 fail\n");		
				return ret;
			}
			if(read_buf[0] == 0x00){
				break;
			}
		}

		
		if(i >= 3)	{
			return FALSE;
		}
		
	}
	  
	  return TRUE;
}
uint8_t bq27520_normal_update_DFI(void)
{
	uint8_t  instruction_bak[2][96], read_buf[2] = {0xFF, 0xFF}, write_buf[2];
	uint16_t DFI_checksum, DFI_checksum_RB, checksum;
	uint32_t sum_of_dfi_bak;
	int row,i,num,j;
	boolean ret;
	uint8_t *instruction_bak_ptr = (uint8_t *)instruction_bak, *read_buf_ptr = read_buf, *write_buf_ptr = write_buf;
	j=0;
	printk("bq27520_calibration_DFI---read and erase instruction flash\n");
	if(1){
		if(!bq27520_enable_rom_mode(TRUE)){
			 return FALSE;
		}
	}
 	 
	for(row = 0; row < 2; row++){
		*write_buf_ptr = 0x00;
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x00, write_buf_ptr, 1);
		if (!ret)	{
			printk("bq27520 write  data 0x00 with command 0x00 fail\n");		
			return ret;
		}
		*write_buf_ptr = row;  
		*(write_buf_ptr + 1) = 0x00; 

		
		
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x01, write_buf_ptr, 2);
		if (!ret)	{
			printk("bq27520 write  data row,0x00 with command 0x01 fail\n");		
			return ret;
		}
	   	
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x64, write_buf_ptr, 2);
		if (!ret)	{
			printk("bq27520 write  data row,0x00 with command 0x64 fail\n");		
			return ret;
		}
	    	
		printk("bq27520 read instruction flash into instruction_bak\n");
		ret=bq27520_read_retry(I2C_ADDR_ROM_MODE, 0x04, instruction_bak_ptr + row * sizeof(instruction_bak[row]), sizeof(instruction_bak[row]));
		if (!ret)	{
			printk("bq27520 read instruction flash fail\n");		
			return ret;
		}
		
		msleep(20);
	}

	if(!bq27520_erase_instruction_flash()){
		return FALSE;
	}

	printk("bq27520_calibration_DFI---start writing image\n");
	for(i = 0; i < 3; i++){
		DFI_checksum = 0;

		if(!bq27520_erase_data_flash() ||
		!bq27520_dfi_write_data_flash(&DFI_checksum)){
			return FALSE;
		}

		
		*write_buf_ptr = 0x08;
		*(write_buf_ptr + 1) = 0x00;
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x00, write_buf_ptr, 1);
		if (!ret)	{
			printk("***data flash checksum***bq27520 write  data 0x08 with command 0x00 fail\n");		
			return ret;
		}
		ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x64, write_buf_ptr, 2);
		if (!ret)	{
			printk("***data flash checksum***bq27520 write  data 0x08,0x00 with command 0x64 fail\n");		
			return ret;
		}
		
		msleep(20);

		
		ret=bq27520_read_retry(I2C_ADDR_ROM_MODE, 0x04, read_buf_ptr, 2);
		if (!ret)	{
			printk("***data flash checksum***bq27520 read data  with command 0x04 fail\n");		
			return ret;
		}
		DFI_checksum_RB = ((read_buf[1] << 8) & 0xFF00) | read_buf[0];

		
		if(DFI_checksum == DFI_checksum_RB){
			break;
		}
	}
	if(i >= 3)	{
		return FALSE;
	}
	  
	printk("bq27520_calibration_DFI---end writing image---start\n");
	for(row = 1; row >= 0; row--){
		for(i = 0; i < 3; i++){
			
			*write_buf_ptr = 0x02;
			ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x00, write_buf_ptr, 1);
			if (!ret)	{
				printk("***restore instruction flash***bq27520 write  data 0x02 with command 0x00 fail\n");		
				return ret;
			}
			
			*write_buf_ptr = row;
			*(write_buf_ptr + 1) = 0x00;
			ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x01, write_buf_ptr, 2);
			if (!ret)	{
				printk("***restore instruction flash***bq27520 write  data row,0x00 with command 0x01 fail\n");		
				return ret;
			}
			
			ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x04, instruction_bak_ptr + row * sizeof(instruction_bak[row]), sizeof(instruction_bak[row]));
			if (!ret)	{
				printk("***restore instruction flash***bq27520 write data IFrowdata with command 0x04 fail\n");		
				return ret;
			}
			
			sum_of_dfi_bak = 0;
			for(num = 0; num < 96; num++){
				sum_of_dfi_bak += *(instruction_bak_ptr + row * sizeof(instruction_bak[row]) + num);
			}

			
			checksum = (uint16_t)((0x02 + row + sum_of_dfi_bak) % 0x10000);

			*write_buf_ptr = checksum & 0x00FF;
			*(write_buf_ptr + 1) = (checksum & 0xFF00) >> 8;

			ret=bq27520_write_retry(I2C_ADDR_ROM_MODE, 0x64, write_buf_ptr, 2);
			if (!ret)	{
				printk("***restore instruction flash***bq27520 write  data LSB,MSB with command 0x64 fail\n");		
				return ret;
			}
			
			msleep(20);

			
			*read_buf_ptr = 0xFF;
			ret=bq27520_read_retry(I2C_ADDR_ROM_MODE, 0x66, read_buf_ptr, 1);
			if (!ret)	{
				printk("***restore instruction flash***bq27520 read data  with command 0x66 fail\n");		
				return ret;
			}
			if(read_buf[0] == 0x00){
				break;
			}
		}

		
		if(i >= 3)	{
			return FALSE;
		}
		
	}
	  
	  return TRUE;
}

uint8_t bq27520_version_DFI(void)
{
	boolean ret=FALSE;	
	uint8_t block_data_control=0x00;
	uint8_t data_flash_class=0x39;
	uint8_t data_flash_block=0x00;
	uint8_t version=0;

	ret=bq27520_write_retry(I2C_ADDR_NORMAL_MODE, 0x61, &block_data_control, 1);
	if (!ret)	{
		printk("***read DFI version***bq27520 write  data 0x00 with command 0x61 fail\n");		
		return ret;
	}
	ret=bq27520_write_retry(I2C_ADDR_NORMAL_MODE, 0x3E, &data_flash_class, 1);
	if (!ret)	{
		printk("***read DFI version***bq27520 write  data 0x39 with command 0x3e fail\n");		
		return ret;
	}
	ret=bq27520_write_retry(I2C_ADDR_NORMAL_MODE, 0x3F, &data_flash_block, 1);
	if (!ret)	{
		printk("***read DFI version***bq27520 write  data 0x00 with command 0x3f fail\n");		
		return ret;
	}
	ret=bq27520_read_retry(I2C_ADDR_NORMAL_MODE, 0x40, &version, 1);
	if (!ret)	{
		printk("***read DFI version***bq27520 read  data with command 0x40 fail\n");			
		return ret;
	}
	
	return version;
}
uint8_t bq27520_calibration_DFI(void)
{
	uint16_t cmd =0;
	int ret;
	
	if(0==in_rom_mode){
		
		ret=bq27520_gauge_IC_check();
		if(ret==FALSE){
			printk("**************\n Gauge IC not correct,stop updating DFI.\n**************\n");
			return FALSE;	
		}

		ret=bq27520_normal_update_DFI();
		if(!ret){
			printk("**************\n fail to write Gauge data flash in normal mode.\n**************\n");
			return FALSE;
		}

		
		if(!bq27520_enable_rom_mode(FALSE)){
			return FALSE;
		}
		msleep(20);
		
		cmd = 0x0041;  
		if(!bq27520_write_retry(I2C_ADDR_NORMAL_MODE, 0x00, (uint8_t*)&cmd, 2)){
		 	return FALSE;
		}

		msleep(50);
	}
	else{
		
		ret=bq27520_rom_update_DFI();
		if(!ret){
			printk("**************\n fail to write Gauge data flash in rom mode.\n**************\n");
			return FALSE;
		}
		
		if(!bq27520_enable_rom_mode(FALSE)){
			return FALSE;
		}
		msleep(20);
		
		cmd = 0x0041;  
		if(!bq27520_write_retry(I2C_ADDR_NORMAL_MODE, 0x00, (uint8_t*)&cmd, 2)){
		 	return FALSE;
		}

		msleep(50);
		mutex_lock(&gauge_mutex);
		in_rom_mode = 0;
		mutex_unlock(&gauge_mutex);
	}
	
	
	
	 
		
	
	bq27520_version_DFI();

	return TRUE;
	
}

int bq27520_copy_dfi_user_to_kernel(const union power_supply_propval *val)
{

	const char *dfi_ptr=val->strval;
	const char  *mach_type_ptr=NULL;
	uint16_t dfi_size=0;
	uint16_t dfi_start_addr=0;
	int i=0;
	
	const struct bq27520_platform_data *pdata;
	struct i2c_client *client;
	boolean ret=FALSE;
	uint16_t dfi_version=0;
	
	mutex_lock(&dfi_mutex);
	update_dfi_status = 1;
	client = di_global->client;
	pdata = client->dev.platform_data;	
	if (pdata->gauge_detection_config)
		mach_type_ptr = pdata->gauge_detection_config(dfi_ptr);
	if(mach_type_ptr){
		
		dfi_version=dfi_version|(*(mach_type_ptr+5));
		dfi_version=(dfi_version<<8)|(*(mach_type_ptr+4));
		if(0){
			if(bq27520_version_DFI()==dfi_version)
				update_dfi_status = 0;
				mutex_unlock(&dfi_mutex);
				return TRUE;
		}
		
		dfi_size=(dfi_size)|*(mach_type_ptr+9);	
		dfi_size=(dfi_size<<8)|*(mach_type_ptr+8);	
		dfi_start_addr=(dfi_start_addr)|*(mach_type_ptr+7);	
		dfi_start_addr=(dfi_start_addr<<8)|*(mach_type_ptr+6);	

		yDataFlashImage= kmalloc(sizeof(uint8_t)*dfi_size,GFP_KERNEL);
		if(yDataFlashImage==NULL){
			printk(KERN_INFO "%s:couldnt request enough memory!\n", __func__);
			update_dfi_status = 0;
			mutex_unlock(&dfi_mutex);
			return ret;
		}		
		memset(yDataFlashImage,0,sizeof(uint8_t)*dfi_size);	
		memcpy(yDataFlashImage,dfi_ptr+dfi_start_addr,sizeof(uint8_t)*dfi_size);
		while(i<dfi_size){
			
			i++;
		}
	}else{
		printk(KERN_INFO "%s:const char point: mach_type_ptr is NULL.\n", __func__);
		update_dfi_status = 0;
		mutex_unlock(&dfi_mutex);
		return ret;
	}
	
	if(1){
		ret=bq27520_calibration_DFI();
		if(!ret){
			printk("***calibration DFI fail\n");	
			kfree(yDataFlashImage);
			update_dfi_status = 0;
			mutex_unlock(&dfi_mutex);
			return ret;
		}else
			printk("***bq27520 DFI write completed!***\n");
	}
	
	msleep(20);
	update_dfi_status = 0;
	mutex_unlock(&dfi_mutex);
	kfree(yDataFlashImage);
	ret=TRUE;
	return ret;
}
#endif

static int bq27x00_read(u8 reg, int *rt_value, int b_single,
			struct bq27x00_device_info *di)
{
	return di->bus->read(reg, rt_value, b_single, di);
}


static int bq27x00_battery_temperature(struct bq27x00_device_info *di)
{
	int ret;
	int temp = 0;

	ret = bq27x00_read(BQ27x00_REG_TEMP, &temp, 0, di);
	
	if (ret) {
		dev_err(di->dev, "error reading temperature\n");
		return ret;
	}

	if (di->chip == BQ27500)
		return temp - 2731;
	else
		return ((temp >> 2) - 273) * 10;
}


static int bq27x00_battery_voltage(struct bq27x00_device_info *di)
{
	int ret;
	int volt = 0;

	ret = bq27x00_read(BQ27x00_REG_VOLT, &volt, 0, di);
	if (ret) {
		dev_err(di->dev, "error reading voltage\n");
		return ret;
	}

	return volt * 1000;
}


static int bq27x00_battery_current(struct bq27x00_device_info *di)
{
	int ret;
	int curr = 0;
	int flags = 0;

	ret = bq27x00_read(BQ27x00_REG_AI, &curr, 0, di);
	if (ret) {
		dev_err(di->dev, "error reading current\n");
		return 0;
	}

	if (di->chip == BQ27500) {
		
		curr = (int)(s16)curr;
	} else {
		ret = bq27x00_read(BQ27x00_REG_FLAGS, &flags, 0, di);
		if (ret < 0) {
			dev_err(di->dev, "error reading flags\n");
			return 0;
		}
		if (flags & BQ27000_FLAG_CHGS) {
			dev_dbg(di->dev, "negative current!\n");
			curr = -curr;
		}
	}

	return curr * 1000;
}


static int bq27x00_battery_rsoc(struct bq27x00_device_info *di)
{
	int ret;
	int rsoc = 0;

	if (di->chip == BQ27500)
		ret = bq27x00_read(BQ27500_REG_SOC, &rsoc, 0, di);
	else
		ret = bq27x00_read(BQ27000_REG_RSOC, &rsoc, 1, di);
	if (ret) {
		dev_err(di->dev, "error reading relative State-of-Charge\n");
		return ret;
	}
	return rsoc;
}

static int bq27x00_battery_status(struct bq27x00_device_info *di,
				  union power_supply_propval *val)
{
	int flags = 0;
	int status;
	int ret;

	ret = bq27x00_read(BQ27x00_REG_FLAGS, &flags, 0, di);
	if (ret < 0) {
		dev_err(di->dev, "error reading flags\n");
		return ret;
	}

	if (di->chip == BQ27500) {
		if (flags & BQ27500_FLAG_FC)
			status = POWER_SUPPLY_STATUS_FULL;
		else if (flags & BQ27500_FLAG_DSC)
			status = POWER_SUPPLY_STATUS_DISCHARGING;
		else
			status = POWER_SUPPLY_STATUS_CHARGING;
	} else {
		if (flags & BQ27000_FLAG_CHGS)
			status = POWER_SUPPLY_STATUS_CHARGING;
		else
			status = POWER_SUPPLY_STATUS_DISCHARGING;
	}

	val->intval = status;
	return 0;
}


static int bq27x00_battery_time(struct bq27x00_device_info *di, int reg,
				union power_supply_propval *val)
{
	int tval = 0;
	int ret;

	ret = bq27x00_read(reg, &tval, 0, di);
	if (ret) {
		dev_err(di->dev, "error reading register %02x\n", reg);
		return ret;
	}

	if (tval == 65535)
		return -ENODATA;

	val->intval = tval * 60;
	return 0;
}

#define to_bq27x00_device_info(x) container_of((x), \
				struct bq27x00_device_info, bat);

static int bq27x00_battery_get_property(struct power_supply *psy,
					enum power_supply_property psp,
					union power_supply_propval *val)
{
	int ret = 0;
	struct bq27x00_device_info *di = to_bq27x00_device_info(psy);

	switch (psp) {
	case POWER_SUPPLY_PROP_STATUS:
		ret = bq27x00_battery_status(di, val);
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_NOW:
	case POWER_SUPPLY_PROP_PRESENT:
		val->intval = bq27x00_battery_voltage(di);
		if (psp == POWER_SUPPLY_PROP_PRESENT)
			val->intval = val->intval <= 0 ? 0 : 1;
		break;
	case POWER_SUPPLY_PROP_CURRENT_NOW:
		val->intval = bq27x00_battery_current(di);
		break;
	case POWER_SUPPLY_PROP_CAPACITY:
		val->intval = bq27x00_battery_rsoc(di);
		break;
	case POWER_SUPPLY_PROP_TEMP:
		val->intval = bq27x00_battery_temperature(di);
		break;
	case POWER_SUPPLY_PROP_TIME_TO_EMPTY_NOW:
		ret = bq27x00_battery_time(di, BQ27x00_REG_TTE, val);
		break;
	case POWER_SUPPLY_PROP_TIME_TO_EMPTY_AVG:
		ret = bq27x00_battery_time(di, BQ27x00_REG_TTECP, val);
		break;
	case POWER_SUPPLY_PROP_TIME_TO_FULL_NOW:
		ret = bq27x00_battery_time(di, BQ27x00_REG_TTF, val);
		break;
	case POWER_SUPPLY_PROP_ENERGY_NOW:
		ret = bq27520_gauge_id(di,val);
		break;
	default:
		return -EINVAL;
	}

	return ret;
}

static int bq27x00_battery_set_property(struct power_supply *psy,
			    enum power_supply_property psp,
			    const union power_supply_propval *val)
{
	int ret = 0;
	switch (psp) {
	case POWER_SUPPLY_PROP_DFI_BUF:	
		ret=bq27520_copy_dfi_user_to_kernel(val);
		break;
	default:
		return -EINVAL;
	}
	return ret;
}

static void bq27x00_powersupply_init(struct bq27x00_device_info *di)
{
	di->bat.type = POWER_SUPPLY_TYPE_BATTERY;
	di->bat.properties = bq27x00_battery_props;
	di->bat.num_properties = ARRAY_SIZE(bq27x00_battery_props);
	di->bat.get_property = bq27x00_battery_get_property;
	
	
	di->bat.set_property =  bq27x00_battery_set_property;
	
	di->bat.external_power_changed = NULL;
}


 #if 1
 
 static int bq27x00_read_i2c(u8 reg, int *rt_value, int b_single,
			struct bq27x00_device_info *di)
{
	struct i2c_client *client = di->client;
	int err;
	unsigned char data[2];
	
	struct i2c_msg msgs[] = {
		[0] = {
			.addr   = client->addr,
			.flags  = 0,
			.buf    = (void *)&reg,
			.len    = 1
		},
		[1] = {
			.addr   = client->addr,
			.flags  = I2C_M_RD,
			.buf    = data,
			.len    = 2
		}
	};
	if (!b_single)
		msgs[1].len = 2;
	else
		msgs[1].len = 1;
	memset(data,0,sizeof(data));
	if(!client)
		return -ENODEV;
	err=i2c_transfer(client->adapter, msgs,2);
	if(err>=0){
		*rt_value = get_unaligned_le16(data);
		return 0;
	}
	return err;
}
  
#else
static int bq27x00_read_i2c(u8 reg, int *rt_value, int b_single,
			struct bq27x00_device_info *di)
{
	struct i2c_client *client = di->client;
	struct i2c_msg msg[1];
	unsigned char data[2];
	int err;

	if (!client->adapter)
		return -ENODEV;

	msg->addr = client->addr;
	msg->flags = 0;
	msg->len = 1;
	msg->buf = data;

	data[0] = reg;
	err = i2c_transfer(client->adapter, msg, 1);

	if (err >= 0) {
		if (!b_single)
			msg->len = 2;
		else
			msg->len = 1;

		msg->flags = I2C_M_RD;
		err = i2c_transfer(client->adapter, msg, 1);
		if (err >= 0) {
			if (!b_single)
				*rt_value = get_unaligned_le16(data);
			else
				*rt_value = data[0];

			return 0;
		}
	}
	return err;
}
#endif
#ifdef BQ27520_REGISTER_TO_MSM

#define I2C_ERROR_RETRY_MAX				5
#define BAT_TEMPERATURE_MAX_CELCIUS_TOLERANCE	780	
#define BAT_TEMPERATURE_MIN_CELCIUS_TOLERANCE	-260	
#define BAT_DEFAULT_TEMPERATURE_CELCIUS		280	
#define BAT_VOLTAGE_MAX_MV_TOLERANCE		5600	
#define BAT_VOLTAGE_MIN_MV_TOLERANCE		1610	
#define BAT_DEFAULT_VOLTAGE_MV			3700	
#define BAT_CURRENCE_MAX_MA_TOLERANCE		1430	
#define BAT_CURRENCE_MIN_MA_TOLERANCE		-13000	
#define BAT_DEFAULT_CURRENCE_MA			600	
#define BAT_CAPACITY_MAX_TOLERANCE			100	
#define BAT_CAPACITY_MIN_TOLERANCE			0	
#define BAT_DEFAULT_CAPACITY				40	

int bq27520_is_battery_id_valid(void)
{

	
	return 1;

	
}

int bq27520_is_charging_battery_temp_within_range(void)
{
	int therm_celcius;
	int i;

	
	if(1==update_dfi_status)
		return 1;
	
	
	msleep(20);
	for(i=0; i<I2C_ERROR_RETRY_MAX; i++){
		
		therm_celcius = bq27x00_battery_temperature(di_global);
		if(therm_celcius<BAT_TEMPERATURE_MIN_CELCIUS_TOLERANCE||
			therm_celcius>BAT_TEMPERATURE_MAX_CELCIUS_TOLERANCE){
			if(KNOWN_SPECIAL_CONDITION)
				break;
			msleep(500);
		}
		else
			break;		
	}
   	if(i==I2C_ERROR_RETRY_MAX){
		printk(KERN_INFO "%s:temperature=%d !\n", __func__,therm_celcius);
		BUG();
	}
	pr_debug("%s: therm_celcius is %d\n", __func__, therm_celcius);
	
	if ( therm_celcius >= BQ27520_BATT_CHG_THERM_OPERATIONAL_MIN_CELCIUS
		&& therm_celcius <= BQ27520_BATT_CHG_THERM_OPERATIONAL_MAX_CELCIUS)
		return 1;

	return 0;
}

int bq27520_is_discharging_battery_temp_within_range(void)
{
	int therm_celcius;
	int i;

	
	if(1==update_dfi_status)
		return 1;
	
	
	msleep(20);
	for(i=0; i<I2C_ERROR_RETRY_MAX; i++){
		
		therm_celcius = bq27x00_battery_temperature(di_global);
		if(therm_celcius<BAT_TEMPERATURE_MIN_CELCIUS_TOLERANCE||
			therm_celcius>BAT_TEMPERATURE_MAX_CELCIUS_TOLERANCE){
			if(KNOWN_SPECIAL_CONDITION)
				break;
			msleep(500);
		}
		else
			break;		
	}
	if(i==I2C_ERROR_RETRY_MAX){
		printk(KERN_INFO "%s:temperature=%d !\n", __func__,therm_celcius);
		BUG();
	}
	pr_debug("%s: therm_celcius is %d\n", __func__, therm_celcius);
	
	if ( therm_celcius >= BQ27520_BATT_DISCHG_THERM_OPERATIONAL_MIN_CELCIUS
		&& therm_celcius <= BQ27520_BATT_DISCHG_THERM_OPERATIONAL_MAX_CELCIUS)
		return 1;

	return 0;
}

int bq27520_is_battery_present(void)
{
	int ret;
	int flags;
	int count=0;
	int present=0;
	

	
	if(1==update_dfi_status)
		return 1;
	
	
	while(count < 5){
		ret=bq27x00_read(BQ27x00_REG_FLAGS, &flags, 0, di_global);
		
		if(TST_BIT(flags,3)){
			present=1;
			break;
		}else{
			count++;
		}
		msleep(500);
	}	
	if(!present)
		printk(KERN_INFO"bq27520  battery is not present,flags=0x%x\n",flags);
	return present;
}

int bq27520_get_battery_temperature(void)
{
	int therm_celcius;
	int i;

	
	if(1==update_dfi_status)
		return BAT_DEFAULT_TEMPERATURE_CELCIUS;
		
	
	msleep(20);
	for(i=0; i<I2C_ERROR_RETRY_MAX; i++){
		therm_celcius = bq27x00_battery_temperature(di_global);
		if(therm_celcius<BAT_TEMPERATURE_MIN_CELCIUS_TOLERANCE||
			therm_celcius>BAT_TEMPERATURE_MAX_CELCIUS_TOLERANCE){
			if(KNOWN_SPECIAL_CONDITION)
				break;
			msleep(500);
		}
		else
			break;		
	}	
	if(i==I2C_ERROR_RETRY_MAX){
		printk(KERN_INFO "%s:temperature=%d !\n", __func__,therm_celcius);
		BUG();
	}
	
	
	return therm_celcius;	
}

int bq27520_get_battery_mvolts(void)
{
	int vbatt_mv;
	int i;

	
	if(1==update_dfi_status)
		return BAT_DEFAULT_VOLTAGE_MV;
		
	
	msleep(20);
	for(i=0; i<I2C_ERROR_RETRY_MAX; i++){
		vbatt_mv = bq27x00_battery_voltage(di_global);
		vbatt_mv/=1000;
		if(vbatt_mv<BAT_VOLTAGE_MIN_MV_TOLERANCE||
			vbatt_mv>BAT_VOLTAGE_MAX_MV_TOLERANCE){
			if(KNOWN_SPECIAL_CONDITION)
				break;
			msleep(500);
		}
		else
			break;		
	}
	if(i==I2C_ERROR_RETRY_MAX){
		printk(KERN_INFO "%s:vbatt_mv=%d !\n", __func__,vbatt_mv);
		BUG();
	}
	pr_debug("%s: vbatt_mv is %d\n", __func__, vbatt_mv);
	if (vbatt_mv > 0)
		return vbatt_mv;
	
	return 0;
}
int bq27520_get_battery_current(void)
{
	int vbatt_ma;
	int i;

	
	if(1==update_dfi_status)
		return BAT_DEFAULT_CURRENCE_MA;
	
	
	msleep(20);
	for(i=0; i<I2C_ERROR_RETRY_MAX; i++){
		vbatt_ma = bq27x00_battery_current(di_global);
		vbatt_ma /=1000;
		if(vbatt_ma<BAT_CURRENCE_MIN_MA_TOLERANCE||
			vbatt_ma>BAT_CURRENCE_MAX_MA_TOLERANCE){
			if(KNOWN_SPECIAL_CONDITION)
				break;
			msleep(500);
		}
		else
			break;		
	}
	if(i==I2C_ERROR_RETRY_MAX){
		printk(KERN_INFO "%s:vbatt_ma=%d !\n", __func__,vbatt_ma);
		BUG();
	}
	pr_debug("%s: vbatt_mv is %d\n", __func__, vbatt_ma);
	
	return vbatt_ma;
	
}
int bq27520_get_battery_capacity(void)
{
	int capacity;
	int ret;
	int i;

	
	if(1==update_dfi_status)
		return BAT_DEFAULT_CAPACITY;
	
	
	msleep(20);
	for(i=0; i<I2C_ERROR_RETRY_MAX; i++){
		capacity = bq27x00_battery_rsoc(di_global);	
		if(capacity<BAT_CAPACITY_MIN_TOLERANCE||
			capacity>BAT_CAPACITY_MAX_TOLERANCE){
			if(KNOWN_SPECIAL_CONDITION)
				break;
			msleep(500);
		}
		else
			break;		
	}
	if(i==I2C_ERROR_RETRY_MAX){
		printk(KERN_INFO "%s:capacity=%d !\n", __func__,capacity);		
		BUG();
	}
	ret = bq27x00_battery_voltage(di_global);
	
	#ifdef BQ27520_DUMMY_BATTERY_SUPPLY
	if(capacity<BQ27520_CAPACITY_DUMMY_BATTERY_MONITOR&&
		ret>BQ27520_CAPACITY_IS_ZERO_VOLTAGE_THRESHOLD)
		capacity+=1;
	#endif
	
 	 
 	
	if(ret<BQ27520_CAPACITY_IS_ZERO_VOLTAGE_THRESHOLD){
		capacity = 0;
		printk(KERN_INFO "%s:voltage<%duv,set capacity as 0 !\n", __func__,
			BQ27520_CAPACITY_IS_ZERO_VOLTAGE_THRESHOLD);
	}
	

	
	if(0){
		 if(battery_low_flag == 1){
		 	capacity = 0;
			printk(KERN_INFO "%s:battery is low,set capacity as 0 !\n", __func__);
		}
		battery_low_flag = 0;
	}
	 
	 
	return capacity;
	
}

 

int bq27520_get_gauge_ID(void)
{
	int ret;
	union power_supply_propval val;
	
	ret = bq27520_gauge_id(di_global,&val);
	return val.intval;	
}

int bq27520_enable_impedance_track_algorithm(void)
{
	int ret=0;
	union power_supply_propval val;

	printk(KERN_INFO "%s:IT enable start....\n", __func__);
	
	msleep(2000);
	ret = bq27520_gauge_IT_enable(di_global,&val);
	if(ret<0){
		printk(KERN_INFO "%s:IT enable fail.\n", __func__);
		return ret;
	}
	msleep(100);
	ret = bq27520_gauge_reset(di_global,&val);
	if(ret<0){
		printk(KERN_INFO "%s:gauge reset fail.\n", __func__);
		return ret;
	}
	printk(KERN_INFO "%s:IT enable end....\n", __func__);
	return ret;
}

int bq27520_disable_impedance_track_algorithm(void)
{
	int ret=0;
	union power_supply_propval val;

	printk(KERN_INFO "%s:IT disable .\n", __func__);
	
	msleep(2000);
	ret = bq27520_gauge_IT_disable(di_global,&val);
	if(ret<0){
		printk(KERN_INFO "%s:IT enable fail.\n", __func__);
		return ret;
	}
	msleep(100);
	ret = bq27520_gauge_reset(di_global,&val);
	if(ret<0){
		printk(KERN_INFO "%s:gauge reset fail.\n", __func__);
		return ret;
	}
	return ret;
}


int bq27520_check_impedance_track_algorithm(void)
{

	uint8_t buf[2];	
	uint8_t temp[4];	
	uint32_t status = 0;
	boolean ret;
	int retval = -1;
		
	buf[0] = 0x00;
	buf[1] = 0x00;
	memset(temp,0,sizeof(temp));
	ret = bq27520_write_retry(I2C_ADDR_NORMAL_MODE,0x00,buf,sizeof(buf));
	if (!ret)	{
		printk("bq27520 write control register fail\n");
		return retval;
	}
	msleep(100);
	ret=bq27520_read_retry(I2C_ADDR_NORMAL_MODE, 0x00,temp,sizeof(temp));
	if (!ret)	{
		printk("bq27520 read control register fail\n");
		return retval;
	}
	
	status = (status)|temp[3];
	status = (status<<24)|temp[2];
	status = (status<<16)|temp[1];
	status = (status<<8)|temp[0];
	
	if(TST_BIT(status,0))
		retval = 1;
	else
		retval = 0;
	
	return retval;
}
static struct msm_battery_gauge bq27520_batt_gauge = {
	.get_battery_mvolts = bq27520_get_battery_mvolts,
	.get_battery_temperature = bq27520_get_battery_temperature,
	.is_battery_present = bq27520_is_battery_present,
	.is_battery_temp_within_range = bq27520_is_charging_battery_temp_within_range,
	
	.is_discharging_battery_temp_within_range = bq27520_is_discharging_battery_temp_within_range,
	
	.is_battery_id_valid = bq27520_is_battery_id_valid,
   	.copy_dfi_from_user_to_kernel = bq27520_copy_dfi_user_to_kernel,
};
#endif

 

 int bq27520_gauge_firmware(void)
 {
	struct bq27x00_device_info *di;
	union power_supply_propval val;
	di=di_global;
	bq27520_gauge_firmware_version(di,&val);
	return val.intval;
 }

 int bq27520_battery_remain_capacity(void)

 {
 	int ret;
	int mAh = 0;
	
	ret = bq27x00_read(BQ27520_REG_REMAIN_CAPACITY, &mAh, 0, di_global);
	
	if (ret) {
		dev_err(di_global->dev, "error reading relative State-of-Charge\n");
		return ret;
	}

	return mAh; 
 }
 
int bq27520_battery_full_charge_capacity(void)
{
 	int ret;
	int mAh = 0;
	
	ret = bq27x00_read(BQ27520_REG_FULL_CHARGE_CAPACITY, &mAh, 0, di_global);
	
	if (ret) {
		dev_err(di_global->dev, "error reading relative State-of-Charge\n");
		return ret;
	}

	return mAh; 
}

int bq27520_gauge_flags(void)
{
	int flags = 0;
	int ret;

	ret = bq27x00_read(BQ27x00_REG_FLAGS, &flags, 0, di_global);
	if (ret < 0) {
		dev_err(di_global->dev, "error reading flags\n");
		return ret;
	}
	return flags;
}

int bq27520_gauge_control_status(void)
{

	char buf[2];	
	char temp[4];	
	int retval;
	int ret=0;
		
	buf[0]=0x00;
	buf[1]=0x00;
	memset(temp,0,sizeof(temp));
	retval=bq27520_write(0x00,buf,sizeof(buf),di_global);
	if (retval<0){
		printk("bq27520 write i2c error\n");	
		return -ENODATA;
	}
	msleep(20);
	retval = bq27520_read(0x00, temp, sizeof(temp), di_global);
	if(retval<0)
		printk("bq27520 read control register 0X00 error\n");
	else{
		ret=(ret)|temp[3];
		ret=(ret<<24)|temp[2];
		ret=(ret<<16)|temp[1];
		ret=(ret<<8)|temp[0];
	}
	
	return ret;
}

  
  
  static char data_buf[280];
  char *bq27520_soc_verification(void)
  {
  	int temperature;
	int voltage_mv;
	int remain_capacity;
	int full_capacity;
	int avg_current_ma;
	int soc;
	struct timeval tv;
	struct rtc_time tm;

	memset(data_buf,0,sizeof(data_buf));
	do_gettimeofday(&tv);
	rtc_time_to_tm(tv.tv_sec,&tm);
	
	temperature = bq27520_get_battery_temperature();
	voltage_mv = bq27520_get_battery_mvolts();
	remain_capacity = bq27520_battery_remain_capacity();
	full_capacity = bq27520_battery_full_charge_capacity();
	avg_current_ma = bq27520_get_battery_current();
	soc = bq27x00_battery_rsoc(di_global);

	snprintf(data_buf, sizeof(data_buf), "%2d/%3d/%4d:%2d::%2d::%2d    %8d  %8d   %8d  %8d %8d  %8d",
		tm.tm_mon,tm.tm_mday,tm.tm_year,tm.tm_hour, tm.tm_min, tm.tm_sec,
		temperature,voltage_mv,remain_capacity,full_capacity,avg_current_ma,soc);
	return data_buf;
  }
  
 
static int bq27520_request_gpio(int gpio[])
{
	int retval=0;
		
	retval = gpio_request(gpio[0], "BATT_LOW");
	if (retval) {
		printk(KERN_ERR "request BATT_LOW fail\n");
		gpio_free(gpio[0]);
	}	
	
	retval = gpio_request(gpio[1], "BATT_SOC_INT");
	if (retval) {
		printk(KERN_ERR "request BATT_SOC_INT fail\n");
		gpio_free(gpio[1]);
	}
	 return retval;

}

static int bq27520_free_gpio(int gpio[])
{
	gpio_free(gpio[0]);
	gpio_free(gpio[1]);
	return 0;
}


static void bq27520_bttery_low_work(struct work_struct *bq27520_work)
{
	printk(KERN_INFO "%s:battery low handler.\n", __func__);
	if(KNOWN_SPECIAL_CONDITION)
		return ;
	
	msm_charger_notify_event(NULL, CHG_BATT_BAT_LOW);

}
 static irqreturn_t bq27520_battery_low_handler(int irq, void *dev_id)
{
	
	if(1==update_dfi_status)
		return IRQ_HANDLED;
	
	schedule_delayed_work(&battery_low_work,HZ*2);
	return IRQ_HANDLED;
}

extern void arch_power_off(void);

static void bq27520_soc_int_work(struct work_struct *bq27520_work)
{		
	int flags;
	int count=0;
	int present=0;

	printk(KERN_INFO "%s:bq27520_soc_int_work.\n", __func__);
		 
	while(count < 5){
		bq27x00_read(BQ27x00_REG_FLAGS, &flags, 0, di_global);
				 
		if(TST_BIT(flags,3)){
			present=1;
			break;
		}else{
			count++;
		}
		msleep(2);
	}	
	if(!present) {
		msm_charger_notify_event(NULL, CHG_BATT_SOC_INT);
		arch_power_off();
	}
	else
		printk(KERN_INFO "%s:bq27520 battery is present.\n", __func__);
	

}

extern void disable_default_ramdump(void);

static irqreturn_t bq27520_soc_int_handler(int irq, void *dev_id)
{
	
	if(1==update_dfi_status)
		return IRQ_HANDLED;
	
	disable_default_ramdump();
	
	printk(KERN_INFO "%s:bq27520_soc_int_handler.\n", __func__);
	schedule_delayed_work(&soc_int_work,HZ*1);
	return IRQ_HANDLED;
}


#ifdef CHECK_POWERON_CONDITION
static void poweron_condition_check_work(struct work_struct *bq27520_work)
{		
	disable_default_ramdump();
	printk(KERN_INFO "%s, disable_default_ramdump in low battery.\n", __func__);
	
	printk(KERN_INFO "%s: not power on due to low battery.\n", __func__);
	arch_power_off();
}
#endif

int get_Battery_valid(void)
{
	return Battery_valid;
}
EXPORT_SYMBOL(get_Battery_valid);


static int bq27x00_battery_probe(struct i2c_client *client,
				 const struct i2c_device_id *id)
{

	char *name;
	struct bq27x00_device_info *di;
	struct bq27x00_access_methods *bus;
	int num;
	int retval = 0;
	uint8_t rom_buf=0xff;
	uint16_t normal_buf=0x0000;
	boolean ret_normal_ok=FALSE;
	boolean ret_rom_ok = FALSE;

	struct bq27520_platform_data *bq27520_gauge_data;	
	int bat_low_int;
	int soc_int;
	
	
	retval = idr_pre_get(&battery_id, GFP_KERNEL);
	if (retval == 0)
		return -ENOMEM;
	mutex_lock(&battery_mutex);
	retval = idr_get_new(&battery_id, client, &num);
	mutex_unlock(&battery_mutex);
	if (retval < 0)
		return retval;

	name = kasprintf(GFP_KERNEL, "%s-%d", id->name, num);
	if (!name) {
		dev_err(&client->dev, "failed to allocate device name\n");
		retval = -ENOMEM;
		goto batt_failed_1;
	}

	di = kzalloc(sizeof(*di), GFP_KERNEL);
	if (!di) {
		dev_err(&client->dev, "failed to allocate device info data\n");
		retval = -ENOMEM;
		goto batt_failed_2;
	}
	di->id = num;
	di->chip = id->driver_data;

	bus = kzalloc(sizeof(*bus), GFP_KERNEL);
	if (!bus) {
		dev_err(&client->dev, "failed to allocate access method "
					"data\n");
		retval = -ENOMEM;
		goto batt_failed_3;
	}

	i2c_set_clientdata(client, di);
	di->dev = &client->dev;
	di->bat.name = name;
	bus->read = &bq27x00_read_i2c;
	di->bus = bus;
	di->client = client;
	di_global=di;
	#ifdef BQ27520_REGISTER_TO_MSM
		
 	mutex_lock(&gauge_mutex);
	ret_normal_ok = bq27520_read_retry(I2C_ADDR_NORMAL_MODE,0x00,(uint8_t*)&normal_buf, 2);
	if(!ret_normal_ok){
		ret_rom_ok = bq27520_read_retry(I2C_ADDR_ROM_MODE, 0x00, &rom_buf, 1);
		if(ret_rom_ok)
			in_rom_mode = 1;
		else
			goto batt_failed_4;
	}
	mutex_unlock(&gauge_mutex);
	printk(KERN_INFO "%s:bq27520 in_rom_mode=%d\n", __func__,in_rom_mode);	
		
	
#ifdef CHECK_POWERON_CONDITION
	if( (3 != get_mfg_mode()) && (1 != get_mfg_mode()) && (1 != in_rom_mode)
		&&(4 != get_mfg_mode())) {
		int voltage;
		int capacity;
		
		voltage = bq27520_get_battery_mvolts();
		capacity = bq27520_get_battery_capacity();
		printk(KERN_INFO "%s, voltage %d, capacity %d.\n", __func__, voltage, capacity);
		
		if( (3400 > voltage) && (5 > capacity) ) {
			Battery_valid = 0;
			
			INIT_DELAYED_WORK(&poweron_condition_check_work_work, poweron_condition_check_work);
			schedule_delayed_work(&poweron_condition_check_work_work, HZ*2);
		}
	}
	printk(KERN_INFO "%s:poweron_condition_check_work_work checked.\n", __func__);	
#endif

	msm_battery_gauge_register(&bq27520_batt_gauge);
	if(0){
		bq27x00_powersupply_init(di);
		retval = power_supply_register(&client->dev, &di->bat);
		if (retval) {
			dev_err(&client->dev, "failed to register battery\n");
			goto batt_failed_4;
		}
	}
	#else	
	if(1){
		bq27x00_powersupply_init(di);
		retval = power_supply_register(&client->dev, &di->bat);
		if (retval) {
			dev_err(&client->dev, "failed to register battery\n");
			goto batt_failed_4;
		}
	}
	#endif

	  
	INIT_DELAYED_WORK(&battery_low_work, bq27520_bttery_low_work);
	bq27520_gauge_data = di->client->dev.platform_data;
	bq27520_data = bq27520_gauge_data;
	bq27520_request_gpio(bq27520_gauge_data->bat_gpio_int);
	bat_low_int = gpio_to_irq(*(bq27520_gauge_data->bat_gpio_int+0));

	retval = bq27520_gauge_data->gauge_request_battery_low_irq(bat_low_int, bq27520_battery_low_handler,  
		IRQF_TRIGGER_RISING,"BAT_LOW", client);
	if(retval < 0){
		printk(KERN_ERR "bq27520 request battery low interrupt fail\n");
		gpio_free(*(bq27520_gauge_data->bat_gpio_int+0));		
	}	
	set_irq_wake(bat_low_int, 1);
	
	  
	
	INIT_DELAYED_WORK(&soc_int_work, bq27520_soc_int_work);
	soc_int = gpio_to_irq(*(bq27520_gauge_data->bat_gpio_int+1));
	
	if(bq27520_version_DFI()>BQ27520_OLD_DFI_VERSION){
		retval = bq27520_gauge_data->gauge_request_soc_int_irq(soc_int, bq27520_soc_int_handler,  
			IRQF_TRIGGER_FALLING,"SOC_INT", client);
		if(retval < 0){
			printk(KERN_ERR "bq27520 request soc_int interrupt fail\n");
			gpio_free(*(bq27520_gauge_data->bat_gpio_int+1));		
		}else
			printk(KERN_ERR "bq27520 request soc_int interrupt success\n");
		set_irq_wake(soc_int, 1);
	}
	
	dev_info(&client->dev, "support ver. %s enabled\n", DRIVER_VERSION);
	
	retval = bq27520_gauge_control_status();
	printk(KERN_INFO "%s:control status register=0x%x.\n", __func__,retval);
	

	return 0;

batt_failed_4:
	kfree(bus);
batt_failed_3:
	kfree(di);
batt_failed_2:
	kfree(name);
batt_failed_1:
	mutex_lock(&battery_mutex);
	idr_remove(&battery_id, num);
	mutex_unlock(&battery_mutex);

	return retval;

}

static int bq27x00_battery_remove(struct i2c_client *client)
{
	struct bq27x00_device_info *di = i2c_get_clientdata(client);
	struct bq27520_platform_data *bq27520_gauge_data;

	cancel_delayed_work_sync(&battery_low_work);

	#ifdef BQ27520_REGISTER_TO_MSM
	msm_battery_gauge_unregister(&bq27520_batt_gauge);
	if(0){
		power_supply_unregister(&di->bat);
	}
	#else
	power_supply_unregister(&di->bat);
	#endif
	
	kfree(di->bat.name);

	mutex_lock(&battery_mutex);
	idr_remove(&battery_id, di->id);
	mutex_unlock(&battery_mutex);

	bq27520_gauge_data = di->client->dev.platform_data;
	bq27520_free_gpio(bq27520_gauge_data->bat_gpio_int);
	kfree(di);

	return 0;
}


#ifdef CONFIG_PM
static int bq27520_suspend(struct device *dev)
{	
	
	return 0;
}

static int bq27520_resume(struct device *dev)
{
	
	
	return 0;
}

static const struct dev_pm_ops bq27520_pm_ops = {
	.suspend = bq27520_suspend,
	.resume = bq27520_resume,
};
#endif

static const struct i2c_device_id bq27x00_id[] = {

	{"bq27x00-battery",BQ27500},
	{},
};

static struct i2c_driver bq27x00_battery_driver = {
	.driver = {
		.name = "bq27x00-battery",
#ifdef CONFIG_PM
		 .pm = &bq27520_pm_ops,
#endif
	},
	.probe = bq27x00_battery_probe,
	.remove = bq27x00_battery_remove,
	.id_table = bq27x00_id,
};

static int __init bq27x00_battery_init(void)
{
	int ret;

	ret = i2c_add_driver(&bq27x00_battery_driver);
	if (ret)
		printk(KERN_ERR "Unable to register BQ27x00 driver\n");
	return ret;
}
module_init(bq27x00_battery_init);

static void __exit bq27x00_battery_exit(void)
{
	i2c_del_driver(&bq27x00_battery_driver);
}
module_exit(bq27x00_battery_exit);

MODULE_AUTHOR("Rodolfo Giometti <giometti@linux.it>");
MODULE_DESCRIPTION("BQ27x00 battery monitor driver");
MODULE_LICENSE("GPL");
