/* Copyright (c) 2010 Code Aurora Forum. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 */

#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/module.h>
#include <linux/debugfs.h>
#include <linux/workqueue.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <mach/max8903b_charge.h>
#include <linux/power_supply.h>
#include <linux/platform_device.h>
#include <linux/gpio/gpio_def.h>


#include <linux/pwm.h>
#include <linux/pmic8058-pwm.h>
#include <linux/mfd/pmic8058.h>
#include <linux/reboot.h>
#include <linux/msm-charger.h>

#include <mach/msm_hsusb.h>

#define MAX8903B_USB_CHARGE_RATING_500	 500
#define MAX8903B_USB_CHARGE_RATING_100 	100
#define MAX8903B_AC_CHARGE_RATING  		1200
#define MAX8903B_RESOURCE_NUM 		12
#define MAX8903B_CHG_PERIOD			((HZ) * 30)
#define MAX8903B_DEBGUG			1
#ifdef   MAX8903B_DEBGUG
#define _degug_printf	printk
#else
#define _degug_printf(a...)
#endif
enum charger_supply_type {
	MAX8903B_AC,
	MAX8903B_USB,
};

struct max8903b_struct {	
	
	struct delayed_work charge_work;

	int present;
	int batt_present;
	bool charging;
	bool fast_charging;
	int chgcurrent;
	int term_current;
	int max_system_voltage;
	int min_system_voltage;

	int chg_type;
	int chg_current;

	int valid_n_gpio;
	struct dentry *dent;
	struct msm_hardware_charger adapter_hw_chg;
};

struct resource  *max8903b_resource[MAX8903B_RESOURCE_NUM];
struct max8903b_struct *max8903b_chg_global;
struct delayed_work max8903b_charge_work;

enum gpio_output{
	MAX8903B_GPIO_CHG_IUSB=0,
	MAX8903B_GPIO_CHG_USUS,
	MAX8903B_GPIO_CHG_DCM,
	MAX8903B_GPIO_CHG_EN,
	MAX8903B_GPIO_CHG_FAULT,
	MAX8903B_GPIO_DC_PWR_OK,
	MAX8903B_GPIO_CHG_STAT,
};

extern int bq27520_get_battery_mvolts(void);
extern int bq27520_get_battery_current(void);
extern int bq27520_get_battery_temperature(void);
extern struct pm8058_chip	*g_pm_chip_for_all;
extern enum chg_type query_usb_chg_type(void);





static int max8903b_request_gpio(struct resource *res[ ])
{
	int status;

	
	status = gpio_request(res[0]->start, res[0]->name);
	if (status) {
		printk(KERN_ERR"request CHG_IUSB fail\n");
		goto free_gpio_io0;
	}
	
	status = gpio_request(res[1]->start, res[1]->name);
	if (status) {
		printk(KERN_ERR "request CHG_USUS fail\n");
		goto free_gpio_io1;
	}
	
	status = gpio_request(res[2]->start, res[2]->name);
	if (status) {
		printk(KERN_ERR "request CHG_DCM fail\n");
		goto free_gpio_io2;
	}
	
	status = gpio_request(res[3]->start, res[3]->name);
	if (status) {
		printk(KERN_ERR "request CHG_EN fail\n");
		goto free_gpio_io3;
	}
	
	status = gpio_request(res[4]->start, res[4]->name);
	if (status) {
		printk(KERN_ERR "request CHG_FAULT fail\n");
		goto free_gpio_io4;
	}
	
	status = gpio_request(res[5]->start, res[5]->name);
	if (status) {
		printk(KERN_ERR "request DC_PWR_OK fail\n");
		goto free_gpio_io5;
	}
	
	status = gpio_request(res[6]->start, res[6]->name);
	if (status) {
		printk(KERN_ERR "request CHG_STAT fail\n");
		goto free_gpio_io6;
	}

	return 0;
free_gpio_io6:
	gpio_free(res[6]->start);
free_gpio_io5:
	gpio_free(res[5]->start);
free_gpio_io4:
	gpio_free(res[4]->start);
free_gpio_io3:
	gpio_free(res[3]->start);
free_gpio_io2:
	gpio_free(res[2]->start);
free_gpio_io1:
	gpio_free(res[1]->start);
free_gpio_io0:
	gpio_free(res[0]->start);

	return 1;

}



static int max8903b_set_gpio(struct resource *res[ ],int  gpio,int value )
{
	int status;
	switch (gpio) {
		case MAX8903B_GPIO_CHG_IUSB:
			status = gpio_direction_output(res[0]->start,value);
			if (status) {
				printk(KERN_ERR "failed to configure gpio CHG_IUSB/IO135");
				goto free_gpio0;
			}			
			break;
		case MAX8903B_GPIO_CHG_USUS:
			status = gpio_direction_output(res[1]->start,value);
			if (status) {
				printk(KERN_ERR "failed to configure gpio CHG_USUS/IO136");
				goto free_gpio1;
			}
			break;
		case MAX8903B_GPIO_CHG_DCM:
			status = gpio_direction_output(res[2]->start,value);
			if (status) {
				printk(KERN_ERR "failed to configure gpio CHG_DCM/IO137");
				goto free_gpio2;
			}
			break;
		case MAX8903B_GPIO_CHG_EN:
			status = gpio_direction_output(res[3]->start,value);
			if (status) {
				printk(KERN_ERR "failed to configure gpio CHG_EN/IO138");
				goto free_gpio3;
			}
			break;

		
		case MAX8903B_GPIO_CHG_FAULT:
			status = gpio_direction_input(res[4]->start);
			if (status) {
				printk(KERN_ERR "failed to configure gpio CHG_FAULT/IO124");
				goto free_gpio4;
			}
			break;
		
		case MAX8903B_GPIO_DC_PWR_OK:
			status = gpio_direction_input(res[5]->start);
			if (status) {
				printk(KERN_ERR "failed to configure gpio DC_PWR_OK/IO125");
				goto free_gpio5;
			}
			break;
		
		case MAX8903B_GPIO_CHG_STAT:
			status = gpio_direction_input(res[6]->start);
			if (status) {
				printk(KERN_ERR "failed to configure gpio CHG_STAT/IO134");
				goto free_gpio6;
			}
			break;
		default:
			return -EINVAL;
	}
	return 1;

free_gpio6:
	gpio_free(res[6]->start);	
free_gpio5:
	gpio_free(res[5]->start);
free_gpio4:
	gpio_free(res[4]->start);
free_gpio3:
	gpio_free(res[3]->start);
free_gpio2:
	gpio_free(res[2]->start);
free_gpio1:
	gpio_free(res[1]->start);
free_gpio0:
	gpio_free(res[0]->start);

	return -EINVAL;
	
}



static int max8903b_stop_charging(struct msm_hardware_charger *hw_chg)
{


	if (!(max8903b_chg_global->charging))
		
		return 0;
	
	max8903b_set_gpio(max8903b_resource,MAX8903B_GPIO_CHG_EN,0);	
	max8903b_chg_global->charging = false;
	cancel_delayed_work(&max8903b_chg_global->charge_work);

	return 0;

}



static int max8903b_start_charging(struct msm_hardware_charger *hw_chg,
		int chg_voltage, int chg_current)
{

	_degug_printf(KERN_ERR "max8903b_start_charging,chg_current=%d\n",chg_current);
	if (max8903b_chg_global->charging)
		
		return 0;

	
	if(chg_current==0){
		return 1;
	}else if(chg_current<=500){
		max8903b_set_gpio(max8903b_resource,MAX8903B_GPIO_CHG_DCM,0);
		max8903b_set_gpio(max8903b_resource,MAX8903B_GPIO_CHG_IUSB,0);
		max8903b_chg_global->adapter_hw_chg.charger_type= CHARGER_TYPE_USB;
	}else{
		max8903b_set_gpio(max8903b_resource,MAX8903B_GPIO_CHG_DCM,1);	
		max8903b_chg_global->adapter_hw_chg.charger_type= CHARGER_TYPE_AC;
	}	
	
	max8903b_set_gpio(max8903b_resource,MAX8903B_GPIO_CHG_USUS,0);
	
	max8903b_set_gpio(max8903b_resource,MAX8903B_GPIO_CHG_EN,1);	

	schedule_delayed_work(&max8903b_chg_global->charge_work,MAX8903B_CHG_PERIOD);
	max8903b_chg_global->charging = true;
	
	return 0;	
}

 

static int max8903b_set_charging_current(struct msm_hardware_charger *hw_chg,int chg_current)
{
	int fast_current;
	_degug_printf(KERN_ERR "$$max8903b set fast-charging current=%d\n",chg_current);
	fast_current=chg_current;
	if(fast_current==500){
		max8903b_set_gpio(max8903b_resource,MAX8903B_GPIO_CHG_DCM,0);
		max8903b_set_gpio(max8903b_resource,MAX8903B_GPIO_CHG_IUSB,0);
	}else if(fast_current>500){
		max8903b_set_gpio(max8903b_resource,MAX8903B_GPIO_CHG_DCM,1);	
	}
	return 1;
}



static int max8903b_get_charging_current(struct msm_hardware_charger *hw_chg)
{	
	int chg_type;
	int ret;
	
	chg_type=max8903b_chg_global->adapter_hw_chg.charger_type;
	_degug_printf(KERN_ERR "$$max8903b get fast-charging current,chg_type=%d\n",chg_type);
	switch(chg_type){
		case CHARGER_TYPE_USB:
			ret= MAX8903B_USB_CHARGE_RATING_500;
			break;
		case CHARGER_TYPE_AC:
			ret= MAX8903B_AC_CHARGE_RATING;
			break;
		default:
			return MAX8903B_USB_CHARGE_RATING_500;
	}
	return ret;
	
}

 
static int max8903b_charging_switched(struct msm_hardware_charger *hw_chg)
{
	return 0;
}

static int max8903b_platform_get_resource(struct platform_device *plat_dev,struct resource *res_array[ ])
{
	res_array[0] = platform_get_resource_byname(plat_dev, IORESOURCE_IRQ, "CHG_IUSB/IO135");
	if (res_array[0] == NULL) {
		printk(KERN_INFO "%s:couldnt find resource CHG_IUSB/IO135\n", __func__);
		return 1;
	} 
	
	res_array[1]= platform_get_resource_byname(plat_dev, IORESOURCE_IRQ, "CHG_USUS/IO136");
	if (res_array[1] == NULL) {
		printk(KERN_INFO "%s:couldnt find resource CHG_USUS/IO136\n", __func__);
		return 1;
	} 
	
	res_array[2] = platform_get_resource_byname(plat_dev, IORESOURCE_IRQ, "CHG_DCM/IO137");
	if (res_array[2] == NULL) {
		printk(KERN_INFO "%s:couldnt find resource CHG_DCM/IO137\n", __func__);
		return 1;
	} 
	
	res_array[3] = platform_get_resource_byname(plat_dev, IORESOURCE_IRQ, "CHG_EN/IO138");
	if (res_array[3] == NULL) {
		printk(KERN_INFO "%s:couldnt find resource CHG_EN/IO138\n", __func__);
		return 1;
	}
	res_array[4] = platform_get_resource_byname(plat_dev, IORESOURCE_IRQ, "CHG_FAULT/IO124");
	if (res_array[4] == NULL) {
		printk(KERN_INFO "%s:couldnt find resource CHG_FAULT/IO124\n", __func__);
		return 1;
	}
	res_array[5] = platform_get_resource_byname(plat_dev, IORESOURCE_IRQ, "DC_PWR_OK/IO125");
	if (res_array[5] == NULL) {
		printk(KERN_INFO "%s:couldnt find resource DC_PWR_OK/IO125\n", __func__);
		return 1;
	}
	res_array[6] = platform_get_resource_byname(plat_dev, IORESOURCE_IRQ, "CHG_STAT/IO134");
	if (res_array[6] == NULL) {
		printk(KERN_INFO "%s:couldnt find resource CHG_STAT/IO134\n", __func__);
		return 1;
	}
	
	res_array[7]= platform_get_resource_byname(plat_dev, IORESOURCE_IRQ, "CHGVAL");
	if (res_array[7] == NULL) {
		printk(KERN_INFO "%s:couldnt find resource CHGVAL\n", __func__);
		return 1;
	} 
	res_array[8]= platform_get_resource_byname(plat_dev, IORESOURCE_IRQ, "CHGINVAL");
	if (res_array[8] == NULL) {
		printk(KERN_INFO "%s:couldnt find resource CHGINVAL\n", __func__);
		return 1;
	} 

	return 0;
}



static int max8903b_is_chg_plugged_in(void)
{
	int ret;
	ret = pm8058_irq_get_rt_status(g_pm_chip_for_all,max8903b_resource[7]->start);
	if(1 == ret)
		return 1;
	else
		return 0;
	
}

 
 
static irqreturn_t max8903_chg_chgval_handler(int irq, void *dev_id)
{
	_degug_printf(KERN_ERR "$$max8903b chgval\n");
	if (max8903b_is_chg_plugged_in()) {	
		if (!max8903b_chg_global->present) {
			
			msm_charger_notify_event(&max8903b_chg_global->adapter_hw_chg,CHG_INSERTED_EVENT);
			max8903b_chg_global->present = 1;
			max8903b_chg_global->adapter_hw_chg.charger_plug_status=1;
		}
	} else{
		if (max8903b_chg_global->present) {
			
			msm_charger_notify_event(&max8903b_chg_global->adapter_hw_chg,CHG_REMOVED_EVENT);
			max8903b_chg_global->present = 0;
			max8903b_chg_global->adapter_hw_chg.charger_plug_status=0;
			 
			max8903b_chg_global->adapter_hw_chg.charger_type= CHARGER_TYPE_INVALID;
			 
   			  
			 #if 0
			if(mfg_mode == 3){
				kernel_power_off();
			}
			#endif
			
		}
	}
	cancel_delayed_work_sync(&max8903b_charge_work);
	schedule_delayed_work(&max8903b_chg_global->charge_work,MAX8903B_CHG_PERIOD);
	return IRQ_HANDLED;
}
 
 

static irqreturn_t max8903_chg_chginval_handler(int irq, void *dev_id)
{
	_degug_printf(KERN_ERR "$$max8903b chginval\n");
	if (max8903b_chg_global->present) {
		if (!max8903b_is_chg_plugged_in()) {
			msm_charger_notify_event(&max8903b_chg_global->adapter_hw_chg,CHG_REMOVED_EVENT);
			max8903b_chg_global->present = 0;
		}
	}

	return IRQ_HANDLED;
}
 

  
static irqreturn_t max8903_chg_fault_handler(int irq, void *dev_id)
{
	_degug_printf(KERN_ERR "$$max8903b chg_fault\n");
	return IRQ_HANDLED;
}
  

  
static irqreturn_t max8903_dc_pwr_ok_handler(int irq, void *dev_id)
{
	_degug_printf(KERN_ERR "$$max8903b dc_pwr_ok\n");
	return IRQ_HANDLED;
}
  

 
static irqreturn_t max8903_chg_state_handler(int irq, void *dev_id)
{
	_degug_printf(KERN_ERR "$$max8903b chg_state\n");
	max8903b_chg_global->fast_charging = true;
	return IRQ_HANDLED;
}
 
   
    
static int max8903b_charge_done_event(void)
{	int current_ma;
	int voltage_mv;
	int ret=0;
	current_ma=bq27520_get_battery_current();
	voltage_mv=bq27520_get_battery_mvolts();
	if(((current_ma>0)&&(current_ma<150))&&(voltage_mv>4200))
		ret=1;
	return ret;

}
    
   
     
static void max8903b_charge_check_status(struct work_struct *max8903b_work)
{
	int charge_done ;
	charge_done=max8903b_charge_done_event();
	if (max8903b_chg_global->charging) {
		if(charge_done){
			msm_charger_notify_event(&max8903b_chg_global->adapter_hw_chg,	CHG_DONE_EVENT);
		}
		if(max8903b_chg_global->fast_charging){
			msm_charger_notify_event(&max8903b_chg_global->adapter_hw_chg,	CHG_BATT_BEGIN_FAST_CHARGING);
		}		
	}
	schedule_delayed_work(&max8903b_chg_global->charge_work,MAX8903B_CHG_PERIOD);	
}
     

     
static void max8903b_off_charging_work(struct work_struct *max8903b_work)
{
	if (max8903b_is_chg_plugged_in()) {	
		if (!max8903b_chg_global->present) {
			
			msm_charger_notify_event(&max8903b_chg_global->adapter_hw_chg,CHG_INSERTED_EVENT);
			max8903b_chg_global->present = 1;
			max8903b_chg_global->adapter_hw_chg.charger_plug_status=1;
		}
	} else{
		if (max8903b_chg_global->present) {
			
			msm_charger_notify_event(&max8903b_chg_global->adapter_hw_chg,CHG_REMOVED_EVENT);
			max8903b_chg_global->present = 0;
			max8903b_chg_global->adapter_hw_chg.charger_plug_status=0;
		}
	}
}

    
    
     
static int __devinit max8903b_request_irqs(struct resource *res[ ])
{
	int ret;
	int chg_fault_int;
	int chg_pwr_ok_int;
	int chg_state_int;

	
	chg_fault_int = gpio_to_irq(res[4]->start);

	ret = request_irq(chg_fault_int, max8903_chg_fault_handler, IRQF_TRIGGER_FALLING | IRQF_TRIGGER_RISING,
		             res[4]->name, NULL);
	if(ret < 0){
		printk(KERN_ERR "max8903b request_irq chg_fault_int fail\n");
		goto err_out;
	}
	set_irq_wake(chg_fault_int, 1);

	
	chg_pwr_ok_int = gpio_to_irq(res[5]->start);

	ret = request_irq(chg_pwr_ok_int, max8903_dc_pwr_ok_handler, IRQF_TRIGGER_FALLING | IRQF_TRIGGER_RISING,
		             res[5]->name, NULL);
	if(ret < 0){
		printk(KERN_ERR "max8903b request_irq chg_pwr_ok_int fail\n");
		goto err_out;
	}
	set_irq_wake(chg_pwr_ok_int, 1);
	
	
	chg_state_int = gpio_to_irq(res[6]->start);

	ret = request_irq(chg_state_int, max8903_chg_state_handler, IRQF_TRIGGER_FALLING | IRQF_TRIGGER_RISING,
		             res[6]->name, NULL);
	if(ret < 0){
		printk(KERN_ERR "max8903b request_irq chg_state_int fail\n");
		goto err_out;
	}
	set_irq_wake(chg_state_int, 1);	

	
	printk("chg_state:%s=%d,chgval:%s=%d,chginval:%s=%d\n",res[6]->name,res[6]->start,
					res[7]->name,res[7]->start,res[8]->name,res[8]->start);
	ret = request_threaded_irq(res[7]->start, NULL,max8903_chg_chgval_handler,
				  IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING,
				 res[7]->name, NULL);
	if (ret < 0) {
		printk(KERN_ERR "max8903b request CHGVAL interrupt fail\n");
		goto err_out;
	}
	set_irq_wake(res[7]->start, 1);
	if(0){
		
		ret = request_threaded_irq(res[8]->start, NULL,max8903_chg_chginval_handler,
					  IRQF_TRIGGER_RISING,
					 res[8]->name, NULL);
		if (ret < 0) {
			printk(KERN_ERR "max8903b request CHGINVAL interrupt fail\n");
			goto err_out;
		}
		set_irq_wake(res[8]->start, 1);
	}
	return 0;

err_out:
	free_irq(res[8]->start, NULL);
	free_irq(res[7]->start, NULL);
	gpio_free(res[6]->start);
	gpio_free(res[5]->start);
	gpio_free(res[4]->start);
	
	return -EINVAL;
}
     
static int __devinit max8903b_charge_probe(struct platform_device *plat_dev)
{
	int ret = 0;	

	struct max8903b_struct *max8903b_chg;
	max8903b_chg = kzalloc(sizeof(*max8903b_chg), GFP_KERNEL);

	if (!max8903b_chg) {
		ret = -ENOMEM;
		goto out;
	}

	ret=max8903b_platform_get_resource(plat_dev,max8903b_resource);
	if(ret)
		goto out;

	ret = max8903b_request_gpio(max8903b_resource);
	if(ret)
		goto out;

	INIT_DELAYED_WORK(&max8903b_chg->charge_work, max8903b_charge_check_status);
	INIT_DELAYED_WORK(&max8903b_charge_work,max8903b_off_charging_work);

	max8903b_chg->fast_charging = false;
	max8903b_chg->adapter_hw_chg.type = CHG_TYPE_USB;
	max8903b_chg->adapter_hw_chg.rating = 2;
	max8903b_chg->adapter_hw_chg.name = "max8903b-adapter";
	max8903b_chg->adapter_hw_chg.start_charging = max8903b_start_charging;
	max8903b_chg->adapter_hw_chg.stop_charging =max8903b_stop_charging;
	max8903b_chg->adapter_hw_chg.charging_switched = max8903b_charging_switched;
	max8903b_chg->adapter_hw_chg.set_charging_current = max8903b_set_charging_current;
	max8903b_chg->adapter_hw_chg.get_charging_current = max8903b_get_charging_current;
	max8903b_chg->adapter_hw_chg.charger_type=CHARGER_TYPE_INVALID;
	max8903b_chg_global=max8903b_chg;
	if(1){
		ret = msm_charger_register(&max8903b_chg->adapter_hw_chg);
		if (ret) {		
			goto out;
		}
	}
	ret=max8903b_request_irqs(max8903b_resource);
	if (ret) {
		pr_err("%s: couldnt register interrupts\n", __func__);
		goto out;
	}
     
	schedule_delayed_work(&max8903b_charge_work, (HZ) * 2);
    
	return 0;
out:	
	kfree(max8903b_chg);
	return ret;

	
}

static int __devexit max8903b_remove(struct platform_device *pdev )
{
	
	free_irq(max8903b_resource[7]->start, NULL);
	free_irq(max8903b_resource[6]->start, NULL);
	free_irq(max8903b_resource[5]->start, NULL);
	free_irq(max8903b_resource[4]->start, NULL);
	
	gpio_free(max8903b_resource[6]->start);
	gpio_free(max8903b_resource[5]->start);
	gpio_free(max8903b_resource[4]->start);
	gpio_free(max8903b_resource[3]->start);
	gpio_free(max8903b_resource[2]->start);
	gpio_free(max8903b_resource[1]->start);
	gpio_free(max8903b_resource[0]->start);

	cancel_delayed_work_sync(&max8903b_charge_work);
	cancel_delayed_work_sync(&max8903b_chg_global->charge_work);
	msm_charger_notify_event(&max8903b_chg_global->adapter_hw_chg, CHG_REMOVED_EVENT);
	msm_charger_unregister(&max8903b_chg_global->adapter_hw_chg);
	kfree(max8903b_chg_global);
	return 0;
}

#ifdef CONFIG_PM
static int max8903b_suspend(struct device *dev)
{	
	
	if (max8903b_chg_global->charging)
		return -EBUSY;
	else{
		gpio_free(max8903b_resource[6]->start);
		gpio_free(max8903b_resource[5]->start);
		gpio_free(max8903b_resource[4]->start);
		gpio_free(max8903b_resource[3]->start);
		gpio_free(max8903b_resource[2]->start);
		gpio_free(max8903b_resource[1]->start);
		gpio_free(max8903b_resource[0]->start);
		cancel_delayed_work_sync(&max8903b_chg_global->charge_work);
		_degug_printf(KERN_ERR "max8903b--suspend\n");
	}
	return 0;
}

static int max8903b_resume(struct device *dev)
{	
	max8903b_request_gpio(max8903b_resource);
	schedule_delayed_work(&max8903b_chg_global->charge_work,MAX8903B_CHG_PERIOD);	
	_degug_printf(KERN_ERR "max8903b--resume\n");
	return 0;
}
static struct dev_pm_ops max8903b_pm_ops = {
	.suspend	= max8903b_suspend,
	.resume	= max8903b_resume,
};
#endif
static struct platform_driver max8903b_charge_driver = {
	.driver	= {
		.name	= "max8903b_charge",
		.owner	= THIS_MODULE,
		#ifdef CONFIG_PM
		.pm	= &max8903b_pm_ops,
		#endif
	},
	.probe    = max8903b_charge_probe,
	.remove = __devexit_p(max8903b_remove),
};


static int __init max8903b_charge_init(void)
{
	int ret;
	ret = platform_driver_register(&max8903b_charge_driver);
	return ret;
}
static void __exit max8903b_charge_exit(void)
{
	platform_driver_unregister(&max8903b_charge_driver);
}



module_init(max8903b_charge_init);
module_exit(max8903b_charge_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("John Ruan, Qisda Incorporated");
MODULE_DESCRIPTION("QSD Battery driver");
