/* Copyright (c) 2010-2011, Code Aurora Forum. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/err.h>
#include <linux/mfd/pmic8058.h>
#include <linux/interrupt.h>
#include <linux/power_supply.h>
#include <linux/delay.h>
#include <linux/bitops.h>
#include <linux/debugfs.h>
#include <linux/msm-charger.h>
#include <linux/time.h>
#include <linux/slab.h>
#ifdef CONFIG_HAS_WAKELOCK
#include <linux/wakelock.h>
#endif

#include <linux/msm_adc.h>


#include <asm/atomic.h>

#include <mach/msm_hsusb.h>

#include <linux/gpio/gpio_def.h>
#include <linux/i2c/bq27520_gauge.h>
#include <linux/i2c/smb136_charge.h>
#include <asm-generic/gpio.h>
#include <mach/hw_id.h>
#include <linux/hrtimer.h>

#define MSM_CHG_MAX_EVENTS		16
#define CHARGING_TEOC_MS		9000000
#define UPDATE_TIME_MS			60000
#define RESUME_CHECK_PERIOD_MS		60000

#define DEFAULT_BATT_MAX_V		4200
#define DEFAULT_BATT_MIN_V		3200
#define DEFAULT_BATT_RESUME_V		4100
#define DEFAULT_BATT_RESUME_C		96

#define MSM_CHARGER_GAUGE_MISSING_VOLTS 3500
#define MSM_CHARGER_GAUGE_MISSING_TEMP  35
#define FAST_CHARGING_THRESHOLD_VOLTAGE_MV		3400
#define FAST_CHARGING_THRESHOLD_CURRENT_MA	500
#define MSM_BATT_EARLY_CHARGING_VOLTAGE		3100

#define IGNORE_BAT_PARA_MONITOR_BEFORE_CHARGING	1

 
#define LARGE_CONSUME_NUMBER				6
#define LARGE_CONSUME_SAFETY_TIME			5


 #define FAST_UPDATE_BATTERY_PARA_COUNT		6
 #define FAST_UPDATE_BATTERY_PARA_TIME		5
 
/**
 * enum msm_battery_status
 * @BATT_STATUS_ABSENT: battery not present
 * @BATT_STATUS_ID_INVALID: battery present but the id is invalid
 * @BATT_STATUS_DISCHARGING: battery is present and is discharging
 * @BATT_STATUS_TRKL_CHARGING: battery is being trickle charged
 * @BATT_STATUS_FAST_CHARGING: battery is being fast charged
 * @BATT_STATUS_JUST_FINISHED_CHARGING: just finished charging,
 *		battery is fully charged. Do not begin charging untill the
 *		voltage falls below a threshold to avoid overcharging
 * @BATT_STATUS_TEMPERATURE_OUT_OF_RANGE: battery present,
					no charging, temp is hot/cold
 */
enum msm_battery_status {
	BATT_STATUS_ABSENT,
	BATT_STATUS_ID_INVALID,
	BATT_STATUS_CURRENT_OUT_OF_RANGE,	
	BATT_STATUS_TEMPERATURE_OUT_OF_RANGE,
	BATT_STATUS_VOLTAGE_OUT_OF_RANGE,
	BATT_STATUS_LARGE_CONSUME,
	BATT_STATUS_TIME_OUT,	
	BATT_STATUS_DISCHARGING,
	BATT_STATUS_EARLY_CHARGING,
	BATT_STATUS_TRKL_CHARGING,
	BATT_STATUS_FAST_CHARGING,
	BATT_STATUS_TAPER_CHARGING,
	BATT_STATUS_JUST_FINISHED_CHARGING,
	BATT_STATUS_TERMINATION_CHARGING,
	BATT_STATUS_RECHARGING,
};

struct msm_hardware_charger_priv {
	struct list_head list;
	struct msm_hardware_charger *hw_chg;
	enum msm_hardware_charger_state hw_chg_state;
	unsigned int max_source_current;
	struct power_supply psy;
#ifdef CONFIG_HAS_WAKELOCK
	struct wake_lock wl;
#endif
};

#ifdef CONFIG_HAS_WAKELOCK
	struct wake_lock heart_wl;
#endif

struct msm_charger_event {
	enum msm_hardware_charger_event event;
	struct msm_hardware_charger *hw_chg;
};

struct msm_charger_mux {
	int inited;
	struct list_head msm_hardware_chargers;
	int count_chargers;
	struct mutex msm_hardware_chargers_lock;

	struct device *dev;

	unsigned int max_voltage;
	unsigned int min_voltage;
	unsigned int resume_voltage;
	unsigned int resume_capacity;

	unsigned int safety_time;
	
	unsigned int usb_safety_time;
	unsigned int ac_safety_time;
	
	unsigned int termination_current_time;
	struct delayed_work teoc_work;
	struct delayed_work termination_current_work;

	int stop_resume_check;
	int resume_count;
	struct delayed_work resume_work;

	unsigned int update_time;
	int stop_update;
	struct delayed_work update_heartbeat_work;

	
	struct delayed_work large_consume_work;
	struct delayed_work over_current_work;
	
	
	int battery_low;
	
	
	int large_consume_count;
	int large_consume_array[LARGE_CONSUME_NUMBER];
	int large_consume_time;
	
	
	int fast_update_time;
	int fast_update_count;
	
	struct mutex status_lock;
	enum msm_battery_status batt_status;
	struct msm_hardware_charger_priv *current_chg_priv;
	struct msm_hardware_charger_priv *current_mon_priv;

	unsigned int (*get_batt_capacity_percent) (void);

	struct msm_charger_event *queue;
	int tail;
	int head;
	spinlock_t queue_lock;
	int queue_count;
	struct work_struct queue_work;
	struct workqueue_struct *event_wq_thread;
	
	struct hrtimer vbus_draw_debounce_timer;
	int vbus_draw_debounce_time_ms;
	int usb_chg_current;
};

static struct msm_charger_mux msm_chg = {
	.vbus_draw_debounce_time_ms = 0,
	.usb_chg_current = 0,
};

static struct msm_battery_gauge *msm_batt_gauge;

static struct msm_hardware_charger_priv *usb_hw_chg_priv = NULL;


#define MSM_BATT_THERM_MAX_CHARGING_CELCIUS			480
#define MSM_BATT_THERM_MIN_CHARGING_CELCIUS			-20
#define MSM_BATT_THERM_MAX_DISCHARGING_CELCIUS		600
#define MSM_BATT_THERM_MIN_DISCHARGING_CELCIUS		-200
#define ENGINEER_MODE_BIT(x)  			(1<<(x))
extern unsigned int get_mfg_mode(void);
extern int msm_is_chg_plugged_in(void);
int in_pre_charge_stage = 1;
static struct mutex battery_low_lock;
struct msm_power_monitor{
 
 	
 	int battery_capacity;
	int battery_voltage;
	int battery_current;
	int battery_temperature;
	int gauge_firmware_version;


	
	int battery_remaining_capacity;
	int battery_full_capacity;
	int gauge_flags;

	
	int control_state;  
	
	char good_ocv[8];
	
	char wait_battery_id[8];
	
	char battery_detected[8];
	
	char charge_threshold1_reached[8];
	
	char charge_threshold_final_reached[8];
	
	
	char usb[15];
	char wall_charger[15];

	
	
	char battery_temp_too_high[15];
	char battery_temp_too_low[15];
	char battery_health[10];
	char is_charging[5];
	
	
 	char usb_current_limit[6];
	
	char usb_suspend[15];
	
	char current_limit_mode[15];
	
	char charging_enable[8];
	
	char chg_fault[10];
	
	char dc_pwr_ok[20];
	
	char chg_status[15];
	
 	
	int dfi_version;
	
 
};

static struct msm_hardware_charger *msm_engineer_mode_charger_data;
char *em_ptr;

 
static int vbus_draw_mA=0;
 
int msm_charger_type_monitor(void)
{
	struct msm_hardware_charger_priv *priv;
	int ret=0;
	priv = msm_chg.current_chg_priv;
	if(priv){
		if(priv->max_source_current>FAST_CHARGING_THRESHOLD_CURRENT_MA)
			ret = POWER_SUPPLY_TYPE_MAINS;
		else if(priv->max_source_current<=FAST_CHARGING_THRESHOLD_CURRENT_MA
			&&priv->max_source_current>0)
			ret = POWER_SUPPLY_TYPE_USB;
		else
			ret = POWER_SUPPLY_TYPE_USB;
	}
	else 
		ret = POWER_SUPPLY_TYPE_USB;
	return ret;
}
 
 
static int msm_chg_read_adc(int channel, int *mv_reading)
{
	int ret;
	void *h;
	struct adc_chan_result adc_chan_result;
	struct completion  conv_complete_evt;

	pr_debug("%s: called for %d\n", __func__, channel);
	ret = adc_channel_open(channel, &h);
	if (ret) {
		pr_err("%s: couldnt open channel %d ret=%d\n",
					__func__, channel, ret);
		goto out;
	}
	init_completion(&conv_complete_evt);
	ret = adc_channel_request_conv(h, &conv_complete_evt);
	if (ret) {
		pr_err("%s: couldnt request conv channel %d ret=%d\n",
						__func__, channel, ret);
		goto out;
	}
	wait_for_completion(&conv_complete_evt);
	ret = adc_channel_read_result(h, &adc_chan_result);
	if (ret) {
		pr_err("%s: couldnt read result channel %d ret=%d\n",
						__func__, channel, ret);
		goto out;
	}
	ret = adc_channel_close(h);
	if (ret) {
		pr_err("%s: couldnt close channel %d ret=%d\n",
						__func__, channel, ret);
	}
	if (mv_reading)
		*mv_reading = adc_chan_result.measurement;

	pr_debug("%s: done for %d\n", __func__, channel);
	return adc_chan_result.physical;
out:
	pr_debug("%s: done for %d\n", __func__, channel);
	return -EINVAL;

}

int msm_chg_get_vph_pwr_mvolts(void)
{
	int vph_pwr_mv;
	vph_pwr_mv = msm_chg_read_adc(CHANNEL_ADC_VPH_PWR, NULL);
	printk("msm_chg_get_vph_pwr_mvolts vph_pwr_mv=%d\n",vph_pwr_mv);
	return vph_pwr_mv;
}
EXPORT_SYMBOL(msm_chg_get_vph_pwr_mvolts);



extern struct pm8058_chip	*g_pm_chip_for_all;
#define PM8058_COIN_CELL_CHARGING_SWITCH 0x2F
int msm_chg_coin_cell_charging_switch(int on_off)
{
	unsigned char temp;
	int ret;
	
	
	if(!g_pm_chip_for_all) {
		printk("msm_chg_coin_cell_charging_switch, g_pm_chip_for_all is not initialized yet!! try later.\n");
		return -1;
	}

	ret = pm8058_read(g_pm_chip_for_all, PM8058_COIN_CELL_CHARGING_SWITCH, &temp, 1);
	if (ret)
		return ret;
	printk("msm_chg_coin_cell_charging_switch, temp=0x%x\n",temp);
	
	if (on_off)
		temp = 0x03;
	else
		temp = 0x00;
	printk("msm_chg_coin_cell_charging_switch, temp=0x%x\n",temp);

	return pm8058_write(g_pm_chip_for_all, PM8058_COIN_CELL_CHARGING_SWITCH, &temp, 1);
}
EXPORT_SYMBOL(msm_chg_coin_cell_charging_switch);

int msm_chg_coin_cell_get_voltage(void)
{
	int coin_cell_mv;
	coin_cell_mv = msm_chg_read_adc(CHANNEL_ADC_VCOIN, NULL);
	printk("msm_chg_coin_cell_get_voltage coin_cell_mv=%d\n",coin_cell_mv);
	return coin_cell_mv;
}
EXPORT_SYMBOL(msm_chg_coin_cell_get_voltage);



int msm_chg_get_xo_thermal(void)
{
	int xo_thermal;
	xo_thermal = msm_chg_read_adc(CHANNEL_ADC_XOTHERM, NULL);
	
	xo_thermal /=1000;
	printk(KERN_INFO "%s:xo_thermal=%d\n", __func__,xo_thermal);
	return xo_thermal;
}
int msm_chg_get_msm_thermal(void)
{
	int msm_thermal;
	msm_thermal = msm_chg_read_adc(CHANNEL_ADC_MSM_THERM, NULL);
	printk(KERN_INFO "%s:msm_thermal=%d\n", __func__,msm_thermal);
	return msm_thermal;
}

static int is_chg_capable_of_charging(struct msm_hardware_charger_priv *priv)
{
	if (priv->hw_chg_state == CHG_READY_STATE
	    || priv->hw_chg_state == CHG_CHARGING_STATE)
		return 1;

	return 0;
}

static int is_batt_status_capable_of_charging(void)
{
	if (msm_chg.batt_status == BATT_STATUS_ABSENT
	    || msm_chg.batt_status == BATT_STATUS_TEMPERATURE_OUT_OF_RANGE
	    || msm_chg.batt_status == BATT_STATUS_ID_INVALID
	    || msm_chg.batt_status == BATT_STATUS_JUST_FINISHED_CHARGING)
		return 0;
	return 1;
}

static int is_batt_status_charging(void)
{
	if (msm_chg.batt_status == BATT_STATUS_TRKL_CHARGING
	    || msm_chg.batt_status == BATT_STATUS_FAST_CHARGING
	       || msm_chg.batt_status == BATT_STATUS_TAPER_CHARGING)
		return 1;
	return 0;
}

static int is_battery_present(void)
{
	if (msm_batt_gauge && msm_batt_gauge->is_battery_present)
		return msm_batt_gauge->is_battery_present();
	else {
		pr_err("msm-charger: no batt gauge batt=absent\n");
		return 0;
	}
}

static int is_battery_temp_within_range(void)
{
	if (msm_batt_gauge && msm_batt_gauge->is_battery_temp_within_range)
		return msm_batt_gauge->is_battery_temp_within_range();
	else {
		pr_err("msm-charger no batt gauge batt=out_of_temperatur\n");
		return 0;
	}
}
static int is_discharging_battery_temp_within_range(void)
{
	if (msm_batt_gauge && msm_batt_gauge->is_discharging_battery_temp_within_range)
		return msm_batt_gauge->is_discharging_battery_temp_within_range();
	else {
		pr_err("msm-charger no batt gauge discharging batt=out_of_temperatur\n");
		return 0;
	}
}
static int is_battery_id_valid(void)
{
	if (msm_batt_gauge && msm_batt_gauge->is_battery_id_valid)
		return msm_batt_gauge->is_battery_id_valid();
	else {
		pr_err("msm-charger no batt gauge batt=id_invalid\n");
		return 0;
	}
}

static int get_prop_battery_mvolts(void)
{
	if (msm_batt_gauge && msm_batt_gauge->get_battery_mvolts)
		return msm_batt_gauge->get_battery_mvolts();
	else {
		pr_err("msm-charger no batt gauge assuming 3.5V\n");
		return MSM_CHARGER_GAUGE_MISSING_VOLTS;
	}
}

static int get_battery_temperature(void)
{
	if (msm_batt_gauge && msm_batt_gauge->get_battery_temperature)
		return msm_batt_gauge->get_battery_temperature();
	else {
		pr_err("msm-charger no batt gauge assuming 35 deg G\n");
		return MSM_CHARGER_GAUGE_MISSING_TEMP;
	}
}

static int get_prop_batt_capacity(void)
{
	if (msm_batt_gauge && msm_batt_gauge->get_batt_remaining_capacity)
		return msm_batt_gauge->get_batt_remaining_capacity();

	return msm_chg.get_batt_capacity_percent();
}

static int get_prop_batt_health(void)
{
	int status = 0;

	if (msm_chg.batt_status == BATT_STATUS_TEMPERATURE_OUT_OF_RANGE)
		status = POWER_SUPPLY_HEALTH_OVERHEAT;
	else if (msm_chg.batt_status == BATT_STATUS_CURRENT_OUT_OF_RANGE)
		status = POWER_SUPPLY_HEALTH_OVERCURRENT;
	else if(msm_chg.batt_status == BATT_STATUS_TIME_OUT)
		status = POWER_SUPPLY_HEALTH_TIMEOUT;
	else if(msm_chg.batt_status == BATT_STATUS_VOLTAGE_OUT_OF_RANGE)
		status = POWER_SUPPLY_HEALTH_OVERVOLTAGE;
	else
		status = POWER_SUPPLY_HEALTH_GOOD;

	return status;

}

static int get_prop_charge_type(void)
{
	int status = 0;

	if (msm_chg.batt_status == BATT_STATUS_TRKL_CHARGING)
		status = POWER_SUPPLY_CHARGE_TYPE_TRICKLE;
	else if (msm_chg.batt_status == BATT_STATUS_FAST_CHARGING)
		status = POWER_SUPPLY_CHARGE_TYPE_FAST;
	else
		status = POWER_SUPPLY_CHARGE_TYPE_NONE;

	return status;
}

static int get_prop_batt_status(void)
{
	int status = 0;

	if (msm_batt_gauge && msm_batt_gauge->get_battery_status) {
		status = msm_batt_gauge->get_battery_status();
		if (status == POWER_SUPPLY_STATUS_CHARGING ||
			status == POWER_SUPPLY_STATUS_FULL ||
			status == POWER_SUPPLY_STATUS_DISCHARGING)
			return status;
	}

	if (is_batt_status_charging())
		status = POWER_SUPPLY_STATUS_CHARGING;
	else if (msm_chg.batt_status ==
		 BATT_STATUS_JUST_FINISHED_CHARGING
			 && msm_chg.current_chg_priv != NULL)
		status = POWER_SUPPLY_STATUS_FULL;
	
	else if (msm_chg.batt_status ==
		 BATT_STATUS_TERMINATION_CHARGING
			 && msm_chg.current_chg_priv != NULL)
		status = POWER_SUPPLY_STATUS_FULL;
	else if (msm_chg.batt_status ==
		 BATT_STATUS_RECHARGING
			 && msm_chg.current_chg_priv != NULL)
		status = POWER_SUPPLY_STATUS_FULL;
	
	else if (msm_chg.batt_status ==
		 BATT_STATUS_EARLY_CHARGING
			 && msm_chg.current_chg_priv != NULL)
		status = POWER_SUPPLY_STATUS_NOT_CHARGING;
	else
		status = POWER_SUPPLY_STATUS_DISCHARGING;

	return status;
}

 /* This function should only be called within handle_event or resume */
static void update_batt_status(void)
{
	if (is_battery_present()) {
		if (is_battery_id_valid()) {
			if (msm_chg.batt_status == BATT_STATUS_ABSENT
				|| msm_chg.batt_status
					== BATT_STATUS_ID_INVALID) {
				msm_chg.stop_resume_check = 0;
				msm_chg.batt_status = BATT_STATUS_DISCHARGING;
			}
		} else
			msm_chg.batt_status = BATT_STATUS_ID_INVALID;
	 } else
		msm_chg.batt_status = BATT_STATUS_ABSENT;
}

 
 static int get_engineer_mode_data(void)
 {
 	struct msm_power_monitor power_monitor;	
	int val=0;
	
	char em_data_buf[sizeof(struct msm_power_monitor)+50];
	em_ptr=em_data_buf;
	
	memset(em_data_buf,0,sizeof(em_data_buf));
	
	power_monitor.battery_capacity = bq27520_get_battery_capacity();
	power_monitor.battery_voltage = get_prop_battery_mvolts();
	power_monitor.battery_current = bq27520_get_battery_current();
	power_monitor.battery_temperature = get_battery_temperature();
	power_monitor.gauge_firmware_version =bq27520_gauge_firmware();
	
	power_monitor.battery_remaining_capacity=bq27520_battery_remain_capacity();
	power_monitor.battery_full_capacity=bq27520_battery_full_charge_capacity();
	power_monitor.gauge_flags=bq27520_gauge_flags();
	power_monitor.control_state=bq27520_gauge_control_status();
	
	val=bq27520_gauge_flags();
	if(val&ENGINEER_MODE_BIT(5))
		strcpy(power_monitor.good_ocv,"true");
	else	
		strcpy(power_monitor.good_ocv,"false");
	if(val&ENGINEER_MODE_BIT(4))
		strcpy(power_monitor.wait_battery_id,"true");
	else
		strcpy(power_monitor.wait_battery_id,"false");
	if(val&ENGINEER_MODE_BIT(3))
		strcpy(power_monitor.battery_detected,"true");
	else	
		strcpy(power_monitor.battery_detected,"false");
	if(val&ENGINEER_MODE_BIT(2))
		strcpy(power_monitor.charge_threshold1_reached,"true");
	else	
		strcpy(power_monitor.charge_threshold1_reached,"false");
	if(val&ENGINEER_MODE_BIT(1))
		strcpy(power_monitor.charge_threshold_final_reached,"true");
	else	
		strcpy(power_monitor.charge_threshold_final_reached,"false");

	
	if(msm_engineer_mode_charger_data){
		if(msm_engineer_mode_charger_data->charger_type==CHARGER_TYPE_AC){
			strcpy(power_monitor.wall_charger,"detected");
			strcpy(power_monitor.usb,"no detected");
		}
		else if(msm_engineer_mode_charger_data->charger_type==CHARGER_TYPE_USB){
			strcpy(power_monitor.wall_charger,"no detected");
			strcpy(power_monitor.usb,"detected");
		}
		else{
			strcpy(power_monitor.wall_charger,"no detected");
			strcpy(power_monitor.usb,"no detected");
		}		
	}

	
	
	
	if(is_batt_status_charging()){
		if(is_battery_temp_within_range())
			strcpy(power_monitor.battery_health,"good");
		else
			strcpy(power_monitor.battery_health,"over heat");
		
		if(get_battery_temperature()>MSM_BATT_THERM_MAX_CHARGING_CELCIUS)
			strcpy(power_monitor.battery_temp_too_high,"too high");
		else
			strcpy(power_monitor.battery_temp_too_high,"not too high");
		if(get_battery_temperature()<MSM_BATT_THERM_MIN_CHARGING_CELCIUS)
			strcpy(power_monitor.battery_temp_too_low,"too low");
		else
			strcpy(power_monitor.battery_temp_too_low,"not too low");
	}
	else{
		if(is_discharging_battery_temp_within_range())
			strcpy(power_monitor.battery_health,"good");
		else
			strcpy(power_monitor.battery_health,"over heat");

		if(get_battery_temperature()>MSM_BATT_THERM_MAX_DISCHARGING_CELCIUS)
			strcpy(power_monitor.battery_temp_too_high,"too high");
		else
			strcpy(power_monitor.battery_temp_too_high,"not too high");
		if(get_battery_temperature()<MSM_BATT_THERM_MIN_DISCHARGING_CELCIUS)
			strcpy(power_monitor.battery_temp_too_low,"too low");
		else
			strcpy(power_monitor.battery_temp_too_low,"not too low");		
	}
	
	if(is_batt_status_charging())
		strcpy(power_monitor.is_charging,"yes");
	else
		strcpy(power_monitor.is_charging,"no");
	
	power_monitor.dfi_version = bq27520_version_DFI();
#ifdef CONFIG_MAX8903B_CHARGE
	if(is_batt_status_charging()){
		val=gpio_get_value_cansleep(CHG_IUSB);
		if(msm_engineer_mode_charger_data->charger_type==CHARGER_TYPE_USB){
			if(val)
				strcpy(power_monitor.usb_current_limit,"100mA");
			else
				strcpy(power_monitor.usb_current_limit,"500mA");
		}
		 
		if(msm_engineer_mode_charger_data->charger_type==CHARGER_TYPE_AC){
			strcpy(power_monitor.usb_current_limit,"NULL");
		}
		 
		val=gpio_get_value_cansleep(CHG_USUS);
		if(val)
			strcpy(power_monitor.usb_suspend,"suspended");
		else
			strcpy(power_monitor.usb_suspend,"no suspended");
		val=gpio_get_value_cansleep(CHG_DCM);	
		if(val)
			strcpy(power_monitor.current_limit_mode,"wall charger");
		else
			strcpy(power_monitor.current_limit_mode,"usb");
		val=gpio_get_value_cansleep(CHG_EN);
		if(val)
			strcpy(power_monitor.charging_enable,"enable");
		else
			strcpy(power_monitor.charging_enable,"disable");
		val=gpio_get_value_cansleep(CHG_FAULT);
		if(val)
			strcpy(power_monitor.chg_fault,"no fault");
		else
			strcpy(power_monitor.chg_fault,"fault");
		val=gpio_get_value_cansleep(DC_PWR_OK);
		if(val)
			strcpy(power_monitor.dc_pwr_ok,"no input detected");
		else
			strcpy(power_monitor.dc_pwr_ok,"input detected");

		val=gpio_get_value_cansleep(CHG_STAT);
		if(val)
			strcpy(power_monitor.chg_status,"error");
		else
			strcpy(power_monitor.chg_status,"fast-charging");
	}
	
	else{
		strcpy(power_monitor.usb_current_limit,"NULL");
		strcpy(power_monitor.usb_suspend,"NULL");
		strcpy(power_monitor.current_limit_mode,"NULL");
		strcpy(power_monitor.charging_enable,"NULL");
		strcpy(power_monitor.chg_fault,"NULL");
		strcpy(power_monitor.dc_pwr_ok,"NULL");
		strcpy(power_monitor.chg_status,"NULL");
	}
	
	  
	
#else
	strcpy(power_monitor.usb_current_limit,"NULL");
	strcpy(power_monitor.usb_suspend,"NULL");
	strcpy(power_monitor.current_limit_mode,"NULL");
	strcpy(power_monitor.charging_enable,"NULL");
	strcpy(power_monitor.chg_fault,"NULL");
	strcpy(power_monitor.dc_pwr_ok,"NULL");
	strcpy(power_monitor.chg_status,"NULL");
	
	  
#endif
	
	snprintf(em_data_buf, sizeof(em_data_buf), "%d,%d,%d,%d,0x%x,%d,%d,0x%x,0x%x,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%d",
		power_monitor.battery_capacity,
		power_monitor.battery_voltage,
		power_monitor.battery_current,
		power_monitor.battery_temperature,
		power_monitor.gauge_firmware_version,
		power_monitor.battery_remaining_capacity,
		power_monitor.battery_full_capacity,
		power_monitor.gauge_flags,
		power_monitor.control_state,
		power_monitor.good_ocv,
		power_monitor.wait_battery_id,
		power_monitor.battery_detected,
		power_monitor.charge_threshold1_reached,
		power_monitor.charge_threshold_final_reached,
		power_monitor.usb,
		power_monitor.wall_charger,
		power_monitor.battery_temp_too_high,
		power_monitor.battery_temp_too_low,
		power_monitor.battery_health,
		power_monitor.is_charging,
		power_monitor.usb_current_limit,
		power_monitor.usb_suspend,
		power_monitor.current_limit_mode,
		power_monitor.charging_enable,
		power_monitor.chg_fault,
		power_monitor.dc_pwr_ok,
		power_monitor.chg_status,
		power_monitor.dfi_version
		);

	return 0;
 }
 
static enum power_supply_property msm_power_props[] = {
	POWER_SUPPLY_PROP_PRESENT,
	POWER_SUPPLY_PROP_ONLINE,
	 
	POWER_SUPPLY_PROP_TYPE,
	 
};

static char *msm_power_supplied_to[] = {
	"battery",
};

static int msm_power_get_property(struct power_supply *psy,
				  enum power_supply_property psp,
				  union power_supply_propval *val)
{
	struct msm_hardware_charger_priv *priv;

	priv = container_of(psy, struct msm_hardware_charger_priv, psy);
	switch (psp) {
	case POWER_SUPPLY_PROP_PRESENT:
		val->intval = !(priv->hw_chg_state == CHG_ABSENT_STATE);
		break;
	case POWER_SUPPLY_PROP_ONLINE:
		val->intval = (priv->hw_chg_state == CHG_READY_STATE)
			|| (priv->hw_chg_state == CHG_CHARGING_STATE);
		break;
	 
	case POWER_SUPPLY_PROP_TYPE:
		break;
	 
	default:
		return -EINVAL;
	}
	return 0;
}

static enum power_supply_property msm_batt_power_props[] = {
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_CHARGE_TYPE,
	POWER_SUPPLY_PROP_HEALTH,
	POWER_SUPPLY_PROP_PRESENT,
	POWER_SUPPLY_PROP_TECHNOLOGY,
	POWER_SUPPLY_PROP_VOLTAGE_MAX_DESIGN,
	POWER_SUPPLY_PROP_VOLTAGE_MIN_DESIGN,
	POWER_SUPPLY_PROP_VOLTAGE_NOW,
	POWER_SUPPLY_PROP_CAPACITY,
	
	POWER_SUPPLY_PROP_CURRENT_NOW,
	POWER_SUPPLY_PROP_TEMP,
	 
 	
};

static enum power_supply_property msm_common_power_props[] = {
	
	POWER_SUPPLY_PROP_DFI_BUF,
	 
 	
	POWER_SUPPLY_PROP_CHARGER_TYPE_INSERTED,
	POWER_SUPPLY_PROP_CHARGER_PLUG_STATUS,
	POWER_SUPPLY_PROP_VPH_PWR,
	
	POWER_SUPPLY_PROP_COIN_VOLTAGE,
	
	POWER_SUPPLY_PROP_ENGINEER_MODE_DATA,
	
	 
	 POWER_SUPPLY_PROP_ENABLE_CHARGE,	
	 
	 
	POWER_SUPPLY_PROP_GAUGE_ID,	
	POWER_SUPPLY_PROP_CHARGER_ID,
	 
	
  	POWER_SUPPLY_PROP_COIN_CHG_ENABLE,
	
	
	 POWER_SUPPLY_PROP_IT_ENABLE,
	
	
	POWER_SUPPLY_PROP_XO_THERMAL,
	POWER_SUPPLY_PROP_MSM_THERMAL,
	
	
	POWER_SUPPLY_PROP_DFI_VERSION,
	
	
	 POWER_SUPPLY_PROP_IT_DISABLE,
	
	 
	POWER_SUPPLY_PROP_SOC_VERIFICATION,
	 
	 
	POWER_SUPPLY_PROP_INPUT_LIMIT_CURRENT,
	 
	POWER_SUPPLY_PROP_IT_CHECK,
};

static int msm_batt_power_get_property(struct power_supply *psy,
				       enum power_supply_property psp,
				       union power_supply_propval *val)
{
	switch (psp) {
	case POWER_SUPPLY_PROP_STATUS:
		val->intval = get_prop_batt_status();
		break;
	case POWER_SUPPLY_PROP_CHARGE_TYPE:
		val->intval = get_prop_charge_type();
		break;
	case POWER_SUPPLY_PROP_HEALTH:
		val->intval = get_prop_batt_health();
		break;
	case POWER_SUPPLY_PROP_PRESENT:
		val->intval = !(msm_chg.batt_status == BATT_STATUS_ABSENT);
		break;
	case POWER_SUPPLY_PROP_TECHNOLOGY:
		val->intval = POWER_SUPPLY_TECHNOLOGY_NiMH;
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_MAX_DESIGN:
		val->intval = msm_chg.max_voltage * 1000;
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_MIN_DESIGN:
		val->intval = msm_chg.min_voltage * 1000;
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_NOW:
		val->intval = get_prop_battery_mvolts();
		break;
	case POWER_SUPPLY_PROP_CAPACITY:
		val->intval = get_prop_batt_capacity();
		break;
	case POWER_SUPPLY_PROP_CURRENT_NOW:		
		val->intval = bq27520_get_battery_current();
		break;
	case POWER_SUPPLY_PROP_TEMP:
		val->intval = bq27520_get_battery_temperature();
		break;
	
	default:

		return -EINVAL;
	}
	return 0;
}

static int msm_common_power_get_property(struct power_supply *psy,
				       enum power_supply_property psp,
				       union power_supply_propval *val)
{
	switch (psp) {
	
	
	case POWER_SUPPLY_PROP_CHARGER_TYPE_INSERTED:
		if(msm_engineer_mode_charger_data)
			val->intval=msm_engineer_mode_charger_data->charger_type;
		break;
	case POWER_SUPPLY_PROP_CHARGER_PLUG_STATUS:
		if(msm_engineer_mode_charger_data)
			val->intval=msm_engineer_mode_charger_data->charger_plug_status;
		break;
	case POWER_SUPPLY_PROP_ENGINEER_MODE_DATA:
		get_engineer_mode_data();
		if(em_ptr)
			val->strval=em_ptr;
		break;
	
	 
	case POWER_SUPPLY_PROP_VPH_PWR:
		val->intval = msm_chg_get_vph_pwr_mvolts();
		break;
	 
	 
	 case POWER_SUPPLY_PROP_GAUGE_ID:
		val->intval = bq27520_get_gauge_ID();	
		break;
	 case POWER_SUPPLY_PROP_CHARGER_ID:
	 	if(usb_hw_chg_priv)
			val->intval = usb_hw_chg_priv->hw_chg->get_charger_IC_ID(usb_hw_chg_priv->hw_chg);	 		
	 	break;
	 

	
	 case POWER_SUPPLY_PROP_COIN_VOLTAGE:
		val->intval = msm_chg_coin_cell_get_voltage();
		break;
	
	
	case POWER_SUPPLY_PROP_XO_THERMAL:
		val->intval = msm_chg_get_xo_thermal();
		break;
	case POWER_SUPPLY_PROP_MSM_THERMAL:
		val->intval = msm_chg_get_msm_thermal();
		break;
	
	
	case POWER_SUPPLY_PROP_DFI_VERSION:
		if(1!=get_mfg_mode())
			return -EINVAL;
		val->intval = bq27520_version_DFI();
		break;
	
	 
	case POWER_SUPPLY_PROP_SOC_VERIFICATION:
		val->strval = bq27520_soc_verification();
		break;
	 
	case POWER_SUPPLY_PROP_IT_CHECK:
		if(1!=get_mfg_mode())
			return -EINVAL;
		val->intval = bq27520_check_impedance_track_algorithm();
		break;
	default:
		return -EINVAL;
		
	}
	return 0;
}


static int msm_start_charging_factory_test(void)
{
	int ret;
	struct msm_hardware_charger_priv *priv;
	unsigned int current_voltage;
	priv = msm_chg.current_chg_priv;
	 current_voltage=0;
	#ifdef CONFIG_SMB136_CHARGE
		
	   
   	  current_voltage = get_prop_battery_mvolts();	
 	if((current_voltage<FAST_CHARGING_THRESHOLD_VOLTAGE_MV)
		 	&&(priv->max_source_current>FAST_CHARGING_THRESHOLD_CURRENT_MA)){
		 priv->hw_chg->set_charging_current(priv->hw_chg,
					FAST_CHARGING_THRESHOLD_CURRENT_MA);
		 msm_chg.battery_low = 1;
	}
	else{
		 priv->hw_chg->set_charging_current(priv->hw_chg,
						FAST_CHARGING_THRESHOLD_CURRENT_MA*2);
		msm_chg.battery_low = 0;
	}
	  
   	  #endif

	ret = priv->hw_chg->start_charging(priv->hw_chg, msm_chg.max_voltage,
					 priv->max_source_current);
	if (ret) {
		dev_err(msm_chg.dev, "%s couldnt start chg error = %d\n",
			priv->hw_chg->name, ret);
	} 

	return ret;
}
static int msm_stop_charging_factory_test(struct msm_hardware_charger_priv *priv)
{
	return priv->hw_chg->stop_charging(priv->hw_chg);
}
static int msm_common_power_set_property(struct power_supply *psy,
				       enum power_supply_property psp,
				      const union power_supply_propval *val)
{
	int ret=0;
	int retval=0;
	switch (psp) {
	case POWER_SUPPLY_PROP_DFI_BUF:
		if(1!=get_mfg_mode())
			return -EINVAL;
				
		if (delayed_work_pending(&msm_chg.update_heartbeat_work) )
				cancel_delayed_work(&msm_chg.update_heartbeat_work);
		
		if (msm_batt_gauge && msm_batt_gauge->copy_dfi_from_user_to_kernel)
			ret=msm_batt_gauge->copy_dfi_from_user_to_kernel(val);
		queue_delayed_work(msm_chg.event_wq_thread,&msm_chg.update_heartbeat_work,
			      round_jiffies_relative(msecs_to_jiffies(msm_chg.update_time)));
		if(ret==0)
			retval= -EINVAL;
		break;
	
	case POWER_SUPPLY_PROP_ENABLE_CHARGE:	
				
		if((strncmp(val->strval, "1", 1) == 0)&&msm_chg.current_chg_priv){
			ret=msm_start_charging_factory_test();
			if(ret!=0)
				retval= -EINVAL;
			
			if (delayed_work_pending(&msm_chg.teoc_work) )
				cancel_delayed_work(&msm_chg.teoc_work);
			
		}
		
		else if((strncmp(val->strval, "0", 1) == 0)&&msm_chg.current_chg_priv){
			ret=msm_stop_charging_factory_test(msm_chg.current_chg_priv);
			if(ret!=0)
				retval= -EINVAL;
			
			if (delayed_work_pending(&msm_chg.teoc_work) )
				cancel_delayed_work(&msm_chg.teoc_work);
			
		}		
		break;
	 

	
	case POWER_SUPPLY_PROP_COIN_CHG_ENABLE:	
				
		if(strncmp(val->strval, "1", 1) == 0){
			ret=msm_chg_coin_cell_charging_switch(1);
			if(ret<0)
				retval= -EINVAL;
		}
		
		else if(strncmp(val->strval, "0", 1) == 0){
			ret=msm_chg_coin_cell_charging_switch(0);
			if(ret<0)
				retval= -EINVAL;
		}	
		break;
	
	
	 case POWER_SUPPLY_PROP_IT_ENABLE:
	 	if(1!=get_mfg_mode())
			return -EINVAL;
	 			
		if (delayed_work_pending(&msm_chg.update_heartbeat_work) )
				cancel_delayed_work(&msm_chg.update_heartbeat_work);
		
	 	if(strncmp(val->strval, "1", 1) == 0){
			ret=bq27520_enable_impedance_track_algorithm();
			if(ret<0)
				retval= -EINVAL;
		}
		queue_delayed_work(msm_chg.event_wq_thread,&msm_chg.update_heartbeat_work,
			      round_jiffies_relative(msecs_to_jiffies(msm_chg.update_time)));
		break;
		
	 
	case POWER_SUPPLY_PROP_IT_DISABLE:
		if(1!=get_mfg_mode())
			return -EINVAL;
				
		if (delayed_work_pending(&msm_chg.update_heartbeat_work) )
				cancel_delayed_work(&msm_chg.update_heartbeat_work);
		
		if(strncmp(val->strval, "0", 1) == 0){
			ret=bq27520_disable_impedance_track_algorithm();
			if(ret<0)
				retval= -EINVAL;
		}
		queue_delayed_work(msm_chg.event_wq_thread,&msm_chg.update_heartbeat_work,
			      round_jiffies_relative(msecs_to_jiffies(msm_chg.update_time)));
		break;
	 
	  
	 case POWER_SUPPLY_PROP_INPUT_LIMIT_CURRENT:
	 	if (delayed_work_pending(&msm_chg.update_heartbeat_work) )
			cancel_delayed_work(&msm_chg.update_heartbeat_work);
	 	if(msm_chg.current_chg_priv&&msm_chg.current_chg_priv->hw_chg){
			if(strncmp(val->strval, "0", 1) == 0){
				
				ret=msm_chg.current_chg_priv->hw_chg->set_charging_mode(msm_chg.current_chg_priv->hw_chg,0);
				if(ret!=0)
					retval= -EINVAL;
			}
		 	else if(strncmp(val->strval, "100", 3) == 0){	
				
				ret=msm_chg.current_chg_priv->hw_chg->set_charging_mode(msm_chg.current_chg_priv->hw_chg,100);
				if(ret!=0)
					retval= -EINVAL;
			}
			else if(strncmp(val->strval, "500", 3) == 0){	
				
				ret=msm_chg.current_chg_priv->hw_chg->set_charging_mode(msm_chg.current_chg_priv->hw_chg,500);
				if(ret!=0)
					retval= -EINVAL;
			}
			else if(strncmp(val->strval, "950", 3) == 0){	
				
				ret=msm_chg.current_chg_priv->hw_chg->set_charging_mode(msm_chg.current_chg_priv->hw_chg,950);
				if(ret!=0)
					retval= -EINVAL;
			}
			else
				retval= -EINVAL;
		}
	 	break;	
	  
	default:
		return -EINVAL;
	}
	return retval;
}


static struct power_supply msm_psy_batt = {
	.name = "battery",
	.type = POWER_SUPPLY_TYPE_BATTERY,
	.properties = msm_batt_power_props,
	.num_properties = ARRAY_SIZE(msm_batt_power_props),
	.get_property = msm_batt_power_get_property,
};

 

static struct power_supply msm_psy_common = {
	.name = "common",
	.type = POWER_SUPPLY_TYPE_BATTERY,
	.properties = msm_common_power_props,
	.num_properties = ARRAY_SIZE(msm_common_power_props),
	.get_property = msm_common_power_get_property,
	 
	.set_property = msm_common_power_set_property,
	
};

static int usb_chg_current;

static void (*notify_vbus_state_func_ptr)(int);
static int usb_notified_of_insertion;

static DEFINE_SPINLOCK(charger_usb_lock);
void _msm_charger_vbus_draw(unsigned int mA)
{
	printk(KERN_INFO "%s:_mA =%d\n", __func__,mA);
	if (usb_hw_chg_priv) {
		usb_hw_chg_priv->max_source_current = mA;
		msm_charger_notify_event(usb_hw_chg_priv->hw_chg,
						CHG_ENUMERATED_EVENT);
	}
	else
		pr_err("%s called early;charger isnt initialized\n", __func__);

}

/* this is passed to the hsusb via platform_data msm_otg_pdata */
int msm_charger_register_vbus_sn(void (*callback)(int))
{
	pr_debug(KERN_INFO "%s\n", __func__);
	notify_vbus_state_func_ptr = callback;
	return 0;
}

/* this is passed to the hsusb via platform_data msm_otg_pdata */
void msm_charger_unregister_vbus_sn(void (*callback)(int))
{
	pr_debug(KERN_INFO "%s\n", __func__);
	notify_vbus_state_func_ptr = NULL;
}

static void notify_usb_of_the_plugin_event(struct msm_hardware_charger_priv
					   *hw_chg, int plugin)
{
	plugin = !!plugin;
	if (plugin == 1 && usb_notified_of_insertion == 0) {
		usb_notified_of_insertion = 1;
		if (notify_vbus_state_func_ptr) {
			dev_dbg(msm_chg.dev, "%s notifying plugin\n", __func__);
			(*notify_vbus_state_func_ptr) (plugin);
		} else
			dev_dbg(msm_chg.dev, "%s unable to notify plugin\n",
				__func__);
		usb_hw_chg_priv = hw_chg;
	 
		if(vbus_draw_mA>0){
			_msm_charger_vbus_draw(vbus_draw_mA);			
		}
	
	}
	if (plugin == 0 && usb_notified_of_insertion == 1) {
		if (notify_vbus_state_func_ptr) {
			dev_dbg(msm_chg.dev, "%s notifying unplugin\n",
				__func__);
			(*notify_vbus_state_func_ptr) (plugin);
		} else
			dev_dbg(msm_chg.dev, "%s unable to notify unplugin\n",
				__func__);
		usb_notified_of_insertion = 0;
		usb_hw_chg_priv = NULL;
		
		spin_lock(&charger_usb_lock);
		vbus_draw_mA=0;
		spin_unlock(&charger_usb_lock);
	
	}
}

static unsigned int msm_chg_get_batt_capacity_percent(void)
{
	   
	#ifdef CONFIG_BQ27520_GAUGE_QSD
		int capacity_percent;
		capacity_percent = bq27520_get_battery_capacity();
		return capacity_percent;
	#else 	
	unsigned int current_voltage = get_prop_battery_mvolts();
	unsigned int low_voltage = msm_chg.min_voltage;
	unsigned int high_voltage = msm_chg.max_voltage;

	if (current_voltage <= low_voltage)
		return 0;
	else if (current_voltage >= high_voltage)
		return 100;
	else
		return (current_voltage - low_voltage) * 100
		    / (high_voltage - low_voltage);
	#endif
	  
}

#ifdef DEBUG
static inline void debug_print(const char *func,
			       struct msm_hardware_charger_priv *hw_chg_priv)
{
	dev_dbg(msm_chg.dev,
		"%s current=(%s)(s=%d)(r=%d) new=(%s)(s=%d)(r=%d) batt=%d En\n",
		func,
		msm_chg.current_chg_priv ? msm_chg.current_chg_priv->
		hw_chg->name : "none",
		msm_chg.current_chg_priv ? msm_chg.
		current_chg_priv->hw_chg_state : -1,
		msm_chg.current_chg_priv ? msm_chg.current_chg_priv->
		hw_chg->rating : -1,
		hw_chg_priv ? hw_chg_priv->hw_chg->name : "none",
		hw_chg_priv ? hw_chg_priv->hw_chg_state : -1,
		hw_chg_priv ? hw_chg_priv->hw_chg->rating : -1,
		msm_chg.batt_status);
}
#else
static inline void debug_print(const char *func,
			       struct msm_hardware_charger_priv *hw_chg_priv)
{
}
#endif

static struct msm_hardware_charger_priv *find_best_charger(void)
{
	struct msm_hardware_charger_priv *hw_chg_priv;
	struct msm_hardware_charger_priv *better;
	int rating;

	better = NULL;
	rating = 0;

	list_for_each_entry(hw_chg_priv, &msm_chg.msm_hardware_chargers, list) {
		if (is_chg_capable_of_charging(hw_chg_priv)) {
			if (hw_chg_priv->hw_chg->rating > rating) {
				rating = hw_chg_priv->hw_chg->rating;
				better = hw_chg_priv;
			}
		}
	}

	return better;
}

static int msm_charging_switched(struct msm_hardware_charger_priv *priv)
{
	int ret = 0;

	if (priv->hw_chg->charging_switched)
		ret = priv->hw_chg->charging_switched(priv->hw_chg);
	return ret;
}

static int msm_stop_charging(struct msm_hardware_charger_priv *priv)
{
	return priv->hw_chg->stop_charging(priv->hw_chg);
}

static void msm_enable_system_current(struct msm_hardware_charger_priv *priv)
{
	if (priv->hw_chg->start_system_current)
		priv->hw_chg->start_system_current(priv->hw_chg,
					 priv->max_source_current);
}

static void msm_disable_system_current(struct msm_hardware_charger_priv *priv)
{
	if (priv->hw_chg->stop_system_current)
		priv->hw_chg->stop_system_current(priv->hw_chg);
}

/* the best charger has been selected -start charging from current_chg_priv */
static int msm_start_charging(void)
{
	int ret;
	struct msm_hardware_charger_priv *priv;
	unsigned int current_voltage;
	priv = msm_chg.current_chg_priv;
	 current_voltage=0;
	 
	if(1==get_mfg_mode())
		return -1;
	

	
	if(msm_chg.current_chg_priv->max_source_current<=500
		&&msm_chg.current_chg_priv->max_source_current>0)
		msm_chg.safety_time = msm_chg.usb_safety_time;
	else if(msm_chg.current_chg_priv->max_source_current>500)
		msm_chg.safety_time = msm_chg.ac_safety_time;
	
	
	  
	if( (msm_chg.resume_capacity < get_prop_batt_capacity() ) 
		&&(msm_chg.resume_voltage < get_prop_battery_mvolts()) ){
		printk(KERN_INFO "%s:cpacity is 100,no need charge.\n", __func__);
		priv->hw_chg_state = CHG_CHARGING_STATE;	
		msm_charger_notify_event(priv->hw_chg,CHG_DONE_EVENT);
		return -1;
	}
	 	
	
	#ifdef CONFIG_SMB136_CHARGE
		
	   
   	  current_voltage = get_prop_battery_mvolts();	
 	if((current_voltage<FAST_CHARGING_THRESHOLD_VOLTAGE_MV)
		 	&&(priv->max_source_current>FAST_CHARGING_THRESHOLD_CURRENT_MA)){
		 priv->hw_chg->set_charging_current(priv->hw_chg,
					FAST_CHARGING_THRESHOLD_CURRENT_MA);
		 msm_chg.battery_low = 1;
	}
	else{
		 priv->hw_chg->set_charging_current(priv->hw_chg,
						FAST_CHARGING_THRESHOLD_CURRENT_MA*2);
		msm_chg.battery_low = 0;
	}
	  
   	  #endif

	ret = priv->hw_chg->start_charging(priv->hw_chg, msm_chg.max_voltage,
					 priv->max_source_current);
	if (ret) {
		dev_err(msm_chg.dev, "%s couldnt start chg error = %d\n",
			priv->hw_chg->name, ret);
	} else{
		priv->hw_chg_state = CHG_CHARGING_STATE;			
		
	}

	return ret;
}

static void handle_charging_done(struct msm_hardware_charger_priv *priv)
{
	 
	dev_info(msm_chg.dev, "%s: handler charging done.\n",__func__);
	if(priv)
		dev_info(msm_chg.dev, "%s: hw_chg->name=%s\n",
			__func__,priv->hw_chg->name);
	
	if (msm_chg.current_chg_priv == priv) {			
		if(msm_chg.batt_status == BATT_STATUS_JUST_FINISHED_CHARGING){
			
			if (msm_chg.current_chg_priv->hw_chg_state == CHG_CHARGING_STATE
				||msm_chg.current_chg_priv->hw_chg_state == CHG_READY_STATE)
				if (msm_stop_charging(msm_chg.current_chg_priv)) {
					dev_err(msm_chg.dev, "%s couldnt stop chg\n",
						msm_chg.current_chg_priv->hw_chg->name);
				}
			msm_chg.current_chg_priv->hw_chg_state = CHG_READY_STATE;

			msm_chg.batt_status = BATT_STATUS_TERMINATION_CHARGING;
			dev_info(msm_chg.dev, "%s: stopping safety timer work\n",
					__func__);
			
			if (delayed_work_pending(&msm_chg.teoc_work) )
				cancel_delayed_work(&msm_chg.teoc_work);
			
			if (delayed_work_pending(&msm_chg.termination_current_work) )
				cancel_delayed_work(&msm_chg.termination_current_work);
			dev_info(msm_chg.dev, "%s: starting resume timer work",
					__func__);
			queue_delayed_work(msm_chg.event_wq_thread,
						&msm_chg.resume_work,
					      round_jiffies_relative(msecs_to_jiffies
							     (RESUME_CHECK_PERIOD_MS)));			
		}
		else{
			
			msm_chg.current_chg_priv->hw_chg_state = CHG_READY_STATE;
			msm_chg.batt_status = BATT_STATUS_JUST_FINISHED_CHARGING;
			if (delayed_work_pending(&msm_chg.teoc_work) )
				cancel_delayed_work(&msm_chg.teoc_work);
			queue_delayed_work(msm_chg.event_wq_thread,
					&msm_chg.termination_current_work,
				      round_jiffies_relative(msecs_to_jiffies
							     (msm_chg.termination_current_time)));
		}
	}
}

static void teoc(struct work_struct *work)
{
	/* we have been charging too long - stop charging */
	dev_info(msm_chg.dev, "%s: safety timer work expired\n", __func__);

	mutex_lock(&msm_chg.status_lock);
	if (msm_chg.current_chg_priv != NULL
	    && msm_chg.current_chg_priv->hw_chg_state == CHG_CHARGING_STATE) {
	    	
	    	if (msm_stop_charging(msm_chg.current_chg_priv)) {
			dev_err(msm_chg.dev, "%s couldnt stop chg\n",
				msm_chg.current_chg_priv->hw_chg->name);
		}
		msm_chg.current_chg_priv->hw_chg_state = CHG_READY_STATE;
	    	msm_chg.batt_status = BATT_STATUS_DISCHARGING;		
	 
	 msm_charger_notify_event(msm_chg.current_chg_priv->hw_chg,CHG_BATT_TIME_OUT);
	  
	}
	mutex_unlock(&msm_chg.status_lock);
}

static void termination_current_timer_expired(struct work_struct *work)
{

	dev_info(msm_chg.dev, "%s: termination current timer work expired\n", __func__);

	mutex_lock(&msm_chg.status_lock);
	if (msm_chg.current_chg_priv != NULL
	    && msm_chg.current_chg_priv->hw_chg_state == CHG_READY_STATE) {
	    msm_chg.batt_status = BATT_STATUS_DISCHARGING;
	    
	   if (msm_stop_charging(msm_chg.current_chg_priv)) {
			dev_err(msm_chg.dev, "%s couldnt stop chg\n",
				msm_chg.current_chg_priv->hw_chg->name);
		}
		msm_chg.current_chg_priv->hw_chg_state = CHG_READY_STATE;
	    	msm_chg.batt_status = BATT_STATUS_DISCHARGING;	
	}
	mutex_unlock(&msm_chg.status_lock);
}
static void handle_battery_inserted(void)
{
	msm_chg.stop_resume_check = 0;
	/* if a charger is already present start charging */
	if (msm_chg.current_chg_priv != NULL &&
	    is_batt_status_capable_of_charging() &&
	    !is_batt_status_charging()) {
		if (msm_start_charging()) {
			dev_err(msm_chg.dev, "%s couldnt start chg\n",
				msm_chg.current_chg_priv->hw_chg->name);
			return;
		}
		msm_chg.batt_status = BATT_STATUS_TRKL_CHARGING;

		dev_info(msm_chg.dev, "%s: starting safety timer work\n",
				__func__);
		queue_delayed_work(msm_chg.event_wq_thread,
					&msm_chg.teoc_work,
				      round_jiffies_relative(msecs_to_jiffies
							     (msm_chg.
							      safety_time)));
	}
}

#define MSM_CHARGER_RESUME_COUNT 5
static void resume_charging(struct work_struct *work)
{
	dev_dbg(msm_chg.dev, "%s resuming charging %d", __func__,
						msm_chg.resume_count);

	if (msm_chg.stop_resume_check) {
		msm_chg.stop_resume_check = 0;
		pr_err("%s stopping resume", __func__);
		return;
	}

	update_batt_status();
	
	
	
	if (msm_chg.batt_status != BATT_STATUS_TERMINATION_CHARGING) {
		pr_err("%s called outside JFC state", __func__);
		return;
	}
	printk(KERN_INFO "%s: voltage=%d,capacity=%d.\n", __func__, get_prop_battery_mvolts(),get_prop_batt_capacity());
	if (get_prop_battery_mvolts() < msm_chg.resume_voltage
		||get_prop_batt_capacity() < msm_chg.resume_capacity)
		msm_chg.resume_count++;
	else
		msm_chg.resume_count = 0;

	
	if (get_prop_battery_mvolts() < msm_chg.min_voltage + 500) {
		pr_err("%s: batt lost voltage rapidly -force resume charging\n",
					__func__);
		msm_chg.resume_count += MSM_CHARGER_RESUME_COUNT + 1;
	}

	if (msm_chg.resume_count > MSM_CHARGER_RESUME_COUNT) {
		
		
		mutex_lock(&msm_chg.status_lock);
		msm_chg.batt_status = BATT_STATUS_DISCHARGING;
		msm_chg.resume_count = 0;
		if(0){
			handle_battery_inserted();
			power_supply_changed(&msm_psy_batt);
			if (msm_chg.current_chg_priv != NULL)
				power_supply_changed(&msm_chg.current_chg_priv->psy);
		}
		if(msm_chg.current_chg_priv!= NULL)
			msm_charger_notify_event(msm_chg.current_chg_priv->hw_chg,
							CHG_BATT_NEEDS_RECHARGING);
		mutex_unlock(&msm_chg.status_lock);
	} else {
		
		dev_info(msm_chg.dev, "%s: rescheduling resume timer work",
				__func__);
		queue_delayed_work(msm_chg.event_wq_thread,
					&msm_chg.resume_work,
				      round_jiffies_relative(msecs_to_jiffies
						     (RESUME_CHECK_PERIOD_MS)));
	}
}



static void handle_charger_over_current(void)
{
	
	if (msm_chg.current_chg_priv != NULL
	    && msm_chg.current_chg_priv->hw_chg_state == CHG_CHARGING_STATE) {
		if (msm_stop_charging(msm_chg.current_chg_priv)) {
			dev_err(msm_chg.dev, "%s couldnt stop chg",
				msm_chg.current_chg_priv->hw_chg->name);
		}
		msm_chg.current_chg_priv->hw_chg_state = CHG_READY_STATE;

		dev_info(msm_chg.dev, "%s: stopping safety timer work",
				__func__);
		
		if (delayed_work_pending(&msm_chg.teoc_work) )
			cancel_delayed_work(&msm_chg.teoc_work);
		
		if (delayed_work_pending(&msm_chg.termination_current_work) )
			cancel_delayed_work(&msm_chg.termination_current_work);
		if (delayed_work_pending(&msm_chg.large_consume_work) )
			cancel_delayed_work(&msm_chg.large_consume_work);
		
	}
	
	msm_chg.stop_resume_check = 1;
			
	if (delayed_work_pending(&msm_chg.resume_work) )
			cancel_delayed_work(&msm_chg.resume_work);
	
}
  

static void large_consume_work_fun(struct work_struct *work)
{		
	
	if (msm_chg.current_chg_priv != NULL 
		&&msm_chg.batt_status == BATT_STATUS_LARGE_CONSUME) {
		if (msm_start_charging()) {
			dev_err(msm_chg.dev, "%s couldnt start chg\n",
				msm_chg.current_chg_priv->hw_chg->name);
			return;
		}
		msm_chg.batt_status = BATT_STATUS_TRKL_CHARGING;

		dev_info(msm_chg.dev, "%s: large consume detection terminates,starting safety timer work\n",
			__func__);
		queue_delayed_work(msm_chg.event_wq_thread,
					&msm_chg.teoc_work,
				      round_jiffies_relative(msecs_to_jiffies
							     (msm_chg.
							      safety_time)));
	}
}

static void handle_battery_large_consume(void)
{	
	msm_chg.large_consume_count = 0;
	
	if (msm_chg.current_chg_priv != NULL
	    && msm_chg.current_chg_priv->hw_chg_state == CHG_CHARGING_STATE) {
		if (msm_stop_charging(msm_chg.current_chg_priv)) {
			dev_err(msm_chg.dev, "%s couldnt stop chg",
				msm_chg.current_chg_priv->hw_chg->name);
		}
		msm_chg.current_chg_priv->hw_chg_state = CHG_READY_STATE;

		dev_info(msm_chg.dev, "%s: large consume is detected,stopping safety timer work",
				__func__);
		queue_delayed_work(msm_chg.event_wq_thread,
					&msm_chg.large_consume_work,
				      round_jiffies_relative(msecs_to_jiffies
							     (msm_chg.large_consume_time)));
		
		if (delayed_work_pending(&msm_chg.teoc_work) )
			cancel_delayed_work(&msm_chg.teoc_work);
		
		if (delayed_work_pending(&msm_chg.termination_current_work) )
			cancel_delayed_work(&msm_chg.termination_current_work);
	}
	msm_chg.stop_resume_check = 1;
			
	if (delayed_work_pending(&msm_chg.resume_work) )
			cancel_delayed_work(&msm_chg.resume_work);
	
}

  
  
static void handle_charger_over_voltage(void)
{
	
	if (msm_chg.current_chg_priv != NULL
	    && msm_chg.current_chg_priv->hw_chg_state == CHG_CHARGING_STATE) {
		if (msm_stop_charging(msm_chg.current_chg_priv)) {
			dev_err(msm_chg.dev, "%s couldnt stop chg",
				msm_chg.current_chg_priv->hw_chg->name);
		}
		
		msm_chg.current_chg_priv->hw_chg_state = CHG_READY_STATE;
		

		dev_info(msm_chg.dev, "%s: stopping safety timer work",
				__func__);
		
		if (delayed_work_pending(&msm_chg.teoc_work) )
			cancel_delayed_work(&msm_chg.teoc_work);
		
		if (delayed_work_pending(&msm_chg.termination_current_work) )
			cancel_delayed_work(&msm_chg.termination_current_work);
		if (delayed_work_pending(&msm_chg.large_consume_work) )
			cancel_delayed_work(&msm_chg.large_consume_work);
	}
	
	msm_chg.stop_resume_check = 1;
			
	if (delayed_work_pending(&msm_chg.resume_work) )
			cancel_delayed_work(&msm_chg.resume_work);
	
}
  
static void handle_battery_removed(void)
{
	/* if a charger is charging the battery stop it */
	if (msm_chg.current_chg_priv != NULL
	    && msm_chg.current_chg_priv->hw_chg_state == CHG_CHARGING_STATE) {
		if (msm_stop_charging(msm_chg.current_chg_priv)) {
			dev_err(msm_chg.dev, "%s couldnt stop chg\n",
				msm_chg.current_chg_priv->hw_chg->name);
		}
		msm_chg.current_chg_priv->hw_chg_state = CHG_READY_STATE;

		dev_info(msm_chg.dev, "%s: stopping safety timer work\n",
				__func__);
		
		if (delayed_work_pending(&msm_chg.teoc_work) )
			cancel_delayed_work(&msm_chg.teoc_work);
		
		if (delayed_work_pending(&msm_chg.termination_current_work) )
			cancel_delayed_work(&msm_chg.termination_current_work);
		if (delayed_work_pending(&msm_chg.large_consume_work) )
			cancel_delayed_work(&msm_chg.large_consume_work);
	}
	msm_chg.stop_resume_check = 1;
			
	if (delayed_work_pending(&msm_chg.resume_work) )
			cancel_delayed_work(&msm_chg.resume_work);
	
}
#define CHG_MAX_CHG_CURRENT_FOR_BATTERY		1400
#define CHG_MIN_CHG_CURRENT_FOR_BATTERY		200
#define MSM_BATT_CHG_THERM_OPERATIONAL_MAX_CELCIUS 		480
#define MSM_BATT_DISCHG_THERM_OPERATIONAL_MAX_CELCIUS 		600
#define MSM_BATT_TERMINATION_CURRENT				50
static int msm_array_average(int *array , int size)
{
	int i;
	int vl_sum=0;

	for(i=0;i<size;i++){
		vl_sum+=array[i];
	}

	return(vl_sum/size);
}
static void msm_array_shift_element_to_last(int *array, int size, int last_element)
{
	int i;
	for (i=1;i<size;i++){
		array[i-1] = array[i];
	}
	array[size-1] = last_element;
} 
static int msm_large_consume_detection(int chg_current)
{
	
	int average = 500;
	int ret = 0;
		
	if(chg_current <CHG_MIN_CHG_CURRENT_FOR_BATTERY){
		if(msm_chg.large_consume_count>=LARGE_CONSUME_NUMBER){			
			
			msm_array_shift_element_to_last(msm_chg.large_consume_array,
				LARGE_CONSUME_NUMBER,chg_current);
			average = msm_array_average(msm_chg.large_consume_array,LARGE_CONSUME_NUMBER);
			if(average < CHG_MIN_CHG_CURRENT_FOR_BATTERY){
				printk(KERN_INFO "%s:large consume:average current=%d.\n", __func__,average);
				ret = 1;
			}
		}
		else{
			
			msm_chg.large_consume_array[msm_chg.large_consume_count] = chg_current;
			msm_chg.large_consume_count++;
		}
	}
	else	
		msm_chg.large_consume_count = 0;

	return ret;		
}
static void update_heartbeat(struct work_struct *work)
{
	int temperature;
	#ifdef CONFIG_SMB136_CHARGE
	unsigned int current_voltage;
	int temp_range;
	int chg_current;
	struct msm_hardware_charger_priv *priv;
	priv = msm_chg.current_chg_priv;
	
	if(1==get_mfg_mode())
		return;
	
	
	
	if((3 == get_mfg_mode())&&(1 == in_pre_charge_stage)){		
		if(get_prop_battery_mvolts()>MSM_BATT_EARLY_CHARGING_VOLTAGE){
			mutex_lock(&battery_low_lock);
		 	in_pre_charge_stage = 0;	
			mutex_unlock(&battery_low_lock);
		}
	}
	
	
	#ifdef CONFIG_HAS_WAKELOCK
	wake_lock(&heart_wl);
	#endif
	
	#endif
	if (msm_chg.batt_status == BATT_STATUS_ABSENT
		|| msm_chg.batt_status == BATT_STATUS_ID_INVALID) {
		if (is_battery_present())
			if (is_battery_id_valid()) {
				msm_chg.batt_status = BATT_STATUS_DISCHARGING;
				handle_battery_inserted();
			}
	} else {
		if (!is_battery_present()) {
			msm_chg.batt_status = BATT_STATUS_ABSENT;
			handle_battery_removed();
		}
		/*
		 * check battery id because a good battery could be removed
		 * and replaced with a invalid battery.
		 */
		if (!is_battery_id_valid()) {
			msm_chg.batt_status = BATT_STATUS_ID_INVALID;
			handle_battery_removed();
		}
	}
	pr_debug("msm-charger %s batt_status= %d\n",
				__func__, msm_chg.batt_status);

	if (msm_chg.current_chg_priv
		&& msm_chg.current_chg_priv->hw_chg_state
			== CHG_CHARGING_STATE) {
		temperature = get_battery_temperature();
		/* TODO implement JEITA SPEC*/
	}

#ifdef CONFIG_SMB136_CHARGE
		
	if(!priv&&is_battery_present()){
		temp_range = is_discharging_battery_temp_within_range();
		if(1!=temp_range&&msm_chg.batt_status != BATT_STATUS_TEMPERATURE_OUT_OF_RANGE){
			if(get_battery_temperature()>MSM_BATT_DISCHG_THERM_OPERATIONAL_MAX_CELCIUS){
				printk(KERN_INFO "discharging battery over temperature:%d\n",get_battery_temperature());
				msm_chg.batt_status = BATT_STATUS_TEMPERATURE_OUT_OF_RANGE;
			}
		}	
		else if(1==temp_range&&msm_chg.batt_status == BATT_STATUS_TEMPERATURE_OUT_OF_RANGE)
			msm_chg.batt_status = BATT_STATUS_DISCHARGING;
	}	
	
	
	   
	 current_voltage = get_prop_battery_mvolts();	
	if(is_batt_status_charging()){
		if(priv){
			 if((current_voltage<FAST_CHARGING_THRESHOLD_VOLTAGE_MV)
			 	&&(priv->max_source_current>FAST_CHARGING_THRESHOLD_CURRENT_MA)
			 	&&0==msm_chg.battery_low){
			 				 	
				 priv->hw_chg->set_charging_current(priv->hw_chg,
						FAST_CHARGING_THRESHOLD_CURRENT_MA);	
				msm_chg.battery_low = 1;
			 }
			 if((current_voltage>FAST_CHARGING_THRESHOLD_VOLTAGE_MV)
			 	&&(priv->max_source_current>FAST_CHARGING_THRESHOLD_CURRENT_MA)
			 	&&1==msm_chg.battery_low){
			 				 	
			 	priv->hw_chg->set_charging_current(priv->hw_chg,
						FAST_CHARGING_THRESHOLD_CURRENT_MA*2);
				msm_chg.battery_low = 0;
			 }
		}
	}
	    
   	  
	 
	if(priv){
		 
		
		
		 if(msm_chg.batt_status == BATT_STATUS_TRKL_CHARGING) {
		 	if(priv->hw_chg->fast_charging_event(priv->hw_chg))
				msm_charger_notify_event(priv->hw_chg,CHG_BATT_BEGIN_FAST_CHARGING);
		}
	
		 if(msm_chg.batt_status == BATT_STATUS_FAST_CHARGING) {
		 	if(priv->hw_chg->taper_charging_event(priv->hw_chg))
				msm_charger_notify_event(priv->hw_chg,CHG_BATT_BEGIN_TAPER_CHARGING);
		 }
		 
		
		
		
		dev_info(msm_chg.dev, "%s: msm_chg.batt_status(1)=%d,hw_chg_state=%d\n",__func__,
				msm_chg.batt_status,msm_chg.current_chg_priv->hw_chg_state);
		if(is_batt_status_charging()){
			if( (100==get_prop_batt_capacity())
				&&(msm_chg.resume_voltage < get_prop_battery_mvolts()) ){
				printk(KERN_INFO "%s:cpacity is 100,battery is full.\n", __func__);
				msm_charger_notify_event(priv->hw_chg,CHG_DONE_EVENT);
			}
		}
			
		if(msm_chg.batt_status == BATT_STATUS_TAPER_CHARGING){	
			if((priv->hw_chg->charging_done_event(priv->hw_chg))
				&&(bq27520_get_battery_current()<MSM_BATT_TERMINATION_CURRENT)
				&&(msm_chg.resume_capacity < get_prop_batt_capacity() )){
				printk(KERN_INFO "%s:cpacity is 100,and current< termation_current,battery is full.\n", __func__);
				msm_charger_notify_event(priv->hw_chg,CHG_DONE_EVENT);
			}
		}
		
		if(msm_chg.batt_status == BATT_STATUS_JUST_FINISHED_CHARGING||
			msm_chg.batt_status == BATT_STATUS_RECHARGING){
			chg_current = bq27520_get_battery_current();
			if(chg_current < MSM_BATT_TERMINATION_CURRENT){
				handle_charging_done(priv);				
			}
		}
		
	  	if (msm_chg.current_chg_priv->hw_chg_state== CHG_CHARGING_STATE||
			msm_chg.current_chg_priv->hw_chg_state == CHG_READY_STATE) {
			 
			if (is_battery_present()){
				
				temp_range = is_battery_temp_within_range();
				if(1!=temp_range&&msm_chg.batt_status != BATT_STATUS_TEMPERATURE_OUT_OF_RANGE){
					if(get_battery_temperature()>MSM_BATT_CHG_THERM_OPERATIONAL_MAX_CELCIUS){
						printk(KERN_INFO "battery over temperature:%d\n",get_battery_temperature());
						msm_charger_notify_event(priv->hw_chg,CHG_BATT_TEMP_OUTOFRANGE);
					}
				}				
				
				chg_current = bq27520_get_battery_current();
					if(is_battery_present()&&chg_current>CHG_MAX_CHG_CURRENT_FOR_BATTERY&&
					msm_chg.batt_status != BATT_STATUS_CURRENT_OUT_OF_RANGE){
					printk(KERN_INFO "over current=%d\n",chg_current);
					msm_charger_notify_event(priv->hw_chg,CHG_BATT_CURRENT_OUTOFRANGE);
				}
							
				
				 
				 
				 if(msm_chg.fast_update_count<= 0){
					 if(priv->max_source_current>FAST_CHARGING_THRESHOLD_CURRENT_MA
					 	&&msm_chg.batt_status == BATT_STATUS_FAST_CHARGING){					 	
						if(msm_large_consume_detection(chg_current)){
						 	msm_charger_notify_event(priv->hw_chg,CHG_BATT_LARGE_CONSUME);	
						}				
					}
		  		}
			}

		}
	}
	 	  

#endif

	/* notify that the voltage has changed
	 * the read of the capacity will trigger a
	 * voltage read*/
	power_supply_changed(&msm_psy_batt);

	if (msm_chg.stop_update) {
		msm_chg.stop_update = 0;
		return;
	}

	
	if(msm_chg.fast_update_count> 0){
		queue_delayed_work(msm_chg.event_wq_thread,
				&msm_chg.update_heartbeat_work,
			      round_jiffies_relative(msecs_to_jiffies
						     (msm_chg.fast_update_time)));
		msm_chg.fast_update_count--;
	}	
	else{	
		queue_delayed_work(msm_chg.event_wq_thread,
					&msm_chg.update_heartbeat_work,
				      round_jiffies_relative(msecs_to_jiffies
							     (msm_chg.update_time)));
	}
	dev_info(msm_chg.dev, "%s: msm_chg.batt_status(2)=%d\n",__func__,msm_chg.batt_status);
	
	
	
	#ifdef CONFIG_HAS_WAKELOCK
	wake_unlock(&heart_wl);
	#endif
	
}

/* set the charger state to READY before calling this */
static void handle_charger_ready(struct msm_hardware_charger_priv *hw_chg_priv)
{
	struct msm_hardware_charger_priv *old_chg_priv = NULL;

	debug_print(__func__, hw_chg_priv);
	 
	if (msm_chg.current_chg_priv != NULL
	    && hw_chg_priv->hw_chg->rating >
	    msm_chg.current_chg_priv->hw_chg->rating) {
		/*
		 * a better charger was found, ask the current charger
		 * to stop charging if it was charging
		 */
		if (msm_chg.current_chg_priv->hw_chg_state ==
		    CHG_CHARGING_STATE) {
			if (msm_stop_charging(msm_chg.current_chg_priv)) {
				dev_err(msm_chg.dev, "%s couldnt stop chg\n",
					msm_chg.current_chg_priv->hw_chg->name);
				return;
			}
			if (msm_charging_switched(msm_chg.current_chg_priv)) {
				dev_err(msm_chg.dev, "%s couldnt stop chg\n",
					msm_chg.current_chg_priv->hw_chg->name);
				return;
			}
		}
		msm_chg.current_chg_priv->hw_chg_state = CHG_READY_STATE;
		old_chg_priv = msm_chg.current_chg_priv;
		msm_chg.current_chg_priv = NULL;
	}

	if (msm_chg.current_chg_priv == NULL) {
		msm_chg.current_chg_priv = hw_chg_priv;
		dev_info(msm_chg.dev,
			 "%s: best charger = %s\n", __func__,
			 msm_chg.current_chg_priv->hw_chg->name);

		msm_enable_system_current(msm_chg.current_chg_priv);
		/*
		 * since a better charger was chosen, ask the old
		 * charger to stop providing system current
		 */
		if (old_chg_priv != NULL)
			msm_disable_system_current(old_chg_priv);
		
		  update_batt_status();
		
		 
		
		  #ifdef IGNORE_BAT_PARA_MONITOR_BEFORE_CHARGING
	
		{
			kernel_hw_info hw_info_kernel;
			int hw_type;
			hw_type = get_hw_info_in_kernel(&hw_info_kernel);			
		
			if(hw_info_kernel.board_version == BOARD_EVT2){
				msm_chg.batt_status = BATT_STATUS_DISCHARGING;
				printk("hardware type=%d\n",hw_info_kernel.board_version);
			}
		}
		#endif
		

		if (!is_batt_status_capable_of_charging())
			return;

		/* start charging from the new charger */
		if (!msm_start_charging()) {
			/* if we simply switched chg continue with teoc timer
			 * else we update the batt state and set the teoc
			 * timer */
			if (!is_batt_status_charging()) {
				dev_info(msm_chg.dev,
				       "%s: starting safety timer\n", __func__);
				queue_delayed_work(msm_chg.event_wq_thread,
							&msm_chg.teoc_work,
						      round_jiffies_relative
						      (msecs_to_jiffies
						       (msm_chg.safety_time)));
				msm_chg.batt_status = BATT_STATUS_TRKL_CHARGING;
			}
		} else {
			/* we couldnt start charging from the new readied
			 * charger */
			if (is_batt_status_charging())
				msm_chg.batt_status = BATT_STATUS_DISCHARGING;
		}
	}
}

static void handle_charger_removed(struct msm_hardware_charger_priv
				   *hw_chg_removed, int new_state)
{
	struct msm_hardware_charger_priv *hw_chg_priv;
	
	dev_info(msm_chg.dev, "%s:charger removed,new_state=%d\n", __func__,new_state);
	
	debug_print(__func__, hw_chg_removed);

	if (msm_chg.current_chg_priv == hw_chg_removed) {
		msm_disable_system_current(hw_chg_removed);
		
		if (msm_chg.current_chg_priv->hw_chg_state == CHG_CHARGING_STATE
			||msm_chg.current_chg_priv->hw_chg_state == CHG_READY_STATE) {
			if (msm_stop_charging(hw_chg_removed)) {
				dev_err(msm_chg.dev, "%s couldnt stop chg\n",
					msm_chg.current_chg_priv->hw_chg->name);
			}
		}
		msm_chg.current_chg_priv = NULL;
	}

	hw_chg_removed->hw_chg_state = new_state;

	if (msm_chg.current_chg_priv == NULL) {
		hw_chg_priv = find_best_charger();
		if (hw_chg_priv == NULL) {
			dev_info(msm_chg.dev, "%s: no chargers\n", __func__);
			/* if the battery was Just finished charging
			 * we keep that state as is so that we dont rush
			 * in to charging the battery when a charger is
			 * plugged in shortly. */
			if (is_batt_status_charging())
				msm_chg.batt_status = BATT_STATUS_DISCHARGING;
			
			if(msm_chg.batt_status == BATT_STATUS_VOLTAGE_OUT_OF_RANGE
				||msm_chg.batt_status == BATT_STATUS_CURRENT_OUT_OF_RANGE
				|| msm_chg.batt_status == BATT_STATUS_TEMPERATURE_OUT_OF_RANGE
				||msm_chg.batt_status == BATT_STATUS_TIME_OUT
				||msm_chg.batt_status == BATT_STATUS_LARGE_CONSUME
				||msm_chg.batt_status == BATT_STATUS_JUST_FINISHED_CHARGING
				||msm_chg.batt_status == BATT_STATUS_TERMINATION_CHARGING
				||msm_chg.batt_status == BATT_STATUS_RECHARGING)
				msm_chg.batt_status = BATT_STATUS_DISCHARGING;
			
		} else {
			msm_chg.current_chg_priv = hw_chg_priv;
			msm_enable_system_current(hw_chg_priv);
			dev_info(msm_chg.dev,
				 "%s: best charger = %s\n", __func__,
				 msm_chg.current_chg_priv->hw_chg->name);

			if (!is_batt_status_capable_of_charging())
				return;

			if (msm_start_charging()) {
				/* we couldnt start charging for some reason */
				msm_chg.batt_status = BATT_STATUS_DISCHARGING;
			}
		}
	}

	/* if we arent charging stop the safety timer */
	if (!is_batt_status_charging()) {
		dev_info(msm_chg.dev, "%s: stopping safety timer work\n",
				__func__);
		
		if (delayed_work_pending(&msm_chg.teoc_work) )
			cancel_delayed_work(&msm_chg.teoc_work);
				
		if (delayed_work_pending(&msm_chg.termination_current_work) )
			cancel_delayed_work(&msm_chg.termination_current_work);
		if (delayed_work_pending(&msm_chg.large_consume_work) )
			cancel_delayed_work(&msm_chg.large_consume_work);
	}
			
	if (delayed_work_pending(&msm_chg.resume_work) )
			cancel_delayed_work(&msm_chg.resume_work);
	
}

static void handle_event(struct msm_hardware_charger *hw_chg, int event)
{
	struct msm_hardware_charger_priv *priv = NULL;

	/*
	 * if hw_chg is NULL then this event comes from non-charger
	 * parties like battery gauge
	 */
	
	if (hw_chg)
		dev_info(msm_chg.dev, "%s %d from %s\n", __func__, event,hw_chg->name);
	else
		dev_info(msm_chg.dev, "%s %d \n", __func__, event);
	
	if (hw_chg)
		priv = hw_chg->charger_private;

	mutex_lock(&msm_chg.status_lock);

	switch (event) {
	case CHG_INSERTED_EVENT:
		if (priv->hw_chg_state != CHG_ABSENT_STATE) {
			dev_info(msm_chg.dev,
				 "%s insertion detected when cbl present",
				 hw_chg->name);
			break;
		}
		if(hw_chg)
			wake_lock(&priv->wl);
		update_batt_status();
		if (hw_chg->type == CHG_TYPE_USB) {
			priv->hw_chg_state = CHG_PRESENT_STATE;
			notify_usb_of_the_plugin_event(priv, 1);
			if (usb_chg_current) {
				priv->max_source_current = usb_chg_current;
				usb_chg_current = 0;
				/* usb has already indicated us to charge */
				priv->hw_chg_state = CHG_READY_STATE;
				handle_charger_ready(priv);
			}
		} else {
			priv->hw_chg_state = CHG_READY_STATE;
			handle_charger_ready(priv);
		}
		
		msm_chg.fast_update_count = FAST_UPDATE_BATTERY_PARA_COUNT;
		if (delayed_work_pending(&msm_chg.update_heartbeat_work) )
				cancel_delayed_work(&msm_chg.update_heartbeat_work);
		update_heartbeat((struct work_struct *)NULL);
		
		break;
	case CHG_ENUMERATED_EVENT:	/* only in USB types */
		if (!msm_is_chg_plugged_in()) {
			dev_info(msm_chg.dev, "%s enum withuot presence\n",
				 hw_chg->name);
			break;
		}
		update_batt_status();
		dev_dbg(msm_chg.dev, "%s enum with %dmA to draw\n",
			 hw_chg->name, priv->max_source_current);
		if (priv->max_source_current == 0) {
			/* usb subsystem doesnt want us to draw
			 * charging current */
			/* act as if the charge is removed */
			
			dev_info(msm_chg.dev, "%s priv->max_source_current is 0.\n", hw_chg->name);
			
			if (priv->hw_chg_state != CHG_PRESENT_STATE)
				handle_charger_removed(priv, CHG_PRESENT_STATE);
		} else {			
			if (priv->hw_chg_state != CHG_READY_STATE) {
				
				if(priv->hw_chg_state == CHG_CHARGING_STATE)
					handle_charger_removed(priv, CHG_PRESENT_STATE);
				
				priv->hw_chg_state = CHG_READY_STATE;
				handle_charger_ready(priv);
			}
		}
		break;
	case CHG_REMOVED_EVENT:
		if (priv->hw_chg_state == CHG_ABSENT_STATE) {
			dev_info(msm_chg.dev, "%s cable already removed\n",
				 hw_chg->name);
			break;
		}
		update_batt_status();
		if (hw_chg->type == CHG_TYPE_USB) {
			usb_chg_current = 0;
			notify_usb_of_the_plugin_event(priv, 0);
		}
		
		dev_info(msm_chg.dev, "%s cable removed\n", hw_chg->name);
		
		handle_charger_removed(priv, CHG_ABSENT_STATE);

		msm_chg.fast_update_count = FAST_UPDATE_BATTERY_PARA_COUNT;
		if (delayed_work_pending(&msm_chg.update_heartbeat_work) )
				cancel_delayed_work(&msm_chg.update_heartbeat_work);
		
		update_heartbeat((struct work_struct *)NULL);
		
		
		msleep(250);
		if(hw_chg)
			wake_unlock(&priv->wl);
		break;
	case CHG_DONE_EVENT:
		if (priv->hw_chg_state == CHG_CHARGING_STATE)
			handle_charging_done(priv);
		break;
	case CHG_BATT_BEGIN_FAST_CHARGING:
		/* only update if we are TRKL charging */
		if (msm_chg.batt_status == BATT_STATUS_TRKL_CHARGING)
			msm_chg.batt_status = BATT_STATUS_FAST_CHARGING;
		break;
	case CHG_BATT_NEEDS_RECHARGING:
		msm_chg.batt_status = BATT_STATUS_DISCHARGING;
		handle_battery_inserted();
		
		msm_chg.batt_status = BATT_STATUS_RECHARGING;
		priv = msm_chg.current_chg_priv;
		break;
	case CHG_BATT_TEMP_OUTOFRANGE:
		/* the batt_temp out of range can trigger
		 * when the battery is absent */
		if (!is_battery_present()
		    && msm_chg.batt_status != BATT_STATUS_ABSENT) {
			msm_chg.batt_status = BATT_STATUS_ABSENT;
			handle_battery_removed();
			break;
		}
		if (msm_chg.batt_status == BATT_STATUS_TEMPERATURE_OUT_OF_RANGE)
			break;
		msm_chg.batt_status = BATT_STATUS_TEMPERATURE_OUT_OF_RANGE;
		handle_battery_removed();
		break;
	case CHG_BATT_TEMP_INRANGE:
		if (msm_chg.batt_status != BATT_STATUS_TEMPERATURE_OUT_OF_RANGE)
			break;
		msm_chg.batt_status = BATT_STATUS_ID_INVALID;
		/* check id */
		if (!is_battery_id_valid())
			break;
		/* assume that we are discharging from the battery
		 * and act as if the battery was inserted
		 * if a charger is present charging will be resumed */
		msm_chg.batt_status = BATT_STATUS_DISCHARGING;
		handle_battery_inserted();
		break;
	case CHG_BATT_INSERTED:
		if (msm_chg.batt_status != BATT_STATUS_ABSENT)
			break;
		/* debounce */
		if (!is_battery_present())
			break;
		msm_chg.batt_status = BATT_STATUS_ID_INVALID;
		if (!is_battery_id_valid())
			break;
		/* assume that we are discharging from the battery */
		msm_chg.batt_status = BATT_STATUS_DISCHARGING;
		/* check if a charger is present */
		handle_battery_inserted();
		break;
	case CHG_BATT_REMOVED:
		if (msm_chg.batt_status == BATT_STATUS_ABSENT)
			break;
		/* debounce */
		if (is_battery_present())
			break;
		msm_chg.batt_status = BATT_STATUS_ABSENT;
		handle_battery_removed();
		break;
	
	case CHG_BATT_CURRENT_OUTOFRANGE:
		if (msm_chg.batt_status == BATT_STATUS_CURRENT_OUT_OF_RANGE)
			break;
		msm_chg.batt_status = BATT_STATUS_CURRENT_OUT_OF_RANGE;
		handle_charger_over_current();
		break;
	case CHG_BATT_CURRENT_INRANGE:
		if (msm_chg.batt_status != BATT_STATUS_CURRENT_OUT_OF_RANGE)
			break;
		msm_chg.batt_status = BATT_STATUS_ID_INVALID;
		
		if (!is_battery_id_valid())
			break;
		
		msm_chg.batt_status = BATT_STATUS_DISCHARGING;
		handle_battery_inserted();
		break;
	case CHG_BATT_BEGIN_TAPER_CHARGING:
		
		if (msm_chg.batt_status == BATT_STATUS_FAST_CHARGING)
			msm_chg.batt_status = BATT_STATUS_TAPER_CHARGING;
		break;
	case CHG_BATT_LARGE_CONSUME:
		if (msm_chg.batt_status == BATT_STATUS_LARGE_CONSUME)
			break;
		msm_chg.batt_status = BATT_STATUS_LARGE_CONSUME;
		handle_battery_large_consume();
		break;
	case CHG_BATT_NOT_LARGE_CONSUME:
		if (msm_chg.batt_status != BATT_STATUS_LARGE_CONSUME)
			break;
		msm_chg.batt_status = BATT_STATUS_ID_INVALID;
		
		if (!is_battery_id_valid())
			break;
		
		msm_chg.batt_status = BATT_STATUS_DISCHARGING;
		handle_battery_inserted();
		break;
	case CHG_BATT_TIME_OUT:
		if (msm_chg.batt_status == BATT_STATUS_TIME_OUT)
			break;
		msm_chg.batt_status = BATT_STATUS_TIME_OUT;
		break;
	case CHG_INPUT_OVER_VOLTAGE_EVENT:
		if (priv->hw_chg_state == CHG_ABSENT_STATE) {
			dev_info(msm_chg.dev, "%s cable already removed",
				 hw_chg->name);
			break;
		}
		msm_chg.batt_status = BATT_STATUS_VOLTAGE_OUT_OF_RANGE;
		handle_charger_over_voltage();
		break;
	case CHG_BATT_SOC_INT:		
		if (msm_chg.batt_status == BATT_STATUS_ABSENT)
			break;
		msm_chg.batt_status = BATT_STATUS_ABSENT;
		handle_battery_removed();
		
		break;
	
	case CHG_BATT_BAT_LOW:
		break;
	case CHG_BATT_STATUS_CHANGE:
		/* TODO  battery SOC like battery-alarm/charging-full features
		can be added here for future improvement */
		break;
	}
	dev_dbg(msm_chg.dev, "%s %d done batt_status=%d\n", __func__,
		event, msm_chg.batt_status);
	 
	 if(msm_chg.current_chg_priv)
	 	dev_info(msm_chg.dev, "%s batt_status=%d,chg_status=%d\n", __func__,
			msm_chg.batt_status,msm_chg.current_chg_priv->hw_chg_state);
	 

	/* update userspace */
	
	
	if (msm_batt_gauge)
		power_supply_changed(&msm_psy_batt);
	if (priv)
		power_supply_changed(&priv->psy);

	mutex_unlock(&msm_chg.status_lock);
}

static int msm_chg_dequeue_event(struct msm_charger_event **event)
{
	unsigned long flags;
	
	
	
	spin_lock_irqsave(&msm_chg.queue_lock, flags);
	if (msm_chg.queue_count == 0) {
		spin_unlock_irqrestore(&msm_chg.queue_lock, flags);
		return -EINVAL;
	}
	*event = &msm_chg.queue[msm_chg.head];
	msm_chg.head = (msm_chg.head + 1) % MSM_CHG_MAX_EVENTS;
	pr_debug("%s dequeueing %d\n", __func__, (*event)->event);
	msm_chg.queue_count--;
	spin_unlock_irqrestore(&msm_chg.queue_lock, flags);
	return 0;
}

static int msm_chg_enqueue_event(struct msm_hardware_charger *hw_chg,
			enum msm_hardware_charger_event event)
{
	unsigned long flags;
	
	
	
	spin_lock_irqsave(&msm_chg.queue_lock, flags);
	if (msm_chg.queue_count == MSM_CHG_MAX_EVENTS) {
		spin_unlock_irqrestore(&msm_chg.queue_lock, flags);
		pr_err("%s: queue full cannot enqueue %d\n",
				__func__, event);
		return -EAGAIN;
	}
	pr_debug("%s queueing %d\n", __func__, event);
	msm_chg.queue[msm_chg.tail].event = event;
	msm_chg.queue[msm_chg.tail].hw_chg = hw_chg;
	msm_chg.tail = (msm_chg.tail + 1)%MSM_CHG_MAX_EVENTS;
	msm_chg.queue_count++;
	spin_unlock_irqrestore(&msm_chg.queue_lock, flags);
	return 0;
}

static void process_events(struct work_struct *work)
{
	struct msm_charger_event *event;
	int rc;

	do {
		rc = msm_chg_dequeue_event(&event);
		if (!rc)
			handle_event(event->hw_chg, event->event);
	} while (!rc);
}

static enum hrtimer_restart vbus_draw_debounce_timer_handler(struct hrtimer *timer)
{
	int is_charger_present = 0;
	
	printk(KERN_INFO "%s: msm_chg.usb_chg_current in %d.\n", __func__, msm_chg.usb_chg_current);
	
	spin_lock(&charger_usb_lock);
	
	is_charger_present = msm_is_chg_plugged_in();
	
	if(!is_charger_present) {
		spin_unlock(&charger_usb_lock);
		return HRTIMER_NORESTART;
	}
	
	
	if(100 == msm_chg.usb_chg_current) {
		msm_chg.usb_chg_current = 500;
	}
	else if(0 == msm_chg.usb_chg_current) {
		msm_chg.usb_chg_current = 500;
	}
	printk(KERN_INFO "%s: msm_chg.usb_chg_current update to %d.\n", __func__, msm_chg.usb_chg_current);
	
	if(msm_chg.usb_chg_current>0)
		vbus_draw_mA = msm_chg.usb_chg_current;	

	
	if (usb_hw_chg_priv) {
		if(usb_hw_chg_priv->hw_chg) {
			usb_hw_chg_priv->max_source_current = msm_chg.usb_chg_current;
			msm_charger_notify_event(usb_hw_chg_priv->hw_chg,
						CHG_ENUMERATED_EVENT);
			
			spin_unlock(&charger_usb_lock);
			return HRTIMER_NORESTART;
		}
	}
	
	usb_chg_current = msm_chg.usb_chg_current;
	spin_unlock(&charger_usb_lock);

	return HRTIMER_NORESTART;
}


void msm_charger_vbus_draw(unsigned int mA)
{

	printk(KERN_INFO "%s:mA =%d\n", __func__,mA);
	
	spin_lock(&charger_usb_lock);
	if(mA>0)
		vbus_draw_mA=mA;
	if(msm_chg.vbus_draw_debounce_time_ms) {
		hrtimer_cancel(&msm_chg.vbus_draw_debounce_timer);
		
		msm_chg.usb_chg_current = mA;
		hrtimer_start(&msm_chg.vbus_draw_debounce_timer,
				ktime_set(msm_chg.vbus_draw_debounce_time_ms / 1000,
				(msm_chg.vbus_draw_debounce_time_ms % 1000) * 1000000),
				HRTIMER_MODE_REL);
		printk(KERN_INFO "%s timer started, %d ms\n", __func__,msm_chg.vbus_draw_debounce_time_ms);
	}
	else {	
		usb_chg_current = mA;
	}
	spin_unlock(&charger_usb_lock);
}

static int __init determine_initial_batt_status(void)
{
	int rc;

	if (is_battery_present())
		if (is_battery_id_valid())
			
			/*
			if (is_battery_temp_within_range())
				msm_chg.batt_status = BATT_STATUS_DISCHARGING;
			else
				msm_chg.batt_status
				    = BATT_STATUS_TEMPERATURE_OUT_OF_RANGE;
			*/
			if(is_discharging_battery_temp_within_range())
				msm_chg.batt_status = BATT_STATUS_DISCHARGING;
			else{
				if(get_battery_temperature()>MSM_BATT_DISCHG_THERM_OPERATIONAL_MAX_CELCIUS){
					printk(KERN_INFO "%s:discharging battery over temperature:%d.\n", __func__,
						get_battery_temperature());
					msm_chg.batt_status = BATT_STATUS_TEMPERATURE_OUT_OF_RANGE;
				}
			}
		else
			msm_chg.batt_status = BATT_STATUS_ID_INVALID;
	else
		msm_chg.batt_status = BATT_STATUS_ABSENT;

	if (is_batt_status_capable_of_charging())
		handle_battery_inserted();

	rc = power_supply_register(msm_chg.dev, &msm_psy_batt);
	if (rc < 0) {
		dev_err(msm_chg.dev, "%s: power_supply_register failed"
			" rc=%d\n", __func__, rc);
		return rc;
	}
	
	
	rc = power_supply_register(msm_chg.dev, &msm_psy_common);
	if (rc < 0) {
		dev_err(msm_chg.dev, "%s: %s power_supply_register failed"
			" rc=%d\n", __func__, msm_psy_common.name,rc);
		return rc;
	}
	

	/* start updaing the battery powersupply every msm_chg.update_time
		* milliseconds */
	queue_delayed_work(msm_chg.event_wq_thread,
				&msm_chg.update_heartbeat_work,
			      round_jiffies_relative(msecs_to_jiffies
						     (msm_chg.update_time)));

	pr_debug("%s:OK batt_status=%d\n", __func__, msm_chg.batt_status);
	return 0;
}

static int __devinit msm_charger_probe(struct platform_device *pdev)
{
	msm_chg.dev = &pdev->dev;
	if (pdev->dev.platform_data) {
		unsigned int milli_secs;

		struct msm_charger_platform_data *pdata
		    =
		    (struct msm_charger_platform_data *)pdev->dev.platform_data;
		
		msm_chg.usb_safety_time =  pdata->usb_safety_time * 60 * MSEC_PER_SEC;
		msm_chg.ac_safety_time =  pdata->ac_safety_time * 60 * MSEC_PER_SEC;
		milli_secs = pdata->usb_safety_time * 60 * MSEC_PER_SEC;
		
		if (milli_secs > jiffies_to_msecs(MAX_JIFFY_OFFSET)) {
			dev_warn(&pdev->dev, "%s: safety time too large"
				 "%dms\n", __func__, milli_secs);
			milli_secs = jiffies_to_msecs(MAX_JIFFY_OFFSET);
		}
		msm_chg.safety_time = milli_secs;

		milli_secs = pdata->update_time * 60 * MSEC_PER_SEC;
		if (milli_secs > jiffies_to_msecs(MAX_JIFFY_OFFSET)) {
			dev_warn(&pdev->dev, "%s: safety time too large"
				 "%dms\n", __func__, milli_secs);
			milli_secs = jiffies_to_msecs(MAX_JIFFY_OFFSET);
		}
		msm_chg.termination_current_time = pdata->termination_current_time * 60 * MSEC_PER_SEC;
		msm_chg.update_time = milli_secs;

		msm_chg.max_voltage = pdata->max_voltage;
		msm_chg.min_voltage = pdata->min_voltage;
		msm_chg.resume_voltage = pdata->resume_voltage;
		msm_chg.get_batt_capacity_percent =
		    pdata->get_batt_capacity_percent;
	}
	if (msm_chg.safety_time == 0)
		msm_chg.safety_time = CHARGING_TEOC_MS;
	if (msm_chg.update_time == 0)
		msm_chg.update_time = UPDATE_TIME_MS;
	if (msm_chg.max_voltage == 0)
		msm_chg.max_voltage = DEFAULT_BATT_MAX_V;
	if (msm_chg.min_voltage == 0)
		msm_chg.min_voltage = DEFAULT_BATT_MIN_V;
	if (msm_chg.resume_voltage == 0)
		msm_chg.resume_voltage = DEFAULT_BATT_RESUME_V;
	if (msm_chg.resume_capacity == 0)
		msm_chg.resume_capacity = DEFAULT_BATT_RESUME_C;
	if (msm_chg.get_batt_capacity_percent == NULL)
		msm_chg.get_batt_capacity_percent =
			msm_chg_get_batt_capacity_percent;

	mutex_init(&msm_chg.status_lock);
	mutex_init(&battery_low_lock);
	INIT_DELAYED_WORK(&msm_chg.teoc_work, teoc);
	INIT_DELAYED_WORK(&msm_chg.termination_current_work, termination_current_timer_expired);
	INIT_DELAYED_WORK(&msm_chg.resume_work, resume_charging);
	INIT_DELAYED_WORK(&msm_chg.update_heartbeat_work, update_heartbeat);
	INIT_DELAYED_WORK(&msm_chg.large_consume_work, large_consume_work_fun);
	
	
	msm_chg.battery_low = 0;
	
	msm_chg.large_consume_count = 0;
	
	msm_chg.fast_update_time = FAST_UPDATE_BATTERY_PARA_TIME * MSEC_PER_SEC;
	msm_chg.fast_update_count = 0;
	
	msm_chg.large_consume_time =LARGE_CONSUME_SAFETY_TIME * 60 * MSEC_PER_SEC;
	hrtimer_init(&msm_chg.vbus_draw_debounce_timer, CLOCK_MONOTONIC, HRTIMER_MODE_REL);
	msm_chg.vbus_draw_debounce_timer.function = vbus_draw_debounce_timer_handler;
	msm_chg.vbus_draw_debounce_time_ms = 1500;

	return 0;
}

static int __devexit msm_charger_remove(struct platform_device *pdev)
{
	mutex_destroy(&msm_chg.status_lock);
	mutex_destroy(&battery_low_lock);
	power_supply_unregister(&msm_psy_batt);
	return 0;
}

int msm_charger_notify_event(struct msm_hardware_charger *hw_chg,
			     enum msm_hardware_charger_event event)
{
	msm_chg_enqueue_event(hw_chg, event);
	queue_work(msm_chg.event_wq_thread, &msm_chg.queue_work);
	return 0;
}
EXPORT_SYMBOL(msm_charger_notify_event);

int msm_charger_register(struct msm_hardware_charger *hw_chg)
{
	struct msm_hardware_charger_priv *priv;
	int rc = 0;

	if (!msm_chg.inited) {
		pr_err("%s: msm_chg is NULL,Too early to register\n", __func__);
		return -EAGAIN;
	}

	if (hw_chg->start_charging == NULL
		|| hw_chg->stop_charging == NULL
		|| hw_chg->name == NULL
		|| hw_chg->rating == 0) {
		pr_err("%s: invalid hw_chg\n", __func__);
		return -EINVAL;
	}

	priv = kzalloc(sizeof *priv, GFP_KERNEL);
	if (priv == NULL) {
		dev_err(msm_chg.dev, "%s kzalloc failed\n", __func__);
		return -ENOMEM;
	}

	priv->psy.name = hw_chg->name;
	if (hw_chg->type == CHG_TYPE_USB)
		priv->psy.type = POWER_SUPPLY_TYPE_USB;
	else
		priv->psy.type = POWER_SUPPLY_TYPE_MAINS;

	priv->psy.supplied_to = msm_power_supplied_to;
	priv->psy.num_supplicants = ARRAY_SIZE(msm_power_supplied_to);
	priv->psy.properties = msm_power_props;
	priv->psy.num_properties = ARRAY_SIZE(msm_power_props);
	priv->psy.get_property = msm_power_get_property;

#ifdef CONFIG_HAS_WAKELOCK
	wake_lock_init(&priv->wl, WAKE_LOCK_SUSPEND, priv->psy.name);
#endif
	rc = power_supply_register(NULL, &priv->psy);
	if (rc) {
		dev_err(msm_chg.dev, "%s power_supply_register failed\n",
			__func__);
		goto out;
	}
	
	msm_engineer_mode_charger_data=hw_chg;
	
	priv->hw_chg = hw_chg;
	priv->hw_chg_state = CHG_ABSENT_STATE;
	INIT_LIST_HEAD(&priv->list);
	mutex_lock(&msm_chg.msm_hardware_chargers_lock);
	list_add_tail(&priv->list, &msm_chg.msm_hardware_chargers);
	mutex_unlock(&msm_chg.msm_hardware_chargers_lock);
	hw_chg->charger_private = (void *)priv;
	return 0;

out:
	wake_lock_destroy(&priv->wl);
	kfree(priv);
	return rc;
}
EXPORT_SYMBOL(msm_charger_register);

void msm_battery_gauge_register(struct msm_battery_gauge *batt_gauge)
{
	if (msm_batt_gauge) {
		msm_batt_gauge = batt_gauge;
		pr_err("msm-charger %s multiple battery gauge called\n",
								__func__);
	} else {
		msm_batt_gauge = batt_gauge;
		determine_initial_batt_status();
	}
}
EXPORT_SYMBOL(msm_battery_gauge_register);

void msm_battery_gauge_unregister(struct msm_battery_gauge *batt_gauge)
{
	msm_batt_gauge = NULL;
}
EXPORT_SYMBOL(msm_battery_gauge_unregister);

int msm_charger_unregister(struct msm_hardware_charger *hw_chg)
{
	struct msm_hardware_charger_priv *priv;

	priv = (struct msm_hardware_charger_priv *)(hw_chg->charger_private);
	mutex_lock(&msm_chg.msm_hardware_chargers_lock);
	list_del(&priv->list);
	mutex_unlock(&msm_chg.msm_hardware_chargers_lock);
	wake_lock_destroy(&priv->wl);
	power_supply_unregister(&priv->psy);
	kfree(priv);
	return 0;
}
EXPORT_SYMBOL(msm_charger_unregister);

static int msm_charger_suspend(struct device *dev)
{
	dev_dbg(msm_chg.dev, "%s suspended\n", __func__);
	msm_chg.stop_update = 1;
	
	cancel_delayed_work(&msm_chg.update_heartbeat_work);
	mutex_lock(&msm_chg.status_lock);
	handle_battery_removed();
	mutex_unlock(&msm_chg.status_lock);
	return 0;
}

static int msm_charger_resume(struct device *dev)
{
	dev_dbg(msm_chg.dev, "%s resumed\n", __func__);
	msm_chg.stop_update = 0;
	/* start updaing the battery powersupply every msm_chg.update_time
	 * milliseconds */
	
	queue_delayed_work(msm_chg.event_wq_thread,
				&msm_chg.update_heartbeat_work,
			      round_jiffies_relative(msecs_to_jiffies
						     (1* MSEC_PER_SEC)));
	mutex_lock(&msm_chg.status_lock);
	handle_battery_inserted();
	mutex_unlock(&msm_chg.status_lock);
	return 0;
}

static SIMPLE_DEV_PM_OPS(msm_charger_pm_ops,
		msm_charger_suspend, msm_charger_resume);

static struct platform_driver msm_charger_driver = {
	.probe = msm_charger_probe,
	.remove = __devexit_p(msm_charger_remove),
	.driver = {
		   .name = "msm-charger",
		   .owner = THIS_MODULE,
		   .pm = &msm_charger_pm_ops,
	},
};

static int __init msm_charger_init(void)
{
	int rc;

	INIT_LIST_HEAD(&msm_chg.msm_hardware_chargers);
	msm_chg.count_chargers = 0;
	mutex_init(&msm_chg.msm_hardware_chargers_lock);

	msm_chg.queue = kzalloc(sizeof(struct msm_charger_event)
				* MSM_CHG_MAX_EVENTS,
				GFP_KERNEL);
	if (!msm_chg.queue) {
		rc = -ENOMEM;
		goto out;
	}
	msm_chg.tail = 0;
	msm_chg.head = 0;
	spin_lock_init(&msm_chg.queue_lock);
	msm_chg.queue_count = 0;
	INIT_WORK(&msm_chg.queue_work, process_events);
	msm_chg.event_wq_thread = create_workqueue("msm_charger_eventd");
	if (!msm_chg.event_wq_thread) {
		rc = -ENOMEM;
		goto free_queue;
	}
	rc = platform_driver_register(&msm_charger_driver);
	if (rc < 0) {
		pr_err("%s: FAIL: platform_driver_register. rc = %d\n",
		       __func__, rc);
		goto destroy_wq_thread;
	}
	msm_chg.inited = 1;

#ifdef CONFIG_HAS_WAKELOCK
	wake_lock_init(&heart_wl, WAKE_LOCK_SUSPEND, "heart_beat");
#endif

	return 0;

destroy_wq_thread:
	destroy_workqueue(msm_chg.event_wq_thread);
free_queue:
	kfree(msm_chg.queue);
out:
	return rc;
}

static void __exit msm_charger_exit(void)
{
	flush_workqueue(msm_chg.event_wq_thread);
	destroy_workqueue(msm_chg.event_wq_thread);
	kfree(msm_chg.queue);
#ifdef CONFIG_HAS_WAKELOCK
	wake_lock_destroy(&heart_wl);
#endif
	platform_driver_unregister(&msm_charger_driver);
}

module_init(msm_charger_init);
module_exit(msm_charger_exit);

MODULE_LICENSE("GPL v2");
MODULE_AUTHOR("Abhijeet Dharmapurikar <adharmap@codeaurora.org>");
MODULE_DESCRIPTION("Battery driver for Qualcomm MSM chipsets.");
MODULE_VERSION("1.0");
