/* Copyright (c) 2010 Code Aurora Forum. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 *
 */

#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/module.h>
#include <linux/debugfs.h>
#include <linux/workqueue.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/i2c/smb136_charge.h>
#include <linux/power_supply.h>
#include <linux/gpio/gpio_def.h>
#include <linux/mfd/pmic8058.h>
#include <mach/msm_hsusb.h>
#include <linux/pmic8058-charger.h>

#include <linux/reboot.h>



#include <linux/msm-charger.h>

 #include <linux/gpio.h>
 #include <linux/gpio/gpio_def.h>
#ifdef CONFIG_HAS_WAKELOCK
#include <linux/wakelock.h>
#endif


#define SMB136_MASK(BITS, POS)  		((unsigned char)(((1 << BITS) - 1) << POS))

#define CHG_CURRENT_REG			0x00
#define FAST_CHG_CURRENT_MASK		SMB136_MASK(3, 5)
#define PRE_CHG_CURRENT_MASK			SMB136_MASK(2, 3)
#define TERM_CHG_CURRENT_MASK		SMB136_MASK(2, 1)

#define FAST_CHG_CURRENT_LOW_BATTERY	0X0F
#define FAST_CHG_CURRENT_NOT_LOW_BATTERY	0XAF

#define INPUT_CURRENT_LIMIT_REG		0x01
#define IN_CURRENT_MASK			SMB136_MASK(3, 5)
#define IN_CURRENT_LIMIT_EN_BIT		BIT(2)
#define IN_CURRENT_DET_THRESH_MASK		SMB136_MASK(2, 0)

#define FLOAT_VOLTAGE_REG			0x02
#define STAT_OUT_POLARITY_BIT			BIT(7)
#define FLOAT_VOLTAGE				0xCD 	
#define FLOAT_VOLTAGE_MASK			SMB136_MASK(7, 0)

#define CONTROL_A_REG				0x03
#define AUTO_RECHARGE_DIS_BIT			BIT(7)
#define CURR_CYCLE_TERM_BIT			BIT(6)
#define PRE_TO_FAST_V_MASK			SMB136_MASK(3, 3)
#define TEMP_BEHAV_BIT				BIT(2)
#define THERM_NTC_CURR_MODE_BIT		BIT(1)
#define THERM_NTC_47KOHM_BIT			BIT(0)

#define CONTROL_B_REG				0x04
#define STAT_OUTPUT_MODE_MASK		SMB136_MASK(2, 6)
#define BATT_OV_ENDS_CYCLE_BIT		BIT(5)
#define AUTO_PRE_TO_FAST_DIS_BIT		BIT(4)
#define SAFETY_TIMER_EN_BIT			BIT(3)
#define OTG_LBR_WD_EN_BIT			BIT(2)
#define CHG_WD_TIMER_EN_BIT			BIT(1)
#define IRQ_OP_MASK				BIT(0)

#define PIN_CTRL_REG				0x05
#define AUTO_CHG_EN_BIT			BIT(7)
#define AUTO_LBR_EN_BIT			BIT(6)
#define OTG_LBR_BIT				BIT(5)
#define I2C_PIN_BIT				BIT(4)
#define PIN_EN_CTRL_MASK			SMB136_MASK(2, 2)
#define OTG_LBR_PIN_CTRL_MASK			SMB136_MASK(2, 0)

#define OTG_LBR_CTRL_REG			0x06
#define BATT_MISSING_DET_EN_BIT		BIT(7)
#define AUTO_RECHARGE_THRESH_MASK		BIT(6)
#define USB_DP_DN_DET_EN_MASK		BIT(5)
#define OTG_LBR_BATT_CURRENT_LIMIT_MASK	SMB136_MASK(2, 3)
#define OTG_LBR_UVLO_THRESH_MASK		SMB136_MASK(3, 0)

#define FAULT_INTR_REG				0x07
#define SAFETY_TIMER_EXP_MASK			SMB136_MASK(1, 7)
#define BATT_TEMP_UNSAFE_MASK		SMB136_MASK(1, 6)
#define INPUT_OVLO_IVLO_MASK			SMB136_MASK(1, 5)
#define BATT_OVLO_MASK			SMB136_MASK(1, 4)
#define INTERNAL_OVER_TEMP_MASK		SMB136_MASK(1, 2)
#define ENTER_TAPER_CHG_MASK			SMB136_MASK(1, 1)
#define CHG_MASK				SMB136_MASK(1, 0)

#define CELL_TEMP_MON_REG			0x08
#define THERMISTOR_CURR_MASK			SMB136_MASK(2, 6)
#define LOW_TEMP_CHG_INHIBIT_MASK		SMB136_MASK(3, 3)
#define HIGH_TEMP_CHG_INHIBIT_MASK		SMB136_MASK(3, 0)

#define	SAFETY_TIMER_THERMAL_SHUTDOWN_REG	0x09
#define DCIN_OVLO_SEL_MASK			SMB136_MASK(2, 7)
#define RELOAD_EN_INPUT_VOLTAGE_MASK	SMB136_MASK(1, 6)
#define THERM_SHUTDN_EN_MASK			SMB136_MASK(1, 5)
#define STANDBY_WD_TIMER_EN_MASK		SMB136_MASK(1, 4)
#define COMPLETE_CHG_TMOUT_MASK		SMB136_MASK(2, 2)
#define PRE_CHG_TMOUT_MASK			SMB136_MASK(2, 0)

#define	VSYS_REG				0x0A
#define	VSYS_MASK				SMB136_MASK(3, 4)

#define IRQ_RESET_REG				0x30

#define COMMAND_A_REG				0x31
#define	VOLATILE_REGS_WRITE_PERM_BIT	BIT(7)
#define	POR_BIT				BIT(6)
#define	FAST_CHG_SETTINGS_BIT			BIT(5)
#define	BATT_CHG_EN_BIT			BIT(4)
#define	USBIN_MODE_500_BIT			BIT(3)
#define	USBIN_MODE_HCMODE_BIT		BIT(2)
#define	OTG_LBR_EN_BIT			BIT(1)
#define	STAT_OE_BIT				BIT(0)

#define STATUS_A_REG				0x32
#define INTERNAL_TEMP_IRQ_STAT		BIT(4)
#define DCIN_OV_IRQ_STAT			BIT(3)
#define DCIN_UV_IRQ_STAT			BIT(2)
#define USBIN_OV_IRQ_STAT			BIT(1)
#define USBIN_UV_IRQ_STAT			BIT(0)

#define STATUS_B_REG				0x33
#define USB_PIN_STAT				BIT(7)
#define USB51_MODE_STAT			BIT(6)
#define USB51_HC_MODE_STAT			BIT(5)
#define INTERNAL_TEMP_LIMIT_B_STAT		BIT(4)
#define DC_IN_OV_STAT				BIT(3)
#define DC_IN_UV_STAT				BIT(2)
#define USB_IN_OV_STAT				BIT(1)
#define USB_IN_UV_STAT				BIT(0)

#define	STATUS_C_REG				0x34
#define AUTO_IN_CURR_LIMIT_MASK		SMB136_MASK(4, 4)
#define AUTO_IN_CURR_LIMIT_STAT		BIT(3)
#define AUTO_SOURCE_DET_COMP_STAT_MASK	SMB136_MASK(2, 1)
#define AUTO_SOURCE_DET_RESULT_STAT	BIT(0)

#define	STATUS_D_REG				0x35
#define	VBATT_LESS_THAN_VSYS_STAT		BIT(7)
#define	USB_FAIL_STAT				BIT(6)
#define	BATT_TEMP_STAT_MASK			SMB136_MASK(2, 4)
#define	INTERNAL_TEMP_LIMIT_STAT		BIT(2)
#define	OTG_LBR_MODE_EN_STAT		BIT(1)
#define	OTG_LBR_VBATT_UVLO_STAT		BIT(0)

#define	STATUS_E_REG				0x36
#define	CHARGE_CYCLE_COUNT_STAT		BIT(7)
#define	CHARGER_TERM_STAT			BIT(6)
#define	SAFETY_TIMER_STAT_MASK		SMB136_MASK(2, 4)
#define	CHARGER_ERROR_STAT			BIT(3)
#define	CHARGING_STAT_E			SMB136_MASK(2, 1)
#define	CHARGING_EN				BIT(0)

#define	STATUS_F_REG				0x37
#define	WD_IRQ_ACTIVE_STAT			BIT(7)
#define	OTG_OVERCURRENT_STAT		BIT(6)
#define	BATT_PRESENT_STAT			BIT(4)
#define	BATT_OV_LATCHED_STAT			BIT(3)
#define	CHARGER_OVLO_STAT			BIT(2)
#define	CHARGER_UVLO_STAT			BIT(1)
#define	BATT_LOW_STAT				BIT(0)

#define	STATUS_G_REG				0x38
#define	CHARGE_TIMEOUT_IRQ_STAT		BIT(7)
#define	PRECHARGE_TIMEOUT_IRQ_STAT		BIT(6)
#define	BATT_HOT_IRQ_STAT			BIT(5)
#define	BATT_COLD_IRQ_STAT			BIT(4)
#define	BATT_OV_IRQ_STAT			BIT(3)
#define	TAPER_CHG_IRQ_STAT			BIT(2)
#define	FAST_CHG_IRQ_STAT			BIT(1)
#define	CHARGING_IRQ_STAT			BIT(0)

#define	STATUS_H_REG				0x39
#define	CHARGE_TIMEOUT_STAT			BIT(7)
#define	PRECHARGE_TIMEOUT_STAT		BIT(6)
#define	BATT_HOT_STAT				BIT(5)
#define	BATT_COLD_STAT			BIT(4)
#define	BATT_OV_STAT				BIT(3)
#define	TAPER_CHG_STAT			BIT(2)
#define	FAST_CHG_STAT				BIT(1)
#define	CHARGING_STAT_H			BIT(0)

#define DEV_ID_REG				0x3B
#define DEVICE_IS_SMB136			BIT(3)

#define COMMAND_B_REG				0x3C
#define	THERM_NTC_CURR_VERRIDE		BIT(7)

#define SMB136_CHG_PERIOD			((HZ) * 5)

#define INPUT_CURRENT_REG_DEFAULT		0xE1
#define INPUT_CURRENT_REG_MIN		0x01

#define	COMMAND_A_REG_DEFAULT		0x90
#define	COMMAND_A_REG_OTG_MODE		0x82

#define	PIN_CTRL_REG_DEFAULT			0x04

#define	PIN_CTRL_REG_CHG_OFF			0x04

#define	FAST_CHG_E_STATUS 			0x2
#define  TAPER_CHG_E_STATUS			0x3
#define SMB136_DEFAULT_BATT_RATING   		950
#define SMB136_DEBUG				1
int smb136_in_charging_state=0;
static int smb136_chg_det_irq = -1;
static struct i2c_client *smb136_chg_i2c_client = NULL;
extern volatile struct pm8058_chip	*g_pm_chip_for_all;
struct delayed_work smb136_charge_work;



extern unsigned int get_mfg_mode(void);
extern void arch_power_off(void);
extern  int in_rom_mode;


int msm_is_chg_plugged_in(void);

#define	SMB136_REGISTER_TO_MSM		1
#ifdef SMB136_REGISTER_TO_MSM
#define 	SMB136_CHARGE_USB_500_MA_MODE		500
#define 	SMB136_CHARGE_USB_100_MA_MODE		100
#define	SMB136_CHARGE_WALL_CHARGER_MODE		1200
enum charger_supply_type {
	SMB136_AC,
	SMB136_USB,
};
#endif
struct smb136_data {
	struct i2c_client *client;
	struct delayed_work charge_work;
	struct delayed_work smb136_interrupt_work;

	bool charging;
	int chgcurrent;
	int term_current;
	int max_system_voltage;
	int min_system_voltage;

	int valid_n_gpio;

	struct power_supply psy_batt;
	int batt_status;
	int batt_chg_type;
	int batt_present;
	int min_design;
	int max_design;
	int batt_mah_rating;

	struct power_supply psy_usb;
	int usb_status;

	bool stop_heartbeat;
	bool disabled;
	
	int present;
	int chg_type;
	struct msm_hardware_charger adapter_hw_chg;
	
};

static void (*notify_vbus_state_func_ptr)(int);

static struct smb136_data *usb_smb136_chg;
int smb136_register_vbus_sn(void (*callback)(int))
{
	pr_debug(KERN_INFO "%s\n", __func__);
	notify_vbus_state_func_ptr = callback;
	return 0;
}

void smb136_unregister_vbus_sn(void (*callback)(int))
{
	pr_debug(KERN_INFO "%s\n", __func__);
	notify_vbus_state_func_ptr = NULL;
}
#if 0
static void smb137b_notify_usb_of_the_plugin_event(struct smb137b_data
					*smb137b_chg,
					   int plugin)
{
	plugin = !!plugin;
	if (plugin == 1 && usb_notified_of_insertion == 0) {
		usb_notified_of_insertion = 1;
		if (notify_vbus_state_func_ptr) {
			pr_debug("%s notifying plugin\n", __func__);
			(*notify_vbus_state_func_ptr) (plugin);
		} else
			pr_debug("%s unable to notify plugin\n",
				__func__);
	}
	if (plugin == 0 && usb_notified_of_insertion == 1) {
		if (notify_vbus_state_func_ptr) {
			pr_debug("%s notifying unplugin\n",
				__func__);
			(*notify_vbus_state_func_ptr) (plugin);
		} else
			pr_debug("%s unable to notify unplugin\n",
				__func__);
		usb_notified_of_insertion = 0;
	}
}
#endif
enum charger_stat {
	SMB136_ABSENT,
	SMB136_PRESENT,
	SMB136_ENUMERATED,
};

static enum power_supply_property power_props[] = {
	POWER_SUPPLY_PROP_PRESENT,
	POWER_SUPPLY_PROP_ONLINE,
};

static char *power_supplied_to[] = {
	"battery",
};

static int power_get_property(struct power_supply *psy,
				  enum power_supply_property psp,
				  union power_supply_propval *val)
{
	struct smb136_data *smb136_chg;

	smb136_chg = container_of(psy, struct smb136_data,
			psy_usb);
	switch (psp) {
	case POWER_SUPPLY_PROP_PRESENT:
		val->intval = (smb136_chg->usb_status != SMB136_ABSENT);
		break;
	case POWER_SUPPLY_PROP_ONLINE:
		val->intval = (smb136_chg->usb_status == SMB136_ENUMERATED);
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

static enum power_supply_property batt_power_props[] = {
	POWER_SUPPLY_PROP_STATUS,
	POWER_SUPPLY_PROP_CHARGE_TYPE,
	POWER_SUPPLY_PROP_PRESENT,
	POWER_SUPPLY_PROP_TECHNOLOGY,
	POWER_SUPPLY_PROP_VOLTAGE_MAX_DESIGN,
	POWER_SUPPLY_PROP_VOLTAGE_MIN_DESIGN,
	POWER_SUPPLY_PROP_VOLTAGE_NOW,
	POWER_SUPPLY_PROP_CAPACITY,
};

static int batt_power_get_property(struct power_supply *psy,
				       enum power_supply_property psp,
				       union power_supply_propval *val)
{
	struct smb136_data *smb136_chg;

	smb136_chg = container_of(psy, struct smb136_data,
			psy_batt);
	switch (psp) {
	case POWER_SUPPLY_PROP_STATUS:
		val->intval  =  smb136_chg->batt_status;
		break;
	case POWER_SUPPLY_PROP_CHARGE_TYPE:
		val->intval  =  smb136_chg->batt_chg_type;
		break;
	case POWER_SUPPLY_PROP_PRESENT:
		val->intval  =  smb136_chg->batt_present;
		break;
	case POWER_SUPPLY_PROP_TECHNOLOGY:
		val->intval = POWER_SUPPLY_TECHNOLOGY_LION;
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_MAX_DESIGN:
		val->intval = smb136_chg->max_design;
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_MIN_DESIGN:
		val->intval = smb136_chg->min_design;
		break;
	case POWER_SUPPLY_PROP_VOLTAGE_NOW:
		
		val->intval = 3500000;
		break;
	case POWER_SUPPLY_PROP_CAPACITY:
		
		val->intval = 50;
		break;
	default:
		return -EINVAL;
	}
	return 0;
}

static int smb136_read_reg(struct i2c_client *client, int reg,
	u8 *val)
{
	s32 ret;
	struct smb136_data *smb136_chg;

	smb136_chg = i2c_get_clientdata(client);
	ret = i2c_smbus_read_byte_data(smb136_chg->client, reg);
	if (ret < 0) {
		dev_err(&smb136_chg->client->dev,
			"i2c read fail: can't read from %02x: %d\n", reg, ret);
		return ret;
	} else
		*val = ret;

	return 0;
}

static int smb136_write_reg(struct i2c_client *client, int reg,
	u8 val)
{
	s32 ret;
	struct smb136_data *smb136_chg;

	smb136_chg = i2c_get_clientdata(client);
	ret = i2c_smbus_write_byte_data(smb136_chg->client, reg, val);
	if (ret < 0) {
		dev_err(&smb136_chg->client->dev,
			"i2c write fail: can't write %02x to %02x: %d\n",
			val, reg, ret);
		return ret;
	}
	return 0;
}


 #ifdef SMB136_DEBUG
static void smb136_dbg_print_all_status_regs(struct smb136_data *smb136_chg)
{
	int ret;
	u8 temp;
	printk("*************************************************\n");
	ret = smb136_read_reg(smb136_chg->client, CHG_CURRENT_REG, &temp);
	printk("CHG_CURRENT_REG_00H=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, INPUT_CURRENT_LIMIT_REG, &temp);
	printk("INPUT_CURRENT_LIMIT_REG_01H=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, FLOAT_VOLTAGE_REG, &temp);
	printk("FLOAT_VOLTAGE_REG_02H=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, CONTROL_A_REG, &temp);
	printk("CONTROL_A_REG_03H=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, CONTROL_B_REG, &temp);
	printk("CONTROL_B_REG_04H=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, PIN_CTRL_REG, &temp);
	printk("PIN_CTRL_REG_05H=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, OTG_LBR_CTRL_REG, &temp);
	printk("OTG_LBR_CTRL_REG_06H=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, FAULT_INTR_REG, &temp);
	printk("FAULT_INTR_REG_07H=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, CELL_TEMP_MON_REG, &temp);
	printk("CELL_TEMP_MON_REG_08H=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, SAFETY_TIMER_THERMAL_SHUTDOWN_REG, &temp);
	printk("SAFETY_TIMER_THERMAL_SHUTDOWN_REG_09H=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, VSYS_REG, &temp);
	printk("VSYS_REG_0AH=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, IRQ_RESET_REG, &temp);
	printk("IRQ_RESET_REG_30H=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, COMMAND_A_REG, &temp);
	printk("COMMAND_A_REG_31H=0x%x\n",temp);
	printk("\n");
	ret = smb136_read_reg(smb136_chg->client, STATUS_A_REG, &temp);
	printk("STATUS_A_REG=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, STATUS_B_REG, &temp);
	printk("STATUS_B_REG=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, STATUS_C_REG, &temp);
	printk("STATUS_C_REG=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, STATUS_D_REG, &temp);
	printk("STATUS_D_REG=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, STATUS_E_REG, &temp);
	printk("STATUS_E_REG=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, STATUS_F_REG, &temp);
	printk("STATUS_F_REG=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, STATUS_G_REG, &temp);
	printk("STATUS_G_REG=0x%x\n",temp);
	ret = smb136_read_reg(smb136_chg->client, STATUS_H_REG, &temp);
	printk("STATUS_H_REG=0x%x\n",temp);
	printk("*************************************************\n");
}
 #else
 static void smb136_dbg_print_all_status_regs(struct smb136_data *smb136_chg)
{

}
 #endif


 
static int smb136_start_charging(struct msm_hardware_charger *hw_chg,
		int chg_voltage, int chg_current)
{
	int cmd_val = COMMAND_A_REG_DEFAULT;
	u8 temp = 0;
	int ret = 0;
	struct smb136_data *smb136_chg;	
	
	smb136_chg = container_of(hw_chg, struct smb136_data,adapter_hw_chg);
	
	printk(KERN_INFO "%s:smb136 start charging =%d\n", __func__,chg_current);
		
             
	if (chg_current < 500){
		
		cmd_val &= ~USBIN_MODE_500_BIT;
		smb136_chg->adapter_hw_chg.charger_type=CHARGER_TYPE_USB;
	}else if (chg_current == 500){
		
		cmd_val |= USBIN_MODE_500_BIT;
		smb136_chg->adapter_hw_chg.charger_type=CHARGER_TYPE_USB;		
	}else{
		
		cmd_val |= USBIN_MODE_HCMODE_BIT;
		smb136_chg->adapter_hw_chg.charger_type=CHARGER_TYPE_AC;		
	}	

	smb136_chg->chgcurrent = chg_current;

             
 	 
 	
	ret = smb136_write_reg(smb136_chg->client, COMMAND_A_REG, cmd_val); 
	
	ret = smb136_write_reg(smb136_chg->client,FLOAT_VOLTAGE_REG, FLOAT_VOLTAGE);
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s write 02 float voltage register fail %d\n", __func__, ret);
	}
	
	
 	 ret = smb136_read_reg(smb136_chg->client, CONTROL_A_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read control reg A,ret=%d\n", __func__, ret);
	} 
	temp |= PRE_TO_FAST_V_MASK;
	ret = smb136_write_reg(smb136_chg->client,CONTROL_A_REG, temp);
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s set pre-charge to fast-charge voltage threshold fail,ret=%d\n", __func__, ret);
	}
 	
 	 
	
 	 
	ret = smb136_write_reg(smb136_chg->client,CELL_TEMP_MON_REG, 0X9C);
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s write 08 control register fail %d\n", __func__, ret);
	}
	
	
 	 
	
 	 ret = smb136_read_reg(smb136_chg->client, INPUT_CURRENT_LIMIT_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read input current limit reg %d\n", __func__, ret);
	} 
	temp|=BIT(5);
	temp|=BIT(6);
	temp|=BIT(7);
	ret = smb136_write_reg(smb136_chg->client,INPUT_CURRENT_LIMIT_REG, temp);
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s couldn't write input current limit reg %d\n", __func__, ret);
	}
	


	
	ret = smb136_read_reg(smb136_chg->client, FAULT_INTR_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read fault interrupt register%d\n", __func__, ret);
	} 
	temp&=0x00;
	temp|= INPUT_OVLO_IVLO_MASK;
	ret = smb136_write_reg(smb136_chg->client,FAULT_INTR_REG, temp);
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s couldn't write fault interrupt register%d\n", __func__, ret);
	}

	
	 
	
	ret = smb136_read_reg(smb136_chg->client, CONTROL_B_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read control B register%d\n", __func__, ret);
	} 
	
	
	temp&=0X3F;	
	temp|=BIT(6);
	
	
	temp&=0XFE;
	
	ret = smb136_write_reg(smb136_chg->client,CONTROL_B_REG, temp);
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s couldn't write control B register%d\n", __func__, ret);
	}
	
	
	ret = smb136_read_reg(smb136_chg->client, FAULT_INTR_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read fault interrupt register%d\n", __func__, ret);
	} 
	temp&=0X00;
	temp|= INPUT_OVLO_IVLO_MASK;
	ret = smb136_write_reg(smb136_chg->client,FAULT_INTR_REG, temp);
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s couldn't write fault interrupt register%d\n", __func__, ret);
	}
	
	
	ret = smb136_read_reg(smb136_chg->client, FLOAT_VOLTAGE_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read float register%d\n", __func__, ret);
	} 
	temp|=BIT(7);
	ret = smb136_write_reg(smb136_chg->client,FLOAT_VOLTAGE_REG, temp);
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s couldn't write float register%d\n", __func__, ret);
	}

	
	
	ret = smb136_write_reg(smb136_chg->client,PIN_CTRL_REG, PIN_CTRL_REG_DEFAULT);
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s write 05h control  register fail %d\n", __func__, ret);
	}
	
 	 
	smb136_chg->charging = true;
	smb136_chg->batt_status = POWER_SUPPLY_STATUS_CHARGING;
	smb136_chg->batt_chg_type = POWER_SUPPLY_CHARGE_TYPE_TRICKLE;
	
	ret = smb136_read_reg(smb136_chg->client, STATUS_E_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read status e reg %d\n", __func__, ret);
	} else {
		if (temp & CHARGER_ERROR_STAT) {
			dev_err(&smb136_chg->client->dev,
				"%s chg error E=0x%x\n", __func__, temp);			
		}
		if (((temp & CHARGING_STAT_E) >> 1) >= FAST_CHG_E_STATUS){
			smb136_chg->batt_chg_type= POWER_SUPPLY_CHARGE_TYPE_FAST;
			msm_charger_notify_event(&smb136_chg->adapter_hw_chg,	
				CHG_BATT_BEGIN_FAST_CHARGING);			
		}
	}
	
	smb136_dbg_print_all_status_regs(usb_smb136_chg);
 
#ifndef SMB136_REGISTER_TO_MSM
	schedule_delayed_work(&smb136_chg->charge_work, SMB136_CHG_PERIOD);
#endif
 
  
	return ret;
}

 

  
static int smb136_stop_charging(struct msm_hardware_charger *hw_chg)
{
	int ret = 0;
	u8 temp;
	struct smb136_data *smb136_chg;	
	printk("smb136 stop charging \n");
	smb136_chg = container_of(hw_chg, struct smb136_data,adapter_hw_chg);	
	
	smb136_chg->charging = false;
	smb136_chg->batt_status = POWER_SUPPLY_STATUS_DISCHARGING;
	smb136_chg->batt_chg_type = POWER_SUPPLY_CHARGE_TYPE_NONE;

	
	ret = smb136_read_reg(smb136_chg->client, COMMAND_A_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read status e reg %d\n", __func__, ret);
	}
	temp&=0xEF;
	

	
	
	ret = smb136_write_reg(smb136_chg->client,COMMAND_A_REG, temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't write to command_reg\n", __func__);
		goto out;
	}
	
out:
	return ret;
}

 
static int smb136_charging_switched(struct msm_hardware_charger *hw_chg)
{
	return 0;
}

 
static int smb136_get_fastcharging_current(struct msm_hardware_charger *hw_chg)
{
	int ret=0;
	u8 temp;
	struct smb136_data *smb136_chg;	
	
	smb136_chg = container_of(hw_chg, struct smb136_data,adapter_hw_chg);
	
	ret = smb136_read_reg(smb136_chg->client, STATUS_B_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read status e reg %d\n", __func__, ret);
	}
	if(temp&USB51_HC_MODE_STAT){
		ret=SMB136_CHARGE_WALL_CHARGER_MODE;
	}else{
		if(temp&USB51_MODE_STAT){
			ret=SMB136_CHARGE_USB_500_MA_MODE;			
		}else{
			ret=SMB136_CHARGE_USB_100_MA_MODE;
		}
	}	
	return ret;
}
 

  
static int smb136_set_fastcharging_current(struct msm_hardware_charger *hw_chg, int chg_current)
{
	int ret=0;
	u8 temp;	
	struct smb136_data *smb136_chg;	
	
	smb136_chg = container_of(hw_chg, struct smb136_data,adapter_hw_chg);
	
 	ret = smb136_read_reg(smb136_chg->client, COMMAND_A_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read charge current reg %d\n", __func__, ret);
	} 
	
	temp|= BIT(7);
	ret = smb136_write_reg(smb136_chg->client, COMMAND_A_REG, temp); 
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s write 31h command register A fail %d\n", __func__, ret);
	}
	
	
	ret = smb136_read_reg(smb136_chg->client, CHG_CURRENT_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read charge current reg %d\n", __func__, ret);
	}
	
	if(chg_current==500){
		
		temp&= FAST_CHG_CURRENT_LOW_BATTERY;
		ret = smb136_write_reg(smb136_chg->client,CHG_CURRENT_REG, temp);
		if (ret) {
			dev_err(&smb136_chg->client->dev,
				"%s couldn't write to charge current reg\n", __func__);
		}
	}else if(chg_current>500){
		temp = FAST_CHG_CURRENT_NOT_LOW_BATTERY;
		ret = smb136_write_reg(smb136_chg->client,CHG_CURRENT_REG, temp);
		if (ret) {
			dev_err(&smb136_chg->client->dev,
				"%s couldn't write to charge current reg\n", __func__);
		}	
	}
	
	return ret;
}

static int smb136_set_charging_mode(struct msm_hardware_charger *hw_chg, int chg_current)
{
	int cmd_val = COMMAND_A_REG_DEFAULT;
	int ret = 0;
	u8 temp;
	struct smb136_data *smb136_chg;	
	
	smb136_chg = container_of(hw_chg, struct smb136_data,adapter_hw_chg);
	
	printk(KERN_INFO "%s:smb136 set charging mode,current=%d\n", __func__,chg_current);
	
	
             
	if(chg_current == 0){
		ret = smb136_read_reg(smb136_chg->client, COMMAND_A_REG, &temp);
		if (ret) {
			dev_err(&smb136_chg->client->dev,
				"%s couldn't read status e reg %d\n", __func__, ret);
		}
		temp&=0xEF;	
		ret = smb136_write_reg(smb136_chg->client,COMMAND_A_REG, temp);
		if (ret) {
			dev_err(&smb136_chg->client->dev,
				"%s couldn't write to command_reg\n", __func__);
		}
		goto out;
	}
	else if (chg_current == 100){
		
		cmd_val &= ~USBIN_MODE_500_BIT;
		
	}else if (chg_current == 500){
		
		cmd_val |= USBIN_MODE_500_BIT;			
	}else{
		
		cmd_val |= USBIN_MODE_HCMODE_BIT;			
	}	 	 
 	
	ret = smb136_write_reg(smb136_chg->client, COMMAND_A_REG, cmd_val); 
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s write 31h command  register fail %d\n", __func__, ret);
	}	
	
 	
	ret = smb136_write_reg(smb136_chg->client,PIN_CTRL_REG, PIN_CTRL_REG_DEFAULT);
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s write 05h control  register fail %d\n", __func__, ret);
	}
out:
	return ret;
}
  

  
static int smb136_is_chg_plugged_in(void)
{
	int ret = -1;
	
	ret = msm_is_chg_plugged_in();
				
	if(1 == ret){
		return 1;
	}else{
		
		
		#if 1
		if(3 == get_mfg_mode()){
			arch_power_off();	
		}
		#endif

		return 0;
	}
	
}
  

static irqreturn_t smb136_valid_handler(int irq, void *dev_id)
{	
	struct smb136_data *smb136_chg;
	struct i2c_client *client = dev_id;
	smb136_chg = i2c_get_clientdata(client);
	
	if (smb136_is_chg_plugged_in()) {	
		if (!smb136_chg->present) {
			
			msm_charger_notify_event(&smb136_chg->adapter_hw_chg,CHG_INSERTED_EVENT);
			smb136_chg->present = 1;	
			smb136_chg->adapter_hw_chg.charger_plug_status=1;
			printk(KERN_INFO "%s:power on:plug in.\n", __func__);
		}
	} else{
		if (smb136_chg->present) {
			
			msm_charger_notify_event(&smb136_chg->adapter_hw_chg,CHG_REMOVED_EVENT);
			smb136_chg->present = 0;
			smb136_chg->adapter_hw_chg.charger_plug_status=0;
			  
			smb136_chg->adapter_hw_chg.charger_type=CHARGER_TYPE_INVALID;	
			  
			 
			printk(KERN_INFO "%s:power on:plug out.\n", __func__);
			
		}
	}	
	cancel_delayed_work_sync(&smb136_charge_work);		
	return IRQ_HANDLED;
}


static int smb136_get_charger_id(struct msm_hardware_charger *hw_chg)
{
	int ret=0;
	u8 temp;	
	struct smb136_data *smb136_chg;	
	
	smb136_chg = container_of(hw_chg, struct smb136_data,adapter_hw_chg);
	ret = smb136_read_reg(smb136_chg->client, DEV_ID_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read device id reg %d\n", __func__, ret);
		goto out;
	}
	if(temp&DEVICE_IS_SMB136)
		ret = DEVICE_IS_SMB136;
	return ret;
out:	ret=0;
	return ret;
}

#ifdef CONFIG_DEBUG_FS
static struct dentry *dent;
static int debug_fs_otg;
static int otg_get(void *data, u64 *value)
{
	*value = debug_fs_otg;
	return 0;
}
static int otg_set(void *data, u64 value)
{
	smb136_otg_power(debug_fs_otg);
	return 0;
}
DEFINE_SIMPLE_ATTRIBUTE(smb136_otg_fops, otg_get, otg_set, "%llu\n");

static int disable_get(void *data, u64 *value)
{
	struct smb136_data *smb136_chg = data;

	*value = (u64)smb136_chg->disabled;
	return 0;
}
static int disable_set(void *data, u64 value)
{
	struct smb136_data *smb136_chg = data;

	smb136_chg->disabled = !!value;
	if (smb136_chg->disabled)
		smb136_stop_charging(&smb136_chg->adapter_hw_chg);
	return 0;
}
DEFINE_SIMPLE_ATTRIBUTE(smb136_disable_fops,
					disable_get, disable_set, "%llu\n");

static void smb136_create_debugfs_entries(struct smb136_data *smb136_chg)
{
	dent = debugfs_create_dir("smb136", NULL);
	if (dent) {
		debugfs_create_file("otg", 0644, dent, NULL, &smb136_otg_fops);
		debugfs_create_file("disable", 0644, dent, smb136_chg,
							&smb136_disable_fops);
	}
}
static void smb136_destroy_debugfs_entries(void)
{
	if (dent)
		debugfs_remove_recursive(dent);
}
#else
static void smb136_create_debugfs_entries(struct smb136_data *smb136_chg)
{
}
static void smb136_destroy_debugfs_entries(void)
{
}
#endif

static void smb136_heartbeat(struct work_struct *smb136_work)
{
	int ret;
	struct smb136_data *smb136_chg;
	u8 temp = 0;
	int notify_batt_changed = 0;

	smb136_chg = container_of(smb136_work, struct smb136_data,
			charge_work.work);
	
	if (smb136_chg->stop_heartbeat)
		return;

	dev_dbg(&smb136_chg->client->dev, "%s\n", __func__);


	ret = smb136_read_reg(smb136_chg->client, STATUS_F_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read status f reg %d\n", __func__, ret);
		goto out;
	}

	if (smb136_chg->batt_present != !(temp & BATT_PRESENT_STAT)) {
		smb136_chg->batt_present = !(temp & BATT_PRESENT_STAT);
		notify_batt_changed = 1;		
	}

	if (!smb136_chg->batt_present)
		smb136_chg->batt_chg_type = POWER_SUPPLY_CHARGE_TYPE_NONE;

	if (!smb136_chg->batt_present && smb136_chg->charging)
	{
	#ifndef SMB136_REGISTER_TO_MSM
		smb136_stop_charging(smb136_chg->adapter_hw_chg);
	#endif
	}

	if (smb136_chg->batt_present
		&& smb136_chg->charging
		&& smb136_chg->batt_chg_type
			!= POWER_SUPPLY_CHARGE_TYPE_FAST) {
		ret = smb136_read_reg(smb136_chg->client,
						STATUS_E_REG, &temp);
		if (ret) {
			dev_err(&smb136_chg->client->dev,
				"%s couldn't read cntrl reg\n", __func__);
			goto out;

		} else {
			if (temp & CHARGER_ERROR_STAT) {
				dev_err(&smb136_chg->client->dev,
					"%s error E=0x%x\n", __func__, temp);
			}
			if (((temp & CHARGING_STAT_E) >> 1)>= FAST_CHG_E_STATUS) {
				smb136_chg->batt_chg_type= POWER_SUPPLY_CHARGE_TYPE_FAST;
				notify_batt_changed = 1;
				msm_charger_notify_event(&smb136_chg->adapter_hw_chg,
					CHG_BATT_BEGIN_FAST_CHARGING);				
			} else {
				smb136_chg->batt_chg_type
					= POWER_SUPPLY_CHARGE_TYPE_TRICKLE;				
			}
		}
	}
	if (smb136_chg->batt_present&& smb136_chg->charging	&& smb136_chg->batt_chg_type
		== POWER_SUPPLY_CHARGE_TYPE_FAST) {
		ret = smb136_read_reg(smb136_chg->client,	STATUS_E_REG, &temp);
		if (ret) {
			dev_err(&smb136_chg->client->dev,
				"%s couldn't read cntrl reg\n", __func__);
			goto out;

		}
		if((temp&CHARGER_TERM_STAT)&&((temp & CHARGING_STAT_E)>=TAPER_CHG_E_STATUS)){
			msm_charger_notify_event(&smb136_chg->adapter_hw_chg,
					CHG_DONE_EVENT);
		}		
	}
	
out:
	if (!smb136_chg->stop_heartbeat)
		schedule_delayed_work(&smb136_chg->charge_work,
						SMB136_CHG_PERIOD);
}


static int smb136_battery_is_present(struct msm_hardware_charger *hw_chg)
{
	int ret=0;
	int retval = 0;
	struct smb136_data *smb136_chg;
	u8 temp = 0;	

	smb136_chg = container_of(hw_chg, struct smb136_data,adapter_hw_chg);	

	dev_dbg(&smb136_chg->client->dev, "%s\n", __func__);

	ret = smb136_read_reg(smb136_chg->client, STATUS_F_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read status f reg %d\n", __func__, ret);		
	}

	if (smb136_chg->batt_present != !(temp & BATT_PRESENT_STAT)) {
		smb136_chg->batt_present = !(temp & BATT_PRESENT_STAT);
	}
	retval = smb136_chg->batt_present;

	return retval;
}
static int smb136_fast_charging_event(struct msm_hardware_charger *hw_chg)
{
	int ret = 0;
	int retval = 0;
	u8 temp;
	int present;
	struct smb136_data *smb136_chg;	
	
	smb136_chg = container_of(hw_chg, struct smb136_data,adapter_hw_chg);	
	
	dev_dbg(&smb136_chg->client->dev, "%s\n", __func__);

	present = smb136_battery_is_present(&(smb136_chg->adapter_hw_chg));
	if (present) {
		ret = smb136_read_reg(smb136_chg->client,	STATUS_E_REG, &temp);
		if (ret) 
			dev_err(&smb136_chg->client->dev,	"%s couldn't read cntrl reg\n", __func__);
			
		else {			
			if (temp & CHARGER_ERROR_STAT) {
				dev_err(&smb136_chg->client->dev,	"%s error E=0x%x\n", __func__, temp);
			}
			if (((temp & CHARGING_STAT_E) >> 1)>= FAST_CHG_E_STATUS) 
				retval= 1;
		}
	}

	return retval;
}

static int smb136_taper_charging_event(struct msm_hardware_charger *hw_chg)
{
	int ret = 0;
	int retval = 0;
	u8 temp;
	int present;
	struct smb136_data *smb136_chg;	
	smb136_chg = container_of(hw_chg, struct smb136_data,adapter_hw_chg);	
	
	dev_dbg(&smb136_chg->client->dev, "%s\n", __func__);

	present = smb136_battery_is_present(&(smb136_chg->adapter_hw_chg));
	if (present) {
		ret = smb136_read_reg(smb136_chg->client,	STATUS_E_REG, &temp);
		if (ret) 
			dev_err(&smb136_chg->client->dev,	"%s couldn't read cntrl reg\n", __func__);
			
		else {
			if (temp & CHARGER_ERROR_STAT) {
				dev_err(&smb136_chg->client->dev,	"%s error E=0x%x\n", __func__, temp);
			}
			if (((temp & CHARGING_STAT_E) >> 1)>= TAPER_CHG_E_STATUS) 
				retval= 1;
		}
	}

	return retval;
}
static int smb136_charging_done_event(struct msm_hardware_charger *hw_chg)
{
	int ret = 0;
	int retval = 0;
	u8 temp;
	int present;
	struct smb136_data *smb136_chg;	
	
	smb136_chg = container_of(hw_chg, struct smb136_data,adapter_hw_chg);	
	
	dev_dbg(&smb136_chg->client->dev, "%s\n", __func__);

	present = smb136_battery_is_present(&(smb136_chg->adapter_hw_chg));
	if (present) {
		ret = smb136_read_reg(smb136_chg->client,	STATUS_E_REG, &temp);
		if (ret) 
			dev_err(&smb136_chg->client->dev,	"%s couldn't read cntrl reg\n", __func__);
			
		else {			
			if (temp & CHARGER_ERROR_STAT) {
				dev_err(&smb136_chg->client->dev,	"%s error E=0x%x\n", __func__, temp);
			}
			if(temp&CHARGER_TERM_STAT)
				retval = 1;
		}
	}
	if(retval)
		printk(KERN_INFO "%s:STATUS_E_REG=0x%x.\n", __func__,temp);

	return retval;
}

  
 
static irqreturn_t smb136_chg_fault_handler(int irq, void *dev_id)
{	
	struct smb136_data *smb136_chg;
	struct i2c_client *client = dev_id;
	
	smb136_chg = i2c_get_clientdata(client);
	printk(KERN_INFO "%s:chg_stat handler.\n", __func__);
	
	schedule_delayed_work(&smb136_chg->smb136_interrupt_work,HZ);	
	return IRQ_HANDLED;
}
static void smb136_interrput_status_work(struct work_struct *smb136_work)
{

	int ret;
	struct smb136_data *smb136_chg;
	u8 temp = 0;
	int usbin_ov_trip = 0;
	int usbin_ov = 0;
	
	smb136_chg = container_of(smb136_work, struct smb136_data,
			smb136_interrupt_work.work);
	
	if(smb136_chg->client)
		printk(KERN_INFO "%s:chg_fault interrupt ocurr,%s\n", __func__,smb136_chg->client->name);
	else{
		return;
	}

	dev_dbg(&smb136_chg->client->dev, "%s\n", __func__);

	
	ret = smb136_write_reg(smb136_chg->client,IRQ_RESET_REG, 0X00);
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s write IRQ reset register fail %d\n", __func__, ret);
	}
	
	ret = smb136_read_reg(smb136_chg->client, STATUS_A_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read status f reg %d\n", __func__, ret);
	}
	else
		printk(KERN_INFO "%s:STATUS_A_REG=0x%x.\n", __func__,temp);
	if(temp&USBIN_OV_IRQ_STAT){
		usbin_ov_trip = 1;
		printk(KERN_INFO "%s:USBIN OV IRQ.\n", __func__);
	}
	if(temp&USBIN_UV_IRQ_STAT){
		printk(KERN_INFO "%s:USBIN UV IRQ.\n", __func__);
	}
	
	ret = smb136_read_reg(smb136_chg->client, STATUS_B_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read status f reg %d\n", __func__, ret);
	}
	else
		printk(KERN_INFO "%s:STATUS_B_REG=0x%x.\n", __func__,temp);
	if(temp&USBIN_OV_IRQ_STAT){
		usbin_ov = 1;
		printk(KERN_INFO "%s:STATUS_B_REG:USBIN OV .\n", __func__);
	}
	if(temp&USBIN_UV_IRQ_STAT){
		printk(KERN_INFO "%s:STATUS_B_REG:USBIN UV.\n", __func__);
	}

	if(usbin_ov){
		printk(KERN_INFO "%s:USBIN over voltage!\n", __func__);
		msm_charger_notify_event(&smb136_chg->adapter_hw_chg,CHG_INPUT_OVER_VOLTAGE_EVENT);
	}

}



static void smb136_off_charging_work(struct work_struct *smb136_work)
{

	if (smb136_is_chg_plugged_in()) {	
		if (!usb_smb136_chg->present) {
			
			msm_charger_notify_event(&usb_smb136_chg->adapter_hw_chg,CHG_INSERTED_EVENT);
			usb_smb136_chg->present = 1;
			usb_smb136_chg->adapter_hw_chg.charger_plug_status=1;
			printk(KERN_INFO "%s:off charging:plug in.\n", __func__);
			
		}
	} else{
		if (usb_smb136_chg->present) {
			
			msm_charger_notify_event(&usb_smb136_chg->adapter_hw_chg,CHG_REMOVED_EVENT);
			usb_smb136_chg->present = 0;
			usb_smb136_chg->adapter_hw_chg.charger_plug_status=0;
			printk(KERN_INFO "%s:off charging:plug out.\n", __func__);
		}
	}
#if 1
	
	{
		int ret = -1;
	
		ret = request_threaded_irq(smb136_chg_i2c_client->irq, NULL,
					   smb136_valid_handler,
					   IRQF_TRIGGER_FALLING | IRQF_TRIGGER_RISING,
					   "smb136_charger_valid", smb136_chg_i2c_client);
		
		if (ret) {
			dev_err(&smb136_chg_i2c_client->dev,"%s request_threaded_irq failed for %d ret =%d\n",
				__func__, smb136_chg_i2c_client->irq, ret);
		}
		
		enable_irq_wake(smb136_chg_det_irq);	
		
	}
#endif
}

static int smb136_request_gpio(int gpio)
{
	int ret=0;
	
	ret = gpio_request(gpio, "CHG_STAT");
	if (ret) {
		printk(KERN_ERR "request CHG_STAT/IO30 fail\n");
		gpio_free(gpio);
	}
	
	return ret;

}

static int smb136_free_gpio(int gpio)
{	
	gpio_free(gpio);
	return 0;
}
static int __devinit smb136_probe(struct i2c_client *client,
				    const struct i2c_device_id *id)
{
	const struct smb136_platform_data *pdata;
	struct smb136_data *smb136_chg;
	u8 temp;
	int ret = 0;
	int chg_fault_int;
	pdata = client->dev.platform_data;
	

	if (pdata == NULL) {
		dev_err(&client->dev, "%s no platform data\n", __func__);
		ret = -EINVAL;
		goto out;
	}

	if (!i2c_check_functionality(client->adapter,
				I2C_FUNC_SMBUS_BYTE_DATA)) {
		ret = -EIO;
		goto out;
	}

	smb136_chg = kzalloc(sizeof(*smb136_chg), GFP_KERNEL);
	if (!smb136_chg) {
		ret = -ENOMEM;
		goto out;
	}

	INIT_DELAYED_WORK(&smb136_chg->charge_work, smb136_heartbeat);
	INIT_DELAYED_WORK(&smb136_charge_work, smb136_off_charging_work);	
	INIT_DELAYED_WORK(&smb136_chg->smb136_interrupt_work, smb136_interrput_status_work);
	smb136_chg->client = client;
	
	smb136_chg->batt_mah_rating = pdata->batt_mah_rating;
	if (smb136_chg->batt_mah_rating == 0)
		smb136_chg->batt_mah_rating = SMB136_DEFAULT_BATT_RATING;

	if (pdata->chg_detection_config)
		ret = pdata->chg_detection_config();
	if (ret) {
		dev_err(&client->dev, "%s valid config failed ret=%d\n",
			__func__, ret);
		goto free_smb136_chg;
	}
	i2c_set_clientdata(client, smb136_chg);
		
	ret = smb136_read_reg(smb136_chg->client, DEV_ID_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read device id reg %d\n", __func__, ret);
		goto free_smb136_chg;
	}
	if(!(temp&DEVICE_IS_SMB136)){
		dev_err(&smb136_chg->client->dev,
			"%s device id is error: %d\n", __func__, temp);
		goto free_smb136_chg;
	}
	printk(KERN_INFO "%s:read smb136 id ok.\n", __func__);
		

	
	smb136_chg_det_irq=client->irq;
	smb136_chg_i2c_client=client;
	
	
	
#if 0
	smb136_off_charging_work((struct work_struct *)NULL);
	
	ret = request_threaded_irq(client->irq, NULL,
				   smb136_valid_handler,
				   IRQF_TRIGGER_FALLING | IRQF_TRIGGER_RISING,
				   "smb136_charger_valid", client);
	
	if (ret) {
		dev_err(&client->dev,"%s request_threaded_irq failed for %d ret =%d\n",
			__func__, client->irq, ret);
		goto free_valid_irq;
	}
	
	enable_irq_wake(smb136_chg_det_irq);	
	
#endif

	
	ret = smb136_write_reg(client, COMMAND_A_REG, 0x80);

	temp = 0;
	ret = smb136_read_reg(smb136_chg->client, OTG_LBR_CTRL_REG, &temp);
	temp |= BATT_MISSING_DET_EN_BIT;
	ret |= smb136_write_reg(client, OTG_LBR_CTRL_REG, temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't set registers for batt missing %d\n",
			__func__, ret);
		goto free_valid_irq;
	}  
 	
	
	ret = smb136_read_reg(smb136_chg->client, CONTROL_B_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read control B register%d\n", __func__, ret);
	} 
	
	
	temp&=0X3F;	
	temp|=BIT(6);
	
	
	temp&=0XFE;
	
	ret = smb136_write_reg(smb136_chg->client,CONTROL_B_REG, temp);
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s couldn't write control B register%d\n", __func__, ret);
	}
	
	
	ret = smb136_read_reg(smb136_chg->client, FAULT_INTR_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read fault interrupt register%d\n", __func__, ret);
	} 
	temp&=0X00;
	temp|= INPUT_OVLO_IVLO_MASK;
	ret = smb136_write_reg(smb136_chg->client,FAULT_INTR_REG, temp);
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s couldn't write fault interrupt register%d\n", __func__, ret);
	}
	
	
	ret = smb136_read_reg(smb136_chg->client, FLOAT_VOLTAGE_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read float register%d\n", __func__, ret);
	} 
	temp|=BIT(7);
	ret = smb136_write_reg(smb136_chg->client,FLOAT_VOLTAGE_REG, temp);
	if(ret){
		dev_err(&smb136_chg->client->dev,
			"%s couldn't write float register%d\n", __func__, ret);
	}

	
	
	

	smb136_chg->valid_n_gpio = pdata->valid_n_gpio;

	smb136_request_gpio(smb136_chg->valid_n_gpio);		
	chg_fault_int = gpio_to_irq(smb136_chg->valid_n_gpio);
	
	ret = request_irq(chg_fault_int, smb136_chg_fault_handler,  IRQF_TRIGGER_FALLING,
		            	"CHG_STAT", client);
	if(ret < 0){
		printk(KERN_ERR "smb136 request_irq chg_fault_int fail\n");
		goto free_valid_gpio;
	}
	set_irq_wake(chg_fault_int, 1);		

	
	
	smb136_chg->psy_batt.name = "battery";
	smb136_chg->psy_batt.type = POWER_SUPPLY_TYPE_BATTERY;
	smb136_chg->psy_batt.properties = batt_power_props;
	smb136_chg->psy_batt.num_properties = ARRAY_SIZE(batt_power_props);
	smb136_chg->psy_batt.get_property = batt_power_get_property;

	smb136_chg->psy_usb.name = "usb";
	smb136_chg->psy_usb.type = POWER_SUPPLY_TYPE_USB;
	smb136_chg->psy_usb.supplied_to = power_supplied_to;
	smb136_chg->psy_usb.num_supplicants = ARRAY_SIZE(power_supplied_to);
	smb136_chg->psy_usb.properties = power_props;
	smb136_chg->psy_usb.num_properties = ARRAY_SIZE(power_props);
	smb136_chg->psy_usb.get_property = power_get_property;

	
	smb136_chg->min_design = 3200;
	smb136_chg->max_design = 4200;

	smb136_chg->batt_status = POWER_SUPPLY_STATUS_DISCHARGING;
	smb136_chg->batt_chg_type = POWER_SUPPLY_CHARGE_TYPE_NONE;

	smb136_chg->adapter_hw_chg.type = CHG_TYPE_USB;
	smb136_chg->adapter_hw_chg.rating = 1;
	smb136_chg->adapter_hw_chg.name = "smb136-adapter";
	smb136_chg->adapter_hw_chg.start_charging = smb136_start_charging;
	smb136_chg->adapter_hw_chg.stop_charging = smb136_stop_charging;
	smb136_chg->adapter_hw_chg.charging_switched = smb136_charging_switched;
	smb136_chg->adapter_hw_chg.get_charging_current = smb136_get_fastcharging_current;
	smb136_chg->adapter_hw_chg.set_charging_current = smb136_set_fastcharging_current;
	smb136_chg->adapter_hw_chg.charger_type=CHARGER_TYPE_INVALID;
	
	smb136_chg->adapter_hw_chg.get_charger_IC_ID=smb136_get_charger_id;
	
	 
	smb136_chg->adapter_hw_chg.battery_is_present= smb136_battery_is_present;
	smb136_chg->adapter_hw_chg.fast_charging_event = smb136_fast_charging_event;
	smb136_chg->adapter_hw_chg.taper_charging_event = smb136_taper_charging_event;
	smb136_chg->adapter_hw_chg.charging_done_event = smb136_charging_done_event;	
	 
	smb136_chg->adapter_hw_chg.is_chg_plugged_rt = msm_is_chg_plugged_in;
	smb136_chg->adapter_hw_chg.set_charging_mode = smb136_set_charging_mode;

	ret = smb136_read_reg(smb136_chg->client, STATUS_F_REG, &temp);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read status f reg %d\n", __func__, ret);
		goto free_valid_irq;
	}
	if (!(temp & BATT_PRESENT_STAT))
		smb136_chg->batt_present = 1;

	usb_smb136_chg = smb136_chg;
	
	ret = msm_charger_register(&smb136_chg->adapter_hw_chg);
	if (ret) {
		dev_err(&smb136_chg->client->dev,
			"%s couldn't read status f reg %d\n", __func__, ret);
		goto free_valid_irq;
	}
	schedule_delayed_work(&smb136_charge_work, (HZ) * 2);
	smb136_dbg_print_all_status_regs(usb_smb136_chg);
	
	smb136_create_debugfs_entries(smb136_chg);
	dev_dbg(&smb136_chg->client->dev, "%s start heartbeat\n", __func__);

	return 0;
	
free_valid_irq:
	free_irq(client->irq, client);
free_valid_gpio:
	gpio_free(smb136_chg->valid_n_gpio);
free_smb136_chg:
	kfree(smb136_chg);
out:
	return ret;
}

void smb136_otg_power(int on)
{
	int ret;

	pr_debug("%s Enter on=%d\n", __func__, on);
	ret=0;
	#ifndef SMB136_REGISTER_TO_MSM	
	if (on) {
		ret = smb136_write_reg(usb_smb136_chg->client,
					PIN_CTRL_REG, PIN_CTRL_REG_CHG_OFF);
		if (ret) {
			pr_err("%s turning off charging in pin_ctrl err=%d\n",
								__func__, ret);
			
			return;
		}

		ret = smb136_write_reg(usb_smb136_chg->client,
			COMMAND_A_REG, COMMAND_A_REG_OTG_MODE);
		if (ret)
			pr_err("%s failed turning on OTG mode ret=%d\n",
								__func__, ret);
	} else {
		ret = smb136_write_reg(usb_smb136_chg->client,
			COMMAND_A_REG, COMMAND_A_REG_DEFAULT);
		if (ret)
			pr_err("%s failed turning off OTG mode ret=%d\n",
								__func__, ret);
		ret = smb136_write_reg(usb_smb136_chg->client,
				PIN_CTRL_REG, PIN_CTRL_REG_DEFAULT);
		if (ret)
			pr_err("%s failed writing to pn_ctrl ret=%d\n",
								__func__, ret);
	}
	#endif
}

void smb136_vbus_draw(unsigned int mA)
{
	pr_debug("%s mA=%d\n", __func__, mA);
	if (usb_smb136_chg == NULL) {
		pr_err("%s Called without charger notifying USB\n", __func__);
		return;
	}
	printk("smb136 vbus_draw=%d\n",mA);
	#ifndef SMB136_REGISTER_TO_MSM
	if (mA == 0)
		smb136_stop_charging(usb_smb136_chg);
	else
		smb136_start_charging(usb_smb136_chg, mA);	
	#endif
}


static int __devexit smb136_remove(struct i2c_client *client)
{
	const struct smb136_platform_data *pdata;
	struct smb136_data *smb136_chg = i2c_get_clientdata(client);

	smb136_chg->stop_heartbeat = true;
	smp_mb();
	pdata = client->dev.platform_data;
	free_irq(client->irq, client);
	
	msm_charger_unregister(&usb_smb136_chg->adapter_hw_chg);
	cancel_delayed_work_sync(&smb136_charge_work);	
 
#ifndef SMB136_REGISTER_TO_MSM
	cancel_delayed_work_sync(&smb136_chg->charge_work);
#endif
	cancel_delayed_work_sync(&smb136_chg->smb136_interrupt_work);
 
	smb136_destroy_debugfs_entries();
	smb136_free_gpio(smb136_chg->valid_n_gpio );
	kfree(smb136_chg);
	return 0;
}

#ifdef CONFIG_PM
static int smb136_suspend(struct device *dev)
{
	struct smb136_data *smb136_chg = dev_get_drvdata(dev);

	dev_dbg(&smb136_chg->client->dev, "%s\n", __func__);
 
  #ifndef SMB136_REGISTER_TO_MSM
	if (delayed_work_pending(&smb136_chg->charge_work)) {
		smb136_chg->stop_heartbeat = true;
		smp_mb();
		cancel_delayed_work_sync(&smb136_chg->charge_work);		
	}
#endif

	if (delayed_work_pending(&smb136_chg->smb136_interrupt_work)) {
		smp_mb();
		cancel_delayed_work_sync(&smb136_chg->smb136_interrupt_work);		
	}

 
  
  
	
	return 0;
}

static int smb136_resume(struct device *dev)
{
	struct smb136_data *smb136_chg = dev_get_drvdata(dev);

	dev_dbg(&smb136_chg->client->dev, "%s\n", __func__);
	smb136_chg->stop_heartbeat = false;
 
  #ifndef SMB136_REGISTER_TO_MSM
	schedule_delayed_work(&smb136_chg->charge_work,
					SMB136_CHG_PERIOD);
  #endif
   

  
	
	return 0;
}

static const struct dev_pm_ops smb136_pm_ops = {
	.suspend = smb136_suspend,
	.resume = smb136_resume,
};
#endif

static const struct i2c_device_id smb136_id[] = {
	{"smb136", 0},
	{},
};
MODULE_DEVICE_TABLE(i2c, smb136_id);

static struct i2c_driver smb136_driver = {
	.driver = {
		   .name = "smb136",
		   .owner = THIS_MODULE,
#ifdef CONFIG_PM
		   .pm = &smb136_pm_ops,
#endif
	},
	.probe = smb136_probe,
	.remove = __devexit_p(smb136_remove),
	.id_table = smb136_id,
};


#if 1
int msm_is_chg_plugged_in(void)
{
	int count = 3;
	int ret;
	int wait_led_count = 20;
	
	
	for( ; (!g_pm_chip_for_all) && (!wait_led_count); wait_led_count --) 
		msleep(150);

	while ((ret =
		pm8058_irq_get_rt_status((struct pm8058_chip *)g_pm_chip_for_all, smb136_chg_det_irq)) == -EAGAIN
	       && count--) {
		printk("%s trycount=%d\n", __func__, count);
		cpu_relax();
	}
	if (ret == -EAGAIN)
		return 0;
	else
		return ret;
}
EXPORT_SYMBOL(msm_is_chg_plugged_in);

static int __init smb136_init(void)
{
	return i2c_add_driver(&smb136_driver);
}
module_init(smb136_init);

static void __exit smb136_exit(void)
{
	return i2c_del_driver(&smb136_driver);
}
module_exit(smb136_exit);

MODULE_AUTHOR("Abhijeet Dharmapurikar <adharmap@codeaurora.org>");
MODULE_DESCRIPTION("Driver for SMB136 Charger chip");
MODULE_LICENSE("GPL v2");
MODULE_ALIAS("i2c:smb136");

#else

#ifndef CONFIG_PM8058_CHARGER
enum pmic_chg_interrupts {
	CHGVAL_IRQ,
	CHGINVAL_IRQ,
	VBATDET_LOW_IRQ,
	VCP_IRQ,
	CHGILIM_IRQ,
	ATC_DONE_IRQ,
	ATCFAIL_IRQ,
	AUTO_CHGDONE_IRQ,
	AUTO_CHGFAIL_IRQ,
	CHGSTATE_IRQ,
	FASTCHG_IRQ,
	CHG_END_IRQ,
	BATTTEMP_IRQ,
	CHGHOT_IRQ,
	CHGTLIMIT_IRQ,
	CHG_GONE_IRQ,
	VCPMAJOR_IRQ,
	VBATDET_IRQ,
	BATFET_IRQ,
	BATT_REPLACE_IRQ,
	BATTCONNECT_IRQ,
	PMIC_CHG_MAX_INTS
};

struct pm8058_charger {
	struct pmic_charger_pdata *pdata;
	struct pm8058_chip *pm_chip;
	struct device *dev;

	int pmic_chg_irq[PMIC_CHG_MAX_INTS];

	int inited;
	int present;
};

static struct pm8058_charger pm8058_chg;

static int pm_chg_get_rt_status(int irq)
{
	int count = 3;
	int wait_led_count = 15;
	int ret;
	static int fitst_call = 1;
	
	if(fitst_call) {
		for( ;!wait_led_count ; wait_led_count--)
			msleep(200);
			
		fitst_call = 0;
	}

	while ((ret =
		pm8058_irq_get_rt_status(g_pm_chip_for_all, smb136_chg_det_irq)) == -EAGAIN
	       && count--) {
		dev_info(pm8058_chg.dev, "%s trycount=%d\n", __func__, count);
		cpu_relax();
	}
	if (ret == -EAGAIN)
		return 0;
	else
		return ret;
}

int msm_is_chg_plugged_in(void)
{
	return pm_chg_get_rt_status(pm8058_chg.pmic_chg_irq[CHGVAL_IRQ]);
}

static int smb136_init(void)
{
	return i2c_add_driver(&smb136_driver);
}

static void smb136_exit(void)
{
	return i2c_del_driver(&smb136_driver);
}

static int __devexit pm8058_charger_remove(struct platform_device *pdev)
{
	smb136_exit();
	return 0;
}

static int __devinit pm8058_charger_probe(struct platform_device *pdev)
{
	struct pm8058_chip *pm_chip;
	struct resource *res;

	pm_chip = platform_get_drvdata(pdev);
	if (pm_chip == NULL) {
		pr_err("%s:no parent data passed in.\n", __func__);
		return -EFAULT;
	}

	pm8058_chg.pm_chip = pm_chip;
	pm8058_chg.pdata = pdev->dev.platform_data;
	pm8058_chg.dev = &pdev->dev;

	res = platform_get_resource_byname(pdev, IORESOURCE_IRQ, "CHGVAL");
	if (res == NULL) {
		dev_err(pm8058_chg.dev,
			"%s:couldnt find resource CHGVAL\n", __func__);
	}
	else {
		pm8058_chg.pmic_chg_irq[CHGVAL_IRQ] = res->start;
	}

	pm8058_chg.inited = 1;

	smb136_init();
	
	return 0;
}

static struct platform_driver pm8058_charger_driver = {
	.probe = pm8058_charger_probe,
	.remove = __devexit_p(pm8058_charger_remove),
	.driver = {
		   .name = "pm8058-charger",
		   .owner = THIS_MODULE,
	},
};

static int __init pm8058_charger_init(void)
{
	return platform_driver_register(&pm8058_charger_driver);
}

static void __exit pm8058_charger_exit(void)
{

}

late_initcall(pm8058_charger_init);
module_exit(pm8058_charger_exit);

MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("PMIC8058 BATTERY driver");
MODULE_VERSION("1.0");
MODULE_ALIAS("platform:pm8058_charger");

#else
  #error not supported this_kind_of configuration, check it.
#endif

#endif