
#include <linux/delay.h>
#include <linux/pwm.h>
#include <linux/module.h>
#include <linux/gpio/gpio_def.h>
#include <mach/gpio.h>
#include <linux/gpio.h>
#include <linux/workqueue.h>
#include <linux/pmic8058-othc.h>
#include <linux/mfd/pmic8901.h>
#include <linux/mfd/pmic8058.h>
#include <linux/regulator/pmic8058-regulator.h>
#include <linux/regulator/pmic8901-regulator.h>
#include <linux/regulator/consumer.h>
#include <linux/regulator/machine.h>
#include "mdp.h"
#include "mdp4.h"
#include "msm_fb.h"

#ifdef CONFIG_FB_MSM_MDP40
#define LCDC_BASE	0xC0000
#else
#define LCDC_BASE	0xE0000
#endif

#include "lcd_sysfs.h"
#ifdef CONFIG_PMIC8058_PWM
#include <linux/pmic8058-pwm.h>
static struct pwm_device *bl_pwm;
#endif

#define BL_LEVEL			30

enum {
	DEBUG_BACKLIGHT = 1U << 0,
};
static int debug_mask = 0;
module_param_named(debug_mask, debug_mask, int, S_IRUGO | S_IWUSR | S_IWGRP);



enum spi_trans_type
{
	WRITE_HIGH_BYTE = 0,
	WRITE_LOW_BYTE,
	WRITE_DATA,
	READ_DATA, 
	SPI_TYPE_MAX
};

static int spi_cmd_table[] = {0x20, 0x00, 0x40, 0xc0 };	
struct work_struct backlight_work;  
static int bl_delay;
extern unsigned get_mfg_mode(void);


static int lcdc_gpio_config(int on)
{
	int gpio_num;
	
	if (on) {
		for(gpio_num=0 ; gpio_num<=27 ; gpio_num++)
			gpio_request(gpio_num, "lcdc_gpio");
	} else {
		for(gpio_num=0 ; gpio_num<=27 ; gpio_num++)
			gpio_free(gpio_num);
	}

	return 0;
}

static int spi_gpio_config(int on)
{
	struct gpio lcdc_ctrl_gpios[] = {
		{LCDC_SPI_CLK, GPIOF_OUT_INIT_LOW, "LCDC_SPI_CLK"}, 
		{LCDC_SPI_SDI, GPIOF_OUT_INIT_LOW, "LCDC_SPI_SDI"},
		{LCDC_SPI_CSI, GPIOF_OUT_INIT_HIGH, "LCDC_SPI_CSI"},
		{LCDC_SPI_SDO, GPIOF_DIR_IN, "LCDC_SPI_SDO"}, 
		{LCD_RESET_N, GPIOF_OUT_INIT_LOW, "LCD_RESET_N"}
	};
	
	if (on) {
		gpio_request_array(lcdc_ctrl_gpios, ARRAY_SIZE(lcdc_ctrl_gpios));
	} else {
		gpio_free_array(lcdc_ctrl_gpios, ARRAY_SIZE(lcdc_ctrl_gpios));
	}

	return 0;
}

#ifdef  CONFIG_PMIC8058_PWM
#define PWM_FREQ_HZ 		6000
#define PWM_PERIOD_USEC 	(USEC_PER_SEC / PWM_FREQ_HZ)
#define PWM_DUTY_LEVEL 		(PWM_PERIOD_USEC / BL_LEVEL)
#endif


static void lcdc_set_backlight(struct msm_fb_data_type *mfd)
{
#ifdef  CONFIG_PMIC8058_PWM
	int bl_level;
	int ret = 0;
	if (bl_delay) {
		mdelay(70);
		bl_delay = 0;
	}
	if (debug_mask & DEBUG_BACKLIGHT)
		printk("%s:%d \n", __func__, mfd->bl_level);

	bl_level = mfd->bl_level;
	
	if (bl_level > BL_LEVEL) {
		bl_level = BL_LEVEL/2;
	} else if (bl_level < 0) {
		bl_level = 0;
	}
	
	bl_duty_cycle = (bl_level *100)/BL_LEVEL;


	if (bl_pwm) {
		pwm_config(bl_pwm, PWM_DUTY_LEVEL*bl_level, PWM_PERIOD_USEC);
		if (ret)
			pr_err("pwm_config() failed!\n");

		ret = pwm_enable(bl_pwm);
		if (ret)
			pr_err("pwm_enable() failed!\n");
	}
#endif

	return;
}

extern int get_Battery_valid(void);

static void backlight_delay_work(struct work_struct *work) 
{  
	struct msm_fb_data_type mfd; 

	if (2 == get_mfg_mode()) 
		mfd.bl_level = 15;
	else
		mfd.bl_level = 9;
		
	if(!get_Battery_valid())
		mfd.bl_level = 0;
		
	mdelay(120);
	lcdc_set_backlight(&mfd); 
 } 

static int lcdc_reset(void)
{
	gpio_set_value_cansleep(LCD_RESET_N,1);
	udelay(30);
	gpio_set_value_cansleep(LCD_RESET_N,0);
	udelay(30);	
	gpio_set_value_cansleep(LCD_RESET_N,1);

	return 0;
}

static void lcdc_spi_write(uint8_t data, uint32 type)
{
	uint8_t bits;
	uint8_t bnum = 8;
	uint8_t cmd;

	if (type >= SPI_TYPE_MAX) {
		printk("%s: unknown command!\n", __func__);
		return ;
	}

	cmd = spi_cmd_table[type];
	
	gpio_set_value_cansleep(LCDC_SPI_CSI, 0);	
	udelay(1);
	
	bits = 0x80;
	while (bnum) {
		gpio_set_value_cansleep(LCDC_SPI_CLK, 0); 
		if (cmd & bits)
			gpio_set_value_cansleep(LCDC_SPI_SDI, 1);
		else
			gpio_set_value_cansleep(LCDC_SPI_SDI, 0);
		udelay(1);
		gpio_set_value_cansleep(LCDC_SPI_CLK, 1); 
		udelay(1);
		bits >>= 1;
		bnum--;
	}

	
	bnum = 8;
	bits = 0x80;
	while (bnum) {
		gpio_set_value_cansleep(LCDC_SPI_CLK, 0); 
		if (data & bits)
			gpio_set_value_cansleep(LCDC_SPI_SDI, 1);
		else
			gpio_set_value_cansleep(LCDC_SPI_SDI, 0);
		udelay(1);
		gpio_set_value_cansleep(LCDC_SPI_CLK, 1); 
		udelay(1);
		bits >>= 1;
		bnum--;
	}

	gpio_set_value_cansleep(LCDC_SPI_CLK, 0); 
	udelay(1);	
   	gpio_set_value_cansleep(LCDC_SPI_CSI, 1);	
   	udelay(1);

	return;

}


static int lcdc_write(uint16 address,uint8_t data)
{  
	uint8_t high_byte, low_byte;

	low_byte =  (uint8_t)address;
	high_byte = (uint8_t)(address >> 8);
	lcdc_spi_write(high_byte, WRITE_HIGH_BYTE);
	lcdc_spi_write(low_byte, WRITE_LOW_BYTE);
	lcdc_spi_write(data, WRITE_DATA);

  	return 0;
}

#ifdef CONFIG_DETECT_PANEL
static void lcdc_spi_read(uint8 *data, uint8_t type)
{
	uint8_t bits;
	uint8_t bnum = 8;
	uint8_t dbit=0;
	uint8_t cmd;
	
	if (type >= SPI_TYPE_MAX) {
		printk("%s: unknown command!\n", __func__);
		return ;
	}

	cmd = spi_cmd_table[type];
	
	gpio_set_value_cansleep(LCDC_SPI_CSI, 0);	
	udelay(1);
	
	
	bits = 0x80;
	while (bnum) {
		gpio_set_value_cansleep(LCDC_SPI_CLK, 0); 
		if (cmd & bits)	
			gpio_set_value_cansleep(LCDC_SPI_SDI, 1);
		else
			gpio_set_value_cansleep(LCDC_SPI_SDI, 0);
		udelay(1);
		gpio_set_value_cansleep(LCDC_SPI_CLK, 1); 
		udelay(1);
		bits >>= 1;
		bnum--;
	}

	
	bnum = 8;
	bits=0;
	while (bnum) {
		bits <<= 1;
		gpio_set_value_cansleep(LCDC_SPI_CLK, 0); 
		udelay(1);
		dbit = gpio_get_value(LCDC_SPI_SDO);		
		gpio_set_value_cansleep(LCDC_SPI_CLK, 1); 
		udelay(1);
		bits |= dbit;
		bnum--;
	}
	
	*data = bits & 0xFF;

	gpio_set_value_cansleep(LCDC_SPI_CLK, 0); 
	udelay(1);	
   	gpio_set_value_cansleep(LCDC_SPI_CSI, 1);	
   	udelay(1);

	return;
}


static int lcdc_read(uint16 address,uint8_t *data)
{  
	uint8_t high_byte, low_byte;

	low_byte =  (uint8_t)address;
	high_byte = (uint8_t)(address >> 8);
	lcdc_spi_write(high_byte, WRITE_HIGH_BYTE);
	lcdc_spi_write(low_byte, WRITE_LOW_BYTE);	
	lcdc_spi_read(data, READ_DATA);

	return 0;	
}


static int lcdc_detect_panel(void)
{
	uint8_t value=0xff;
	
	lcdc_read(0x0a00, &value);	
	if (value == 0x9c) {		
		printk("%s: Detected Auo panel!\n", __func__);
		return 0;
	}

	printk("%s: fail to communicate with panel!\n", __func__);
	return 1;
}
#endif

static int lcdc_init(void)
{
	int ret=0;

	
	lcdc_write(0x1100, 0);	
	lcdc_write(0xC000, 0x8A);
	lcdc_write(0xC002, 0x8A);
	lcdc_write(0xC200, 0x02);
	lcdc_write(0xC202, 0x32);
	lcdc_write(0xC100, 0x40);
	lcdc_write(0xC700, 0x8B);
	mdelay(120);
	lcdc_write(0x2900, 0);  	
	lcdc_write(0x2C00, 0); 	

	lcdc_write(0x3600, 0x03);  	
	return ret;
}


static int lcdc_power(int on)
{
	static struct regulator *vreg_vdd;
	int rc;

	if (on) {
		vreg_vdd = regulator_get(NULL, "8058_l15");
		if (IS_ERR(vreg_vdd)) {
			printk(" unable to get vreg_vdd \n");
			return 0;
		}
		rc = regulator_set_voltage(vreg_vdd, 3000000, 3000000);
		if (rc) {
			printk(" unable to set vreg_vdd result is %d \n", rc);
			if (vreg_vdd)
				regulator_put(vreg_vdd);
			return 0;
		}
		if (regulator_enable(vreg_vdd)) {
			printk("unable to enable vreg_vdd result \n");
			return 0;
		}
	} else {
		regulator_disable(vreg_vdd);
		regulator_put(vreg_vdd);
	}
	   	
	return 0;

}

 int lcdc_panel_on(struct platform_device *pdev)
{
	int ret = 0;
	static int count = 0;
	struct msm_fb_data_type mfd; 
	
	printk("%s\n", __func__);

   	lcdc_power(1);
	
	lcdc_reset();		
	mdelay(20);		
	mdelay(40);		
	lcdc_init();		
	
#ifdef CONFIG_DETECT_PANEL
	lcdc_detect_panel();	
#endif

	
	if (!count) {
		
		mfd.bl_level = 0;
		lcdc_set_backlight(&mfd);	

		

			schedule_work(&backlight_work);
		count = 1;
		return ret;
	}

	bl_delay = 1;
	
	return ret;
}


 int lcdc_panel_off(struct platform_device *pdev)
{
	int ret = 0;

	printk("%s\n", __func__);

	mdelay(20);		
	lcdc_write(0x2800, 0);	
	lcdc_write(0x1000, 0);	
	mdelay(40);		
	lcdc_write(0x4f00, 0x01);	
	gpio_set_value_cansleep(LCD_RESET_N,0);	

	mdelay(10);
	lcdc_power(0);		
	
	
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_ON, FALSE);
	MDP_OUTP(MDP_BASE + LCDC_BASE, 0);
	mdp_pipe_ctrl(MDP_CMD_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
	mdp_pipe_ctrl(MDP_OVERLAY0_BLOCK, MDP_BLOCK_POWER_OFF, FALSE);
	mdp_histogram_ctrl(FALSE);
	return ret;
}


static int lcdc_panel_probe(struct platform_device *pdev)
{
	int rc = 0;
#ifdef CONFIG_PMIC8058_PWM
	bl_pwm = pwm_request(0, "lcdc-backlight");

	if (bl_pwm == NULL || IS_ERR(bl_pwm)) {
		pr_err("%s pwm_request() failed!\n", __func__);
		bl_pwm = NULL;
	}

	bl_frequency = PWM_FREQ_HZ;
	
	printk("%s: bl_pwm = 0x%p \n", __func__, bl_pwm);
#endif

	
	spi_gpio_config(1);	
	lcdc_gpio_config(1);
	
	msm_fb_add_device(pdev);

	return rc;
}

static int __devexit lcdc_panel_remove(struct platform_device *pdev)
{
	return 0;
}

static struct platform_driver this_driver = {
	.probe  = lcdc_panel_probe,
	.remove = lcdc_panel_remove,
	.driver = {
		.name   = "lcdc_wvga",
	},
};

static struct msm_fb_panel_data panel_data = {
	.on = lcdc_panel_on,
	.off = lcdc_panel_off,
	.set_backlight = lcdc_set_backlight,
};

static struct platform_device this_device = {
	.name   = "lcdc_wvga",
	.id	= 1,
	.dev	= {
		.platform_data = &panel_data,
	}
};

static int __init lcdc_panel_init(void)
{
	int ret;
	struct msm_panel_info *pinfo;
       
	printk("%s\n", __func__);
	
	ret = platform_driver_register(&this_driver);
	if (ret)
		return ret;

	pinfo = &panel_data.panel_info;

	pinfo->xres = 480;
	pinfo->yres = 800;
	pinfo->type = LCDC_PANEL;
	pinfo->pdest = DISPLAY_1;
	pinfo->wait_cycle = 0;
	pinfo->bpp = 24;
	pinfo->fb_num = 2;
	pinfo->clk_rate = 25600000;
	pinfo->bl_max = BL_LEVEL;
	pinfo->bl_min = 1;

	pinfo->lcdc.h_back_porch = 2;
	pinfo->lcdc.h_front_porch = 2;
	pinfo->lcdc.h_pulse_width = 2;
	pinfo->lcdc.v_back_porch = 2;
	pinfo->lcdc.v_front_porch = 2;
	pinfo->lcdc.v_pulse_width = 2;
	pinfo->lcdc.border_clr = 0;     
	pinfo->lcdc.underflow_clr = 0xff; 
	pinfo->lcdc.hsync_skew = 0;

	strcpy(pinfo->version, "AUO_NT35582_V0");
	INIT_WORK(&backlight_work, backlight_delay_work);
	
	ret = platform_device_register(&this_device);
	if (ret)
		platform_driver_unregister(&this_driver);

	return ret;
}
module_init(lcdc_panel_init);
